/////////////////////////////////////////////////////////////////////////////
//
//           F L I G H T S A F E T Y   I N T E R N A T I O N A L
//                     Simulation Systems Division
//                      2700 North Hemlock Circle
//                     Broken Arrow, Oklahoma 74012
//                          (918) 259-4000
/////////////////////////////////////////////////////////////////////////////
//
// DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
// Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
// technical or operational data or information from automatic dissemination
// under the International Exchange Program or by other means.  This protection
// covers information required solely for administrative or operational
// purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
//
// WARNING:  This document contains technical data whose export is restricted
// by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
// Executive Order 12470.  Violation of these export control laws is subject
// to severe criminal penalties.  Dissemination of this document is controlled
// under DOD Directive 5230.25
//
/////////////////////////////////////////////////////////////////////////////
//
//
// Filename         : ApproachTrack.cpp
//
// Date             : 09 August 1999
//
// Engineer         : Billy Baker
//
// Revision         : $Revision: 1.5 $
//
// Description      : ApproachTrack.cpp contains the implementation of the 
//                    CApproachTrack class. This class is used in 
//                    conjunction with the approach plot to show the 
//                    top, side, and vref data.  Rather than using
//                    OpenGL's transformation methods, an R1 R2 T1 R3 T2
//                    matrix is precomputed.  The rotations rely on data
//                    that only changes when the runway changes.  And, T1
//                    has a fixed translation.  The values that don't change
//                    are member variables with accessors and mutators.  As
//                    these members changes, a number of sin and cos variables
//                    are also updated.
//
// Classification   : UNCLASSIFIED
//
// Requirements     : None.
//
// Components Used  : math.h
//
// Operational 
//    Restrictions  : Machine dependencies/restrictions
//                        None.
//                    Design dependencies/restrictions
//                        None.
//                    Operations containing dependencies/restrictions
//                        None.
//                    Compiler dependencies/restrictions
//                        None.
//                    Other conditions for proper execution
//                        None.
//
// Environment      : Operating system(s) - Microsoft Windows NT 4.0 with
//                                              NT service pack 3, 4, or 5
//                                          Microsoft Windows NT 2000
//
//                    Compiler(s) - Visual C++ 6.0
//
//                    Architechure(s) - Intel Pentium, Pentium II
/////////////////////////////////////////////////////////////////////////////

/////////////////////////////////////////////////////////////////////////////
//
//                              R e v i s i o n   H i s t o r y
//
/////////////////////////////////////////////////////////////////////////////
// $Log: approachtrack.cpp $                                                                   
// Revision 1.5  2000/01/27 20:00:15  billyb                                                                   
// Changed byte by byte memset to 4 byte by 4 byte memset.                                                                   
// Revision 1.4  2000/01/27 19:54:51  billyb                                                                   
// Corrected possible memory leaks/corruption with copy                                                                   
// constructor.  Changed memcpy to non-byte by byte copy.                                                                   
// Changed list types to match new type in CVariant.                                                                   
// Revision 1.3  1999/11/26 21:02:52  billyb                                                                   
// Changed signature of UpdateData.  Changed UpdateData                                                                   
// to work with the new format for the persistant CVariant data.                                                                   
// Revision 1.2  1999/10/11 19:53:13  billyb                                                                   
// Revision 1.2  1999/09/14 19:38:04  billyb                                                                   
// Added extra conditionals to prevent trying to add                                                                    
// or draw out of bounds data.  Added VerticesAdded                                                                    
// accessor for m_ulVerticesAdded which is used with                                                                    
// a full list after a clear.                                                                   
/////////////////////////////////////////////////////////////////////////////
#include "..\core\stdafx.h"
#include "ApproachTrack.h"
#include "..\core\DataConversion.h"
#include "OpenGLMap.h"
#include "ApproachPlot.h"
#include "ToleranceData.h"

CApproachTrack::CApproachTrack()
{
    m_dTouchdownLon                     = 0.0;
    m_dTouchdownLat                     = 0.0;
    m_dLocalizerLon                     = 0.0;
    m_dLocalizerLat                     = 0.0;
    m_fLocalizerHeading                 = 0.0f;
    m_fLocalizerDeviation               = 0.0f;
    m_fLocalizerToGlideslopeDistance    = 0.0f;
    m_fScale                            = 1.0f;
    m_vertices                          = NULL;
    m_ulArraySize                       = 0;
    m_ulVerticesAdded                   = 0;
    m_fAspectRatioForVrefPlot           = 1.0f;
    m_fElevation                        = 0.0f;
    m_fMaxVrefDeviationShown            = 1.0f;
    m_fVRef                             = 0.0f;
    m_nApproachType                     = 1;
    cos_of_loc_hdg                      = 0.0;
    sin_of_loc_hdg                      = 0.0;
    m_fNMToGlideslope                   = 0.0f;
    m_fNMToTouchdown                    = 0.0f;
    m_hWnd                              = NULL;
    m_eTDC                              = ApproachPlot;
    m_nApproachViewType                 = 0;

   m_ulLastTimeValue = 0;
   m_times           = NULL;

    ArraySize(13500);
}

CApproachTrack::CApproachTrack(const CApproachTrack& rApproachTrack)
{
    m_dTouchdownLon                     = 0.0;
    m_dTouchdownLat                     = 0.0;
    m_dLocalizerLon                     = 0.0;
    m_dLocalizerLat                     = 0.0;
    m_fLocalizerHeading                 = 0.0f;
    m_fLocalizerDeviation               = 0.0f;
    m_fLocalizerToGlideslopeDistance    = 0.0f;
    m_fScale                            = 1.0f;
    m_vertices                          = NULL;
    m_ulArraySize                       = 0;
    m_ulVerticesAdded                   = 0;
    m_fAspectRatioForVrefPlot           = 1.0f;
    m_fElevation                        = 0.0f;
    m_fMaxVrefDeviationShown            = 1.0f;
    m_fVRef                             = 0.0f;
    m_nApproachType                     = 1;
    cos_of_loc_hdg                      = 0.0;
    sin_of_loc_hdg                      = 0.0;
    m_fNMToGlideslope                   = 0.0f;
    m_fNMToTouchdown                    = 0.0f;
    m_hWnd                              = NULL;
    m_eTDC                              = ApproachPlot;
    m_nApproachViewType                 = 0;

    m_ulLastTimeValue = 0;
    m_times           = NULL;

    ArraySize(13500);

    *this = rApproachTrack;
}

CApproachTrack::~CApproachTrack()
{
    if (m_vertices != NULL)
    {
        delete [] m_vertices;

        m_vertices = NULL;
    }

   if (m_times != NULL)
   {
       delete [] m_times;

       m_times = NULL;
   }
}

CApproachTrack& CApproachTrack::operator=(const CApproachTrack& rApproachTrack)
{
	// no work if they are the same
	if (this != &rApproachTrack)
	{
        if (m_vertices != NULL)
        {
            delete [] m_vertices;
            m_vertices          = NULL;
            m_ulVerticesAdded   = 0;
        }

        if (m_times != NULL)
        {
            delete [] m_times;
            m_times = NULL;
        }

        TouchdownLon(rApproachTrack.m_dTouchdownLon);
        TouchdownLat(rApproachTrack.m_dTouchdownLat);

        LocalizerHeading(rApproachTrack.m_fLocalizerHeading);

        Scale(rApproachTrack.m_fScale);

        ArraySize(rApproachTrack.m_ulArraySize);

        ApproachType(rApproachTrack.m_nApproachType);

        AspectRatioForVrefPlot(rApproachTrack.m_fAspectRatioForVrefPlot);

        Elevation(rApproachTrack.m_fElevation);

        LocalizerLat(rApproachTrack.m_dLocalizerLat);
        LocalizerLon(rApproachTrack.m_dLocalizerLon);

        LocalizerToGlideslopeDistance(rApproachTrack.m_fLocalizerToGlideslopeDistance);

        MaxVrefDeviationShown(rApproachTrack.m_fMaxVrefDeviationShown);

        VRef(rApproachTrack.m_fVRef);

        m_ulVerticesAdded = rApproachTrack.m_ulVerticesAdded;

        unsigned long ul = 0;
        unsigned long ulSize = m_ulArraySize;
        unsigned long ulTimeIndex = 0;
        while (ulSize)
        {
            m_vertices[ul]      = rApproachTrack.m_vertices[ul];
            m_vertices[ul + 1]  = rApproachTrack.m_vertices[ul + 1];
            m_vertices[ul + 2]  = rApproachTrack.m_vertices[ul + 2];

            m_times[ulTimeIndex] = rApproachTrack.m_times[ulTimeIndex];

            ++ulTimeIndex;

            ul += 3;
            --ulSize;
        }

        Hwnd(rApproachTrack.m_hWnd);
        TypeOfTrack(rApproachTrack.m_eTDC);
        ApproachViewType(rApproachTrack.m_nApproachViewType);
	}

	return *this;
}
    
bool CApproachTrack::operator==(const CApproachTrack& rApproachTrack)
{
    bool bRetVal = true;

    if (m_dTouchdownLon != rApproachTrack.m_dTouchdownLon)
        bRetVal = false;

    if (m_dTouchdownLat != rApproachTrack.m_dTouchdownLat)
        bRetVal = false;

    if (m_fLocalizerHeading != rApproachTrack.m_fLocalizerHeading)
        bRetVal = false;

    if (m_fScale != rApproachTrack.m_fScale)
        bRetVal = false;

    if (m_ulVerticesAdded != rApproachTrack.m_ulVerticesAdded)
        bRetVal = false;

    if (m_ulArraySize != rApproachTrack.m_ulArraySize)
        bRetVal = false;

    if (m_nApproachType != rApproachTrack.m_nApproachType)
        bRetVal = false;

    if (m_fAspectRatioForVrefPlot != rApproachTrack.m_fAspectRatioForVrefPlot)
        bRetVal = false;

    if (m_fElevation != rApproachTrack.m_fElevation)
        bRetVal = false;

    if (m_dLocalizerLat != rApproachTrack.m_dLocalizerLat)
        bRetVal = false;

    if (m_dLocalizerLon != rApproachTrack.m_dLocalizerLon)
        bRetVal = false;

    if (m_fLocalizerToGlideslopeDistance != rApproachTrack.m_fLocalizerToGlideslopeDistance)
        bRetVal = false;

    if (m_fMaxVrefDeviationShown != rApproachTrack.m_fMaxVrefDeviationShown)
        bRetVal = false;

    if (m_fVRef   != rApproachTrack.m_fVRef)
        bRetVal = false;

    if (m_hWnd != rApproachTrack.m_hWnd)
        bRetVal = false;

    if (m_eTDC != rApproachTrack.m_eTDC)
        bRetVal = false;

    if (m_nApproachViewType != rApproachTrack.m_nApproachViewType)
        bRetVal = false;

    return bRetVal;
}

void  CApproachTrack::Draw(void)
{
    if (m_ulVerticesAdded <= 0 || m_ulVerticesAdded > m_ulArraySize)
        return;

    glColor3ub(255, 0, 0);

    glVertexPointer(3, GL_FLOAT, 0, (const void*)m_vertices);
    glDrawArrays(GL_LINE_STRIP, 0, m_ulVerticesAdded);
}

void CApproachTrack::UpdateData()
{
    if (m_ulArraySize < 1)
        return;

    // Get the pointers for start of new data and last clear.
    bool bOk;
    _FSI_STL::list<struct TolData>::iterator itStart;
    _FSI_STL::list<struct TolData>::iterator itClear;

    CToleranceData::ClearIt(m_hWnd, m_eTDC, bOk, itClear);
    if (!bOk)
        return;

    CToleranceData::StartIt(m_hWnd, m_eTDC, bOk, itStart);
    if (!bOk)
        return;

    unsigned short usAdded      = CToleranceData::Added(m_hWnd, m_eTDC);
/*
    bool bAddClears = false;
    // See if we need to add data starting at the last clear.
    if (m_ulVerticesAdded == 0 && itClear != itStart)
    {
        bAddClears = true;
    }
*/
    if (CToleranceData::ClearSinceLastUpdate(m_hWnd, m_eTDC))
        m_ulVerticesAdded = 0;

    unsigned long ulCurIndex    = 3 * m_ulVerticesAdded;
    unsigned long ulLoop        = usAdded;
    struct TolData td;
/*
    // Add data since last clear.  Should occur if the vertices currently added
    // is 0 and the clear pointer is not at the same point as the start pointer.
    if (bAddClears)
    {
        while (itClear != itStart)
        {
           td = (*itClear);

           AddVertice(td, ulCurIndex);

           ulCurIndex += 3;

           ++itClear;
           ++m_ulVerticesAdded;

            // Check to see if the maximum number of vertices have been
            // added and there is still more clear data to add.  If so,
            // then decrease the number of vertices added by one and move all
            // of the data down.  Also, move the current index back down 3.
            if (m_ulVerticesAdded == m_ulArraySize && itClear != itStart)
            {
               ulCurIndex -= 3;

               --m_ulVerticesAdded;

				int i = 0;
				int j = 3;
				int k = m_ulVerticesAdded * 3;
                int nTimeIndex1 = 0;
                int nTimeIndex2 = 1;
				while (i < k)
				{
					m_vertices[i]	= m_vertices[j];
					m_vertices[i+1] = m_vertices[j+1];
					m_vertices[i+2] = m_vertices[j+2];

                    m_times[nTimeIndex1]    = m_times[nTimeIndex2];

					j += 3;
					i += 3;

                    ++nTimeIndex1;
                    ++nTimeIndex2;
				}
            }
        }
    }
*/
    // Max number of vertices added and more need to be added.
    // Move the data down the number of vertices that need to
    // be added.
    if ((m_ulVerticesAdded + usAdded) > m_ulArraySize && usAdded > 0)
    {
        int nNumberOfVerticesBeyondMax = (m_ulVerticesAdded + usAdded - m_ulArraySize);

		// If greater than the max size, then all of the data will get replaced
		// and a move is not necessary.
		if (nNumberOfVerticesBeyondMax < m_ulArraySize)
		{
			// Make sure that VerticesAdded stays non-negative.
			// To go negative would mean that the amount beyond the
			// end is greater than the number already in the array.  Thus,
			// all of the vertices in the array will get removed and the
			// number added would be zero.
			if (nNumberOfVerticesBeyondMax < m_ulVerticesAdded)
			{
				m_ulVerticesAdded -= nNumberOfVerticesBeyondMax;
				int i = 0;
				int j = nNumberOfVerticesBeyondMax * 3;
				int k = m_ulVerticesAdded * 3;
                int nTimeIndex1 = 0;
                int nTimeIndex2 = 1;
				while (i < k)
				{
					m_vertices[i]	= m_vertices[j];
					m_vertices[i+1] = m_vertices[j+1];
					m_vertices[i+2] = m_vertices[j+2];

                    m_times[nTimeIndex1]    = m_times[nTimeIndex2];

					j += 3;
					i += 3;

                    ++nTimeIndex1;
                    ++nTimeIndex2;
				}
			}
			else
			{
				m_ulVerticesAdded = 0;
				if (usAdded > m_ulArraySize)
				{
					usAdded = m_ulArraySize;
					ulLoop = usAdded;
				}
			}

			ulCurIndex = 3 * m_ulVerticesAdded;
		}
		else
		{
			while (nNumberOfVerticesBeyondMax > m_ulArraySize)
			{
				--usAdded;
				++itStart;
				--nNumberOfVerticesBeyondMax;
			}

			if (usAdded > m_ulArraySize)
				usAdded -= m_ulArraySize;

			m_ulVerticesAdded	= 0;
			ulCurIndex			= 0;
			ulLoop				= m_ulArraySize;
		}
    }

    // All deques are assumed to have single element arrays as data.
    while (ulLoop)
    {
        td = (*itStart);

        AddVertice(td, ulCurIndex);

        ulCurIndex += 3;

        ++itStart;
        --ulLoop;
    }

    if (usAdded > 0)
        m_ulVerticesAdded += usAdded;

    // If the last time value minus the fist time value in
    // the array of times is greater than or equal to the
    // array size which should be the maximum time distance
    // between points then move all of the data down one.
    int nTimeIndex = 0;
    while (nTimeIndex < m_ulVerticesAdded &&
           (m_ulLastTimeValue - m_times[nTimeIndex]) >= m_ulArraySize)
    {
        ++nTimeIndex;
    }

    m_ulVerticesAdded -= nTimeIndex;

    int i = 0;
	int j = nTimeIndex * 3;
	int k = m_ulVerticesAdded * 3;
    int nTimeIndex1 = 0;
    int nTimeIndex2 = nTimeIndex;
	while (i < k)
	{
		m_vertices[i]	= m_vertices[j];
		m_vertices[i+1] = m_vertices[j+1];
		m_vertices[i+2] = m_vertices[j+2];

        m_times[nTimeIndex1]    = m_times[nTimeIndex2];

		j += 3;
		i += 3;

        ++nTimeIndex1;
        ++nTimeIndex2;
    }
}

void CApproachTrack::AddVertice(struct TolData& td, unsigned long ulCurIndex)
{
    // No error checking.  Called from UpdateData.
    double  dXVal = 0.0;
    double  dYVal = 0.0;
    double  xgas, ygas, agas, bgas;
    double  ac_latitude;
    double  ac_longitude;
    float   ac_altitude;
    float   ac_airspeed;

    m_ulLastTimeValue = td.s_ulTime;
    m_times[ulCurIndex / 3] = m_ulLastTimeValue;

    ac_latitude  = td.s_dLat;
    ac_longitude = td.s_dLon;

    // The xgas term is the north-south distance from the aircraft to the selected ground station, in nautical miles.  
    xgas              =  (m_dLocalizerLat  -  ac_latitude)    *  NM_PER_DEGREE;
    // The ygas term is the east-west distance from the aircraft to the selected ground station, in nautical miles.
    ygas              =  (m_dLocalizerLon  -  ac_longitude)   *  NM_PER_DEGREE  *  cos(((m_dLocalizerLat +  ac_latitude) /  2.0)  *  DEG_TO_RAD);

    // The agas term is the distance from the aircraft to the facility, in nautical miles, measured along the axis of the facility heading. 
    // Positive values indicate the aircraft is on the front course, with negative values indicate the aircraft is on the back course.
    agas              =  ygas  *  sin_of_loc_hdg +  xgas  *  cos_of_loc_hdg;
    // The bgas term is the distance from the aircraft to the facility centerline, in nautical miles, measured perpendicular to the 
    // facility centerline.  Positive values indicate the aircraft is to the right of the facility centerline, and negative values 
    // indicate the aircraft is to the left of the facility centerline.
    bgas              =  xgas  *  sin_of_loc_hdg -  ygas  *  cos_of_loc_hdg;

    if (m_nApproachType & LOCALIZER_DIRECTIONAL_AID)
    {
        // Finally, m_nm_to_gs is the distance from the aircraft to the glideslope shack measured along the localizer's heading.
        m_fNMToGlideslope          =  agas  -  m_fLocalizerToGlideslopeDistance  /  NMI_FT;

        dXVal = m_fNMToGlideslope * m_fScale / 3600.0;
    }
    else
    {
        // The x_distance term is the north-south distance from the aircraft to the touchdown point, in nautical miles.  
        double   x_distance        =  (m_dTouchdownLat   -  ac_latitude)    *  NM_PER_DEGREE;
        // The y_distance term is the east-west distance from the aircraft to the touchdown point, in nautical miles.  
        double   y_distance        =  (m_dTouchdownLon   -  ac_longitude)   *  NM_PER_DEGREE  *  cos(((m_dTouchdownLat +  ac_latitude) /  2.0)  *  DEG_TO_RAD);
        // The ac_to_td_bearing term is the angle between the aircraft and the touchdown point.
        float    ac_to_td_bearing  =  atan2(y_distance, x_distance) *  RAD_TO_DEG;
        // Calculate the distance to the touchdown point, in nautical miles.
        m_fNMToTouchdown           =  sqrt(x_distance   *  x_distance  +  y_distance  *  y_distance) *  cos((ac_to_td_bearing   -  m_fLocalizerHeading) *  DEG_TO_RAD);

        dXVal = m_fNMToTouchdown * m_fScale / 3600.0;
    }

    m_fLocalizerDeviation      =  bgas  *  NMI_FT;  // Calculate the localizer deviation.

    if (m_nApproachViewType == 1) // From approach plot, side view plot.
    {
       ac_altitude  =  td.s_fAlt;

       float  ft_above_touchdown_value   =  ac_altitude -  m_fElevation;

       dYVal    =  -ALTITUDE_SCALE_FACTOR  *  (ft_above_touchdown_value  /  NMI_FT)  *  (m_fScale   /  3600.0f);
    }
    else if (m_nApproachViewType == 2) // From approach plot, VREF plot.
    {
       ac_airspeed  =  td.s_fAirpseed;

       float  vref_deviation =  ac_airspeed -  m_fVRef;

       dYVal    =  -((vref_deviation /  m_fMaxVrefDeviationShown)  *  m_fAspectRatioForVrefPlot);
    }
    else // From approach plot, top view plot.
       dYVal    =  (m_fLocalizerDeviation / NMI_FT) * m_fScale / 3600.0;

    m_vertices[ulCurIndex]      = dXVal;
    m_vertices[ulCurIndex + 1]  = dYVal;
    m_vertices[ulCurIndex + 2]  = 0.0;
}

/////////////////////////////////////////////////////////////////////////////
//
// void CApproachTrack::RecomputeOnAirportRunwayChange()
//
// Inputs           : None.
//
// Return Values    : None.
//
// Date             : 25 September 2000
//
// Engineer         : Billy Baker
//
// Description      : RecomputeOnAirportRunwayChange should be called when
//                    the active aiport/runway is changed while the approach
//                    plot is displayed.  This will recompute all of the
//                    vertices based on the new touchdown.
//
/////////////////////////////////////////////////////////////////////////////
void CApproachTrack::RecomputeOnAirportRunwayChange()
{
    if (m_ulArraySize < 1)
        return;

    // Get the pointers for start of new data and last clear.
    bool bOk;
    _FSI_STL::list<struct TolData>::iterator itStart;
    _FSI_STL::list<struct TolData>::iterator itClear;

    CToleranceData::ClearIt(m_hWnd, m_eTDC, bOk, itClear);
    if (!bOk)
        return;

    CToleranceData::StartIt(m_hWnd, m_eTDC, bOk, itStart);
    if (!bOk)
        return;

    // Reset the number of vertices to zero so that all of the
    // data will be recomputed.
    m_ulVerticesAdded           = 0;

    unsigned long ulCurIndex    = 0;
    struct TolData td;

    // Add data since last clear.  Should occur if the vertices currently added
    // is 0 and the clear pointer is not at the same point as the start pointer.
    while (1)
    {
       td = (*itClear);

       AddVertice(td, ulCurIndex);

       ulCurIndex += 3;

       ++m_ulVerticesAdded;

       if (itClear == itStart)
          break;

       ++itClear;

        // Check to see if the maximum number of vertices have been
        // added and there is still more clear data to add.  If so,
        // then decrease the number of vertices added by one and move all
        // of the data down.  Also, move the current index back down 3.
        if (m_ulVerticesAdded == m_ulArraySize)
        {
           ulCurIndex -= 3;

           --m_ulVerticesAdded;

           int i = 0;
           int j = 3;
           int k = m_ulVerticesAdded * 3;
           while (i < k)
           {
               m_vertices[i]	= m_vertices[j];
               m_vertices[i+1] = m_vertices[j+1];
               m_vertices[i+2] = m_vertices[j+2];

               j += 3;
               i += 3;
           }
        }
    }
}

/////////////////////////////////////////////////////////////////////////////
//
// void CApproachTrack::Clear()
//
// Inputs           : None.
//
// Return Values    : None.
//
// Date             : 13 July 1999
//
// Engineer         : Billy Baker
//
// Description      : This method is used to tell the CApproachTrack 
//                    class that a clear is needed.
//
/////////////////////////////////////////////////////////////////////////////
void CApproachTrack::Clear()
{
    m_ulVerticesAdded = 0;
    CToleranceData::Clear(m_hWnd, m_eTDC);
}

//////////////////////////////////////////////////////////////////////
// Accessors
//////////////////////////////////////////////////////////////////////

double CApproachTrack::TouchdownLon()
{
    return m_dTouchdownLon;
}

double CApproachTrack::TouchdownLat()
{
    return m_dTouchdownLat;
}

double CApproachTrack::LocalizerLon()
{
    return m_dLocalizerLon;
}

double CApproachTrack::LocalizerLat()
{
    return m_dLocalizerLat;
}

float CApproachTrack::LocalizerHeading()
{
    return m_fLocalizerHeading;
}

float CApproachTrack::LocalizerDeviation()
{
    return m_fLocalizerDeviation;
}

float CApproachTrack::LocalizerToGlideslopeDistance()
{
    return m_fLocalizerToGlideslopeDistance;
}

int CApproachTrack::ApproachType()
{
    return m_nApproachType;
}

float CApproachTrack::Scale()
{
    return m_fScale;
}

unsigned long int CApproachTrack::ArraySize()
{
    return m_ulArraySize;
}

float CApproachTrack::NMToGS()
{
    return m_fNMToGlideslope;
}

float CApproachTrack::NMToTouchdown()
{
    return m_fNMToTouchdown;
}

float CApproachTrack::Elevation()
{
    return m_fElevation;
}

float CApproachTrack::VRef()
{
    return m_fVRef;
}

float CApproachTrack::MaxVrefDeviationShown()
{
   return m_fMaxVrefDeviationShown;
}

float CApproachTrack::AspectRatioForVrefPlot()
{
   return m_fAspectRatioForVrefPlot;
}

unsigned long int CApproachTrack::VerticesAdded()
{
    return m_ulVerticesAdded;
}

HWND CApproachTrack::Hwnd()
{
    return m_hWnd;
}

enum TolDataClients CApproachTrack::TypeOfTrack()
{
    return m_eTDC;
}

int CApproachTrack::ApproachViewType()
{
    return m_nApproachViewType;
}

//////////////////////////////////////////////////////////////////////
// Mutators
//////////////////////////////////////////////////////////////////////

void CApproachTrack::TouchdownLon(const double dTouchdownLon)
{
    m_dTouchdownLon   =  dTouchdownLon;
}

void CApproachTrack::TouchdownLat(const double dTouchdownLat)
{
    m_dTouchdownLat   =  dTouchdownLat;
}

void CApproachTrack::LocalizerLon(const double dLocalizerLon)
{
    m_dLocalizerLon   =  dLocalizerLon;
}

void CApproachTrack::LocalizerLat(const double dLocalizerLat)
{
    m_dLocalizerLat   =  dLocalizerLat;
}

void CApproachTrack::LocalizerHeading(const float fLocalizerHeading)
{
    m_fLocalizerHeading             = fLocalizerHeading;

    sin_of_loc_hdg    =  sin(fLocalizerHeading *  DEG_TO_RAD);
    cos_of_loc_hdg    =  cos(fLocalizerHeading *  DEG_TO_RAD);
}

void CApproachTrack::LocalizerToGlideslopeDistance(const float fLocalizerToGlideslopeDistance)
{
    m_fLocalizerToGlideslopeDistance = fLocalizerToGlideslopeDistance;
}

void CApproachTrack::ApproachType(const int nApproachType)
{
    m_nApproachType = nApproachType;
}

void CApproachTrack::Scale(const float fScale)
{
    if (fScale == 0.0)
        return;

    // m_scale and scale are defined in CApproachPlot to be
    // 7200/user defined scale.  Thus, if the map scale
    // goes from 30 to 60 then all of the points need to
    // be multiplied by 0.5.  Since increasing the real
    // scale makes m_scale and scale smaller (7200/30 >
    // 7200/60), the order here is new scale/old scale.
    float fScaleRatio = fScale/m_fScale;

    // Scale all of the points already in the array to
    // be drawn for the track.
    unsigned long ulIndex   = 0;
    unsigned long ulSize    = m_ulVerticesAdded;
    while (ulSize)
    {
       m_vertices[ulIndex]      *= fScaleRatio; 
       m_vertices[ulIndex + 1]  *= fScaleRatio; 
       m_vertices[ulIndex + 2]  *= fScaleRatio; 

       ulIndex += 3;
       --ulSize;
    }

    m_fScale           = fScale;
}

void CApproachTrack::ArraySize(const unsigned long int ulArraySize)
{
    if (ulArraySize != m_ulArraySize && m_vertices != NULL)
    {
        delete [] m_vertices;
        m_vertices = NULL;
    }

    if (ulArraySize != m_ulArraySize && m_times != NULL)
    {
        delete [] m_times;
        m_times = NULL;
    }

    if (m_vertices == NULL)
    {
        // Make the array one vertices larger than the size requested to append
        // the first vertice to the end for use with GL_LINE_STRIP.
        m_vertices = new GLfloat[ulArraySize * 3];
    }

    if (m_times == NULL)
        m_times = new unsigned long int[ulArraySize];

    memset(m_times, 0, ulArraySize);

    // Set everyone to 0.0.
    unsigned long ul = 0;
    unsigned long ulSize = ulArraySize;
    while (ulSize)
    {
        m_vertices[ul]      = 0.0f;
        m_vertices[ul + 1]  = 0.0f;
        m_vertices[ul + 2]  = 0.0f;

        ul += 3;
        --ulSize;
    }

    m_ulArraySize = ulArraySize;
}

void CApproachTrack::Elevation(const float fElevation)
{
    m_fElevation  =  fElevation;
}

void CApproachTrack::VRef(const float fVRef)
{
    m_fVRef =  fVRef;
}

void CApproachTrack::MaxVrefDeviationShown(const float fMaxVrefDeviationShown)
{
   m_fMaxVrefDeviationShown   =  fMaxVrefDeviationShown;
}

void CApproachTrack::AspectRatioForVrefPlot(const float fAspectRatioForVrefPlot)
{
   m_fAspectRatioForVrefPlot  =  fAspectRatioForVrefPlot;
}

void CApproachTrack::Hwnd(HWND hWnd)
{
    m_hWnd = hWnd;
}

void CApproachTrack::TypeOfTrack(enum TolDataClients eTDC)
{
    m_eTDC = eTDC;
}

void CApproachTrack::ApproachViewType(const int nApproachViewType)
{
    m_nApproachViewType = nApproachViewType;
}


