// ApproachPlot.cpp: implementation of the CApproachPlot class.
//
//////////////////////////////////////////////////////////////////////

#include "..\core\stdafx.h"
#include "ApproachTrack.h"
#include "ApproachPlot.h"
#include "..\core\DataConversion.h"

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

CApproachPlot::CApproachPlot()
{
   m_exPtUpperLeft      =  CExtentsPoint(CPoint(0,0));
   m_exPtLowerRight     =  CExtentsPoint(CPoint(960,480));

   m_stlStrWidgetName   =  _FSI_STL::string("Approach_Plot");

   m_listGraphicalElementVars.clear();
   m_listGraphicalElementVars.push_back("approach_scale");
   m_listGraphicalElementVars.push_back("active_airport");
   m_listGraphicalElementVars.push_back("active_runway");
   m_listGraphicalElementVars.push_back("ac_latitude");
   m_listGraphicalElementVars.push_back("ac_longitude");
   m_listGraphicalElementVars.push_back("ac_altitude");
   m_listGraphicalElementVars.push_back("ac_heading");
   m_listGraphicalElementVars.push_back("pitch");
   m_listGraphicalElementVars.push_back("ias");
   m_listGraphicalElementVars.push_back("vRef");
   m_listGraphicalElementVars.push_back("Erase Track");

   m_listGraphicalElementVars.push_back("Left Landing Gear Green");
   m_listGraphicalElementVars.push_back("Nose Landing Gear Green");
   m_listGraphicalElementVars.push_back("Right Landing Gear Green");

   m_listGraphicalElementVars.push_back("ac_latitude_test");
   m_listGraphicalElementVars.push_back("ac_longitude_test");
   m_listGraphicalElementVars.push_back("ac_altitude_test");
   m_listGraphicalElementVars.push_back("ac_heading_test");
   m_listGraphicalElementVars.push_back("pitch_test");
   m_listGraphicalElementVars.push_back("ias_test");
   m_listGraphicalElementVars.push_back("vRef_test");

   // Expose these objects for color customization
   m_listColorSelectable.push_back("SCALE_INDICATOR");
   m_listColorSelectable.push_back("ACTIVE_ILS");
   m_listColorSelectable.push_back("FAKE_ILS");


   m_vRef                  =  110.0f;
   m_vRef_cv               =  m_vRef;
   m_max_ias_delta         =  20.0f;
   m_max_deviation_shown   =  m_max_ias_delta   +  10.0f;   // Moves limits so they aren't at the edge of the graph.
   m_max_airspeed_shown    =  m_vRef  +  m_max_ias_delta;
   m_min_airspeed_shown    =  m_vRef  -  m_max_ias_delta;

   m_data_available        =  false;

   m_bClear                =  true;

   m_scale                 =  20.0f;
   m_scale_cv              =  m_scale;

   m_active_airport_id.Format("KRND");
   m_active_airport_id_cv  =  m_active_airport_id;
   m_active_runway_id.Format("14R");
   m_active_runway_id_cv   =  m_active_runway_id;

   m_mapHorzRecompute[TOP_VIEW]  = true;
   m_mapHorzRecompute[SIDE_VIEW] = true;
   m_mapHorzRecompute[VREF_VIEW] = true;

   m_mapVertRecompute[TOP_VIEW]  = true;
   m_mapVertRecompute[SIDE_VIEW] = true;
   m_mapVertRecompute[VREF_VIEW] = true;

   m_bool_fake_ils				   =	false;
}


CApproachPlot::~CApproachPlot()
{
}


CWidget  *CApproachPlot::CreateObject()
{
   return   new   CApproachPlot();
}


void  CApproachPlot::Setup(void)
{
   COpenGLMap::Initialize(GetSafeHwnd());

   m_inner_marker.Format("%c", INNER_MARKER);
   m_middle_marker.Format("%c", MIDDLE_MARKER);
   m_outer_marker.Format("%c", OUTER_MARKER);

   m_inner_marker_side_view.Format("%c", INNER_MARKER_SIDE_VIEW);
   m_middle_marker_side_view.Format("%c", MIDDLE_MARKER_SIDE_VIEW);
   m_outer_marker_side_view.Format("%c", OUTER_MARKER_SIDE_VIEW);

   m_airplane.Format("%c", THE_PLANE);
   m_airplane_side_view.Format("%c", ELEVATOR_TRIM);

   m_airport_name_title.Format("AIRPORT NAME\n");
   m_active_airport_id_title.Format("AIRPORT ID\n");
   m_active_runway_id_title.Format("RUNWAY ID\n");
   m_active_localizer_id_title.Format("LOC ID\n");
   m_active_localizer_freq_title.Format("LOC FREQ\n");
   m_loc_hdg_title.Format("LOC HDG\n");
   m_runway_hdg_title.Format("RUNWAY HDG\n");


   m_nm_to_td_title.Format("NM TO TD\n");
   m_ft_above_touchdown_title.Format("FT ABOVE TOUCHDOWN\n");
   m_vref_deviation_title.Format("V-REF DEVIATION\n");
   m_localizer_deviation_title.Format("LOC DEVIATION\n");
   m_glideslope_deviation_title.Format("G/S DEVIATION\n");

   m_maximum_NM_shown   =  5.0f;


   compass.Initialize();
   compass.SetLabelRadius(0.05f);
   compass.SetOuterRadius(0.10f);
   compass.SetLabelProperties(0.35f, 1.0f, true);
   compass.SetTickColor(128, 128, 128);
   compass.SetTickLengths(0.02f, 0.01f, 0.01f *  m_aspect_ratio);

   TICK_INFO   tick_info;

   tick_info.location   =  0.0f;
   tick_info.type       =  MAJOR;
   strcpy(tick_info.ident, "N");
   compass.TickList.AddTail(tick_info);

   tick_info.location   =  90.0f;
   tick_info.type       =  MAJOR;
   strcpy(tick_info.ident, "E");
   compass.TickList.AddTail(tick_info);

   tick_info.location   =  180.0f;
   tick_info.type       =  MAJOR;
   strcpy(tick_info.ident, "S");
   compass.TickList.AddTail(tick_info);

   tick_info.location   =  270.0f;
   tick_info.type       =  MAJOR;
   strcpy(tick_info.ident, "W");
   compass.TickList.AddTail(tick_info);

   compass.GenerateLists();

   m_track.ApproachViewType(TOP_VIEW);

   m_altitude_track.ApproachViewType(SIDE_VIEW);

   m_airspeed_track.ApproachViewType(VREF_VIEW);

   // May not be the right place if screen dependence is
   // required.
   if (m_track.Hwnd() == NULL)
      m_track.Hwnd(reinterpret_cast<HWND>(1));

   if (m_altitude_track.Hwnd() == NULL)
      m_altitude_track.Hwnd(reinterpret_cast<HWND>(1));

   if (m_airspeed_track.Hwnd() == NULL)
      m_airspeed_track.Hwnd(reinterpret_cast<HWND>(1));
}


void  CApproachPlot::Render(void)
{
   BeginDraw(1,1,1);

   // Split the window into 5 pieces, 3 small and 2 big, all of them are the same width.
   // 1. Looked Up Data                5%    of the height of the original viewing area.
   // 2. Top View of the Localizer     42.5% of the height of the original viewing area.
   // 3. Calculated Data               5%    of the height of the original viewing area.
   // 4. Side View of the Glideslope   42.5% of the height of the original viewing area.
   // 5. Vref Plot                     5%    of the height of the original viewing area.

   // Draw lines to separate the 5 areas of the view.
   // Changed 255,255,255 to 254,254,254 to support printing
   glRGB(254, 254, 254);
   glBegin(GL_LINES);
      glVertex2f(-1.0f, 0.90f);
      glVertex2f( 1.0f, 0.90f);

      glVertex2f(-1.0f, 0.05f);
      glVertex2f( 1.0f, 0.05f);

      glVertex2f(-1.0f, -0.05f);
      glVertex2f( 1.0f, -0.05f);

      glVertex2f(-1.0f, -0.90f);
      glVertex2f( 1.0f, -0.90f);
   glEnd();


   // Setup the Looked Up Data View.
   glViewport(rectViewCoord.left, rectView.Height() - rectViewCoord.bottom + (int)(0.95f * m_height), (int)m_width,  (int)(0.05f * m_height));
   glScissor( rectViewCoord.left, rectView.Height() - rectViewCoord.bottom + (int)(0.95f * m_height), (int)m_width,  (int)(0.05f * m_height));
   // Since we just cut the view vertically, modify the aspect ratio accordingly, but DON'T FORGET TO PUT IT BACK!
   m_aspect_ratio *= 0.05f;

   DrawLookedUpData();

   // Reset the aspect ratio.
   m_aspect_ratio *= 1.0f  /  0.05f;
   // Reset the viewport to the entire window.
   glViewport(rectViewCoord.left, rectView.Height() - rectViewCoord.bottom, (int)m_width, (int)m_height);
   glScissor( rectViewCoord.left, rectView.Height() - rectViewCoord.bottom, (int)m_width, (int)m_height);



   // Setup the Top View of the Localizer.
   glViewport(rectViewCoord.left, rectView.Height() - rectViewCoord.bottom + (int)(0.525f * m_height), (int)m_width,  (int)(0.425f * m_height));
   glScissor( rectViewCoord.left, rectView.Height() - rectViewCoord.bottom + (int)(0.525f * m_height), (int)m_width,  (int)(0.425f * m_height));
   // Since we just cut the view vertically, modify the aspect ratio accordingly, but DON'T FORGET TO PUT IT BACK!
   m_aspect_ratio *= 0.425f;

   DrawTopView();

   // Reset the aspect ratio.
   m_aspect_ratio *= 1.0f  /  0.425f;
   // Reset the viewport to the entire window.
   glViewport(rectViewCoord.left, rectView.Height() - rectViewCoord.bottom, (int)m_width, (int)m_height);
   glScissor( rectViewCoord.left, rectView.Height() - rectViewCoord.bottom, (int)m_width, (int)m_height);



   // Setup the Calcutated Data View.
   glViewport(rectViewCoord.left, rectView.Height() - rectViewCoord.bottom + (int)(0.475f * m_height), (int)m_width,  (int)(0.05f * m_height));
   glScissor( rectViewCoord.left, rectView.Height() - rectViewCoord.bottom + (int)(0.475f * m_height), (int)m_width,  (int)(0.05f * m_height));
   // Since we just cut the view vertically, modify the aspect ratio accordingly, but DON'T FORGET TO PUT IT BACK!
   m_aspect_ratio *= 0.05f;

   DrawCalculatedData();

   // Reset the aspect ratio.
   m_aspect_ratio *= 1.0f  /  0.05f;
   // Reset the viewport to the entire window.
   glViewport(rectViewCoord.left, rectView.Height() - rectViewCoord.bottom, (int)m_width, (int)m_height);
   glScissor( rectViewCoord.left, rectView.Height() - rectViewCoord.bottom, (int)m_width, (int)m_height);



   // Setup the Side View of the Glideslope
   glViewport(rectViewCoord.left, rectView.Height() - rectViewCoord.bottom + (int)(0.05f * m_height), (int)m_width,  (int)(0.425f * m_height));
   glScissor( rectViewCoord.left, rectView.Height() - rectViewCoord.bottom + (int)(0.05f * m_height), (int)m_width,  (int)(0.425f * m_height));
   // Since we just cut the view in half vertically, modify the aspect ratio accordingly, but DON'T FORGET TO PUT IT BACK!
   m_aspect_ratio *= 0.425f;

   DrawSideView();

   // Reset the aspect ratio.
   m_aspect_ratio *= 1.0f  /  0.425f;
   // Reset the viewport to the entire window.
   glViewport(rectViewCoord.left, rectView.Height() - rectViewCoord.bottom, (int)m_width, (int)m_height);
   glScissor( rectViewCoord.left, rectView.Height() - rectViewCoord.bottom, (int)m_width, (int)m_height);



   // Setup the Vref Plot View
   glViewport(rectViewCoord.left, rectView.Height() - rectViewCoord.bottom + (int)(0.0f * m_height), (int)m_width,  (int)(0.05f * m_height));
   glScissor( rectViewCoord.left, rectView.Height() - rectViewCoord.bottom + (int)(0.0f * m_height), (int)m_width,  (int)(0.05f * m_height));
   // Since we just cut the view vertically, modify the aspect ratio accordingly, but DON'T FORGET TO PUT IT BACK!
   m_aspect_ratio *= 0.05f;

   DrawVrefPlot();

   // Reset the aspect ratio.
   m_aspect_ratio *= 1.0f  /  0.05f;
   // Reset the viewport to the entire window.
   glViewport(rectViewCoord.left, rectView.Height() - rectViewCoord.bottom, (int)m_width, (int)m_height);
   glScissor( rectViewCoord.left, rectView.Height() - rectViewCoord.bottom, (int)m_width, (int)m_height);
}


void CApproachPlot::ChangeValue(const CString& rstrElementVar, CChangeValue* pCV)
{
   if (pCV == NULL)
   {
       return;
   }
   CVariant* pVariant = pCV->Variant();
   static   int   track_size     =  0;

   if (rstrElementVar      == CString("approach_scale"))
   {
      m_scale_cv                 =  *pVariant;
      m_maximum_NM_shown_cv      =  m_scale_cv;    // Sometimes it's handy to know the actual number of miles being shown.

      if (m_scale_cv == 0.0f)
      {
         m_scale_cv              =  7200.0f;
         m_maximum_NM_shown_cv   =  1.0f;
      }

      m_scale_cv  =  7200.0f  /  m_scale_cv;
   }
   else if (rstrElementVar   == CString("active_airport"))
   {
      CString  airport_id((char *)((_FSI_STL::string)(*pVariant)).c_str());
      airport_id.MakeUpper();

      m_active_airport_id_cv  =  airport_id;

      int   length   =  m_active_airport_id_cv.GetLength();
      for (; length < 4; length++)
      {
         m_active_airport_id_cv  += " ";
      }
   }
   else if (rstrElementVar   == CString("active_runway"))
   {
      CString  runway_id((char *)((_FSI_STL::string)(*pVariant)).c_str());
      runway_id.MakeUpper();

      m_active_runway_id_cv   =  runway_id;

      int   length   =  m_active_runway_id_cv.GetLength();
      for (; length < 4; length++)
      {
         m_active_runway_id_cv   += " ";
      }
   }
   else if (rstrElementVar   == CString("Left Landing Gear Green"))
   {
      m_bool_left_gear_down   =  *pVariant;
   }
   else if (rstrElementVar   == CString("Right Landing Gear Green"))
   {
      m_bool_right_gear_down  =  *pVariant;
   }
   else if (rstrElementVar   == CString("Nose Landing Gear Green"))
   {
      m_bool_nose_gear_down   =  *pVariant;
   }
   else if (rstrElementVar    == CString("ac_latitude"))
   {
      m_ac_latitude_cv     =  *pVariant;
   }
   else if (rstrElementVar    == CString("ac_longitude"))
   {
      m_ac_longitude_cv    =  *pVariant;
   }
   else if (rstrElementVar    == CString("ac_altitude"))
   {
      m_ac_altitude_cv     =  *pVariant;
   }
   else if (rstrElementVar    == CString("ac_heading"))
   {
      m_ac_heading_cv      =  *pVariant;
   }
   else if (rstrElementVar    == CString("pitch"))
   {
      m_pitch_cv           =  *pVariant;
   }
   else if (rstrElementVar    == CString("ias"))
   {
      m_ias_cv             =  *pVariant;
   }
   else if (rstrElementVar    == CString("vRef"))
   {
      m_vRef_cv            =  *pVariant;
   }
   else if (rstrElementVar   == CString("Erase Track"))
   {
       m_bClear                =  !m_bClear;
   }

}


bool CApproachPlot::UpdateRenderVariables()
{
   bool  bRetVal  =  CWidget::UpdateRenderVariables();

   if (m_bEditing)
      return bRetVal;

   if (m_bClear   == true)
   {
        if (m_track.VerticesAdded() > 0)
            m_track.Clear();

        if (m_altitude_track.VerticesAdded() > 0)
            m_altitude_track.Clear();

        if (m_airspeed_track.VerticesAdded() > 0)
            m_airspeed_track.Clear();

        m_bClear =  false;
   }

   if (m_scale != m_scale_cv)
   {
      m_track.Scale(m_scale_cv);
      m_altitude_track.Scale(m_scale_cv);
      m_airspeed_track.Scale(m_scale_cv);

      m_mapHorzRecompute[TOP_VIEW]  = true;
      m_mapHorzRecompute[SIDE_VIEW] = true;
      m_mapHorzRecompute[VREF_VIEW] = true;
      m_mapVertRecompute[TOP_VIEW]  = true;
      m_mapVertRecompute[SIDE_VIEW] = true;
      m_mapVertRecompute[VREF_VIEW] = true;

      m_mapHorzIndicators.clear();
      m_mapVertIndicators.clear();
   }

   if (m_active_airport_id_cv != m_active_airport_id  || m_active_runway_id_cv  != m_active_runway_id)
   {
      m_active_airport_id  =  m_active_airport_id_cv;
      m_active_runway_id   =  m_active_runway_id_cv;

      m_mapHorzRecompute[TOP_VIEW]  = true;
      m_mapHorzRecompute[SIDE_VIEW] = true;
      m_mapHorzRecompute[VREF_VIEW] = true;
      m_mapVertRecompute[TOP_VIEW]  = true;
      m_mapVertRecompute[SIDE_VIEW] = true;
      m_mapVertRecompute[VREF_VIEW] = true;

      m_mapHorzIndicators.clear();
      m_mapVertIndicators.clear();

      CString  temp(m_active_airport_id);
      temp.MakeReverse();
      int   int_ident   =  (int)*(int *)LPCTSTR(temp);
      rdbStation  *pStation   =  rdbRadioDB.Airport(int_ident);

      if (pStation)
      {
         rdbAPT   *pAirport      =  (rdbAPT *)pStation;
         m_airport_name          =  pAirport->chName;
         m_airport_name.SetAt(29, '\0');
         float airport_mag_var   =  (float)(pAirport->sMvarX10 /  10.0f);

         pStation =  rdbRadioDB.Runway(int_ident);
         bool  runway_found   =  false;
         while (pStation   && !runway_found)
         {
            rdbRWY      *pRunway =  (rdbRWY  *)pStation;

            CString     runway_id;
            runway_id   =  (char)((pRunway->nIdent &  0xff000000) >> 24);
            runway_id   += (char)((pRunway->nIdent &  0x00ff0000) >> 16);
            runway_id   += (char)((pRunway->nIdent &  0x0000ff00) >> 8);
            runway_id   += (char)((pRunway->nIdent &  0x000000ff));
            runway_id   += "\0";

            if (runway_id  == m_active_runway_id)
            {
               runway_found      =  true;
               m_data_available  =  true;

               m_runway_latitude             =  BamsToDegrees(pRunway->nLat);
               m_runway_longitude            =  BamsToDegrees(pRunway->nLon);
               m_runway_heading              =  (float)(pRunway->sThdgx10  /  10.0f);
               float magnetic_runway_heading =  m_runway_heading  - airport_mag_var;
               m_runway_hdg.Format("%3.0f", (magnetic_runway_heading <  0.0f) ?  360.0f   +  magnetic_runway_heading :  magnetic_runway_heading);
               m_runway_length               =  pRunway->sLen;
               m_runway_displaced_threshold  =  pRunway->sDispThr;
               m_runway_width                =  pRunway->sWidth;

               // We want the runway length to be the length available for landing.
               // This does not include the displaced threshold so subtract it out.
               m_runway_length               -= m_runway_displaced_threshold;

               m_elevation             =  (float)(pRunway->nType2    &  0x0000ffff);

               m_active_localizer_id   =  (char)((pRunway->nLocIdent &  0xff000000) >> 24);
               m_active_localizer_id   += (char)((pRunway->nLocIdent &  0x00ff0000) >> 16);
               m_active_localizer_id   += (char)((pRunway->nLocIdent &  0x0000ff00) >> 8);
               m_active_localizer_id   += (char)((pRunway->nLocIdent &  0x000000ff));
               m_active_localizer_id   += "\0";

               if (m_active_localizer_id  == "    ")
               {
                  m_approach_type         =  NON_PRECISION;
                  m_active_localizer_id   =  "N/A";

                  m_localizer_latitude    =  m_runway_latitude;
                  m_localizer_longitude   =  m_runway_longitude;

                  m_localizer_heading                 =  m_runway_heading;
                  m_loc_hdg.Format("N/A");
                  m_active_localizer_freq.Format("N/A");
                  m_active_localizer_id.Format("N/A");

#ifdef   HAF
                  m_localizer_to_glideslope_distance  =  0.0f;
                  m_localizer_width_angle             =  0.0f;
                  m_glideslope_angle_from_ground      =  0.0f;
                  m_glideslope_width_angle            =  0.0f;
                  m_bool_fake_ils                     =  false;
#else
                  m_approach_type         =  FULL_PRECISION;
                  m_localizer_to_glideslope_distance  =  m_runway_length   + 1000;
                  m_localizer_width_angle             =  4.0f;
                  m_glideslope_angle_from_ground      =  3.0f;
                  m_glideslope_width_angle            =  1.40f;
                  m_bool_fake_ils                     =  true;

                  m_localizer_latitude   =  m_runway_latitude    +  ((m_localizer_to_glideslope_distance  /  NMI_FT)  *  cos(m_runway_heading *  DEG_TO_RAD))   /  NM_PER_DEGREE;
                  m_localizer_longitude  =  m_runway_longitude   +  ((m_localizer_to_glideslope_distance  /  NMI_FT)  *  sin(m_runway_heading *  DEG_TO_RAD))   /  NM_PER_DEGREE  /  cos(((m_runway_latitude +  m_localizer_latitude)  /  2.0)  *  DEG_TO_RAD);
#endif

                  m_glideslope_distance               =  1000.0f;

                  // INNER marker
                  m_inner_marker_latitude    =  0.0;
                  m_inner_marker_longitude   =  0.0;
                  m_inner_marker_heading     =  0.0;

                  // MIDDLE marker
                  m_middle_marker_latitude   =  0.0;
                  m_middle_marker_longitude  =  0.0;
                  m_middle_marker_heading    =  0.0;

                  // OUTER marker
                  m_outer_marker_latitude    =  0.0;
                  m_outer_marker_longitude   =  0.0;
                  m_outer_marker_heading     =  0.0;
               }
               else
               {
                  m_bool_fake_ils                     =  false;
                  rdbStation  *pStationILS   =  rdbRadioDB.ILS(int_ident);
                  bool  ils_found   =  false;
                  while (pStationILS   && !ils_found)
                  {
                     rdbILS   *pILS =  (rdbILS  *)pStationILS;

                     CString        localizer_id;
                     localizer_id   =  (char)((pILS->nIdent &  0xff000000) >> 24);
                     localizer_id   += (char)((pILS->nIdent &  0x00ff0000) >> 16);
                     localizer_id   += (char)((pILS->nIdent &  0x0000ff00) >> 8);
                     localizer_id   += (char)((pILS->nIdent &  0x000000ff));
                     localizer_id   += "\0";

                     if (localizer_id  == m_active_localizer_id)
                     {
                        ils_found   =  true;

                        if (pILS->nType2  &  0x00200000)    // Is there a glideslope available?
                        {
                           m_approach_type   =  FULL_PRECISION;
                        }
                        else
                        {
                           m_approach_type   =  LOCALIZER_ONLY;
                        }

                        m_localizer_latitude    =  BamsToDegrees(pILS->nLat);
                        m_localizer_longitude   =  BamsToDegrees(pILS->nLon);

                        m_localizer_heading                 =  (float)(pILS->sThdgX10  /  10.0f);
                        float ils_mag_var                   =  (float)(pILS->sMvarX10  /  10.0f);
                        float magnetic_localizer_heading    =  m_localizer_heading  -  ils_mag_var;
                        m_loc_hdg.Format("%3.0f", (magnetic_localizer_heading <  0.0f) ?  360.0f   +  magnetic_localizer_heading :  magnetic_localizer_heading);
                        float magnetic_runway_heading       =  m_runway_heading     -  ils_mag_var;
                        m_runway_hdg.Format("%3.0f", (magnetic_runway_heading <  0.0f) ?  360.0f   +  magnetic_runway_heading    :  magnetic_runway_heading);

                        m_active_localizer_freq.Format("%3.2f", (float)(pILS->sFreqX100 /  100.0f));
                        m_active_localizer_id.Format("%s", localizer_id);

                        // If the runway and localizer are more than 3 apart then it is assumed that this is a LDA facility.
                        if (fabs(m_runway_heading  -  m_localizer_heading) >  3.0 &&
                            fabs(m_runway_heading  -  m_localizer_heading) <  357.0 )
                        {
                           m_approach_type   |= LOCALIZER_DIRECTIONAL_AID;
                           m_elevation       =  (float)(pILS->sElev);         // Runway and Loc elevation may be different.
                        }

                        m_localizer_to_glideslope_distance  =  (float)pILS->sLgd;
                        m_localizer_width_angle             =  (float)pILS->sLbwX100   /  100.0f;
                        m_glideslope_angle_from_ground      =  (float)pILS->sGsaX100   /  100.0f;
                        m_glideslope_width_angle            =  (float)pILS->sGsw       /  100.0f;

//                        if (pILS->sGsDist)
                           m_glideslope_distance   =  (float)pILS->sGsDist;
//                        else
//                           m_glideslope_distance   =  1000.0f;
                        // Account for displaced threshold...
                        m_glideslope_distance      -= m_runway_displaced_threshold;

                        // INNER marker
                        m_inner_marker_latitude    =  BamsToDegrees(pILS->nImLat);
                        m_inner_marker_longitude   =  BamsToDegrees(pILS->nImLon);
                        m_inner_marker_heading     =  (double)pILS->sThdgX10   /  10.0;

                        // MIDDLE marker
                        m_middle_marker_latitude   =  BamsToDegrees(pILS->nMmLat);
                        m_middle_marker_longitude  =  BamsToDegrees(pILS->nMmLon);
                        m_middle_marker_heading    =  (double)pILS->sThdgX10   /  10.0;

                        // OUTER marker
                        m_outer_marker_latitude    =  BamsToDegrees(pILS->nOmLat);
                        m_outer_marker_longitude   =  BamsToDegrees(pILS->nOmLon);
                        m_outer_marker_heading     =  (double)pILS->sThdgX10   /  10.0;

                     }
                     pStationILS = rdbRadioDB.ILSNext();
                  }
               }
            }
            pStation = rdbRadioDB.RunwayNext();
         }
         if (!runway_found)
         {
            CString  message;
            message.Format("Sorry, no runway with an ident of \"%s\" could be found at %s.", m_active_runway_id, m_active_airport_id);
            COpenGLMap::m_stlStrError     =  (LPCTSTR)message;
            COpenGLMap::m_stlStrCaption   =  "Invalid Ident";
            AfxBeginThread(COpenGLMap::MsgThread, (LPVOID)0);
         }
      }
      else
      {
         CString  message;
         message.Format("Sorry, no airport with an ident of \"%s\" could be found.", m_active_airport_id);
         COpenGLMap::m_stlStrError     =  (LPCTSTR)message;
         COpenGLMap::m_stlStrCaption   =  "Invalid Ident";
         AfxBeginThread(COpenGLMap::MsgThread, (LPVOID)0);
      }

      // Calculate the latitude and longitude of the touchdown point.  This is based upon the runway latitude, longitude, heading and the distance
      // from the end of the runway to the glideslope shack (m_glideslope_distance - which accounts for the displaced threshold).
      m_touchdown_latitude    =  m_runway_latitude    +  ((m_glideslope_distance /  NMI_FT)  *  cos(m_runway_heading *  DEG_TO_RAD))   /  NM_PER_DEGREE;
      m_touchdown_longitude   =  m_runway_longitude   +  ((m_glideslope_distance /  NMI_FT)  *  sin(m_runway_heading *  DEG_TO_RAD))   /  NM_PER_DEGREE  /  cos(((m_runway_latitude +  m_touchdown_latitude)   /  2.0)  *  DEG_TO_RAD);

      // Calculate the latitude and longitude of the glideslope shack.  This is based upon the localizer latitude, longitude, heading and the 
      // distance from the localizer to the glideslope shack (m_localizer_to_glideslope_distance).
      m_glideslope_latitude   =  m_localizer_latitude    +  ((-m_localizer_to_glideslope_distance  /  NMI_FT)  *  cos(m_localizer_heading *  DEG_TO_RAD))   /  NM_PER_DEGREE;
      m_glideslope_longitude  =  m_localizer_longitude   +  ((-m_localizer_to_glideslope_distance  /  NMI_FT)  *  sin(m_localizer_heading *  DEG_TO_RAD))   /  NM_PER_DEGREE  /  cos(((m_localizer_latitude +  m_glideslope_latitude)  /  2.0)  *  DEG_TO_RAD);

      m_track.TouchdownLat(m_touchdown_latitude);
      m_track.TouchdownLon(m_touchdown_longitude);
      m_track.LocalizerLat(m_localizer_latitude);
      m_track.LocalizerLon(m_localizer_longitude);
      m_track.LocalizerHeading(m_localizer_heading);
      m_track.LocalizerToGlideslopeDistance(m_localizer_to_glideslope_distance);
      m_track.ApproachType(m_approach_type);

      m_track.RecomputeOnAirportRunwayChange();
   
      m_altitude_track.TouchdownLat(m_touchdown_latitude);
      m_altitude_track.TouchdownLon(m_touchdown_longitude);
      m_altitude_track.LocalizerLat(m_localizer_latitude);
      m_altitude_track.LocalizerLon(m_localizer_longitude);
      m_altitude_track.LocalizerHeading(m_localizer_heading);
      m_altitude_track.LocalizerToGlideslopeDistance(m_localizer_to_glideslope_distance);
      m_altitude_track.ApproachType(m_approach_type);
      m_altitude_track.Elevation(m_elevation);

      m_altitude_track.RecomputeOnAirportRunwayChange();

      m_airspeed_track.TouchdownLat(m_touchdown_latitude);
      m_airspeed_track.TouchdownLon(m_touchdown_longitude);
      m_airspeed_track.LocalizerLat(m_localizer_latitude);
      m_airspeed_track.LocalizerLon(m_localizer_longitude);
      m_airspeed_track.LocalizerHeading(m_localizer_heading);
      m_airspeed_track.LocalizerToGlideslopeDistance(m_localizer_to_glideslope_distance);
      m_airspeed_track.ApproachType(m_approach_type);
      m_airspeed_track.VRef(m_vRef);
      m_airspeed_track.MaxVrefDeviationShown(m_max_deviation_shown);

      m_airspeed_track.AspectRatioForVrefPlot(m_aspect_ratio * 0.05);

      m_airspeed_track.RecomputeOnAirportRunwayChange();
   }

   m_track.UpdateData();

   m_altitude_track.UpdateData();

   m_airspeed_track.UpdateData();


   if (m_track.VerticesAdded()   >  0)
   {
      m_nm_to_gs                    =  m_track.NMToGS();
      m_nm_to_td_value              =  m_track.NMToTouchdown();
      m_localizer_deviation_value   =  m_track.LocalizerDeviation();
   }
   else
   {
      double   xgas, ygas, agas, bgas;
      double   sin_of_loc_hdg    =  sin(m_localizer_heading *  DEG_TO_RAD);
      double   cos_of_loc_hdg    =  cos(m_localizer_heading *  DEG_TO_RAD);


      // The xgas term is the north-south distance from the aircraft to the selected ground station, in nautical miles.  
      xgas              =  (m_localizer_latitude   -  m_ac_latitude_cv)    *  NM_PER_DEGREE;
      // The ygas term is the east-west distance from the aircraft to the selected ground station, in nautical miles.
      ygas              =  (m_localizer_longitude  -  m_ac_longitude_cv)   *  NM_PER_DEGREE  *  cos(((m_localizer_latitude +  m_ac_latitude_cv) /  2.0)  *  DEG_TO_RAD);

      // The agas term is the distance from the aircraft to the facility, in nautical miles, measured along the axis of the facility heading. 
      // Positive values indicate the aircraft is on the front course, with negative values indicate the aircraft is on the back course.
      agas              =  ygas  *  sin_of_loc_hdg +  xgas  *  cos_of_loc_hdg;
      // The bgas term is the distance from the aircraft to the facility centerline, in nautical miles, measured perpendicular to the 
      // facility centerline.  Positive values indicate the aircraft is to the right of the facility centerline, and negative values 
      // indicate the aircraft is to the left of the facility centerline.
      bgas              =  xgas  *  sin_of_loc_hdg -  ygas  *  cos_of_loc_hdg;

      if (m_approach_type  &  LOCALIZER_DIRECTIONAL_AID)
      {
         // Finally, m_nm_to_gs is the distance from the aircraft to the glideslope shack measured along the localizer's heading.
         m_nm_to_gs     =  agas  -  m_localizer_to_glideslope_distance  /  NMI_FT;
      }
      else
      {
         // The x_distance term is the north-south distance from the aircraft to the touchdown point, in nautical miles.  
         double   x_distance        =  (m_touchdown_latitude   -  m_ac_latitude_cv)    *  NM_PER_DEGREE;
         // The y_distance term is the east-west distance from the aircraft to the touchdown point, in nautical miles.  
         double   y_distance        =  (m_touchdown_longitude   -  m_ac_longitude_cv)   *  NM_PER_DEGREE  *  cos(((m_touchdown_latitude +  m_ac_latitude_cv) /  2.0)  *  DEG_TO_RAD);
         // The ac_to_td_bearing term is the angle between the aircraft and the touchdown point.
         float    ac_to_td_bearing  =  atan2(y_distance, x_distance) *  RAD_TO_DEG;
         // Calculate the distance to the touchdown point, in nautical miles.
         m_nm_to_td_value           =  sqrt(x_distance   *  x_distance  +  y_distance  *  y_distance) *  cos((ac_to_td_bearing   -  m_localizer_heading) *  DEG_TO_RAD);
      }

      m_localizer_deviation_value   =  bgas  *  NMI_FT;  // Calculate the localizer deviation.
   }

   m_scale                 =  m_scale_cv;
   m_maximum_NM_shown      =  m_maximum_NM_shown_cv;

   m_ac_latitude           =  m_ac_latitude_cv;
   m_ac_longitude          =  m_ac_longitude_cv;
   m_ac_altitude           =  m_ac_altitude_cv;
   m_ac_heading            =  m_ac_heading_cv; 
   m_pitch                 =  m_pitch_cv;
   m_ias                   =  m_ias_cv;
   m_vRef                  =  m_vRef_cv;

   m_max_ias_delta         =  20.0f;
   m_max_deviation_shown   =  m_max_ias_delta   +  10.0f;   // Moves limits so they aren't at the edge of the graph.
   m_max_airspeed_shown    =  m_vRef  +  m_max_ias_delta;
   m_min_airspeed_shown    =  m_vRef  -  m_max_ias_delta;

   // Format the m_nm_to_td_value.
   m_nm_to_td.Format("%5.2f", m_nm_to_td_value);

   // Calculate Feet Above Touchdown...
   m_ft_above_touchdown_value =  m_ac_altitude  -  m_elevation;
   m_ft_above_touchdown.Format("%5.1f", m_ft_above_touchdown_value);


   // Determine if we're getting closer or further away from the glideslope shack.
   if (m_approach_type  &  LOCALIZER_DIRECTIONAL_AID)
   {
      static   float lpv_nm_to_gs   =  0.0f;
      if (m_nm_to_gs  >  lpv_nm_to_gs)
         m_going_away   =  GOING_AWAY;
      else if (m_nm_to_gs  <  lpv_nm_to_gs)
         m_going_away   =  GOING_TOWARD;
      lpv_nm_to_gs      =  m_nm_to_gs;
   }
   else
   {
      static   float lpv_nm_to_td   =  0.0f;
      if (m_nm_to_td_value >  lpv_nm_to_td)
         m_going_away   =  GOING_AWAY;
      else if (m_nm_to_td_value <  lpv_nm_to_td)
         m_going_away   =  GOING_TOWARD;
      lpv_nm_to_td      =  m_nm_to_td_value;
   }


   float    distance;   // Either the distance to touchdown or the distance to the glideslope shack (LDA).
   if (m_approach_type  &  LOCALIZER_DIRECTIONAL_AID)
      distance =  m_nm_to_gs;
   else
      distance =  m_nm_to_td_value;
   // Calculate the glideslope deviation and limit the range of the glideslope to 25 NM.
   if (distance   >= 0.0f  && distance <  25.0f)
   {
      // The altitude the aircraft should be at, at a given distance from the glideslope shack.
      float desired_altitude  =  distance *  NMI_FT   *  tan(m_glideslope_angle_from_ground  *  DEG_TO_RAD);

      float glideslope_deviation_value =  m_ft_above_touchdown_value -  desired_altitude;
      if (glideslope_deviation_value   >  0.0f)
         m_glideslope_deviation.Format("%5.1f FT ABOVE", glideslope_deviation_value);
      else if (fabs(glideslope_deviation_value) <= 0.09f)
         m_glideslope_deviation.Format("%5.1f FT",       fabs(glideslope_deviation_value));
      else
         m_glideslope_deviation.Format("%5.1f FT BELOW", -glideslope_deviation_value);
   }
   else
   {
      m_glideslope_deviation.Format("OUT OF RANGE");
   }


   // Show the localizer deviation and limit the range of the localizer to 32 NM.
   if (distance   >= 0.0f  && distance <  32.0f)
   {
      if (m_localizer_deviation_value  <  0.0f)
         m_localizer_deviation.Format("%5.1f FT LEFT",   -m_localizer_deviation_value);
      else if (fabs(m_localizer_deviation_value)   <= 0.09f)
         m_localizer_deviation.Format("%5.1f FT",        fabs(m_localizer_deviation_value));
      else
         m_localizer_deviation.Format("%5.1f FT RIGHT",  m_localizer_deviation_value);
   }
   else
   {
      m_localizer_deviation.Format("OUT OF RANGE");
   }

   // Calculate vRef deviation.
   m_vref_deviation.Format("%3.1f KTS", m_ias   -  m_vRef);

   // Determine if gear is down and draw the appropriate symbol.
   if (m_bool_nose_gear_down  && m_bool_left_gear_down   && m_bool_right_gear_down)
      m_airplane_side_view.Format("%c", SIDE_VIEW_W_GEAR_DOWN);
   else
      m_airplane_side_view.Format("%c", ELEVATOR_TRIM);

   return bRetVal;
}


void  CApproachPlot::DrawLookedUpData()
{
   glPushMatrix();
   {
      if (m_aspect_ratio  <= 1.0f)
         glScalef(m_aspect_ratio, 1.0f, 1.0f);
      else if (m_aspect_ratio != 0.0f)
         glScalef(1.0f, 1.0f  /  m_aspect_ratio, 1.0f);

      glPushMatrix();
         glTranslatef(-0.75f  /  m_aspect_ratio, 0.50f, 0.0f);
         airportName.Draw(m_airport_name_title  +  m_airport_name, true, 0.0f, 9.0f);
      glPopMatrix();
      glPushMatrix();
         glTranslatef(-0.40f  /  m_aspect_ratio, 0.50f, 0.0);
         airportId.Draw(m_active_airport_id_title  +  m_active_airport_id, true, 0.0f, 9.0f);
      glPopMatrix();

      glPushMatrix();
         glTranslatef(-0.15f  /  m_aspect_ratio, 0.50f, 0.0f);
         runwayId.Draw(m_active_runway_id_title +  m_active_runway_id, true, 0.0f, 9.0f);
      glPopMatrix();
      glPushMatrix();
         glTranslatef( 0.10f  /  m_aspect_ratio, 0.50f, 0.0f);
         localizerId.Draw(m_active_localizer_id_title +  m_active_localizer_id, true, 0.0f, 9.0f);
      glPopMatrix();
      glPushMatrix();
         glTranslatef( 0.35f  /  m_aspect_ratio, 0.50f, 0.0f);
         localizerFreq.Draw(m_active_localizer_freq_title   +  m_active_localizer_freq, true, 0.0f, 9.0f);
      glPopMatrix();
      glPushMatrix();
         glTranslatef( 0.60f  /  m_aspect_ratio, 0.50f, 0.0f);
         locHdg.Draw(m_loc_hdg_title +  m_loc_hdg, true, 0.0f, 9.0f);
      glPopMatrix();
      glPushMatrix();
         glTranslatef( 0.80f  /  m_aspect_ratio, 0.50f, 0.0f);
         runwayHdg.Draw(m_runway_hdg_title +  m_runway_hdg, true, 0.0f, 9.0f);
      glPopMatrix();
   }
   glPopMatrix();
}


void  CApproachPlot::DrawTopView()
{
   glPushMatrix();
   {
      // Offset everything so that the localizer shack is located at the right edge of the drawing area.
      glTranslatef(1.0f, 0.0f, 0.0f);
 
      glPushMatrix();
      {
         if (m_aspect_ratio != 0.0f)
            glScalef(1.0f, 1.0f  /  m_aspect_ratio, 1.0f);

         // Draw the vertical deviation grid lines.
         glPushMatrix();
         {
            glTranslatef(-1.90f, 0.0f, 0.0f);   // Near the left edge.
            DrawVerticalScale(TOP_VIEW);
         }
         glPopMatrix();


         // Draw a compass rotated to the localizer heading.
         glPushMatrix();
         {
            glTranslatef(-0.20f, 0.20f, 0.0f);
            glRotated(m_localizer_heading -  90.0,    0.0,  0.0,  1.0);
            compass.Draw();
         }
         glPopMatrix();


         // Move to what it is we really want to see...
         AdjustTheView();


         // Draw the runway.
         glPushMatrix();
         {
            glRotated(m_runway_longitude,  0.0,  1.0,  0.0);
            glRotated(-m_runway_latitude,  1.0,  0.0,  0.0);
            glTranslated(0.0, 0.0,  0.95  *  m_scale);
            glRotated(270.0 - m_runway_heading,  0.0,  0.0,  1.0);

            DrawRunway(TOP_VIEW);
         }
         glPopMatrix();


         // Draw the aiming point markers and the horizontal scale.
         glPushMatrix();
         {
            glRotated(m_touchdown_longitude,  0.0,  1.0,  0.0);
            glRotated(-m_touchdown_latitude,  1.0,  0.0,  0.0);
            glTranslated(0.0, 0.0,  0.95  *  m_scale);
            glRotated(270.0 - m_runway_heading,  0.0,  0.0,  1.0);

            DrawAimingPointMarkers();
            DrawHorizontalScale(TOP_VIEW);

            // If the scale is one mile or less, show the runway number on the runway.
            if (m_maximum_NM_shown  <= 1.0f)
            {
               glPushMatrix();
                  glTranslatef(((m_glideslope_distance -  100.0f)   /  NMI_FT)  *  (m_scale /  3600.0f), 0.0f, 0.0f);
                  glScalef(1.5f, 1.0f, 1.0f);
                  CString  runway_id(m_active_runway_id);
                  runway_id.TrimRight();     // Trim trailing spaces.
                  runway_id.TrimLeft('0');   // Trim leading zeros.
                  if (runway_id.GetLength()  == 1)
                     runwayId.Draw(runway_id, true, 90.0f,   (0.30f  )  *  (m_scale /  3600.0f));
                  else if (runway_id.GetLength()  == 2)
                     runwayId.Draw(runway_id, true, 90.0f,   (0.25f  )  *  (m_scale /  3600.0f));
                  else
                     runwayId.Draw(runway_id, true, 90.0f,   (0.15f  )  *  (m_scale /  3600.0f));
               glPopMatrix();
            }
         }
         glPopMatrix();


         // Draw the inner marker...
         glPushMatrix();
            glRotated(m_inner_marker_longitude,  0.0,  1.0,  0.0);
            glRotated(-m_inner_marker_latitude,  1.0,  0.0,  0.0);
            glTranslated(0.0, 0.0,  0.95  *  m_scale);
            glRotated(270.0 - m_inner_marker_heading,  0.0,  0.0,  1.0);
            inner_marker.Draw(m_inner_marker);
         glPopMatrix();

      
         // Draw the middle marker...
         glPushMatrix();
            glRotated(m_middle_marker_longitude,  0.0,  1.0,  0.0);
            glRotated(-m_middle_marker_latitude,  1.0,  0.0,  0.0);
            glTranslated(0.0, 0.0,  0.95  *  m_scale);
            glRotated(270.0 - m_middle_marker_heading,  0.0,  0.0,  1.0);
            middle_marker.Draw(m_middle_marker);
         glPopMatrix();

      
         // Draw the outer marker...
         glPushMatrix();
            glRotated(m_outer_marker_longitude,  0.0,  1.0,  0.0);
            glRotated(-m_outer_marker_latitude,  1.0,  0.0,  0.0);
            glTranslated(0.0, 0.0,  0.95  *  m_scale);
            glRotated(270.0 - m_outer_marker_heading,  0.0,  0.0,  1.0);
            outer_marker.Draw(m_outer_marker);
         glPopMatrix();


         // Draw the localizer beam...
         DrawLocalizerBeam();


         // Draw the airplane.
         // Method #1 below would be the most obvious way to position the aircraft, but...
         // due to the curvature of the earth, this method results in SEEING a deviation from the centerline of the localizer
         // beam that - calcutations tell us - doesn't really exist.  So Method #2, which takes advantage of the calculations
         // already performed to get at the number of nautical miles from either the touchdown point or from the glideslope shack and 
         // the localizer deviation, is used to plot the aircraft position.

         /* Method #1
            glPushMatrix();
            {
               glRotated(m_ac_longitude,  0.0,  1.0,  0.0);
               glRotated(-m_ac_latitude,  1.0,  0.0,  0.0);
               glTranslated(0.0, 0.0,  0.95  *  m_scale);
               airplane.Draw(m_airplane, true, -m_ac_heading, 1.0f);
            }
            glPopMatrix();
         */

         // Method #2
         glPushMatrix();
         {
            float computed_horizontal_offset =  0.0f;
            bool  out_of_range               =  false;

            if (m_approach_type  &  LOCALIZER_DIRECTIONAL_AID)
            {
               glRotated(m_glideslope_longitude,  0.0,  1.0,  0.0);
               glRotated(-m_glideslope_latitude,  1.0,  0.0,  0.0);
               glTranslated(0.0, 0.0,  0.95  *  m_scale);
               glRotated(270.0 - m_localizer_heading,  0.0,  0.0,  1.0);

               // Determine if the aircraft is off the scale - and 'move' the aircraft if it is.
               out_of_range   =  LimitHorizontalRange(m_nm_to_gs, m_localizer_to_glideslope_distance, computed_horizontal_offset, airplane);
            }
            else
            {
               glRotated(m_touchdown_longitude,  0.0,  1.0,  0.0);
               glRotated(-m_touchdown_latitude,  1.0,  0.0,  0.0);
               glTranslated(0.0, 0.0,  0.95  *  m_scale);
               glRotated(270.0 - m_localizer_heading,  0.0,  0.0,  1.0);

               // Determine if the aircraft is off the scale - and 'move' the aircraft if it is.
               out_of_range   =  LimitHorizontalRange(m_nm_to_td_value, m_glideslope_distance, computed_horizontal_offset, airplane);
            }

            // Draw the track with the previous rotates and translates.
            m_track.Draw();

            // Draw the airplane.
            glPushMatrix();
            {
               float vertical_limit             =  (m_localizer_deviation_value /  NMI_FT)  *  (m_scale /  3600.0f);
               float max_deviation_to_the_right =  -(m_maximum_NM_shown *  NMI_FT   *  m_aspect_ratio /  2.0f);
               float max_deviation_to_the_left  =   (m_maximum_NM_shown *  NMI_FT   *  m_aspect_ratio /  2.0f);
               float vertical_offset            =  100.0f   *  m_maximum_NM_shown;  //  At 1NM scale this is 100ft at 10NM scale this is 1000ft.

               // If right of the maximum deviation to the right for this scale, graphically move the aircraft vertical_offset lower 
               // than the maximum deviation so it doesn't half disappear, and paint it DIM-RED.
               if (m_localizer_deviation_value        <  max_deviation_to_the_right)
               {
                  vertical_limit =  ((max_deviation_to_the_right  +  vertical_offset)  /  NMI_FT)  *  (m_scale /  3600.0f);
                  airplane.SetForegroundColor(128, 0, 0);
               }
               // If within vertical_offset of the maximum deviation to the right for this scale, graphically keep the aircraft at 
               // vertical_offset lower than the maximum deviation so it doesn't half disappear, and leave it WHITE.
               else if (m_localizer_deviation_value   <  max_deviation_to_the_right +  vertical_offset)
               {
                  vertical_limit =  ((max_deviation_to_the_right  +  vertical_offset)  /  NMI_FT)  *  (m_scale /  3600.0f);
                  if (! out_of_range)
                     // Changed 255,255,255 to 254,254,254 to support printing
                     airplane.SetForegroundColor(254, 254, 254);
               }
               // If right of the maximum deviation to the left for this scale, graphically move the aircraft vertical_offset higher 
               // than the maximum deviation so it doesn't half disappear, and paint it DIM-RED.
               else if (m_localizer_deviation_value   >  max_deviation_to_the_left)
               {
                  vertical_limit =  ((max_deviation_to_the_left   -  vertical_offset)  /  NMI_FT)  *  (m_scale /  3600.0f);
                  airplane.SetForegroundColor(128, 0, 0);
               }
               // If within vertical_offset of the maximum deviation to the left for this scale, graphically keep the aircraft at 
               // vertical_offset higher than the maximum deviation so it doesn't half disappear, and leave it WHITE.
               else if (m_localizer_deviation_value   >  max_deviation_to_the_left -  vertical_offset)
               {
                  vertical_limit =  ((max_deviation_to_the_left   -  vertical_offset)  /  NMI_FT)  *  (m_scale /  3600.0f);
                  if (! out_of_range)
                     // Changed 255,255,255 to 254,254,254 to support printing
                     airplane.SetForegroundColor(254, 254, 254);
               }
               else if (! out_of_range)
                  // Changed 255,255,255 to 254,254,254 to support printing
                  airplane.SetForegroundColor(254, 254, 254);

               glTranslated(computed_horizontal_offset, vertical_limit, 0.0);
               glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
               airplane.Draw(m_airplane, true, -m_ac_heading - (270.0 - m_localizer_heading), 1.0f);
               glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
            }
            glPopMatrix();
         }
         glPopMatrix();
      }
      glPopMatrix();
   }
   glPopMatrix();
}


void  CApproachPlot::DrawCalculatedData()
{
   glPushMatrix();
   {
      if (m_aspect_ratio  <= 1.0f)
         glScalef(m_aspect_ratio, 1.0f, 1.0f);
      else if (m_aspect_ratio != 0.0f)
         glScalef(1.0f, 1.0f  /  m_aspect_ratio, 1.0f);

      glPushMatrix();
         glTranslatef(-0.45f  /  m_aspect_ratio, 0.50f, 0.0f);
         nmToTd.Draw(m_nm_to_td_title  +  m_nm_to_td, true, 0.0f, 9.0f);
      glPopMatrix();
      glPushMatrix();
         glTranslatef(-0.15f  /  m_aspect_ratio, 0.50f, 0.0);
         ftAboveTouchdown.Draw(m_ft_above_touchdown_title  +  m_ft_above_touchdown,   true, 0.0f, 9.0f);
      glPopMatrix();
      glPushMatrix();
         glTranslatef( 0.15f  /  m_aspect_ratio, 0.50f, 0.0f);
         localizerDeviation.Draw(m_localizer_deviation_title   +  m_localizer_deviation,  true, 0.0f, 9.0f);
      glPopMatrix();
      glPushMatrix();
         glTranslatef( 0.50f  /  m_aspect_ratio, 0.50f, 0.0f);
         glideslopeDeviation.Draw(m_glideslope_deviation_title  +  m_glideslope_deviation, true, 0.0f, 9.0f);
      glPopMatrix();
      glPushMatrix();
         glTranslatef( 0.80f  /  m_aspect_ratio, 0.50f, 0.0f);
         vRefDeviation.Draw(m_vref_deviation_title +  m_vref_deviation, true, 0.0f, 9.0f);
      glPopMatrix();
   }
   glPopMatrix();
}


void  CApproachPlot::DrawSideView()
{
   glPushMatrix();
   {
      // Offset everything so that the localizer shack is located at the right edge of the drawing area and 
      // so that the runway is located at the bottom of the drawing area.
      glTranslatef(1.0f, -1.0f, 0.0f);

      glPushMatrix();
      {
         if (m_aspect_ratio != 0.0f)
            glScalef(1.0f, 1.0f  /  m_aspect_ratio, 1.0f);

         // Draw the vertical scale depicting altitude.
         glPushMatrix();
         {
            glTranslatef(-1.90f, 0.0f, 0.0f);
            DrawVerticalScale(SIDE_VIEW);
         }
         glPopMatrix();


         // Move to what it is we really want to see...
         AdjustTheView();


         // Draw the Runway.
         glPushMatrix();
         {
            glRotated(m_runway_longitude,  0.0,  1.0,  0.0);
            glRotated(-m_runway_latitude,  1.0,  0.0,  0.0);
            glTranslated(0.0, 0.0,  0.95  *  m_scale);
            // Glideslope has same bearing as localizer so use m_localizer_heading not m_runway_heading.   
            glRotated(270.0 - m_localizer_heading,  0.0,  0.0,  1.0);

            DrawRunway(SIDE_VIEW);
         }
         glPopMatrix();

         // Draw the glideslope, horizontal scale and the airplane.
         glPushMatrix();
         {
            float computed_horizontal_offset =  0.0f;
            bool  out_of_range               =  false;

            // If this is an LDA...
            if (m_approach_type   &  LOCALIZER_DIRECTIONAL_AID)
            {
               // Draw the glideslope and the horizontal scale with respect to the distance from the localizer.
               glRotated(m_localizer_longitude,  0.0,  1.0,  0.0);
               glRotated(-m_localizer_latitude,  1.0,  0.0,  0.0);
               glTranslated(0.0, 0.0,  0.95  *  m_scale);
               glRotated(270.0 - m_localizer_heading,  0.0,  0.0,  1.0);

               float offset   =  (m_localizer_to_glideslope_distance /  NMI_FT)  *  (m_scale /  3600.0f);
               glTranslatef(offset, 0.0f, 0.0f);   // Move to the touchdown point.
               
               DrawHorizontalScale(SIDE_VIEW);
               DrawGlideslopeBeam();

               // Determine if the aircraft is off the scale - and 'move' the aircraft if it is.
               out_of_range   =  LimitHorizontalRange(m_nm_to_gs, m_localizer_to_glideslope_distance, computed_horizontal_offset, airplane_side_view);
            }
            else  //  If this isn't a LDA.
            {
               // Draw the glideslope and the horizontal scale with respect to the touchdown point.
               glRotated(m_touchdown_longitude,  0.0,  1.0,  0.0);
               glRotated(-m_touchdown_latitude,  1.0,  0.0,  0.0);
               glTranslated(0.0, 0.0,  0.95  *  m_scale);
               // Glideslope has same bearing as localizer so use m_localizer_heading not m_runway_heading.   
               glRotated(270.0 - m_localizer_heading,  0.0,  0.0,  1.0);

               DrawHorizontalScale(SIDE_VIEW);
               DrawGlideslopeBeam();

               // Determine if the aircraft is off the scale - and 'move' the aircraft if it is.
               out_of_range   =  LimitHorizontalRange(m_nm_to_td_value, m_glideslope_distance, computed_horizontal_offset, airplane_side_view);
            }

            m_altitude_track.Draw();

            // Draw the airplane.
            glPushMatrix();
            {
               float height            =  ALTITUDE_SCALE_FACTOR   *  (m_ft_above_touchdown_value   /  NMI_FT)  *  (m_scale /  3600.0f);
               float max_altitude      =  ((m_maximum_NM_shown *  NMI_FT   /  ALTITUDE_SCALE_FACTOR)  *  m_aspect_ratio);
               float vertical_offset   =  4.0f  *  m_maximum_NM_shown;  //  At 1NM scale this is 4ft at 10NM scale this is 40ft.

               // If above the maximum altitude for this scale, graphically move the aircraft vertical_offset lower than the maximum 
               // altitude so it doesn't half disappear, and paint it DIM-RED.
               if (m_ft_above_touchdown_value   >  max_altitude)
               {
                  height   =  ALTITUDE_SCALE_FACTOR   *  ((max_altitude -  vertical_offset)  /  NMI_FT)  *  (m_scale /  3600.0f);
                  airplane_side_view.SetForegroundColor(128, 0, 0);
               }
               // If within vertical_offset of the maximum altitude for this scale, graphically keep the aircraft at vertical_offset 
               // lower than the maximum altitude so it doesn't half disappear, and leave it WHITE.
               else if (m_ft_above_touchdown_value    >  max_altitude   -  vertical_offset)
               {
                  height   =  ALTITUDE_SCALE_FACTOR   *  ((max_altitude -  vertical_offset)  /  NMI_FT)  *  (m_scale /  3600.0f);
                  if (! out_of_range)
                     // Changed 255,255,255 to 254,254,254 to support printing
                     airplane_side_view.SetForegroundColor(254, 254, 254);
               }
               // If below the ground, graphically move the aircaft to the vertical_offset, so it doesn't sink into the ground, 
               // and paint it DIM-RED.
               else if (m_ft_above_touchdown_value    <  0.0f)
               {
                  height   =  ALTITUDE_SCALE_FACTOR   *  (vertical_offset   /  NMI_FT)  *  (m_scale /  3600.0f);
                  airplane_side_view.SetForegroundColor(128, 0, 0);
               }
               // If below vertical_offset, graphically keep the aircaft at vertical_offset, so it doesn't sink into the ground, 
               // and leave it WHITE.
               else if (m_ft_above_touchdown_value    <  vertical_offset)
               {
                  height   =  ALTITUDE_SCALE_FACTOR   *  (vertical_offset   /  NMI_FT)  *  (m_scale /  3600.0f);
                  if (! out_of_range)
                     // Changed 255,255,255 to 254,254,254 to support printing
                     airplane_side_view.SetForegroundColor(254, 254, 254);
               }
               else if (! out_of_range)
                  // Changed 255,255,255 to 254,254,254 to support printing
                  airplane_side_view.SetForegroundColor(254, 254, 254);

               // Move to the appropriate position...
               glTranslatef(computed_horizontal_offset, -height, 0.0f);
               // Re-orientate the ELEVATOR_TRIM symbol so the tail is pointing up when the nose is pointing to the right.
               glRotatef(180.0f, 1.0f, 0.0f, 0.0f);      
               // If going away from the glideslope shack then turn the aircraft symbol around.
               if (m_going_away  == GOING_AWAY)
                  glRotatef(180.0,  0.0f, 1.0f, 0.0f);   
               // Finally, draw the aircraft symbol.
               airplane_side_view.Draw(m_airplane_side_view,   true, -m_pitch, 1.15f);
            }
            glPopMatrix();
         }
         glPopMatrix();



         // Draw the inner marker...
         glPushMatrix();
            glRotated(m_inner_marker_longitude,  0.0,  1.0,  0.0);
            glRotated(-m_inner_marker_latitude,  1.0,  0.0,  0.0);
            glTranslated(0.0, 0.0,  0.95  *  m_scale);
            glRotated(270.0 - m_inner_marker_heading,  0.0,  0.0,  1.0);
            inner_marker.Draw(m_inner_marker_side_view,  true, 180.0f,  25.0f /  m_maximum_NM_shown);
         glPopMatrix();

      
         // Draw the middle marker...
         glPushMatrix();
            glRotated(m_middle_marker_longitude,  0.0,  1.0,  0.0);
            glRotated(-m_middle_marker_latitude,  1.0,  0.0,  0.0);
            glTranslated(0.0, 0.0,  0.95  *  m_scale);
            glRotated(270.0 - m_middle_marker_heading,  0.0,  0.0,  1.0);
            middle_marker.Draw(m_middle_marker_side_view,   true, 180.0f,  50.0f /  m_maximum_NM_shown);
         glPopMatrix();

      
         // Draw the outer marker...
         glPushMatrix();
            glRotated(m_outer_marker_longitude,  0.0,  1.0,  0.0);
            glRotated(-m_outer_marker_latitude,  1.0,  0.0,  0.0);
            glTranslated(0.0, 0.0,  0.95  *  m_scale);
            glRotated(270.0 - m_outer_marker_heading,  0.0,  0.0,  1.0);
            outer_marker.Draw(m_outer_marker_side_view,  true, 180.0f,  200.0f   /  m_maximum_NM_shown);
         glPopMatrix();
      }
      glPopMatrix();
   }
   glPopMatrix();
}


void  CApproachPlot::DrawVrefPlot()
{
   glPushMatrix();
   {
      // Offset everything so that the localizer shack is located at the right edge of the drawing area.
      glTranslatef(1.0f, 0.0f, 0.0f);

      glPushMatrix();
      {
         if (m_aspect_ratio != 0.0f)
            glScalef(1.0f, 1.0f  /  m_aspect_ratio, 1.0f);


         // Draw the vertical deviation grid lines.
         glPushMatrix();
         {
            glTranslatef(-1.90f, 0.0f, 0.0f);   // Near the left edge.
            DrawVerticalScale(VREF_VIEW);
         }
         glPopMatrix();


         // Move to what it is we really want to see...
         AdjustTheView();


         glPushMatrix();
         {
            float computed_horizontal_offset =  0.0f;
            bool  out_of_range               =  false;
            float vert_offset                =  0.0f;

            // If this is an LDA...
            if (m_approach_type   &  LOCALIZER_DIRECTIONAL_AID)
            {
               // Draw the horizontal scale with respect to the distance from the localizer.
               glRotated(m_localizer_longitude,  0.0,  1.0,  0.0);
               glRotated(-m_localizer_latitude,  1.0,  0.0,  0.0);
               glTranslated(0.0, 0.0,  0.95  *  m_scale);
               glRotated(270.0 - m_localizer_heading,  0.0,  0.0,  1.0);

               float offset   =  (m_localizer_to_glideslope_distance /  NMI_FT)  *  (m_scale /  3600.0f);
               glTranslatef(offset, 0.0f, 0.0f);   // Move to the touchdown point.
               
               DrawHorizontalScale(VREF_VIEW);

               // Determine if the aircraft is off the scale - and 'move' the aircraft if it is.
               out_of_range   =  LimitHorizontalRange(m_nm_to_gs, m_localizer_to_glideslope_distance, computed_horizontal_offset, vRefIndicator);
            }
            else
            {
               // Draw the horizontal scale with respect to the touchdown point.
               glRotated(m_touchdown_longitude,  0.0,  1.0,  0.0);
               glRotated(-m_touchdown_latitude,  1.0,  0.0,  0.0);
               glTranslated(0.0, 0.0,  0.95  *  m_scale);
               glRotated(270.0 - m_localizer_heading,  0.0,  0.0,  1.0);

               DrawHorizontalScale(VREF_VIEW);

               // Determine if the aircraft is off the scale - and 'move' the aircraft if it is.
               out_of_range   =  LimitHorizontalRange(m_nm_to_td_value, m_glideslope_distance, computed_horizontal_offset, vRefIndicator);
            }


            m_airspeed_track.Draw();

            // Draw the vRef indicator.
            glPushMatrix();
            {
               // If coming in too fast - flat line the graph at the maximum and change the indicator to RED.
               if (m_ias   -  m_vRef   >  m_max_ias_delta)
               {
                  vert_offset =  ((m_max_airspeed_shown  -  m_vRef)  /  m_max_deviation_shown)  *  m_aspect_ratio;
                  vRefIndicator.SetForegroundColor(255, 0, 0);
               }
               // If coming in too slow - flat line the graph at the minimum and change the indicator to RED.
               else if (m_vRef   -  m_ias >  m_max_ias_delta)
               {
                  vert_offset =  ((m_min_airspeed_shown  -  m_vRef)  /  m_max_deviation_shown)  *  m_aspect_ratio;
                  vRefIndicator.SetForegroundColor(255, 0, 0);
               }
               // If coming in within tolerance - change the indicator to white.
               else
               {
                  vert_offset =  ((m_ias  -  m_vRef)  /  m_max_deviation_shown)  *  m_aspect_ratio;
                  if (! out_of_range)
                     // Changed 255,255,255 to 254,254,254 to support printing
                     vRefIndicator.SetForegroundColor(254, 254, 254);
               }

               // Move to the appropriate position...
               glTranslatef(computed_horizontal_offset, -vert_offset, 0.0f);
               // Re-orientate the ELEVATOR_TRIM symbol so the tail is pointing up when the nose is pointing to the right.
               glRotatef(180.0f, 1.0f, 0.0f, 0.0f);
               // If going away from the glideslope shack then turn the aircraft symbol around.
               if (m_going_away  == GOING_AWAY)
                  glRotatef(180.0,  0.0f, 1.0f, 0.0f);
               // Finally, draw the aircraft symbol.
               vRefIndicator.Draw(m_airplane_side_view,   true, -m_pitch, 1.15f);
            }
            glPopMatrix();
         }
         glPopMatrix();
      }
      glPopMatrix();
   }
   glPopMatrix();
}


void  CApproachPlot::DrawAimingPointMarkers()
{
   float runway_width         =  (m_runway_width   /  NMI_FT)  *  (m_scale /  3600.0f);
   float aiming_point_width   =  (50.0f            /  NMI_FT)  *  (m_scale /  3600.0f);
   float aiming_point_length  =  (100.0f           /  NMI_FT)  *  (m_scale /  3600.0f);

   // Changed 255,255,255 to 254,254,254 to support printing
   glRGB(254, 254, 254);

   glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
      glBegin(GL_QUADS);
         glVertex2f(-aiming_point_length  /  2.0f, -runway_width  /  2.0);
         glVertex2f( aiming_point_length  /  2.0f, -runway_width  /  2.0);
         glVertex2f( aiming_point_length  /  2.0f, -runway_width  /  2.0   +  aiming_point_width);
         glVertex2f(-aiming_point_length  /  2.0f, -runway_width  /  2.0   +  aiming_point_width);

         glVertex2f(-aiming_point_length  /  2.0f, runway_width   /  2.0   -  aiming_point_width);
         glVertex2f( aiming_point_length  /  2.0f, runway_width   /  2.0   -  aiming_point_width);
         glVertex2f( aiming_point_length  /  2.0f, runway_width   /  2.0);
         glVertex2f(-aiming_point_length  /  2.0f, runway_width   /  2.0);
      glEnd();
   glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
}


void  CApproachPlot::DrawHorizontalScale(enum approach_view_type which_view)
{
   // It has been determined that having a distance to touchdown scale drawn 
   // at a LDA facility presents a mind-bending view of the real world.  So 
   // if this is a LDA facility - don't show the distance to touchdown scale.
   if (!(m_approach_type   &  LOCALIZER_DIRECTIONAL_AID))
   {
      // Check to see if we need to recompute the data for this view.
      if (m_mapHorzRecompute[which_view])
      {
         float          fScaleDiv3600    = m_scale / 3600.0f;
         float          fMaxNMShownRatio = m_maximum_NM_shown * m_aspect_ratio;
         float          fResult          = fMaxNMShownRatio * fScaleDiv3600;
         ScaleIndicator local_indicator;
         local_indicator.label.SetForegroundColor(m_mapColorSelectables["SCALE_INDICATOR"].s_color.Red(),
                                     m_mapColorSelectables["SCALE_INDICATOR"].s_color.Green(),
                                     m_mapColorSelectables["SCALE_INDICATOR"].s_color.Blue());
         // Changed 0,0,0 to 1,1,1 to support printing
         local_indicator.label.SetBackgroundColor(true, 1, 1, 1);

         // Draw the scale indicators starting at the aiming point location...
         for (int ii =  0; ii <  2  *  m_maximum_NM_shown;  ii++)
         {
            // Set the text for the label
            if (ii == 0)                                    // Label the touch down point (aiming point marker location, aka glideslope shack)
               local_indicator.strLabel.Format("TD");
            else if (m_maximum_NM_shown   == 1  && ii %  2) // If on a 1NM scale and if ii is odd, then label the 0.5NM mark.
               local_indicator.strLabel.Format("%1.1f NM", ((float)ii /  2.0f));
            else if (ii %  2)                               // If odd, then these are the 0.5NM markings:  don't label them.
               local_indicator.strLabel.Format("");
            else                                            // Label all the NM markings.
               local_indicator.strLabel.Format("%d NM", ii/2);

            // Set the x position for the label.
            local_indicator.fX =  ((float)ii  /  2.0f) *  (fScaleDiv3600);

            // Set the y position for the label.
            if (which_view == TOP_VIEW)
               local_indicator.fY =   0.925f   *  fResult / 2.0f;
            else if (which_view == SIDE_VIEW)
               local_indicator.fY =  -0.975f   *  fResult;
            else if (which_view == VREF_VIEW)
               local_indicator.fY =  -0.175f   *  fResult;

            // Store the data.
            m_mapHorzIndicators[which_view].push_back(local_indicator);
         }

         // Data computed.  Don't recompute unless scale or
         // airport/runway changes.
         m_mapHorzRecompute[which_view] = false;
     }


      _FSI_STL::list<ScaleIndicator>::iterator lIt = 
                                       m_mapHorzIndicators[which_view].begin();
      glRGB(64, 64, 64);
      glBegin(GL_LINES);
         for (; lIt != m_mapHorzIndicators[which_view].end(); lIt++)
         {
             glVertex2f((*lIt).fX,  1 + (*lIt).fY);
             glVertex2f((*lIt).fX, -1 + (*lIt).fY);
         }
      glEnd();

      lIt = m_mapHorzIndicators[which_view].begin();
      glPushMatrix();
         // Draw the scale labels.  The fPrevY and fPrevX value are used
         // to allow the translations to occur from one label
         // to the next rather than pushing, translating, drawing,
         // and then popping for each label.  Now, push once,
         // translate, draw, translate, draw, etc, pop once.
         float fPrevX = 0.0f;
         float fPrevY = 0.0f;
         for (; lIt != m_mapHorzIndicators[which_view].end(); lIt++)
         {
             glTranslatef((*lIt).fX - fPrevX, (*lIt).fY - fPrevY, 0.0f);
             (*lIt).label.Draw((*lIt).strLabel, true, 180.0f, 0.3f);
             fPrevX = (*lIt).fX;
             fPrevY = (*lIt).fY;
         }
      glPopMatrix();
   }
}


void  CApproachPlot::DrawVerticalScale(enum approach_view_type which_view)
{
   // Check to see if we need to recompute the data for this view.
   if (m_mapVertRecompute[which_view])
   {
      int   ii =  0;
      float fScaleDiv3600 = m_scale / 3600.0f;
      float offset;
      ScaleIndicator local_indicator;
      local_indicator.label.SetForegroundColor(m_mapColorSelectables["SCALE_INDICATOR"].s_color.Red(),
                                  m_mapColorSelectables["SCALE_INDICATOR"].s_color.Green(),
                                  m_mapColorSelectables["SCALE_INDICATOR"].s_color.Blue());
      // Changed 0,0,0 to 1,1,1 to support printing
      local_indicator.label.SetBackgroundColor(true, 1, 1, 1);

      // The x position is always the same.
      local_indicator.fX = m_maximum_NM_shown * fScaleDiv3600;

      if (which_view == TOP_VIEW)
      {
         float change_in_deviation        =  m_maximum_NM_shown   *  100.0f;
         float max_deviation_to_the_right = -(m_maximum_NM_shown  *  NMI_FT   *  m_aspect_ratio /  2.0f) +  change_in_deviation;
         float max_deviation_to_the_left  =  (m_maximum_NM_shown  *  NMI_FT   *  m_aspect_ratio /  2.0f) -  change_in_deviation;

         for (float deviation =  change_in_deviation; 
              deviation   <= max_deviation_to_the_left; 
              deviation   += change_in_deviation)
         {
            offset   =  (deviation / NMI_FT)  * fScaleDiv3600;

            // Set the text and the y position for the label
            local_indicator.strLabel.Format("%5.f FT", deviation);
            local_indicator.fY = offset;

            // Set the color for the label.
            if (ii %  2)
                local_indicator.color = CColor("032032032");
            else
                local_indicator.color = CColor("048048048");

            // Store the data.
            m_mapVertIndicators[which_view].push_back(local_indicator);

            // Set the y position for the label
            local_indicator.fY = -offset;

            // Set the text for the label.
            if (ii++ %  2)
                local_indicator.color = CColor("032032032");
            else
                local_indicator.color = CColor("048048048");

            // Store the data.
            m_mapVertIndicators[which_view].push_back(local_indicator);
         }   

         // Data computed.  Don't recompute unless scale or
         // airport/runway changes.
         m_mapVertRecompute[which_view] = false;
      }
      else if (which_view == SIDE_VIEW)
      {
         float change_in_altitude   =  m_maximum_NM_shown   *  25.0f;
         float min_altitude         =  change_in_altitude;
         float max_altitude         =  ((m_maximum_NM_shown *  NMI_FT   /  ALTITUDE_SCALE_FACTOR)  *  m_aspect_ratio)   -  change_in_altitude;

         for (float  height   =  min_altitude;  
              height   <= max_altitude;  
              height   += change_in_altitude)
         {
            offset   =  ALTITUDE_SCALE_FACTOR  *  (height  /  NMI_FT)  *  fScaleDiv3600;

            // Set the text and the y position for the label
            local_indicator.strLabel.Format("%5.f FT", height);
            local_indicator.fY = offset;

            // Set the color for the label.
            if (ii++ %  2)
                local_indicator.color = CColor("032032032");
            else
                local_indicator.color = CColor("048048048");

            // Store the data.
            m_mapVertIndicators[which_view].push_back(local_indicator);
         }

         // Data computed.  Don't recompute unless scale or
         // airport/runway changes.
         m_mapVertRecompute[which_view] = false;
      }
      else if (which_view == VREF_VIEW)
      {
         for (float  airspeed =  m_min_airspeed_shown;  
              airspeed  <= m_max_airspeed_shown;  
              airspeed += m_max_ias_delta)
         {
            offset   =  ((airspeed   -  m_vRef)  /  m_max_deviation_shown)   *  m_aspect_ratio;

            // Set the text and the y position for the label
            local_indicator.strLabel.Format("%5.f KTS", airspeed);
            local_indicator.fY = offset;

            // Set the color for the label.
            if (ii++ %  2)
                local_indicator.color = CColor("128128128");
            else
                local_indicator.color = CColor("048048048");

            // Store the data.
            m_mapVertIndicators[which_view].push_back(local_indicator);
         }

         // Data computed.  Don't recompute unless scale or
         // airport/runway changes.
         m_mapVertRecompute[which_view] = false;
      }
   }

   _FSI_STL::list<ScaleIndicator>::iterator lIt = 
                                    m_mapVertIndicators[which_view].begin();
   glBegin(GL_LINES);
      // Draw all of the lines as stored in m_mapVertIndicators.
      // The data has already had the translation applied.
      for (; lIt != m_mapVertIndicators[which_view].end(); lIt++)
      {
          glRGB((*lIt).color.Red(), (*lIt).color.Green(), (*lIt).color.Blue());
          glVertex2f(-1.0f,      (*lIt).fY);
          glVertex2f( (*lIt).fX, (*lIt).fY);
      }
   glEnd();

   lIt = m_mapVertIndicators[which_view].begin();
   glPushMatrix();
      // Draw the scale labels.  The fPrevY value is used
      // to allow the translations to occur from one label
      // to the next rather than pushing, translating, drawing,
      // and then popping for each label.  Now, push once,
      // translate, draw, translate, draw, etc, pop once.
      float fPrevY = 0.0f;
      for (; lIt != m_mapVertIndicators[which_view].end(); lIt++)
      {
          glTranslatef(0.0f, (*lIt).fY - fPrevY, 0.0f);
          (*lIt).label.Draw((*lIt).strLabel, true, 0.0f, 0.3f);
          fPrevY = (*lIt).fY;
      }
   glPopMatrix();
}


void  CApproachPlot::DrawLocalizerBeam()
{
   // Draw the localizer fan if its width is not zero.
   if (m_localizer_width_angle)
   {
      glPushMatrix();
      {
         glRotated(m_localizer_longitude, 0.0,  1.0,  0.0);
         glRotated(-m_localizer_latitude, 1.0,  0.0,  0.0);
         glTranslated(0.0, 0.0,  0.95  *  m_scale);
         glRotated(270.0   -  m_localizer_heading,    0.0,  0.0,  1.0);

         if (m_bool_fake_ils)
         {
            glRGB(m_mapColorSelectables["FAKE_ILS"].s_color.Red(),
                  m_mapColorSelectables["FAKE_ILS"].s_color.Green(),
                  m_mapColorSelectables["FAKE_ILS"].s_color.Blue());
         }
         else
         {
            glRGB(m_mapColorSelectables["ACTIVE_ILS"].s_color.Red(),
                  m_mapColorSelectables["ACTIVE_ILS"].s_color.Green(),
                  m_mapColorSelectables["ACTIVE_ILS"].s_color.Blue());
         }

         // The commented lines are the original.  The vertex data has 
         // the result of multiplying a rotation matrix around z by ii.
         glBegin(GL_LINES);
         for (float  ii =  m_localizer_width_angle / 2.0f; ii >= -m_localizer_width_angle / 2.0f; ii -= m_localizer_width_angle / 4.0f)
         {
            glVertex2f(0.0f, 0.0f);
            glVertex2f(10.0f * cos(ii * DEG_TO_RAD), 10.0f * sin(ii * DEG_TO_RAD));
//            glPushMatrix();
//               glRotatef(ii, 0.0f, 0.0f, 1.0f);
//               glBegin(GL_LINES);
//                  glVertex2f(0.0f, 0.0f);
//                  glVertex2f(10.0f, 0.0f);
//               glEnd();
//            glPopMatrix();
         }
         glEnd();
      }
      glPopMatrix();
   }
}


void  CApproachPlot::DrawGlideslopeBeam()
{
   // Draw the glideslope fan if its width is not zero and its angle from the ground isn't zero.
   if (m_glideslope_width_angle  && m_glideslope_angle_from_ground)
   {
      if (m_bool_fake_ils)
      {
         glRGB(m_mapColorSelectables["FAKE_ILS"].s_color.Red(),
               m_mapColorSelectables["FAKE_ILS"].s_color.Green(),
               m_mapColorSelectables["FAKE_ILS"].s_color.Blue());
      }
      else
      {
         glRGB(m_mapColorSelectables["ACTIVE_ILS"].s_color.Red(),
               m_mapColorSelectables["ACTIVE_ILS"].s_color.Green(),
               m_mapColorSelectables["ACTIVE_ILS"].s_color.Blue());
      }

      glPushMatrix();
      {
         glScalef(1.0f, ALTITUDE_SCALE_FACTOR, 1.0f);
         glRotatef(-m_glideslope_angle_from_ground, 0.0f, 0.0f, 1.0f);

         float fResult = m_maximum_NM_shown *  (m_scale /  3600.0f);

         // The commented lines are the original.  The vertex data has 
         // the result of multiplying a rotation matrix around z by angle.
         // The point 0,0 is not affected by the rotation.  The second
         // vertex becomes the big quantity times cos(angle) and the
         // big quantity times sin(angle)--fResult * cos(angle), fResult *
         // sin(angle).
         glBegin(GL_LINES);
         for (float  angle =  m_glideslope_width_angle / 2.0f; angle >= -m_glideslope_width_angle / 2.0f; angle -= m_glideslope_width_angle / 4.0f)
         {
            glVertex2f( 0.0f, 0.0f);
            glVertex2f( fResult * cos(angle * DEG_TO_RAD), fResult * sin(angle * DEG_TO_RAD));
//            glPushMatrix();
//               glRotatef(angle, 0.0f, 0.0f, 1.0f);
//               glBegin(GL_LINES);
//                  glVertex2f( 0.0f, 0.0f);
//                  glVertex2f(m_maximum_NM_shown *  (m_scale /  3600.0f), 0.0f);
//               glEnd();
//            glPopMatrix();
         }
         glEnd();
      }
      glPopMatrix();
   }
}

void  CApproachPlot::DrawRunway(enum approach_view_type which_view)
{
   glRGB(96, 96, 96);
   if (which_view == TOP_VIEW)
   {
      float runway_length  =  (m_runway_length  /  NMI_FT)  *  (m_scale /  3600.0f);
      float runway_width   =  (m_runway_width   /  NMI_FT)  *  (m_scale /  3600.0f);

      glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
         glBegin(GL_QUADS);
            glVertex2f(0.0f,           -runway_width  /  2.0);
            glVertex2f(-runway_length, -runway_width  /  2.0);
            glVertex2f(-runway_length,  runway_width  /  2.0);
            glVertex2f(0.0f,            runway_width  /  2.0);
         glEnd();
         // Draw an extended centerline for the runway.
         glBegin(GL_LINES);
            glVertex2f(0.0f, 0.0f);
            glVertex2f(m_maximum_NM_shown *  (m_scale /  3600.0f), 0.0f);
         glEnd();
      glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
   }
   else if (which_view == SIDE_VIEW)
   {
      float runway_length     =  (m_runway_length  /  NMI_FT)  *  (m_scale /  3600.0f);
      float runway_thickness  =  0.02f;   // Purely a magic number that makes the runway look good.  It doesn't get scaled (on purpose!).

      glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
         glBegin(GL_QUADS);
            glVertex2f(0.0f,           -runway_thickness /  2.0);
            glVertex2f(-runway_length, -runway_thickness /  2.0);
            glVertex2f(-runway_length,  runway_thickness /  2.0);
            glVertex2f(0.0f,            runway_thickness /  2.0);
         glEnd();
      glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
   }
}


void  CApproachPlot::AdjustTheView(void)
{
   // Unless this is a LDA facility, shift the whole drawing so that the glideslope shack is near the right edge of the screen.
   if (!(m_approach_type   &  LOCALIZER_DIRECTIONAL_AID))
   {
      float clipping_point =  0.0f;
      if (m_localizer_to_glideslope_distance <  m_runway_length)
         clipping_point =  ((m_localizer_to_glideslope_distance -  m_glideslope_distance)  /  NMI_FT)  *  (m_scale /  3600.0f);
      else
         clipping_point =  ((m_runway_length -  m_glideslope_distance)  /  NMI_FT)  *  (m_scale /  3600.0f);
      glTranslatef(clipping_point,  0.0f, 0.0f);
   }


   // Determine what to look at ... runway or localizer.
   if (m_approach_type  == NON_PRECISION)
   {
      // Force the centerline of runway to be horizontal on the screen (and therefore all other items drawn to be rotated accordingly).
      glRotated(m_runway_heading -  90.0,    0.0,  0.0,  1.0);

      // Remember we are using real lat/lon to draw, so rotate the 'Earth' so we see the desired runway.
      glRotated( m_runway_latitude,    1.0,  0.0,  0.0);
      glRotated(-m_runway_longitude,   0.0,  1.0,  0.0);
   }
   else
   {
      // Force the centerline of localizer fan to be horizontal on the screen (and therefore all other items drawn to be rotated accordingly).
      glRotated(m_localizer_heading -  90.0,    0.0,  0.0,  1.0);

      // Remember we are using real lat/lon to draw, so rotate the 'Earth' so we see the desired localizer.
      glRotated( m_localizer_latitude,    1.0,  0.0,  0.0);
      glRotated(-m_localizer_longitude,   0.0,  1.0,  0.0);
   }
}


bool  CApproachPlot::LimitHorizontalRange(float real_world_distance_to_ac, float glideslope_distance, float &computed_horizontal_offset, COpenGLtext &icon)
{
   float right_horizontal_offset =  (125.0f  *  m_maximum_NM_shown   -  glideslope_distance) /  NMI_FT;
   float left_horizontal_offset  =  ( 60.0f  *  m_maximum_NM_shown   +  glideslope_distance) /  NMI_FT;
   bool  out_of_range            =  false;

   // If the real_world_distance_to_ac is greater than the maximum distance shown for this scale minus the glideslope distance, 
   // (since the scale is offset by the glideslope distance), then keep the icon at the left edge of the screen and paint it DIM-RED.
   if (real_world_distance_to_ac >  m_maximum_NM_shown   -  glideslope_distance  /  NMI_FT)
   {
      computed_horizontal_offset =  (m_maximum_NM_shown  -  left_horizontal_offset) *  (m_scale /  3600.0f);
      icon.SetForegroundColor(128, 0, 0);
      out_of_range   =  true;
   }
   // If within left_horizontal_offset of the maximum distance shown for this scale, then keep the icon at the left 
   // edge of the screen and keep it WHITE.
   else if (real_world_distance_to_ac  >  m_maximum_NM_shown   -  left_horizontal_offset)
   {
      computed_horizontal_offset =  (m_maximum_NM_shown  -  left_horizontal_offset) *  (m_scale /  3600.0f);
      // Changed 255,255,255 to 254,254,254 to support printing
      icon.SetForegroundColor(254, 254, 254);
   }
   // If the real_world_distance_to_ac is less than the glideslope distance, (since the scale is offset by the glideslope 
   // distance), then keep the aircraft at the right edge of the screen and paint it DIM-RED.
   else if (real_world_distance_to_ac  <  -glideslope_distance /  NMI_FT)
   {
      computed_horizontal_offset =  right_horizontal_offset *  (m_scale /  3600.0f);
      icon.SetForegroundColor(128, 0, 0);
      out_of_range   =  true;
   }
   // If within right_horizontal_offset then keep the aircraft at the right edge of the screen and keep it WHITE.
   else if (real_world_distance_to_ac  <  right_horizontal_offset)
   {
      computed_horizontal_offset =  right_horizontal_offset *  (m_scale /  3600.0f);
      // Changed 255,255,255 to 254,254,254 to support printing
      icon.SetForegroundColor(254, 254, 254);
   }
   else
   {
      computed_horizontal_offset =  real_world_distance_to_ac  *  (m_scale /  3600.0f);
      // Changed 255,255,255 to 254,254,254 to support printing
      icon.SetForegroundColor(254, 254, 254);
   }

   return   out_of_range;
}
