/////////////////////////////////////////////////////////////////////////////
//
//           F L I G H T S A F E T Y   I N T E R N A T I O N A L
//                     Simulation Systems Division
//                      2700 North Hemlock Circle
//                     Broken Arrow, Oklahoma 74012
//                          (918) 259-4000
/////////////////////////////////////////////////////////////////////////////
//
// DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
// Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
// technical or operational data or information from automatic dissemination
// under the International Exchange Program or by other means.  This protection
// covers information required solely for administrative or operational
// purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
//
// WARNING:  This document contains technical data whose export is restricted
// by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
// Executive Order 12470.  Violation of these export control laws is subject
// to severe criminal penalties.  Dissemination of this document is controlled
// under DOD Directive 5230.25
//
/////////////////////////////////////////////////////////////////////////////
//
//
// Filename         : AircraftTrack.cpp
//
// Date             : 14 September 1999
//
// Engineer         : Billy Baker
//
// Revision         : $Revision: 1.6 $
//
// Description      : AircraftTrack.cpp contains the implementation of 
//                    the CAircraftTrack class.  This class is to be 
//                    used with the areamap to plot the lat/lon of the 
//                    aircraft(s).  It is similar to CApproachTrack.
//
// Classification   : UNCLASSIFIED
//
// Requirements     : None.
//
// Components Used  : _FSI_STL::list, math.h, Core::CVariant, .
//
// Operational 
//    Restrictions  : Machine dependencies/restrictions
//                        None.
//                    Design dependencies/restrictions
//                        None.
//                    Operations containing dependencies/restrictions
//                        None.
//                    Compiler dependencies/restrictions
//                        None.
//                    Other conditions for proper execution
//                        None.
//
// Environment      : Operating system(s) - Microsoft Windows NT 4.0 with
//                                              NT service pack 3, 4, or 5
//                                          Microsoft Windows NT 2000
//
//                    Compiler(s) - Visual C++ 6.0
//
//                    Architechure(s) - Intel Pentium, Pentium II
/////////////////////////////////////////////////////////////////////////////

/////////////////////////////////////////////////////////////////////////////
//
//                              R e v i s i o n   H i s t o r y
//
/////////////////////////////////////////////////////////////////////////////
// $Log: aircrafttrack.cpp $                                                                     
// Revision 1.6  2000/01/27 20:02:02  billyb                                                                     
// Changed byte by byte memset to 8 byte by 8 byte memset.                                                                     
// Revision 1.5  2000/01/27 19:53:58  billyb                                                                     
// Corrected possible memory leaks/corruption with copy                                                                     
// constructor.  Changed memcpy to non-byte by byte copy.                                                                     
// Changed list types to match new type in CVariant.                                                                     
// Revision 1.4  1999/11/26 21:01:34  billyb                                                                     
// Changed signature of UpdateData.  Changed UpdateData                                                                     
// to work with the new format of the data lists for CVariant                                                                     
// persistant data.                                                                     
// Revision 1.3  1999/10/19 05:28:17  billyb                                                                     
// Changed float f's to double d's.                                                                     
// Revision 1.2  1999/09/14 19:19:52  billyb                                                                     
// Added comments.  Replaced include for OpenGLText with                                                                     
// gl\gl.h.  Added extra conditionals to prevent trying to add                                                                      
// or draw out of bounds data.  Added VerticesAdded                                                                      
// accessor for m_ulVerticesAdded which is used with                                                                      
// a full list after a clear.                                                                     
/////////////////////////////////////////////////////////////////////////////
#include "..\core\stdafx.h"
#include <list>
#include <gl\gl.h>
#include "..\core\Variant.h"
#include "AircraftTrack.h"
#include "..\core\DataConversion.h"
#include "OpenGLMap.h"

#include "..\comms\commsshared.h"

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

CAircraftTrack::CAircraftTrack()
{
   m_dScale          =  1.0f;
   m_vertices        =  NULL;
   m_ulArraySize     =  0;
   m_ulVerticesAdded =  0;

   m_red             =  255;
   m_green           =  255;
   m_blue            =  255;

   m_hWnd            =  NULL;
   m_eTDC            =  AreaMapWing;

   m_bUseAlt         = false;

   m_fElevation      = 0.0f;
   m_pHeightOfTerrain = NULL;

   m_ulLastTimeValue = 0;
   m_times           = NULL;

   ArraySize(13500);
}

CAircraftTrack::CAircraftTrack(const CAircraftTrack& rAircraftTrack)
{
   m_dScale          =  1.0f;
   m_vertices        =  NULL;
   m_ulArraySize     =  0;
   m_ulVerticesAdded =  0;

   m_red             =  255;
   m_green           =  255;
   m_blue            =  255;

   m_hWnd            =  NULL;
   m_eTDC            =  AreaMapWing;

   m_bUseAlt         = false;

   m_fElevation      = 0.0f;
   m_pHeightOfTerrain = NULL;

   m_ulLastTimeValue = 0;
   m_times           = NULL;

   ArraySize(13500);

    *this = rAircraftTrack;
}

CAircraftTrack::~CAircraftTrack()
{
   if (m_vertices != NULL)
   {
       delete [] m_vertices;

       m_vertices = NULL;
   }

   if (m_times != NULL)
   {
       delete [] m_times;

       m_times = NULL;
   }
}

CAircraftTrack& CAircraftTrack::operator=(const CAircraftTrack& rAircraftTrack)
{
	// no work if they are the same
	if (this != &rAircraftTrack)
	{
        if (m_vertices != NULL)
        {
            delete [] m_vertices;
            m_vertices          = NULL;
            m_ulVerticesAdded   = 0;
        }

        if (m_times != NULL)
        {
            delete [] m_times;
            m_times = NULL;
        }

        Scale(rAircraftTrack.m_dScale);

        ArraySize(rAircraftTrack.m_ulArraySize);

        m_ulVerticesAdded = rAircraftTrack.m_ulVerticesAdded;

        SetColor(rAircraftTrack.m_red, rAircraftTrack.m_green, rAircraftTrack.m_blue);

        unsigned long ul = 0;
        unsigned long ulSize = m_ulArraySize;
        unsigned long ulTimeIndex = 0;
        while (ulSize)
        {
            m_vertices[ul]      = rAircraftTrack.m_vertices[ul];
            m_vertices[ul + 1]  = rAircraftTrack.m_vertices[ul + 1];
            m_vertices[ul + 2]  = rAircraftTrack.m_vertices[ul + 2];

            m_times[ulTimeIndex] = rAircraftTrack.m_times[ulTimeIndex];

            ++ulTimeIndex;

            ul += 3;
            --ulSize;
        }

        Hwnd(rAircraftTrack.m_hWnd);
        TypeOfTrack(rAircraftTrack.m_eTDC);
        UseAltitude(rAircraftTrack.m_bUseAlt);
	}

	return *this;
}
    
bool CAircraftTrack::operator==(const CAircraftTrack& rAircraftTrack)
{
    bool bRetVal = true;

    if (m_dScale != rAircraftTrack.m_dScale)
        bRetVal = false;

    if (m_ulVerticesAdded != rAircraftTrack.m_ulVerticesAdded)
        bRetVal = false;

    if (m_ulArraySize != rAircraftTrack.m_ulArraySize)
        bRetVal = false;

    if (m_red  != rAircraftTrack.m_red)
        bRetVal = false;

    if (m_green  != rAircraftTrack.m_green)
        bRetVal = false;

    if (m_blue  != rAircraftTrack.m_blue)
        bRetVal = false;

    if (m_hWnd != rAircraftTrack.m_hWnd)
        bRetVal = false;

    if (m_eTDC != rAircraftTrack.m_eTDC)
        bRetVal = false;

    if (m_bUseAlt != rAircraftTrack.m_bUseAlt)
        bRetVal = false;

    return bRetVal;
}

void  CAircraftTrack::Draw(void)
{
   if (m_ulVerticesAdded <= 0 || m_ulVerticesAdded > m_ulArraySize)
      return;

   glColor3ub(m_red, m_green, m_blue);

   glVertexPointer(3, GL_FLOAT, 0, (const void*)m_vertices);
   glDrawArrays(GL_LINE_STRIP, 0, m_ulVerticesAdded);
}

void CAircraftTrack::UpdateData()
{
    if (m_ulArraySize < 1)
        return;

    // Get the pointers for start of new data and last clear.
    bool bOk;
    _FSI_STL::list<struct TolData>::iterator itStart;
    _FSI_STL::list<struct TolData>::iterator itClear;

    CToleranceData::ClearIt(m_hWnd, m_eTDC, bOk, itClear);
    if (!bOk)
        return;

    CToleranceData::StartIt(m_hWnd, m_eTDC, bOk, itStart);

    if (!bOk)
        return;

    if (m_bUseAlt)
    {
        if (m_pHeightOfTerrain == NULL)
            m_pHeightOfTerrain = CCommsShared::FindVariant("height_of_terrain");

        if (m_pHeightOfTerrain != NULL)
            m_fElevation = (float)(*m_pHeightOfTerrain);
    }

    if (CToleranceData::ClearSinceLastUpdate(m_hWnd, m_eTDC))
        m_ulVerticesAdded = 0;

    unsigned short usAdded      = CToleranceData::Added(m_hWnd, m_eTDC);

    bool bAddClears = false;
    // See if we need to add data starting at the last clear.
    if (m_ulVerticesAdded == 0 && itClear != itStart)
    {
        bAddClears = true;
    }

    unsigned long ulCurIndex    = 3 * m_ulVerticesAdded;
    unsigned long ulLoop        = usAdded;
    struct TolData td;

    // Add data since last clear.  Should occur if the vertices currently added
    // is 0 and the clear pointer is not at the same point as the start pointer.
    if (bAddClears)
    {
        while (itClear != itStart)
        {
           td = (*itClear);

           AddVertice(td, ulCurIndex);

           ulCurIndex += 3;

           ++itClear;
           ++m_ulVerticesAdded;

           // Check to see if the maximum number of vertices have been
           // added and there is still more clear data to add.  If so,
           // then decrease the number of vertices added by one and move all
           // of the data down.  Also, move the current index back down 3.
           if (m_ulVerticesAdded == m_ulArraySize && itClear != itStart)
           {
               ulCurIndex -= 3;

               --m_ulVerticesAdded;

				int i = 0;
				int j = 3;
				int k = m_ulVerticesAdded * 3;
                int nTimeIndex1 = 0;
                int nTimeIndex2 = 1;
				while (i < k)
				{
					m_vertices[i]	= m_vertices[j];
					m_vertices[i+1] = m_vertices[j+1];
					m_vertices[i+2] = m_vertices[j+2];

                    m_times[nTimeIndex1]    = m_times[nTimeIndex2];

					j += 3;
					i += 3;

                    ++nTimeIndex1;
                    ++nTimeIndex2;
				}
           }
        }
    }

    // Max number of vertices added and more need to be added.
    // Move the data down the number of vertices that need to
    // be added.
    if ((m_ulVerticesAdded + usAdded) >= m_ulArraySize && usAdded > 0)
    {
        int nNumberOfVerticesBeyondMax = (m_ulVerticesAdded + usAdded - m_ulArraySize);

		// If greater than the max size, then all of the data will get replaced
		// and a move is not necessary.
		if (nNumberOfVerticesBeyondMax < m_ulArraySize)
		{
			// Make sure that VerticesAdded stays non-negative.
			// To go negative would mean that the amount beyond the
			// end is greater than the number already in the array.  Thus,
			// all of the vertices in the array will get removed and the
			// number added would be zero.
			if (nNumberOfVerticesBeyondMax < m_ulVerticesAdded)
			{
				m_ulVerticesAdded -= nNumberOfVerticesBeyondMax;
				int i = 0;
				int j = nNumberOfVerticesBeyondMax * 3;
				int k = m_ulVerticesAdded * 3;
                int nTimeIndex1 = 0;
                int nTimeIndex2 = 1;
				while (i < k)
				{
					m_vertices[i]	= m_vertices[j];
					m_vertices[i+1] = m_vertices[j+1];
					m_vertices[i+2] = m_vertices[j+2];

                    m_times[nTimeIndex1]    = m_times[nTimeIndex2];

					j += 3;
					i += 3;

                    ++nTimeIndex1;
                    ++nTimeIndex2;
                }
			}
			else
			{
				m_ulVerticesAdded = 0;
				if (usAdded > m_ulArraySize)
				{
					usAdded = m_ulArraySize;
					ulLoop = usAdded;
				}
			}

			ulCurIndex = 3 * m_ulVerticesAdded;
		}
		else
		{
			while (nNumberOfVerticesBeyondMax > m_ulArraySize)
			{
				--usAdded;
				++itStart;
				--nNumberOfVerticesBeyondMax;
			}

			if (usAdded > m_ulArraySize)
				usAdded -= m_ulArraySize;

			m_ulVerticesAdded	= 0;
			ulCurIndex			= 0;
			ulLoop				= m_ulArraySize;
		}
    }

    // All deques are assumed to have single element arrays as data.
    while (ulLoop)
    {
       td = (*itStart);

       AddVertice(td, ulCurIndex);

       ulCurIndex += 3;

       ++itStart;
       --ulLoop;
    }

    if (usAdded > 0)
        m_ulVerticesAdded += usAdded;

    // If the last time value minus the fist time value in
    // the array of times is greater than or equal to the
    // array size which should be the maximum time distance
    // between points then move all of the data down one.
    int nTimeIndex = 0;
    while (nTimeIndex < m_ulVerticesAdded &&
           (m_ulLastTimeValue - m_times[nTimeIndex]) >= m_ulArraySize)
    {
        ++nTimeIndex;
    }

    m_ulVerticesAdded -= nTimeIndex;

    int i = 0;
	int j = nTimeIndex * 3;
	int k = m_ulVerticesAdded * 3;
    int nTimeIndex1 = 0;
    int nTimeIndex2 = nTimeIndex;
	while (i < k)
	{
		m_vertices[i]	= m_vertices[j];
		m_vertices[i+1] = m_vertices[j+1];
		m_vertices[i+2] = m_vertices[j+2];

        m_times[nTimeIndex1]    = m_times[nTimeIndex2];

		j += 3;
		i += 3;

        ++nTimeIndex1;
        ++nTimeIndex2;
    }
}

void CAircraftTrack::AddVertice(struct TolData& td, unsigned long ulCurIndex)
{
    // No error checking.  Called from UpdateData.
    double dcosLon;
    double dsinLon;
    double dcosLat;
    double dsinLat;

    double dLonRad;
    double dLatRad;

    double dZTrans;
    double dSurface = 0.95 * m_dScale;

    double dTemp            = (m_dScale / 3600.0) / NMI_FT;

    
    m_ulLastTimeValue = td.s_ulTime;
    m_times[ulCurIndex / 3] = m_ulLastTimeValue;

    // Precompute all of the vertices having computed the two
    // rotations and the translate needed to get the vertex
    // to the correct position on the screen.
    dLonRad =  td.s_dLon  *  DEG_TO_RAD;
    dLatRad = -td.s_dLat  *  DEG_TO_RAD;

    dcosLon = cos(dLonRad);
    dsinLon = sin(dLonRad);
    dcosLat = cos(dLatRad);
    dsinLat = sin(dLatRad);

    // Add the altitude to the distance from the center of the earth.
    dZTrans = dSurface;
    if (true == m_bUseAlt)
    {
       dZTrans  += (td.s_fAlt -  m_fElevation)  *  dTemp;
    }

    // Values come from multiplying R1 * R2 * T1 where R1 is the rotation for
    // the longitude, R2 is the rotation for the negative latitude, and
    // T1 is a translation of (0, 0, fZTrans) where fZTrans is defined
    // above.  The point that needs to be rendered after applying R1, R2, and
    // T1 is (0, 0, 0).  Thus, precomputing the matrices decreases the number
    // of operations since many of the terms fall out.
    m_vertices[ulCurIndex]     = dsinLon * ( dZTrans) * dcosLat; 
    m_vertices[ulCurIndex + 1] =           (-dZTrans) * dsinLat, 
    m_vertices[ulCurIndex + 2] = dcosLon * ( dZTrans) * dcosLat;
}

/////////////////////////////////////////////////////////////////////////////
//
// void CAircraftTrack::Clear()
//
// Inputs           : None.
//
// Return Values    : None.
//
// Date             : 13 July 1999
//
// Engineer         : Billy Baker
//
// Description      : This method is used to tell the CAircraftTrack 
//                    class that a clear is needed.
//
/////////////////////////////////////////////////////////////////////////////
void CAircraftTrack::Clear()
{
    m_ulVerticesAdded = 0;
    CToleranceData::Clear(m_hWnd, m_eTDC);
}

//////////////////////////////////////////////////////////////////////
// Accessors
//////////////////////////////////////////////////////////////////////

double CAircraftTrack::Scale()
{
    return m_dScale;
}

unsigned long int CAircraftTrack::ArraySize()
{
    return m_ulArraySize;
}

unsigned long int CAircraftTrack::VerticesAdded()
{
    return m_ulVerticesAdded;
}

HWND CAircraftTrack::Hwnd()
{
    return m_hWnd;
}

enum TolDataClients CAircraftTrack::TypeOfTrack()
{
    return m_eTDC;
}

bool CAircraftTrack::UseAltitude()
{
    return m_bUseAlt;
}

//////////////////////////////////////////////////////////////////////
// Mutators
//////////////////////////////////////////////////////////////////////

void CAircraftTrack::Scale(const double dScale)
{
    if (dScale == 0.0)
        return;

    // m_dScale and scale are defined in CAircraftPlot to be
    // 7200/user defined scale.  Thus, if the map scale
    // goes from 30 to 60 then all of the points need to
    // be multiplied by 0.5.  Since increasing the real
    // scale makes m_dScale and scale smaller (7200/30 >
    // 7200/60), the order here is new scale/old scale.
    double dScaleRatio = dScale/m_dScale;

    // Scale all of the points already in the array to
    // be drawn for the track.
    unsigned long int ulIndex   = 0;
    unsigned long int ulSize    = m_ulVerticesAdded;
    while (ulSize)
    {
       m_vertices[ulIndex]      *= dScaleRatio; 
       m_vertices[ulIndex + 1]  *= dScaleRatio; 
       m_vertices[ulIndex + 2]  *= dScaleRatio; 

       ulIndex += 3;
       --ulSize;
    }

    m_dScale           = dScale;
}

void CAircraftTrack::ArraySize(const unsigned long int ulArraySize)
{
    if (ulArraySize != m_ulArraySize && m_vertices != NULL)
    {
        delete [] m_vertices;
        m_vertices = NULL;
    }

    if (ulArraySize != m_ulArraySize && m_times != NULL)
    {
        delete [] m_times;
        m_times = NULL;
    }

    if (m_vertices == NULL)
    {
        // Make the array one vertices larger than the size requested to append
        // the first vertice to the end for use with GL_LINE_STRIP.
        m_vertices = new GLfloat[ulArraySize * 3];
    }

    if (m_times == NULL)
        m_times = new unsigned long int[ulArraySize];

    memset(m_times, 0, ulArraySize);

    // Set everyone to 0.0.
    unsigned long ul = 0;
    unsigned long ulSize = ulArraySize;
    while (ulSize)
    {
        m_vertices[ul]      = 0.0;
        m_vertices[ul + 1]  = 0.0;
        m_vertices[ul + 2]  = 0.0;

        ul += 3;
        --ulSize;
    }

    m_ulArraySize = ulArraySize;
}

void CAircraftTrack::SetColor(int red, int green, int blue)
{
   m_red    =  red;
   m_green  =  green;
   m_blue   =  blue;
}

void CAircraftTrack::Hwnd(HWND hWnd)
{
    m_hWnd = hWnd;
}

void CAircraftTrack::TypeOfTrack(enum TolDataClients eTDC)
{
    m_eTDC = eTDC;
}

void CAircraftTrack::UseAltitude(bool bUseAlt)
{
    m_bUseAlt = bUseAlt;
}
