// 3DMapping.cpp: implementation of the C3DMapping class.
//
//////////////////////////////////////////////////////////////////////

#include "..\core\stdafx.h"
#include "..\core\dataconversion.h"
#include "3DMapping.h"


//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

C3DMapping::C3DMapping()
{
   m_exPtUpperLeft      =  CExtentsPoint(CPoint(0,0));
   m_exPtLowerRight     =  CExtentsPoint(CPoint(640,640));

   m_stlStrWidgetName   =  _FSI_STL::string("ThreeD_Map");

   m_listGraphicalElementVars.push_back("3d_map_scale");
   m_listGraphicalElementVars.push_back("3d_map_spin");
   m_listGraphicalElementVars.push_back("3d_map_camera");
   m_listGraphicalElementVars.push_back("3d_map_spider_web");
   m_listGraphicalElementVars.push_back("3d_map_scale_aircraft");
   m_listGraphicalElementVars.push_back("3d_map_spider_web_center_airport_id");
   m_listGraphicalElementVars.push_back("3d_map_zoom");
   m_listGraphicalElementVars.push_back("ac_pitch_lead");
   m_listGraphicalElementVars.push_back("ac_roll_lead");
   m_listGraphicalElementVars.push_back("ac_pitch_wing");
   m_listGraphicalElementVars.push_back("ac_roll_wing");
   m_listGraphicalElementVars.push_back("height_of_terrain");
   m_listGraphicalElementVars.push_back("Left Landing Gear Green");
   m_listGraphicalElementVars.push_back("Nose Landing Gear Green");
   m_listGraphicalElementVars.push_back("Right Landing Gear Green");
   m_listGraphicalElementVars.push_back("field_of_view");
   m_listGraphicalElementVars.push_back("camera_position");


   glEnable(GL_CULL_FACE);

   leadModel.Initialize(NOT_ANY);
   leadModel.SetGearPosition(NOSE_GEAR,    0.0f);
   leadModel.SetGearPosition(LEFT_GEAR,    0.0f);
   leadModel.SetGearPosition(RIGHT_GEAR,   0.0f);

   wingModel.Initialize(NAVY);
   wingModel.SetGearPosition(NOSE_GEAR,    0.0f);
   wingModel.SetGearPosition(LEFT_GEAR,    0.0f);
   wingModel.SetGearPosition(RIGHT_GEAR,   0.0f);

   glDisable(GL_CULL_FACE);


   m_3d_scale                       =  7200.0   /  25.0;
   m_3d_scale_cv                    =  m_3d_scale;
   m_scale_cv                       =  7200.0   /  10.0;

   m_tilt_angle                     =  -90.0f;
   
   m_spin_angle                     =  0.0f;
   m_spin_angle_cv                  =  m_spin_angle;

   m_camera_altitude                =  0.0f;

   m_camera_angle                   =  0.0f;
   m_camera_angle_cv                =  m_camera_angle;

   m_height_of_terrain              =  1000.0f;
   m_height_of_terrain_cv           =  m_height_of_terrain;

   m_ac_pitch_lead                  =  0.0f;
   m_ac_pitch_lead_cv               =  m_ac_pitch_lead;

   m_ac_roll_lead                   =  0.0f;
   m_ac_roll_lead_cv                =  m_ac_roll_lead;


   m_bool_nose_gear_down            =  false;
   m_bool_nose_gear_down_cv         =  m_bool_nose_gear_down;

   m_bool_left_gear_down            =  false;
   m_bool_left_gear_down_cv         =  m_bool_left_gear_down;

   m_bool_right_gear_down           =  false;
   m_bool_right_gear_down_cv        =  m_bool_right_gear_down;

   m_active_airport_latitude        =  0.0;
   m_active_airport_longitude       =  0.0;

   m_zoom_ratio                     =  1.0f;

   m_pquad  =  gluNewQuadric();
}


C3DMapping::~C3DMapping()
{
   glMatrixMode(GL_PROJECTION);
   glLoadIdentity();
   glMatrixMode(GL_MODELVIEW);
   glLoadIdentity();
   gluDeleteQuadric(m_pquad);
}


CWidget* C3DMapping::CreateObject()
{
   return new C3DMapping();
}


void  C3DMapping::BeginDrawing()
{
//   BeginDraw(64, 144, 255);
}


void  C3DMapping::SetupPerspective()
{
   glViewport(rectViewCoord.left,   rectView.Height() -  rectViewCoord.bottom,   (int)m_width,  (int)m_height);
   glScissor(rectViewCoord.left,    rectView.Height() -  rectViewCoord.bottom,   (int)m_width,  (int)m_height);


   glMatrixMode(GL_PROJECTION);
   glLoadIdentity();

   gluPerspective(m_field_of_view   /  m_zoom_ratio, 1.0,  0.0001, 1000000.0);
   m_z_offset  =  1.0f  /  tan((m_field_of_view /  2.0f  /  m_zoom_ratio)   *  DEG_TO_RAD);
   glMatrixMode(GL_MODELVIEW);
   glLoadIdentity();


   switch   (m_long_camera_position)
   {
      case  STUDENT:
      {
         float    altitude_offset   =  sin(m_camera_angle   *  DEG_TO_RAD) +  m_camera_altitude;
         double   sin_roll          =  sin(-m_ac_roll_wing  *  DEG_TO_RAD);
         double   cos_roll          =  cos(-m_ac_roll_wing  *  DEG_TO_RAD);

         gluLookAt(  0.0,        m_camera_altitude,   0.0,
                     0.0,        altitude_offset,     -1.0,
                     sin_roll,   cos_roll,            0.0);
      }
      break;

      case  TOWER:
      {
         gluLookAt(  0.0,  m_camera_altitude,   0.0,
                     0.0,  0.0,                 -1.0,
                     0.0,  1.0,                 0.0);
      }
      break;

      case  CHASE_PLANE:
      {
         gluLookAt(  0.0,  m_camera_altitude,   m_z_offset,
                     0.0,  m_camera_altitude,   0.0,
                     0.0,  1.0,                 0.0);
      }
      break;
   }


   glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
   glPushMatrix();
      glLoadIdentity();
      glTranslated(0.0, 0.0,  -m_z_offset);
      glRGB(64, 144, 255);    // Sky Blue
//      glRGB(0, 0, 0);
      glBegin(GL_QUADS);
         glVertex2f(-1.0f,-1.0f);
         glVertex2f( 1.0f,-1.0f);
         glVertex2f( 1.0f, 1.0f);
         glVertex2f(-1.0f, 1.0f);
      glEnd();
      glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
   glPopMatrix();

}


void  C3DMapping::MoveViewingPosition()
{
   glRotated(m_tilt_angle,  1.0,  0.0,  0.0);
   glRotated(m_spin_angle,  0.0,  0.0,  1.0);
}


void  C3DMapping::DrawEarth()
{
   glTranslated(0.0, 0.0,  -0.95 *  m_3d_scale);

   glPushMatrix();
   {
      glScaled(m_3d_scale,   m_3d_scale,   m_3d_scale);
      glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
      glRGB(0, 48, 0);
      gluSphere(m_pquad,  0.95f,  36,   180);
      glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
   }
   glPopMatrix();

   glPushMatrix();
      glLoadIdentity();
      glTranslated(0.0, 0.0,  -m_z_offset);

      glTranslated(-0.80, 0.95, 0.0);
      m_SpinAngle.Draw(m_CStr_spin_angle,             true, 0.0f, 0.45f);
      glTranslated( 0.40, 0.0,  0.0);
      m_CameraAngle.Draw(m_CStr_camera_angle,         true, 0.0f, 0.45f);
      glTranslated( 0.40, 0.0,  0.0);
      m_CameraAltitude.Draw(m_CStr_camera_altitude,   true, 0.0f, 0.45f);
      glTranslated( 0.40, 0.0,  0.0);
      m_DeltaAltitude.Draw(m_CStr_delta_altitude,     true, 0.0f, 0.45f);
      glTranslated( 0.40, 0.0,  0.0);
      m_Scale.Draw(m_CStr_scale,                      true, 0.0f, 0.45f);
   glPopMatrix();
}


void  C3DMapping::EndDrawing()
{
   EndDraw();

   glMatrixMode(GL_PROJECTION);
   glLoadIdentity();
   glMatrixMode(GL_MODELVIEW);
   glLoadIdentity();
}


void  C3DMapping::DrawLeadAircraft()
{
   glRotated(m_lead_ac_longitude,  0.0,  1.0,  0.0);
   glRotated(-m_lead_ac_latitude,  1.0,  0.0,  0.0);
   glTranslated(0.0, 0.0, 0.95 * m_3d_scale);

   glEnable(GL_DEPTH_TEST);
   glClear(GL_DEPTH_BUFFER_BIT);

   glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);


   if (m_lead_ac_altitude  -  m_height_of_terrain  >  1000.0)
   {
      switch   (m_long_camera_position)
      {
         case  STUDENT:
         {
            glRGB(255, 255, 64);
            gluSphere(m_pquad,   0.0075f,  36,   18);
         }
         break;

         case  TOWER:
         {
            glRGB(255, 255, 64);
            gluSphere(m_pquad,   0.00075f,  36,   18);
         }
         break;

         case  CHASE_PLANE:
         {
            glRGB(255, 255, 64);
            gluSphere(m_pquad,   0.0075f,  36,   18);
         }
         break;
      }
   }


   double   z_distance  =  ((m_lead_ac_altitude -  m_height_of_terrain)   /  NMI_FT)  *  (m_3d_scale /  3600.0);
   glTranslated(0.0, 0.0, z_distance);
   glRotatef(-m_ac_heading_lead  +  180.0f,  0.0f, 0.0f, 1.0f);
   glRotatef(m_ac_pitch_lead, 1.0f, 0.0f, 0.0f);   // Pitch
   glRotatef(m_ac_roll_lead,  0.0f, 1.0f, 0.0f);   // Roll
   glRotatef(90.0f,  1.0f, 0.0f, 0.0f);

   if (m_3d_map_scale_aircraft)  // Scales with map scale...
   {
      glScaled(1.0   /  140000.0,   1.0   /  140000.0,   1.0   /  140000.0);
      glScaled((m_3d_scale /  3600.0),   (m_3d_scale /  3600.0),   (m_3d_scale /  3600.0));
   }
   else                          // Fixed size... independant of map scale.
   {
      glScaled(1.0   /  2250.0,  1.0   /  2250.0,  1.0   /  2250.0);
   }

   glEnable(GL_CULL_FACE);
   leadModel.Render();
   glDisable(GL_CULL_FACE);

   glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
}


void  C3DMapping::DrawWingAircraft()
{
   glRotated(m_ac_longitude,  0.0,  1.0,  0.0);
   glRotated(-m_ac_latitude,  1.0,  0.0,  0.0);
   glTranslated(0.0, 0.0, 0.95 * m_3d_scale);


   glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);


   if (m_ac_altitude -  m_height_of_terrain  >  1000.0)
   {
      switch   (m_long_camera_position)
      {
         case  TOWER:
         {
            glRGB(255, 128, 255);
            gluSphere(m_pquad,   0.00075f,  36,   18);
         }
         break;

         case  CHASE_PLANE:
         {
            glRGB(255, 128, 255);
            gluSphere(m_pquad,   0.0075f,  36,   18);
         }
         break;
      }
   }

   double   z_distance  =  ((m_ac_altitude   -  m_height_of_terrain) /  NMI_FT)  *  (m_3d_scale /  3600.0);
   glTranslated(0.0, 0.0, z_distance);
   glRotatef(-m_ac_heading_wing  +  180.0f,  0.0f, 0.0f, 1.0f);
   glRotatef(m_ac_pitch_wing, 1.0f, 0.0f, 0.0f);   // Pitch
   glRotatef(m_ac_roll_wing,  0.0f, 1.0f, 0.0f);   // Roll
   glRotatef(90.0f,  1.0f, 0.0f, 0.0f);

   if (m_3d_map_scale_aircraft)  // Scales with map scale...
   {
      glScaled(1.0   /  140000.0,   1.0   /  140000.0,   1.0   /  140000.0);
      glScaled((m_3d_scale /  3600.0), (m_3d_scale /  3600.0), (m_3d_scale /  3600.0));
   }
   else                          // Fixed size... independant of map scale.
   {
      glScaled(1.0   /  2250.0,  1.0   /  2250.0,  1.0   /  2250.0);
   }


   switch   (m_long_camera_position)
   {
      case  TOWER:
      case  CHASE_PLANE:
      {
         glEnable(GL_CULL_FACE);
         wingModel.Render();
         glDisable(GL_CULL_FACE);
      }
      break;
   }

   glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
   glDisable(GL_DEPTH_TEST);
}


void C3DMapping::DrawSymbols(void)
{
   glPushMatrix();
      CAreaMap::DrawSymbols();
   glPopMatrix();
}


void C3DMapping::DrawLeadTrack(void)
{
   if (m_sym_disp_lead_ac_track_on)
   {
      // Aircraft Track information display.
      glPushMatrix();
         m_aircraftLeadTrack.Draw();
      glPopMatrix();

      glPushMatrix();
         DrawLeadAircraft();
      glPopMatrix();
   }
}

void C3DMapping::DrawWingTrack(void)
{
   if (m_sym_disp_wing_ac_track_on)
   {
      // Aircraft Track information display.
      glPushMatrix();
         m_aircraftWingTrack.Draw();
      glPopMatrix();

      glPushMatrix();
         DrawWingAircraft();
      glPopMatrix();
   }
}


void C3DMapping::ChangeValue(const CString& rstrElementVar, CChangeValue* pCV)
{
   if (pCV == NULL)
   {
       return;
   }
   CVariant* pVariant = pCV->Variant();

   if (rstrElementVar      == "3d_map_scale")
   {
      m_3d_scale_cv        =  *pVariant;

      if (m_3d_scale_cv    == 0.0)
         m_3d_scale_cv     =  7200.0;

      m_3d_scale_cv        =  7200.0   /  m_3d_scale_cv;
   }
   else if (rstrElementVar    == "3d_map_spin")
   {
      if ((_FSI_STL::string)*pVariant  == "Right")
         m_spin_angle_cv   += (m_field_of_view  *  0.5);
      else if ((_FSI_STL::string)*pVariant  == "Left")
         m_spin_angle_cv   -= (m_field_of_view  *  0.5);
      else if ((_FSI_STL::string)*pVariant   == "Front")
         m_spin_angle_cv   =  0.0f;
   }
   else if (rstrElementVar    == "3d_map_camera")
   {
      if ((_FSI_STL::string)*pVariant  == "Up")
         m_camera_angle_cv += 10.0f;
      else if ((_FSI_STL::string)*pVariant   == "Down")
         m_camera_angle_cv -= 10.0f;
      else if ((_FSI_STL::string)*pVariant   == "Front")
         m_camera_angle_cv =  0.0f;
   }
   else if (rstrElementVar    == "3d_map_spider_web")
   {
      m_3d_map_spider_web_cv  =  *pVariant;
   }
   else if (rstrElementVar    == "3d_map_scale_aircraft")
   {
      m_3d_map_scale_aircraft_cv =  *pVariant;
   }
   else if (rstrElementVar    == "3d_map_spider_web_center_airport_id")
   {
      CString  airport_id((char *)((_FSI_STL::string)(*pVariant)).c_str());
      CenterSpiderWebOnAirport(airport_id);
   }
   else if (rstrElementVar    == "3d_map_zoom")
   {
      if ((_FSI_STL::string)*pVariant  == "Zoom Out")
      {
         if (7200.0f /  m_3d_scale_cv  <  20.0f)
            m_3d_scale_cv  *= 0.5f;

         if (m_zoom_ratio  >  0.5f)
            m_zoom_ratio   *= 0.5f;
      }
      else if ((_FSI_STL::string)*pVariant   == "Zoom In")
      {
         m_3d_scale_cv  *= 2.0f;
         m_zoom_ratio   *= 2.0f;
      }
   }
   else if (rstrElementVar    == "ac_pitch_lead")
   {
      m_ac_pitch_lead_cv      =  -(float)(*pVariant);
   }
   else if (rstrElementVar    == "ac_roll_lead")
   {
      m_ac_roll_lead_cv       =  -(float)(*pVariant);
   }
   else if (rstrElementVar    == "ac_pitch_wing")
   {
      m_ac_pitch_wing_cv      =  -(float)(*pVariant);
   }
   else if (rstrElementVar    == "ac_roll_wing")
   {
      m_ac_roll_wing_cv       =  -(float)(*pVariant);
   }
   else if (rstrElementVar    == "height_of_terrain")
   {
      m_height_of_terrain_cv  =  *pVariant;
   }
   else if (rstrElementVar    == "Left Landing Gear Green")
   {
      m_bool_left_gear_down_cv   =  *pVariant;
   }
   else if (rstrElementVar    == "Right Landing Gear Green")
   {
      m_bool_right_gear_down_cv  =  *pVariant;
   }
   else if (rstrElementVar    == "Nose Landing Gear Green")
   {
      m_bool_nose_gear_down_cv   =  *pVariant;
   }
   else if (rstrElementVar    == "field_of_view")
   {
      m_field_of_view_cv      =  *pVariant;
      if (m_field_of_view_cv  == 0.0)
         m_field_of_view_cv   =  0.01f;
   }
   else if (rstrElementVar   == "camera_position")
   {
      m_long_camera_position_cv  =  *pVariant;
      m_spin_angle               =  0.0f;
      m_spin_angle_cv            =  0.0f;
   }
   else
   {
      CAreaMap::ChangeValue(rstrElementVar, pCV);
   }
}


bool C3DMapping::UpdateRenderVariables()
{
   m_3d_scale                          =  m_3d_scale_cv;
   m_spin_angle                        =  m_spin_angle_cv;
   m_camera_angle                      =  m_camera_angle_cv;
   m_3d_map_scale_aircraft             =  m_3d_map_scale_aircraft_cv;
   m_field_of_view                     =  m_field_of_view_cv;

   m_ac_pitch_lead                     =  m_ac_pitch_lead_cv;
   m_ac_roll_lead                      =  m_ac_roll_lead_cv;

   m_ac_pitch_wing                     =  m_ac_pitch_wing_cv;
   m_ac_roll_wing                      =  m_ac_roll_wing_cv;

   m_height_of_terrain                 =  m_height_of_terrain_cv;

   m_bool_nose_gear_down               =  m_bool_nose_gear_down_cv;
   m_bool_left_gear_down               =  m_bool_left_gear_down_cv;
   m_bool_right_gear_down              =  m_bool_right_gear_down_cv;

   wingModel.SetGearPosition(NOSE_GEAR,    (float)m_bool_nose_gear_down);
   wingModel.SetGearPosition(LEFT_GEAR,    (float)m_bool_left_gear_down);
   wingModel.SetGearPosition(RIGHT_GEAR,   (float)m_bool_right_gear_down);

   m_ac_altitude                       =  m_ac_altitude_cv;
   m_lead_ac_altitude                  =  m_lead_ac_altitude_cv;


   m_area_map_spider_web_cv            =  m_3d_map_spider_web_cv;

   m_touch_reposition_requested        =  false;
   m_touch_storm_location_requested    =  false;
   m_spider_web_at_cursor_requested    =  false;
   m_map_center_at_cursor_requested    =  false;

   m_sym_disp_ils_on_cv                =  true;
   m_sym_disp_vor_on_cv                =  true;
   m_sym_disp_markers_on_cv            =  true;
   m_sym_disp_ndb_on_cv                =  true;
   m_sym_disp_victor_routes_on_cv      =  false;
   m_sym_disp_jet_routes_on_cv         =  false;
   m_sym_disp_waypoints_on_cv          =  true;
   m_sym_disp_airports_on_cv           =  true;
   m_sym_disp_ils_text_on_cv           =  false;
   m_sym_disp_vor_text_on_cv           =  false;
   m_sym_disp_ndb_text_on_cv           =  false;
   m_sym_disp_lat_long_grid_on_cv      =  false;
   m_sym_disp_mouse_lat_long_on_cv     =  false;
   m_sym_disp_lead_ac_track_on_cv      =  true;
   m_sym_disp_wing_ac_track_on_cv      =  true;



   float delta_altitude =  m_lead_ac_altitude   -  m_ac_altitude;
   m_CStr_delta_altitude.Format( "WING A/C IS\n%.0f FT %s LEAD",  fabs(delta_altitude),   (delta_altitude   >  0) ?  "BELOW"  :  "ABOVE");

   m_long_camera_position              =  m_long_camera_position_cv;


   switch   (m_long_camera_position)
   {
      case  STUDENT:
      {
         m_spin_angle                     += m_ac_heading_wing_cv;
         m_3d_scale                       =  7200.0   /  20.0;
         m_area_map_latitude_center_cv    =  m_ac_latitude_cv;
         m_area_map_longitude_center_cv   =  -m_ac_longitude_cv;
         m_camera_altitude                =  ((m_ac_altitude      -  m_height_of_terrain) /  NMI_FT)  *  (m_3d_scale /  3600.0);
         m_tilt_angle                     =  -90.0f;
         m_camera_angle                   -= m_ac_pitch_wing;

         if (m_zoom_ratio  >= 1.0f)
            m_CStr_scale.Format("ZOOM\n%.0f X", m_zoom_ratio); // Show zoom factor.
         else
            m_CStr_scale.Format("ZOOM\n%.1f X", m_zoom_ratio); // Show zoom factor.
      }
      break;

      case  TOWER:
      {
         GetAirportLatLon(m_active_airport_id_cv, m_active_airport_latitude, m_active_airport_longitude);

         // The xgas term is the north-south distance from the aircraft to the center of the map, in nautical miles.  
         double   xgas  =  (m_active_airport_latitude    -  m_ac_latitude_cv)    *  NM_PER_DEGREE;
         // The ygas term is the east-west distance from the aircraft to the center of the map, in nautical miles.  
         double   ygas  =  (m_active_airport_longitude   -  m_ac_longitude_cv)   *  NM_PER_DEGREE  *  cos(((m_active_airport_latitude  +  m_ac_latitude_cv) /  2.0)  *  DEG_TO_RAD);

         m_spin_angle                     =  atan2(-ygas, -xgas) *  RAD_TO_DEG;
         m_spin_angle_cv                  =  m_spin_angle;
         m_3d_scale                       =  7200.0   /  20.0;
         m_area_map_latitude_center_cv    =  m_active_airport_latitude;
         m_area_map_longitude_center_cv   =  -m_active_airport_longitude;
         m_camera_altitude                =  (250.0   /  NMI_FT)  *  (m_3d_scale /  3600.0);
         m_tilt_angle                     =  -90.0f;

         float ac_to_map_center_distance  =  sqrt(xgas   *  xgas  +  ygas  *  ygas) *  NMI_FT;
         float elevation_offset           =  (m_ac_altitude -  m_height_of_terrain  -  250.0);
         m_camera_angle                   =  atan2(elevation_offset, ac_to_map_center_distance) *  RAD_TO_DEG;
         m_tilt_angle                     -= m_camera_angle;

         if (m_zoom_ratio  >= 1.0f)
            m_CStr_scale.Format("ZOOM\n%.0f X", m_zoom_ratio); // Show zoom factor.
         else
            m_CStr_scale.Format("ZOOM\n%.1f X", m_zoom_ratio); // Show zoom factor.
      }
      break;

      case  CHASE_PLANE:
      {
         m_spin_angle                     =  0.0f;
         m_spin_angle_cv                  =  0.0f;
         m_3d_scale                       =  m_3d_scale_cv;
         m_area_map_latitude_center_cv    =  m_ac_latitude_cv;
         m_area_map_longitude_center_cv   =  -m_ac_longitude_cv;
         m_camera_altitude                =  ((m_ac_altitude      -  m_height_of_terrain) /  NMI_FT)  *  (m_3d_scale /  3600.0);
         m_tilt_angle                     =  -90.0f;
         m_camera_angle                   =  0.0f;
         m_zoom_ratio                     =  1.0f;

         float scale =  7200.0f  /  m_3d_scale;                         // Gets us back to NM.
         if (scale   <  1.0f)
            m_CStr_scale.Format("SCALE\n%.0f FT", (scale *  NMI_FT));   // Show scale in FT.
         else
            m_CStr_scale.Format("SCALE\n%.1f NM", scale);               // Show scale in NM.
      }
      break;
   }


   if (m_camera_angle)
      m_CStr_camera_angle.Format("LOOKING\n%.0f%c %s",   fabs(m_camera_angle), DEGREES, (m_camera_angle > 0)   ?  "UP"  :  "DOWN");
   else
      m_CStr_camera_angle.Format("LOOKING\nFORWARD");


   if (m_spin_angle_cv)
      m_CStr_spin_angle.Format("LOOKING\n%.0f%c %s",     fabs(m_spin_angle_cv), DEGREES, (m_spin_angle_cv > 0)   ?  "RIGHT"  :  "LEFT");
   else
      m_CStr_spin_angle.Format("LOOKING\nFORWARD");


   m_scale_cv  =  m_3d_scale;


   m_CStr_camera_altitude.Format("CAMERA ALTITUDE\n%.0f FT",  (m_camera_altitude   *  NMI_FT)  /  (m_3d_scale /  3600.0));


   m_aircraftLeadTrack.UseAltitude(true);
   m_aircraftWingTrack.UseAltitude(true);
   bool  bRetVal  =  CAreaMap::UpdateRenderVariables();
   m_aircraftLeadTrack.UseAltitude(false);
   m_aircraftWingTrack.UseAltitude(false);

   return   bRetVal;
}
