#include "..\core\stdafx.h"
#include "PlugInManager.h"
#include <io.h>


//Method:      CPlugInManager
//Purpose:     Default Constructor
//Parameters:  NONE
//Returns:     NONE
CPlugInManager::CPlugInManager()
{
   m_PlugIns.clear();
}


//Method:      CPlugInManager
//Purpose:     Destructor
//Parameters:  NONE
//Returns:     NONE
CPlugInManager::~CPlugInManager()
{
   UnloadAll();
}


//Method:      GetInstance
//Purpose:     Called to get a reference to the only instance of this class.
//Parameters:  NONE
//Returns:     Reference to the only instance of this class.
CPlugInManager& CPlugInManager::GetInstance()
{
   static CPlugInManager PIM;
   return PIM;
}


//Method:      LoadPlugins
//Purpose:     Called by the application to load all plugins into memory at once.
//Parameters:  NONE
//Returns:     NONE
void CPlugInManager::LoadPlugins()
{
   if (m_PlugIns.size() == 0)
   {
      LoadPlugin("ImageTypePdf.dll");
      LoadPlugin("ImageTypeTiff.dll");
   }
}


//Method:      LoadPlugin
//Purpose:     Called by LoadPlugins...used to help load an individual plugin.
//Parameters:  strFilename - path and filename to the plugin (DLL) to load.
//Returns:     True if the plugin was loaded into memory.
bool CPlugInManager::LoadPlugin(const std::string& strFilename)
{
   //Load the DLL into memory.
   HMODULE hDll = ::LoadLibrary (strFilename.c_str());
   DWORD err = GetLastError();
    if ( hDll == NULL )
      return false;//Error

   //Query the DLL to determine what interfaces and implementations it supports.
   PlugInInfo* pPlugInInfo = new PlugInInfo();
   {
      //Query to see if the DLL exports the required GetInterfaceImplementations function.
      bool (*pFunction)(char[64],char[64]);
      pFunction = (bool (*)(char[64],char[64]))::GetProcAddress(hDll,"GetInterfaceImplementations");
      if(pFunction == NULL)
      {
         ::FreeLibrary(hDll);
         delete pPlugInInfo;
         return false;
      }

      //Each call to GetInterfaceImplementations should return a different pair of Interface/Implementation
      //strings.  Function should return true while there is more pairs to be fetched.
      char csInterface[64], csImplementation[64];
      while(pFunction(csInterface,csImplementation))
      {  
         InterfaceInfo* pInterfaceInfo = new InterfaceInfo();
         pInterfaceInfo->strInterface = csInterface;
         pInterfaceInfo->strImplementation = csImplementation;
         pPlugInInfo->InterfaceInfoArray.push_back(pInterfaceInfo);
      }
   }

   //Save the pointer and DLL handle for use later.
   pPlugInInfo->hDll = hDll;
   m_PlugIns.push_back(pPlugInInfo);

   return true;
}


//Method:      UnloadAll
//Purpose:     Called by the application to unload all plugins from memory.
//Parameters:  NONE
//Returns:     NONE
void CPlugInManager::UnloadAll()
{
   std::vector<PlugInInfo*>::iterator a;
   for(a = m_PlugIns.begin();a!=m_PlugIns.end();++a)
   {
      PlugInInfo* pPlugInInfo = *a;
      ::FreeLibrary(pPlugInInfo->hDll);

      std::vector<InterfaceInfo*>::iterator b;
      for(b=pPlugInInfo->InterfaceInfoArray.begin();b!=pPlugInInfo->InterfaceInfoArray.end();++b)
      {
         InterfaceInfo* pInterfaceInfo = *b;
         delete pInterfaceInfo;
      }
      pPlugInInfo->InterfaceInfoArray.clear();
      delete pPlugInInfo;
   }
   m_PlugIns.clear();
}


//Method:      FindModule
//Purpose:     Called to find a module that supports a specified interface and instance name.
//Parameters:  strInterface - Name of the supported interface (e.g. IUserFuncPlugIn_01, IiosObjPlugIn_01, etc...)
//Returns:     HMODULE - handle to the DLL that supports the specified Interface and Implementation.
//          HMODULE is NULL if none is found.
HMODULE CPlugInManager::FindModule(std::string& strInterface)
{
   //For each plugin loaded, check to see if it supports the passed function name.
   for (m_itFindModule=m_PlugIns.begin(); m_itFindModule!=m_PlugIns.end(); ++m_itFindModule)
   {
      PlugInInfo& info = **m_itFindModule;
      std::vector<InterfaceInfo*>::iterator b;
      for(b=info.InterfaceInfoArray.begin();b!=info.InterfaceInfoArray.end();++b)
      {
         InterfaceInfo& info2 = **b;
         if(info2.strInterface == strInterface)
         {
            ++m_itFindModule;
            return info.hDll;
         }
      }
   }
   return NULL;
}


//Method:      FindNextModule
//Purpose:     Called to find a module that supports a specified interface and instance name.
//Parameters:  csInterface - Name of the supported interface (e.g. IUserFuncPlugIn_01, IiosObjPlugIn_01, etc...)
//Returns:     HMODULE - handle to the DLL that supports the specified Interface and Implementation.
//          HMODULE is NULL if none is found.
HMODULE CPlugInManager::FindNextModule(std::string& strInterface)
{
   //For each plugin loaded, check to see if it supports the passed function name.
   std::vector<PlugInInfo*>::iterator it;
   for (it=m_itFindModule; it!=m_PlugIns.end(); ++it)
   {
      PlugInInfo& info = **it;
      std::vector<InterfaceInfo*>::iterator b;
      for(b=info.InterfaceInfoArray.begin();b!=info.InterfaceInfoArray.end();++b)
      {
         InterfaceInfo& info2 = **b;
         if(info2.strInterface == strInterface)
         {
            ++m_itFindModule;    
            return info.hDll;
         }
      }
   }
   return NULL;
}

