// GeoChart.cpp : implementation file
//

#include <direct.h>
#include "..\core\stdafx.h"
#include "GeoChart.h"

#include "..\core\dib.h"
#include "..\core\LinkPage.h"
#include "..\comms\CommsShared.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

/////////////////////////////////////////////////////////////////////////////
// CGeoChart

CGeoChart::CGeoChart():
 m_rChart(0.0, 0.0, 0.0, 0.0)
,m_fZoomFactor(1.0f)
,m_fZoomInLimit(1.0f)
,m_fZoomOutLimit(12.8f)
,m_fZoomStep(0.8f)
,m_fZoomSliderStep(0.12f)
,m_ptMoveStart(0,0)
,m_ptPanOffset(0,0)
,m_CursorPoint(0,0)
,m_bLeftMouseDown(false)
,m_bMouseOverWindow(false)
,m_dWingShipLatitude(0.0)
,m_dWingShipLongitude(0.0)
,m_fWingShipHeading(0.0f)
,m_ptWingShipCenter(0,0)
,m_ptLastPassWingShipCenter(0,0)
,m_dLeadShipLatitude(0.0)
,m_dLeadShipLongitude(0.0)
,m_fLeadShipHeading(0.0f)
,m_ptLeadShipCenter(0,0)
,m_ptLastPassLeadShipCenter(0,0)
,m_nAreaIndex(0)
,m_nChartIndex(0)
,m_chart_center_at_aircraft_requested(false)
,m_hChartInfoDll(NULL)
,m_map_rotation(0.0)
,m_bClearLead(true)
,m_bClearWing(true)
,m_bShowLeadTrack(true)
,m_bShowWingTrack(true)
,m_bDeleting(false)
,m_ChartChange(false)
,m_pGeoChartDataManager(NULL)
,m_MapToScale(true)
,m_Initial(false)
,m_winRect(0,0,0,0)
,m_MapOrigin(0,0)
,m_ScreenOrigin(0,0)
,m_DefaultDiameter(50.0f)
,m_ChartFitToScreenAreaIndex(5)
,m_pChartBitmap(NULL)
,m_pChartDC(NULL)
{
   QueryPerformanceCounter ( &t_last_pass );

   m_listWidgetPropPages.push_back(CLinkPage::CreateObject);

   m_listGraphicalElementVars.clear();
   m_listGraphicalElementVars.push_back("Wing Ship Latitude");
   m_listGraphicalElementVars.push_back("Wing Ship Longitude");
   m_listGraphicalElementVars.push_back("Wing Ship Heading");
   m_listGraphicalElementVars.push_back("Lead Ship Latitude");
   m_listGraphicalElementVars.push_back("Lead Ship Longitude");
   m_listGraphicalElementVars.push_back("Lead Ship Heading");
   m_listGraphicalElementVars.push_back("Chart Area");
   m_listGraphicalElementVars.push_back("Chart List");
   m_listGraphicalElementVars.push_back("Chart Center Aircraft");
   m_listGraphicalElementVars.push_back("Chart Center Aircraft State");
   m_listGraphicalElementVars.push_back("Chart Center At Lat Lon");
   m_listGraphicalElementVars.push_back("Chart Center Latitude");
   m_listGraphicalElementVars.push_back("Chart Center Longitude");
   m_listGraphicalElementVars.push_back("Chart Zoom Level");
   m_listGraphicalElementVars.push_back("Chart Scale");
   m_listGraphicalElementVars.push_back("Chart Map Name");
   m_listGraphicalElementVars.push_back("Lead Track Erase");
   m_listGraphicalElementVars.push_back("Wing Track Erase");
   m_listGraphicalElementVars.push_back("Lead Track Show");
   m_listGraphicalElementVars.push_back("Wing Track Show");
   m_listGraphicalElementVars.push_back("Chart Change");
   m_listGraphicalElementVars.push_back("Chart Change State");
   m_listGraphicalElementVars.push_back("Sortie Load Count");

    // Must be set once for InitReinit() to work correctly.
   m_pDib          = NULL;
   CGeoChart::InitReinit();
}

void CGeoChart::InitReinit()
{
   if (m_pDib != NULL)
   {
      m_pDib->DeleteObject();
      delete m_pDib;
   }

   m_pDib              = NULL;
   m_strImage          = "Untitled.bmp";
   m_strComponent      = "IOS";
   m_strSubComponent   = "PAGES";
}

CGeoChart::~CGeoChart()
{
   m_bDeleting = true;

   if (m_pDib != NULL)
   {
      m_pDib->DeleteObject();
      delete m_pDib;
   }

   if (m_pGeoChartDataManager)
   {
      HANDLE temp_mutex = m_pGeoChartDataManager->m_mutex;

      // Before we set m_pGeoChartDataManager to NULL, wait until everyone else is done using it.
      WaitForSingleObject(temp_mutex, INFINITE /*ms*/);
      m_pGeoChartDataManager  =  NULL;
      ReleaseMutex(temp_mutex);
   }

	if (m_pChartBitmap)
		m_pChartBitmap->DeleteObject();
	m_pChartBitmap =  NULL;

   if (m_pChartDC)
      m_pChartDC->DeleteDC();
   m_pChartDC  =  NULL;
}


BEGIN_MESSAGE_MAP(CGeoChart, CButtonWidget)
   //{{AFX_MSG_MAP(CGeoChart)
   ON_WM_LBUTTONDOWN()
   ON_WM_LBUTTONUP()
   ON_WM_MOUSEWHEEL()
   ON_WM_MOUSEMOVE()
   //}}AFX_MSG_MAP
   ON_MESSAGE(WM_MOUSELEAVE,OnMouseLeave)
   ON_MESSAGE(WM_MOUSEHOVER,OnMouseHover)
   ON_MESSAGE(WM_KILLFOCUS, OnKillFocus)
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CGeoChart message handlers
void CGeoChart::Initialize(CXMLWidget*& rpXMLWidget, CWnd* pWnd, 
                               const long lId, bool bEditMode)
{
   CWidget::Initialize(rpXMLWidget,pWnd, lId, bEditMode);

   CGeoChart::InitReinit();

   if (rpXMLWidget != NULL)
   {
      ResetProperties();
   }

   if (GetSafeHwnd() == NULL)
   {
     Create(NULL,
         WS_CHILD | WS_VISIBLE  | BS_OWNERDRAW | CS_OWNDC, 
         CRect(m_exPtUpperLeft, m_exPtLowerRight),
         (CWnd*)pWnd, lId);
   }
   else
   {
     SetParent(pWnd);
     MoveWindow(m_exPtUpperLeft.X(), m_exPtUpperLeft.Y(),
                m_exPtLowerRight.X() - m_exPtUpperLeft.X(),
                m_exPtLowerRight.Y() - m_exPtUpperLeft.Y(), FALSE);
     ShowWindow(SW_NORMAL);
   }

   GetClientRect(&m_winRect);    // Several functions make use of m_winRect 

   CDC* pDCur = GetDC();
   if (m_pDC->GetSafeHdc() != NULL)
   {
      m_pDC->DeleteDC();
   }
   m_pDC->CreateCompatibleDC(pDCur);

   if (m_pBMP->GetSafeHandle() != NULL)
   {
      m_pBMP->DeleteObject();
   }
   m_pBMP->CreateCompatibleBitmap(pDCur, m_exPtLowerRight.X() - m_exPtUpperLeft.X(), m_exPtLowerRight.Y() - m_exPtUpperLeft.Y());
   m_pChartDC     =  m_pDC;
   m_pChartBitmap =  m_pBMP;
   ReleaseDC(pDCur);


   m_pWnd = this;
   m_mapValidAddresses[this] = m_pWnd->GetSafeHwnd();

   m_aircraftLeadTrack.UseAltitude(false);
   m_aircraftLeadTrack.TypeOfTrack(GeoChartLead);
   m_aircraftLeadTrack.SetColor(0, 255, 255);

   m_aircraftWingTrack.UseAltitude(false);
   m_aircraftWingTrack.TypeOfTrack(GeoChartWing);
   m_aircraftWingTrack.SetColor(255, 0, 255);
   
   CString  path("IOS^IMAGES^");
   CWidget::ExpandPath(path);

   m_bmpWingShipOutOfBounds.Load(path  + "WingShipOutOfBounds.bmp");
   m_bmpWingShipInBounds.Load(path     + "WingShipInBounds.bmp");
   m_bmpLeadShipOutOfBounds.Load(path  + "LeadShipOutOfBounds.bmp");
   m_bmpLeadShipInBounds.Load(path     + "LeadShipInBounds.bmp");

   char  *dllPath =  getenv("GEOCHART");
   std::string strChartInfoDLL(dllPath);
   strChartInfoDLL   += "\\bin";
   char  pwd[MAX_PATH];                
   _getcwd(pwd, sizeof(pwd));                // save off previous working directory
   _chdir(strChartInfoDLL.c_str());          // change directories so all dependencies can be found
   strChartInfoDLL   += "\\ChartInfo.dll";
   m_hChartInfoDll = LoadLibrary(strChartInfoDLL.c_str());

   if (m_hChartInfoDll != NULL)
   {
      //DLL is loaded, get a pointer to the CreateChartInfo function.
      IChartInfoV2* (*pCreateChartInfoV2)();
      pCreateChartInfoV2 = (IChartInfoV2* (*)())::GetProcAddress(m_hChartInfoDll,"CreateChartInfoV2");
      if (pCreateChartInfoV2 != NULL)
      {
         //Create an instance of the chart info.
         m_pGeoChartDataManager->m_pChartInfo = pCreateChartInfoV2();
      }
   }
   //Load all ImageType Plugins for use in rendering the charts.
   m_ImageTypeManager.LoadPlugins();
   // change back to previous working directory
   _chdir(pwd);


   CString  ChartInfoPath("IOS^CHARTINFO^");
   CWidget::ExpandPath(ChartInfoPath);
   m_pGeoChartDataManager->SetChartPath(ChartInfoPath);
   std::string strChartInfoFile(ChartInfoPath + "ChartInfo.xml");

   m_vecAreas.clear();

   if (m_pGeoChartDataManager->Read(strChartInfoFile.c_str()))
   {
      m_bChartInfoIsLoaded = true;

      int nAreas(-1);
      m_pGeoChartDataManager->GetNumberOfAreas(nAreas);
      for (int a(0); a < nAreas;++a)
      {
         std::string strTemp(m_pGeoChartDataManager->GetAreaName(a));
         m_vecAreas.push_back(strTemp);
      }

      // Push these Areas into the GeoChartDataManager so the combo box for selecting one can be updated.
      if (m_pGeoChartDataManager)
      {
         m_pGeoChartDataManager->m_area_filtered.clear();
         m_pGeoChartDataManager->m_number_of_areas =  nAreas;

         _FSI_STL::vector<_FSI_STL::string>::iterator   lIt   =  m_vecAreas.begin();
         int   count(0);
         {
            _FSI_STL::list<_FSI_STL::string> list;
            list.push_back("Sortie - Referenced");
            m_pGeoChartDataManager->m_list_of_areas[count++]   =  list;
            m_pGeoChartDataManager->m_area_filtered.push_back(false);
         }
         {
            _FSI_STL::list<_FSI_STL::string> list;
            list.push_back("Nearby");
            m_pGeoChartDataManager->m_list_of_areas[count++]   =  list;
            m_pGeoChartDataManager->m_area_filtered.push_back(false);
         }
         for (; lIt < m_vecAreas.end(); lIt++)
         {
            _FSI_STL::list<_FSI_STL::string> list;
            std::string areaName((*lIt));
            int   position =  areaName.find("[filter]");
            if (position   != std::string::npos)
            {
               areaName =  areaName.substr(0, position-1);
               m_pGeoChartDataManager->m_area_filtered.push_back(true);
            }
            else
               m_pGeoChartDataManager->m_area_filtered.push_back(false);
            list.push_back(areaName.c_str());
            m_pGeoChartDataManager->m_list_of_areas[count++]   =  list;
         }
      }
   }
}


void CGeoChart::DrawItem(LPDRAWITEMSTRUCT lpDrawItemStruct) 
{
   CRect rect(&(lpDrawItemStruct->rcItem));
   CDC *pDC = CDC::FromHandle(lpDrawItemStruct->hDC);

   pDC->SetBkMode(TRANSPARENT);

   if (m_pDib != NULL && m_bChartLoaded == true)
   {
      //Coordinates in which we will draw.
      CRect& rWindow = (CRect)&(lpDrawItemStruct->rcItem);

      //Create an off screen device context in which to render.
      HDC   hMemoryDC(NULL);
      hMemoryDC = CreateCompatibleDC(lpDrawItemStruct->hDC);
      if(hMemoryDC == NULL)
         return;

      //Create a bitmap in which to render.
      HBITMAP  hBitmap(NULL);
      hBitmap = CreateCompatibleBitmap(lpDrawItemStruct->hDC, rWindow.Width(), rWindow.Height());
      if(hBitmap == NULL)
         return;
      HGDIOBJ hOldBitmap = SelectObject(hMemoryDC, hBitmap);

      // Get the image created by PrepareChart and blit it into the current memory DC we are preparing
      {
         BOOL valid = BitBlt(hMemoryDC,       // handle to destination device context
               0,                // x-coordinate of upper-left corner of dest. rectangle
               0,                // y-coordinate of upper-left corner of dest. rectangle
               rWindow.Width(),  // width of destination rectangle
               rWindow.Height(), // height of destination rectangle
               m_pChartDC->GetSafeHdc(),    // handle to source device context
               0,                // x-coordinate of upper-left corner of source rectangle
               0,                // y-coordinate of upper-left corner of source rectangle
               SRCCOPY           // raster operation code
              );
      }

      if (m_MapToScale)
      {
         // Draw the track.
         if (m_bShowLeadTrack)
            DrawTrack(hMemoryDC, m_aircraftLeadTrack);
         if (m_bShowWingTrack)
            DrawTrack(hMemoryDC, m_aircraftWingTrack);

         // Get the Aircraft position back in Map Cooridinates so we can see if it is within the overall Chart area (m_rChart extents).
         CPoint   testPt(0,0);
         testPt.x  =  ScreenToMap((int)m_MapOrigin.x, (int)m_ScreenOrigin.x, (int)m_ptWingShipCenter.x, 1.0f/m_fZoomFactor);
         testPt.y  =  ScreenToMap((int)m_MapOrigin.y, (int)m_ScreenOrigin.y, (int)m_ptWingShipCenter.y, 1.0f/m_fZoomFactor);
         // Draw the WingShip
         if (m_rChart.PtInRect(testPt))
         {
            m_bmpWingShipInBounds.Draw(hMemoryDC, m_ptWingShipCenter, m_fWingShipHeading + m_map_rotation);
            m_chart_center_at_aircraft_state =  true;
         }
         else
         {
            m_chart_center_at_aircraft_state =  false;
            m_bmpWingShipOutOfBounds.Draw(hMemoryDC, m_ptWingShipCenter, m_fWingShipHeading  + m_map_rotation);

            // Setup the pen with which to draw a red crosshatched line to indicate the ownship is in the general direction of a particular side of the chart
            LOGBRUSH lbNormal; 
            lbNormal.lbStyle  =  BS_HATCHED;
            lbNormal.lbColor  =  RGB(255,0,0);
            lbNormal.lbHatch  =  HS_DIAGCROSS; 
            HPEN  hPenNormal  =  ExtCreatePen(PS_GEOMETRIC|PS_SOLID,(DWORD)25, &lbNormal, 0, NULL); 
            SelectObject(hMemoryDC, hPenNormal); 

            if (m_ptWingShipCenter.x   >  m_winRect.right)                                // if the ownship is right of the right edge of the chart
            {
               MoveToEx(hMemoryDC,  m_winRect.right, m_winRect.top,  (LPPOINT)NULL);      // draw a line from the top to bottom of the widget along the right side
               LineTo(hMemoryDC,    m_winRect.right, m_winRect.bottom);                   // to indicate the ownship is in that general direction
            }
            else if (m_ptWingShipCenter.x   <  m_winRect.left)                            // if the ownship is left of the left edge of the chart
            {                                                                             
               MoveToEx(hMemoryDC,  m_winRect.left, m_winRect.top,  (LPPOINT)NULL);       // draw a line from the top to bottom of the widget along the left side
               LineTo(hMemoryDC,    m_winRect.left, m_winRect.bottom);                    // to indicate the ownship is in that general direction
            }

            if (m_ptWingShipCenter.y   >  m_winRect.bottom)                               // if the ownship is below the bottom edge of the chart
            {                                                                             
               MoveToEx(hMemoryDC,  m_winRect.left,  m_winRect.bottom,  (LPPOINT)NULL);   // draw a line from the left to right of the widget along the bottom side
               LineTo(hMemoryDC,    m_winRect.right, m_winRect.bottom);                   // to indicate the ownship is in that general direction
            }                                                                             
            else if (m_ptWingShipCenter.y   <  m_winRect.top)                             // if the ownship is above the top edge of the chart
            {                                                                             
               MoveToEx(hMemoryDC,  m_winRect.left,   m_winRect.top,  (LPPOINT)NULL);     // draw a line from the left to right  of the widget along the top side
               LineTo(hMemoryDC,    m_winRect.right,  m_winRect.top);                     // to indicate the ownship is in that general direction
            }
            DeleteObject(hPenNormal); 
         }

         testPt.x  =  ScreenToMap((int)m_MapOrigin.x, (int)m_ScreenOrigin.x, (int)m_ptLeadShipCenter.x, 1.0f/m_fZoomFactor);
         testPt.y  =  ScreenToMap((int)m_MapOrigin.y, (int)m_ScreenOrigin.y, (int)m_ptLeadShipCenter.y, 1.0f/m_fZoomFactor);
         // Draw the LeadShip
         if (m_rChart.PtInRect(testPt))
            m_bmpLeadShipInBounds.Draw(hMemoryDC, m_ptLeadShipCenter, m_fLeadShipHeading + m_map_rotation);
         else
            m_bmpLeadShipOutOfBounds.Draw(hMemoryDC, m_ptLeadShipCenter, m_fLeadShipHeading  + m_map_rotation);
      }
      else
      {
         {
            RECT& rc = (CRect)&(lpDrawItemStruct->rcItem);
            SetTextAlign(hMemoryDC, TA_CENTER|TA_BASELINE);
            SetTextColor(hMemoryDC, RGB(255,0,0));
            HFONT hFont,hTmp;
            hFont =  CreateFont(40,0,0,0,FW_BOLD,0,0,0,0,0,0,2,0,"SYSTEM_FIXED_FONT");
            hTmp  =  (HFONT)SelectObject(hMemoryDC, hFont);
            CString  m_csMessage("Chart Not To Scale");
            TextOut(hMemoryDC, (rc.left + rc.right) / 2, ((rc.top + rc.bottom) * 0.965), m_csMessage, m_csMessage.GetLength());
            DeleteObject(SelectObject(hMemoryDC, hTmp));
         }
         {
            RECT& rc = (CRect)&(lpDrawItemStruct->rcItem);
            SetTextAlign(hMemoryDC, TA_CENTER|TA_BASELINE);
            SetTextColor(hMemoryDC, RGB(255,0,0));
            HFONT hFont,hTmp;
            hFont =  CreateFont(20,0,0,0,FW_BOLD,0,0,0,0,0,0,2,0,"SYSTEM_FIXED_FONT");
            hTmp  =  (HFONT)SelectObject(hMemoryDC, hFont);
            CString  m_csMessage("Current A/C position not displayed");
            TextOut(hMemoryDC, (rc.left + rc.right) / 2, ((rc.top + rc.bottom) * 0.995), m_csMessage, m_csMessage.GetLength());
            DeleteObject(SelectObject(hMemoryDC, hTmp));
         }
      }
       
      /*
      // This code draws a RED X across the widget thus helping determine the middle of the widget 
      // which is sometimes useful for debugging.
      // Setup the pen with which to draw a big X
      LOGBRUSH lbNormal; 
      HPEN     hPenNormal; 
      lbNormal.lbStyle = BS_SOLID; 
      lbNormal.lbColor = RGB(255,0,0);
      lbNormal.lbHatch = 0; 
      hPenNormal  =  ExtCreatePen(PS_GEOMETRIC|PS_SOLID,(DWORD)3, &lbNormal, 0, NULL); 
      (HPEN)SelectObject(hMemoryDC, hPenNormal); 
      MoveToEx(hMemoryDC, 0, 0,(LPPOINT)NULL);
      LineTo(hMemoryDC, rWindow.Width(), rWindow.Height());
      MoveToEx(hMemoryDC, rWindow.Width(), 0,(LPPOINT)NULL);
      LineTo(hMemoryDC, 0, rWindow.Height());
      DeleteObject(hPenNormal); 
      */

      //Blit everything to the screen DC
      BOOL valid = BitBlt(lpDrawItemStruct->hDC,    // handle to destination device context
            0,                // x-coordinate of upper-left corner of dest. rectangle
            0,                // y-coordinate of upper-left corner of dest. rectangle
            rWindow.Width(),  // width of destination rectangle
            rWindow.Height(), // height of destination rectangle
            hMemoryDC,        // handle to source device context
            0,                // x-coordinate of upper-left corner of source rectangle
            0,                // y-coordinate of upper-left corner of source rectangle
            SRCCOPY           // raster operation code
           );

      //Clean-up.
      SelectObject(hMemoryDC,hOldBitmap);
      DeleteObject(hBitmap);
      DeleteDC(hMemoryDC);
   }
   else
   {
      pDC->Rectangle(rect);
      pDC->DrawText("GeoChart", 8, rect, DT_CENTER | DT_VCENTER | DT_SINGLELINE);
   }

   // Move and invalidate the children for the different states.
   if (m_bLButtonDown != m_bLButtonDownLastPass)
   {
      m_bLButtonDownLastPass = m_bLButtonDown;
      if (m_bTraverseUpdate == false)
      {
         TraverseBaseWidgets((CWidget*)this, true);
      }
   }
   else
   {
      if (m_bTraverseUpdate == false)
      {
         TraverseBaseWidgets((CWidget*)this, false);
      }
   }

   m_bTraverseUpdate = false;
}

void CGeoChart::ResetProperties()
{
    CWidget::ResetProperties();
    CWidget::ExtentsProperties();

    CXMLElement* pXMLElement = NULL;
    POSITION pos = NULL;

    // Find the element.
    if (m_pXMLWidget->FindElement(pXMLElement, pos, _FSI_STL::string("LINK")) == true)
    {
        m_strImage = pXMLElement->ElementValue().c_str();
        STRING2STRING_MAP::iterator s2sIt = NULL;
        if (pXMLElement->FindAttribute(s2sIt,_FSI_STL::string("COMPONENT")) == true)
        {
            m_strComponent = (*s2sIt).second.c_str();
        }
        else
        {
            pXMLElement->AddAttribute(_FSI_STL::string("COMPONENT"),
                                      _FSI_STL::string("IOS"));
        }

        if (pXMLElement->FindAttribute(s2sIt,_FSI_STL::string("SUBCOMPONENT")) == true)
        {
            m_strSubComponent = (*s2sIt).second.c_str();
        }
        else
        {
            pXMLElement->AddAttribute(_FSI_STL::string("SUBCOMPONENT"),
                                      _FSI_STL::string("PAGES"));
        }
    }
    else
    {
        // No element.  Add a new one.
        m_pXMLWidget->AddElement(_FSI_STL::string("LINK"), 
                                 _FSI_STL::string((LPCTSTR)m_strImage),
                                 NULL);
        pXMLElement = NULL;
        if (m_pXMLWidget->FindElement(pXMLElement, pos, _FSI_STL::string("LINK")) == true)
        {
            pXMLElement->AddAttribute(_FSI_STL::string("COMPONENT"),
                                      _FSI_STL::string("IOS"));
            pXMLElement->AddAttribute(_FSI_STL::string("SUBCOMPONENT"),
                                      _FSI_STL::string("PAGES"));
        }
    }

    if (m_pDib != NULL)
    {
        m_pDib->DeleteObject();
        delete m_pDib;
    }
    m_pDib = new CDib;
    CString strFileName(m_strComponent + "^" + m_strSubComponent + "^" + m_strImage);

    ExpandPath(strFileName);
}

void CGeoChart::OnLButtonDown(UINT nFlags, CPoint point) 
{
   if (m_bEditing == true)
   {
      CButtonWidget::OnLButtonDown(nFlags, point);    
   }
   else
   {
      CButtonWidget::OnLButtonDown(nFlags, point);

      NMHDR nmhdr;
      nmhdr.idFrom = GetDlgCtrlID();
      nmhdr.hwndFrom = GetSafeHwnd();
      nmhdr.code = WM_LBUTTONDOWN;
      GetParent()->SendMessage(WM_NOTIFY,GetDlgCtrlID(),(long)&nmhdr);

      m_bLeftMouseDown  =  true;
      m_ptMoveStart.x   =  point.x;
      m_ptMoveStart.y   =  point.y; 

      m_MapOrigin.x  =  ScreenToMap((int)m_MapOrigin.x, (int)m_ScreenOrigin.x, (int)m_ptMoveStart.x, 1.0f/m_fZoomFactor);
      m_MapOrigin.y  =  ScreenToMap((int)m_MapOrigin.y, (int)m_ScreenOrigin.y, (int)m_ptMoveStart.y, 1.0f/m_fZoomFactor);
	  m_ScreenOrigin =  m_ptMoveStart;
	  WriteScreenPositions();

   }
}

void CGeoChart::OnLButtonUp(UINT nFlags, CPoint point) 
{
   if (m_bEditing == true)
   {
   }
   else
   {
      CButtonWidget::OnLButtonUp(nFlags, point);
      CButton::OnLButtonUp(nFlags, point);
      NMHDR nmhdr;
      nmhdr.idFrom = GetDlgCtrlID();
      nmhdr.hwndFrom = GetSafeHwnd();
      nmhdr.code = WM_LBUTTONUP;
      GetParent()->SendMessage(WM_NOTIFY,GetDlgCtrlID(),(long)&nmhdr);
      m_bLeftMouseDown = false;
   }
}


/////////////////////////////////////////////////////////////////////////////
//
// void CGeoChart::ChangeValue()
//
// Inputs           : CString& rstrElementVar - the internal name of a 
//                                              variable that needs to be 
//                                              updated.
//                    CChangeValue* pCV       - pointer to data used to 
//                                              update an internal 
//                                              variable.
//
// Return Values    : None.
//
// Description      : CCB::ChangeValue() is a common framework 
//                    method that is used by the Comms thread to send 
//                    new data to the graphical elements.  Since it is 
//                    called from another thread, the special internal 
//                    variables are used to store the updates.  Before 
//                    the graphical element is redrawn, 
//                    UpdateRenderVariables is called to copy the 
//                    updated values to the values used for rendering.  
//                    This causes more memory to be used but prevents 
//                    the use of synchronization objects that may stall 
//                    the Comms thread.
//
/////////////////////////////////////////////////////////////////////////////
void CGeoChart::ChangeValue(const CString& rstrElementVar, CChangeValue* pCV)
{
   if (pCV == NULL)
   {
      return;
   }
   CVariant* pVariant = pCV->Variant();

   if (rstrElementVar   == "Wing Ship Latitude")
   {
      m_varWingShipLatitude_cv.Value((double)*(pCV->Variant()));
   }
   else if (rstrElementVar   == "Wing Ship Longitude")
   {
      m_varWingShipLongitude_cv.Value((double)*(pCV->Variant()));
   }
   else if (rstrElementVar   == "Wing Ship Heading")
   {
      m_varWingShipHeading_cv.Value((float)*(pCV->Variant()));
   }
   else if (rstrElementVar   == "Lead Ship Latitude")
   {
      m_varLeadShipLatitude_cv.Value((double)*(pCV->Variant()));
   }
   else if (rstrElementVar   == "Lead Ship Longitude")
   {
      m_varLeadShipLongitude_cv.Value((double)*(pCV->Variant()));
   }
   else if (rstrElementVar   == "Lead Ship Heading")
   {
      m_varLeadShipHeading_cv.Value((float)*(pCV->Variant()));
   }
   else if (rstrElementVar    == "Lead Track Show")
   {
      m_bShowLeadTrack  =  *pVariant;
   }
   else if (rstrElementVar    == "Wing Track Show")
   {
      m_bShowWingTrack  =  *pVariant;
   }
   else if (rstrElementVar    == "Lead Track Erase")
   {
      std::string command((_FSI_STL::string)*pVariant);
      if (command == "Erase")
      {
         m_bClearLead   =  true;
      }
   }
   else if (rstrElementVar    == "Wing Track Erase")
   {
      std::string command((_FSI_STL::string)*pVariant);
      if (command == "Erase")
      {
         m_bClearWing   =  true;
      }
   }
   else if (rstrElementVar   == "Chart Area")
   {
      if (!m_pGeoChartDataManager->m_bInitialUpdate)
         m_varChartArea_cv.Value((long)*(pCV->Variant()));
   }
   else if (rstrElementVar   == "Chart List")
   {
      if (!m_pGeoChartDataManager->m_bInitialUpdate)
         m_varChartList_cv.Value((long)*(pCV->Variant()));
   }
   else if (rstrElementVar   == "Chart Center Aircraft")
   {
      m_chart_center_at_aircraft_requested   =  *pVariant;
      m_varChartCenterAtAircraftRequest_cv.Value(m_chart_center_at_aircraft_requested);
      if (m_chart_center_at_aircraft_requested)
      {
         CalculateZoomPanRect();    // Move the chart into proper position and size
         PrepareChart();
      }
   }
   else if (rstrElementVar   == "Chart Center Aircraft State")
   {
      m_chart_center_at_aircraft_state =  *pVariant;
   }
   else if (rstrElementVar   == "Chart Center At Lat Lon")
   {
      //m_chart_center_at_lat_lon_requested =  *pVariant;
   }
   else if (rstrElementVar   == "Chart Center Latitude")
   {
      m_varChartCenterLat_cv  =  *pVariant;
   }
   else if (rstrElementVar   == "Chart Center Longitude")
   {
      m_varChartCenterLon_cv  =  *pVariant;
   }
   else if (rstrElementVar   == "Chart Zoom Level")
   {
      m_varChartZoomLevel_cv  =  *pVariant;
   }
   else if (rstrElementVar   == "Chart Scale")
   {
      m_varChartScale_cv  =  *pVariant;
	  ChangeScale();
   }
   else if (rstrElementVar   == "Chart Map Name")
   {
      m_ChartMapName          =  (char *)((_FSI_STL::string)(*pVariant)).c_str();
   }
   else if (rstrElementVar   == "Chart Change")
   {
      m_ChartChange           =  true;
   }
   else
   {
      CWidget::ChangeValue(rstrElementVar, pCV);
   }
}

// Force update to variables controlling Pub Type and Available dropdowns
// by sending mouse click event
void CGeoChart::UpdateChartSelectionControls()
{
   _FSI_STL::list<CAction*>::iterator lIt = m_listActions.begin();
   for (; lIt != m_listActions.end(); lIt++)
   {
      CIOAction* pIOAction = dynamic_cast<CIOAction*>(*lIt);
      if (pIOAction != NULL)
      {
         CString str(pIOAction->GraphicalElementName().c_str());
         if (str  == "Chart Area")
            pIOAction->OnLButtonUp();
         else if (str == "Chart List")
            pIOAction->OnLButtonUp();
      }
   }
}


/////////////////////////////////////////////////////////////////////////////
//
// bool CGeoChart::UpdateRenderVariables()
//
// Inputs           : None.
//
// Return Values    : true  - all of the render variables were updated.
//                    false - a problem occurred updating the render 
//                            variables.
//
// Description      : CCBWidget::UpdateRenderVariables is a 
//                    common framework method that is called by the 
//                    Comms thread to make sure that all of the 
//                    variables used for rendering have the most up to 
//                    date data.
//
/////////////////////////////////////////////////////////////////////////////
bool CGeoChart::UpdateRenderVariables()
{
   if (m_pGeoChartDataManager == NULL || m_bDeleting)
      return false;

   HANDLE temp_mutex = m_pGeoChartDataManager->m_mutex;

   WaitForSingleObject(temp_mutex, INFINITE);

   if (m_bDeleting)
   {
      ReleaseMutex(temp_mutex);
      return false;
   }

   bool bRetVal = CWidget::UpdateRenderVariables();

   _FSI_STL::list<CAction*>::iterator lIt = m_listActions.begin();
   for (; lIt != m_listActions.end(); lIt++)
   {
      CIOAction* pIOAction = dynamic_cast<CIOAction*>(*lIt);
      if (pIOAction != NULL)
      {
         CString str(pIOAction->GraphicalElementName().c_str());
         if (str  == "Chart Center Aircraft State")
         {
			 UpdateChartCenterAtAircraftState(m_chart_center_at_aircraft_state);
         }
         else if (str == "Chart Center Aircraft")
         {
			UpdateChartCenterAtAircraftRequested(m_chart_center_at_aircraft_requested);
         }
         else if (str == "Chart Center Latitude")
         {
			UpdateChartCenterLatitude();
         }
         else if (str == "Chart Center Longitude")
         {
			UpdateChartCenterLongitude();
         }
      }
   }

   if (bRetVal == true)
   {
      bool  bChartChanged(false);

      m_dWingShipLatitude    =  m_varWingShipLatitude_cv;
      m_dWingShipLongitude   =  m_varWingShipLongitude_cv;
      m_fWingShipHeading     =  m_varWingShipHeading_cv;

	  // Use these 3 lines when running on Dev PC to spoof aircraft position normally sent by Host
	  /*m_dWingShipLatitude    =  30.715192;
      m_dWingShipLongitude   =  -87.032964;
      m_fWingShipHeading     =  47.0;*/

      m_dLeadShipLatitude    =  m_varLeadShipLatitude_cv;
      m_dLeadShipLongitude   =  m_varLeadShipLongitude_cv;
      m_fLeadShipHeading     =  m_varLeadShipHeading_cv;

      if (m_pGeoChartDataManager)
      {
         m_pGeoChartDataManager->m_ac_lat =  m_dWingShipLatitude;
         m_pGeoChartDataManager->m_ac_lon =  m_dWingShipLongitude;

         if (  m_pGeoChartDataManager->m_bInitialUpdate  && 
               m_dWingShipLatitude     != 0.0            && 
               m_dWingShipLongitude    != 0.0
            )
         {
            m_pGeoChartDataManager->CreateLocalChartList();

			// Make sure there are charts in the local list. Don't try to display one
			// if there are none. If there is at least one, select the first one to display
			// as the initial chart.
            if (m_nAreaIndex   =  m_pGeoChartDataManager->m_local_charts.size() > 1)
			{
               m_nAreaIndex   =  m_pGeoChartDataManager->m_local_charts[0].area;
               m_nChartIndex  =  m_pGeoChartDataManager->m_local_charts[0].chart;
               m_varChartArea_cv.Value((long)m_nAreaIndex+2L);
               m_varChartList_cv.Value((long)m_nChartIndex+1L);
               m_varChartArea =  m_varChartArea_cv;   
               m_varChartList =  m_varChartList_cv;
               m_ChartChange  =  false;
			   m_fZoomFactor  =  1.0f;
			   SynchronizeZoom();
               m_varChartCenterLat_cv.Value(m_dWingShipLatitude);
               m_varChartCenterLon_cv.Value(m_dWingShipLongitude);
               UpdateChartSelectionControls();
               LookupChart();
               m_chart_center_at_aircraft_requested   =  true;
               m_varChartCenterAtAircraftRequest_cv.Value(m_chart_center_at_aircraft_requested);
			}
			// If the local charts list is empty, just update the dropdown menus.
			// User can then select from available charts to display. No initial chart
			// will be displayed.
            else
			{
	           m_varChartArea_cv.Value(0L);
               m_varChartList_cv.Value(0L);
               UpdateChartSelectionControls();
			}

            m_pGeoChartDataManager->m_bInitialUpdate  =  false;
         }
         else
         {
            if (m_varChartArea_cv   != m_varChartArea)
            {
               // Don't update m_nAreaIndex here, wait for m_nChartIndex to change first
               m_varChartArea =  m_varChartArea_cv;   
            }

            if (m_varChartList_cv != m_varChartList)
            {
               m_varChartList    =  m_varChartList_cv ;
            }

            if (m_ChartChange)
            {
               m_ChartChange     =  false;
               m_nChartIndex     =  (int)(long)m_varChartList - 1;

               // Special cases:  
               // (m_selected_area == 0) implies that the first list item has been selected.  
               // We placed "Sortie - Referenced" in that position - this triggers a lookup of the charts 
               // that were specified by the loaded sortie.
               //
               // (m_selected_area == 1)  implies that the second list item has been selected.  
               // We placed "Nearby" in that position - this triggers a lookup of the charts that contain 
               // the aircraft's current position.
               //
               if (m_pGeoChartDataManager->m_selected_area  <= 1)
               {
                  if (m_nChartIndex < 0)
                     m_nChartIndex  =  0;
                  else
                  {
                     m_nAreaIndex   =  m_pGeoChartDataManager->m_local_charts[m_nChartIndex].area;
                     m_nChartIndex  =  m_pGeoChartDataManager->m_local_charts[m_nChartIndex].chart;
                     bChartChanged  =  true;
                     LookupChart();
                     if (m_nAreaIndex  == m_ChartFitToScreenAreaIndex)
                        m_chart_center_at_aircraft_requested   =  false;
                     else
                        m_chart_center_at_aircraft_requested   =  true;
                     m_varChartCenterAtAircraftRequest_cv.Value(m_chart_center_at_aircraft_requested);
                  }
               }
               else
               {
                  if (m_nChartIndex < 0)
                     m_nChartIndex  =  0;
                  else
                  {
                     m_nAreaIndex   =  (int)(long)m_varChartArea - 2;
                     bChartChanged  =  true;
                     LookupChart();
                  }
               }

               // Trigger action so that comms variable attached to "Chart Change State" get
               // set back to zero - thus disabling the "Change" button on the GUI.
               _FSI_STL::list<CAction*>::iterator lIt = m_listActions.begin();
               for (; lIt != m_listActions.end(); lIt++)
               {
                  CIOAction* pIOAction = dynamic_cast<CIOAction*>(*lIt);
                  if (pIOAction != NULL)
                  {
                     CString str(pIOAction->GraphicalElementName().c_str());
                     if (str  == "Chart Change State")
                     {
                        // Force a "click", which in turn will cause GetValue() to be called, 
                        // which in turn will update the above variables.
                        pIOAction->OnLButtonUp();
                     }
                  }
               }
            }
         }
      }

      // Determine the point at which the Wing Ship needs to be drawn.
      {
         CTrackPoint ptScreen;
         CTrackPoint ptWorld(m_dWingShipLatitude,m_dWingShipLongitude,0.0);
         bool  valid(GetTrackPoint(ptWorld,ptScreen));
         if (valid)
         {
            m_ptWingShipCenter.x = (int)ptScreen.GetLon();
            m_ptWingShipCenter.y = (int)ptScreen.GetLat();
         }
         else
         {
            char  buf[256];
            sprintf(buf, "UpdateRenderVariables: GetTrackPoint returned false for chart (%d, %d) at (%f, %f) \n", m_nAreaIndex, m_nChartIndex, m_dWingShipLatitude, m_dWingShipLongitude);
            OutputDebugString(buf);
         }
      }

      // Determine the point at which the Lead Ship needs to be drawn.
      {
         CTrackPoint ptScreen;
         CTrackPoint ptWorld(m_dLeadShipLatitude,m_dLeadShipLongitude,0.0);
         bool  valid(GetTrackPoint(ptWorld,ptScreen));
         if (valid)
         {
            m_ptLeadShipCenter.x = (int)ptScreen.GetLon();
            m_ptLeadShipCenter.y = (int)ptScreen.GetLat();
         }
      }

      if (m_aircraftLeadTrack.Hwnd() == NULL)
         m_aircraftLeadTrack.Hwnd(reinterpret_cast<HWND>(1));
      if (m_bClearLead == true && m_aircraftLeadTrack.VerticesAdded() > 0)
         m_aircraftLeadTrack.Clear();
      m_bClearLead   =  false;
      m_aircraftLeadTrack.SetTrackCallback((ITrackCallback *)this);
      m_aircraftLeadTrack.UpdateData();
      PrepTrack(m_aircraftLeadTrack, bChartChanged);

      if (m_aircraftWingTrack.Hwnd() == NULL)
         m_aircraftWingTrack.Hwnd(reinterpret_cast<HWND>(1));
      if (m_bClearWing == true && m_aircraftWingTrack.VerticesAdded() > 0)
         m_aircraftWingTrack.Clear();
      m_bClearWing   =  false;
      m_aircraftWingTrack.SetTrackCallback((ITrackCallback *)this);
      m_aircraftWingTrack.UpdateData();
      PrepTrack(m_aircraftWingTrack, bChartChanged);
   }
   ReleaseMutex(m_pGeoChartDataManager->m_mutex);
   return bRetVal;
}

//////////////////////////////////////////////////////////////////////////////////////
//Method:      CalculateZoomPanRect
//Purpose:     Move the chart into proper position and size
//Parameters:  NONE
//Returns:     NONE  
void CGeoChart::CalculateZoomPanRect()
{
   if (m_pGeoChartDataManager == NULL)
      return;

   CPoint   widgetCenter(m_winRect.CenterPoint());

   bool newChart(ChartChanged());

   // Get the chart's width and height once...
   double   imgW(m_rChart.right  -  m_rChart.left);
   double   imgH(m_rChart.bottom -  m_rChart.top);

   // The following code determines if we are in initial update, if we have changed charts, and the type of chart open.
   // If this is not a new chart or intial update, return to the zoom level we were at when/if we had the geoCharts
   // window open previously. If the aircraft is within the chart, center the chart on the aircraft location, else
   // return to the last location we were zoomed into.

   // If initial update or chart changed and it's an approach plate
   if (m_pGeoChartDataManager->m_bInitialUpdate || (newChart && (m_MapToScale   == false || m_nAreaIndex   == m_ChartFitToScreenAreaIndex)))
   {
	  m_fZoomFactor     =  (float)__max(imgW   /  m_winRect.Width(), imgH   /  m_winRect.Height());

      m_MapOrigin.x     =  (int)(imgW / 2.0);
      m_MapOrigin.y     =  (int)(imgH / 2.0);
      m_ScreenOrigin.x  =  widgetCenter.x;
      m_ScreenOrigin.y  =  widgetCenter.y;
	  m_Initial = true;
   }
   else if (m_MapToScale && m_nAreaIndex != m_ChartFitToScreenAreaIndex) // Not initial update or approach plate
   {
      double   dLat(m_dWingShipLatitude), dLon(m_dWingShipLongitude);
      double   dLatCtr(dLat), dLonCtr(dLon);
      bool     valid(m_pGeoChartDataManager->GetPixelFromLatLon(m_nAreaIndex, m_nChartIndex, dLat, dLon));
      CPoint   testPt((int)dLon, (int)dLat);
      if (valid   && m_rChart.PtInRect(testPt))                         // Aircraft is within this chart
	  {
         double   YCtr     =  dLat;
         double   LatY     =  dLatCtr  -  (m_DefaultDiameter   /  60.0);   // default diameter / NM in a degree
         // Convert lat/lon to unscaled cartesian coordinates
         m_pGeoChartDataManager->GetPixelFromLatLon(m_nAreaIndex, m_nChartIndex, LatY, dLonCtr);
         double   Y2(LatY);
         double   deltaY(Y2 -  YCtr);

		 // If last pass was initial update or we changed charts
		 if (m_Initial || newChart)
		 {
            m_Initial = false;
            m_fZoomFactor     =  (float)__max((deltaY   /  m_winRect.Height()), (deltaY /  m_winRect.Width()));
		 }
		 else
		 {
			 ReadZoomLevel();
		 }

		 // Center chart on aircraft
         m_MapOrigin.x     =  (int)dLon;
         m_MapOrigin.y     =  (int)dLat;
         m_ScreenOrigin.x  =  widgetCenter.x;
         m_ScreenOrigin.y  =  widgetCenter.y;
	  }
      else if (m_Initial || newChart) // Aircraft not within chart. Check for initial update or chart change
	  {
			 m_Initial = false;
		     m_fZoomFactor  =  4.0f;                   // No way to compute the ZoomFactor so this is a guess that looks pretty good

             if (testPt.x   >  (int)imgW)              // if the ownship is right of the current chart
			 {
                m_MapOrigin.x     =  (int)(imgW);      // place the right side of the chart along the right side of the widget
                m_ScreenOrigin.x  =  m_winRect.right;  
			 }
             else if (testPt.x <  0)                   // if the ownship is left of the current chart
			 {
                m_MapOrigin.x     =  0;                // place the left side of the chart along the left side of the widget
                m_ScreenOrigin.x  =  0;
			 }
             else                                      // if the ownship is with the boundaries (left/right) of the current chart
			 {
                m_MapOrigin.x     =  testPt.x;         // place the center (left/right) of the chart at the center (left/right) of the widget
                m_ScreenOrigin.x  =  widgetCenter.x;
			 }

             if (testPt.y   >  (int)imgH)              // if the ownship is below the current chart
			 {                                         
                m_MapOrigin.y     =  (int)(imgH);      // place the bottom side of the chart along the bottom side of the widget
                m_ScreenOrigin.y  =  m_winRect.bottom; 
			 }                                         
             else if (testPt.y <  0)                   // if the ownship is above the current chart
			 {                                         
                m_MapOrigin.y     =  0;                // place the top side of the chart along the top side of the widget
                m_ScreenOrigin.y  =  m_winRect.top;    
			 }                                         
             else                                      // if the ownship is with the boundaries (top/bottom) of the current chart
			 {                                         
                m_MapOrigin.y     =  testPt.y;         // place the center (top/bottom) of the chart at the center (top/bottom) of the widget
                m_ScreenOrigin.y  =  widgetCenter.y;
			 }
	  }
	  else // Aircraft not within chart, not initial update or next pass, and no chart change
	  {
		  ReadZoomLevel();
		  ReadScreenPositions();
	  }
   }
   else // Approach plate, not initial update, and no chart change
   {
	   ReadZoomLevel();
	   {
          double   dLat(m_dWingShipLatitude), dLon(m_dWingShipLongitude);
          bool     valid(m_pGeoChartDataManager->GetPixelFromLatLon(m_nAreaIndex, m_nChartIndex, dLat, dLon));
		  CPoint   testPt((int)dLon, (int)dLat);
          if (valid && m_rChart.PtInRect(testPt)) // Aircraft is within chart
		  {
             m_MapOrigin.x  =  (int)dLon;
             m_MapOrigin.y  =  (int)dLat;
             m_ScreenOrigin.x  =  widgetCenter.x;
             m_ScreenOrigin.y  =  widgetCenter.y;
		  }
	      else // Aircraft is not within chart
		  {
			  ReadScreenPositions();
		  }
	  }
   }

   if (m_chart_center_at_aircraft_requested)
   {
      double   dLat(m_dWingShipLatitude), dLon(m_dWingShipLongitude);
      bool     valid(m_pGeoChartDataManager->GetPixelFromLatLon(m_nAreaIndex, m_nChartIndex, dLat, dLon));
      if (valid)
      {
         m_MapOrigin.x  =  (int)dLon;
         m_MapOrigin.y  =  (int)dLat;
         m_ScreenOrigin.x  =  widgetCenter.x;
         m_ScreenOrigin.y  =  widgetCenter.y;
      }
      m_chart_center_at_aircraft_requested   =  false;
      m_varChartCenterAtAircraftRequest_cv.Value(m_chart_center_at_aircraft_requested);
   }

   // Save all chart parameters so we can return to this zoom level and possibly this location
   // if we close the GeoCharts window then re-open it.
   SynchronizeZoom();
   WriteZoomLevel();
   WriteScreenPositions();
   WriteAreaChartIndexes();

}

//Method:      OnMouseWheel
//Purpose:     Called by the windows message WM_MOUSEWHEEL.  Method is used to 
//             calculate the degree by which we need to zoom (m_fZoomFactor)
//Parameters:  zDelta = (short) HIWORD(wParam);    // wheel rotation
//             CPoint pt                           // x and y position of pointer
//Returns:     
BOOL CGeoChart::OnMouseWheel(UINT nFlags, short zDelta, CPoint pt) 
{
   bool bContinueZoom(true);
   CPoint   tmpPt(pt);
   ScreenToClient(&tmpPt);
   CPoint   tmpMapOrigin(0,0);
   tmpMapOrigin.x =  ScreenToMap((int)m_MapOrigin.x, (int)m_ScreenOrigin.x, (int)tmpPt.x, 1.0f/m_fZoomFactor);
   tmpMapOrigin.y =  ScreenToMap((int)m_MapOrigin.y, (int)m_ScreenOrigin.y, (int)tmpPt.y, 1.0f/m_fZoomFactor);
   if (!m_rChart.PtInRect(tmpMapOrigin))   // Only allow zooming when zoom point is within map area
      return 0;

   m_CursorPoint.x   =  pt.x;
   m_CursorPoint.y   =  pt.y;
   ScreenToClient(&m_CursorPoint);

   m_MapOrigin.x    =  ScreenToMap((int)m_MapOrigin.x, (int)m_ScreenOrigin.x, (int)m_CursorPoint.x, 1.0f/m_fZoomFactor);
   m_MapOrigin.y    =  ScreenToMap((int)m_MapOrigin.y, (int)m_ScreenOrigin.y, (int)m_CursorPoint.y, 1.0f/m_fZoomFactor);
   m_ScreenOrigin   =  m_CursorPoint;
   WriteScreenPositions();

   //***********************************************************************
   //Determine the ZoomFactor that will be used by the StretchBlit function.
   float fStep(m_fZoomStep);
   if (zDelta < 0)//zoom in attempted
   {
      // Range check to make sure we do not zoom too far in.
      if (m_fZoomFactor < m_fZoomInLimit)
	     bContinueZoom = false;
      else
	  {
         m_fZoomFactor *= fStep;
		 WriteZoomLevel();
	     SynchronizeZoom();
	     PrepareChart();
	  }
   }
   else if (zDelta > 0)//zoom out attempted.
   {
      //Range check to make sure we do not zoom too far out.
   
      //Get the size of the scaled chart.
      CRect rChart(0,0,0,0);
      rChart.right  = (int)((float)m_rChart.right  *  (1.0f/m_fZoomFactor));
      rChart.bottom = (int)((float)m_rChart.bottom *  (1.0f/m_fZoomFactor));

      //Check to see if both the height and the width of the window is smaller than
      //the window we are drawing in.  If so, don't allow any more zoom out.
      if (m_winRect.Width()   >  rChart.Width() && m_winRect.Height()   >  rChart.Height())
	     bContinueZoom = false;
      else
      {
         fStep = 1.0f/fStep;     
         m_fZoomFactor *= fStep;
		 WriteZoomLevel();
	     SynchronizeZoom();
	     PrepareChart();
      }
   }

   return 0;
}

//Method:      ChangeScale
// Use this to set zoom factor using zoom slider or edit box
//Returns:     
void CGeoChart::ChangeScale(void) 
{
   m_fZoomFactor = m_fZoomOutLimit - (float(m_varChartScale_cv) * m_fZoomSliderStep);
   WriteZoomLevel();
   PrepareChart();
}

void CGeoChart::OnMouseMove(UINT nFlags, CPoint point) 
{
   
   if (m_bLeftMouseDown)
   {
      // Calculate the offset for the panning change.
      m_CursorPoint.x   =  point.x;
      m_CursorPoint.y   =  point.y;
      m_ptPanOffset.x   =  (m_CursorPoint.x  -  m_ScreenOrigin.x);
      m_ptPanOffset.y   =  (m_CursorPoint.y  -  m_ScreenOrigin.y);

      // Convert edges of chart to scaled screen cooridinates
      int   lEdge =  MapToScreen((int)m_MapOrigin.x, (int)m_ScreenOrigin.x, (int)m_rChart.left,    1.0/m_fZoomFactor);
      int   rEdge =  MapToScreen((int)m_MapOrigin.x, (int)m_ScreenOrigin.x, (int)m_rChart.right,   1.0/m_fZoomFactor);
      int   tEdge =  MapToScreen((int)m_MapOrigin.y, (int)m_ScreenOrigin.y, (int)m_rChart.top,     1.0/m_fZoomFactor);
      int   bEdge =  MapToScreen((int)m_MapOrigin.y, (int)m_ScreenOrigin.y, (int)m_rChart.bottom,  1.0/m_fZoomFactor);

      // Enforce horizontal panning limits
      int   windowCenterX(m_winRect.Width()  /  2);            // Determine X center of window.
      bool  movingLeft(m_ptPanOffset.x <  0);                  // Determine direction of current movement.
      if (movingLeft && rEdge <  windowCenterX)                // If moving left, don't allow right edge to go left of center.
         m_ptPanOffset.x   =  0;                               // Restore rectangle size and position.
      else if (!movingLeft && lEdge >  windowCenterX)          // If moving right, don't allow left edge to go right of center.
         m_ptPanOffset.x   =  0;                               // Restore rectangle size and position.

      // Enforce vertical panning limits (only when panning).  
      int   windowCenterY(m_winRect.Height() /  2);            // Determine Y center of window.
      bool  movingDown(m_ptPanOffset.y >  0);                  // Determine direction of current movement.
      if (movingDown && tEdge >  windowCenterY)                // If moving down, don't allow top edge to go below center.
         m_ptPanOffset.y   =  0;                               // Restore rectangle size and position.
      else if (!movingDown && bEdge <  windowCenterY)          // If moving up, don't allow bottom edge to go above center.
         m_ptPanOffset.y   =  0;                               // Restore rectangle size and position.

      m_ScreenOrigin += m_ptPanOffset;                         // Apply offsets (if any)

      WriteScreenOrigins();

      PrepareChart();
   }

   //Needed to post the WM_MOUSEHOVER and WM_MOUSELEAVE messages.  This will help
   //the window grab focus when the mouse enters the area of the window.
   TRACKMOUSEEVENT   tme;  
   tme.cbSize        =  sizeof(TRACKMOUSEEVENT); 
   tme.dwFlags       =  TME_LEAVE   |  TME_HOVER; 
   tme.dwHoverTime   =  200;
   tme.hwndTrack     =  m_hWnd;  
   TrackMouseEvent(&tme); 
   SetFocus();

}


LRESULT CGeoChart::OnMouseLeave(WPARAM,LPARAM)
{
   m_bLeftMouseDown     =  false;
   m_bMouseOverWindow   =  false;
   ::SetFocus(NULL);
   return 0;
}


LRESULT CGeoChart::OnMouseHover(WPARAM,LPARAM)
{
   if (!m_bMouseOverWindow)
   {
      SetForegroundWindow();
      m_bMouseOverWindow   =  true;
   }
   return 0;
}

LRESULT CGeoChart::OnKillFocus(WPARAM,LPARAM)
{
   m_bLeftMouseDown     =  false;
   m_bMouseOverWindow   =  false;
   return 0;
}


//Function:    GetTrackPoint
//Purpose:     This function determines where on the screen to draw a Lat/Lon point.
//             It basicly tells me the pixel the Lat/Lon point is at.
//Parameters:  ptWorld  - Point that contains the world coordinates in Latitude, Longitude, and Altitude.
//             ptScreen - Point that will receive the screen coordinates (X=Lon, Y=Lat, Z=Alt).
//Returns:     true if returned data is valid, false indicates invalid data
bool CGeoChart::GetTrackPoint(const CTrackPoint ptWorld, CTrackPoint& ptScreen)
{
   bool  valid(false);

   if (m_pGeoChartDataManager == NULL)
      return   false;

   DWORD retcode  =  WaitForSingleObject(m_pGeoChartDataManager->m_mutex, INFINITE);

   if (!m_bDeleting)
   {
      try
      {
         double   dLat(ptWorld.GetLat());
         double   dLon(ptWorld.GetLon());
         valid =  m_pGeoChartDataManager->GetPixelFromLatLon(m_nAreaIndex, m_nChartIndex, dLat, dLon);
         if (valid)
         {
            dLon  =  MapToScreen((double)m_MapOrigin.x, (double)m_ScreenOrigin.x, dLon, 1.0/m_fZoomFactor);
            dLat  =  MapToScreen((double)m_MapOrigin.y, (double)m_ScreenOrigin.y, dLat, 1.0/m_fZoomFactor);
            ptScreen.SetLon(dLon);
            ptScreen.SetLat(dLat);
         }
      }
      catch (...)
      {
      }
   }
   ReleaseMutex(m_pGeoChartDataManager->m_mutex);
   return   valid;
}


bool CGeoChart::GetTrackPointNoOffsets(const CTrackPoint ptWorld, CTrackPoint& ptScreen)
{
   bool  valid(false);
   
   if (m_pGeoChartDataManager == NULL)
      return   false;

   DWORD retcode  =  WaitForSingleObject(m_pGeoChartDataManager->m_mutex, INFINITE);
   if (!m_bDeleting)
   {
      try
      {
         double   dLat(ptWorld.GetLat());
         double   dLon(ptWorld.GetLon());
         valid =  m_pGeoChartDataManager->GetPixelFromLatLon(m_nAreaIndex, m_nChartIndex, dLat, dLon);
         if (valid)
         {
            ptScreen.SetLon(dLon);
            ptScreen.SetLat(dLat);
         }
      }
      catch (...)
      {
      }
   }
   ReleaseMutex(m_pGeoChartDataManager->m_mutex);
   return   valid;
}


bool CGeoChart::LookupChart(void)
{
   if (m_pGeoChartDataManager == NULL)
      return   false;

   int   numberOfCharts(-1);
   m_pGeoChartDataManager->GetNumberOfCharts(m_nAreaIndex, numberOfCharts);

   if (numberOfCharts   >  0)
   {
      // Get the name of the specified chart 
      std::string strFullFilename(m_pGeoChartDataManager->GetChartFile(m_nAreaIndex,m_nChartIndex,true));

      // Load the chart.
      if (m_ImageTypeManager.Open(strFullFilename.c_str()))
      {
         m_pGeoChartDataManager->GetMapRotationAngle(m_nAreaIndex, m_nChartIndex, m_map_rotation);

         GeoReferenceType_T   refType;
         m_pGeoChartDataManager->GetGeoReferenceType(m_nAreaIndex, m_nChartIndex, refType);
         if (refType == None)
            m_MapToScale   =  false;
         else
            m_MapToScale   =  true;

         // Get the chart's width and height once...
         double   imgW(m_ImageTypeManager.GetImageWidth());
         double   imgH(m_ImageTypeManager.GetImageHeight());
         m_rChart.left     =  0.0;
         m_rChart.top      =  0.0;
         m_rChart.right    =  imgW;
         m_rChart.bottom   =  imgH;

         m_bChartLoaded =  true;

         CalculateZoomPanRect();    // Move the chart into proper position and size
         PrepareChart();
      }
      else
      {
         m_bChartLoaded = false;
         char  buf[256];
         sprintf(buf, "LookupChart: m_ImageTypeManager.Open(\"%s\") returned false\n", strFullFilename.c_str());
         OutputDebugString(buf);
      }
   }
   return   m_bChartLoaded;
}


CVariant* CGeoChart::GetValue(const CString &rstrValue)
{
   if (rstrValue == "Chart Center Aircraft State")
   {
      m_varChartCenterAtAircraftState_cv.Value(m_chart_center_at_aircraft_state);
      return   &m_varChartCenterAtAircraftState_cv;
   }
   else if (rstrValue == "Chart Center Aircraft")
   {
      return   &m_varChartCenterAtAircraftRequest_cv;
   }
   else if (rstrValue == "Chart Center At Lat Lon")
   {
      m_varChartCenterAtLatLonRequest_cv.Value(false);
      return   &m_varChartCenterAtLatLonRequest_cv;
   }
   else if (rstrValue   == "Chart Center Latitude")
   {
      return   &m_varChartCenterLat_cv;
   }
   else if (rstrValue   == "Chart Center Longitude")
   {
      return   &m_varChartCenterLon_cv;
   }
   else if (rstrValue   == "Chart Zoom Level")
   {
      return   &m_varChartZoomLevel_cv;
   }
   /*else if (rstrValue   == "Chart Scale")
   {
      return   &m_varChartScale_cv;
   }*/
   else if (rstrValue   == "Chart Change State")
   {
      m_varChartChangeState_cv.Value(0L);    // Only need to set this to 0 thus disabling thee "Change" button.
      return   &m_varChartChangeState_cv;
   }
   else if (m_pGeoChartDataManager->m_bInitialUpdate)
   {
      // Only write these values if we are initializing - this causes the list box to reflect initial chart selection
      if (rstrValue  == "Chart Area")
      {
         return   &m_varChartArea_cv;
      }
      else if (rstrValue  == "Chart List")
      {
         return   &m_varChartList_cv;
      }
   }

   return   NULL;
}


void CGeoChart::PrepTrack(CAircraftTrackRaw &whichTrack, bool bChartChanged)
{
   if (m_pGeoChartDataManager == NULL)
      return;

   DWORD retcode  =  WaitForSingleObject(m_pGeoChartDataManager->m_mutex, INFINITE);

   if (!m_bDeleting)
   {
      try
      {
         int   size           =  whichTrack.VerticesAdded() *  3;
         float *vertices      =  whichTrack.GetVertices();
         TolDataClients type  =  whichTrack.TypeOfTrack();

         std::vector<CPoint>  vecPointsInTrack;
         vecPointsInTrack.clear();

         if (bChartChanged)
            whichTrack.Recalculate();

         for (int jj = 0; jj < size-3; jj+=3)
         {
   	      CTrackPoint ptScreen(vertices[jj], vertices[jj+1], vertices[jj+2]);
            CPoint      point(0,0);
            point.x  =  ptScreen.GetLon();
            point.y  =  ptScreen.GetLat();
            point.x  =  MapToScreen((double)m_MapOrigin.x, (double)m_ScreenOrigin.x, (double)point.x, 1.0/m_fZoomFactor);
            point.y  =  MapToScreen((double)m_MapOrigin.y, (double)m_ScreenOrigin.y, (double)point.y, 1.0/m_fZoomFactor);

            vecPointsInTrack.push_back(point);
         }
         m_mapTrackTypeToPointsInTrack[type] =  vecPointsInTrack;
      }
      catch (...)
      {
      }
   }
   ReleaseMutex(m_pGeoChartDataManager->m_mutex);
}


void CGeoChart::DrawTrack(HDC hDC, CAircraftTrackRaw &whichTrack)
{
   if (m_pGeoChartDataManager == NULL)
      return;

   DWORD retcode  =  WaitForSingleObject(m_pGeoChartDataManager->m_mutex, INFINITE);

   if (!m_bDeleting)
   {
      try
      {
         int   count(0), nSize(0);
         TolDataClients type  =  whichTrack.TypeOfTrack();
         if (m_mapTrackTypeToPointsInTrack.find(type) == m_mapTrackTypeToPointsInTrack.end())
         {
            ReleaseMutex(m_pGeoChartDataManager->m_mutex);
            return;
         }

         std::vector<CPoint>  &vecPointsInTrack =  m_mapTrackTypeToPointsInTrack[type];
         nSize =  vecPointsInTrack.size();   // Number of points stored in memory.

         if (nSize > 0)
         {
            int   index;
            const int   tolerance(10);
            HPEN  hPenOld;
            // Setup a pen with which to draw the (background) track.
            LOGBRUSH lbBackground; 
            HPEN     hPenBackground; 
            lbBackground.lbStyle = BS_SOLID; 
            lbBackground.lbColor = RGB(32, 32, 32);
            lbBackground.lbHatch = 0; 
            hPenBackground =  ExtCreatePen(PS_GEOMETRIC|PS_SOLID,(DWORD)5, &lbBackground, 0, NULL); 
            hPenOld  =  (HPEN)SelectObject(hDC, hPenBackground);
            MoveToEx(hDC, vecPointsInTrack[0].x, vecPointsInTrack[0].y,(LPPOINT)NULL);
            int   oldX(0), oldY(0);
            // Loop through every point and draw the track.
            for (index=1; index < nSize; index++)
            {
               int nX(vecPointsInTrack[index].x);
               int nY(vecPointsInTrack[index].y);
               if (abs(nX - oldX) > tolerance || abs(nY - oldY) > tolerance)
               {
                  LineTo(hDC,nX,nY);
                  oldX  =  nX;
                  oldY  =  nY;
                  count++;
               }
            }
            LineTo(hDC, vecPointsInTrack[nSize-1].x, vecPointsInTrack[nSize-1].y);
            DeleteObject(hPenBackground); 

            // Setup the pen with which to draw the (normal) track.
            LOGBRUSH lbNormal; 
            HPEN     hPenNormal; 
            lbNormal.lbStyle = BS_SOLID; 
            lbNormal.lbColor = RGB(whichTrack.m_red,whichTrack.m_green,whichTrack.m_blue);
            lbNormal.lbHatch = 0; 
            hPenNormal  =  ExtCreatePen(PS_GEOMETRIC|PS_SOLID,(DWORD)3, &lbNormal, 0, NULL); 
            (HPEN)SelectObject(hDC, hPenNormal); 
            MoveToEx(hDC, vecPointsInTrack[0].x, vecPointsInTrack[0].y,(LPPOINT)NULL);
            oldX  =  oldY  =  count =  0;
            // Loop through every point and draw the track.
            for (index=1; index < nSize; index++)
            {
               int nX(vecPointsInTrack[index].x);
               int nY(vecPointsInTrack[index].y);
               if (abs(nX - oldX) > tolerance || abs(nY - oldY) > tolerance)
               {
                  LineTo(hDC,nX,nY);
                  oldX  =  nX;
                  oldY  =  nY;
                  count++;
               }
            }
            LineTo(hDC, vecPointsInTrack[nSize-1].x, vecPointsInTrack[nSize-1].y);
            DeleteObject(hPenNormal); 

            SelectObject(hDC, hPenOld);      // Clean up before leaving.
         }
      }
      catch (...)
      {
      }
   }
   ReleaseMutex(m_pGeoChartDataManager->m_mutex);
}


void CGeoChart::PrepareChart(void)
{
   HDC      hMemoryDC   =  m_pChartDC->GetSafeHdc();
   HGDIOBJ  hOldBitmap  =  m_pChartDC->SelectObject(m_pChartBitmap);

   // Erase the backgound with the selected color before proceeding.
   {
      //Setup the pen with which to draw.
      LOGBRUSH lbNormal; 
      lbNormal.lbStyle = BS_SOLID;
      // Set approach plate pen color to white, otherwise black
      if (m_nAreaIndex == m_ChartFitToScreenAreaIndex)
      {
         lbNormal.lbColor = RGB(255,255,255);
      }
      else
      {
         lbNormal.lbColor = RGB(0,0,0);
      }
      lbNormal.lbHatch = 0; 
      HPEN hPen(NULL); 
      hPen = ExtCreatePen(PS_GEOMETRIC|PS_SOLID,(DWORD)1, &lbNormal, 0, NULL); 
      if(hPen == NULL)
         return;
      HGDIOBJ hOldPen = SelectObject(hMemoryDC, hPen); 

      //Setup a brush with which to draw.
      HBRUSH hBrush(NULL);
      // Set approach plate brush color to white, otherwise black
      if (m_nAreaIndex == m_ChartFitToScreenAreaIndex)
      {
         hBrush = CreateSolidBrush(RGB(255,255,255));
      }
      else
      {
         hBrush = CreateSolidBrush(RGB(0,0,0));
      }
      if(hBrush == NULL)
         return;
      HGDIOBJ hOldBrush = SelectObject(hMemoryDC, hBrush); 

      //Flood fill rectangle (window) with the erase color.
      Rectangle(hMemoryDC, 0, 0, m_winRect.Width(), m_winRect.Height());

      //Clean up....
      SelectObject(hMemoryDC,hOldPen);
      SelectObject(hMemoryDC,hOldBrush);
      DeleteObject(hPen); 
      DeleteObject(hBrush); 
   }

   SetLastError(0);                                // Clear any existing errors
   // Draw the Chart in an offscreen DC in a bitmap
   m_ImageTypeManager.Draw(hMemoryDC, m_MapOrigin.x, m_MapOrigin.y, 1.0f/m_fZoomFactor, m_ScreenOrigin.x, m_winRect.Width()-m_ScreenOrigin.x, m_ScreenOrigin.y, m_winRect.Height()-m_ScreenOrigin.y);

   int   drawError(GetLastError());                // Check to see if the Draw code produced an error.
   if (drawError  == ERROR_NOT_ENOUGH_MEMORY)
   {
      // Put a message on the screen indicating what happened and how we are going to fix it for user...
      RECT& rc = m_winRect;
      Rectangle(hMemoryDC, rc.left, rc.top, rc.right, rc.bottom);
      SetTextAlign(hMemoryDC, TA_CENTER|TA_BASELINE);
      CString  m_csMessage("Exceeded maximum zoom level - returning to previous zoom level!");
      TextOut(hMemoryDC, (rc.left + rc.right) / 2, (rc.top + rc.bottom) / 2, m_csMessage, m_csMessage.GetLength());
      OutputDebugStringA(m_csMessage);

      m_fZoomFactor  /= m_fZoomStep;      // Back out the last zoom that caused the error
	  SynchronizeZoom();
	  WriteZoomLevel();
      CalculateZoomPanRect();             // Move the chart into proper position and size

      Invalidate(FALSE);                  // Force redraw.
      return;
   }
   else if (drawError)
   {
      LPVOID   lpMsgBuf;
      LPVOID   lpDisplayBuf;

      FormatMessage(FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM | FORMAT_MESSAGE_IGNORE_INSERTS,
                     NULL, drawError, MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), (LPTSTR) &lpMsgBuf, 0, NULL);

      // Display the error message
      lpDisplayBuf   =  (LPVOID)LocalAlloc(LMEM_ZEROINIT, (lstrlen((LPCTSTR)lpMsgBuf) + 40) * sizeof(TCHAR)); 
      sprintf((LPTSTR)lpDisplayBuf, "Draw failed with error %d: %s", drawError, lpMsgBuf); 

      // Put a message on the screen indicating what happened...
      RECT& rc = m_winRect;
      Rectangle(hMemoryDC, rc.left, rc.top, rc.right, rc.bottom);
      SetTextAlign(hMemoryDC, TA_CENTER|TA_BASELINE);
      CString  m_csMessage((LPTSTR)lpDisplayBuf);
      TextOut(hMemoryDC, (rc.left + rc.right) / 2, (rc.top + rc.bottom) / 2, m_csMessage, m_csMessage.GetLength());
      OutputDebugStringA(m_csMessage);

      LocalFree(lpMsgBuf);
      LocalFree(lpDisplayBuf);

      Invalidate(FALSE);                  // Force redraw.
      return;
   }

   SelectObject(hMemoryDC, hOldBitmap);
}
