#include "..\core\stdafx.h"
#include <afxwin.h>
#include "AcBitmap.h"
#include "math.h"

//Method:		CAcBitmap
//Purpose:		Default Constructor
//Parameters:	NONE
//Returns:		NONE
CAcBitmap::CAcBitmap()
{
	Initialize();
}


//Method:		~CAcBitmap
//Purpose:		Default Destructor
//Parameters:	NONE
//Returns:		NONE
CAcBitmap::~CAcBitmap()
{}


//Method:		Draw
//Purpose:		Renders to the screen the bitmap of the aircraft if loaded.  If not the default
//				draw renders a red cross to the screen.
//Parameters:	hDC - handle to the device context in which you want to render.
//				ptCenter - X,Y point representing the center of the image.
//				fRotation - Angle by which to rotate the image.  This will normally be heading (1-360).
//Returns:		NONE
void CAcBitmap::Draw(HDC hDC, POINT ptCenter, float fRotation)
{
	int nHalfWidth = m_nWidth/2;
	int nHalfHeight = m_nHeight/2;

	//Update the coordinates of the bitmap based on the current center.
	m_rCoordinates.left   = ptCenter.x - nHalfWidth;
	m_rCoordinates.right  = ptCenter.x + nHalfWidth;
	m_rCoordinates.top    = ptCenter.y - nHalfHeight;
	m_rCoordinates.bottom = ptCenter.y + nHalfHeight;

	//Set the points to the corners of the existing image.
	POINT lpPoint[3];
	lpPoint[0].x = nHalfWidth;
	lpPoint[0].y = nHalfHeight;
	lpPoint[1].x = -nHalfWidth;
	lpPoint[1].y = nHalfHeight;
	lpPoint[2].x = nHalfWidth;
	lpPoint[2].y = -nHalfHeight;
							  
	//Calculate the points for the heading rotation.
	double  dDegToRad = 57.29577951;
	double  dAngleSin = sin(fmod(fRotation - 90, 360) / dDegToRad);
	double  dAngleCos = cos(fmod(fRotation - 90, 360) / dDegToRad);
	for(int nCount = 0; nCount < 3; nCount ++)
	{
		double  dX = ((lpPoint[nCount].x * dAngleSin) + (lpPoint[nCount].y * dAngleCos));
		double  dY = ((lpPoint[nCount].y * dAngleSin) - (lpPoint[nCount].x * dAngleCos));

		lpPoint[nCount].x = (int)dX + ptCenter.x;
		lpPoint[nCount].y = (int)dY + ptCenter.y;
	}
	
	//The draw, rotation, and mask all occurs here with this blt.
	PlgBlt(hDC,lpPoint,m_hDC,0,0,m_nWidth,m_nHeight,m_hBmpMono,0,0);      
}


//Method:		GetCoordinates
//Purpose:		Returns a reference to a rectangle that contains the current coordinates of the bitmap.
//Parameters:	NONE
//Returns:		Reference to a rectangle
RECT CAcBitmap::GetCoordinates()
{
	return m_rCoordinates;
}


//Method:		Initialize
//Purpose:		Resets the memory before loading a bitmap into memory.
//Parameters:	NONE
//Returns:		NONE
void CAcBitmap::Initialize()
{
	m_nWidth = m_nHeight = 0;

	//Delete the bitmap and reset its handle.
	if (m_hBmp)
		DeleteObject(m_hBmp);
	m_hBmp = NULL;

	//Delete the device context and reset its handle.
	if(m_hDC)
		DeleteObject(m_hDC);
	m_hDC = NULL;

	//Now create a device context compatible with the current screen.
	m_hDC = CreateCompatibleDC(NULL);
}


//Method:		Load
//Purpose:		Default Load, ensures that something will draw to the screen in the event a bitmap can
//				not be loaded from the disk.
//Parameters:	NONE
//Returns:		NONE
void CAcBitmap::Load()
{
	//Clear memory in the event this is not the first load.
	Initialize();

	m_nHeight = m_nWidth = 85;//A^2 + B^2 = C^2 where A&B=60
	
	// Populate bitmapinfo header with the information needed to draw a cross.
	BITMAPINFO bmInfo;
	bmInfo.bmiHeader.biSize = sizeof(BITMAPINFOHEADER);
	bmInfo.bmiHeader.biWidth = ((((int) m_nWidth * 8) + 31) & ~31) >> 3;//Pads to a multiple of 4;
	bmInfo.bmiHeader.biHeight = m_nHeight;
	bmInfo.bmiHeader.biPlanes = 1;
	bmInfo.bmiHeader.biBitCount = 32;
	bmInfo.bmiHeader.biCompression = BI_RGB;

	// Create the DIB section.
	m_hBmp = CreateDIBSection(m_hDC,
							  (BITMAPINFO*)&bmInfo,
							  DIB_RGB_COLORS,
							  (VOID **)&m_pBits,
							  0,
							  0);

	// Select the bitmap into the buffer
	if(m_hBmp)
		::SelectObject(m_hDC,m_hBmp);
	else
		return;//ERROR!!!

	//Flood the bitmap with black.
	SetBkColor(m_hDC, RGB(0, 0, 0));
		
	//***************************************************
	//Draw a generic aircraft to the bitmap.
	CPen pen;
	pen.CreatePen(PS_SOLID,4,RGB(0,0,255));
	CDC  dc;
   dc.SetOutputDC(m_hDC);
	dc.SelectObject(&pen);
	
	//Draw the aircraft.
	//Body
	dc.MoveTo(43,13);
	dc.LineTo(43,63);
	
	//Wings
	dc.MoveTo(43,28);
	dc.LineTo(13,48);
	dc.MoveTo(43,28);
	dc.LineTo(73,48);

	//Tail
	dc.MoveTo(43,63);
	dc.LineTo(33,73);
	dc.MoveTo(43,63);
	dc.LineTo(53,73);
	//***************************************************

	//Create a monochrome bitmap for use later in the PlgBlt.
	{
		m_hDcMono = CreateCompatibleDC(m_hDC);

		//Create the compatible mono bitmap.
		m_hBmpMono = (HBITMAP) CreateCompatibleBitmap(m_hDcMono, m_nWidth, m_nHeight);
		SelectObject(m_hDcMono, m_hBmpMono);

		//Blt the aircraft bitmap into the mono bitmap.  The magic is in the NOTSRCCOPY.
		BitBlt(m_hDcMono,0,0,m_nWidth,m_nHeight,m_hDC,0,0,NOTSRCCOPY);
	}
}


//Method:		Load
//Purpose:		Custom Load, allows user to specify a bitmap file to load into memory.
//Parameters:	szFilename - Full path to the file you wish to load.
//Returns:		NONE
void CAcBitmap::Load(const char* szFilename)
{
	//Clear memory in the event this is not the first load.
	Initialize();

	//Get a file pointer to the bitmap.
	FILE* pFile = NULL;
	pFile = fopen(szFilename,"rb");

	//Make sure the file was opened.  If not, perform the default load.
	if(pFile == NULL)
	{
		Load();
		return;
	}

	//Load the bitmap file header into memory. (Used to get the size of the file.)
	BITMAPFILEHEADER bitmapFileHeader;
	fread((void*)&bitmapFileHeader, sizeof(BITMAPFILEHEADER),1,pFile);

	//Get a pointer to the BITMAPINFOHEADER and read it into memory.
	BITMAPINFOHEADER bitmapInfoHeader;
	fread((void*)&bitmapInfoHeader, sizeof(BITMAPINFOHEADER),1,pFile);
	
	//Set the size of the DC for the height and width of the bitmap.  Because the bitmap can be rotated, the
	//area needed for the height and width is greater that the actual height and width of the bitmap.  To determine
	//the area that is needed, we need to calculate the length of the hypotenuse using the height and width of
	//the actual bitmap and use this length to set the height and width of the DC and bitmap.
	double dLengthOfHypotenuse = sqrt((double)((bitmapInfoHeader.biHeight*bitmapInfoHeader.biHeight)+(bitmapInfoHeader.biWidth*bitmapInfoHeader.biWidth)));//A^2 + B^2 = C^2  where C is dLengthOfHypotenuse.	
	m_nHeight = m_nWidth = (int)dLengthOfHypotenuse + 1;

	//Calculate the number of palette entries.
	int nPaletteEntries = 1 << bitmapInfoHeader.biBitCount;
	if(bitmapInfoHeader.biBitCount > 8)
		nPaletteEntries = 0;
	else if(bitmapInfoHeader.biClrUsed !=0)
		nPaletteEntries = bitmapInfoHeader.biClrUsed;

	
	//Read in the Palette information if it exists.
	RGBQUAD* pRGBQuad = NULL;
	LOGPALETTE *pLogPalette = NULL;
	UINT uUsage = DIB_RGB_COLORS;
	if(nPaletteEntries > 0)
	{
		//Initialize the memory for the palette information.
		pRGBQuad = new RGBQUAD[nPaletteEntries];
		if(pRGBQuad == NULL)
			return;//Not enough memory to load the palette.

		//Read the palette data into memory.
		fread(pRGBQuad, sizeof(RGBQUAD),nPaletteEntries,pFile);	

		//If there are palette entries, we need to realize the palette so we can use these colors.
		pLogPalette = (LOGPALETTE*) new char[sizeof(LOGPALETTE)+(sizeof(PALETTEENTRY)*nPaletteEntries)];

		if(pLogPalette == NULL)
			return;//Error initializing memory for the logpalette.

		pLogPalette->palVersion = 0x300;
		pLogPalette->palNumEntries = nPaletteEntries;

		for(int a=0;a<nPaletteEntries;a++)
		{
			pLogPalette->palPalEntry[a].peRed = pRGBQuad[a].rgbRed;
			pLogPalette->palPalEntry[a].peGreen = pRGBQuad[a].rgbGreen;
			pLogPalette->palPalEntry[a].peBlue = pRGBQuad[a].rgbBlue;
		}

		HPALETTE hPalette = CreatePalette(pLogPalette);
		SelectPalette(m_hDC,hPalette,TRUE);
		RealizePalette(m_hDC);
	
		uUsage = DIB_PAL_COLORS;
	}

	//Initialize the memory for the bit information.
	char* pBits = NULL;
	int nBits = bitmapFileHeader.bfSize - sizeof(BITMAPFILEHEADER) - sizeof(BITMAPINFOHEADER) - (sizeof(RGBQUAD)*nPaletteEntries);
	pBits = new char[nBits];
	if(pBits == NULL)
		return;//Not enough memory to load the bitmap bits.

	//Read the bit data into memory.
	fread(pBits, sizeof(char),nBits,pFile);	
	
	//Close the bitmap file.
	fclose(pFile);


	// Populate bitmapinfo header with the information needed to draw a cross.
	BITMAPINFO bmInfo;
	bmInfo.bmiHeader.biSize = sizeof(BITMAPINFOHEADER);
	bmInfo.bmiHeader.biWidth = ((((int) m_nWidth * 8) + 31) & ~31) >> 3;//Pads to a multiple of 4;
	bmInfo.bmiHeader.biHeight = m_nHeight;
	bmInfo.bmiHeader.biPlanes = 1;
	bmInfo.bmiHeader.biBitCount = bitmapInfoHeader.biBitCount;
	bmInfo.bmiHeader.biCompression = BI_RGB;

	// Create the DIB section.
	m_hBmp = CreateDIBSection(m_hDC,
							  (BITMAPINFO*)&bmInfo,
							  uUsage,
							  (VOID **)&m_pBits,
							  0,
							  0);

	// Select the bitmap into the buffer
	if(m_hBmp)
		::SelectObject(m_hDC,m_hBmp);
	else
		return;//ERROR!!!

	if(nPaletteEntries > 0)
	{
		UINT nTemp = SetDIBColorTable(m_hDC,0,nPaletteEntries,pRGBQuad);
		if(nTemp == 0)
			DWORD dwTemp = GetLastError();

		nTemp = SetDIBits(m_hDC,m_hBmp,0,bitmapInfoHeader.biHeight,pBits,(BITMAPINFO*)&bmInfo,uUsage);
		if(nTemp == 0)
			DWORD dwTemp = GetLastError();
	}
 
	
	//Flood the bitmap with black.
	SetBkColor(m_hDC, RGB(0,0,0));


	//Now that the bitmap is loaded into memory, we need to blt it into the memory DC.
	int nXDest = 0;
	int nYDest = 0;
	nXDest = (m_nWidth - bitmapInfoHeader.biWidth)/2;
	nYDest = (m_nHeight - bitmapInfoHeader.biHeight)/2;
	StretchDIBits(m_hDC,nXDest,nYDest,bitmapInfoHeader.biWidth,bitmapInfoHeader.biHeight,0,0,bitmapInfoHeader.biWidth,bitmapInfoHeader.biHeight,pBits,(const BITMAPINFO*)&bitmapInfoHeader,uUsage,SRCCOPY);
	
	//Create a monochrome bitmap for use later in the PlgBlt.
	{
		m_hDcMono = CreateCompatibleDC(m_hDC);

		//Create the compatible mono bitmap.
		m_hBmpMono = (HBITMAP) CreateCompatibleBitmap(m_hDcMono, m_nWidth, m_nHeight);
		SelectObject(m_hDcMono, m_hBmpMono);

		//Blt the aircraft bitmap into the mono bitmap.  The magic is in the NOTSRCCOPY.
		BitBlt(m_hDcMono,0,0,m_nWidth,m_nHeight,m_hDC,0,0,NOTSRCCOPY);
	}

	//Clean up memory before exiting.
	delete []pLogPalette;
	delete []pRGBQuad;
	delete []pBits;
}
