// Engine3Gauge.cpp: implementation of the CEngine3Gauge class.
//
//////////////////////////////////////////////////////////////////////

#include "..\core\stdafx.h"
#include "Engine3Gauge.h"
#include "..\core\dataconversion.h"

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif

#ifndef  HAF
   // FLASHER determines how long the NACWS threat alert is filled/unfilled.
   static   const int   FLASHER                       =  1;
   // TOTAL_FLASH_TIME is the number of seconds to flash a TA when it first becomes a TA.
   static   const int   TOTAL_FLASH_TIME              =  10;
   // RETURN_TO_NORMAL_MODE_TIMER is the number of seconds after nacws_selected switch is released before the display switches back to the engine gauge.
   static   const int   RETURN_TO_NORMAL_MODE_TIMER   =  7;    
#endif

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

CEngine3Gauge::CEngine3Gauge()
{
   m_stlStrWidgetName   =  _FSI_STL::string("Engine_Oil");
   m_exPtUpperLeft      =  CExtentsPoint(CPoint(0,0));
   m_exPtLowerRight     =  CExtentsPoint(CPoint(200,200));


   
   m_CStr_temperature.Format("%.0f%cC", 0.0f, DEGREES);
   m_CStr_voltage.Format("%.1f VOLTS", 0.0f);
   m_CStr_amperage.Format("%+.0f AMPS", 0.0f);

   m_CStr_temperature_cv         =  m_CStr_temperature;
   m_CStr_voltage_cv             =  m_CStr_voltage;
   m_CStr_amperage_cv            =  m_CStr_amperage;

   m_temperature_angle           =  -135.0f;

   m_oilPressure                 =  100.0f;
   m_oilPressure_cv              =  m_oilPressure;
   if (m_oilPressure       >  0.0f  && m_oilPressure  <  200.0f)
      m_oilPressure_angle_cv  =  (270.0f/200.0f *  m_oilPressure)   -  135.0f;
   else if (m_oilPressure  >= 200.0f)
      m_oilPressure_angle_cv  =  135.0f;
   else
      m_oilPressure_angle_cv  =  -135.0f;

   m_hydraulicPressure           =  2000.0f;
   m_hydraulicPressure_cv        =  m_hydraulicPressure;
   if (m_hydraulicPressure  >  1700.0f   && m_hydraulicPressure <  3600.0f)
      m_hydraulicPressure_angle_cv  =  (270.0f/1900.0f   *  (m_hydraulicPressure   -  1700.0f))   -  135.0f;
   else if (m_hydraulicPressure  >  3600.0f)
      m_hydraulicPressure_angle_cv  =  135.0f;
   else
      m_hydraulicPressure_angle_cv  =  -135.0f;

   m_CStr_oilPressure.Format("%.0f\nPSI", m_oilPressure);
   m_CStr_hydraulicPressure.Format("%.0f\nPSI", m_hydraulicPressure);
   m_CStr_oilPressure_cv         =  m_CStr_oilPressure;
   m_CStr_hydraulicPressure_cv   =  m_CStr_hydraulicPressure;

   m_temperature_angle_cv        =  m_temperature_angle;
   m_oilPressure_angle_cv        =  m_oilPressure_angle;
   m_hydraulicPressure_angle_cv  =  m_hydraulicPressure_angle;

   m_listGraphicalElementVars.clear();
   m_listGraphicalElementVars.push_back("temperature");
   m_listGraphicalElementVars.push_back("oilPressure");
   m_listGraphicalElementVars.push_back("hydraulicPressure");
   m_listGraphicalElementVars.push_back("voltage");
   m_listGraphicalElementVars.push_back("amperage");

#ifndef  HAF
   m_listGraphicalElementVars.push_back("Nacws_off_sw");
   m_listGraphicalElementVars.push_back("Nacws_selected");
   m_listGraphicalElementVars.push_back("Nacws_Display_Range");
   m_listGraphicalElementVars.push_back("Nacws_System_Status");
   m_listGraphicalElementVars.push_back("Nacws_Test");
   m_listGraphicalElementVars.push_back("Nacws_Status");
   m_listGraphicalElementVars.push_back("Nacws_System_Mode");
   m_listGraphicalElementVars.push_back("Nacws_Detection_Mode");
   m_listGraphicalElementVars.push_back("Nacws_Intr_Range_1");
   m_listGraphicalElementVars.push_back("Nacws_Intr_Range_2");
   m_listGraphicalElementVars.push_back("Nacws_Intr_Rel_Alt_1");
   m_listGraphicalElementVars.push_back("Nacws_Intr_Rel_Alt_2");
   m_listGraphicalElementVars.push_back("Nacws_Intr_Disp_Type_1");
   m_listGraphicalElementVars.push_back("Nacws_Intr_Disp_Type_2");
   m_listGraphicalElementVars.push_back("Nacws_Intr_Brg_1");
   m_listGraphicalElementVars.push_back("Nacws_Intr_Brg_2");
   m_listGraphicalElementVars.push_back("Nacws_Intr_Active_1");
   m_listGraphicalElementVars.push_back("Nacws_Intr_Active_2");
   m_listGraphicalElementVars.push_back("Nacws_On");
   m_listGraphicalElementVars.push_back("Tas_Inhibit_Nacws");

   m_float_Nacws_Display_Range         =  20.0f;
   m_float_Nacws_Display_Range_cv      =  m_float_Nacws_Display_Range;
   m_bool_nacws_selected               =  false;
   m_bool_nacws_selected_cv            =  m_bool_nacws_selected;
   m_CTime_nacws_deselected_startTime  =  0;

   m_NACWS_list[0].flash_counter       =  0;
   m_NACWS_list[0].flash_counter_sign  =  1;

   m_NACWS_list[1].flash_counter       =  0;
   m_NACWS_list[1].flash_counter_sign  =  1;
  
#endif

   for (int ii =  0; ii < MAX_NUMBER_OF_LISTS; ii++)
   {
      m_int_lists[ii]   =  -1;
   }
}

CEngine3Gauge::~CEngine3Gauge()
{
}

void  CEngine3Gauge::Setup(void)
{
   CPanel::Initialize(GetSafeHwnd());
   SetupTemperatureTicks();
   SetupOilPressureTicks();
   SetupHydraulicPressureTicks();

#ifndef  HAF
   SetupNacwsOuterTicks();
   SetupNacwsInnerTicks();
   range.SetForegroundColor(0, 255, 255);
   range.SetBackgroundColor(true, 0, 0, 0);
#endif

   m_CStr_needle.Format("%c", NOMINAL_NEEDLE);
   m_CStr_temperature_title         =  "TEMP";
   m_CStr_pressure_title            =  "PRESS";
   m_CStr_oilTitle                  =  "OIL";
   m_CStr_hydraulic_pressure_title.Format("HYDR\nPRESS");

   for (int ii =  0; ii < MAX_NUMBER_OF_LISTS; ii++)
   {
      if (m_int_lists[ii]  != -1)
         glDeleteLists(m_int_lists[ii], 1);
   }

   m_int_lists[BEZEL]   =  glGenLists(1);
   glNewList(m_int_lists[BEZEL], GL_COMPILE);
   {
      // Draw the standard stuff, (bezel, screw heads, and primary title).
      CPanel::Draw();
   }
   glEndList();

   m_int_lists[STATIC_TICKS]  =  glGenLists(1);
   glNewList(m_int_lists[STATIC_TICKS], GL_COMPILE);
   {
      // Temperature portion of Gauge.
      glPushMatrix();
         glTranslatef(-0.4f, 0.40f, 0.0f);
         // White
         temperatureTicks.Draw();
         temperatureTicks.DrawArc(false, -135.0f, -112.5f, 1.5f);
         // Green
         temperatureGreenTicks.Draw();
         temperatureGreenTicks.DrawArc(false, -112.5f, 101.25f, 1.5f);
         // Yellow
         temperatureYellowTicks.Draw();
         temperatureYellowTicks.DrawArc(false, 101.25f, 112.5f, 1.5f);
         // Red
         temperatureRedTicks.Draw();

         temperatureLabel.SetForegroundColor(192, 192, 192);
         glTranslatef(0.0f, -0.25f, 0.0f);
         temperatureLabel.Draw(m_CStr_temperature_title,   true, 0.0f, 1.6f, 1.0f);
      glPopMatrix();

            // Oil Title
      glPushMatrix();
         oilTitle.SetForegroundColor(192, 192, 192);
         glTranslatef(0.00f, 0.75f, 0.0f);
         oilTitle.Draw(m_CStr_oilTitle,   true, 0.0f, 1.6f, 1.0f);
      glPopMatrix();

      // Oil Pressure portion of Gauge
      glPushMatrix();
         glTranslatef(0.4f, 0.40f, 0.0f);
         oilPressureTicks.Draw();
         oilPressureTicks.DrawArc(false, -135.0f, -82.35f, 1.5f);
         oilPressureTicks.DrawArc(false, 40.5f, 135.0f, 1.5f);

         oilPressureGreenTicks.Draw();
         oilPressureGreenTicks.DrawArc(false, -13.5f, 40.5, 1.5f);

         oilPressureYellowTicks.Draw();
         oilPressureYellowTicks.DrawArc(false, -82.35f, -13.5, 1.5f);

         oilPressureRedTicks.Draw();

         oilPressureLabel.SetForegroundColor(192, 192, 192);
         glTranslatef(0.0f, -0.25f, 0.0f);
         oilPressureLabel.Draw(m_CStr_pressure_title,  true, 0.0f, 1.6f, 1.0f);
      glPopMatrix();

      // HydraulicPressure portion of Gauge
      glPushMatrix();
         glTranslatef(-0.4f, -0.30f, 0.0f);
         hydraulicPressureTicks.Draw();
         hydraulicPressureTicks.DrawArc(false, -135.0f, 32.7f, 1.5f);
         hydraulicPressureTicks.DrawArc(false, 66.8f, 120.0f, 1.5f);

         hydraulicPressureGreenTicks.Draw();
         hydraulicPressureGreenTicks.DrawArc(false, 32.7f, 66.8f, 1.5f);

         hydraulicPressureYellowTicks.Draw();
         hydraulicPressureYellowTicks.DrawArc(false, -135.0f, -120.8f, 1.5f);
         hydraulicPressureYellowTicks.DrawArc(false, 120.8f, 135.0f, 1.5f);

         hydraulicPressureLabel.SetForegroundColor(192, 192, 192);
         glTranslatef(0.0f, -0.25f, 0.0f);
         hydraulicPressureLabel.Draw(m_CStr_hydraulic_pressure_title, true, 0.0f, 1.6f, 1.0f);
      glPopMatrix();
   }
   glEndList();

}


void  CEngine3Gauge::Render(void)
{
   BeginDraw();

   glPushMatrix();
   {
      glCallList(m_int_lists[BEZEL]);

#ifndef  HAF
      if (m_bool_nacws_selected && !m_bool_TAS_installed)
      {
         DrawNacwsDisplay();
      }
      else
      {
         m_CTime_nacws_deselected_startTime  =  0;    // Reset the timer so that NACWS display doesn't inadvertantly display.
#endif

         glCallList(m_int_lists[STATIC_TICKS]);

         // Draw tick marks and labels.
         // Temperature portion of Gauge.
         glPushMatrix();
            glTranslatef(-0.4f, 0.40f, 0.0f);
            temperatureValue.Draw(m_CStr_temperature,   true, 0.0f, 1.75f, 1.0f);

            if (m_temperature_angle   >= 112.5f)
               temperatureRedTicks.DrawArc(false, 112.5f, m_temperature_angle, 1.5f);

            glRotatef(-m_temperature_angle, 0.0f, 0.0f, 1.0f);
            glTranslatef( 0.0f, 0.125f, 0.0f);
            temperatureNeedle.Draw(m_CStr_needle,  true, 0.0f, 14.0f, 1.0f);
         glPopMatrix();


         // Oil Pressure portion of Gauge
         glPushMatrix();
            glTranslatef(0.4f, 0.40f, 0.0f);

            if (m_oilPressure <  40.0f)
               oilPressureRedTicks.DrawArc(false, m_oilPressure_angle,  -82.35f, 1.5f);

            glPushMatrix();
               glTranslatef(0.0f, 0.05f, 0.0f);
               oilPressureValue.Draw(m_CStr_oilPressure,   true, 0.0f, 1.75f, 1.0f);
            glPopMatrix();

            glRotatef(-m_oilPressure_angle, 0.0f, 0.0f, 1.0f);
            glTranslatef(0.0f, 0.125f, 0.0f);
            oilPressureNeedle.Draw(m_CStr_needle,  true, 0.0f, 14.0f, 1.0f);
         glPopMatrix();


         // HydraulicPressure portion of Gauge
         glPushMatrix();
            glTranslatef(-0.4f, -0.30f, 0.0f);

            glPushMatrix();
               glTranslatef(0.0f, 0.05f, 0.0f);
               hydraulicPressureValue.Draw(m_CStr_hydraulicPressure,   true, 0.0f, 1.75f, 1.0f);
            glPopMatrix();

            glRotatef(-m_hydraulicPressure_angle, 0.0f, 0.0f, 1.0f);
            glTranslatef(0.0f, 0.125f, 0.0f);
            hydraulicPressureNeedle.Draw(m_CStr_needle,  true, 0.0f, 14.0f, 1.0f);
         glPopMatrix();


         // Voltage
         glPushMatrix();
            glTranslatef(0.45f, -0.10f, 0.0f);
            voltageValue.Draw(m_CStr_voltage,  true, 0.0f, 1.75f, 1.0f);
         glPopMatrix();


         // Amperage
         glPushMatrix();
            glTranslatef(0.45f, -0.30f, 0.0f);
            amperageValue.Draw(m_CStr_amperage,   true, 0.0f, 1.75f, 1.0f);
         glPopMatrix();


#ifndef  HAF
         if (!m_bool_TAS_installed)
         {
            if (!m_bool_nacws_off   && m_bool_nacws_on)       // If NACWS has NOT been turned off...
            {
               // NACWS System Mode
               glPushMatrix();
                  glTranslatef(0.45f, -0.50f, 0.0f);
                  nacwsSystemMode.Draw(m_CStr_nacwsSystemMode, true, 0.0f, 1.6f, 1.0f);
               glPopMatrix();
            }

            // NACWS Status
            glPushMatrix();
               glTranslatef(0.05f, -0.775f, 0.0f);
               nacwsStatus.Draw(m_CStr_nacwsStatus,   true, 0.0f, 1.6f, 1.0f);
            glPopMatrix();      
         }
      }
#endif
   }
   glPopMatrix();

   EndDraw();
}


void  CEngine3Gauge::SetTemperatureValue(float temperature)
{
   m_temperature_cv           =  temperature;

   if (temperature   >  128.0f   || temperature <  -70.0f)
      m_CStr_temperature_cv.Format("---%cC", DEGREES);
   else
      m_CStr_temperature_cv.Format("%.0f%cC", temperature, DEGREES);

   if (temperature  >  0.0f   && temperature <  120.0f)
      m_temperature_angle_cv  =  (270.0f/120.0f *  temperature)   -  135.0f;
   else if (temperature >= 120.0f)
      m_temperature_angle_cv  =  135.0f;
   else
      m_temperature_angle_cv  =  -135.0f;
}


void  CEngine3Gauge::SetOilPressureValue(float oilPressure)
{
   m_oilPressure_cv           =  oilPressure;
   m_CStr_oilPressure_cv.Format("%.0f\nPSI", oilPressure);

   if (oilPressure   >  0.0f  && oilPressure <  200.0f)
      m_oilPressure_angle_cv  =  (270.0f/200.0f *  oilPressure)   -  135.0f;
   else if (oilPressure >= 200.0f)
      m_oilPressure_angle_cv  =  135.0f;
   else
      m_oilPressure_angle_cv  =  -135.0f;
}


void  CEngine3Gauge::SetHydraulicPressureValue(float hydraulicPressure)
{
   m_hydraulicPressure_cv  =  ROUND(hydraulicPressure, 0.1);   // Round to the nearest 10.

   m_CStr_hydraulicPressure_cv.Format("%.0f\nPSI", m_hydraulicPressure_cv);

   if (m_hydraulicPressure_cv >  1700.0f  && m_hydraulicPressure_cv  <  3600.0f)
      m_hydraulicPressure_angle_cv  =  (270.0f/1900.0f   *  (m_hydraulicPressure_cv -  1700.0f))   -  135.0f;
   else if (hydraulicPressure >= 3600.0f)
      m_hydraulicPressure_angle_cv  =  135.0f;
   else
      m_hydraulicPressure_angle_cv  =  -135.0f;
}


void  CEngine3Gauge::SetVoltageValue(float voltage)
{
   m_voltage_cv  =  voltage;
   if (m_voltage_cv  >  0.0f  && m_voltage_cv  <  10.0f)
      m_CStr_voltage_cv.Format(" %.1f VOLTS", m_voltage_cv);
   else
      m_CStr_voltage_cv.Format("%.1f VOLTS", m_voltage_cv);
}


void  CEngine3Gauge::SetAmperageValue(float amperage)
{
   m_amperage_cv =  amperage;
   float fabs_amperage  =  fabs(amperage);

   if (fabs_amperage  <  10.0f)
      m_CStr_amperage_cv.Format("  %+.0f AMPS", m_amperage_cv);
   else if (fabs_amperage  <  100.0f)
      m_CStr_amperage_cv.Format(" %+.0f AMPS", m_amperage_cv);
   else
      m_CStr_amperage_cv.Format("%+.0f AMPS", m_amperage_cv);
}


void  CEngine3Gauge::SetupTemperatureTicks(void)
{
   temperatureTicks.Initialize();
   temperatureTicks.SetLabelRadius(0.45f);
   temperatureTicks.SetOuterRadius(0.35f);
   temperatureTicks.SetLabelProperties(1.6f, 1.0f);
   temperatureTicks.SetTickColor(255, 255, 255);

   TICK_INFO   tick_info;

   strcpy(tick_info.ident, "");

   tick_info.location   =  225.0f;
   tick_info.type       =  MINOR;
//   strcpy(tick_info.ident, "0");
   temperatureTicks.TickList.AddTail(tick_info);

   temperatureTicks.GenerateLists();


   temperatureGreenTicks.Initialize();
   temperatureGreenTicks.SetLabelRadius(0.45f);
   temperatureGreenTicks.SetOuterRadius(0.35f);
   temperatureGreenTicks.SetLabelProperties(1.6f, 1.0f);
   temperatureGreenTicks.SetTickColor(0, 128, 0);

   tick_info.location   =  247.5f;
   tick_info.type       =  MINOR;
//   strcpy(tick_info.ident, "10");
   temperatureGreenTicks.TickList.AddTail(tick_info);

   tick_info.location   =  270.0f;
   tick_info.type       =  MINOR;
//   strcpy(tick_info.ident, "20");
   temperatureGreenTicks.TickList.AddTail(tick_info);

   tick_info.location   =  292.5f;
   tick_info.type       =  MINOR;
//   strcpy(tick_info.ident, "30");
   temperatureGreenTicks.TickList.AddTail(tick_info);

   tick_info.location   =  315.0f;
   tick_info.type       =  MINOR;
//   strcpy(tick_info.ident, "40");
   temperatureGreenTicks.TickList.AddTail(tick_info);

   tick_info.location   =  337.5f;
   tick_info.type       =  MINOR;
//   strcpy(tick_info.ident, "50");
   temperatureGreenTicks.TickList.AddTail(tick_info);

   tick_info.location   =  0.0f;
   tick_info.type       =  MINOR;
//   strcpy(tick_info.ident, "60");
   temperatureGreenTicks.TickList.AddTail(tick_info);

   tick_info.location   =  22.5f;
   tick_info.type       =  MINOR;
//   strcpy(tick_info.ident, "70");
   temperatureGreenTicks.TickList.AddTail(tick_info);

   tick_info.location   =  45.0f;
   tick_info.type       =  MINOR;
//   strcpy(tick_info.ident, "80");
   temperatureGreenTicks.TickList.AddTail(tick_info);

   tick_info.location   =  67.5f;
   tick_info.type       =  MINOR;
//   strcpy(tick_info.ident, "90");
   temperatureGreenTicks.TickList.AddTail(tick_info);

   tick_info.location   =  90.0f;
   tick_info.type       =  MINOR;
//   strcpy(tick_info.ident, "100");
   temperatureGreenTicks.TickList.AddTail(tick_info);

   tick_info.location   =  101.25f;
   tick_info.type       =  NO_TICK;
//   strcpy(tick_info.ident, "");
   temperatureGreenTicks.TickList.AddTail(tick_info);

   temperatureGreenTicks.GenerateLists();


   temperatureYellowTicks.Initialize();
   temperatureYellowTicks.SetLabelRadius(0.45f);
   temperatureYellowTicks.SetOuterRadius(0.35f);
   temperatureYellowTicks.SetLabelProperties(1.6f, 1.0f);
   temperatureYellowTicks.SetTickColor(255, 255, 0);

   tick_info.location   =  101.25f;
   tick_info.type       =  NO_TICK;
//   strcpy(tick_info.ident, "");
   temperatureYellowTicks.TickList.AddTail(tick_info);

   tick_info.location   =  112.5f;
   tick_info.type       =  NO_TICK;
//   strcpy(tick_info.ident, "110");
   temperatureYellowTicks.TickList.AddTail(tick_info);

   temperatureYellowTicks.GenerateLists();


   temperatureRedTicks.Initialize();
   temperatureRedTicks.SetLabelRadius(0.45f);
   temperatureRedTicks.SetOuterRadius(0.35f);
   temperatureRedTicks.SetLabelProperties(1.6f, 3.25f);
   temperatureRedTicks.SetTickColor(255, 0, 0);

   tick_info.location   =  112.5f;
   tick_info.type       =  INTERMEDIATE;
//   strcpy(tick_info.ident, "110");
   temperatureRedTicks.TickList.AddTail(tick_info);

   tick_info.location   =  135.0f;
   tick_info.type       =  NO_TICK;
//   strcpy(tick_info.ident, "120");
   temperatureRedTicks.TickList.AddTail(tick_info);

   temperatureRedTicks.GenerateLists();
}


void  CEngine3Gauge::SetupOilPressureTicks(void)
{
   oilPressureTicks.Initialize();
   oilPressureTicks.SetLabelRadius(0.45f);
   oilPressureTicks.SetOuterRadius(0.35f);
   oilPressureTicks.SetLabelProperties(1.6f, 1.0f);
   oilPressureTicks.SetTickColor(255, 255, 255);

   TICK_INFO   tick_info;

   strcpy(tick_info.ident, "");


   tick_info.location   =  225.0f;
   tick_info.type       =  MINOR;
//   strcpy(tick_info.ident, "0");
   oilPressureTicks.TickList.AddTail(tick_info);

   tick_info.location   =  252.0f;
   tick_info.type       =  MINOR;
//   strcpy(tick_info.ident, "20");
   oilPressureTicks.TickList.AddTail(tick_info);

   tick_info.location   =  279.0f;
   tick_info.type       =  MINOR;
//   strcpy(tick_info.ident, "40");
   oilPressureTicks.TickList.AddTail(tick_info);

   tick_info.location   =  40.5f;      // 130
   tick_info.type       =  NO_TICK;
//   strcpy(tick_info.ident, "");
   oilPressureTicks.TickList.AddTail(tick_info);

   tick_info.location   =  54.0f;
   tick_info.type       =  MINOR;
//   strcpy(tick_info.ident, "140");
   oilPressureTicks.TickList.AddTail(tick_info);

   tick_info.location   =  81.0f;
   tick_info.type       =  MINOR;
//   strcpy(tick_info.ident, "160");
   oilPressureTicks.TickList.AddTail(tick_info);

   tick_info.location   =  108.0f;
   tick_info.type       =  MINOR;
//   strcpy(tick_info.ident, "180");
   oilPressureTicks.TickList.AddTail(tick_info);

   tick_info.location   =  135.0f;
   tick_info.type       =  MINOR;
//   strcpy(tick_info.ident, "200");
   oilPressureTicks.TickList.AddTail(tick_info);

   oilPressureTicks.GenerateLists();


   oilPressureYellowTicks.Initialize();
   oilPressureYellowTicks.SetLabelRadius(0.45f);
   oilPressureYellowTicks.SetOuterRadius(0.35f);
   oilPressureYellowTicks.SetLabelProperties(1.6f, 1.0f);
   oilPressureYellowTicks.SetTickColor(255, 255, 0);

   tick_info.location   =  306.0f;
   tick_info.type       =  MINOR;
//   strcpy(tick_info.ident, "60");
   oilPressureYellowTicks.TickList.AddTail(tick_info);

   tick_info.location   =  333.0f;
   tick_info.type       =  MINOR;
//   strcpy(tick_info.ident, "80");
   oilPressureYellowTicks.TickList.AddTail(tick_info);

   tick_info.location   =  346.5f;     // 90
   tick_info.type       =  NO_TICK;
//   strcpy(tick_info.ident, "");
   oilPressureYellowTicks.TickList.AddTail(tick_info);

   oilPressureYellowTicks.GenerateLists();



   oilPressureGreenTicks.Initialize();
   oilPressureGreenTicks.SetLabelRadius(0.45f);
   oilPressureGreenTicks.SetOuterRadius(0.35f);
   oilPressureGreenTicks.SetLabelProperties(1.6f, 1.0f);
   oilPressureGreenTicks.SetTickColor(0, 128, 0);

   tick_info.location   =  346.5f;     // 90
   tick_info.type       =  NO_TICK;
//   strcpy(tick_info.ident, "");
   oilPressureGreenTicks.TickList.AddTail(tick_info);

   tick_info.location   =  0.0f;
   tick_info.type       =  MINOR;
//   strcpy(tick_info.ident, "100");
   oilPressureGreenTicks.TickList.AddTail(tick_info);

   tick_info.location   =  27.0f;
   tick_info.type       =  MINOR;
//   strcpy(tick_info.ident, "120");
   oilPressureGreenTicks.TickList.AddTail(tick_info);

   tick_info.location   =  40.5f;      // 130
   tick_info.type       =  NO_TICK;
//   strcpy(tick_info.ident, "");
   oilPressureGreenTicks.TickList.AddTail(tick_info);

   oilPressureGreenTicks.GenerateLists();



   oilPressureRedTicks.Initialize();
   oilPressureRedTicks.SetLabelRadius(0.45f);
   oilPressureRedTicks.SetOuterRadius(0.35f);
   oilPressureRedTicks.SetLabelProperties(1.6f, 3.25f);
   oilPressureRedTicks.SetTickColor(255, 0, 0);

   tick_info.location   =  279.0f;
   tick_info.type       =  INTERMEDIATE;
//   strcpy(tick_info.ident, "180");
   oilPressureRedTicks.TickList.AddTail(tick_info);

   tick_info.location   =  135.0f;
   tick_info.type       =  MINOR;
//   strcpy(tick_info.ident, "200");
   oilPressureRedTicks.TickList.AddTail(tick_info);

   oilPressureRedTicks.GenerateLists();
}


void  CEngine3Gauge::SetupHydraulicPressureTicks(void)
{
   hydraulicPressureTicks.Initialize();
   hydraulicPressureTicks.SetLabelRadius(0.45f);
   hydraulicPressureTicks.SetOuterRadius(0.35f);
   hydraulicPressureTicks.SetLabelProperties(1.6f, 1.0f);
   hydraulicPressureTicks.SetTickColor(255, 255, 255);

   TICK_INFO   tick_info;

   strcpy(tick_info.ident, "");

   tick_info.location   =  239.2f;
   tick_info.type       =  MINOR;
//   strcpy(tick_info.ident, "1800");
   hydraulicPressureTicks.TickList.AddTail(tick_info);

   tick_info.location   =  267.6f;
   tick_info.type       =  MINOR;
//   strcpy(tick_info.ident, "2000");
   hydraulicPressureTicks.TickList.AddTail(tick_info);

   tick_info.location   =  296.1f;
   tick_info.type       =  MINOR;
//   strcpy(tick_info.ident, "2200");
   hydraulicPressureTicks.TickList.AddTail(tick_info);

   tick_info.location   =  324.5f;
   tick_info.type       =  MINOR;
//   strcpy(tick_info.ident, "2400");
   hydraulicPressureTicks.TickList.AddTail(tick_info);

   tick_info.location   =  352.9f;
   tick_info.type       =  MINOR;
//   strcpy(tick_info.ident, "2600");
   hydraulicPressureTicks.TickList.AddTail(tick_info);

   tick_info.location   =  21.3f;
   tick_info.type       =  MINOR;
//   strcpy(tick_info.ident, "2800");
   hydraulicPressureTicks.TickList.AddTail(tick_info);

   tick_info.location   =  27.0f;      // 2880
   tick_info.type       =  NO_TICK;
//   strcpy(tick_info.ident, "");
   hydraulicPressureTicks.TickList.AddTail(tick_info);

   tick_info.location   =  63.0f;      // 3120
   tick_info.type       =  NO_TICK;
//   strcpy(tick_info.ident, "");
   hydraulicPressureTicks.TickList.AddTail(tick_info);

   tick_info.location   =  78.2f;
   tick_info.type       =  MINOR;
//   strcpy(tick_info.ident, "3200");
   hydraulicPressureTicks.TickList.AddTail(tick_info);

   tick_info.location   =  106.6f;
   tick_info.type       =  MINOR;
//   strcpy(tick_info.ident, "3400");
   hydraulicPressureTicks.TickList.AddTail(tick_info);

   tick_info.location   =  120.0f;     // 3500
   tick_info.type       =  NO_TICK;
//   strcpy(tick_info.ident, "");
   hydraulicPressureTicks.TickList.AddTail(tick_info);

   hydraulicPressureTicks.GenerateLists();


   hydraulicPressureGreenTicks.Initialize();
   hydraulicPressureGreenTicks.SetLabelRadius(0.45f);
   hydraulicPressureGreenTicks.SetOuterRadius(0.35f);
   hydraulicPressureGreenTicks.SetLabelProperties(1.6f, 1.0f);
   hydraulicPressureGreenTicks.SetTickColor(0, 128, 0);

   tick_info.location   =  27.0f;      // 2880
   tick_info.type       =  NO_TICK;
//   strcpy(tick_info.ident, "");
   hydraulicPressureGreenTicks.TickList.AddTail(tick_info);

   tick_info.location   =  49.7f;
   tick_info.type       =  MINOR;
//   strcpy(tick_info.ident, "3000");
   hydraulicPressureGreenTicks.TickList.AddTail(tick_info);

   tick_info.location   =  63.0f;      // 3120
   tick_info.type       =  NO_TICK;
//   strcpy(tick_info.ident, "");
   hydraulicPressureGreenTicks.TickList.AddTail(tick_info);

   hydraulicPressureGreenTicks.GenerateLists();


   hydraulicPressureYellowTicks.Initialize();
   hydraulicPressureYellowTicks.SetLabelRadius(0.45f);
   hydraulicPressureYellowTicks.SetOuterRadius(0.35f);
   hydraulicPressureYellowTicks.SetLabelProperties(1.6f, 1.0f);
   hydraulicPressureYellowTicks.SetTickColor(255, 255, 0);

   tick_info.location   =  135.0f;
   tick_info.type       =  MINOR;
//   strcpy(tick_info.ident, "3600");
   hydraulicPressureYellowTicks.TickList.AddTail(tick_info);

   hydraulicPressureYellowTicks.GenerateLists();
}


#ifndef  HAF
void  CEngine3Gauge::SetupNacwsOuterTicks(void)
{
   nacwsOuterTicks.Initialize();
   nacwsOuterTicks.SetOuterRadius(NACWS_OUTER_RING_RADIUS);
   nacwsOuterTicks.SetLabelProperties(1.6f, 1.0f);
   nacwsOuterTicks.SetTickColor(0, 255, 255);
   nacwsOuterTicks.SetTickLengths(0.0375f, 0.0275f, 0.0275f);

   TICK_INFO   tick_info;

   strcpy(tick_info.ident, "");

   tick_info.location   =  0.0f;
   tick_info.type       =  MAJOR_RECTANGLE;
   nacwsOuterTicks.TickList.AddTail(tick_info);

   tick_info.location   =  5.0f;
   tick_info.type       =  MINOR;
   nacwsOuterTicks.TickList.AddTail(tick_info);

   tick_info.location   =  10.0f;
   tick_info.type       =  MINOR;
   nacwsOuterTicks.TickList.AddTail(tick_info);

   tick_info.location   =  15.0f;
   tick_info.type       =  MINOR;
   nacwsOuterTicks.TickList.AddTail(tick_info);

   tick_info.location   =  20.0f;
   tick_info.type       =  MINOR;
   nacwsOuterTicks.TickList.AddTail(tick_info);

   tick_info.location   =  25.0f;
   tick_info.type       =  MINOR;
   nacwsOuterTicks.TickList.AddTail(tick_info);

   tick_info.location   =  30.0f;
   tick_info.type       =  INTERMEDIATE_DIAMOND;
   nacwsOuterTicks.TickList.AddTail(tick_info);

   tick_info.location   =  35.0f;
   tick_info.type       =  MINOR;
   nacwsOuterTicks.TickList.AddTail(tick_info);

   tick_info.location   =  40.0f;
   tick_info.type       =  MINOR;
   nacwsOuterTicks.TickList.AddTail(tick_info);

   tick_info.location   =  45.0f;
   tick_info.type       =  MINOR;
   nacwsOuterTicks.TickList.AddTail(tick_info);

   tick_info.location   =  50.0f;
   tick_info.type       =  MINOR;
   nacwsOuterTicks.TickList.AddTail(tick_info);

   tick_info.location   =  55.0f;
   tick_info.type       =  MINOR;
   nacwsOuterTicks.TickList.AddTail(tick_info);

   tick_info.location   =  60.0f;
   tick_info.type       =  INTERMEDIATE_DIAMOND;
   nacwsOuterTicks.TickList.AddTail(tick_info);

   tick_info.location   =  65.0f;
   tick_info.type       =  MINOR;
   nacwsOuterTicks.TickList.AddTail(tick_info);

   tick_info.location   =  70.0f;
   tick_info.type       =  MINOR;
   nacwsOuterTicks.TickList.AddTail(tick_info);

   tick_info.location   =  75.0f;
   tick_info.type       =  MINOR;
   nacwsOuterTicks.TickList.AddTail(tick_info);

   tick_info.location   =  80.0f;
   tick_info.type       =  MINOR;
   nacwsOuterTicks.TickList.AddTail(tick_info);

   tick_info.location   =  85.0f;
   tick_info.type       =  MINOR;
   nacwsOuterTicks.TickList.AddTail(tick_info);

   tick_info.location   =  90.0f;
   tick_info.type       =  MAJOR_RECTANGLE;
   nacwsOuterTicks.TickList.AddTail(tick_info);

   tick_info.location   =  95.0f;
   tick_info.type       =  MINOR;
   nacwsOuterTicks.TickList.AddTail(tick_info);

   tick_info.location   =  100.0f;
   tick_info.type       =  MINOR;
   nacwsOuterTicks.TickList.AddTail(tick_info);

   tick_info.location   =  105.0f;
   tick_info.type       =  MINOR;
   nacwsOuterTicks.TickList.AddTail(tick_info);

   tick_info.location   =  110.0f;
   tick_info.type       =  MINOR;
   nacwsOuterTicks.TickList.AddTail(tick_info);

   tick_info.location   =  115.0f;
   tick_info.type       =  MINOR;
   nacwsOuterTicks.TickList.AddTail(tick_info);

   tick_info.location   =  120.0f;
   tick_info.type       =  INTERMEDIATE_DIAMOND;
   nacwsOuterTicks.TickList.AddTail(tick_info);

   tick_info.location   =  125.0f;
   tick_info.type       =  MINOR;
   nacwsOuterTicks.TickList.AddTail(tick_info);

   tick_info.location   =  130.0f;
   tick_info.type       =  MINOR;
   nacwsOuterTicks.TickList.AddTail(tick_info);

   tick_info.location   =  135.0f;
   tick_info.type       =  MINOR;
   nacwsOuterTicks.TickList.AddTail(tick_info);

   tick_info.location   =  140.0f;
   tick_info.type       =  MINOR;
   nacwsOuterTicks.TickList.AddTail(tick_info);

   tick_info.location   =  145.0f;
   tick_info.type       =  MINOR;
   nacwsOuterTicks.TickList.AddTail(tick_info);

   tick_info.location   =  150.0f;
   tick_info.type       =  INTERMEDIATE_DIAMOND;
   nacwsOuterTicks.TickList.AddTail(tick_info);

   tick_info.location   =  155.0f;
   tick_info.type       =  MINOR;
   nacwsOuterTicks.TickList.AddTail(tick_info);

   tick_info.location   =  160.0f;
   tick_info.type       =  MINOR;
   nacwsOuterTicks.TickList.AddTail(tick_info);

   tick_info.location   =  165.0f;
   tick_info.type       =  MINOR;
   nacwsOuterTicks.TickList.AddTail(tick_info);

   tick_info.location   =  170.0f;
   tick_info.type       =  MINOR;
   nacwsOuterTicks.TickList.AddTail(tick_info);

   tick_info.location   =  175.0f;
   tick_info.type       =  MINOR;
   nacwsOuterTicks.TickList.AddTail(tick_info);

//   tick_info.location   =  180.0f;
//   tick_info.type       =  MAJOR_RECTANGLE;
//   nacwsOuterTicks.TickList.AddTail(tick_info);

   tick_info.location   =  185.0f;
   tick_info.type       =  MINOR;
   nacwsOuterTicks.TickList.AddTail(tick_info);

   tick_info.location   =  190.0f;
   tick_info.type       =  MINOR;
   nacwsOuterTicks.TickList.AddTail(tick_info);

   tick_info.location   =  195.0f;
   tick_info.type       =  MINOR;
   nacwsOuterTicks.TickList.AddTail(tick_info);

   tick_info.location   =  200.0f;
   tick_info.type       =  MINOR;
   nacwsOuterTicks.TickList.AddTail(tick_info);

   tick_info.location   =  205.0f;
   tick_info.type       =  MINOR;
   nacwsOuterTicks.TickList.AddTail(tick_info);

   tick_info.location   =  210.0f;
   tick_info.type       =  INTERMEDIATE_DIAMOND;
   nacwsOuterTicks.TickList.AddTail(tick_info);

   tick_info.location   =  215.0f;
   tick_info.type       =  MINOR;
   nacwsOuterTicks.TickList.AddTail(tick_info);

   tick_info.location   =  220.0f;
   tick_info.type       =  MINOR;
   nacwsOuterTicks.TickList.AddTail(tick_info);

   tick_info.location   =  225.0f;
   tick_info.type       =  MINOR;
   nacwsOuterTicks.TickList.AddTail(tick_info);

   tick_info.location   =  230.0f;
   tick_info.type       =  MINOR;
   nacwsOuterTicks.TickList.AddTail(tick_info);

   tick_info.location   =  235.0f;
   tick_info.type       =  MINOR;
   nacwsOuterTicks.TickList.AddTail(tick_info);

   tick_info.location   =  240.0f;
   tick_info.type       =  INTERMEDIATE_DIAMOND;
   nacwsOuterTicks.TickList.AddTail(tick_info);

   tick_info.location   =  245.0f;
   tick_info.type       =  MINOR;
   nacwsOuterTicks.TickList.AddTail(tick_info);

   tick_info.location   =  250.0f;
   tick_info.type       =  MINOR;
   nacwsOuterTicks.TickList.AddTail(tick_info);

   tick_info.location   =  255.0f;
   tick_info.type       =  MINOR;
   nacwsOuterTicks.TickList.AddTail(tick_info);

   tick_info.location   =  260.0f;
   tick_info.type       =  MINOR;
   nacwsOuterTicks.TickList.AddTail(tick_info);

   tick_info.location   =  265.0f;
   tick_info.type       =  MINOR;
   nacwsOuterTicks.TickList.AddTail(tick_info);

   tick_info.location   =  270.0f;
   tick_info.type       =  MAJOR_RECTANGLE;
   nacwsOuterTicks.TickList.AddTail(tick_info);

   tick_info.location   =  275.0f;
   tick_info.type       =  MINOR;
   nacwsOuterTicks.TickList.AddTail(tick_info);

   tick_info.location   =  280.0f;
   tick_info.type       =  MINOR;
   nacwsOuterTicks.TickList.AddTail(tick_info);

   tick_info.location   =  285.0f;
   tick_info.type       =  MINOR;
   nacwsOuterTicks.TickList.AddTail(tick_info);

   tick_info.location   =  290.0f;
   tick_info.type       =  MINOR;
   nacwsOuterTicks.TickList.AddTail(tick_info);

   tick_info.location   =  295.0f;
   tick_info.type       =  MINOR;
   nacwsOuterTicks.TickList.AddTail(tick_info);

   tick_info.location   =  300.0f;
   tick_info.type       =  INTERMEDIATE_DIAMOND;
   nacwsOuterTicks.TickList.AddTail(tick_info);

   tick_info.location   =  305.0f;
   tick_info.type       =  MINOR;
   nacwsOuterTicks.TickList.AddTail(tick_info);

   tick_info.location   =  310.0f;
   tick_info.type       =  MINOR;
   nacwsOuterTicks.TickList.AddTail(tick_info);

   tick_info.location   =  315.0f;
   tick_info.type       =  MINOR;
   nacwsOuterTicks.TickList.AddTail(tick_info);

   tick_info.location   =  320.0f;
   tick_info.type       =  MINOR;
   nacwsOuterTicks.TickList.AddTail(tick_info);

   tick_info.location   =  325.0f;
   tick_info.type       =  MINOR;
   nacwsOuterTicks.TickList.AddTail(tick_info);

   tick_info.location   =  330.0f;
   tick_info.type       =  INTERMEDIATE_DIAMOND;
   nacwsOuterTicks.TickList.AddTail(tick_info);

   tick_info.location   =  335.0f;
   tick_info.type       =  MINOR;
   nacwsOuterTicks.TickList.AddTail(tick_info);

   tick_info.location   =  340.0f;
   tick_info.type       =  MINOR;
   nacwsOuterTicks.TickList.AddTail(tick_info);

   tick_info.location   =  345.0f;
   tick_info.type       =  MINOR;
   nacwsOuterTicks.TickList.AddTail(tick_info);

   tick_info.location   =  350.0f;
   tick_info.type       =  MINOR;
   nacwsOuterTicks.TickList.AddTail(tick_info);

   tick_info.location   =  355.0f;
   tick_info.type       =  MINOR;
   nacwsOuterTicks.TickList.AddTail(tick_info);

   nacwsOuterTicks.GenerateLists();
}


void  CEngine3Gauge::SetupNacwsInnerTicks(void)
{
   nacwsInnerTicks.Initialize();
   nacwsInnerTicks.SetOuterRadius(NACWS_INNER_RING_RADIUS);
   nacwsInnerTicks.SetLabelProperties(1.4f, 1.0f);
   nacwsInnerTicks.SetTickColor(0, 255, 255);
   nacwsInnerTicks.SetTickLengths(0.0275f, 0.0275f, 0.0275f);

   TICK_INFO   tick_info;

   strcpy(tick_info.ident, "");

   tick_info.location   =  0.0f;
   tick_info.type       =  MAJOR_RECTANGLE;
   nacwsInnerTicks.TickList.AddTail(tick_info);

   tick_info.location   =  15.0f;
   tick_info.type       =  MINOR;
   nacwsInnerTicks.TickList.AddTail(tick_info);

   tick_info.location   =  30.0f;
   tick_info.type       =  MINOR;
   nacwsInnerTicks.TickList.AddTail(tick_info);

   tick_info.location   =  45.0f;
   tick_info.type       =  MINOR;
   nacwsInnerTicks.TickList.AddTail(tick_info);

   tick_info.location   =  60.0f;
   tick_info.type       =  MINOR;
   nacwsInnerTicks.TickList.AddTail(tick_info);

   tick_info.location   =  75.0f;
   tick_info.type       =  MINOR;
   nacwsInnerTicks.TickList.AddTail(tick_info);

   tick_info.location   =  90.0f;
   tick_info.type       =  MAJOR_RECTANGLE;
   nacwsInnerTicks.TickList.AddTail(tick_info);

   tick_info.location   =  105.0f;
   tick_info.type       =  MINOR;
   nacwsInnerTicks.TickList.AddTail(tick_info);

   tick_info.location   =  120.0f;
   tick_info.type       =  MINOR;
   nacwsInnerTicks.TickList.AddTail(tick_info);

   tick_info.location   =  135.0f;
   tick_info.type       =  MINOR;
   nacwsInnerTicks.TickList.AddTail(tick_info);

   tick_info.location   =  150.0f;
   tick_info.type       =  MINOR;
   nacwsInnerTicks.TickList.AddTail(tick_info);

   tick_info.location   =  165.0f;
   tick_info.type       =  MINOR;
   nacwsInnerTicks.TickList.AddTail(tick_info);

//   tick_info.location   =  180.0f;
//   tick_info.type       =  MAJOR_RECTANGLE;
//   nacwsInnerTicks.TickList.AddTail(tick_info);

   tick_info.location   =  195.0f;
   tick_info.type       =  MINOR;
   nacwsInnerTicks.TickList.AddTail(tick_info);

   tick_info.location   =  210.0f;
   tick_info.type       =  MINOR;
   nacwsInnerTicks.TickList.AddTail(tick_info);

   tick_info.location   =  225.0f;
   tick_info.type       =  MINOR;
   nacwsInnerTicks.TickList.AddTail(tick_info);

   tick_info.location   =  240.0f;
   tick_info.type       =  MINOR;
   nacwsInnerTicks.TickList.AddTail(tick_info);

   tick_info.location   =  255.0f;
   tick_info.type       =  MINOR;
   nacwsInnerTicks.TickList.AddTail(tick_info);

   tick_info.location   =  270.0f;
   tick_info.type       =  MAJOR_RECTANGLE;
   nacwsInnerTicks.TickList.AddTail(tick_info);

   tick_info.location   =  285.0f;
   tick_info.type       =  MINOR;
   nacwsInnerTicks.TickList.AddTail(tick_info);

   tick_info.location   =  300.0f;
   tick_info.type       =  MINOR;
   nacwsInnerTicks.TickList.AddTail(tick_info);

   tick_info.location   =  315.0f;
   tick_info.type       =  MINOR;
   nacwsInnerTicks.TickList.AddTail(tick_info);

   tick_info.location   =  330.0f;
   tick_info.type       =  MINOR;
   nacwsInnerTicks.TickList.AddTail(tick_info);

   tick_info.location   =  345.0f;
   tick_info.type       =  MINOR;
   nacwsInnerTicks.TickList.AddTail(tick_info);

   nacwsInnerTicks.GenerateLists();
}
#endif


CWidget* CEngine3Gauge::CreateObject()
{
   return   new   CEngine3Gauge();
}


void  CEngine3Gauge::ChangeValue(const CString& rstrElementVar, CChangeValue* pCV)
{
   if (pCV  == NULL)
   {
      return;
   }
   CVariant *pVariant   =  pCV->Variant();

   if (rstrElementVar      == "temperature")
   {
      SetTemperatureValue(*pVariant);
   }
   else if (rstrElementVar == "oilPressure")
   {
      SetOilPressureValue(*pVariant);
   }
   else if (rstrElementVar == "hydraulicPressure")
   {
      SetHydraulicPressureValue(*pVariant);
   }
   else if (rstrElementVar == "voltage")
   {
      SetVoltageValue(*pVariant);
   }
   else if (rstrElementVar == "amperage")
   {
      SetAmperageValue(*pVariant);
   }

#ifndef  HAF
   else if (rstrElementVar == "Nacws_off_sw")
   {
      m_bool_nacws_off_cv  =  *pVariant;     // True if Nacws is turned off.
   }
   else if (rstrElementVar == "Nacws_selected")
   {
      if (m_bool_nacws_selected_cv  == true  && (bool)*pVariant   == false)
         m_CTime_nacws_deselected_startTime  =  CTime::GetCurrentTime();

      m_bool_nacws_selected_cv   =  *pVariant;
   }
   else if (rstrElementVar == "Nacws_Display_Range")
   {
      long  selected_range =  *pVariant;

      switch   (selected_range)
      {
         case  0:
            m_float_Nacws_Display_Range_cv   =  1.5f;
            break;
         case  1:
            m_float_Nacws_Display_Range_cv   =  3.0f;
            break;
         case  2:
            m_float_Nacws_Display_Range_cv   =  5.0f;
            break;
         case  3:
            m_float_Nacws_Display_Range_cv   =  10.0f;
            break;
         case  4:
            m_float_Nacws_Display_Range_cv   =  20.0f;
            break;
      }
   }
   else if (rstrElementVar == "Nacws_System_Status")
   {
      m_long_nacws_system_status_cv    =  *pVariant;
   }
   else if (rstrElementVar == "Nacws_Test")
   {
      m_bool_nacws_test_cv             =  *pVariant;
   }
   else if (rstrElementVar == "Nacws_Status")
   {
      m_long_nacws_status_cv           =  *pVariant;
   }
   else if (rstrElementVar == "Nacws_System_Mode")
   {
      m_long_nacws_system_mode_cv      =  *pVariant;
   }
   else if (rstrElementVar == "Nacws_Detection_Mode")
   {
      m_long_nacws_detection_mode_cv   =  *pVariant;
   }
   else if (rstrElementVar == "Nacws_Intr_Range_1")
   {
      m_float_nacws_intr_range_1_cv    =  *pVariant;
   }
   else if (rstrElementVar == "Nacws_Intr_Range_2")
   {
      m_float_nacws_intr_range_2_cv    =  *pVariant;
   }
   else if (rstrElementVar == "Nacws_Intr_Rel_Alt_1")
   {
      m_float_nacws_intr_rel_alt_1_cv  =  *pVariant;
   }
   else if (rstrElementVar == "Nacws_Intr_Rel_Alt_2")
   {
      m_float_nacws_intr_rel_alt_2_cv  =  *pVariant;
   }
   else if (rstrElementVar == "Nacws_Intr_Disp_Type_1")
   {
      m_NACWS_list[0].startTime        =  CTime::GetCurrentTime();

      if (m_long_nacws_intr_disp_type_1_cv   == 1L  && (long)*pVariant  != 1) // last pass it was a threat, this pass it isn't
         m_CTime_nacws_deselected_startTime  =  CTime::GetCurrentTime();

      m_long_nacws_intr_disp_type_1_cv =  *pVariant;
   }
   else if (rstrElementVar == "Nacws_Intr_Disp_Type_2")
   {
      m_NACWS_list[1].startTime        =  CTime::GetCurrentTime();

      if (m_long_nacws_intr_disp_type_2_cv   == 1L  && (long)*pVariant  != 1) // last pass it was a threat, this pass it isn't
         m_CTime_nacws_deselected_startTime  =  CTime::GetCurrentTime();

      m_long_nacws_intr_disp_type_2_cv =  *pVariant;
   }
   else if (rstrElementVar == "Nacws_Intr_Brg_1")
   {
      m_float_nacws_intr_brg_1_cv      =  *pVariant;
   }
   else if (rstrElementVar == "Nacws_Intr_Brg_2")
   {
      m_float_nacws_intr_brg_2_cv      =  *pVariant;
   }
   else if (rstrElementVar == "Nacws_Intr_Active_1")
   {
      m_bool_nacws_intr_active_1_cv    =  *pVariant;
   }
   else if (rstrElementVar == "Nacws_Intr_Active_2")
   {
      m_bool_nacws_intr_active_2_cv    =  *pVariant;
   }
   else if (rstrElementVar == "Nacws_On")
   {
      m_bool_nacws_on_cv               =  *pVariant;
   }
   else if (rstrElementVar == "Tas_Inhibit_Nacws")
   {
      m_bool_TAS_installed             = *pVariant;
   }

#endif
}

/////////////////////////////////////////////////////////////////////////////
//
// bool CEngine3Gauge::UpdateRenderVariables()
//
// Inputs           : None.
//
// Return Values    : success value of the update of the rendering variables.
//
// Date             : 16 June 1999
//
// Engineer         : Billy Baker
//
// Description      : Override of the parent's UpdateRenderVariables.  
//                    If the class is being destroyed while in this method,
//                    then the return value should be false. This method is
//                    called by the COMMS layer before it tells the mainframe
//                    that a screen update is needed.
//
/////////////////////////////////////////////////////////////////////////////
bool CEngine3Gauge::UpdateRenderVariables()
{
   m_temperature                 =  m_temperature_cv;
   m_CStr_temperature            =  m_CStr_temperature_cv;
   m_temperature_angle           =  m_temperature_angle_cv;

   m_oilPressure                 =  m_oilPressure_cv;
   m_CStr_oilPressure            =  m_CStr_oilPressure_cv;
   m_oilPressure_angle           =  m_oilPressure_angle_cv;

   m_hydraulicPressure           =  m_hydraulicPressure_cv;
   m_CStr_hydraulicPressure      =  m_CStr_hydraulicPressure_cv;
   m_hydraulicPressure_angle     =  m_hydraulicPressure_angle_cv;

   m_voltage                     =  m_voltage_cv;
   m_CStr_voltage                =  m_CStr_voltage_cv;

   m_amperage                    =  m_amperage_cv;
   m_CStr_amperage               =  m_CStr_amperage_cv;

   m_bool_exceedence_found       =  false;

#ifndef  HAF
   m_bool_nacws_off              =  m_bool_nacws_off_cv;
   m_float_Nacws_Display_Range   =  m_float_Nacws_Display_Range_cv;
   m_bool_nacws_selected         =  m_bool_nacws_selected_cv;
   m_long_nacws_system_status    =  m_long_nacws_system_status_cv;
   m_bool_nacws_test             =  m_bool_nacws_test_cv;
   m_long_nacws_status           =  m_long_nacws_status_cv;
   m_long_nacws_system_mode      =  m_long_nacws_system_mode_cv;
   m_long_nacws_detection_mode   =  m_long_nacws_detection_mode_cv;
   m_float_nacws_intr_range_1    =  m_float_nacws_intr_range_1_cv;
   m_float_nacws_intr_range_2    =  m_float_nacws_intr_range_2_cv;
   m_float_nacws_intr_rel_alt_1  =  m_float_nacws_intr_rel_alt_1_cv;
   m_float_nacws_intr_rel_alt_2  =  m_float_nacws_intr_rel_alt_2_cv;
   m_long_nacws_intr_disp_type_1 =  m_long_nacws_intr_disp_type_1_cv;
   m_long_nacws_intr_disp_type_2 =  m_long_nacws_intr_disp_type_2_cv;
   m_float_nacws_intr_brg_1      =  m_float_nacws_intr_brg_1_cv;
   m_float_nacws_intr_brg_2      =  m_float_nacws_intr_brg_2_cv;
   m_bool_nacws_intr_active_1    =  m_bool_nacws_intr_active_1_cv;
   m_bool_nacws_intr_active_2    =  m_bool_nacws_intr_active_2_cv;
   m_bool_nacws_on               =  m_bool_nacws_on_cv;
#endif


   if (m_temperature >  128.0f   || m_temperature  <  -70.0f)
   {
      temperatureValue.SetForegroundColor(0,    128,  0);               // Green
      temperatureValue.SetBackgroundColor(false,  0,  0,  0);           // Default
      temperatureNeedle.SetForegroundColor(255,   0,  0);               // Red
      m_bool_exceedence_found    =  true;
   }
   else if (m_temperature >  110.0f   || m_temperature  <  -40.0f)
   {
      temperatureValue.SetForegroundColor(255,  255,  255);             // White
      temperatureValue.SetBackgroundColor(true, 255,  0,  0);           // Red
      temperatureNeedle.SetForegroundColor(255,  0,  0);                // Red
      m_bool_exceedence_found    =  true;
   }
   else if (m_temperature  >  105.0f   && m_temperature  <  110.0f)
   {
      temperatureValue.SetForegroundColor(0,  0,  0);                   // Black
      temperatureValue.SetBackgroundColor(true, 255,  255,  0);         // Yellow
      temperatureNeedle.SetForegroundColor(255,  255,  255);            // White
      m_bool_exceedence_found    =  true;
   }
   else
   {
      temperatureValue.SetForegroundColor(255,  255,  255);             // White
      temperatureValue.SetBackgroundColor(false, 0,  0,  0);            // Default
      temperatureNeedle.SetForegroundColor(255,  255,  255);            // White
   }

   
   if ((m_oilPressure   >= 0.0f  && m_oilPressure  <  39.0f)   || m_oilPressure  >  200.0f)
   {
      oilPressureValue.SetForegroundColor(255,  255,  255);             // White
      oilPressureValue.SetBackgroundColor(true, 255,  0,  0);           // Red
      oilPressureNeedle.SetForegroundColor(255,  0,  0);                // Red
      m_bool_exceedence_found    =  true;
   }
   else if (m_oilPressure  >= 39.0f && m_oilPressure  <  90.0f)
   {
      oilPressureValue.SetForegroundColor(0,  0,  0);                   // Black
      oilPressureValue.SetBackgroundColor(true, 255,  255,  0);         // Yellow
      oilPressureNeedle.SetForegroundColor(255,  255,  255);            // White
      m_bool_exceedence_found    =  true;
   }
   else
   {
      oilPressureValue.SetForegroundColor(255,  255,  255);             // White
      oilPressureValue.SetBackgroundColor(false, 0,  0,  0);            // Default
      oilPressureNeedle.SetForegroundColor(255,  255,  255);            // White
   }


   if (m_hydraulicPressure <  1800.0f  || m_hydraulicPressure  >  3500.0f)
   {
      hydraulicPressureValue.SetForegroundColor(0,  0,  0);             // Black
      hydraulicPressureValue.SetBackgroundColor(true, 255,  255,  0);   // Yellow
      m_bool_exceedence_found    =  true;
   }
   else
   {
      hydraulicPressureValue.SetForegroundColor(255,  255,  255);       // White
      hydraulicPressureValue.SetBackgroundColor(false, 0,  0,  0);      // Default
   }


   if (m_voltage  <  22.0f || (m_voltage  >  29.5  && m_voltage   <  32.3))
   {
      voltageValue.SetForegroundColor(0,  0, 0);                        // Black
      voltageValue.SetBackgroundColor(true, 255,  255,  0);             // Yellow
      m_bool_exceedence_found    =  true;
   }
   else if (m_voltage   >  32.2f)
   {
      voltageValue.SetForegroundColor(255,   255,  255);                // White
      voltageValue.SetBackgroundColor(true, 255,  0,  0);               // Red
      m_bool_exceedence_found    =  true;
   }
   else
   {
      voltageValue.SetForegroundColor(255,   255,  255);                // White
      voltageValue.SetBackgroundColor(false, 0,  0,  0);                // Default
   }


#ifndef  HAF
   // Try to determine whether or not to show the NACWS display or the Engine display.
   m_CTime_nacws_currentTime  =  CTime::GetCurrentTime();
   m_CTimeSpan_elapsedTime    =  m_CTime_nacws_currentTime  -  m_CTime_nacws_deselected_startTime;
   long  total_seconds        =  m_CTimeSpan_elapsedTime.GetTotalSeconds();
   bool  ignore_timer         =  true;

   if (m_bool_nacws_selected)    // If the trigger is pulled - show the NACWS display.  Don't care about anything else.
   {
   }
   else                          // Trigger isn't pulled so...
   {
      if (m_bool_exceedence_found   || m_bool_nacws_off  || !m_bool_nacws_on)
         ignore_timer   =  true;
      else
         ignore_timer   =  false;

      if (((m_long_nacws_intr_disp_type_1   == 1) || (m_long_nacws_intr_disp_type_2   == 1))   && !m_bool_exceedence_found)
         m_bool_nacws_selected   =  true;
      else
         m_bool_nacws_selected   =  false;

      if (!ignore_timer)
      {
         if (m_bool_nacws_selected  ||  total_seconds <  RETURN_TO_NORMAL_MODE_TIMER)
            m_bool_nacws_selected   =  true;
         else
            m_bool_nacws_selected   =  false;
      }
   }


   if (m_bool_nacws_off)                        // NACWS has been turned off...
   {
      if (m_bool_nacws_selected)
         m_CStr_nacwsStatus   =  "NACWS\nOFF";
      else
         m_CStr_nacwsStatus   =  "NACWS OFF";
   }
   else if (!m_bool_nacws_on)                   // NACWS has been failed...
   {
      if (m_bool_nacws_selected)
         m_CStr_nacwsStatus   =  "NACWS\nFAIL";
      else
         m_CStr_nacwsStatus   =  "NACWS FAIL";
   }
   else
   {
      if (m_long_nacws_system_status   == 1  || m_long_nacws_system_status == 3  || m_long_nacws_detection_mode   == 1)
         m_bool_nacws_show_list  =  true;
      else
         m_bool_nacws_show_list  =  false;

      if (m_float_Nacws_Display_Range  <  3.0f)
      {
         m_CStr_nacws_display_inner_range.Format("%.2f", m_float_Nacws_Display_Range   *  0.5);
         m_CStr_nacws_display_outer_range.Format("%.1f", m_float_Nacws_Display_Range);
      }
      else if (m_float_Nacws_Display_Range  ==  10.0f)
      {
         m_CStr_nacws_display_inner_range.Format("%.1f", m_float_Nacws_Display_Range   *  0.5);
         m_CStr_nacws_display_outer_range.Format("%.0f", m_float_Nacws_Display_Range);
      }
      else if (m_float_Nacws_Display_Range  ==  20.0f)
      {
         m_CStr_nacws_display_inner_range.Format("%.0f", m_float_Nacws_Display_Range   *  0.5);
         m_CStr_nacws_display_outer_range.Format("%.0f", m_float_Nacws_Display_Range);
      }
      else
      {
         m_CStr_nacws_display_inner_range.Format("%.1f", m_float_Nacws_Display_Range   *  0.5);
         m_CStr_nacws_display_outer_range.Format("%.1f", m_float_Nacws_Display_Range);
      }

      switch   (m_long_nacws_system_mode)
      {
         case  0:
            if (m_bool_nacws_selected)
               m_CStr_nacwsSystemMode  =  "LANDING";
            else
               m_CStr_nacwsSystemMode  =  "NACWS  \nLANDING";
            break;
         case  1:
            if (m_bool_nacws_selected)
               m_CStr_nacwsSystemMode  =  "ENROUTE";
            else
               m_CStr_nacwsSystemMode  =  "NACWS  \nENROUTE";
            break;
      }

      switch   (m_long_nacws_status)
      {
         case  0:
            if (m_bool_nacws_selected)
               m_CStr_nacwsStatus   =  "NACWS\nFAIL";
            else
               m_CStr_nacwsStatus   =  "NACWS FAIL";
            break;
         case  1:
            if (m_bool_nacws_selected)
               m_CStr_nacwsStatus   =  "NACWS\nINOP";
            else
               m_CStr_nacwsStatus   =  "NACWS INOP";
            break;
         case  2:
            if (m_bool_nacws_selected)
            {
               switch   (m_long_nacws_system_status)
               {
                  case  0:
                     m_CStr_nacwsStatus   =  "";
                     break;
                  case  1:
                     m_CStr_nacwsStatus   =  "GPS LOST";
                     break;
                  case  2:
                     m_CStr_nacwsStatus   =  "HVPS FAIL";
                     break;
                  case  3:
                     m_CStr_nacwsStatus   =  "HEADING LOST";
                     break;
                  case  4:
                     m_CStr_nacwsStatus   =  "RDB FAIL";
                     break;
                  case  5:
                     m_CStr_nacwsStatus   =  "SELF TEST FAIL";
                     break;
                  case  6:
                     m_CStr_nacwsStatus   =  "NO LOCKED RDRS";
                     break;
                  case  7:
                     m_CStr_nacwsStatus   =  "BARO ALT LOST";
                     break;
               }
            }
            else
            {
               m_CStr_nacwsStatus      =  "NACWS DGRD";
            }
            break;
         case  3:
            if ((m_long_nacws_intr_disp_type_1  == 1  || m_long_nacws_intr_disp_type_2 == 1) && !m_bool_nacws_selected)
            {
               m_CStr_nacwsStatus   =  "NACWS TA";
            }
            else if (m_bool_nacws_selected)
            {
               switch   (m_long_nacws_system_status)
               {
                  case  0:
                     m_CStr_nacwsStatus   =  "";
                     break;
                  case  1:
                     m_CStr_nacwsStatus   =  "GPS LOST";
                     break;
                  case  2:
                     m_CStr_nacwsStatus   =  "HVPS FAIL";
                     break;
                  case  3:
                     m_CStr_nacwsStatus   =  "HEADING LOST";
                     break;
                  case  4:
                     m_CStr_nacwsStatus   =  "RDB FAIL";
                     break;
                  case  5:
                     m_CStr_nacwsStatus   =  "SELF TEST FAIL";
                     break;
                  case  6:
                     m_CStr_nacwsStatus   =  "NO LOCKED RDRS";
                     break;
                  case  7:
                     m_CStr_nacwsStatus   =  "BARO ALT LOST";
                     break;
               }
            }
            else
            {
               m_CStr_nacwsStatus   =  "";
            }
            break;
      }
   
      m_NACWS_list[0].bearing             =  m_float_nacws_intr_brg_1;
      m_NACWS_list[0].delta_altitude      =  ROUND(m_float_nacws_intr_rel_alt_1, 0.01);         // Round to the nearest 100.

      m_NACWS_list[0].range               =  m_float_nacws_intr_range_1;
      m_NACWS_list[0].threat              =  (m_long_nacws_intr_disp_type_1   == 1);
      m_NACWS_list[0].unknown_altitude    =  false;
      m_NACWS_list[0].active              =  m_bool_nacws_intr_active_1;


      m_NACWS_list[1].bearing             =  m_float_nacws_intr_brg_2;
      m_NACWS_list[1].delta_altitude      =  ROUND(m_float_nacws_intr_rel_alt_2, 0.01);         // Round to the nearest 100.
      m_NACWS_list[1].range               =  m_float_nacws_intr_range_2;
      m_NACWS_list[1].threat              =  (m_long_nacws_intr_disp_type_2   == 1);
      m_NACWS_list[1].unknown_altitude    =  false;
      m_NACWS_list[1].active              =  m_bool_nacws_intr_active_2;
   }
#endif

   return CWidget::UpdateRenderVariables();
}

#ifndef  HAF
void  CEngine3Gauge::DrawNacwsDisplay(void)
{
   if (m_bool_nacws_off  || !m_bool_nacws_on || m_long_nacws_status  == 0  || m_long_nacws_status  == 1)
   {
      // NACWS Status
      glPushMatrix();
         glTranslatef(0.0f, 0.2f, 0.0f);
         nacwsStatus.Draw(m_CStr_nacwsStatus,   true, 0.0f, 5.0f, 1.0f, true);
      glPopMatrix();
      return;
   }

   if (m_bool_nacws_show_list)
   {
      CString     CStr_symbol_label("ACTIVE MODE");
      CString     CStr_symbol;

      COpenGLtext symbol;
      COpenGLtext symbol_label;

      glPushMatrix();
         glTranslatef(0.0f, 0.75f, 0.0f);
         symbol_label.Draw(CStr_symbol_label, true, 0.0f, 1.5f);                                      // Draw the detection mode.
      glPopMatrix();

      int   number_of_threats_active   =  0;
      if (m_NACWS_list[0].threat)
         number_of_threats_active++;
      if (m_NACWS_list[1].threat)
         number_of_threats_active++;

      for (int ii =  0; ii <  MAX_NACWS_THREATS;   ii++)
      {
         if (m_NACWS_list[ii].active   && m_NACWS_list[ii].threat)
         {
            glPushMatrix();
            {
               if (number_of_threats_active  >  1)
                  glTranslatef(0.0f, 0.50f   -  (ii   *  0.25f), 0.0f);
               else
                  glTranslatef(0.0f, 0.50f, 0.0f);

               if (m_NACWS_list[ii].flash_counter  >  0)                                              // Fill the symbol for FLASHER counts.
               {
                  glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
                  if (m_NACWS_list[ii].flash_counter  >=  FLASHER)
                     m_NACWS_list[ii].flash_counter_sign *= -1;
               }
               else if (m_NACWS_list[ii].flash_counter   <  0)                                        // Don't fill the symbol for FLASHER counts.
               {
                  glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
                  if (m_NACWS_list[ii].flash_counter  <=  -FLASHER)
                     m_NACWS_list[ii].flash_counter_sign *= -1;
               }
               m_NACWS_list[ii].flash_counter   += m_NACWS_list[ii].flash_counter_sign;

               symbol.SetForegroundColor(255, 255, 0);                                                // Yellow.
               symbol_label.SetForegroundColor(255, 255, 0);                                          // Yellow.

               if (fabs(m_NACWS_list[ii].delta_altitude) <= 2700.0f)                                  // Within +/- 2700 ft...
               {
                  // Determine the symbol - based upon the altitude of the potential threat.
                  if (m_NACWS_list[ii].unknown_altitude)                                              // Unknown altitude:
                  {
                     CStr_symbol.Format("%c", CIRCLE);                                                // A circle.
                     symbol.Draw(CStr_symbol, true, 0.0f, 2.0f);                                      // Draw the symbol.
                  }
                  else if (m_NACWS_list[ii].delta_altitude >  300.0f)                                 // Above us:
                  {
                     glPushMatrix();
                        glTranslatef(-0.65f, 0.0f, 0.0f);
                        CStr_symbol.Format("%c", DELTA);                                              // An upward pointing triangle.
                        symbol.Draw(CStr_symbol, true, 0.0f, 2.0f);                                   // Draw the symbol.
                     glPopMatrix();
                  }
                  else if (m_NACWS_list[ii].delta_altitude <   -300.0f)                               // Below us:
                  {
                     glPushMatrix();
                        glTranslatef(-0.65f, 0.0f, 0.0f);
                        CStr_symbol.Format("%c", DELTA);                                              // An upward pointing triangle. 
                        symbol.Draw(CStr_symbol, true, 180.0, 2.0f);                                  // Rotate to point down and draw it.
                     glPopMatrix();
                  }
                  else                                                                                // Within +/- 300 ft:
                  {
                     glPushMatrix();
                        glTranslatef(-0.65f, 0.0f, 0.0f);
                        CStr_symbol.Format("%c", SQUARE);                                             // A square.
                        symbol.Draw(CStr_symbol, true, 0.0f, 2.0f);                                   // Draw the symbol.
                     glPopMatrix();
                  }

                  glPushMatrix();
                     glTranslatef(-0.15f, 0.0f, 0.0f);
                     CStr_symbol_label.Format("%3.1f NM", m_NACWS_list[ii].range);                 // Format the range display.
                     symbol_label.Draw(CStr_symbol_label, true, 0.0f, 1.4f);                       // Draw the range display.
                  glPopMatrix();

                  glPushMatrix();
                     glTranslatef( 0.45f, 0.0f, 0.0f);
                     CStr_symbol_label.Format("%+5.0f FT", m_NACWS_list[ii].delta_altitude);       // Format the altitude display.
                     symbol_label.Draw(CStr_symbol_label, true, 0.0f, 1.4f);                       // Draw the altitude display.
                  glPopMatrix();               
               }
            }
            glPopMatrix();
         }
      }
   }
   else
   {
      glScissor(rectViewCoord.left + (int)(((1.0f   -  NACWS_OUTER_RING_RADIUS)   /  2.0f)   *  m_width)+1, 
                rectView.Height() - rectViewCoord.bottom + (int)(((1.0f   -  NACWS_OUTER_RING_RADIUS)   /  2.0f)  *  m_height), 
                (int)(NACWS_OUTER_RING_RADIUS   *  m_width),  
                (int)(NACWS_OUTER_RING_RADIUS   *  m_height));

      glClearStencil(1);
      glClear(GL_STENCIL_BUFFER_BIT);
      glStencilFunc(GL_ALWAYS, 0, 0);
      glStencilOp(GL_REPLACE, GL_REPLACE, GL_REPLACE);

      GLUquadricObj  *pquad   =  gluNewQuadric();
         gluQuadricDrawStyle(pquad, GLU_FILL);
         glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
         glRGB(0, 0, 0);
         gluDisk(pquad, 0,   NACWS_OUTER_RING_RADIUS,   360, 1);
         glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
      gluDeleteQuadric(pquad);

      glStencilFunc(GL_NOTEQUAL, 1, 1);
      glStencilOp(GL_KEEP, GL_KEEP, GL_KEEP);

      // Draw the outer and inner range rings.
      nacwsOuterTicks.Draw();
      nacwsInnerTicks.Draw();

      CString     CStr_symbol_label;
      CString     CStr_symbol;

      COpenGLtext symbol;
      COpenGLtext symbol_label;

      // Draw the plane at the center of the display.
      CStr_symbol.Format("%c", THE_PLANE);
      symbol.SetForegroundColor(0, 255, 255);
      symbol.Draw(CStr_symbol, true, 0.0f, 2.0f);

      // Set the background of the text to Black so as to mask out whatever the text is on top of.
      symbol_label.SetBackgroundColor(true, 0, 0, 0);
      symbol_label.SetXtoYratio(0.85f);

      for (int ii =  0; ii <  MAX_NACWS_THREATS;   ii++)
      {
         if (m_NACWS_list[ii].active)
         {
            glPushMatrix();
            {
               // Since in OpenGL 0 is real-world East, modify the bearing to compensate.
               float bearing_used_for_rotation  =  90.0f -  m_NACWS_list[ii].bearing;

               glRotatef(bearing_used_for_rotation, 0.0f, 0.0f, 1.0f);

               if (m_NACWS_list[ii].range >  m_float_Nacws_Display_Range)
                  glTranslatef(NACWS_OUTER_RING_RADIUS, 0.0f, 0.0f);
               else
                  glTranslatef(m_NACWS_list[ii].range /  m_float_Nacws_Display_Range   *  NACWS_OUTER_RING_RADIUS, 0.0f, 0.0f);

               if (fabs(m_NACWS_list[ii].delta_altitude) <= 2700.0f)                                  // Within +/- 2700 ft...
               {
                  if (m_NACWS_list[ii].threat)                                                        // This is a threat alert symbol.
                  {
                     if (m_NACWS_list[ii].flash_counter  >  0)                                        // Fill the symbol for FLASHER counts.
                     {
                        glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
                        if (m_NACWS_list[ii].flash_counter  >=  FLASHER)
                           m_NACWS_list[ii].flash_counter_sign *= -1;
                     }
                     else if (m_NACWS_list[ii].flash_counter   <  0)                                  // Don't fill the symbol for FLASHER counts.
                     {
                        glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
                        if (m_NACWS_list[ii].flash_counter  <=  -FLASHER)
                           m_NACWS_list[ii].flash_counter_sign *= -1;
                     }
                     m_NACWS_list[ii].flash_counter   += m_NACWS_list[ii].flash_counter_sign;

                     symbol.SetForegroundColor(255, 255, 0);                                          // Yellow.
                     symbol_label.SetForegroundColor(255, 255, 0);                                    // Yellow.
                  }
                  else                                                                                // Not a threat.
                  {
                     glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
                     symbol.SetForegroundColor(255, 255, 255);                                        // White.
                     symbol_label.SetForegroundColor(255, 255, 255);                                  // White.
                  }


                  // Determine the symbol - based upon the altitude of the potential threat.
                  if (m_NACWS_list[ii].unknown_altitude)                                              // Unknown altitude:
                  {
                     CStr_symbol.Format("%c", CIRCLE);                                                // A circle.
                     symbol.Draw(CStr_symbol, true, -bearing_used_for_rotation, 2.0f);                // Draw the symbol.
                  }
                  else if (m_NACWS_list[ii].delta_altitude >  300.0f)                                 // Above us:
                  {
                     CStr_symbol.Format("%c", DELTA);                                                 // An upward pointing triangle.
                     symbol.Draw(CStr_symbol, true, -bearing_used_for_rotation, 2.0f);                // Draw the symbol.

                     if (m_NACWS_list[ii].delta_altitude <= 900.0f)
                     {
                        CStr_symbol_label.Format("%+03.0f", m_NACWS_list[ii].delta_altitude  /  100.0f); // Format the altitude display.
                        float x_offset =  0.135f   *  sin(bearing_used_for_rotation *  DEG_TO_RAD);      // X offset for the label position.
                        float y_offset =  0.135f   *  cos(bearing_used_for_rotation *  DEG_TO_RAD);      // Y offset for the label position.
                        glTranslatef(x_offset,  y_offset,  0.0f);                                        // Move so we can label the symbol.
                        symbol_label.Draw(CStr_symbol_label, true, -bearing_used_for_rotation, 1.4f);    // Draw the altitude display.
                     }
                  }
                  else if (m_NACWS_list[ii].delta_altitude <   -300.0f)                               // Below us:
                  {
                     CStr_symbol.Format("%c", DELTA);                                                 // An upward pointing triangle. 
                     symbol.Draw(CStr_symbol, true, 180.0 - bearing_used_for_rotation, 2.0f);         // Rotate to point down and draw it.

                     if (m_NACWS_list[ii].delta_altitude >= -900.0f)
                     {
                        CStr_symbol_label.Format("%+03.0f", m_NACWS_list[ii].delta_altitude  /  100.0f); // Format the altitude display.
                        float x_offset =  -0.135f  *  sin(bearing_used_for_rotation *  DEG_TO_RAD);      // X offset for the label position.
                        float y_offset =  -0.135f  *  cos(bearing_used_for_rotation *  DEG_TO_RAD);      // Y offset for the label position.
                        glTranslatef(x_offset,  y_offset,  0.0f);                                        // Move so we can label the symbol.
                        symbol_label.Draw(CStr_symbol_label, true, -bearing_used_for_rotation, 1.4f);    // Draw the altitude display.
                     }
                  }
                  else                                                                                // Within +/- 300 ft:
                  {
                     CStr_symbol.Format("%c", SQUARE);                                                // A square.
                     symbol.Draw(CStr_symbol, true, -bearing_used_for_rotation, 2.0f);                // Draw the symbol.
                  }
               }
            }
            glPopMatrix();
         }
      }

      glStencilFunc(GL_ALWAYS, 0, 0);
      glStencilOp(GL_KEEP, GL_KEEP, GL_KEEP);

      // Reset the Scissor area back to the entire window.
      glScissor(rectViewCoord.left, rectView.Height() - rectViewCoord.bottom, (int)m_width, (int)m_height);

      glPushMatrix();
         glTranslatef(0.0f, -(NACWS_INNER_RING_RADIUS-0.025f), 0.0f);
         range.Draw(m_CStr_nacws_display_inner_range, true, 0.0f, 1.4f);
      glPopMatrix();
      glPushMatrix();
         glTranslatef(0.0f, -(NACWS_OUTER_RING_RADIUS-0.025f), 0.0f);
         range.Draw(m_CStr_nacws_display_outer_range, true, 0.0f, 1.4f);
      glPopMatrix();
   }

   // NACWS Status
   glPushMatrix();
      glTranslatef(0.0f, -0.8f, 0.0f);
      nacwsStatus.Draw(m_CStr_nacwsStatus,   true, 0.0f, 1.25f, 1.0f);
   glPopMatrix();      

   // NACWS System Mode
   glPushMatrix();
      glTranslatef(0.55f, -0.70f, 0.0f);
      nacwsSystemMode.Draw(m_CStr_nacwsSystemMode, true, 0.0f, 1.4f, 1.0f);
   glPopMatrix();
}
#endif
