// AdiGauge.cpp: implementation of the CAdiGauge class.
//
//////////////////////////////////////////////////////////////////////

#include "..\core\stdafx.h"
#include "AdiGauge.h"
#include "..\core\dataconversion.h"
#include "winbase.h"

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

CAdiGauge::CAdiGauge()
{
   m_stlStrWidgetName            =  _FSI_STL::string("ADI");
   m_exPtUpperLeft               =  CExtentsPoint(CPoint(0,0));
   m_exPtLowerRight              =  CExtentsPoint(CPoint(200,200));

   m_adi_mode                    =  1;
   m_attitude_fail               =  0;
   m_cp_500_fail                 =  0;
   m_glideslope_offset           =  0.0f;
   m_long_GS_status              =  0;
   m_bool_GS_fail                =  false;
   m_bool_Nav_Pwr_Fail           =  false;
   m_lateralPosition_offset      =  0.0f;
   m_long_lateral_offset_valid   =  0;
   m_long_lateral_offset_red_x   =  0;
   m_marker_beacon               =  0;
   m_pitch_angle                 =  0.0f;
   m_rate_of_turn                =  0.0f;
   m_roll_angle                  =  0.0f;
   m_slip_and_skid               =  0.0f;
   m_support_HSI_composite_mode  =  0;

   m_adi_mode_cv                    =  m_adi_mode;
   m_attitude_fail_cv               =  m_attitude_fail;
   m_cp_500_fail_cv                 =  m_cp_500_fail;
   m_glideslope_offset_cv           =  m_glideslope_offset;
   m_long_GS_status_cv              =  m_long_GS_status;
   m_bool_GS_fail_cv                =  m_bool_GS_fail;
   m_bool_Nav_Pwr_Fail_cv           =  m_bool_Nav_Pwr_Fail;
   m_lateralPosition_offset_cv      =  m_lateralPosition_offset;
   m_long_lateral_offset_valid_cv   =  m_long_lateral_offset_valid;
   m_long_lateral_offset_red_x_cv   =  m_long_lateral_offset_red_x;
   m_marker_beacon_cv               =  m_marker_beacon;
   m_pitch_angle_cv                 =  m_pitch_angle;
   m_roll_angle_cv                  =  m_roll_angle;
   m_support_HSI_composite_mode_cv  =  m_support_HSI_composite_mode;
   m_slip_and_skid_cv               =  m_slip_and_skid;
   m_rate_of_turn_cv                =  m_rate_of_turn;

   m_bool_FHDG_Flag                 =  false;
   m_bool_FHDG_Flag_cv              =  m_bool_FHDG_Flag;

   m_bool_XTALK_Flag                =  true;
   m_bool_XTALK_Flag_cv             =  m_bool_XTALK_Flag;

   m_bool_EADI_Power                =  true;
   m_bool_EADI_Power_cv             =  m_bool_EADI_Power;

   m_listGraphicalElementVars.clear();
   m_listGraphicalElementVars.push_back("ac_pitch_wing");
   m_listGraphicalElementVars.push_back("ac_roll_wing");
   m_listGraphicalElementVars.push_back("Glideslope");
   m_listGraphicalElementVars.push_back("Lateral Offset");
   m_listGraphicalElementVars.push_back("Lateral Offset Valid");
   m_listGraphicalElementVars.push_back("Lateral Offset Red X");
   m_listGraphicalElementVars.push_back("ADI Mode");
   m_listGraphicalElementVars.push_back("GPS Annunciation Mode");
   m_listGraphicalElementVars.push_back("Marker Beacon");
   m_listGraphicalElementVars.push_back("Slip Ball");
   m_listGraphicalElementVars.push_back("Rate of Turn");
   m_listGraphicalElementVars.push_back("Back Course");

   // For the HSI (composite mode)
   m_listGraphicalElementVars.push_back("Selected Course");
   m_listGraphicalElementVars.push_back("Selected Course Valid");
   m_listGraphicalElementVars.push_back("Selected Course Dtk");

   m_listGraphicalElementVars.push_back("Heading Bug");
   m_listGraphicalElementVars.push_back("Navigation Source");
   m_listGraphicalElementVars.push_back("HSI Mode");
   m_listGraphicalElementVars.push_back("Heading Failure");
   m_listGraphicalElementVars.push_back("Heading Invalid");
   m_listGraphicalElementVars.push_back("Attitude Fail");
   m_listGraphicalElementVars.push_back("CP 500 Fail");
   m_listGraphicalElementVars.push_back("Heading Select Knob Failure");
   m_listGraphicalElementVars.push_back("To From");
   m_listGraphicalElementVars.push_back("AC Heading");
   m_listGraphicalElementVars.push_back("Time To Go");
   m_listGraphicalElementVars.push_back("Ils_Selected");

   m_listGraphicalElementVars.push_back("Primary DME");
   m_listGraphicalElementVars.push_back("Primary DME Status");
   m_listGraphicalElementVars.push_back("Ground Speed");
   m_listGraphicalElementVars.push_back("Ground Speed Status");
   m_listGraphicalElementVars.push_back("Time to Go Status");
   m_listGraphicalElementVars.push_back("Primary DME Frequency");
   m_listGraphicalElementVars.push_back("Primary GPS Alert");
   m_listGraphicalElementVars.push_back("GS Scale Status");
   m_listGraphicalElementVars.push_back("GS Rcvr Fail");
   m_listGraphicalElementVars.push_back("Nav Pwr Fail");

   m_listGraphicalElementVars.push_back("EADI_Power");
   m_listGraphicalElementVars.push_back("FHDG_Flag");
   m_listGraphicalElementVars.push_back("XTALK_Flag");

   m_hsiGauge  =  NULL;

   m_list                  =  -1;
   m_pitch_scale_list      =  -1;
   m_roll_indicators_list  =  -1;

   m_pquad  =  gluNewQuadric();
}

CAdiGauge::~CAdiGauge()
{
   if (m_hsiGauge)
      delete   m_hsiGauge;

   gluDeleteQuadric(m_pquad);
}

void  CAdiGauge::Setup(void)
{
   if (GetSafeHwnd())
      CPanel::Initialize(GetSafeHwnd());

   gluQuadricDrawStyle(m_pquad, GLU_FILL);

   symbolicAircraft.SetForegroundColor(192,  192,  0);   // Yellow
   m_symbolic_aircraft.Format("%c", SYMBOLIC_AIRCRAFT);

   zeroDegreeIndex.SetForegroundColor(192,   192,  0);   // Yellow
   m_zero_degree_index.Format("%c", DELTA);

   whiteSkyPointer.SetForegroundColor(255,   255,  255); // White
   m_white_sky_pointer.Format("%c", DELTA);

   chevron.SetForegroundColor(255,  0, 0);               // Red
   m_chevron   =  "V";

   XTALK.SetForegroundColor(192, 192, 0);                // Yellow
   XTALK.SetBoxOutlineColor(192, 192, 0);                // Yellow
   m_XTALK     =  "XTALK";


   rollTicks.Initialize();
   rollTicks.SetLabelRadius(0.60f);
   rollTicks.SetOuterRadius(0.75f);
   rollTicks.SetLabelProperties(1.3f, 1.0f);
   rollTicks.SetTickLengths(0.15f, 0.09f, 0.05f);


   TICK_INFO   tick_info;

   tick_info.location   =  270.0f;
   tick_info.type       =  INTERMEDIATE;
   strcpy(tick_info.ident, "");
   rollTicks.TickList.AddTail(tick_info);

   tick_info.location   =  300.0f;
   tick_info.type       =  INTERMEDIATE;
   strcpy(tick_info.ident, "");
   rollTicks.TickList.AddTail(tick_info);

   tick_info.location   =  315.0f;
   tick_info.type       =  MINOR;
   strcpy(tick_info.ident, "");
   rollTicks.TickList.AddTail(tick_info);

   tick_info.location   =  330.0f;
   tick_info.type       =  INTERMEDIATE;
   strcpy(tick_info.ident, "");
   rollTicks.TickList.AddTail(tick_info);

   tick_info.location   =  340.0f;
   tick_info.type       =  MINOR;
   strcpy(tick_info.ident, "");
   rollTicks.TickList.AddTail(tick_info);

   tick_info.location   =  350.0f;
   tick_info.type       =  MINOR;
   strcpy(tick_info.ident, "");
   rollTicks.TickList.AddTail(tick_info);

   tick_info.location   =  0.0f;
   tick_info.type       =  INTERMEDIATE;
   strcpy(tick_info.ident, "");
   rollTicks.TickList.AddTail(tick_info);

   tick_info.location   =  10.0f;
   tick_info.type       =  MINOR;
   strcpy(tick_info.ident, "");
   rollTicks.TickList.AddTail(tick_info);

   tick_info.location   =  20.0f;
   tick_info.type       =  MINOR;
   strcpy(tick_info.ident, "");
   rollTicks.TickList.AddTail(tick_info);

   tick_info.location   =  30.0f;
   tick_info.type       =  INTERMEDIATE;
   strcpy(tick_info.ident, "");
   rollTicks.TickList.AddTail(tick_info);

   tick_info.location   =  45.0f;
   tick_info.type       =  MINOR;
   strcpy(tick_info.ident, "");
   rollTicks.TickList.AddTail(tick_info);

   tick_info.location   =  60.0f;
   tick_info.type       =  INTERMEDIATE;
   strcpy(tick_info.ident, "");
   rollTicks.TickList.AddTail(tick_info);

   tick_info.location   =  90.0f;
   tick_info.type       =  INTERMEDIATE;
   strcpy(tick_info.ident, "");
   rollTicks.TickList.AddTail(tick_info);

   rollTicks.GenerateLists();


   // Bottom half of the gauge.
   bottomTicks.Initialize();
   bottomTicks.SetLabelRadius(0.60f);
   bottomTicks.SetOuterRadius(0.75f);
   bottomTicks.SetLabelProperties(1.3f, 1.0f);


   // Lateral Deviation Indicator
   lateralPositionIndicator.SetForegroundColor(0,  192,  0);    // Green
   m_lateralPosition_indicator.Format("%c", BACKGROUND_CELL);

   lateralPositionTicks.Initialize();
   lateralPositionTicks.SetTickStyle(LINEAR, HORIZONTAL);
   lateralPositionTicks.SetLabelProperties(1.25f, 1.0f);
   lateralPositionTicks.SetLabelRadius(0.20f);
   lateralPositionTicks.SetTickLengths(0.0375f, 0.0275f, 0.0175f);
   strcpy(tick_info.ident, "");

   tick_info.location   =  -0.50f;
   tick_info.type       =  INTERMEDIATE_CIRCLE;
   lateralPositionTicks.TickList.AddTail(tick_info);

   tick_info.location   =  -0.25f;
   tick_info.type       =  MINOR_CIRCLE;
   lateralPositionTicks.TickList.AddTail(tick_info);

   tick_info.location   =  0.0f;
   tick_info.type       =  INTERMEDIATE_DIAMOND;
   lateralPositionTicks.TickList.AddTail(tick_info);

   tick_info.location   =  0.25f;
   tick_info.type       =  MINOR_CIRCLE;
   lateralPositionTicks.TickList.AddTail(tick_info);

   tick_info.location   =  0.50f;
   tick_info.type       =  INTERMEDIATE_CIRCLE;
   lateralPositionTicks.TickList.AddTail(tick_info);
   strcpy(tick_info.ident, "");

   lateralPositionTicks.GenerateLists();


   // Glideslope Indicator
   glideslopeIndicator.SetForegroundColor(0,  192,  0);    // Green
   m_glideslope_indicator.Format("%c", DELTA);

   glideslopeTicks.Initialize();
   glideslopeTicks.SetTickStyle(LINEAR, VERTICAL);
   glideslopeTicks.SetLabelProperties(1.25f, 1.0f);
   glideslopeTicks.SetLabelRadius(0.15f);
   glideslopeTicks.SetTickLengths(0.0375f, 0.0275f, 0.0175f);

   strcpy(tick_info.ident, "");
   tick_info.location   =  0.30f;
   tick_info.type       =  INTERMEDIATE_CIRCLE;
   strcpy(tick_info.ident, "^GS");
   glideslopeTicks.TickList.AddTail(tick_info);
   strcpy(tick_info.ident, "");

   tick_info.location   =  0.15f;
   tick_info.type       =  MINOR_CIRCLE;
   glideslopeTicks.TickList.AddTail(tick_info);

   tick_info.location   =  0.0f;
   tick_info.type       =  MAJOR;
   glideslopeTicks.TickList.AddTail(tick_info);

   tick_info.location   =  -0.15f;
   tick_info.type       =  MINOR_CIRCLE;
   glideslopeTicks.TickList.AddTail(tick_info);

   tick_info.location   =  -0.30f;
   tick_info.type       =  INTERMEDIATE_CIRCLE;
   glideslopeTicks.TickList.AddTail(tick_info);

   glideslopeTicks.GenerateLists();


   // Rate of Turn Indicator
   rateOfTurn.SetForegroundColor(255,  255,  255);
   m_rateOfTurn.Format("%c", RATE_OF_TURN);

   rateOfTurnTicks.Initialize();
   rateOfTurnTicks.SetTickStyle(LINEAR, HORIZONTAL);
   rateOfTurnTicks.SetLabelProperties(1.25f, 1.0f);
   rateOfTurnTicks.SetLabelRadius(0.20f);
   rateOfTurnTicks.SetTickLengths(0.0675f, 0.0475f, 0.0275f);
   strcpy(tick_info.ident, "");

   tick_info.location   =  -0.25f;
   tick_info.type       =  MAJOR_RECTANGLE;
   rateOfTurnTicks.TickList.AddTail(tick_info);

   tick_info.location   =  0.0f;
   tick_info.type       =  MAJOR_RECTANGLE;
   rateOfTurnTicks.TickList.AddTail(tick_info);

   tick_info.location   =  0.25f;
   tick_info.type       =  MAJOR_RECTANGLE;
   rateOfTurnTicks.TickList.AddTail(tick_info);
   strcpy(tick_info.ident, "");

   rateOfTurnTicks.GenerateLists();


   // Slip and Skid Indicator
   slipAndSkid.SetForegroundColor(255,  255,  255);
   m_slipAndSkid.Format("%c", CIRCLE);

   slipAndSkidTicks.Initialize();
   slipAndSkidTicks.SetTickStyle(LINEAR, HORIZONTAL);
   slipAndSkidTicks.SetLabelProperties(1.25f, 1.0f);
   slipAndSkidTicks.SetLabelRadius(0.20f);
   slipAndSkidTicks.SetTickLengths(0.0675f, 0.0475f, 0.0275f);
   strcpy(tick_info.ident, "");

   tick_info.location   =  -0.04f;
   tick_info.type       =  MINOR;
   slipAndSkidTicks.TickList.AddTail(tick_info);

   tick_info.location   =  0.04f;
   tick_info.type       =  MINOR;
   slipAndSkidTicks.TickList.AddTail(tick_info);
   strcpy(tick_info.ident, "");

   slipAndSkidTicks.GenerateLists();


   markerBeacon.SetBoxOutlineColor(255, 255, 255);

   xOut.SetForegroundColor(255, 0, 0);
   m_xOut.Format("X");

   attitudeFail.SetBoxOutlineColor(255, 0, 0);
   attitudeFail.SetForegroundColor(255, 0, 0);
   m_attitudeFail.Format("ATTITUDE FAIL");

   controlPanelFailed.SetBoxOutlineColor(255, 0, 0);
   controlPanelFailed.SetForegroundColor(255, 0, 0);
   m_controlPanelFailed.Format("CP");

   backCourseIndicator.SetForegroundColor(0, 192,  0);   // Green
   m_backCourseIndicator.Format("BC");

   if (m_list != -1)
      glDeleteLists(m_list, 1);

   m_list = glGenLists(1);
   glNewList(m_list, GL_COMPILE);
   {
      CPanel::Draw();   // Draw the standard stuff, (bezel, screw heads, and primary title).
   }
   glEndList();


   if (m_pitch_scale_list  != -1)
      glDeleteLists(m_pitch_scale_list, 1);

   m_pitch_scale_list = glGenLists(1);
   glNewList(m_pitch_scale_list, GL_COMPILE);
   {
      float ii =  0.0f;
      glPushMatrix();
      for (ii = -180.0f; ii <= 180.0f; ii += 2.5f)
      { 
         int   fabs_ii  =  (int)fabs(ii);

         if (fabs_ii >  90)
            fabs_ii  =  180   -  fabs_ii;

         if (ii   == 0.0f)
            continue;
         else if (fabs_ii  == 85)
            m_pitch_scale  =  "";
         else if (fabs_ii  %  10 == 0)
            m_pitch_scale.Format("%d ----- %d", fabs_ii, fabs_ii);
         else if (ii == 5.0f  || ii == -5.0f)
            m_pitch_scale.Format("%d --- %d", fabs_ii, fabs_ii);
         else if (fabs_ii  %  5  == 0)
            m_pitch_scale  =  " --- ";
         else if ((fmod(ii, 2.5f)   == 0.0f) && (ii <= 30.0f && ii >= -30.0f))
            m_pitch_scale  =  " -- ";
         else
            continue;

         glPushMatrix();
         {
            glRotatef(-90.0f, 0.0f, 1.0f, 0.0f);
            glRotatef(ii, 0.0f, 0.0f, 1.0f);
            glTranslatef(1.2f, 0.0f, 0.0f);
            glRotatef(90.0f, 0.0f, 1.0f, 0.0f);

            // Draw Pitch Scale or Circles at +-90
            if (ii == 90.0f || ii == -90.0f)
            {
               glRGB(255, 255, 255);         // White
               glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
               gluDisk(m_pquad, 0.030f,  0.040f, 30, 1);
               glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
            }
            else
            {
               if (ii   >  90.0f || ii <  -90.0f)
                  pitchScale.Draw(m_pitch_scale, true, 180.0f, 1.0f, 1.0f);
               else
                  pitchScale.Draw(m_pitch_scale, true, 0.0f, 1.0f, 1.0f);
            }

            if (ii == 45.0f   || ii == 65.0f || ii == 85.0f)
               chevron.Draw(m_chevron,   true, 0.0f, 3.0f, 3.0f);      // V
            else if (ii == -45.0f   || ii == -65.0f   || ii == -85.0f)
               chevron.Draw(m_chevron,   true, 180.0f, 3.0f, 3.0f);    // ^
         }
         glPopMatrix();
      }
      glPopMatrix();
   }
   glEndList();

   
   if (m_roll_indicators_list != -1)
      glDeleteLists(m_roll_indicators_list, 1);

   m_roll_indicators_list  = glGenLists(1);
   glNewList(m_roll_indicators_list, GL_COMPILE);
   {
      rollTicks.DrawTickMarks(false);
      rollTicks.DrawTickLabels();
      rollTicks.DrawArc(true, -90.0f, 90.0f);
      bottomTicks.DrawArc(true, 90.0f, 270.0f);

      // Zero Degree Index marker for roll reference.
      glPushMatrix();
         glTranslatef(0.0f, 0.65f, 0.0f);
         glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
            zeroDegreeIndex.Draw(m_zero_degree_index, true, 180.0f, 2.0f, 1.0f);
         glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
      glPopMatrix();
   }
   glEndList();

   if (m_pWnd)
   {
      m_hsiGauge     =  new   CEHSIGauge;

      if (m_hsiGauge)
      {
         m_hsiGauge->rectView       =  rectView;
         m_hsiGauge->rectViewCoord  =  rectViewCoord;
         m_hsiGauge->Setup();
         m_hsiGauge->m_width        =  m_width;
         m_hsiGauge->m_height       =  m_height;
         m_hsiGauge->m_aspect_ratio =  m_aspect_ratio;
      }
   }
}


void  CAdiGauge::Render(void)
{
   BeginDraw();

   glCallList(m_list);

   DrawAdiSpecifics();

   if (!m_bool_EADI_Power)
   {
      glColor4ub(255, 255, 255, 64);
      glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
         glBegin(GL_QUADS);
            glVertex2f(-1.0f, 1.0f);
            glVertex2f( 1.0f, 1.0f);
            glVertex2f( 1.0f, -1.0f);
            glVertex2f(-1.0f, -1.0f);
         glEnd();
      glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
   }

   EndDraw();
}

void  CAdiGauge::SetPitchAngle(float pitch)
{
   m_pitch_angle_cv  =  pitch;
}

void  CAdiGauge::SetRollAngle(float roll)
{
   m_roll_angle_cv  =  roll;
}

void  CAdiGauge::SetGlideslopeOffset(float offset)
{
   if (offset        >  2.5f)
      offset         =  2.5f  *  0.15f;
   else if (offset   <  -2.5f)
      offset         =  -2.5f *  0.15f;
   else
      offset         *= 0.15f;

   m_glideslope_offset_cv  =  offset;
}

void  CAdiGauge::SetLateralPositionOffset(float offset)
{
   if (offset        >  2.5f)
      offset         =  2.5f  *  0.25f;
   else if (offset   <  -2.5f)
      offset         =  -2.5f *  0.25f;
   else
      offset         *= 0.25f;

   m_lateralPosition_offset_cv   =  offset;
//   m_lateralPosition_offset_cv   =  offset   *  0.25f;
}

CWidget* CAdiGauge::CreateObject()
{
   return new CAdiGauge();
}

void CAdiGauge::ChangeValue(const CString& rstrElementVar, CChangeValue* pCV)
{
   static   CVariant hsi_mode;

   if (pCV == NULL)
   {
      return;
   }
   CVariant* pVariant = pCV->Variant();

   if (rstrElementVar      == "ac_pitch_wing")              // Pitch
   {
      SetPitchAngle(-(float)*pVariant);
   }
   else if (rstrElementVar == "ac_roll_wing")               // Roll
   {
      SetRollAngle(*pVariant);
   }
   else if (rstrElementVar == "Glideslope")                 // Glideslope Offset
   {
      SetGlideslopeOffset(*pVariant);
   }
   else if (rstrElementVar == "Lateral Offset")             // Lateral Offset
   {
      SetLateralPositionOffset(*pVariant);
   }
   else if (rstrElementVar == "ADI Mode")                   // Adi Mode (composite or normal)
   {
      m_adi_mode_cv     =  (long)*pVariant;
   }
   else if (rstrElementVar == "GPS Annunciation Mode")      // GPS Annunciation Mode (enroute or approach)
   {
      m_gps_annunciation_mode_cv =  *pVariant;
   }
   else if (rstrElementVar == "Marker Beacon")              // Marker Beacon
   {
      m_marker_beacon_cv      =  *pVariant;

      switch   (m_marker_beacon_cv)
      {
         case  0:
            markerBeacon_cv.SetForegroundColor(0, 0, 0);
            m_markerBeacon_cv.Format("   ");
            break;
         case  1:
            markerBeacon_cv.SetForegroundColor(0, 255, 255);
            m_markerBeacon_cv.Format("OM");
            break;
         case  2:
            markerBeacon_cv.SetForegroundColor(232, 189, 45);
            m_markerBeacon_cv.Format("MM");
            break;
         case  3:
            markerBeacon_cv.SetForegroundColor(255, 255, 255);
            m_markerBeacon_cv.Format("IM");
            break;
         case  4:
            markerBeacon_cv.SetForegroundColor(255, 0, 0);
            m_markerBeacon_cv.Format("%cM", OMI);
            break;
      }
   }
   else if (rstrElementVar == "Attitude Fail")        // Attitude Fail
   {
      m_attitude_fail_cv      =  *pVariant;
   }
   else if (rstrElementVar == "CP 500 Fail")          // CP 500 Fail
   {
      m_cp_500_fail_cv        =  *pVariant;
   }
   else if (rstrElementVar == "GS Scale Status")      // GS Scale Status
   {
      m_long_GS_status_cv     =  *pVariant;
   }
   else if (rstrElementVar == "GS Rcvr Fail")         // GS failure
   {
      m_bool_GS_fail_cv       =  (bool)*pVariant;
   }
   else if (rstrElementVar == "Nav Pwr Fail")         // Power failure
   {
      m_bool_Nav_Pwr_Fail_cv  =  (bool)*pVariant;
   }
   else if (rstrElementVar == "Lateral Offset Valid") // Lateral Offset Valid
   {
      m_long_lateral_offset_valid_cv   =  *pVariant;
   }
   else if (rstrElementVar == "Lateral Offset Red X") // Lateral Offset Red X
   {
      m_long_lateral_offset_red_x_cv   =  *pVariant;
   }
   else if (rstrElementVar == "Slip Ball")            // Slip Ball
   {
      m_slip_and_skid_cv      =  *pVariant;
      if (m_slip_and_skid_cv  >  3.0f)
         m_slip_and_skid_cv   =  0.20f;
      else if (m_slip_and_skid_cv   <  -3.0f)
         m_slip_and_skid_cv   =  -0.20f;
      else
         m_slip_and_skid_cv   =  (0.20f /  3.0f)  *  m_slip_and_skid_cv;
   }
   else if (rstrElementVar == "Rate of Turn")         // Rate of Turn
   {
      m_rate_of_turn_cv       =  *pVariant;
      if (m_rate_of_turn_cv   >  4.5f)
         m_rate_of_turn_cv    =  0.375f;
      else if (m_rate_of_turn_cv <  -4.5f)
         m_rate_of_turn_cv    =  -0.375f;
      else
         m_rate_of_turn_cv    =  (0.25f /  3.0f)  *  m_rate_of_turn_cv;
   }
   else if (rstrElementVar == "Ils_Selected")      // Ils_Selected
   {
      m_ils_selected_cv    =  (bool)*pVariant;
   }
   else if (rstrElementVar == "Back Course")       // Back Course
   {
      m_back_course_cv     =  (bool)*pVariant;
   }
   else if (rstrElementVar == "HSI Mode")          // HSI Mode
   {
      hsi_mode.Value((long)NORMAL_HSI);
   }
   else if (rstrElementVar == "EADI_Power")        // EADI_Power
   {
      m_bool_EADI_Power_cv =  (bool)*pVariant;
   }
   else if (rstrElementVar == "FHDG_Flag")               // FHDG_Flag
   {
      m_bool_FHDG_Flag_cv  =  (bool)*pVariant;
   }   
   else if (rstrElementVar == "XTALK_Flag")              // XTALK_Flag
   {
      m_bool_XTALK_Flag_cv =  (bool)*pVariant;
   }   
   


   if (m_hsiGauge && m_pWnd)
   {
      m_hsiGauge->SetSupportAdiCompositeMode(m_adi_mode_cv  == ADI_MODE_COMPOSITE);
      
      if (rstrElementVar == "HSI Mode")             // HSI Mode
      {
         CChangeValue   CV;
         CV.Variant(&hsi_mode);
         m_hsiGauge->ChangeValue(rstrElementVar, &CV);
      }
      else
         m_hsiGauge->ChangeValue(rstrElementVar, pCV);
   }
}

void CAdiGauge::DrawAdiSpecifics(void)
{
   glPushMatrix();
   {
      glCallList(m_list);

      if (m_pWnd == NULL)
      {
         // Only scale when called from the EHSI gauge.
         if (m_aspect_ratio  <= 1.0f)
            glScalef(m_aspect_ratio, 1.0f, 1.0f);
         else if (m_aspect_ratio != 0.0f)
            glScalef(1.0f, 1.0f  /  m_aspect_ratio, 1.0f);
      }
  
      if (m_attitude_fail)
      {
         glPushMatrix();
            glTranslatef(0.0f, 0.30f, 0.0f);
            attitudeFail.Draw(m_attitudeFail, true, 0.0f, 1.25f, 1.75, true);
         glPopMatrix();
         bottomTicks.DrawArc(true, 0.0f, 360.0f);
      }
      else
      {
         glPushMatrix();
         {
            glRotatef(m_roll_angle, 0.0f, 0.0f, 1.0f);
            DrawPitchScale();
         }
         glPopMatrix();

         // Roll indicator.
         glCallList(m_roll_indicators_list);
      }

      // Symbolic Aircraft
      symbolicAircraft.Draw(m_symbolic_aircraft, true, 0.0f, 20.0f, 4.0f);

      if (m_gps_annunciation_mode   == GPS_ANNUNCIATION_APPROACH  || m_ils_selected)   // If in approach mode...
      {
         if (m_adi_mode != ADI_MODE_COMPOSITE  && !m_bool_Nav_Pwr_Fail
             && !m_support_HSI_composite_mode)    // If not in composite mode, draw localizer.
            DrawLocalizer();

         if (m_ils_selected  && !m_back_course  && !m_bool_Nav_Pwr_Fail)   // ILS selected, not on back course - show GS indicator.
            DrawGlideslope();
      }

      if (m_marker_beacon)
      {
         glPushMatrix();
            glTranslatef(0.68f, -0.50f, 0.0f);
            markerBeacon.Draw(m_markerBeacon, true, 0.0f, 1.0f, 1.0f, true);
         glPopMatrix();
      }

      if (m_adi_mode != ADI_MODE_COMPOSITE   && !m_support_HSI_composite_mode)         // If not in composite mode, draw rate of turn.
      {
         // Rate of Turn Indicator
         glPushMatrix();
            glTranslatef(0.0f, -0.75f, 0.0f);
            rateOfTurnTicks.Draw();

            if (m_attitude_fail)
            {
               glScalef(30.0f, 2.5f, 1.0f);
               xOut.Draw(m_xOut);
            }
            else
            {
               glTranslatef(m_rate_of_turn, -0.07f, 0.0f);
               rateOfTurn.Draw(m_rateOfTurn, true, 0.0f, 1.5f, 1.0f);
            }
         glPopMatrix();
      }

      // If this is really the AdiGauge and not the ADI portion of a HSI gauge in composite mode.
      if (m_pWnd)
      {
         // Slip and Skid Indicator
         glPushMatrix();
            glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
               glTranslatef(0.0f, -0.97f, 0.0f);

               glRGB(8, 8, 8);
               glBegin(GL_QUADS);
                  glVertex2f(-0.20f,   -0.04f);
                  glVertex2f(-0.20f,    0.04f);
                  glVertex2f( 0.20f,    0.04f);
                  glVertex2f( 0.20f,   -0.04f);
               glEnd();

               slipAndSkidTicks.Draw();

               glTranslatef(m_slip_and_skid, 0.0f, 0.0f);
               slipAndSkid.Draw(m_slipAndSkid, true, 0.0f, 1.0f, 1.0f);
            glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
         glPopMatrix();
      }
   }
   glPopMatrix();

   // Control Panel Failure Annuciator
   if (m_cp_500_fail)
   {
      glPushMatrix();
         glTranslatef(-0.75f, -0.375f, 0.0f);
         controlPanelFailed.Draw(m_controlPanelFailed, true, 0.0f, 1.2f, 1.5, true);
      glPopMatrix();
   }

   // XTALK
   if (m_bool_XTALK_Flag)
   {
      glPushMatrix();
         glTranslatef(-0.700f, -0.550f, 0.0f);
         XTALK.Draw(m_XTALK, true, 0.0f, 1.001f, 1.5f, true);
      glPopMatrix();
   }

   if (m_adi_mode == ADI_MODE_COMPOSITE || m_support_HSI_composite_mode)
   {
      if (m_hsiGauge && m_pWnd)
         m_hsiGauge->DrawHsiSpecifics();
   }
}


void  CAdiGauge::SetSupportHsiCompositeMode(bool support)
{
   m_support_HSI_composite_mode_cv  =  support;
}


void  CAdiGauge::DrawPitchScale(void)
{
   float ii             =  0.0f;
   float offset         =  0.0f;

   glPushMatrix();
      glClearStencil(1);
      glClear(GL_STENCIL_BUFFER_BIT);
      glStencilFunc(GL_ALWAYS, 0, 0);
      glStencilOp(GL_REPLACE, GL_REPLACE, GL_REPLACE);

      // Draw a filled circle to mark the area to clip the rest of the drawing to.
      glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
      glRGB(0, 0, 0);
      gluDisk(m_pquad, 0,   0.6f,   30, 1);

      glStencilFunc(GL_NOTEQUAL, 1, 1);
      glStencilOp(GL_KEEP, GL_KEEP, GL_KEEP);

      // Perform calculations and logic to determine what needs to be blue and what needs to be black.
      float theAngle =  (float)(acos(2.0f * sin(m_pitch_angle * DEG_TO_RAD)) * RAD_TO_DEG);

      if (m_pitch_angle <= -25.0)
         theAngle =  147.5f;
      else if (m_pitch_angle  >= 25.0f)
         theAngle =  32.5f;

      float x  =  0.60f *  (float)sin(theAngle  *  DEG_TO_RAD);
      float y  =  0.60f *  (float)cos(theAngle  *  DEG_TO_RAD);

      glRGB(64, 144, 255);       // Sky blue.
      gluPartialDisk(m_pquad, 0.0f,  0.60f, 30, 1, -theAngle, 2.0f  *  theAngle);  // Draw an arc (pie-shape).

      // Change the color as necessary so as to hide the center portion of the arc (pie-shape).
      if (theAngle   <= 90.0f)
         glRGB(0, 0, 0);         // Black ground.
      else
         glRGB(64, 144, 255);    // Sky blue.

      // Draw a Triangle to hide the center portion of the arc (pie-shape).
      glBegin(GL_POLYGON);
         glVertex2f( 0.0f,  0.0f);
         glVertex2f( x,  y);
         glVertex2f(-x,  y);
      glEnd();


      if (m_pitch_angle <= -25.0 || m_pitch_angle >= 25.0)
         glRGB(0, 0, 0);         // Black horizon line.
      else
         glRGB(255, 255, 255);   // White horizon line.

      // Draw a line representing the horizon.
      glBegin(GL_LINE_STRIP);
         glVertex2f(-x, y);
         glVertex2f( x, y);
      glEnd();

      glRGB(255, 255, 255);         // White
      // Draw Sky Pointer (Indicates Roll).
      glPushMatrix();
         glTranslatef(0.0f, 0.55f, 0.0f);
         whiteSkyPointer.Draw(m_white_sky_pointer, true, 0.0f, 2.0f, 1.0f);
      glPopMatrix();

      glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);

      // Rotate to position and draw the scale.
      glRotatef(-m_pitch_angle, 1.0f, 0.0f, 0.0f);
      glCallList(m_pitch_scale_list);

      glStencilFunc(GL_ALWAYS, 0, 0);
      glStencilOp(GL_KEEP, GL_KEEP, GL_KEEP);
   glPopMatrix();
}


void  CAdiGauge::DrawGlideslope(void)
{
   // Glideslope Indicator
   if (m_long_GS_status)
   {
      glPushMatrix();
         glTranslatef(0.75f, 0.0f, 0.0f);
         glideslopeTicks.Draw();
         glTranslatef(0.025f, m_glideslope_offset, 0.0f);
         glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
            glideslopeIndicator.Draw(m_glideslope_indicator, true, 90.0f, 1.75f, 1.0f);
         glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
      glPopMatrix();
   }
   else if (m_bool_GS_fail)
   {
      glPushMatrix();
         glTranslatef(0.75f, 0.0f, 0.0f);
         glideslopeTicks.Draw();
         glScalef(2.5f, 20.0f, 1.0f);
         xOut.Draw(m_xOut, true, 0.0f, 1.0f, 1.75f, false);
      glPopMatrix();
   }
}


void  CAdiGauge::DrawLocalizer(void)
{
   // Lateral Position Indicator
   glPushMatrix();
      if (m_gps_annunciation_mode   == GPS_ANNUNCIATION_APPROACH)
         m_CStr_lateralPositionType =  "GPS";
      else
         m_CStr_lateralPositionType =  "LOC";

      if (m_back_course)   // If we're doing a back course, show the BC annunciator and reverse the lateral position offset.
      {
         glPushMatrix();
            glTranslatef(-0.10f, 0.0f, 0.0f);
            backCourseIndicator.Draw(m_backCourseIndicator, true, 0.0f, 1.0f, 1.0, false);
            m_lateralPosition_offset   *= -1.0f;
         glPopMatrix();
      }

      if (!m_long_lateral_offset_valid)
      {
         if (m_long_lateral_offset_red_x)
         {
            glTranslatef(0.0f, -0.665f, 0.0f);
            lateralPositionTicks.Draw();
            glPushMatrix();
               glTranslatef(0.75, 0.0f, 0.0f);
               lateralPositionType.Draw(m_CStr_lateralPositionType, true, 0.0f, 1.0f, 1.0f, false);
            glPopMatrix();
            glScalef(30.0f, 2.5f, 1.0f);
            xOut.Draw(m_xOut);
         }
      }
      else
      {
         glTranslatef(0.0f, -0.665f, 0.0f);
         lateralPositionTicks.Draw();
         glPushMatrix();
            glTranslatef(0.75, 0.0f, 0.0f);
            lateralPositionType.Draw(m_CStr_lateralPositionType, true, 0.0f, 1.0f, 1.0f, false);
         glPopMatrix();
         glTranslatef(m_lateralPosition_offset, 0.0f, 0.0f);
         lateralPositionIndicator.Draw(m_lateralPosition_indicator, true, 0.0f, 1.0f, 1.0f);
      }
   glPopMatrix();
}


float CAdiGauge::TimeThisFunction(function_ptr function_to_time)
{
   LARGE_INTEGER  theFrequency;
   QueryPerformanceFrequency(&theFrequency);
   LARGE_INTEGER  theStartCount, theEndCount, diffCount;
   float milliseconds;

   QueryPerformanceCounter(&theStartCount);

   (this->*function_to_time)();
   
   QueryPerformanceCounter(&theEndCount);
   diffCount.QuadPart   =  theEndCount.QuadPart -  theStartCount.QuadPart;
   milliseconds         =  (float)((float)diffCount.QuadPart /  (float)theFrequency.QuadPart)   *  1000.0f;

   return   milliseconds;
}

/////////////////////////////////////////////////////////////////////////////
//
// bool CAdiGauge::UpdateRenderVariables()
//
// Inputs           : None.
//
// Return Values    : success value of the update of the rendering variables.
//
// Date             : 16 June 1999
//
// Engineer         : Billy Baker
//
// Description      : Override of the parent's UpdateRenderVariables.  
//                    If the class is being destroyed while in this method,
//                    then the return value should be false. This method is
//                    called by the COMMS layer before it tells the mainframe
//                    that a screen update is needed.
//
/////////////////////////////////////////////////////////////////////////////
bool CAdiGauge::UpdateRenderVariables()
{
   m_pitch_angle                 =  m_pitch_angle_cv;
   m_roll_angle                  =  m_roll_angle_cv;
   m_glideslope_offset           =  m_glideslope_offset_cv;
   m_lateralPosition_offset      =  m_lateralPosition_offset_cv;

   m_adi_mode                    =  m_adi_mode_cv;
   m_gps_annunciation_mode       =  m_gps_annunciation_mode_cv;
   m_marker_beacon               =  m_marker_beacon_cv;
   markerBeacon                  =  markerBeacon_cv;
   m_markerBeacon                =  m_markerBeacon_cv;
   m_attitude_fail               =  m_attitude_fail_cv;
   m_cp_500_fail                 =  m_cp_500_fail_cv;
   m_long_GS_status              =  m_long_GS_status_cv;
   m_bool_GS_fail                =  m_bool_GS_fail_cv;
   m_bool_Nav_Pwr_Fail           =  m_bool_Nav_Pwr_Fail_cv;
   m_long_lateral_offset_valid   =  m_long_lateral_offset_valid_cv;
   m_long_lateral_offset_red_x   =  m_long_lateral_offset_red_x_cv;
   m_support_HSI_composite_mode  =  m_support_HSI_composite_mode_cv;
   m_slip_and_skid               =  m_slip_and_skid_cv;
   m_rate_of_turn                =  m_rate_of_turn_cv;
   m_ils_selected                =  m_ils_selected_cv;
   m_back_course                 =  m_back_course_cv;

   m_bool_EADI_Power             =  m_bool_EADI_Power_cv;
   m_bool_FHDG_Flag              =  m_bool_FHDG_Flag_cv;
   m_bool_XTALK_Flag             =  m_bool_XTALK_Flag_cv;

   if (m_hsiGauge != NULL)
   {
       // Don't care about the return value because it should be false since the
       // hsi is not a valid window.
       m_hsiGauge->UpdateRenderVariables();
   }

   return CWidget::UpdateRenderVariables();
}
