// VerticalSpeedGauge.cpp: implementation of the CHoneywellTAS class.
//
//////////////////////////////////////////////////////////////////////

#include "..\core\stdafx.h"
#include "..\core\dataconversion.h"
#include "Honeywell_TAS.h"
#include "winbase.h"
#include <float.h>

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif

//////////////////////////////////////////////////////////////////////
// Variables ICD:
//
//   HOST VARIABLE NAME           TYPE  NOTES
//
//   "Vsi_Power"                  bool  false=Off, true=On
//   "vertical_speed"             float FPM (host var: "rate of climb")
//   "Nacws_Display_Range"        int   0=normal, 1=above, 2=below  (actual range is set with a local variable)
//   "Nacws_Status"               int   0=NORMAL,          1=TCAS TEST,     2=TCAS STBY,     3=TCAS FLAG,
//                                      4=NORMAL+VSI_FAIL, 5=TEST+VSI_FAIL, 6=STBY+VSI_FAIL, 7=TCAS/VSI FAIL
//   "Nacws_Intr_Range_1"         float  0.0 .. 1000.0 NM
//   "Nacws_Intr_Range_2"         float  0.0 .. 1000.0 NM
//   "Nacws_Intr_Rel_Alt_1"       float  -10000.0 .. 10000.0 FT
//   "Nacws_Intr_Rel_Alt_2"       float  -10000.0 .. 10000.0 FT
//   "Nacws_Intr_Disp_Type_1"     int   bits 0-1: 0=non-threat tfc (open diamond), 1=TA alert (amber dot),
//                                                2=TA alert/no bearing (amber text), 3=proximate traffic (solid diamond)
//                                      bit    2: reserved,
//                                      bits 3-4: 0=level, 1=climbing, 2=descending
//   "Nacws_Intr_Disp_Type_2"     int   bits 0-1: 0=non-threat tfc (open diamond), 1=TA alert (amber dot),               
//                                      bit    2: reserved,
//                                                2=TA alert/no bearing (amber text), 3=proximate traffic (solid diamond)
//                                      bits 3-4: 0=level, 1=climbing, 2=descending
//   "Nacws_Intr_Brg_1"           float 0.0 .. 360.0
//   "Nacws_Intr_Brg_2"           float 0.0 .. 360.0
//   "Nacws_Intr_Active_1"        bool  false=Off, true=Display
//   "Nacws_Intr_Active_2"        bool  false=Off, true=Display


#define CYAN    0, 255, 255
#define WHITE 255, 255, 255
#define AMBER 255, 255,   0 // yellow per CDR
//#define AMBER 255, 191,   0  // true amber

#define FONT_VSI_LEGEND 1.46f, 1.00f
#define FONT_1000FPM    1.14f, 1.00f
#define FONT_FLAGS      1.70f, 1.25f


//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

CHoneywellTAS::CHoneywellTAS()
{
   m_stlStrWidgetName   =  _FSI_STL::string("Honeywell_TAS");
   m_exPtUpperLeft      =  CExtentsPoint(CPoint(0,0));
   m_exPtLowerRight     =  CExtentsPoint(CPoint(200,200));

   m_listGraphicalElementVars.clear();
   m_listGraphicalElementVars.push_back("vertical_speed"         ); // host var "rate of climb"
   m_listGraphicalElementVars.push_back("Vsi_Power"              );
   m_listGraphicalElementVars.push_back("Nacws_Display_Range"    );
   m_listGraphicalElementVars.push_back("Nacws_Status"           );
   //m_listGraphicalElementVars.push_back("Nacws_Test"             );
   m_listGraphicalElementVars.push_back("Nacws_Intr_Range_1"     );
   m_listGraphicalElementVars.push_back("Nacws_Intr_Range_2"     );
   m_listGraphicalElementVars.push_back("Nacws_Intr_Rel_Alt_1"   );
   m_listGraphicalElementVars.push_back("Nacws_Intr_Rel_Alt_2"   );
   m_listGraphicalElementVars.push_back("Nacws_Intr_Brg_1"       );
   m_listGraphicalElementVars.push_back("Nacws_Intr_Brg_2"       );
   m_listGraphicalElementVars.push_back("Nacws_Intr_Active_1"    );
   m_listGraphicalElementVars.push_back("Nacws_Intr_Active_2"    );
   m_listGraphicalElementVars.push_back("Nacws_Intr_Disp_Type_1" );
   m_listGraphicalElementVars.push_back("Nacws_Intr_Disp_Type_2" );
   m_listGraphicalElementVars.push_back("Tas_Range_Sel"          );


   m_float_vertical_speed     =  0.0f;
   m_float_vertical_speed_cv  =  m_float_vertical_speed;

   m_list             =  -1;
   m_bool_Power       = true;
   m_bool_Power_cv    = true;

   m_float_Nacws_Display_Range         =  10.0f;
   m_float_Nacws_Display_Range_cv      =  m_float_Nacws_Display_Range;

   m_long_View = 0;
   m_long_View_cv = m_long_View;

   m_vsi_valid_cv = m_vsi_valid = true;
   m_long_tcas_flag = TCAS_INVALID;

   m_long_nacws_status = m_long_nacws_status_cv = TCAS_INVALID;
   m_bool_nacws_test   = m_bool_nacws_test_cv   = false;

   m_range_internal = Range_10NM;  // 10 NM default range

   memset(m_nacws_list, 0, sizeof m_nacws_list);

   m_pquad = gluNewQuadric(); // disk for stencil
}

CHoneywellTAS::~CHoneywellTAS()
{
   gluDeleteQuadric(m_pquad);
}


void  CHoneywellTAS::Setup(void)
{
   CPanel::Initialize(GetSafeHwnd());

   const float tick_lengths = 0.10f;

   theTicks.Initialize();
   theTicks.SetLabelRadius(0.68f);
   theTicks.SetOuterRadius(0.78f);
   theTicks.SetLabelProperties(2.5f, 1.5f);
   theTicks.SetTickLengths(tick_lengths, tick_lengths * 0.625f, tick_lengths * 0.4375f);
   theTicks.SetTickColor(WHITE);

   theLabels.Initialize();
   theLabels.SetLabelRadius(0.85f);
   theLabels.SetOuterRadius(0.78f);
   theLabels.SetLabelProperties(2.0f, 1.25f);
   theLabels.SetTickLengths(tick_lengths, tick_lengths * 0.625f, tick_lengths * 0.4375f);
   theLabels.SetTickColor(WHITE);

   struct tick_data_t {
      float     location;
      char*     label;
      tick_type type;
   } tick_data[] = {
      { -172.5f, ""  , MAJOR        }, // -6000
      { -167.4f, ""  , INTERMEDIATE }, // -5500
      { -161.8f, ""  , MAJOR        }, // -5000
      { -155.6f, ""  , INTERMEDIATE }, // -4500
      { -148.7f, "4" , MAJOR        }, // -4000
      { -140.9f, ""  , INTERMEDIATE }, // -3500
      { -131.9f, ""  , MAJOR        }, // -3000
      { -121.2f, ""  , INTERMEDIATE }, // -2500
      { -108.1f, "2" , MAJOR        }, // -2000
      {  -91.3f, ""  , INTERMEDIATE }, // -1500
      {  -67.5f, "1" , MAJOR        }, // -1000
      {  -60.8f, ""  , MINOR        }, //  -900
      {  -54.0f, ""  , MINOR        }, //  -800
      {  -47.3f, ""  , MINOR        }, //  -700
      {  -40.5f, ""  , MINOR        }, //  -600
      {  -33.8f, ".5", INTERMEDIATE }, //  -500
      {  -27.0f, ""  , MINOR        }, //  -400
      {  -20.3f, ""  , MINOR        }, //  -300
      {  -13.5f, ""  , MINOR        }, //  -200
      {   -6.8f, ""  , MINOR        }, //  -100
      {    0.0f, "0" , MAJOR        }, //     0 FPM
      {    6.8f, ""  , MINOR        }, //   100
      {   13.5f, ""  , MINOR        }, //   200
      {   20.3f, ""  , MINOR        }, //   300
      {   27.0f, ""  , MINOR        }, //   400
      {   33.8f, ".5", INTERMEDIATE }, //   500
      {   40.5f, ""  , MINOR        }, //   600
      {   47.3f, ""  , MINOR        }, //   700
      {   54.0f, ""  , MINOR        }, //   800
      {   60.8f, ""  , MINOR        }, //   900
      {   67.5f, "1" , MAJOR        }, //  1000
      {   91.3f, ""  , INTERMEDIATE }, //  1500
      {  108.1f, "2" , MAJOR        }, //  2000
      {  121.2f, ""  , INTERMEDIATE }, //  2500
      {  131.9f, ""  , MAJOR        }, //  3000
      {  140.9f, ""  , INTERMEDIATE }, //  3500
      {  148.7f, "4" , MAJOR        }, //  4000
      {  155.6f, ""  , INTERMEDIATE }, //  4500
      {  161.8f, ""  , MAJOR        }, //  5000
      {  167.4f, ""  , INTERMEDIATE }, //  5500
      {  172.5f, ""  , MAJOR        }, //  6000
      {  180.0f, "6" , NO_TICK      }  //  6000
   };

   TICK_INFO   tick_info;

   const num_entries = sizeof tick_data / sizeof tick_data_t;
   const float tick_rotation_offset = -90.0f;

   for (int n=0; n < num_entries; n++)
   {
      tick_info.location   =  tick_data[n].location + tick_rotation_offset;
      tick_info.type       =  tick_data[n].type;
      strcpy(tick_info.ident, tick_data[n].label);
      theTicks.TickList.AddTail(tick_info);
      theLabels.TickList.AddTail(tick_info);
   }

   theTicks.GenerateLists();
   theLabels.GenerateLists();

   m_verticalSpeedNeedle.Format("%c", VSI_NEEDLE);
   m_verticalSpeed_title.Format("VERTICAL\nSPEED");
   fpm_x1000   =  ("1000 FPM");


   if (m_list != -1)
      glDeleteLists(m_list, 1);

   m_list = glGenLists(1);
   glNewList(m_list, GL_COMPILE);
   {
      glPushMatrix();
      {
         // Draw the standard stuff, (bezel, screw heads).
         CPanel::Draw();

         // Draw tick marks and labels.
         theTicks.DrawTickMarks(false);
         theLabels.DrawTickLabels();
         glLineWidth(1.5f);
         theLabels.DrawArc(true, 0.0f, 360.0f, 1.0f);

      }
      glPopMatrix();
   }
   glEndList();

   m_Cstr_ac_symbol.Format("%c", VSI_AIRCRAFT_SYMBOL);
   aircraftSymbol.SetForegroundColor(CYAN);

}


void  CHoneywellTAS::Render(void)
{

   BeginDraw();

   glPushAttrib( GL_POLYGON_BIT | GL_ENABLE_BIT | GL_LINE_BIT | GL_POINT_BIT );

   glPushMatrix();
      glCallList(m_list);  // STATIC ITEMS
   glPopMatrix();

   if (m_long_tcas_flag == TCAS_STBY    ||
       m_long_tcas_flag == TCAS_TEST    ||
       m_long_tcas_flag == TCAS_TA_ONLY   )
   {
      glPushMatrix();                       // RANGE
         glTranslatef(0.6f, 0.83f, 0.0f);
         range.SetForegroundColor(CYAN);
         range.Draw(m_CStr_nacws_display_range, true, 0.0f, FONT_FLAGS);
      glPopMatrix();

      if (m_long_View != VIEW_NORMAL)
      {
         glPushMatrix();         // VIEW (ABOVE / BELOW)
            glTranslatef(-0.6f, 0.83f, 0.0f);
            view.SetForegroundColor(CYAN);
            view.Draw(m_CStr_nacws_display_view, true, 0.0f, FONT_FLAGS);
         glPopMatrix();
      }
   }



   glPushMatrix();           // "TCAS" / "TA ONLY" / "TEST" / "TCAS STBY"
      glTranslatef(-0.7f, -0.62f, 0.0f);
      tcasFlag.Draw(m_Cstr_tcas_flag, true, 0.0f, FONT_FLAGS);
   glPopMatrix();

   if (m_long_tcas_flag != TCAS_TEST    &&
       m_long_tcas_flag != TCAS_TA_ONLY   )
   {
      glPushMatrix();
         glTranslatef(0.0f, 0.31f, 0.0f);    // "VERTICAL SPEED"
         verticalSpeed_title.SetForegroundColor(CYAN);
         verticalSpeed_title.Draw(m_verticalSpeed_title,   true, 0.0f, FONT_VSI_LEGEND);
      glPopMatrix();
      glPushMatrix();
         glTranslatef(0.0f, -0.23f, 0.0f);  // "1000 FPM"
         fpmX1000_title.SetForegroundColor(CYAN);
         fpmX1000_title.Draw(fpm_x1000, true, 0.0f, FONT_1000FPM);
      glPopMatrix();
   }
   else // DISPLAY THE TAS SYMBOLOGY
   {

      // Set up stencil to keep drawing inside the range circle
      glClearStencil(1);
      glClear(GL_STENCIL_BUFFER_BIT);
      glStencilFunc(GL_ALWAYS, 0, 0);
      glStencilOp(GL_REPLACE, GL_REPLACE, GL_REPLACE);
      glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
      gluQuadricDrawStyle(m_pquad, GLU_FILL);
      glRGB(0, 0, 0);
      gluDisk(m_pquad, 0,   0.679f,   30, 1);
      glStencilFunc(GL_NOTEQUAL, 1, 1);
      glStencilOp(GL_KEEP, GL_KEEP, GL_KEEP);


      const float ac_pos = -0.22f;           // where the aircraft symbol is on the display (y position)
      const float range_ring_mult = 0.36f;   // how the range ring scale maps to the display

      glPushAttrib(GL_POLYGON_BIT);      
      glEnable(GL_POLYGON_SMOOTH);
      glPushMatrix();         // AIRCRAFT SYMBOL
         glTranslatef(0.0f, ac_pos, 0.0f);
         aircraftSymbol.Draw(m_Cstr_ac_symbol, true, 0.0f, 1.0f);
      glPopMatrix();
      glPopAttrib();

      float aspect_scale = (m_exPtLowerRight.X() - m_exPtUpperLeft.X()) / 200.0f;  // RANGE RING DOTS
      glEnable(GL_POINT_SMOOTH);
      glPointSize(3.0f * aspect_scale);
      glPushMatrix();
         glTranslatef(0.0f, ac_pos, 0.0f);

         float range_multiplier = 5.0 / m_float_Nacws_Display_Range;
         glScalef(range_multiplier, range_multiplier, 1.0f);

         if (m_float_Nacws_Display_Range < 40.0f)  // only display dots below 40 nm
         {
            glPushMatrix();
               glScalef(range_ring_mult, range_ring_mult, 1.0f);
               glColor3f(CYAN);
               glBegin(GL_POINTS);                // precomputed sin/cos table for range dots
                  glVertex2f( 0.00f,  1.00f);
                  glVertex2f( 0.50f,  0.87f);
                  glVertex2f( 0.87f,  0.50f);
                  glVertex2f( 1.00f,  0.00f);

                  if (m_float_Nacws_Display_Range > 3.0f)
                  {
                     glVertex2f( 0.87f, -0.50f);
                     glVertex2f( 0.50f, -0.87f);
                     glVertex2f( 0.00f, -1.00f);
                     glVertex2f(-0.50f, -0.87f);
                     glVertex2f(-0.87f, -0.50f);
                  }

                  glVertex2f(-1.00f,  0.00f);
                  glVertex2f(-0.87f,  0.50f);
                  glVertex2f(-0.50f,  0.87f);
               glEnd();
            glPopMatrix();
         }

         for (int z = 0; z < MAX_NACWS_THREATS; z++)
         {
            if (!m_nacws_list[z].active) continue;

            float tgt_bearing     = m_nacws_list[z].bearing;
            float tgt_range       = m_nacws_list[z].range;
            float tgt_delta_alt   = m_nacws_list[z].delta_altitude;
            int   climb_descend   = m_nacws_list[z].climb_descend;
            int   disp_type       = m_nacws_list[z].tgt_type;

            if ((disp_type != TGT_TA        ) &&   // reject filtered traffic except TA's
                (disp_type != TGT_TA_NB     ) &&
                (!filteredGood(tgt_delta_alt)))
               continue;


            if (disp_type == TGT_TA)   // Keep all TA threats on-screen
            {
               double range_limit_coefficient = 0.25*(cos(fabs(tgt_bearing)*DEG_TO_RAD) - 0.12 * fabs(sin(tgt_bearing*DEG_TO_RAD)) + 3.0);
               double range_limit             = m_float_Nacws_Display_Range * range_limit_coefficient;

               if (tgt_range > range_limit) 
                  tgt_range = range_limit;
            }

            glPushMatrix();
               glRotatef(-tgt_bearing, 0.0f, 0.0f, 1.0f);
               glTranslatef(0.0f, tgt_range / 2.0 * range_ring_mult, 0.0f);
               CString CStr_symbol, CStr_alt, CStr_trend;
               COpenGLtext altSymbol;

               if (tgt_delta_alt > 0)
                  CStr_alt.Format("+%02.0f", ROUND(tgt_delta_alt, 0.01) / 100.0f);
               else
                  CStr_alt.Format("-%02.0f", ROUND(fabs(tgt_delta_alt), 0.01) / 100.0f);

               if (climb_descend)
               {
                  CStr_trend.Format("%c", climb_descend < 1 ? DOWNARROW : UPARROW);
               }

               altSymbol.SetForegroundColor(WHITE);

               if (disp_type == TGT_TA)  // TA ALERT
               {
                  glColor4ub(AMBER, 255);
                  glPointSize(10.0f * aspect_scale);
                  glBegin(GL_POINTS);
                     glVertex3f(0.0f, 0.0f, 0.0f);
                  glEnd();
               altSymbol.SetForegroundColor(AMBER);
               symbol.SetForegroundColor(AMBER);
               }
               else if (disp_type == TGT_OTHER)    // Non-threat
               {
                  symbol.SetForegroundColor(WHITE);
                  CStr_symbol.Format("%c", DIAMOND);
                  glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
                  symbol.Draw(CStr_symbol, true, tgt_bearing, 2.0f/range_multiplier, 2.0f); // Draw the symbol.
               }
               else if (disp_type == TGT_PROX)
               {
                  symbol.SetForegroundColor(WHITE);
                  CStr_symbol.Format("%c", FILLED_DIAMOND);             // Proximate traffic 
                  symbol.Draw(CStr_symbol, true, tgt_bearing, 2.0f/range_multiplier, 1.0f); // Draw the symbol.
               }

               if (disp_type == TGT_OTHER || disp_type == TGT_PROX || disp_type == TGT_TA)
               {
                  glPushMatrix();
                     glRotatef(tgt_bearing, 0.0f, 0.0f, 1.0f);  // draw altitude above or below the symbol
                     glTranslatef(0.0f, tgt_delta_alt > 0.0f ? 0.12f/range_multiplier : -0.12f/range_multiplier, 0.0f);
                     altSymbol.Draw(CStr_alt, true, 0.0f, 2.0f/range_multiplier, 1.5f);
                  glPopMatrix();

                  if (climb_descend)
                  {  
                     glPushMatrix();
                        glRotatef(tgt_bearing, 0.0f, 0.0f, 1.0f);  // the climbing or descending symbol
                        glTranslatef(0.10f/range_multiplier, 0.0f, 0.0f);
                        altSymbol.Draw(CStr_trend, true, 0.0f, 1.8f/range_multiplier, 2.0f);
                     glPopMatrix();   
                  }
               }

            glPopMatrix();
               if (disp_type == TGT_TA_NB) // Traffic with no bearing;
               {
                  glPushMatrix();   // display non-bearing traffic advisories
                     CString CStr_nbta;
                     CStr_nbta.Format("%02.1fNM/",tgt_range);
                     CStr_nbta = CStr_nbta + CStr_alt + CStr_trend;
                     glTranslatef(0.0f, (ac_pos - 0.030f)/range_multiplier , 0.0f);

                     if (disp_type == TGT_TA_NB)
                        symbol.SetForegroundColor(AMBER);
                     else
                        symbol.SetForegroundColor(WHITE);

                     symbol.Draw(CStr_nbta, true, 0.0f, 1.5f/range_multiplier, 1.3f);
                  glPopMatrix();
               }
         }

      glPopMatrix();



      glStencilFunc(GL_ALWAYS, 0, 0);
      glStencilOp(GL_KEEP, GL_KEEP, GL_KEEP);
   }

   if (m_vsi_valid)
   {
      glPushMatrix();         // NEEDLE
         glRotatef(-m_float_verticalSpeed_angle, 0.0f, 0.0f, 1.0f);
         verticalSpeedNeedle.Draw(m_verticalSpeedNeedle, true, 0.0f, 1.0f);
      glPopMatrix();
   }
   else  
   {
      COpenGLtext vsi_invalid; // "VSI" FLAG
      glPushMatrix();
         glTranslatef(0.65f, -0.62f, 0.0f);
         vsi_invalid.SetForegroundColor(AMBER);
         vsi_invalid.Draw((CString)" \nVSI", true, 0.0f, FONT_FLAGS);
      glPopMatrix();
   }


   if (!m_bool_Power)         // SHADING FOR POWER "OFF"
   {
      glColor4ub(WHITE, 64);
      glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
         glBegin(GL_QUADS);
            glVertex2f(-1.0f, 1.0f);
            glVertex2f( 1.0f, 1.0f);
            glVertex2f( 1.0f, -1.0f);
            glVertex2f(-1.0f, -1.0f);
         glEnd();
      glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
   }


   glPopAttrib();

   EndDraw();
}

CWidget* CHoneywellTAS::CreateObject()
{
   return new CHoneywellTAS();
}

void CHoneywellTAS::ChangeValue(const CString& rstrElementVar, CChangeValue* pCV)
{
   if (pCV  == NULL)
   {
      return;
   }
   CVariant *pVariant   =  pCV->Variant();

   if (rstrElementVar   == "vertical_speed")
   {
      m_float_vertical_speed_cv  =  *pVariant;
   }
   else if (rstrElementVar   == "Vsi_Power")
   {
      m_bool_Power_cv         =  *pVariant;
   }
   else if (rstrElementVar == "Nacws_Display_Range")
   {
      long range_and_view = *pVariant;

      m_long_View_cv = range_and_view;

   }
   else if (rstrElementVar == "Nacws_Status")
   {
      m_long_nacws_status_cv = *pVariant;
   }
//   else if (rstrElementVar == "Nacws_Test")
//   {
//      m_bool_nacws_test_cv = *pVariant;
//   }
   else if (rstrElementVar == "Nacws_Intr_Range_1")
   {
      m_nacws_list[0].range_cv = *pVariant;
   }
   else if (rstrElementVar == "Nacws_Intr_Range_2")
   {
      m_nacws_list[1].range_cv = *pVariant;
   }
   else if (rstrElementVar == "Nacws_Intr_Rel_Alt_1")
   {
      m_nacws_list[0].delta_altitude_cv = *pVariant;
   }
   else if (rstrElementVar == "Nacws_Intr_Rel_Alt_2")
   {
      m_nacws_list[1].delta_altitude_cv = *pVariant;
   }
   else if (rstrElementVar == "Nacws_Intr_Brg_1")
   {
      m_nacws_list[0].bearing_cv = *pVariant;
   }
   else if (rstrElementVar == "Nacws_Intr_Brg_2")
   {
      m_nacws_list[1].bearing_cv = *pVariant;
   }
   else if (rstrElementVar == "Nacws_Intr_Active_1")
   {
      m_nacws_list[0].active_cv = *pVariant;
   }
   else if (rstrElementVar == "Nacws_Intr_Active_2")
   {
      m_nacws_list[1].active_cv = *pVariant;
   }
   else if (rstrElementVar == "Nacws_Intr_Disp_Type_1")
   {
      m_nacws_list[0].tgt_type_cv = *pVariant;
   }
   else if (rstrElementVar == "Nacws_Intr_Disp_Type_2")
   {
      m_nacws_list[1].tgt_type_cv = *pVariant;
   }
   else if (rstrElementVar == "Tas_Range_Sel")
   {
      if ((_FSI_STL::string)*pVariant == "UP")
      {
         m_range_internal = __min(m_range_internal++, Range_Max);
      }
      else if ((_FSI_STL::string)*pVariant == "DN")
      {
         m_range_internal = __max(m_range_internal--, Range_Min);
      }

      switch(m_range_internal)
      {
         case 0:
            m_float_Nacws_Display_Range_cv   =   3.0f;  // 1.0 nm N/A in Honeywell TAS.
            break;
         case Range_3NM:
            m_float_Nacws_Display_Range_cv   =   3.0f;
            break;
         case Range_5NM:
            m_float_Nacws_Display_Range_cv   =   5.0f;
            break;
         case Range_10NM:
            m_float_Nacws_Display_Range_cv   =  10.0f;
            break;
         case Range_20NM:
            m_float_Nacws_Display_Range_cv   =  20.0f;
            break;
         case Range_40NM:
            m_float_Nacws_Display_Range_cv   =  40.0f;
            break;
         default:
            m_float_Nacws_Display_Range_cv   =  40.0f;
      }
   }


}

/////////////////////////////////////////////////////////////////////////////
//
// bool CHoneywellTAS::UpdateRenderVariables()
//
// Inputs           : None.
//
// Return Values    : success value of the update of the rendering variables.
//
// Date             : 16 June 1999
//
// Engineer         : Billy Baker
//
// Description      : Override of the parent's UpdateRenderVariables.  
//                    If the class is being destroyed while in this method,
//                    then the return value should be false. This method is
//                    called by the COMMS layer before it tells the mainframe
//                    that a screen update is needed.
//
/////////////////////////////////////////////////////////////////////////////

bool CHoneywellTAS::UpdateRenderVariables()
{
   // Determine the rate at which this function is getting called.
   LARGE_INTEGER  theFrequency;
   QueryPerformanceFrequency(&theFrequency);
   static   LARGE_INTEGER  theStartCount;
   static   bool  one_time_only  =  true;
   LARGE_INTEGER  theEndCount, diffCount;
   float    iteration_rate;

   if (one_time_only)
   {
      QueryPerformanceCounter(&theStartCount);
      one_time_only  =  false;
   }
   
   QueryPerformanceCounter(&theEndCount);
   diffCount.QuadPart   =  theEndCount.QuadPart -  theStartCount.QuadPart;
   iteration_rate       =  (float)((float)diffCount.QuadPart /  (float)theFrequency.QuadPart);
   QueryPerformanceCounter(&theStartCount);


   

   static   float lpv_output        =  0.0f;

   if (iteration_rate   >= 1.0f)    // If for some reason we haven't been called for more than a second, just snap to desired value.
   {
      m_float_vertical_speed  =  m_float_vertical_speed_cv;
      lpv_output              =  m_float_vertical_speed_cv;
   }
   else
   {
      float slew_limit_per_iteration   =  440.0f   *  iteration_rate;
      float fraction                   =  m_float_vertical_speed_cv  *  iteration_rate;
      float calculated_filter_output   =  lpv_output  *  (1.0f -  iteration_rate)   +  fraction;

      if (calculated_filter_output  -  lpv_output  >  slew_limit_per_iteration)  
         lpv_output  += slew_limit_per_iteration;
      else if (lpv_output  -  calculated_filter_output   >  slew_limit_per_iteration)  
         lpv_output  -= slew_limit_per_iteration;
      else
         lpv_output  =  calculated_filter_output;

      m_float_vertical_speed  =  lpv_output;
   }

   const float rotation_offset = -90.0f;

   if (m_float_vertical_speed < -6000.0f)
      m_float_verticalSpeed_angle = -172.5f + rotation_offset;

   else if (m_float_vertical_speed > 6000.0f)
      m_float_verticalSpeed_angle = 172.5f + rotation_offset;

   else if (( -1000.0f <= m_float_vertical_speed) && (m_float_vertical_speed <= 1000.0f))
      m_float_verticalSpeed_angle = (67.5f * m_float_vertical_speed / 1000.0f) + rotation_offset;

   else
      m_float_verticalSpeed_angle = _copysign(1.0, m_float_vertical_speed) *
                   (67.5f + 58.6016 * log(fabs(m_float_vertical_speed / 1000.0))) + rotation_offset;


   m_bool_Power   =  m_bool_Power_cv;
   m_long_View    =  m_long_View_cv;
   m_vsi_valid    =  !(m_long_nacws_status_cv & 0x4); // bit 3 indicates failure

   m_CStr_nacws_display_range.Format(_T("RNG %.0f"), m_float_Nacws_Display_Range);  // RANGE

   switch(m_long_View)  // ABOVE / BELOW
   {
      case VIEW_ABOVE:  m_CStr_nacws_display_view = "ABOVE";
                        break;

      case VIEW_BELOW:  m_CStr_nacws_display_view = "BELOW";
                        break;
      
      default:          m_CStr_nacws_display_view = "";
   }


   m_long_nacws_status = m_long_nacws_status_cv & 0x3;     
   m_bool_nacws_test   = (m_long_nacws_status == TCAS_TEST) || m_bool_nacws_test_cv; ///////////// TEST MODE
   m_float_Nacws_Display_Range = m_float_Nacws_Display_Range_cv;

   static bool host_initiated_test;

   if (m_bool_nacws_test)        // 7 second timer for test mode
   {
      m_timeStart = time(NULL);
      //if (m_long_nacws_status == TCAS_TEST)
      //   host_initiated_test = true;
      //else
         host_initiated_test = false;
   }

   if (m_timeStart != 0)  // timer running?
   {
      if ((time(NULL) - m_timeStart >= 7) || // 7 sec timeout for button sense
          (host_initiated_test && (m_long_nacws_status != TCAS_TEST))) // instant off for host control
      {
         m_timeStart = 0; 
         memset(m_nacws_list, 0, sizeof m_nacws_list);
      }
      else
      {
         m_bool_nacws_test = true;

         m_nacws_list[0].bearing          =     30.0f;
         m_nacws_list[0].range            =      3.8f;
         m_nacws_list[0].delta_altitude   =  -1000.0f;
         m_nacws_list[0].active           =      true;
         m_nacws_list[0].climb_descend    =        -1;
         m_nacws_list[0].tgt_type         =  TGT_PROX;

         m_nacws_list[1].bearing          =    330.0f;
         m_nacws_list[1].range            =      3.8f;
         m_nacws_list[1].delta_altitude   =  +1000.0f;
         m_nacws_list[1].active           =      true;
         m_nacws_list[1].climb_descend    =         0;
         m_nacws_list[1].tgt_type         = TGT_OTHER;

         m_nacws_list[2].bearing          =    270.0f;
         m_nacws_list[2].range            =      2.0f;
         m_nacws_list[2].delta_altitude   =   -200.0f;
         m_nacws_list[2].active           =      true;
         m_nacws_list[2].climb_descend    =        +1;
         m_nacws_list[2].tgt_type         =    TGT_TA;
      }
   
   }
   else
   {
      m_float_Nacws_Display_Range   =  m_float_Nacws_Display_Range_cv;

      for (int i=0; i<MAX_NACWS_THREATS; i++)
      {
         m_nacws_list[i].active         = m_nacws_list[i].active_cv;          // false..true
         m_nacws_list[i].bearing        = m_nacws_list[i].bearing_cv;         // 0.0..360.0
         m_nacws_list[i].delta_altitude = m_nacws_list[i].delta_altitude_cv;  // -10000..10000
         m_nacws_list[i].range          = m_nacws_list[i].range_cv;           // 0.01..1000.0
         m_nacws_list[i].tgt_type       = m_nacws_list[i].tgt_type_cv & 0x07; // low 3 bits are target type
         // map bits 4 and 5: 00 = level (0 is output), 01=climb (1 is output), 10 = descend (-1 is output)
         int climb_descend_bits = (m_nacws_list[i].tgt_type_cv >> 3) & 0x03;
         m_nacws_list[i].climb_descend  = climb_descend_bits == 2 ? -1 : climb_descend_bits;
      }
   }

   m_long_tcas_flag = m_bool_nacws_test ? TCAS_TEST : m_long_nacws_status;

   switch(m_long_tcas_flag)
   {
      case TCAS_STBY   :   m_Cstr_tcas_flag = " TCAS\n   STBY";
                           tcasFlag.SetForegroundColor(CYAN);
                           break;

      case TCAS_TEST   :   m_Cstr_tcas_flag = " \n   TEST";
                           tcasFlag.SetForegroundColor(AMBER);
                           break;

      case TCAS_TA_ONLY:   m_Cstr_tcas_flag = " TA\n   ONLY";
                           tcasFlag.SetForegroundColor(CYAN);
                           break;
      case TCAS_INVALID:
      default          :   m_Cstr_tcas_flag = " \n   TCAS";
                           tcasFlag.SetForegroundColor(AMBER);
   }

   return CWidget::UpdateRenderVariables();
}

bool CHoneywellTAS::filteredGood(float delta_alt)
{
   switch (m_long_View)
   {
      case VIEW_NORMAL: return (fabs(delta_alt) <= 2700.0);
      case VIEW_ABOVE:  return (-2700.0 <= delta_alt && delta_alt <= 9000.0);
      case VIEW_BELOW:  return (-9000.0 <= delta_alt && delta_alt <= 2700.0);
      default:          return true;
   }
}
  

