/////////////////////////////////////////////////////////////////////////////
//
//           F L I G H T S A F E T Y   I N T E R N A T I O N A L
//                     Simulation Systems Division
//                      2700 North Hemlock Circle
//                     Broken Arrow, Oklahoma 74012
//                          (918) 259-4000
/////////////////////////////////////////////////////////////////////////////
//
// DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
// Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
// technical or operational data or information from automatic dissemination
// under the International Exchange Program or by other means.  This protection
// covers information required solely for administrative or operational
// purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
//
// WARNING:  This document contains technical data whose export is restricted
// by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
// Executive Order 12470.  Violation of these export control laws is subject
// to severe criminal penalties.  Dissemination of this document is controlled
// under DOD Directive 5230.25
//
/////////////////////////////////////////////////////////////////////////////
//
//
// Filename         : ZCTProtocol.cpp
//
// Date             : 17 August 1999
//
// Engineer         : Billy Baker
//
// Revision         : $Revision$
//
// Description      : ZCTProcotol.cpp contains the implementation of the 
//                    CZCTProcotol class. This class is a modification 
//                    of the ioshostcomms code used with the ZCT 
//                    developed control loading diagnostic application 
//                    (iosrt) which was also used as an IOS.  The 
//                    modifications were made to make the code 
//                    object-oriented in the sense that it is derived 
//                    from CProtocol and makes some use of the 
//                    CGenericHostSockets class.
//                    
//                    Other modifications include a fixed value for
//                    iospSmsBase (0xF4700000) since no shared memory
//                    is used.  This value is determined by the actual
//                    hardware.  A number of variables were also deleted
//                    since they are in the CGenericHostSockets class.
//                    Since the ZCT does not use fixed packet sizes, the
//                    SendData and ReceiveData methods of CGenericHostSockets
//                    is not used to send and receive data.                    
//
// Classification   : UNCLASSIFIED
//
// Requirements     : None.
//
// Components Used  : CProtocol, CGenericHostSockets.
//
// Operational 
//    Restrictions  : Machine dependencies/restrictions
//                        None.
//                    Design dependencies/restrictions
//                        None.
//                    Operations containing dependencies/restrictions
//                        None.
//                    Compiler dependencies/restrictions
//                        None.
//                    Other conditions for proper execution
//                        None.
//
// Environment      : Operating system(s) - Microsoft Windows NT 4.0 with
//                                              NT service pack 3, 4, or 5
//                                          Microsoft Windows NT 2000
//                                          VxWorks
//
//                    Compiler(s) - Visual C++ 6.0
//                                  Tornado 1.0.1
//
//                    Architechure(s) - Intel Pentium, Pentium II
/////////////////////////////////////////////////////////////////////////////

/////////////////////////////////////////////////////////////////////////////
//
//                              R e v i s i o n   H i s t o r y
//
/////////////////////////////////////////////////////////////////////////////
// $Log$                                                                   //
/////////////////////////////////////////////////////////////////////////////
#include "HostComms.h"
#include "GenericHostSockets.h"
#include "Protocol.h"
#include "ZCTProtocol.h"


//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

CZCTProtocol::CZCTProtocol()
{
    m_pSockets = new CGenericHostSockets(ZCT_PORT, 
                                     1,
                                     //ZCT_MAX_RECEIVE_BUFFER, 
                                     ZCT_RECEIVE_BUFFER_CACHE, 
                                     ZCT_MAX_SEND_BUFFER, 
                                     ZCT_SEND_BUFFER_CACHE);

    memset(&iosCommandReceive, 0, sizeof(iosRtCommand));

    iosbCommandReceiveRequested         = FALSE;
    iosuHostTick                        = 0;
    iosnReceiveDataCount                = 0;
    iospReceiveDataPointer              = NULL;
    iosnReadPointers                    = 0;
    iosnSendBufferCount                 = 0;
    iosnSendBufferPositionIn            = 0;
    iosnSendBufferPositionOut           = 0;
    iosnSetReadPointersTag              = 0;
    iospSmsBase                         = (char*)0xF4700000;
    iosnWritePointers                   = 0;
    m_bCommandComplete                  = false;

    memset(ioshpPlotBuffers,      0, MAX_PLOT_BUFFERS * sizeof(HostPlot));
    memset(iospReadPointers,      0, MAX_COMMAND_DATA * sizeof(int));
    memset(iospReadPointerTypes,  0, MAX_COMMAND_DATA * sizeof(int));
    memset(iospWritePointers,     0, MAX_COMMAND_DATA * sizeof(int));
    memset(iospWritePointerTypes, 0, MAX_COMMAND_DATA * sizeof(int));
}

CZCTProtocol::~CZCTProtocol()
{
    // The base class will take care of deleting m_pSockets.
}

/*
 ******************************************************************************
 *																			  *
 *                          	 Command Functions	                          *
 *																			  *
 ******************************************************************************
 */


// iosCommandNull
// Process the IOSRT_COMMAND_NULL command
// Entry   : nothing
// Returns : nothing
void CZCTProtocol::iosCommandNull()
{
	#ifdef PRINT_STATISTICS
		printf("ZCTProtocol::iosCommandNull :  Received IOSRT_COMMAND_NULL.\n");
	#endif
}

// iosCommandRead
// Process the IOSRT_COMMAND_READ command
// Entry   : nothing
// Returns : nothing
void CZCTProtocol::iosCommandRead()
{
	// Obtain the number of dwords iosRt is requesting to read.
	int	nCount = ntohl(iosCommandReceive.Command.Read.nCount);

	#ifdef PRINT_STATISTICS
		printf("ZCTProtocol::iosCommandRead :  Received IOSRT_COMMAND_READ of "\
               "%d dword(s).\n", nCount);
	#endif

	// Validate the command.
	if((iosnReadPointers == 0) && (nCount != 0))
	{
		// Error, iosRt requested read but it has yet to supply the pointers.
		// In response, send iosRt the HOST_COMMAND_NO_READ_POINTERS command.
		HostCommand		iosCommandSend;

		iosCommandSend.nCommand = htonl(HOST_COMMAND_NO_READ_POINTERS);
		iosCommandSend.nCommandSize = htonl(HOST_COMMAND_HEADER_SIZE);

		iosSendCommand(& iosCommandSend);

		#ifdef PRINT_STATISTICS
			printf("ZCTProtocol::iosCommandRead :  Sent "\
                   "HOST_COMMAND_NO_READ_POINTERS.\n");
		#endif
	}
	else if(nCount > MAX_COMMAND_DATA)
	{
		// Error, iosRt requested read of more data than allowed.
		HostCommand		iosCommandSend;

		iosCommandSend.nCommand = htonl(HOST_COMMAND_READ_OUT_OF_RANGE);
		iosCommandSend.nCommandSize = htonl(HOST_COMMAND_HEADER_SIZE);
		
		iosSendCommand(& iosCommandSend);

		#ifdef PRINT_STATISTICS
			printf("ZCTProtocol::iosCommandRead :  Sent "\
                   "HOST_COMMAND_READ_OUT_OF_RANGE %d.\n", nCount);
		#endif
	}
	else
	{
        // Valid command.  Read the data at the pointer locations specified in the
        // iospReadPointers array previously received from iosRt and send it to iosRt.
        // The type of each data element to send is specified in the pReadPointerType
        // array previously received from iosRt during the set read pointers command.
		HostCommand		iosCommandSend;
		unsigned int *	pDestination = iosCommandSend.Command.ReadReturn.dwData;
		unsigned int *	pEnd = (unsigned int *)& iosCommandSend.Command.ReadReturn.dwData[nCount];
		int *			pSource = iospReadPointers;
		int				nType;
		int *			pTypes = iospReadPointerTypes;

		while(pDestination < pEnd)
		{
			// Determine the read operation on the data type.
			nType = * pTypes ++;

			if(nType & PLOT_BIT)
			{
				// Data type is being plotted, get the plot buffer.
				HostPlot*	pPlot = iosPlotFind(nType);

				// Check the plot element buffer count.
				if(pPlot->nCount)
				{
					// Plot buffer has recorded elements, send them.
					HostCommand		iosCommandPlot;

					iosCommandPlot.nCommand = htonl(HOST_COMMAND_PLOT);
					iosCommandPlot.nCommandSize = htonl(HOST_COMMAND_HEADER_SIZE +
														HOST_COMMAND_PLOT_HEADER_SIZE +
														(pPlot->nCount * (pPlot->nTypeDWords << 2)));
					iosCommandPlot.Command.Plot.pAddress = (void *)(htonl((unsigned long)pPlot->pAddress));
					iosCommandPlot.Command.Plot.nCount = htonl(pPlot->nCount);
					iosCommandPlot.Command.Plot.nType = htonl(pPlot->nType);
					iosCommandPlot.Command.Plot.nTypeDWords = htonl(pPlot->nTypeDWords);
					iosCommandPlot.Command.Plot.uHostTick = htonl(pPlot->uHostTick);
					switch(pPlot->nType)
					{
						case IOSRT_DOUBLE:
						{
							// Double copy is 64 bits.
							int		nCount;
							for(nCount = 0; nCount < (pPlot->nCount << 1); nCount += 2)
							{
								int nTemp = htonl(pPlot->nArray[nCount]);
								iosCommandPlot.Command.Plot.nArray[nCount] = htonl(pPlot->nArray[nCount + 1]);
								iosCommandPlot.Command.Plot.nArray[nCount + 1] = nTemp;
							}
						}
						break;

						default:
						{
							// Default copy is 32 bits.
							int		nCount;
							for(nCount = 0; nCount < pPlot->nCount; nCount ++)
								iosCommandPlot.Command.Plot.nArray[nCount] = htonl(pPlot->nArray[nCount]);
						}
						break;
					};

					iosSendCommand(& iosCommandPlot);

					#ifdef PRINT_STATISTICS
						printf("ZCTProtocol::iosCommandRead :  Sent "\
                               "HOST_COMMAND_PLOT of %d samples of "\
                               "address %p.\n", pPlot->nCount, 
                                                pPlot->pAddress, 
                                                pPlot->nTypeDWords);
					#endif


					// Then reset the element count.
					pPlot->nCount = 0;
				}

				// Next source pointer.
				pSource ++;
			}
			else
			{
				// Data type is being read.
				nType &= DATA_TYPE_BITS;
 
				switch(nType)
				{
					case IOSRT_CHAR:
					{
						int *	pThisSource = (int *) * pSource ++;
						if(pThisSource != NULL)
							* (char *)pDestination = *(char*)pThisSource;

						pDestination += 1;
					}
					break;

					case IOSRT_SHORT:
					case IOSRT_USHORT:
					{
						int *	pThisSource = (int *) * pSource ++;
						if(pThisSource != NULL)
							* (short *)pDestination = (short)htonl(* pThisSource);

						pDestination += 1;
					}
					break;

					case IOSRT_INT:
					case IOSRT_UINT:
					case IOSRT_LOGICAL:
					case IOSRT_FLOAT:
					{
						int *	pThisSource = (int *) * pSource ++;
						if(pThisSource != NULL)
							* (int *)pDestination = htonl(* pThisSource);

						pDestination += 1;
					}
					break;

					case IOSRT_DOUBLE:
					{
						unsigned long * pThisSource = (unsigned long *)* pSource ++;
						if(pThisSource != NULL)
						{
							unsigned long uTemp = htonl(pThisSource[0]);
							((unsigned long *)pDestination)[0] = htonl(pThisSource[1]);
							((unsigned long *)pDestination)[1] = uTemp;
						}

						pDestination += 2;
					}
					break;
				}
			}
		}

		// Send the data to iosRt.
		iosCommandSend.nCommand = htonl(HOST_COMMAND_READ_RETURN);
		iosCommandSend.nCommandSize = htonl(HOST_COMMAND_HEADER_SIZE + 
											HOST_COMMAND_READ_RETURN_HEADER_SIZE +
											(sizeof(iosCommandSend.Command.ReadReturn.dwData[0]) * nCount));
		iosCommandSend.Command.ReadReturn.nCount = htonl(nCount);
		iosCommandSend.Command.ReadReturn.nTag = htonl(iosnSetReadPointersTag);

		iosSendCommand(& iosCommandSend);
	}
}

// iosCommandSetReadPointers
// Process the IOSRT_COMMAND_SET_READ_POINTERS command
// Entry   : nothing
// Returns : nothing
void CZCTProtocol::iosCommandSetReadPointers()
{
    int				nPointers = ntohl(iosCommandReceive.Command.SetReadPointers.nPointers);

    int *			pDestination = iospReadPointers;
    int *			pEnd = & iospReadPointers[nPointers];
    iosRtPointer *	pSource = iosCommandReceive.Command.SetReadPointers.pPointers;
    int *			pTypes = iospReadPointerTypes;

    // Validate the command.
    if(nPointers > MAX_COMMAND_DATA)
    {
	    // Error, iosRt wrote too many read pointers.
	    HostCommand		iosCommandSend;

	    iosCommandSend.nCommand = htonl(HOST_COMMAND_TOO_MANY_READ_POINTERS);
	    iosCommandSend.nCommandSize = htonl(HOST_COMMAND_HEADER_SIZE);
	    
	    iosSendCommand(& iosCommandSend);
    }

    else
    {
        // Valid command.
        #ifdef PRINT_STATISTICS
            printf("ZCTProtocl::iosCommandSetReadPointers :  Received "\
                    "IOSRT_COMMAND_SET_READ_POINTERS of %d "\
                    "pointer(s).\n", nPointers);
        #endif

        iosnSetReadPointersTag = ntohl(iosCommandReceive.Command.SetReadPointers.nTag);

        memset(iospReadPointers, 0, sizeof(iospReadPointers));
        iosPlotInitialize();

        while(pDestination < pEnd)
        {
            int	pType = ntohl(pSource->nPointerType);

            if(pType & PLOT_BIT)
            {
                iosPlotAdd((& iospSmsBase[ntohl(pSource->pPointer)]), pType);
            }

            * pTypes ++ = pType;
            * pDestination ++ = (int)(& iospSmsBase[ntohl(pSource ++->pPointer)]);
        }

        iosnReadPointers = nPointers;
    }
}

// iosCommandSetWritePointers
// Process the IOSRT_COMMAND_SET_WRITE_POINTERS command
// Entry   : nothing
// Returns : nothing
void CZCTProtocol::iosCommandSetWritePointers()
{
    int				nPointers = ntohl(iosCommandReceive.Command.SetWritePointers.nPointers);

    int *			pDestination = iospWritePointers;
    int *			pEnd = & iospWritePointers[nPointers];
    iosRtPointer *	pSource = iosCommandReceive.Command.SetWritePointers.pPointers;
    int *			pTypes = iospWritePointerTypes;

    // Validate the command.
    if(nPointers > MAX_COMMAND_DATA)
    {
        // Error, iosRt wrote too many write pointers.
        HostCommand		iosCommandSend;

        iosCommandSend.nCommand = htonl(HOST_COMMAND_TOO_MANY_WRITE_POINTERS);
        iosCommandSend.nCommandSize = htonl(HOST_COMMAND_HEADER_SIZE);

        iosSendCommand(& iosCommandSend);
    }
    else
    {
        // Valid command.
        #ifdef PRINT_STATISTICS
            printf("ZCTProtocol::iosCommandSetWritePointers :  Received "\
                    "IOSRT_COMMAND_SET_WRITE_POINTERS of %d pointer(s).\n",
                    nPointers);
        #endif

        memset(iospWritePointers, 0, sizeof(iospWritePointers));

        while(pDestination < pEnd)
        {
            * pTypes ++ = ntohl(pSource->nPointerType);

            #ifdef PRINT_STATISTICS
                printf("ZCTProtocol::iosCommandSetWritePointers :  Set "\
                        "write pointer %p.\n", 
                        & iospSmsBase[ntohl(pSource->pPointer)]);
            #endif

            * pDestination ++ = (int)(& iospSmsBase[ntohl(pSource ++->pPointer)]);
        }

        iosnWritePointers = nPointers;
    }
}

// iosCommandWrite
// Process the IOSRT_COMMAND_WRITE command
// Entry   : nothing
// Returns : nothing
void CZCTProtocol::iosCommandWrite()
{
	int	nCount = ntohl(iosCommandReceive.Command.Write.nCount);

	#ifdef PRINT_STATISTICS
		printf("ZCTProtocl::iosCommandWrite :  Received IOSRT_COMMAND_WRITE "\
               "of %d dword(s).\n", nCount);
	#endif

	// Validate the command.
	if((iosnWritePointers == 0) && (nCount != 0))
	{
		// Error, iosRt requested write but it has yet to supply the pointers.
		// In response, send iosRt the HOST_COMMAND_NO_WRITE_POINTERS command.
		HostCommand		iosCommandSend;

		iosCommandSend.nCommand = htonl(HOST_COMMAND_NO_WRITE_POINTERS);
		iosCommandSend.nCommandSize = htonl(HOST_COMMAND_HEADER_SIZE);
		
		iosSendCommand(& iosCommandSend);
	}
	else if(nCount > MAX_COMMAND_DATA)
	{
		// Error, iosRt requested write of more data than allowed.
		HostCommand		iosCommandSend;

		iosCommandSend.nCommand = htonl(HOST_COMMAND_WRITE_OUT_OF_RANGE);
		iosCommandSend.nCommandSize = htonl(HOST_COMMAND_HEADER_SIZE);

		iosSendCommand(& iosCommandSend);
	}
	else
	{
		// Valid command. Write the data received from iosRt to the locations specified
		// in the iospWritePointers array previously received from iosRt.  The type of
		// write to perform is specified in the pWritePointerType array previously received
		// from iosRt during the set write pointers command.
		int *			pDestination = (int *)iospWritePointers;
		int *			pEnd = (int *)& iospWritePointers[nCount];
		unsigned int *	pSource = iosCommandReceive.Command.Write.wrData;
		int *			pTypes = iospWritePointerTypes;

		while(pDestination < pEnd)
		{
			switch((* pTypes ++) & DATA_TYPE_BITS)
			{
				case IOSRT_CHAR:
				{
					int *	pThisDestination = (int *) * pDestination ++;

					#ifdef PRINT_STATISTICS
						printf("ZCTProtocol::iosCommandWrite :  Write char "\
                               "(logical) at %p, %02x.\n", 
                               (int *) pThisDestination, \
                               (char)ntohl(* pSource));
					#endif

					if(pThisDestination != NULL)
						* (char *)pThisDestination = *(char*)pSource;

					pSource += 1;
				}
				break;

				case IOSRT_SHORT:
				case IOSRT_USHORT:
				{
					int *	pThisDestination = (int *) * pDestination ++;

					#ifdef PRINT_STATISTICS
						printf("ZCTProtocol::iosCommandWrite :  Write short "\
                               "at %p, %04x.\n", (int *) pThisDestination, 
                               (short)ntohl(* pSource));
					#endif

					if(pThisDestination != NULL)
						* (short *)pThisDestination = (short)ntohl(* pSource);

					pSource += 1;
				}
				break;

				case IOSRT_INT:
				case IOSRT_UINT:
				case IOSRT_LOGICAL:
				case IOSRT_FLOAT:
				{
					int *	pThisDestination = (int *) * pDestination ++;

					#ifdef PRINT_STATISTICS
						printf("ZCTProtocol::iosCommandWrite :  Write int "\
                               "(uint, float) at %p, %04x.\n", 
                               (int *) pThisDestination, 
                               (int)ntohl(* pSource));
					#endif

					if(pThisDestination != NULL)
						* pThisDestination = ntohl(* pSource);

					pSource += 1;
				}
				break;

				case IOSRT_DOUBLE:
				{
					unsigned long * pThisDestination = (unsigned long *)* pDestination ++;

					#ifdef PRINT_STATISTICS
						printf("ZCTProtocol::iosCommandWrite :  Write double "\
                                "at %p, %08x.\n", (int *) pThisDestination, 
                                (double)ntohl(* pSource));
					#endif

					if(pThisDestination != NULL)
					{
						unsigned long uTemp = ntohl(((unsigned long *)pSource)[0]);
						pThisDestination[0] = ntohl(((unsigned long *)pSource)[1]);
						pThisDestination[1] = uTemp;
					}

					pSource += 2;
				}
				break;
			}
		}
	}
}

/*
 ******************************************************************************
 *                                                                            *
 *                              Plotting Functions                            *
 *                                                                            *
 ******************************************************************************
 */

// iosPlotAdd
// Add a plot variable
// Entry   : variable offset, variable type
// Returns : TRUE if successful, FALSE if not
void CZCTProtocol::iosPlotAdd(void * pAddress, int nType)
{
	// Retreive the buffer number from nType.
	int		nPlotBuffer = (nType >> PLOT_BUFFER_NUMBER_LSB) & (PLOT_BUFFER_NUMBER_BITS);

	// Initialize the buffer.
	ioshpPlotBuffers[nPlotBuffer].pAddress = pAddress;
	ioshpPlotBuffers[nPlotBuffer].nType = nType;
	switch(nType & DATA_TYPE_BITS)
	{
		case IOSRT_DOUBLE:
		{
			ioshpPlotBuffers[nPlotBuffer].nTypeDWords = 2;
		}
		break;

		default:
		{
			ioshpPlotBuffers[nPlotBuffer].nTypeDWords = 1;
		}
		break;
	}
	ioshpPlotBuffers[nPlotBuffer].nCount = 0;
}

// iosPlotData
// Acquire plot data
// Entry   : nothing
// Returns : nothing
void CZCTProtocol::iosPlotData()
{
	// Gather the data for all plot buffers.
	int		nCount;

	for(nCount = 0; nCount < MAX_PLOT_BUFFERS; nCount ++)
	{
		// Check this plot buffer.
		HostPlot* pPlot = & ioshpPlotBuffers[nCount];

		if((pPlot->pAddress) && ((pPlot->nCount) < MAX_PLOT_DATA))
		{
			// Plot buffer active and has not overflowed, write the tick value
			// if the first entry is being recorded.
			if(pPlot->nCount == 0)
				pPlot->uHostTick = iosuHostTick;

			// Then write the value.
			switch(pPlot->nType & DATA_TYPE_BITS)
			{
				case IOSRT_CHAR:
				{
					pPlot->nArray[pPlot->nCount ++] = (char) * (char *)pPlot->pAddress;
				}
				break;

				case IOSRT_SHORT:
				case IOSRT_USHORT:
				{
					pPlot->nArray[pPlot->nCount ++] = (short) * (short *)pPlot->pAddress;
				}
				break;

				case IOSRT_INT:
				case IOSRT_UINT:
				case IOSRT_LOGICAL:
				case IOSRT_FLOAT:
				{
					pPlot->nArray[pPlot->nCount ++] = (int) * (int *)pPlot->pAddress;
				}
				break;

				case IOSRT_DOUBLE:
				{
					((double *)pPlot->nArray)[pPlot->nCount ++] = (double) * (double *)pPlot->pAddress;
				}
				break;
			}
		}
	}
}

// iosPlotFind
// Return the address of a plot buffer
// Entry   : variable type
// Returns : pointer to HostPlot structure
HostPlot* CZCTProtocol::iosPlotFind(int nType)
{
	// Return a pointer to the plot buffer.
	return	& ioshpPlotBuffers[(nType >> PLOT_BUFFER_NUMBER_LSB) & (PLOT_BUFFER_NUMBER_BITS)];
}

// iosPlotInitialize
// Initialize for plotting.
// Entry   : nothing
// Returns : nothing
void CZCTProtocol::iosPlotInitialize()
{
	int		nCount;

	// Initialize all plot buffers.
	for(nCount = 0; nCount < MAX_PLOT_BUFFERS; nCount ++)
	{
		ioshpPlotBuffers[nCount].pAddress = NULL;
		ioshpPlotBuffers[nCount].nCount = 0;
	}
}

/*
 ******************************************************************************
 *                                                                            *
 *                              Send/Receive Functions                        *
 *                                                                            *
 ******************************************************************************
 */
 
// iosReceiveData
// Receive data from iosRt
// Entry   : nothing
// Returns : number of bytes remaining to be read
int CZCTProtocol::iosReceiveData()
{
    if (m_pSockets == NULL)
    {
        return 0;
    }

	while(iosnReceiveDataCount)
	{
		int		nBytesReceived;

		// Receive data.
        if((nBytesReceived = recv(m_pSockets->m_socketAccept, 
                                  iospReceiveDataPointer, iosnReceiveDataCount,
                                  0)) == SOCKET_ERROR)
		{
			// Receive failed, see why.
			int	nError;

			if((nError = GetLastError()) != WSAEWOULDBLOCK)
			{
				// Error during receive.

				switch(nError)
				{
					case WSAESHUTDOWN:
					case WSAECONNRESET:
					{
						#ifdef PRINT_STATISTICS
							printf("ZCTProtocol::iosReceiveData :  iosRt "\
                                   "disconnected.\n");
						#endif
					}
					break;

					default:
					{
						#ifdef PRINT_ERRORS
							printf("ZCTProtocol::iosReceiveData :  Receive "\
                                   "error %d.\n", nError);
						#endif
					}
					break;
				}

				// Close the socket.
				m_pSockets->CloseAcceptSocket();
			}

			// Either an error occured or the socket would block, break out
			// of the receive loop.
			break;
		}
		else if(nBytesReceived == 0)
		{
			// iosRt disconnected, close the socket.
            m_pSockets->CloseAcceptSocket();

			#ifdef PRINT_STATISTICS
				printf("ZCTProtocol::iosReceiveData :  iosRt disconnected.\n");
			#endif

			// Break out of the receive loop.
			break;
		}
		else
		{
			// Update the receive parameters.
			if(nBytesReceived > iosnReceiveDataCount)
			{
				// Error, for some reason the socket received more than it was asked for.
				m_pSockets->CloseAcceptSocket();

				#ifdef PRINT_ERRORS
					printf("ZCTProtocol::iosReceiveData :  Received %d bytes "\
                           "from socket, only asked for %d.\n", 
                           nBytesReceived, iosnReceiveDataCount);
				#endif

				// Break out of the receive loop.
				break;
			}
			else
			{
				// Receive no more bytes from socket than was asked for.
				iosnReceiveDataCount -= nBytesReceived;
				iospReceiveDataPointer += nBytesReceived;
			}
		}
	}

	return iosnReceiveDataCount;
}

// iosSendCommand
// Add a host command to the iosSendData buffer
// Entry   : pointer to command
// Returns : TRUE if successful, FALSE if not
BOOL CZCTProtocol::iosSendCommand(HostCommand* hcCommand)
{
    if (m_pSockets == NULL)
    {
        return FALSE;
    }

	int	nBytesRemaining = ntohl(hcCommand->nCommandSize);
	char * pCommand = (char *)hcCommand;

	/// Check for buffer space.
	if((iosnSendBufferCount + nBytesRemaining) <= ZCT_MAX_SEND_BUFFER)
	{
		// Space available, add command to send buffer.
		while(nBytesRemaining)
		{
			// Check for send buffer wrap.
			int	nBytesToSend = nBytesRemaining;

			if((iosnSendBufferPositionIn + nBytesToSend) > ZCT_MAX_SEND_BUFFER)
			{
				// Send buffer will wrap, set up to send what will fit in end of
				// buffer.
				nBytesToSend = ZCT_MAX_SEND_BUFFER - iosnSendBufferPositionIn;
			}
			
			// Add command to buffer.
			memcpy(& m_pSockets->m_cSendBuffer[iosnSendBufferPositionIn], pCommand, nBytesToSend);

			// Update pointers.
			iosnSendBufferPositionIn += nBytesToSend;
			if(iosnSendBufferPositionIn >= ZCT_MAX_SEND_BUFFER)
				iosnSendBufferPositionIn = 0;

			pCommand += nBytesToSend;

			iosnSendBufferCount += nBytesToSend;
			 
			nBytesRemaining -= nBytesToSend;
		}

		return TRUE;
	}

	return FALSE;
}

// iosSendData
// Send data to iosRt
// Entry   : nothing
// Returns : number of bytes remaining to be sent
int CZCTProtocol::iosSendData()
{
    if (m_pSockets == NULL)
    {
        return 0;
    }

	while(iosnSendBufferCount)
	{
		int	nBytesSent;
		int	nBytesToSend = iosnSendBufferCount;

		// Determine the number of bytes to send in order to avoid buffer wrap.
		if((iosnSendBufferPositionOut + nBytesToSend) > ZCT_MAX_SEND_BUFFER)
			nBytesToSend = ZCT_MAX_SEND_BUFFER - iosnSendBufferPositionOut;


		// Send the bytes.
		if((nBytesSent = send(m_pSockets->m_socketAccept, 
                              &(m_pSockets->m_cSendBuffer[iosnSendBufferPositionOut]), 
                              nBytesToSend, 0)) == SOCKET_ERROR)
		{
			// Send failed, see why.
			int		nError;

			if((nError = GetLastError()) != WSAEWOULDBLOCK)
			{
				// Error, close the socket.
				m_pSockets->CloseAcceptSocket();

				#ifdef PRINT_ERRORS
						printf("ZCTProtocol::iosSendData :  Send error %d.\n", nError);
				#endif
			}

			// Either an error occured or the socket would block, break out
			// of the send loop.
			break;
		}

		else
		{
			// No error or blocking, update the send parameters.
			if(nBytesSent > nBytesToSend)
			{
				/*
				 * Error, for some reason the socket sent more than it was asked to.
				 */

				m_pSockets->CloseAcceptSocket();

				#ifdef PRINT_ERRORS
					printf("ZCTProtocol::iosSendData :  Socket sent %d bytes, only asked to send %d.\n", nBytesSent, nBytesToSend);
				#endif

				// Break out of the send loop.
				break;
			}

			else
			{

				// Send was successful, update the send parameters.
				iosnSendBufferPositionOut += nBytesSent;
				if(iosnSendBufferPositionOut >= ZCT_MAX_SEND_BUFFER)
					iosnSendBufferPositionOut = 0;

				iosnSendBufferCount -= nBytesSent;
			}
		}
	}

	return iosnSendBufferCount;
}

/////////////////////////////////////////////////////////////////////////////
//
// ProcessCommands
//
// Inputs           : None.
//
// Return Values    : None.
//
// Date             : 06 May 1999
//
// Engineer         : Billy Baker
//
// Description      : ProcessCommands should be called after MAX_RECEIVE_BUFFER
//                    has been read from the socket.  ProcessCommands will 
//                    parse the data from the socket and then take the 
//                    appropriate action.
/////////////////////////////////////////////////////////////////////////////
void CZCTProtocol::ProcessCommands()
{
    if (m_pSockets == NULL)
    {
        m_bCommandComplete = false;
        return;
    }

	int		nCommand = ntohl(iosCommandReceive.nCommand);
	int		nCommandSize = ntohl(iosCommandReceive.nCommandSize);

	if((nCommand < IOSRT_FIRST_COMMAND) ||
	   (nCommand > IOSRT_LAST_COMMAND) ||
	   (nCommandSize > IOSRT_COMMAND_MAX_SIZE))
	{
		// Invalid command header, close the socket.
		m_pSockets->CloseAcceptSocket();

		#ifdef PRINT_ERRORS
			if(nCommandSize <= IOSRT_COMMAND_MAX_SIZE)
				printf("ZCTHostComms::ProcessCommands :  Invalid command header %08x.\n", nCommand);
			else
				printf("ZCTHostComms::ProcessCommands :  Invalid command size %d in header %08x.\n", nCommandSize, nCommand);
		#endif

		iosCommandReceive.nCommand = 0;
		iosbCommandReceiveRequested = FALSE;
        m_bCommandComplete = false;

        return;
	}

	// Command header valid, see if command data has been received.
	if(iosbCommandReceiveRequested == FALSE)
	{
		// Have not received the command data, read it.
		iosnReceiveDataCount = nCommandSize - IOSRT_COMMAND_HEADER_SIZE;
		iosbCommandReceiveRequested = TRUE;
        m_bCommandComplete = false;

        return;
	}

    m_bCommandComplete = true;
	iosbCommandReceiveRequested = FALSE;

	// Command received, process it.
	switch(nCommand)
	{
		case IOSRT_COMMAND_NULL:
		{
			// Null command.
			iosCommandNull();
		}
		break;

		case IOSRT_COMMAND_READ:
		{
			// Read command.
			iosCommandRead();
		}
		break;

		case IOSRT_COMMAND_SET_READ_POINTERS:
		{
			// Set read pointers command.
			iosCommandSetReadPointers();
		}
		break;

		case IOSRT_COMMAND_SET_WRITE_POINTERS:
		{
			// Set write pointers command.
			iosCommandSetWritePointers();
		}
		break;

		case IOSRT_COMMAND_WRITE:
		{
			// Write command.
			iosCommandWrite();
		}
		break;

		default:
		{
			// Unknown command.
			#ifdef PRINT_ERRORS
				printf("ZCTHostComms::ProcessCommands :  Received unknown command '%d' from iosRt.\n", nCommand);
			#endif

			// Close the socket.
			m_pSockets->CloseAcceptSocket();
		}
		break;
	}

	iosCommandReceive.nCommand = 0;
}

/////////////////////////////////////////////////////////////////////////////
//
//                          The ZCT HOST
//
/////////////////////////////////////////////////////////////////////////////
CZCTProtocol*       pZCTProtocol = NULL;

#ifdef __cplusplus
extern "C"
{
#endif // __cplusplus

/////////////////////////////////////////////////////////////////////////////
//
// ZCTHostCommsExitRoutine
//
// Inputs           : None.
//
// Return Values    : None for standard C.  VxWorks needs an int.
//
// Date             : 06 May 1999
//
// Engineer         : Billy Baker
//
// Description      : ExitRoutine is used to clean up the socket connections
//                    when the process/task is killed/deleted.
/////////////////////////////////////////////////////////////////////////////
#ifdef WIN32
    void ZCTHostCommsExitRoutine()
    {
        if (pZCTProtocol != NULL)
        {
            delete pZCTProtocol;
        }

        WSACleanup();

        return;
    }
#else
#ifdef VXWORKS
    int	ZCTHostCommsExitRoutine()
    {
        if (pZCTProtocol != NULL)
        {
            delete pZCTProtocol;
        }

        taskDeleteHookDelete(ZCTHostCommsExitRoutine);

        return 0;
    }
#endif
#endif

/////////////////////////////////////////////////////////////////////////////
//
// BOOL ZCTHostCommsInitialize()
//
// Inputs           : None.
//
// Return Values    : TRUE if an ZCT protocol was created.
//                    FALSE if an ZCT protocol was not created. 
//
// Date             : 17 August 1999
//
// Engineer         : Billy Baker
//
// Description      : ZCTHostCommsInitialize() should be called once if 
//                    successful to create an ZCTProtocol.
//
/////////////////////////////////////////////////////////////////////////////
BOOL ZCTHostCommsInitialize()
{
    return TRUE;
}

/////////////////////////////////////////////////////////////////////////////
//
// ZCTHostComms
//
// Inputs           : long int lDelay - number of tries to skip before
//                                      attempting to reconnect.
//
// Return Values    : None..
//
// Date             : 06 May 1999
//
// Engineer         : Billy Baker
//
// Description      : HostComms should be called at a schedule time.  It will
//                    attempt to create the listen and accept sockets.  Once
//                    these sockets are created, it will call 
//                    CGenericHostSockets::ReceiveData, ReadValues, 
//                    CGenericHostSockets::SendData, and then ProcessCommands.
//                    Once the listen socket is created, the host byte ordering
//                    is determined and the byte conversion function pointers
//                    are set.
/////////////////////////////////////////////////////////////////////////////
void ZCTHostComms(long int lDelay)
{
    // Variable for not trying to connect on every pass.  Could be
    // deleted if you want to check every pass.
    static	int				nDelay          = 0;
    static  BOOL            bInitial        = TRUE;
    static  BOOL            bInitialized    = FALSE;

    if (bInitial == TRUE)
    {
        if (pZCTProtocol == NULL)
        {
            pZCTProtocol = new CZCTProtocol();
        }

#ifdef WIN32

        int                 nCount;
	    WSADATA             WSAData;

	    // Initialize the windows socket interface.
	    if((nCount = WSAStartup(MAKEWORD(2,2), &WSAData)) != 0)
	    {
            #ifdef PRINT_ERRORS
		        printf("ZCTHostComms : WSAStartup error %d.\n", nCount);
            #endif

		    return;
	    }

        #ifdef PRINT_STATISTICS
	        printf("ZCTHostComms : WINSOCK.DLL Version %d.%d.\n", HIBYTE(WSAData.wVersion), 
                                                   LOBYTE(WSAData.wVersion));
        #endif

	    // Initialize the atexit handler.
	    atexit(ZCTHostCommsExitRoutine);

#else
    #ifdef VXWORKS

        // Initialize VxWorks end task handler.
	    taskHookInit();
	    taskDeleteHookAdd(ZCTHostCommsExitRoutine);

    #endif
#endif

        bInitial = FALSE;
    }

    if (pZCTProtocol == NULL)
    {
        return;
    }

    if (pZCTProtocol->m_pSockets == NULL)
    {
        return;
    }

	// Validate the listening socket.
	if(!(pZCTProtocol->m_pSockets->IsValidListenSocket()))
	{
        if (nDelay)
        {
            nDelay--;
        }
        else
        {
            // Do not have a listening socket, create one.
            //
		    // Attempt to create listen socket.
		    #ifdef PRINT_STATISTICS
			    printf("ZCTHostCommsInitialize : Attempting to create listen socket.\n");
            #endif

		    if(!pZCTProtocol->m_pSockets->CreateListenSocket())
		    {
			    // Attempt succeeded.
                #ifdef PRINT_STATISTICS
				    printf("\nZCTHostCommsInitialize : Could not create listen socket.\n");
                #endif

                nDelay = lDelay;
		    }
            else
            {
		        // Attempt succeeded.
                #ifdef PRINT_STATISTICS
			        printf("\nZCTHostCommsInitialize : Created listen socket.\n");
                #endif
            }
        }

		return;
	}

	// Validate the accept socket.
	if(!(pZCTProtocol->m_pSockets->IsValidAcceptSocket()))
	{
		 // Do not have an accept socket, attempt to create one.
		 //
		 // This can occur during the first pass, during subsequent passes if
		 // the client has not requested a connection, and after any comms error
		 // occurs, including client disconnect.  The objective is to continue
		 // trying until an accept socket is created.
		if(nDelay)
		{
			// Delay before next attempt.
			nDelay --;
		}
		else
		{
			// Attempt to create accept socket.
			#ifdef PRINT_STATISTICS
				printf("ZCTHostComms : Attempting to create accept socket.\n");
            #endif

			if(!pZCTProtocol->m_pSockets->CreateAcceptSocket())
			{
				// Attempt failed, reset delay.
				nDelay = lDelay;
			}
			else
			{
                bInitialized = TRUE;

				// Attempt succeeded.
                #ifdef PRINT_STATISTICS
				    printf("\nZCTHostComms : Created accept socket.\n");
                #endif
			}
		}

		return;
	}

	// Initialize run parameters.
	if(bInitialized == TRUE)
	{
		// Initialize sms base address.
		if(nDelay)
		{
			// Delay before next attempt.
			nDelay --;
		}

		// Initialize send, receive.
		pZCTProtocol->iosnSendBufferCount = pZCTProtocol->iosnReceiveDataCount = 0;

		// Initialize plotting.
		pZCTProtocol->iosPlotInitialize();

		// Initialization complete.
		bInitialized = FALSE;
		return;
	}

	// Update host tick.
	pZCTProtocol->iosuHostTick ++;

	// Gather plot data.
	pZCTProtocol->iosPlotData();

	// Check on data to send.
	if(pZCTProtocol->iosnSendBufferCount)
	{
		// Data to send, send it.
		pZCTProtocol->iosSendData();
	}

	// Check on data to receive.
	if(pZCTProtocol->iosnReceiveDataCount &&
       pZCTProtocol->m_pSockets->IsValidAcceptSocket())
	{
		// Expecting data to receive, receive it.
		if(pZCTProtocol->iosReceiveData())
		{
			// Still data to receive, wait until next pass.
			//
			// Notes :
			//
			//	When iosReceiveData() returns non-zero, a request for data was made
			// that was larger than a single TCP/IP transmission could transfer.
			// Since all of the data requested is not yet received, there is no sense
			// proceeding to iosRt command processing.
			return;
		}
	}

	// Process iosRt commands.
	if(pZCTProtocol->iosCommandReceive.nCommand)
	{
		// Have received a command header, process the command.

	    // Process any data that was read from the client.
        pZCTProtocol->ProcessCommands();

		if (pZCTProtocol->m_bCommandComplete == false)
		{
			// Command not yet complete, wait until next pass.
			return;
		}
	}

	// Previous command complete, set up to read the next iosRt command header.
	pZCTProtocol->iosCommandReceive.nCommand      = 0;
	pZCTProtocol->iospReceiveDataPointer          = (char *)& (pZCTProtocol->iosCommandReceive);
	pZCTProtocol->iosnReceiveDataCount            = IOSRT_COMMAND_HEADER_SIZE;

	return;
}

#ifdef __cplusplus
}
#endif // __cplusplus
