-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                      JPATS T-6A Flight Training Device
--
--
--  Engineer:  Ted E. Dennison
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------

with System;

-------------------------------------------------------------------------------
-- This package provides facilites for models to register internal variables
-- for communications with the IOS. Each such variable must have an entry in
-- the IOS interface database file.
-------------------------------------------------------------------------------
package JPATS_IOS_Interface is

   type Set_Value is private;

   subtype IOS_String_Var is String (1..4);

   ----------------------------------------------------------------------------
   -- This type is for routines to set the value for registered variables.
   -- The IOS will indirectly call the routine. pData is the address of the
   -- value the IOS would like placed in the variable.
   ----------------------------------------------------------------------------
   type Set_Variable_Proc_Ptr_Type is access procedure (Value : in Set_Value);

   ----------------------------------------------------------------------------
   -- This type is for routines to set the value for registered arrays.
   -- The IOS will indirectly call the routine. pData is the value the IOS
   -- would like placed in the variable. The offsets correspond
   -- to indices.
   ----------------------------------------------------------------------------
   type Set_Array_Proc_Ptr_Type is access procedure (Value     : in Set_Value;
                                                     Offset1d  : in Integer;
                                                     Offset2d  : in Integer;
                                                     Completed : in Boolean);

   ----------------------------------------------------------------------------
   -- Null routines. These are intended for use when you want the variable to
   -- be read-only.
   ----------------------------------------------------------------------------
   Dont_Write_Variable : constant Set_Variable_Proc_Ptr_Type;
   Dont_Write_Array    : constant Set_Array_Proc_Ptr_Type;

   ----------------------------------------------------------------------------
   -- The following conversion routines convert IOS Set_Value's into common
   -- Ada data types.
   ----------------------------------------------------------------------------
   function Convert_To_Integer        (Value : Set_Value) return Integer;
   function Convert_To_Bool           (Value : Set_Value) return Boolean;
   function Convert_To_Float          (Value : Set_Value) return Float;
   function Convert_To_Long_Float     (Value : Set_Value) return Long_Float;
   function Convert_To_Character      (Value : Set_Value) return Character;
   function Convert_To_Ios_String_Var (Value : Set_Value) return IOS_String_Var;
   function Convert_To_Set_Value      (Addr  : System.Address) return Set_Value;
   
   ----------------------------------------------------------------------------
   -- Register the variable with the given name with the IOS, using the given
   -- address to read values and the given routine to update values.
   ----------------------------------------------------------------------------
   procedure Register
     (Name        : in String;
      Variable    : in System.Address;
      Set_Routine : in Set_Variable_Proc_Ptr_Type := Dont_Write_Variable
      );

   ----------------------------------------------------------------------------
   -- Register the array with the given name with the IOS, using the given
   -- address to read values and the given routine to update values.
   ----------------------------------------------------------------------------
   procedure Register
     (Name               : in String;
      Var_Array          : in System.Address;
      Dimension_1_Length : in Positive;
      Dimension_2_Length : in Positive                := 1;
      Set_Routine        : in Set_Array_Proc_Ptr_Type := Dont_Write_Array
      );

   ----------------------------------------------------------------------------
   -- Initialize this facility by reading in the list of interface variables.
   -- This should be called before any calls to Register are made.
   ----------------------------------------------------------------------------
   procedure Initialize;

private
   type Set_Value is new System.Address;

   -- Null write routines
   procedure No_Write_To_Variable (Value : in Set_Value);
   procedure No_Write_To_Array (Value     : in Set_Value;
                                Offset1d  : in Integer;
                                Offset2d  : in Integer;
                                Completed : in Boolean);

   Dont_Write_Variable : constant Set_Variable_Proc_Ptr_Type := No_Write_To_Variable'access;
   Dont_Write_Array    : constant Set_Array_Proc_Ptr_Type    := No_Write_To_Array'access;

end JPATS_IOS_Interface;

