-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                      JPATS T-6A Flight Training Device
--
--
--  Engineer:  Ted E. Dennison
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------

with Ada.Characters.Latin_1;
with Ada.Exceptions;
with Ada.Strings.Fixed;
with Ada.Strings.Maps.Constants;
with Ada.Strings.Unbounded;
with Ada.Text_IO;
with Interfaces.C.Strings;
with System.Address_To_Access_Conversions;

with Bc.Containers.Maps.Unbounded;
with BC.Support.Unmanaged_Storage;

with Token.Analyzer;
with Token.Keyword;
with Token.Real;
with Token.CSV_Field;
with Token.Character_Set;
with Token.End_Of_File;
with Token.Line_Comment;

with Log;
with Simulation_Dictionary;
with Stethoscope;

use type Log.Event_Type;
use type Ada.Strings.Maps.Character_Set;

pragma Elaborate_All (Simulation_Dictionary);

-------------------------------------------------------------------------------
-- This package provides facilites for models to register internal variables
-- for communications with the IOS. Each such variable must have an entry in
-- the IOS interface database file.
-------------------------------------------------------------------------------
package body JPATS_IOS_Interface is


   -- Address conversion routines
   package To_Bool_Ptr       is new System.Address_To_Access_Conversions (Boolean);
   package To_Int_Ptr        is new System.Address_To_Access_Conversions (Integer);
   package To_Float_Ptr      is new System.Address_To_Access_Conversions (Float);
   package To_Long_Float_Ptr is new System.Address_To_Access_Conversions (Long_Float);
   package To_Character_Ptr  is new System.Address_To_Access_Conversions (Character);
   package To_IOS_String_Ptr is new System.Address_To_Access_Conversions (IOS_String_Var);

   type Bool_32 is new Boolean;
   for Bool_32'Size use 32;

   procedure RegisterIOSVariable( Annotation       : Interfaces.C.Char_Array;
                                  Variable_Address : System.address;
                                  Change_Value_Ptr : Set_Variable_Proc_Ptr_Type;
                                  ucSize           : Interfaces.C.char;
                                  Recorded         : Bool_32;
                                  fLow             : Interfaces.C.C_Float;
                                  fHigh            : Interfaces.C.C_Float );
   pragma import( c, RegisterIOSVariable, "RegisterIOSVariable" );

   procedure RegisterIOSVariableEx( Annotation       : Interfaces.C.Char_Array;
                                    Variable_Address : System.address;
                                    Change_Value_Ptr : Set_Array_Proc_Ptr_Type;
                                    ucSize           : Interfaces.C.char;
                                    Recorded         : Bool_32;
                                    fLow             : Interfaces.C.C_Float;
                                    fHigh            : Interfaces.C.C_Float;
                                    lLong1d          : Interfaces.C.Long;
                                    lLong2d          : Interfaces.C.Long );
   pragma import( c, RegisterIOSVariableEX, "RegisterIOSVariableEx" );


   --------------------------------------------------------------------------
   -- A type for keeping the information needed to register an IOS variable
   type IOS_Variable_Type is ( Float_Type,
                               Int_Type,
                               Bool_Type,
                               Long_Float_Type,
                               Ios_String_Var_Type );

   type String_Ptr is access String;

   type Variable_Registration_Info is record
      Annotation    : String_Ptr;
      Recorded      : Bool_32;
      Units         : String_Ptr;
      IOS_Type      : Ios_Variable_Type;
      Minimum_Value : Float;
      Maximum_Value : Float;
   end record;

   Var_Type_String_Array : constant array (Float_Type..Ios_String_Var_Type)
     of String_Ptr := (Float_Type          => new String'("float"),
                       Int_Type            => new String'("int"),
                       Bool_Type           => new String'("uchar"),
                       Long_Float_Type     => new String'("double"),
                       Ios_String_Var_Type => new String'("int")
                       );


   --------------------------------------
   -- Booch Map component declarations --
   --------------------------------------

   -- The number of hash buckets used in the hash algorithm
   Buckets : constant := 128;

   Unmanaged_Pool : BC.Support.Unmanaged_Storage.Pool;


   -- String hash routine. This routine will be used as the primary lookup for
   -- dictionary entries.
   function Hash (Word : in Ada.Strings.Unbounded.Unbounded_String) return Positive;

   package String_Containers is new BC.Containers
     (Item => Ada.Strings.Unbounded.Unbounded_String,
      "="  => Ada.Strings.Unbounded."="
      );

   package Registration_Maps is new String_Containers.Maps
     (Value => Variable_Registration_Info,
      "="   => Ada.Strings.Unbounded."="
      );

   package Maps is new Registration_Maps.Unbounded
     (Hash            => Hash,
      Buckets         => Buckets,
      Storage_Manager => BC.Support.Unmanaged_Storage.Pool,
      Storage         => Unmanaged_Pool
      );

   Registration_Map : Maps.Unbounded_Map;

   -------------------------------------------------
   -- IOS Interface CSV file parsing declarations --
   -------------------------------------------------
   If_File_Name  : constant String := Simulation_Dictionary.Lookup ("IOS_IF_File");

   IF_File : Ada.Text_IO.File_Type;

   type IF_Token_ID is (Comma, Real, True_Id, False_Id, Int_Id, Bool_Id, Float_Id,
                        Long_Float_Id, String_Id, Field, EOL, EOF, Whitespace, Comment);

   package Tokenizer is new Token.Analyzer(IF_Token_ID);

   Syntax : Tokenizer.Syntax :=
     (Comma          => new Token.Keyword.Instance'(Token.Keyword.Get(",")),
      Real           => new Token.Real.Instance'(Token.Real.Get),
      True_Id        => new Token.Keyword.Instance'(Token.Keyword.Get("true")),
      False_Id       => new Token.Keyword.Instance'(Token.Keyword.Get("false")),
      Int_ID         => new Token.Keyword.Instance'(Token.Keyword.Get("int")),
      Bool_Id        => new Token.Keyword.Instance'(Token.Keyword.Get("bool")),
      Float_Id       => new Token.Keyword.Instance'(Token.Keyword.Get("float")),
      Long_Float_Id  => new Token.Keyword.Instance'(Token.Keyword.Get("long_float")),
      String_Id      => new Token.Keyword.Instance'(Token.Keyword.Get("string")),
      Field          => new Token.CSV_Field.Instance'(Token.CSV_Field.Get),
      EOL            => new Token.Character_Set.Instance'
      (Token.Character_Set.Get
       (Set        => Ada.Strings.Maps.To_Set (Token.EOL_Character),
        Reportable => True
        )),
      EOF            => new Token.End_Of_File.Instance'(Token.End_Of_File.Get),
      Whitespace     => new Token.Character_Set.Instance'
      (Token.Character_Set.Get (Token.Character_Set.Standard_Whitespace -
                                Ada.Strings.Maps.To_Set (Token.Eol_Character))),
      Comment        => new Token.Line_Comment.Instance'(Token.Line_Comment.Get ("#"))
      );

   -- Routine to feed text to the analyzer
   function Get_Text_Line return String;


   Analyzer : Tokenizer.Instance := Tokenizer.Initialize
     (Language_Syntax => Syntax,
      Feeder          => Get_Text_Line'Access
      );

   Parse_Error    : exception;
   Error_Message  : Ada.Strings.Unbounded.Unbounded_String;
   Error_Severity : Log.Event_Type;
   Errors         : Natural := 0;

   --------------------------------------------------------
   -- Helper routines for our instantiated free packages.
   --

   -----------------------------------------------------------------------------
   -- String hash routine. This routine will be used as the primary lookup for
   -- dictionary entries.
   -----------------------------------------------------------------------------
   function Hash (Word : in Ada.Strings.Unbounded.Unbounded_String) return Positive is

      Value : Natural := 0;
   begin

      for Char in 1..Ada.Strings.Unbounded.Length(Word) loop
         Value := Value + Character'Pos(Ada.Strings.Unbounded.Element
                                        (Source => Word,
                                         Index  => Char
                                         )) mod
           128;
      end loop;

      return Value + 1;

   end Hash;

   ------------------------------------------------------------------------------------
   -- The routine used to feed data into the token analizer. This routine reads from
   -- the analizer file one line at a time until EOF is reached.
   -- Some other routine must open and close the file.
   ------------------------------------------------------------------------------------
   function Get_Text_Line return String is
      Buffer : String(1..512);
      Data_Length : Integer;
   begin

      Ada.Text_IO.Get_Line (File => IF_File, Item => Buffer, Last => Data_Length);

      if Ada.Text_IO.End_Of_File(IF_File) then
         return Buffer(1..Data_Length) & Token.EOF_Character;
      else
         return Buffer(1..Data_Length) & Token.EOL_Character;
      end if;
   exception
      when Ada.Text_IO.End_Error =>
         return (1 => Token.EOF_Character);

   end Get_Text_Line;

   --------------------------------------
   -- IOS Set_Value conversion routines
   --

   ----------------------------------------------------------------------------
   -- The following conversion routines convert IOS Set_Value's into common
   -- Ada data types.
   ----------------------------------------------------------------------------
   function Convert_To_Integer (Value : Set_Value) return Integer is
   begin
      return To_Int_Ptr.To_Pointer(System.Address(Value)).all;
   end Convert_To_Integer;

   function Convert_To_Bool (Value : Set_Value) return boolean is
   begin

      -- We have to do it this stupid way, due to a problem in how GreenHills handles
      -- booleans with unused bits set.
      if To_Bool_Ptr.To_Pointer (System.Address(Value)).all then
         return True;
      else
         return False;
      end if;

   end Convert_To_Bool;

   function Convert_To_Float (Value : Set_Value) return Float is
   begin
      return To_Float_Ptr.To_Pointer (System.Address(Value)).all;
   end Convert_To_Float;

   function Convert_To_Long_Float (Value : Set_Value) return Long_Float is
   begin
      return To_Long_Float_Ptr.To_Pointer (System.Address(Value)).all;
   end Convert_To_Long_Float;

   function Convert_To_Character (Value : Set_Value) return Character is
   begin
      return To_Character_Ptr.To_Pointer (System.Address(Value)).all;
   end Convert_To_Character;

   function Convert_To_Ios_String_Var (Value : Set_Value) return IOS_String_Var is
   begin
      return To_IOS_String_Ptr.To_Pointer (System.Address(Value)).all;
   end Convert_To_IOS_String_Var;

   function Convert_To_Set_Value (Addr : System.Address) return Set_Value is
   begin
      return Set_Value(Addr);
   end Convert_To_Set_Value;

   ----------------------------------------------------
   -- The default null or "Don't Write" write routines
   ----------------------------------------------------

   procedure No_Write_To_Variable (Value : Set_Value) is
   begin
      null;
   end No_Write_To_Variable;

   procedure No_Write_To_Array (Value     : in Set_Value;
                                Offset1d  : in Integer;
                                Offset2d  : in Integer;
                                Completed : in Boolean)  is
   begin
      null;
   end No_Write_To_Array;


   ----------------------------------------
   -- CSV Interface file parsing routines
   --

   ----------------------------------------------------------------------------
   -- Report a parsing error message.
   ----------------------------------------------------------------------------
   procedure Report (Event    : in String;
                     Severity : in Log.Event_Type
                    ) is
   begin
      Log.Report
        (Event => "error at line" & Integer'Image(Tokenizer.Line (Analyzer)) & ", column" &
         Integer'Image(Tokenizer.Column (Analyzer)) & " in file " & IF_File_Name &
         Ada.Characters.Latin_1.CR & Ada.Characters.Latin_1.LF & " - " & Event,
         Severity => Severity
         );
   end Report;


   ----------------------------------------------------------------------------
   -- Parse over all the rest of the tokens on the current CSV line.
   ----------------------------------------------------------------------------
   procedure Parse_Rest_Of_Line is
   begin
      while Tokenizer.Token (Analyzer) /= EOL loop

         if Tokenizer.Token (Analyzer) = EOF then
            return;
         end if;

         Tokenizer.Find_Next (Analyzer);
      end loop;

      Tokenizer.Find_Next (Analyzer);
   end Parse_Rest_Of_Line;

   ----------------------------------------------------------------------------
   -- Parse the date line.
   ----------------------------------------------------------------------------
   procedure Parse_Date_Line is
   begin
      -- This should be a CSV line with the file's creation date in the first
      -- field

      if Tokenizer.Token (Analyzer) = Field then
         Log.Report ("Parsing " & IF_File_Name & " created on " &
                     Tokenizer.Lexeme (Analyzer)
                     );
      else
         Report
           (Event    => "Creation date expected in first field.",
            Severity => Log.Warning
            );
      end if;

      Parse_Rest_Of_Line;
   end Parse_Date_Line;

   ----------------------------------------------------------------------------
   -- Parse a header line.
   ----------------------------------------------------------------------------
   procedure Parse_Header_Line is
   begin
      -- Right now we ignore header data.
      Parse_Rest_Of_Line;
   end Parse_Header_Line;

   ----------------------------------------------------------------------------
   -- Parse a comma.
   ----------------------------------------------------------------------------
   procedure Parse_Comma is
   begin
      if Tokenizer.Token (Analyzer) /= Comma then
            Error_Severity := Log.Error;
            Error_Message  := Ada.Strings.Unbounded.To_Unbounded_String
              ("Comma expected, found " &
               If_Token_Id'Image(Tokenizer.Token (Analyzer)) & "."
               );
            raise Parse_Error;
      end if;

      Tokenizer.Find_Next (Analyzer);
   end Parse_Comma;

   ----------------------------------------------------------------------------
   -- Parse a CSV field (and ignore the contents).
   ----------------------------------------------------------------------------
   procedure Parse_Field is
   begin
      case Tokenizer.Token (Analyzer) is
         when Comma =>
            Tokenizer.Find_Next (Analyzer);
         when EOL =>
            Error_Severity := Log.Error;
            Error_Message  := Ada.Strings.Unbounded.To_Unbounded_String
              ("Unexpected end of line.");
            raise Parse_Error;
         when EOF =>
            Error_Severity := Log.Fatal;
            Error_Message  := Ada.Strings.Unbounded.To_Unbounded_String
              ("Unexpected end of file.");
            raise Parse_Error;

         when others =>
            Tokenizer.Find_Next (Analyzer);
            Parse_Comma;
      end case;

   end Parse_Field;

   ----------------------------------------------------------------------------
   -- Parse a CSV field and return the string value.
   ----------------------------------------------------------------------------
   function Parse_Field return String is
      Lexeme : constant String := Tokenizer.Lexeme(Analyzer);
   begin
      case Tokenizer.Token (Analyzer) is
         when Comma =>
            Tokenizer.Find_Next (Analyzer);
            return "";
         when EOL =>
            Error_Severity := Log.Error;
            Error_Message  := Ada.Strings.Unbounded.To_Unbounded_String
              ("Unexpected end of line.");
            raise Parse_Error;
         when EOF =>
            Error_Severity := Log.Fatal;
            Error_Message  := Ada.Strings.Unbounded.To_Unbounded_String
              ("Unexpected end of file.");
            raise Parse_Error;

         when others =>
            Tokenizer.Find_Next (Analyzer);
            Parse_Comma;
            return Lexeme;
      end case;

   end Parse_Field;

   ----------------------------------------------------------------------------
   -- Parse a boolean field and return the value.
   ----------------------------------------------------------------------------
   function Parse_Boolean_Field return Bool_32 is
   begin
      case Tokenizer.Token (Analyzer) is
         when True_Id =>
            Tokenizer.Find_Next (Analyzer);
            Parse_Comma;
            return True;
         when False_Id =>
            Tokenizer.Find_Next (Analyzer);
            Parse_Comma;
            return False;
         when others =>
            Report
              (Event    => "expected boolean value (true or false).",
               Severity => Log.Error
               );
            Errors := Errors + 1;

            if Tokenizer.Token (Analyzer) /= Comma then
               Tokenizer.Find_Next (Analyzer);
               Parse_Comma;
            end if;
            raise Parse_Error;
      end case;
   end Parse_Boolean_Field;

   ----------------------------------------------------------------------------
   -- Parse an IOS type of of the current field.
   ----------------------------------------------------------------------------
   procedure Parse_IOS_Type (Value : out IOS_Variable_Type) is
   begin
      case Tokenizer.Token (Analyzer) is
         when Int_Id =>
            Value := Int_Type;
         when Bool_Id =>
            Value := Bool_Type;
         when Float_Id =>
            Value := Float_Type;
         when Long_Float_Id =>
            Value := Long_Float_Type;
         when String_Id =>
            Value := IOS_String_Var_Type;
         when EOL =>
            Error_Severity := Log.Error;
            Error_Message  := Ada.Strings.Unbounded.To_Unbounded_String
              ("Unexpected end of line.");
            raise Parse_Error;
         when EOF =>
            Error_Severity := Log.Fatal;
            Error_Message  := Ada.Strings.Unbounded.To_Unbounded_String
              ("Unexpected end of file.");
            raise Parse_Error;

         when others =>
            -- Report this field as an error, but move on (in case there are
            -- more errors that can be reported).

            Value := IOS_Variable_Type'First;
            Report
              (Event    => "expected one of: int, bool, float, long_float, string.",
               Severity => Log.Error
               );
            Errors := Errors + 1;

            Tokenizer.Find_Next (Analyzer);
            Parse_Comma;
      end case;
   end Parse_IOS_Type;

   ----------------------------------------------------------------------------
   -- Parse a CSV field that contains a data value range.
   ----------------------------------------------------------------------------
   procedure Parse_Range
     (Lower_Value  : out Float;
      Higher_Value : out Float
     ) is
      Field : constant String := Parse_Field;
      Elipsis_Index : constant Natural :=
        Ada.Strings.Fixed.Index
        (Source  => Field,
         Pattern => ".."
         );
   begin
      -- The basic idea is that there should be two numbers, separated by an
      -- elipsis ("..").
      Lower_Value := 0.0;
      Higher_Value := 0.0;
      if Elipsis_Index > 0 then
         Lower_Value  := Float'Value(Field (1..Elipsis_Index - 1));
         Higher_Value := Float'Value(Field (Elipsis_Index + 2..Field'length));
      end if;

   exception
      -- Just ignore the offending field. These values aren't really nessecary.
      when Constraint_Error =>
         null;
   end Parse_Range;

   ----------------------------------------------------------------------------
   -- Parse a line of CSV data.
   ----------------------------------------------------------------------------
   procedure Parse_Data_Line is
      Field_Index : Natural := 1;

      New_Registration_Info : Variable_Registration_Info;
      Mapped_To_Name        : Ada.Strings.Unbounded.Unbounded_String;
   begin

      -- Skip the first 3 fields.
      Parse_Field;
      Parse_Field;
      Parse_Field;

      -- Get the variable annotation and name fields
      New_Registration_Info.Annotation := new String'(Parse_Field);
      Mapped_To_Name := Ada.Strings.Unbounded.To_Unbounded_String
        (Ada.Strings.Fixed.Translate
         (Source  => Parse_Field,
          Mapping => Ada.Strings.Maps.Constants.Lower_Case_Map
          )
         );

      -- Get the Record flag
      New_Registration_Info.Recorded := Parse_Boolean_Field;

      -- Get the type, ignore the next field, then get the units string
      Parse_IOS_Type (New_Registration_Info.Ios_Type);
      Parse_Field;
      New_Registration_Info.Units      := new String'(Parse_Field);

      -- Ignore the next field, then get the range.
      Parse_Field;
      Parse_Range
        (Lower_Value  => New_Registration_Info.Minimum_Value,
         Higher_Value => New_Registration_Info.Maximum_Value
         );

      -- Ignore the rest of the line.
      Parse_Rest_Of_Line;

      -- Map everything to the name.
      Maps.Bind
        (M => Registration_Map,
         I => Mapped_To_Name,
         V => New_Registration_Info
         );
   exception
      when Parse_Error =>

         case Error_Severity is
            when Log.Fatal =>
               raise;
            when Log.Error =>
               Errors := Errors + 1;
            when others =>
               null;
         end case;

         Report
           (Event    => Ada.Strings.Unbounded.To_String (Error_Message),
            Severity => Error_Severity
            );
         Parse_Rest_Of_Line;

      when Bc.Duplicate =>
         Report
           (Event    => "A variable named " & Ada.Strings.Unbounded.To_String (Mapped_To_Name) &
            " has already been processed.",
            Severity => Log.Error
            );
         Errors := Errors + 1;
   end Parse_Data_Line;

   ----------------------------------------------------------------------------
   -- Parse the whole interface file. This routine assumes the first token has
   -- been read in already.
   ----------------------------------------------------------------------------
   procedure Parse_Interface_File is
   begin

      -- The first line contains the creation date, the next three contain column
      -- headings, and the rest are interface variable data.

      Parse_Date_Line;
      Parse_Header_Line;
      Parse_Header_Line;
      Parse_Header_Line;

      while Tokenizer.Token (Analyzer) /= EOF loop
         if Tokenizer.Token (Analyzer) = EOL then
            Tokenizer.Find_Next (Analyzer);
         else
            Parse_Data_Line;
         end if;
      end loop;

   exception
      when Parse_Error =>
         Report
           (Event    => Ada.Strings.Unbounded.To_String (Error_Message),
            Severity => Error_Severity
            );
         if Error_Severity = Log.Fatal then
            raise;
         end if;
   end Parse_Interface_File;


   ----------------------------------------------
   -- Externally visible routines.
   --

   ----------------------------------------------------------------------------
   -- Register the variable with the given name with the IOS, using the given
   -- address to read values and the given routine to update values.
   ----------------------------------------------------------------------------
   procedure Register
     (Name        : in String;
      Variable    : in System.Address;
      Set_Routine : in Set_Variable_Proc_Ptr_Type := Dont_Write_Variable
     ) is

      Variable_Info : Variable_Registration_Info;
      Lower_Case_Name : constant String :=
        Ada.Strings.Fixed.Translate
        (Source  => Name,
         Mapping => Ada.Strings.Maps.Constants.Lower_Case_Map
         );
   begin

      Variable_Info := Maps.Value_Of
        (M => Registration_Map,
         I => Ada.Strings.Unbounded.To_Unbounded_String (Lower_Case_Name)
         );

      RegisterIOSVariable
        (Annotation       => Interfaces.C.To_C (Variable_Info.Annotation.all),
         Variable_Address => Variable,
         Change_Value_Ptr => Set_Routine,
         Ucsize           => Interfaces.C.NUL,
         Recorded         => Variable_Info.Recorded,
         Flow             => Interfaces.C.C_Float(Variable_Info.Minimum_Value),
         Fhigh            => Interfaces.C.C_Float(Variable_Info.Maximum_Value)
         );

      Stethoscope.Register_Signal
        (Name           => Variable_Info.Annotation.all,
         Units          => Variable_Info.Units.all,
         Object_Address => Variable,
         Value_Type     => Var_Type_String_Array(Variable_Info.IOS_Type).all
         );

   exception
      when Bc.Not_Found =>
         Log.Report
           (Event    => """" & Lower_Case_Name & """ is not in " & IF_File_Name,
            Severity => Log.Warning
            );
      when Stethoscope.Operation_Failed =>
         Log.Report
           (Event    => "Failed to register " & Variable_Info.Annotation.all &
            " in " & Variable_Info.Units.all &
            " (" & Var_Type_String_Array(Variable_Info.IOS_Type).all & ") " &
            " with Stethoscope.",
            Severity => Log.Warning);
   end Register;

   ----------------------------------------------------------------------------
   -- Register the array with the given name with the IOS, using the given
   -- address to read values and the given routine to update values.
   ----------------------------------------------------------------------------
   procedure Register
     (Name               : in String;
      Var_Array          : in System.Address;
      Dimension_1_Length : in Positive;
      Dimension_2_Length : in Positive                := 1;
      Set_Routine        : in Set_Array_Proc_Ptr_Type := Dont_Write_Array
     ) is

      Variable_Info : Variable_Registration_Info;

      Lower_Case_Name : constant String :=
        Ada.Strings.Fixed.Translate
        (Source  => Name,
         Mapping => Ada.Strings.Maps.Constants.Lower_Case_Map
         );
   begin

      Variable_Info := Maps.Value_Of
        (M => Registration_Map,
         I => Ada.Strings.Unbounded.To_Unbounded_String (Lower_Case_Name)
         );

      RegisterIOSVariableEx
        (Annotation       => Interfaces.C.To_C (Variable_Info.Annotation.all),
         Variable_Address => Var_Array,
         Change_Value_Ptr => Set_Routine,
         Ucsize           => Interfaces.C.NUL,
         Recorded         => Variable_Info.Recorded,
         Flow             => Interfaces.C.C_Float(Variable_Info.Minimum_Value),
         Fhigh            => Interfaces.C.C_Float(Variable_Info.Maximum_Value),
         Llong1d          => Interfaces.C.Long(Dimension_1_Length),
         Llong2d          => Interfaces.C.Long(Dimension_2_Length)
         );

   exception
      when Bc.Not_Found =>
         Log.Report
           (Event    => """" & Lower_Case_Name & """ is not in " & IF_File_Name,
            Severity => Log.Warning
            );
   end Register;

   ----------------------------------------------------------------------------
   -- Initialize this facility by reading in the list of interface variables.
   -- This should be called before any calls to Register are made.
   ----------------------------------------------------------------------------
   procedure Initialize is
   begin

      Ada.Text_IO.Open
        (Name => IF_File_Name,
         File => IF_File,
         Mode => Ada.Text_IO.In_File
         );

      -- Load up the first token in the file.
      Tokenizer.Find_Next (Analyzer);

      Parse_Interface_File;

      Ada.Text_IO.Close (IF_File);

   end Initialize;


end JPATS_IOS_Interface;
