-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                 JPATS T-6A Texan-II Flight Training Device
--
--
--  Engineer:  Thomas Haley
--
--  Revision:  (Number and date inserted by Clearcase)
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------

with Ada.Characters.Latin_1;
with Ada.Exceptions;
with Ada.Streams;
with System.Address_To_Access_Conversions;

with Scheduler_Collection;
with Interfaces.C;
with JPATS_IOS_Interface;
with Log;
with Mission_Time;

use type Interfaces.C.Short;
use type System.Address;
use type Ada.Streams.Stream_Element_Offset;

package body Jpats_HostComms_Scheduler is

   type Counter_32 is mod 2**32;
   Comm_Updating : Counter_32 := 0;

   type Boolean_32 is new Boolean;
   for Boolean_32'Size use 32;

   This_Scheduler : aliased Instance;

   --------------------------------------------
   -- C routines imported from iosprotocol.cpp
   --
   procedure IOSHostComms;
   pragma import(C, IOSHostComms, "IOSHostComms");

   function IOSHostCommsInitialize return Interfaces.C.Int;
   pragma import(C, IOSHostCommsInitialize, "IOSHostCommsInitialize");

   function IOSHostCommsConnected return Interfaces.C.Int;
   pragma import(C, IOSHostCommsConnected, "IOSHostCommsConnected");

   procedure IOSHostCommsProcessCommands
     (Replay  : in Boolean_32 := False;
      Restore : in Boolean_32 := False
     );
   pragma Import (C, IOSHostCommsProcessCommands, "IOSHostCommsProcessCommands");

   function IOSHostCommsReadValues return Interfaces.C.Int;
   pragma Import (C,IOSHostCommsReadValues, "IOSHostCommsReadValues");

   function IOSHostCommsReceiveBuffer return System.Address;
   pragma Import (C, IOSHostCommsReceiveBuffer, "IOSHostCommsReceiveBuffer");

   function IOSHostCommsMaxReceiveBuffer return Interfaces.C.Short;
   pragma Import (C, IOSHostCommsMaxReceiveBuffer, "IOSHostCommsMaxReceiveBuffer");

   -----------------------------------
   -- Stream Conversion Declarations
   --

   Bytes_Per_Element : constant := (Ada.Streams.Stream_Element'Size + 7) / 8;

   subtype IOS_Buffer_Stream is Ada.Streams.Stream_Element_Array
     (1..Ada.Streams.Stream_Element_Offset
      ( (IOSHostCommsMaxReceiveBuffer + Bytes_Per_Element - 1) / Bytes_Per_Element)
      );

   package IOS_Buffer_Convert is new System.Address_To_Access_Conversions (IOS_Buffer_Stream);

   ---------------------------------------------------
   -- Internal subprograms, tasks and protected types
   --


   ----------------------------------------------------------------------------
   -- The following protected object is used to allow the non-realtime IOS
   -- communicator to wait on an entry for his next update time. The realtime
   -- client can release the communicator without blocking himself.
   ----------------------------------------------------------------------------
   protected type Request_Semaphore is

      -------------------------------------------------------------------------
      -- Issue a request. This entry will not be accepted until the previous
      -- request has completed.
      -------------------------------------------------------------------------
      entry Issue;

      -------------------------------------------------------------------------
      -- Respond to a request. This entry will not be accepted until there is
      -- a request to respond to.
      -------------------------------------------------------------------------
      entry Wait_For;

      -------------------------------------------------------------------------
      -- Mark the request as completed. This routine should not get called
      -- unless Process has already been called.
      -------------------------------------------------------------------------
      procedure Complete;

      -------------------------------------------------------------------------
      -- Return true if there is currently no oustanding uncompleted request.
      -------------------------------------------------------------------------
      function Completed return Boolean;
   private
      Requested  : Boolean := False;
   end Request_Semaphore;
   Request : Request_Semaphore;

   protected body Request_Semaphore is
      -------------------------------------------------------------------------
      -- Issue a request. This entry will not be accepted until the previous
      -- request has completed.
      -------------------------------------------------------------------------
      entry Issue when not Requested is
      begin
         Requested := True;
      end Issue;

      -------------------------------------------------------------------------
      -- Respond to a request. This entry will not be accepted until there is
      -- a request to respond to.
      -------------------------------------------------------------------------
      entry Wait_For when Requested is
      begin
         null;
      end Wait_For;

      -------------------------------------------------------------------------
      -- Mark the request as completed. This routine should not get called
      -- unless Process has already been called.
      -------------------------------------------------------------------------
      procedure Complete is
      begin
         Requested := False;
      end Complete;

      -------------------------------------------------------------------------
      -- Return true if there is currently no oustanding uncompleted request.
      -------------------------------------------------------------------------
      function Completed return Boolean is
      begin
         return not Requested;
      end Completed;
   end Request_Semaphore;

   ----------------------------------------------------------------------------
   -- Low (default) priority task to perform IOS communications.
   ----------------------------------------------------------------------------
   task Communicator is
      -------------------------------------------------------------------------
      -- This entry tells the communicator to initialize the communications.
      -------------------------------------------------------------------------
      entry Initialize;
   end Communicator;

   task body Communicator is
      use type Interfaces.C.Int;
      Trash : Interfaces.C.Int;
   begin
      accept Initialize;

      -- Attempt to get the connection going before we accept any requests
      if IOSHostCommsInitialize = 0 then
         Log.Report (Event    => "IOSHostCommsInitialize failed.",
                     Severity => Log.Fatal
                     );
      else
         Trash := IOSHostCommsConnected;
      end if;

      loop
         while IOSHostCommsConnected = 0 loop
            Log.Report (Event    => "IOS Connection failed. Retrying",
                        Severity => Log.Error
                        );
         end loop;

         -- Wait for a send request, perform it, then mark it complete.
         Request.Wait_For;

         IOSHostComms;

         Request.Complete;

      end loop;
   exception
      when Error : others =>
         Log.Report (Event => "JPATS_HostComms_Scheduler.Communicator terminated due to " &
                     "unhandled exception." & Ada.Characters.Latin_1.Cr &
                     Ada.Characters.Latin_1.Lf &
                     Ada.Exceptions.Exception_Information (Error),
                     Severity => Log.Error);
   end Communicator;

   ----------------------------------------------------------------------------
   -- Retrieve and process any IOS requests. Gather up all host information
   -- requested by the IOS and request that it be sent. The data will actually
   -- be send by a non-realtime task.
   ----------------------------------------------------------------------------
   procedure Perform_Update
     (An_Instance          : in out Instance;
      Replay               : in     Boolean_32 := False
     ) is
      use type Interfaces.C.Int;
   begin

      Comm_Updating := Comm_Updating + 1;
      Mission_Time.Update;

      if Request.Completed then
          if not An_Instance.IOS_Responding then
            An_Instance.IOS_Responding := True;
            Log.Report ("The IOS interface is now up.");
         end if;

         -- Retrieve and process any IOS commands
         IOSHostCommsProcessCommands (Replay => Replay);

         -- Gather the host data for sending
         if IOSHostCommsReadValues /= 0 then
            -- Request the host data be sent
            Request.Issue;
         end if;
      else
         if An_Instance.IOS_Responding then
            An_Instance.IOS_Responding := False;
            Log.Report
              (Event    => "The IOS is not responding.",
               Severity => Log.Warning
               );
         end if;
      end if;
   end Perform_Update;

   -------------------------
   -- External subprograms
   --

   ----------------------------------------------------------------------------
   -- Initialize the IOS/Host communications model.
   ----------------------------------------------------------------------------
   procedure Initialize (An_Instance : in out Instance) is
      use type Interfaces.C.Int;
   begin

      Communicator.Initialize;

      JPATS_IOS_Interface.Register
        (Name        => "Comm Updating",
         Variable    => Comm_Updating'address
         );
   end Initialize;

   ----------------------------------------------------------------------------
   -- Retrieve and process any IOS requests.
   ----------------------------------------------------------------------------
   procedure Update
     (An_Instance          : in out Instance;
      Integration_Constant : in     Float) is
   begin
      Perform_Update (An_Instance);
   end Update;

   ----------------------------------------------------------------------------
   -- This routine is called only during a replay. It keeps the communications
   -- link alive (and allows replay commands), but doesn't accept variable
   -- updates.
   ----------------------------------------------------------------------------
   procedure Update_In_Freeze
     (An_Instance : in out Instance) is
   begin
      Perform_Update
        (An_Instance => An_Instance,
         Replay      => True
         );
   end Update_In_Freeze;

   -------------------------------------------------------------------------------
   -- Routine to save a snapshot of the instance to the given stream.
   --
   -- Note that the predefined attributes 'Write and 'Output can be used on any
   -- object to save its value to the given stream. This routine should be
   -- written so that data is written to the stream in the same order that it will
   -- be read back.
   -------------------------------------------------------------------------------
   procedure Save
     (An_Instance : in out Instance;
      To_Stream   : access Ada.Streams.Root_Stream_Type'Class) is

   begin
      if IOSHostCommsReceiveBuffer /= System.Null_Address then
        Ada.Streams.Write
          (Stream => To_Stream.all,
           Item   => IOS_Buffer_Convert.To_Pointer(IOSHostCommsReceiveBuffer).all
           );
     else
        Log.Report ("No IOS Saving - No Receive buffer available");
     end if;
   end Save;

   -------------------------------------------------------------------------------
   -- Routine to retrieve a snapshot of the instance from the given stream.
   --
   -- Note that the predefined attributes 'Read and 'Input can be used on any
   -- object to retrieve its value in the given stream. This routine should be
   -- written so that data is read from the stream in the same order it was placed
   -- into the stream.
   -------------------------------------------------------------------------------
   procedure Restore
     (An_Instance : in out Instance;
      From_Stream : access Ada.Streams.Root_Stream_Type'Class) is

      Last_Element : Ada.Streams.Stream_Element_Offset;
   begin
      Ada.Streams.Read
       (Stream => From_Stream.all,
        Item   => IOS_Buffer_Convert.To_Pointer(IOSHostCommsReceiveBuffer).all,
        Last   => Last_Element
        );

      if Last_Element > 0 then
         if Last_Element /= IOS_Buffer_Stream'Length then
            Log.Report
              (Event => "Invalid IOS snapshot size. Expected" &
               Integer'Image(IOS_Buffer_Stream'Length) &
               " but read" & Ada.Streams.Stream_Element_Offset'Image(Last_Element),
               Severity => Log.Error
               );
         else
            IOSHostCommsProcessCommands(Restore => True);
         end if;
      end if;

   end Restore;
   -------------------------------------------------------------------------------
   -- Returns the maximum amount of *bytes* that a snapshot takes up. This is
   -- used by Create_Streams to help size some internal buffers.
   -------------------------------------------------------------------------------
   function Snapshot_Size (An_Instance : Instance) return Natural is
   begin
      return Natural(IOSHostCommsMaxReceiveBuffer);
   end Snapshot_Size;


begin
   Scheduler_Collection.Register (This_Scheduler'Access);
end Jpats_HostComms_Scheduler;
