/////////////////////////////////////////////////////////////////////////////
//
//           F L I G H T S A F E T Y   I N T E R N A T I O N A L
//                     Simulation Systems Division
//                      2700 North Hemlock Circle
//                     Broken Arrow, Oklahoma 74012
//                          (918) 259-4000
/////////////////////////////////////////////////////////////////////////////
//
// DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
// Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
// technical or operational data or information from automatic dissemination
// under the International Exchange Program or by other means.  This protection
// covers information required solely for administrative or operational
// purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
//
// WARNING:  This document contains technical data whose export is restricted
// by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
// Executive Order 12470.  Violation of these export control laws is subject
// to severe criminal penalties.  Dissemination of this document is controlled
// under DOD Directive 5230.25
//
/////////////////////////////////////////////////////////////////////////////
//
//
// Filename         : IOSProtocol.cpp
//
// Date             : 06 May 1999
//
// Engineer         : Billy Baker
//
// Revision         : $Revision: 1.10 $
//
// Description      : Protocol.cpp contains the implementation of the methods 
//                    to start the communications process, process commands
//                    from the client, and cleanup the communications process.
//                    The HostComms method should be called at a regular 
//                    interval by another function that schedules the calls
//                    at a periodic rate.
//
//                    The protocol implemented here will send a packet of
//                    MAX_SEND_BUFFER every time that HostComms is called.
//                    The contents of the buffer is determined by commands sent
//                    from the client.  
//
// Classification   : UNCLASSIFIED
//
// Requirements     : None.
//
// Components Used  : CGenericHostSockets, CIOSProtocol.
//
// Operational 
//    Restrictions  : Machine dependencies/restrictions
//                        None.
//                    Design dependencies/restrictions
//                        None.
//                    Operations containing dependencies/restrictions
//                        None.
//                    Compiler dependencies/restrictions
//                        None.
//                    Other conditions for proper execution
//                        None.
//
// Environment      : Operating system(s) - Microsoft Windows NT 4.0 with
//                                              NT service pack 4
//                                          Microsoft Windows NT 2000
//                                          VXWorks
//
//                    Compiler(s) - Visual C++ 6.0
//                                  Tornado 1.0.1
//
//                    Architechure(s) - Pentium II
/////////////////////////////////////////////////////////////////////////////

/////////////////////////////////////////////////////////////////////////////
//
//                              R e v i s i o n   H i s t o r y
//
/////////////////////////////////////////////////////////////////////////////
// $Log: IOSProtocol.cpp $
// Revision 1.10  2000/02/28 18:35:25  billyb
// Changed printfs to fprintfs.
// Revision 1.9  2000/02/28 18:17:06  billyb
// Changed error message define wrappers.  Fixed crash when
// trying to change a variable that had an error.
// Revision 1.8  2000/02/14 09:23:02  billyb
// Changed RegisterIOSVariable to take two ints for two
// dimensional arrays.  Changed CHANGE_VARIABLE to set
// values in two dimensions.
// Revision 1.7  2000/02/10 21:02:12  billyb
// Array updates will send a 0 as the third parameter to
// an ArrayMutator until the last index is written at 
// which time a 1 will be sent.
// Revision 1.6  2000/02/09 20:19:27  billyb
// Changed length of data from the host to a short.  Changed
// CHANGE_VARIABLE to allow for updating an entire array.
// Revision 1.5  2000/01/26 09:30:01  billyb
// Corrected warnings about type mismatches.
// Revision 1.4  2000/01/26 09:25:28  billyb
// Rewrote ProcessCommands to loop over all possible commands.
// Added support for changing an array element.  Removed memcpys.
// Added filling buffer to client with array data.  Added Register function
// and RegisterIOSVariableEx.
// Revision 1.3  1999/11/30 08:10:50  billyb
// Shifted code around for dealing with ADD_VARIABLE errors.
// Added size mismatch error in a commented section.
// Revision 1.2  1999/11/19 23:27:43  billyb
// Fixed incorrect conditionals dealing with last index.
/////////////////////////////////////////////////////////////////////////////
#include "HostComms.h"
#include "GenericHostSockets.h"
#include "Protocol.h"
#include "iosProtocol.h"
#include "log_report.h"

CIOSProtocol::CIOSProtocol()
{
  m_pSockets = new CGenericHostSockets(IOS_PORT, 
                                       IOS_MAX_RECEIVE_BUFFER, 
                                       IOS_RECEIVE_BUFFER_CACHE, 
                                       IOS_MAX_SEND_BUFFER, 
                                       IOS_SEND_BUFFER_CACHE);

  memset(m_ReadAddresses, 0, IOS_MAX_VARIABLES * sizeof(Variable));
  memset(m_Variables, 0, IOS_MAX_VARIABLES * sizeof(Variable));
  memset(m_Errors, 0, IOS_MAX_VARIABLES * sizeof(Error));
}

CIOSProtocol::~CIOSProtocol()
{
  // The base class will take care of deleting m_pSockets.
}

/////////////////////////////////////////////////////////////////////////////
//
// ProcessCommands
//
// Inputs           : None.
//
// Return Values    : None.
//
// Date             : 06 May 1999
//
// Engineer         : Billy Baker
//
// Description      : ProcessCommands should be called after MAX_RECEIVE_BUFFER
//                    has been read from the socket.  ProcessCommands will 
//                    parse the data from the socket and then take the 
//                    appropriate action.
//                    ProcessChanges should be cleared to prevent processing
//                    of normal variable change commands.
//                    ProcessProtocol should be cleared to prevent processing
//                    of everthing *except* normal variable change commands.
/////////////////////////////////////////////////////////////////////////////
void CIOSProtocol::ProcessCommands() {ProcessCommands (true, true);};

void CIOSProtocol::ProcessCommands(BOOL ProcessChanges, BOOL ProcessProtocol)
{
  if (m_pSockets == NULL)
    {
      return;
    }

  char *pcBuf = m_pSockets->m_cReceiveBuffer;
  BOOL bDone  = FALSE;

  // Get the command type
  short shCommand;
  shCommand = *(short *)pcBuf;
  shCommand = m_pSockets->Ntohs(shCommand);
        
  pcBuf += 2;

  while (shCommand < DIAGNOSTICS &&
         shCommand >= RESET      &&
         !bDone)
    {
      switch (shCommand)
        {
        case ADD_VARIABLE:
          {
            // Get the index with which to work
            unsigned short int ushIndex = IOS_MAX_VARIABLES;
            unsigned short int ushLength      = 0;

            if ((pcBuf + 4 - m_pSockets->m_cReceiveBuffer) < IOS_MAX_RECEIVE_BUFFER)
              {
                ushIndex = *(unsigned short *)pcBuf;

                ushIndex = m_pSockets->Ntohs(ushIndex);

                pcBuf += 2;

                // Get the length of the variable string
                ushLength = *(unsigned short*)pcBuf;

                ushLength = m_pSockets->Ntohs(ushLength);

                pcBuf += 2;
              }
            else
              {
                bDone = TRUE;
              }

            if (ushLength > 0                     && 
                (pcBuf + min(255, ushLength) + 2 - m_pSockets->m_cReceiveBuffer) < IOS_MAX_RECEIVE_BUFFER)
              {
                short shLastUniqueCount = m_shUniqueErrors;

                // Copy the variable string
                char cVariableName[256];
                memcpy(cVariableName, pcBuf, min(255, ushLength));
                cVariableName[min(255, ushLength)] = '\0';
                pcBuf += min(255, ushLength);

		if (ProcessProtocol) 
		  {

		    // Get the size that the IOS expects
		    unsigned short ushSize;
		    ushSize = *(unsigned short *)pcBuf;
		    ushSize = m_pSockets->Ntohs(ushSize);
		    pcBuf += 2;

		    BOOL bFound         = FALSE;
		    int nVariablePos    = 0;
		    while (nVariablePos < m_shPossibleVariables && bFound == FALSE)
		      {
			if (strncmp(m_Variables[nVariablePos].s_cName, 
				    cVariableName,
				    MAX_VARIABLE_NAME_LEN - 1) == 0)
			  {
			    bFound = TRUE;
			  }

			nVariablePos++;
		      }

		    nVariablePos--;

                // If the variable name sent can be found and the number of
                // bytes will not overflow the buffer, then add the variable.
                // Also, the maximum number of variables has not been reached,
                // the size from the client equals the size on the host, and
                // the reference count is zero (has not been added once).
		    if (bFound == TRUE                                   &&
			(m_lBytesAdded + ushSize) <= IOS_MAX_SEND_BUFFER && 
			ushIndex < IOS_MAX_VARIABLES                     &&
			//                    ushSize == m_Variables[nVariablePos].s_ushSize   &&
			m_Variables[nVariablePos].s_ushReferenceCount == 0)
		      {
			Variable var;

			// Found the variable.  Add it to the array of read addresses.
			var.s_lAddress          = m_Variables[nVariablePos].s_lAddress;
			m_Variables[nVariablePos].s_ushReferenceCount++;
			var.s_ushSize           = ushSize;
			var.s_cName[0]          = '\0';
			var.s_pChangeValue      = m_Variables[nVariablePos].s_pChangeValue;
			var.s_bPlayBack         = m_Variables[nVariablePos].s_bPlayBack;
			var.s_ushReferenceCount = 1;
			var.s_lLength1D         = m_Variables[nVariablePos].s_lLength1D;
			var.s_lLength2D         = m_Variables[nVariablePos].s_lLength2D;

			m_lBytesAdded += ushSize;

			m_ReadAddresses[ushIndex] = var;
			m_shLastIndex = ushIndex;

			m_shVarCount++;
			m_bVarCountChanged = true;

#ifdef PRINT_STATISTICS

         char msg[256];
         sprintf(msg,"IOSProtocol::ProcessCommands : Added %s at index %d\n", cVariableName, (unsigned int)ushIndex);
         msg[(sizeof msg) - 1] = 0;

         Log_Report(msg, Log_Informational);

			//Log_Report("IOSProtocol::ProcessCommands : Added %s \n", cVariableName, Log_Informational);
#endif
		      }
		    else
		      {
			bool bFillError = false;

			if (bFound == TRUE && (m_lBytesAdded + ushSize) > IOS_MAX_SEND_BUFFER)
			  {
			    if (m_shUniqueErrors < IOS_MAX_VARIABLES)
			      {
				m_Errors[m_shUniqueErrors].s_ucError |= BUFFER_EXCEEDED;
				m_Errors[m_shUniqueErrors].s_ushIndex = ushIndex;
			      }

			    if (shLastUniqueCount == m_shUniqueErrors)
			      {
				m_shUniqueErrors++;
			      }

			    m_shBufferExceededVariables++;
			    m_bBufferExceededChanged = true;

#ifdef PRINT_ERRORS
			    Log_Report("IOSProtocol::ProcessCommands : Could not add %s -- buffer max "
				       "reached\n", cVariableName, Log_Error);
#endif
			  }

			if (ushIndex >= IOS_MAX_VARIABLES)
			  {
			    if (m_shUniqueErrors < IOS_MAX_VARIABLES)
			      {
				m_Errors[m_shUniqueErrors].s_ucError |= MAX_VAR_EXCEEDED;
				m_Errors[m_shUniqueErrors].s_ushIndex = ushIndex;
			      }

			    if (shLastUniqueCount == m_shUniqueErrors)
			      {
				m_shUniqueErrors++;
			      }

			    m_shMaxVarExceededVariables++;
			    m_bMaxVarExceededChanged = true;

#ifdef PRINT_ERRORS

			    Log_Report("IOSProtocol::ProcessCommands : Could not add %s -- max symbols "
				       "reached\n", cVariableName, Log_Error);
#endif
			  }

			// Check the reference count to see if it has been added once.
			if (bFound == TRUE && m_Variables[nVariablePos].s_ushReferenceCount > 0)
			  {
			    bFillError = true;

			    if (m_shUniqueErrors < IOS_MAX_VARIABLES)
			      {
				m_Errors[m_shUniqueErrors].s_ucError |= DUPLICATE_VARIABLE;
				m_Errors[m_shUniqueErrors].s_ushIndex = ushIndex;
			      }

			    if (shLastUniqueCount == m_shUniqueErrors)
			      {
				m_shUniqueErrors++;
			      }

			    m_shDuplicateVariables++;
			    m_bDuplicateChanged = true;

#ifdef PRINT_ERRORS
			    Log_Report("IOSProtocol::ProcessCommands : Could not add %s - "
				       "duplicate\n", cVariableName, Log_Warning);
#endif
			  }

			// If the variable is not found.
			if (bFound == FALSE)
			  {
			    bFillError = true;

			    if (m_shUniqueErrors < IOS_MAX_VARIABLES)
			      {
				m_Errors[m_shUniqueErrors].s_ucError |= NOT_FOUND;
				m_Errors[m_shUniqueErrors].s_ushIndex = ushIndex;
			      }

			    if (shLastUniqueCount == m_shUniqueErrors)
			      {
				m_shUniqueErrors++;
			      }

			    m_shNotFoundVariables++;
			    m_bNotFoundChanged = true;

#ifdef PRINT_NOT_FOUND_ERRORS
			    Log_Report("IOSProtocol::ProcessCommands : %s not found\n", cVariableName, Log_Warning);
#endif
			  }

			/*
			  // If the size from the client doesn't match the size
			  // on the host.
			  if (ushSize != m_Variables[nVariablePos].s_ushSize)
			  {
			  bFillError = true;

			  if (m_shUniqueErrors < IOS_MAX_VARIABLES)
			  {
			  m_Errors[m_shUniqueErrors].s_ucError |= SIZE_MISMATCH;
			  m_Errors[m_shUniqueErrors].s_ushIndex = ushIndex;
			  }

			  if (shLastUniqueCount == m_shUniqueErrors)
			  {
			  m_shUniqueErrors++;
			  }

			  m_shSizeMismatchVariables++;
			  m_bSizeMismatchChanged = true;

			  #ifdef PRINT_ERRORS
			  Log_Report("IOSProtocol::ProcessCommands : Could not add %s - "
			  "size mismatch\n", cVariableName, Log_Error);
			  #endif
			  }
			*/

			// Size may not match the real variable size. 
			// ReadValues will just set ushSize of the buffer
			// to 0.  ushSize is what the client expects to read
			// for this variable even if there is an error with
			// it.  The fill errors are duplicates, size mismatch
			// and not found.  ushSize added to the length of data
			// can not exceed the total buffer.
			if (bFillError == true)
			  {
			    if ((m_lBytesAdded + ushSize) <= IOS_MAX_SEND_BUFFER && 
				ushIndex < IOS_MAX_VARIABLES)
			      {
				Variable var;

				var.s_lAddress              = NULL;
				var.s_ushSize               = ushSize;
				var.s_cName[0]              = '\0';
				var.s_pChangeValue          = NULL;
				var.s_ushReferenceCount     = 1;

				m_lBytesAdded += ushSize;

				m_ReadAddresses[ushIndex]   = var;
				m_shLastIndex               = ushIndex;
			      }
			  }
		      }
		  }
              }
            else if (ushLength > 0)
              {
                bDone = TRUE;
              }
          }
          break;
        case DELETE_VARIABLE:
          {
            // Get the index with which to work
            unsigned short int ushIndex = m_shLastIndex;
            ushIndex++;

            if ((pcBuf + 2 - m_pSockets->m_cReceiveBuffer) < IOS_MAX_RECEIVE_BUFFER)
              {
                ushIndex = *(unsigned short *)pcBuf;

                ushIndex = m_pSockets->Ntohs(ushIndex);

                pcBuf += 2;
              }
            else
              {
                bDone = TRUE;
              }

            if (ProcessProtocol && (ushIndex <= m_shLastIndex))
              {
                m_lBytesAdded -= m_ReadAddresses[ushIndex].s_ushSize;

                if (m_ReadAddresses[ushIndex].s_lAddress != NULL)
                  {
                    m_shVarCount--;
                  }

                int i = ushIndex + 1; 
                while (i <= m_shLastIndex)
                  {
                    m_ReadAddresses[i-1] = m_ReadAddresses[i];
                    i++;
                  }

                m_shLastIndex--;


#ifdef PRINT_STATISTICS
                Log_Report("IOSProtocol::ProcessCommands : Deleted %d\n", ushIndex, Log_Informational);
#endif

              }
            else
              {
#ifdef PRINT_ERRORS
		if (ProcessProtocol) 
		  {
		    Log_Report("IOSProtocol::ProcessCommands : Could not delete %d\n", ushIndex, Log_Error);
		  }
#endif
              }
          }
          break;
        case CHANGE_VARIABLE:
          {
            // Get the index with which to work
            unsigned short int ushIndex = m_shLastIndex;
            ushIndex++;
            unsigned short ushLength    = 0;
            long int lOffset1D           = 0;
            long int lOffset2D           = 0;
            long int lNumberOfIndices   = 0;
            unsigned short int ushSize  = 0;
            char* pcData                = NULL;

            if ((pcBuf + 4 - m_pSockets->m_cReceiveBuffer) < IOS_MAX_RECEIVE_BUFFER)
              {
                ushIndex = *(unsigned short *)pcBuf;

                ushIndex = m_pSockets->Ntohs(ushIndex);

                pcBuf += 2;

                // Get the length of the variable data
                ushLength = *(unsigned short *)pcBuf;

                ushLength = m_pSockets->Ntohs(ushLength);

                // Move to the data.
                pcBuf += 2;
              }
            else
              {
                bDone = TRUE;
              }

            // while there is data to send to a mutator method
            // and the command is CHANGE_VARIABLE
            if (ushLength > 0                     && 
                (pcBuf + ushLength + sizeof(lOffset1D) + sizeof(lOffset2D) - m_pSockets->m_cReceiveBuffer) < IOS_MAX_RECEIVE_BUFFER)
              {
                if (ushIndex <= m_shLastIndex)
                  {
                    pcData = pcBuf;
                    pcBuf += ushLength;

                    lOffset1D = *(long int *)pcBuf;
                    lOffset1D = m_pSockets->Ntohl(lOffset1D);

                    pcBuf += sizeof(lOffset1D);

                    lOffset2D = *(long int *)pcBuf;
                    lOffset2D = m_pSockets->Ntohl(lOffset2D);

                    pcBuf += sizeof(lOffset2D);

                    if (m_ReadAddresses[ushIndex].s_pChangeValue != NULL &&
			(ProcessChanges || !m_ReadAddresses[ushIndex].s_bPlayBack) &&
			(ProcessProtocol || m_ReadAddresses[ushIndex].s_bPlayBack)
			)
                      {
                        if (m_ReadAddresses[ushIndex].s_lLength1D < 1 ||
                            m_ReadAddresses[ushIndex].s_lLength2D < 1)
                          {
                            break;
                          }

                        // Figure out how many indices in the array will probably
                        // be changed. For example, an array of chars has a length of 16.
                        // s_ushSize below would be 16 and s_lLength would be 16.  
                        // If ushLength is 4, then 4 / (16/16) says that four characters
                        // would change.  For an array of 32 bit ints that has a length
                        // of 16, then s_ushSize would be 64 and s_lLength would be 16.
                        // If ushLength is 4, the 4 / (64/16) says that only one index
                        // would change.
                        ushSize = (unsigned short)(m_ReadAddresses[ushIndex].s_ushSize / 
                                                   (m_ReadAddresses[ushIndex].s_lLength1D *
                                                    m_ReadAddresses[ushIndex].s_lLength2D));

                        if (ushSize < 1)
                          {
                            break;
                          }

                        lNumberOfIndices = ushLength / ushSize;

                        // We should be able to update a variable.
                        // This code is specific to the type of simulation with which
                        // the host is running.
                        switch (ushSize)
                          {
                          case 2:
                            {
                              short shTemp;
                              if (m_ReadAddresses[ushIndex].s_lLength1D > 1 ||
                                  m_ReadAddresses[ushIndex].s_lLength2D > 1 )
                                {
                                  int   nIndexStep            = 0;
                                  long  lOffset2D_From_Client  = lOffset2D;
                                  while (nIndexStep < lNumberOfIndices &&
                                         lOffset1D   < m_ReadAddresses[ushIndex].s_lLength1D)
                                    {
                                      while (nIndexStep < lNumberOfIndices &&
                                             lOffset2D  < m_ReadAddresses[ushIndex].s_lLength2D)
                                        {
                                          shTemp = *(short *)pcData;

                                          shTemp = m_pSockets->Ntohs(shTemp);

                                          pcData += 2;

                                          if ((nIndexStep + 1) == lNumberOfIndices)
                                            {
                                              ((ArrayMutator)m_ReadAddresses[ushIndex].s_pChangeValue)(&shTemp, 
                                                                                                       lOffset1D, 
                                                                                                       lOffset2D, 
                                                                                                       1);
                                            }
                                          else
                                            {
                                              ((ArrayMutator)m_ReadAddresses[ushIndex].s_pChangeValue)(&shTemp, 
                                                                                                       lOffset1D, 
                                                                                                       lOffset2D, 
                                                                                                       0);
                                            }

                                          nIndexStep++;
                                          lOffset2D++;
                                        }

                                      lOffset2D = lOffset2D_From_Client;
                                      lOffset1D++;
                                    }
                                }
                              else
                                {
                                  shTemp = *(short *)pcData;

                                  shTemp = m_pSockets->Ntohs(shTemp);

                                  pcData += 2;

                                  ((Mutator)m_ReadAddresses[ushIndex].s_pChangeValue)(&shTemp);
                                }
                            }
                            break;
                          case 4:
                            {
                              long lTemp;
                              if (m_ReadAddresses[ushIndex].s_lLength1D > 1 ||
                                  m_ReadAddresses[ushIndex].s_lLength2D > 1 )
                                {
                                  int   nIndexStep            = 0;
                                  long  lOffset2D_From_Client  = lOffset2D;
                                  while (nIndexStep < lNumberOfIndices &&
                                         lOffset1D   < m_ReadAddresses[ushIndex].s_lLength1D)
                                    {
                                      while (nIndexStep < lNumberOfIndices &&
                                             lOffset2D   < m_ReadAddresses[ushIndex].s_lLength2D)
                                        {
                                          lTemp = *(long *)pcData;

                                          lTemp = m_pSockets->Ntohl(lTemp);

                                          pcData += 4;

                                          if ((nIndexStep + 1) == lNumberOfIndices)
                                            {
                                              ((ArrayMutator)m_ReadAddresses[ushIndex].s_pChangeValue)(&lTemp, 
                                                                                                       lOffset1D, 
                                                                                                       lOffset2D, 
                                                                                                       1);
                                            }
                                          else
                                            {
                                              ((ArrayMutator)m_ReadAddresses[ushIndex].s_pChangeValue)(&lTemp, 
                                                                                                       lOffset1D, 
                                                                                                       lOffset2D, 
                                                                                                       0);
                                            }

                                          nIndexStep++;
                                          lOffset2D++;
                                        }

                                      lOffset2D = lOffset2D_From_Client;
                                      lOffset1D++;
                                    }
                                }
                              else
                                {
                                  lTemp = *(long *)pcData;

                                  lTemp = m_pSockets->Ntohl(lTemp);

                                  pcData += 4;

                                  ((Mutator)m_ReadAddresses[ushIndex].s_pChangeValue)(&lTemp);
                                }
                            }
                            break;
                          case 8:
                            {
                              double dTemp;
                              if (m_ReadAddresses[ushIndex].s_lLength1D > 1 ||
                                  m_ReadAddresses[ushIndex].s_lLength2D > 1 )
                                {
                                  int   nIndexStep            = 0;
                                  long  lOffset2D_From_Client  = lOffset2D;
                                  while (nIndexStep < lNumberOfIndices &&
                                         lOffset1D   < m_ReadAddresses[ushIndex].s_lLength1D)
                                    {
                                      while (nIndexStep < lNumberOfIndices &&
                                             lOffset2D   < m_ReadAddresses[ushIndex].s_lLength2D)
                                        {
                                          dTemp = *(double *)pcData;

                                          dTemp = m_pSockets->Ntohd(dTemp);

                                          pcData += 8;

                                          if ((nIndexStep + 1) == lNumberOfIndices)
                                            {
                                              ((ArrayMutator)m_ReadAddresses[ushIndex].s_pChangeValue)(&dTemp, 
                                                                                                       lOffset1D, 
                                                                                                       lOffset2D, 
                                                                                                       1);
                                            }
                                          else
                                            {
                                              ((ArrayMutator)m_ReadAddresses[ushIndex].s_pChangeValue)(&dTemp, 
                                                                                                       lOffset1D, 
                                                                                                       lOffset2D, 
                                                                                                       0);
                                            }

                                          nIndexStep++;
                                          lOffset2D++;
                                        }

                                      lOffset2D = lOffset2D_From_Client;
                                      lOffset1D++;
                                    }
                                }
                              else
                                {
                                  dTemp = *(double *)pcData;

                                  dTemp = m_pSockets->Ntohd(dTemp);

                                  pcData += 8;

                                  ((Mutator)m_ReadAddresses[ushIndex].s_pChangeValue)(&dTemp);
                                }
                            }
                            break;
                          default:
                            {
                              if (m_ReadAddresses[ushIndex].s_lLength1D > 1 ||
                                  m_ReadAddresses[ushIndex].s_lLength2D > 1 )
                                {
                                  int   nIndexStep            = 0;
                                  long  lOffset2D_From_Client  = lOffset2D;
                                  while (nIndexStep < lNumberOfIndices &&
                                         lOffset1D   < m_ReadAddresses[ushIndex].s_lLength1D)
                                    {
                                      while (nIndexStep < lNumberOfIndices &&
                                             lOffset2D   < m_ReadAddresses[ushIndex].s_lLength2D)
                                        {
                                          if ((nIndexStep + 1) == lNumberOfIndices)
                                            {
                                              ((ArrayMutator)m_ReadAddresses[ushIndex].s_pChangeValue)(pcData, 
                                                                                                       lOffset1D, 
                                                                                                       lOffset2D, 
                                                                                                       1);
                                            }
                                          else
                                            {
                                              ((ArrayMutator)m_ReadAddresses[ushIndex].s_pChangeValue)(pcData, 
                                                                                                       lOffset1D, 
                                                                                                       lOffset2D, 
                                                                                                       0);
                                            }

                                          nIndexStep++;
                                          lOffset2D++;
                                          pcData += ushSize;
                                        }

                                      lOffset2D = lOffset2D_From_Client;
                                      lOffset1D++;
                                    }
                                }
                              else
                                {
                                  ((Mutator)m_ReadAddresses[ushIndex].s_pChangeValue)(pcData);
                                }
                            }
                            break;
                          }
                      }
                  }
                else
                  {
                  }
              }
            else if (ushLength > 0)
              {
                bDone = TRUE;
              }
          }
          break;
        case RESET:
          {
	    if (ProcessProtocol) {
#ifdef PRINT_STATISTICS
	      Log_Report("IOSProtocol::ProcessCommands : Resetting\n", Log_Informational);
#endif

	      m_shVarCount                = 0;
	      m_lBytesAdded               = 0;
	      m_shUniqueErrors            = 0;
	      m_shNotFoundVariables       = 0;
	      m_shDuplicateVariables      = 0;
	      m_shBufferExceededVariables = 0;
	      m_shMaxVarExceededVariables = 0;
	      m_shLastIndex               = 0;
	      m_shSizeMismatchVariables   = 0;

	      m_bNotFoundChanged          = false;
	      m_bDuplicateChanged         = false;
	      m_bVarCountChanged          = false;
	      m_bBufferExceededChanged    = false;
	      m_bMaxVarExceededChanged    = false;
	      m_bSizeMismatchChanged      = false;

	      int i = 0; 
	      while (i < IOS_MAX_VARIABLES)
		{
		  m_Variables[i].s_ushReferenceCount = 0;
		  i++;
		}

	      memset(m_Errors, 0, IOS_MAX_VARIABLES * sizeof(Error));
	    }
          }
          break;
        };

      if (pcBuf + 2 - m_pSockets->m_cReceiveBuffer < IOS_MAX_RECEIVE_BUFFER &&
          bDone == FALSE)
        {
          // Get the command type
          shCommand = *(short *)pcBuf;
          shCommand = m_pSockets->Ntohs(shCommand);

          pcBuf += 2;
        }
      else
        {
          bDone = TRUE;
        }
    }

  return;
}

/////////////////////////////////////////////////////////////////////////////
//
// ReadValues
//
// Inputs           : None.
//
// Return Values    : None..
//
// Date             : 06 May 1999
//
// Engineer         : Billy Baker
//
// Description      : ReadValues will read all of the addresses in 
//                    IOSProtocol.m_ReadAddresses and store the data in the send 
//                    buffer in network byte ordering.  If no variables need to 
//                    be read, the send buffer is simply filled with zeroes.
/////////////////////////////////////////////////////////////////////////////
bool CIOSProtocol::ReadValues()
{
  if (m_pSockets == NULL)
    {
      return false;
    }

  short shCommand;
  shCommand = *(short *)m_pSockets->m_cReceiveBuffer;
  shCommand = m_pSockets->Ntohs(shCommand);

  if (shCommand < CHANGE_VARIABLE)
    {
      // If the state of sending data is about to change, don't send
      // anything.
      return false;
    }

  // Make sure there is something to send.
  memset(m_pSockets->m_cSendBuffer, 0, IOS_MAX_SEND_BUFFER);

  short int shCount       = m_shVarCount;

  char* pcData = m_pSockets->m_cSendBuffer;

  if ((m_bBufferExceededChanged | m_bDuplicateChanged | m_bNotFoundChanged |
       m_bVarCountChanged | m_bSizeMismatchChanged) > 0)
    {
      // The amount of data that is being sent to the client
      // has changed since the last time this method was invoked.
      // Send a diagnostics message to let the client know about
      // the state of things on the host.
      m_bBufferExceededChanged    = false;
      m_bDuplicateChanged         = false;
      m_bNotFoundChanged          = false;
      m_bVarCountChanged          = false;
      m_bSizeMismatchChanged      = false;


      if (IOS_MAX_SEND_BUFFER > 16)
        {
          short sh = DIAGNOSTICS;

          sh = m_pSockets->Htons(sh);
          *(short *)pcData = sh;
          pcData += 2;

          sh = m_shUniqueErrors;
          sh = m_pSockets->Htons(sh);
          *(short *)pcData = sh;
          pcData += 2;

          sh = m_shVarCount;
          sh = m_pSockets->Htons(sh);
          *(short *)pcData = sh;
          pcData += 2;

          sh = m_shNotFoundVariables;
          sh = m_pSockets->Htons(sh);
          *(short *)pcData = sh;
          pcData += 2;

          sh = m_shDuplicateVariables;
          sh = m_pSockets->Htons(sh);
          *(short *)pcData = sh;
          pcData += 2;

          sh = m_shBufferExceededVariables;
          sh = m_pSockets->Htons(sh);
          *(short *)pcData = sh;
          pcData += 2;

          sh = m_shMaxVarExceededVariables;
          sh = m_pSockets->Htons(sh);
          *(short *)pcData = sh;
          pcData += 2;

          sh = m_shSizeMismatchVariables;
          sh = m_pSockets->Htons(sh);
          *(short *)pcData = sh;
          pcData += 2;

          int i = 0;
          while (i < m_shUniqueErrors && 
                 (pcData + 3 - m_pSockets->m_cSendBuffer) < IOS_MAX_SEND_BUFFER)
            {
              sh = m_Errors[i].s_ushIndex;
              sh = m_pSockets->Htons(sh);
              *(short *)pcData = sh;
              pcData += 2;

              *(unsigned char *)pcData = m_Errors[i].s_ucError;
              pcData++;

              i++;
            }

          return true;
        }

      return false;
    }
    
  // Skip the first 2 bytes which will have the number of variables
  // (0 to ...) or a status message (-1 to -N) and other statistics.
  pcData += 2;  
  int nVariableIndex = 0;
  int i = 0;
  unsigned short ushSize;
  while (i <= m_shLastIndex &&
         (pcData + m_ReadAddresses[nVariableIndex].s_ushSize - m_pSockets->m_cSendBuffer) < IOS_MAX_SEND_BUFFER)
    {
      if (m_ReadAddresses[nVariableIndex].s_lAddress != NULL)
        {
          memcpy(pcData, (void *)(m_ReadAddresses[nVariableIndex].s_lAddress), 
                 m_ReadAddresses[nVariableIndex].s_ushSize);

          // Put the data into network byte ordering.
          if ((m_ReadAddresses[nVariableIndex].s_lLength1D * 
               m_ReadAddresses[nVariableIndex].s_lLength2D) > 1)
            {
              char* pcArray = pcData;
              ushSize =  (unsigned short)(m_ReadAddresses[nVariableIndex].s_ushSize / 
                                          (m_ReadAddresses[nVariableIndex].s_lLength1D *
                                           m_ReadAddresses[nVariableIndex].s_lLength2D));
              switch (ushSize)
                {
                case 2:
                  {
                    short sh;
                    long lAddress = m_ReadAddresses[nVariableIndex].s_lAddress;
                    while (lAddress - m_ReadAddresses[nVariableIndex].s_lAddress < m_ReadAddresses[nVariableIndex].s_ushSize)
                      {
                        sh = *(short *)(lAddress);

                        sh = m_pSockets->Htons(sh);

                        *(short *)pcArray = sh;

                        lAddress += 2;
                        pcArray  += 2;
                      }
                  }
                  break;
                case 4:
                  {
                    long l;
                    long lAddress = m_ReadAddresses[nVariableIndex].s_lAddress;
                    while (lAddress - m_ReadAddresses[nVariableIndex].s_lAddress < m_ReadAddresses[nVariableIndex].s_ushSize)
                      {
                        l = *(long *)lAddress;

                        l = m_pSockets->Htonl(l);

                        *(long *)pcArray = l;

                        lAddress += 4;
                        pcArray  += 4;
                      }
                  }
                  break;
                case 8:
                  {
                    double d;
                    long lAddress = m_ReadAddresses[nVariableIndex].s_lAddress;
                    while (lAddress - m_ReadAddresses[nVariableIndex].s_lAddress < m_ReadAddresses[nVariableIndex].s_ushSize)
                      {
                        d = *(double *)lAddress;

                        d = m_pSockets->Htond(d);

                        *(double *)pcArray = d;

                        lAddress += 8;
                        pcArray  += 8;
                      }
                  }
                  break;
                };

            }
          else if ((m_ReadAddresses[nVariableIndex].s_lLength1D * 
                    m_ReadAddresses[nVariableIndex].s_lLength2D) == 1)
            {
              switch (m_ReadAddresses[nVariableIndex].s_ushSize)
                {
                case 2:
                  {
                    short sh;
                        
                    sh = *(short *)m_ReadAddresses[nVariableIndex].s_lAddress;

                    sh = m_pSockets->Htons(sh);

                    *(short *)pcData = sh;
                  }
                  break;
                case 4:
                  {
                    long l;

                    l = *(long *)m_ReadAddresses[nVariableIndex].s_lAddress;

                    l = m_pSockets->Htonl(l);

                    *(long *)pcData = l;
                  }
                  break;
                case 8:
                  {
                    double d;

                    d = *(double *)m_ReadAddresses[nVariableIndex].s_lAddress;

                    d = m_pSockets->Htond(d);

                    *(double *)pcData = d;
                  }
                  break;
                };
            }
        }
      else
        {
          memset(pcData, 0, m_ReadAddresses[nVariableIndex].s_ushSize);
        }

      // move to the next starting position in the send buffer.
      pcData += m_ReadAddresses[nVariableIndex].s_ushSize;
      nVariableIndex++;

      i++;
    }

  // Set the number of variables to send.
  shCount = m_pSockets->Htons(shCount);
  *(short *)m_pSockets->m_cSendBuffer = shCount;

  return true;
}

/////////////////////////////////////////////////////////////////////////////
//
//                          The IOS HOST
//
/////////////////////////////////////////////////////////////////////////////
CIOSProtocol*       pIOSProtocol = NULL;

#ifdef __cplusplus
extern "C"
{
#endif // __cplusplus

  /////////////////////////////////////////////////////////////////////////////
  //
  // IOSHostCommsExitRoutine
  //
  // Inputs           : None.
  //
  // Return Values    : None for standard C.  VxWorks needs an int.
  //
  // Date             : 06 May 1999
  //
  // Engineer         : Billy Baker
  //
  // Description      : ExitRoutine is used to clean up the socket connections
  //                    when the process/task is killed/deleted.
  /////////////////////////////////////////////////////////////////////////////
#ifdef WIN32
  void IOSHostCommsExitRoutine()
  {
    if (pIOSProtocol != NULL)
      {
        delete pIOSProtocol;
      }

    WSACleanup();

    return;
  }
#else
#ifdef VXWORKS
  int   IOSHostCommsExitRoutine()
  {
    if (pIOSProtocol != NULL)
      {
        delete pIOSProtocol;
      }

    taskDeleteHookDelete(IOSHostCommsExitRoutine);

    return 0;
  }
#endif
#endif

  /////////////////////////////////////////////////////////////////////////////
  //
  // IOSHostCommsProcessCommands
  //
  // Inputs           : None.
  //
  // Return Values    : None.
  //
  // Date             : 29 June 2000
  //
  // Engineer         : Ted Dennison
  //
  // Description      : Reads and processes any commands that have been
  //                    received from the IOS. If there is no connection or 
  //                    no commands can be immediately read, it just returns.
  //                    Replay should be set during a replay to prevent
  //                    processing of normal variable change commands.  
  //                    Restore should be set to prevent processing of anything
  //                    *but* normal variable change commands.
  /////////////////////////////////////////////////////////////////////////////
  void IOSHostCommsProcessCommands(BOOL Replay, BOOL Restore)
  {
    if (pIOSProtocol == NULL)
      {
        return;
      }

    if (pIOSProtocol->m_pSockets == NULL)
      {
        return;
      }

    if (!Restore) {

      // Validate the listening socket.
      if(!(pIOSProtocol->m_pSockets->IsValidListenSocket()))
	{
	  return;
	}

      // Validate the accept socket.
      if(!(pIOSProtocol->m_pSockets->IsValidAcceptSocket()))
	{
	  return;
	}

      // Read out any data in the socket, if there is any
      pIOSProtocol->m_pSockets->ReceiveData();

    }

    
    // Process any data that was read from the client.
    pIOSProtocol->ProcessCommands(!Replay, !Restore);

  }

  /////////////////////////////////////////////////////////////////////////////
  //
  // IOSHostCommsReadValues
  //
  // Inputs           : None.
  //
  // Return Values    : None.
  //
  // Date             : 29 June 2000
  //
  // Engineer         : Ted Dennison
  //
  // Description      : Gathers up all values that the IOS has requested
  //                    receipt of into a buffer for sending. This should be 
  //                    called before IOSHostComms. It should not be called 
  //                    again until IOSHostComms returns.
  /////////////////////////////////////////////////////////////////////////////
  BOOL IOSHostCommsReadValues ()
  {
    if (pIOSProtocol == NULL)
      {
        return FALSE;
      }

    if (pIOSProtocol->m_pSockets == NULL)
      {
        return FALSE;
      }

    // Validate the listening socket.
    if(!(pIOSProtocol->m_pSockets->IsValidListenSocket()))
      {
        return FALSE;
      }

    // Validate the accept socket.
    if(!(pIOSProtocol->m_pSockets->IsValidAcceptSocket()))
      {
        return FALSE;
      }

    return (pIOSProtocol->ReadValues());
    
  }

  /////////////////////////////////////////////////////////////////////////////
  //
  // BOOL IOSHostCommsConnected
  //
  // Inputs           : None.
  //
  // Return Values    : TRUE if we are connected to the IOS.
  //                    FALSE if we were not connected and no connection could
  //                    be started.
  //
  // Date             : 29 June 2000
  //
  // Engineer         : Ted Dennison
  //
  // Description      : This routine verifies that a connection exists. If it
  //                    doesn't it tries to create one.
  //
  /////////////////////////////////////////////////////////////////////////////
  BOOL IOSHostCommsConnected ()
  {
    if (pIOSProtocol == NULL)
      {
        return FALSE;
      }

    if (pIOSProtocol->m_pSockets == NULL)
      {
        return FALSE;
      }

    // Validate the listening socket.
    if(!(pIOSProtocol->m_pSockets->IsValidListenSocket()))
      {
        // Do not have a listening socket, create one.
        //
        // Attempt to create listen socket.
#ifdef PRINT_STATISTICS
        Log_Report("IOSHostCommsInitialize : Attempting to create listen socket.\n", Log_Informational);
#endif

        if(!pIOSProtocol->m_pSockets->CreateListenSocket())
          {
            // Attempt failed.
#ifdef PRINT_ERRORS
            Log_Report("\nIOSHostCommsInitialize : Failed to create listen socket.\n", Log_Error);
#endif
            return FALSE;
          }
        else
          {
            // Attempt succeeded.
#ifdef PRINT_STATISTICS
            Log_Report("\nIOSHostComms : Created listen socket.\n", Log_Informational);
#endif
          }
      }

    // Validate the accept socket.
    if(!(pIOSProtocol->m_pSockets->IsValidAcceptSocket()))
      {
        // Do not have an accept socket, attempt to create one.
        //
        // This can occur during the first pass, and after any comms error
        // occurs, including client disconnect.  The objective is to continue
        // trying until an accept socket is created.
#ifdef PRINT_STATISTICS
        Log_Report("IOSHostComms : Attempting to create accept socket.\n", Log_Informational);
#endif
 
        if(!pIOSProtocol->m_pSockets->CreateAcceptSocket())
          {return FALSE;}
        else
          {
            // Attempt succeeded.
#ifdef PRINT_STATISTICS
            Log_Report("\nIOSHostComms : Created accept socket.\n", Log_Informational);
#endif
          }
      }
    return TRUE;
  }

  /////////////////////////////////////////////////////////////////////////////
  //
  // BOOL IOSHostCommsInitialize()
  //
  // Inputs           : None.
  //
  // Return Values    : TRUE if an IOS protocol was created.
  //                    FALSE if an IOS protocol was not created. 
  //
  // Date             : 17 August 1999
  //
  // Engineer         : Billy Baker
  //
  // Description      : IOSHostCommsInitialize() should be called once if 
  //                    successful to create an IOSProtocol.
  //
  /////////////////////////////////////////////////////////////////////////////
  BOOL IOSHostCommsInitialize()
  {
    if (pIOSProtocol == NULL)
      {
        pIOSProtocol = new CIOSProtocol();
      }

#ifdef WIN32

    int      nCount;
    WSADATA  WSAData;

    // Initialize the windows socket interface.
    if((nCount = WSAStartup(MAKEWORD(2,2), &WSAData)) != 0)
      {
#ifdef PRINT_ERRORS
        Log_Report("IOSHostCommsInitialize : WSAStartup error %d.\n", nCount, Log_Error);
#endif
        
        return FALSE;
      }

#ifdef PRINT_STATISTICS
    Log_Report("IOSHostCommsInitialize : WINSOCK.DLL Version %d.%d.\n", 
               HIBYTE(WSAData.wVersion), 
               LOBYTE(WSAData.wVersion), Log_Informational);
#endif

    // Initialize the atexit handler.
    atexit(IOSHostCommsExitRoutine);

#else
#ifdef VXWORKS

    // There is no graceful recovery from this task getting terminated on the host.
    // If there was, the following calls might prove useful:
    //          taskHookInit();
    //          taskDeleteHookAdd(IOSHostCommsExitRoutine);

#endif
#endif

    return TRUE;
  }

  /////////////////////////////////////////////////////////////////////////////
  //
  // IOSHostComms
  //
  // Inputs           : long int lDelay - number of tries to skip before
  //                                      attempting to reconnect.
  //
  // Return Values    : None..
  //
  // Date             : 06 May 1999
  //
  // Engineer         : Billy Baker
  //
  // Description      : HostComms should be called at a schedule time.  It will
  //                    attempt to create the listen and accept sockets.  Once
  //                    these sockets are created, it will call 
  //                    CGenericHostSockets::ReceiveData, ReadValues, 
  //                    CGenericHostSockets::SendData, and then ProcessCommands.
  //                    Once the listen socket is created, the host byte ordering
  //                    is determined and the byte conversion function pointers
  //                    are set.
  /////////////////////////////////////////////////////////////////////////////
  void IOSHostComms()
  {
    
    if (!IOSHostCommsConnected()) {
      return;
    }

    // Fill the buffer to send to the client.
    if (pIOSProtocol->m_pSockets->IsValidAcceptSocket())
      {
        // Send data to the client.
        pIOSProtocol->m_pSockets->SendData();

      }

    return;
  }

  void Register(const char* pcName, void* pReadAddress,
                void* pChangeValue, unsigned char ucSize,
                BOOL playBack, float fLow, float fHigh, 
                long lLength1D, long lLength2D)
  {
    Variable* pVariable = NULL;
    short int shPossibleVariables = 0;

    // Make sure that we have a protocol.
    if (pIOSProtocol == NULL)
      {
        pIOSProtocol = new CIOSProtocol();

        if (pIOSProtocol == NULL)
          {
            return;
          }
      }

    // Get the current number of variables that have been added.
    shPossibleVariables = pIOSProtocol->PossibleVariables();

    // Make sure we haven't exceeded our array.
    if (shPossibleVariables == IOS_MAX_VARIABLES)
      {
        return;
      }

    // Insert the name in sorted order or make sure that it hasn't
    // already been added.
    int nVariableIndex = 0;
    if (shPossibleVariables > 0)
      {
        // Search for the variable name to see if it has already been added or
        // for the correct index to insert in sorted order.
        bool bLessThan = true;
        while (nVariableIndex < shPossibleVariables && bLessThan == true)
          {
            pVariable = &(pIOSProtocol->m_Variables[nVariableIndex]);
            if (strncmp(pVariable->s_cName,
                        pcName,
                        MAX_VARIABLE_NAME_LEN - 1) >= 0)
              {
                // Found the first string that is "greater than" the name
                // to register.
                bLessThan = false;
              }

            nVariableIndex++;
          }

        // Check to see if the name is already in the array.
        if (strncmp(pVariable->s_cName, 
                    pcName,
                    MAX_VARIABLE_NAME_LEN - 1) == 0)
          {
            // It is.  Just return.
            return;
          }

        // Since the for loop incremented past the actual insertion
        // index, decrement the insertion index.
        if (bLessThan == false)
          {
            nVariableIndex--;

            // Move everyone down.
            int i = shPossibleVariables - 1;
            while (i >= nVariableIndex)
              {
                pIOSProtocol->m_Variables[i+1] = pIOSProtocol->m_Variables[i];
                i--;
              }
          }
      }

    // Get the address of the next Variable slot to which the passed 
    // information will be added.
    pVariable = &(pIOSProtocol->m_Variables[nVariableIndex]);

    // Fill in the data.
    strncpy(pVariable->s_cName, pcName, MAX_VARIABLE_NAME_LEN - 1);
    pVariable->s_cName[strlen(pVariable->s_cName)] = '\0';

    pVariable->s_lAddress           = (long)pReadAddress;
    pVariable->s_pChangeValue       = pChangeValue;
    pVariable->s_bPlayBack          = playBack;
    pVariable->s_ushSize            = ucSize;
    pVariable->s_ushReferenceCount  = 0;
    pVariable->s_fLow               = fLow;
    pVariable->s_fHigh              = fHigh;
    pVariable->s_lLength1D          = lLength1D;
    pVariable->s_lLength2D          = lLength2D;

    // Update the number of variables in the array.
    shPossibleVariables++;
    pIOSProtocol->PossibleVariables(shPossibleVariables);
  }

  void RegisterIOSVariableEx(const char* pcName, void* pReadAddress,
                             void* pChangeValue, unsigned char ucSize,
                             BOOL playBack, float fLow, float fHigh, 
                             long lLength1D, long lLength2D)
  {
    Register(pcName, pReadAddress, pChangeValue, ucSize, playBack, fLow, fHigh, 
             lLength1D, lLength2D);
  }

  void RegisterIOSVariable(const char* pcName, void* pReadAddress,
                           void* pChangeValue, unsigned char ucSize,
                           BOOL playBack, float fLow, float fHigh)
  {
    Register(pcName, pReadAddress, pChangeValue, ucSize, playBack, fLow, fHigh, 1, 1);
  }

  char* IOSHostCommsReceiveBuffer ()
  {
    if (pIOSProtocol == NULL)
      {
	return NULL;
      }
    
    if (pIOSProtocol->m_pSockets == NULL)
      {
	return NULL;
      }
    return (pIOSProtocol->m_pSockets->m_cReceiveBuffer);
  }

  short IOSHostCommsMaxReceiveBuffer ()
  {
    return IOS_MAX_RECEIVE_BUFFER;
  }


#ifdef __cplusplus
}
#endif // __cplusplus
