/////////////////////////////////////////////////////////////////////////////
//
//           F L I G H T S A F E T Y   I N T E R N A T I O N A L
//                     Simulation Systems Division
//                      2700 North Hemlock Circle
//                     Broken Arrow, Oklahoma 74012
//                          (918) 259-4000
/////////////////////////////////////////////////////////////////////////////
//
// DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
// Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
// technical or operational data or information from automatic dissemination
// under the International Exchange Program or by other means.  This protection
// covers information required solely for administrative or operational
// purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
//
// WARNING:  This document contains technical data whose export is restricted
// by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
// Executive Order 12470.  Violation of these export control laws is subject
// to severe criminal penalties.  Dissemination of this document is controlled
// under DOD Directive 5230.25
//
/////////////////////////////////////////////////////////////////////////////
//
//
// Filename         : GenericHostSockets.cpp
//
// Date             : 06 May 1999
//
// Engineer         : Billy Baker
//
// Revision         : $Revision: 1.4 $
//
// Description      : GenericHostSockets.cpp contains the implementation of the 
//                    CGenericHostSockets class. CGenericHostSockets is meant to be 
//                    a generic class for creating a TCP strem socket.  Methods
//                    are also provided for reading to and writing data from 
//                    the stream socket.
//          
//                    To compile for VXWorks, the VXWORKS symbols must be 
//                    defined.
//
// Classification   : UNCLASSIFIED
//
// Requirements     : None.
//
// Components Used  : CByteConversion.
//
// Operational 
//    Restrictions  : Machine dependencies/restrictions
//                        None.
//                    Design dependencies/restrictions
//                        None.
//                    Operations containing dependencies/restrictions
//                        None.
//                    Compiler dependencies/restrictions
//                        None.
//                    Other conditions for proper execution
//                        None.
//
// Environment      : Operating system(s) - Microsoft Windows NT 4.0 with
//                                              NT service pack 4
//                                          Microsoft Windows NT 2000
//                                          VXWorks
//
//                    Compiler(s) - Visual C++ 6.0
//                                  Tornado 1.0.1
//
//                    Architechure(s) - Pentium II
/////////////////////////////////////////////////////////////////////////////

/////////////////////////////////////////////////////////////////////////////
//
//                              R e v i s i o n   H i s t o r y
//
/////////////////////////////////////////////////////////////////////////////
// $Log: GenericHostSockets.cpp $
// Revision 1.4  2000/02/28 18:37:25  billyb
// Changed printfs to fprintfs.  Changed listen IP address to INADDR_ANY
// as opposed to using gethostbyname.  Changed listen backlog to 0.  
// Added SO_LINGER to reset the socket on close.
// Revision 1.3  2000/01/26 09:20:39  billyb
// Corrected comments.  Changed reverse methods to avoid
// possibly using a byte by byte copy with memcpy.
// Revision 1.2  1999/11/17 06:42:19  billyb
// Changed cache variable name to reflect its change from long
// to short.
/////////////////////////////////////////////////////////////////////////////
#include "HostComms.h"
#include "GenericHostSockets.h"
#include "log_report.h"

/////////////////////////////////////////////////////////////////////////////
//
// CGenericHostSockets::CGenericHostSockets
//
// Inputs           : None.
//
// Return Values    : None.
//
// Date             : 10 May 1999
//
// Engineer         : Billy Baker
//
// Description      : Default constructor.
/////////////////////////////////////////////////////////////////////////////
CGenericHostSockets::CGenericHostSockets(unsigned short ushPort,
                                 unsigned long int ulMaxReceiveBuffer,
                                 unsigned long int ulReceiveCache,
                                 unsigned long int ulMaxSendBuffer,
                                 unsigned long int ulSendCache,
                                 BYTE_ORDERING eNetworkByteOrdering)
{
    m_ushPort = ushPort;
    m_socketAccept = INVALID_SOCKET;
    m_socketListen = INVALID_SOCKET;
    m_ulMaxReceiveBuffer = ulMaxReceiveBuffer;
    m_ulReceiveCache = ulReceiveCache;
    m_ulMaxSendBuffer = ulMaxSendBuffer;
    m_ushSendCache = (unsigned short)ulSendCache;

    m_cReceiveBuffer = NULL;
    m_cSendBuffer = NULL;

    m_cReceiveBuffer = new char[ulMaxReceiveBuffer];
    m_cSendBuffer = new char[ulMaxSendBuffer];

    m_eNetworkByteOrdering = eNetworkByteOrdering;

    // Determine the byte ordering
    union 
    {
        short s;
        char c[sizeof(short)];
    } un;

    un.s = 0x0102;
    if (sizeof(short) == 2)
    {
        if (un.c[0] == 1 && un.c[1] == 2)
        {
            if (m_eNetworkByteOrdering == LITTLE)
            {
                // Host is big endian and network order desired is big endian. 
                Ntohs = ReverseS;
                Ntohl = ReverseL;
                Ntohd = ReverseD;
                Htons = ReverseS;
                Htonl = ReverseL;
                Htond = ReverseD;

                #ifdef PRINT_STATISTICS

                    Log_Report("GenericHostSockets : Host is big-endian and Network is "
                           "little-endian.  Conversions performed.\n", Log_Informational);

                #endif
            }
            else
            {
                // Host is big endian and network order desired is big endian. 
                Ntohs = NoChangeS;
                Ntohl = NoChangeL;
                Ntohd = NoChangeD;
                Htons = NoChangeS;
                Htonl = NoChangeL;
                Htond = NoChangeD;

                #ifdef PRINT_STATISTICS
                    Log_Report("GenericHostSockets : Host is big-endian and Network is "
                           "big-endian.  No conversions performed.\n", Log_Informational);
                #endif
            }
        }
        else
        {
            if (m_eNetworkByteOrdering == LITTLE)
            {
                // Host is little endian and network order desired is little 
                // endian.
                Ntohs = NoChangeS;
                Ntohl = NoChangeL;
                Ntohd = NoChangeD;
                Htons = NoChangeS;
                Htonl = NoChangeL;
                Htond = NoChangeD;

                #ifdef PRINT_STATISTICS
                    Log_Report("GenericHostSockets : Host is little-endian and Network "
                           "is little-endian.  No conversions performed.\n", Log_Informational);
                #endif
            }
            else
            {
                // Host is little endian and network order desired is big 
                // endian.
                Ntohs = ReverseS;
                Ntohl = ReverseL;
                Ntohd = ReverseD;
                Htons = ReverseS;
                Htonl = ReverseL;
                Htond = ReverseD;

                #ifdef PRINT_STATISTICS
                    Log_Report("GenericHostSockets : Host is little-endian and Network "
                           "is big-endian.  Conversions performed.\n", Log_Informational);
                #endif
            }
        }
    }
}

/////////////////////////////////////////////////////////////////////////////
//
// CGenericHostSockets::~CGenericHostSockets
//
// Inputs           : None.
//
// Return Values    : None.
//
// Date             : 10 May 1999
//
// Engineer         : Billy Baker
//
// Description      : Default destructor.
/////////////////////////////////////////////////////////////////////////////
CGenericHostSockets::~CGenericHostSockets()
{
    CGenericHostSockets::CloseAcceptSocket();	
    CGenericHostSockets::CloseListenSocket();

    if (m_cReceiveBuffer != NULL)
    {
        delete [] m_cReceiveBuffer;
    }

    if (m_cSendBuffer != NULL)
    {
        delete [] m_cSendBuffer;
    }
}

 /////////////////////////////////////////////////////////////////////////////
//
// CGenericHostSockets::CloseAcceptSocket
//
// Inputs           : None.
//
// Return Values    : success value of socket closure.
//
// Date             : 06 May 1999
//
// Engineer         : Billy Baker
//
// Description      : CloseAcceptSocket is called when the task/process is
//                    closed or if an error occurs while reading or writing
//                    the socket.  Also, if errors occur while the Accept
//                    is being created, then CloseAcceptSocket will be called.
/////////////////////////////////////////////////////////////////////////////
BOOL CGenericHostSockets::CloseAcceptSocket()
{
    BOOL    bReturn = TRUE;

    if(m_socketAccept != INVALID_SOCKET)
    {
        // Shutdown the socket's sending and receiving.
        shutdown(m_socketAccept, SD_BOTH);

        // Attempt to close the socket.
        if(closesocket(m_socketAccept))
        {
            #ifdef PRINT_ERRORS
                Log_Report("GenericHostSockets : Close socket error %d.\n", 
                       GetLastError(), Log_Error);
            #endif

            bReturn = FALSE;
        }

        // Socket is now invalid.
        m_socketAccept = INVALID_SOCKET;
    }

    return bReturn;
}

/////////////////////////////////////////////////////////////////////////////
//
// CGenericHostSockets::CloseListenSocket
//
// Inputs           : None.
//
// Return Values    : success value of socket closure.
//
// Date             : 06 May 1999
//
// Engineer         : Billy Baker
//
// Description      : CloseListenSocket is called when the task/process is
//                    closed or if an error occurs while the Listen socket
//                    is being created.
/////////////////////////////////////////////////////////////////////////////
BOOL CGenericHostSockets::CloseListenSocket()
{
    BOOL    bReturn = TRUE;

    if(m_socketListen != INVALID_SOCKET)
    {
        // Close down the sending and receiving.
        shutdown(m_socketListen, SD_BOTH);

        // Attempt to close the socket.
        if(closesocket(m_socketListen))
        {
            #ifdef PRINT_ERRORS
                Log_Report("GenericHostSockets : Close socket error %d.\n", 
                       GetLastError(), Log_Error);
            #endif

            bReturn = FALSE;
        }

        // Socket is now invalid.
        m_socketListen = INVALID_SOCKET;
    }

    return bReturn;
}

/////////////////////////////////////////////////////////////////////////////
//
// CGenericHostSockets::CreateAcceptSocket
//
// Inputs           : None.
//
// Return Values    : success value of socket creation.
//
// Date             : 06 May 1999
//
// Engineer         : Billy Baker
//
// Description      : CreateAcceptSocket is called when the task/process is
//                    started after the creation of the Listen socket.  If the
//                    Accept socket could not be created during the last call
//                    or if an error occured during reading or writing, then
//                    an attempt will be made again to create the accept
//                    socket.  The Accept socket will only be created if a 
//                    client is attempting to connect.
/////////////////////////////////////////////////////////////////////////////
BOOL CGenericHostSockets::CreateAcceptSocket()
{
	// Attempt a new connection.
    if (!IsValidListenSocket())
    {
        return FALSE;
    }

	m_socketAccept = accept(m_socketListen, NULL, NULL);

	if(m_socketAccept == INVALID_SOCKET)
	{
		 // Connection failed, see why.  Note if the error code is
		 // WSAEWOULDBLOCK, the socket is informing the application
		 // that it would have had to wait to create the accept socket
		 // because there was no request to make a connection.  This
		 // is not an error, it is the sockets method of informing the
		 // application there is no connection request.
        int		nError = GetLastError();

		if(nError != WSAEWOULDBLOCK)
		{
			// Failed because an error occured.
			#ifdef PRINT_ERRORS
				Log_Report("GenericHostSockets : Socket accept error %d.\n", 
                       nError, Log_Error);
            #endif
		}

		// Return no create.
		return FALSE;
	}

	// Return create.
	return TRUE;
}

/////////////////////////////////////////////////////////////////////////////
//
// CGenericHostSockets::CreateListenSocket
//
// Inputs           : None.
//
// Return Values    : success value of socket creation.
//
// Date             : 06 May 1999
//
// Engineer         : Billy Baker
//
// Description      : CreateListenSocket is called when the task/process is
//                    started.  If the Listen socket could not be created 
//                    during the last call, then an attempt will be made again 
//                    to create the Listen socket.  Once the socket is created,
//                    the buffer sizes are set, non-blocking mode is set, and
//                    the Nagle algorithm is disabled.  Also, binding occurs 
//                    here.
/////////////////////////////////////////////////////////////////////////////
BOOL CGenericHostSockets::CreateListenSocket()
{
    // Local variables.
    SOCKADDR_IN soiAddressServer;
    int         dwNoBlocking = TRUE;
    long        size = m_ushSendCache * m_ulMaxSendBuffer;

    // Retrieve the IP address and TCP port number for the machine the 
    // application is on.
    #ifdef PRINT_STATISTICS
        char        chBuffer[256];
        int         nCount;

        nCount = sizeof(chBuffer);
        gethostname(chBuffer, nCount);

        Log_Report("GenericHostSockets : Host name %s.\n", chBuffer, Log_Informational);
    #endif

/*  If the INADDR_ANY is not used below, this commented code with
    chBuffer from the PRINT_STATISTICS ifdef can be used to attach
    to a specific IP and port.


    PHOSTENT    pHostEnt;
    pHostEnt = gethostbyname(chBuffer);

#ifdef WIN32
    if(pHostEnt == NULL)
    {
        #ifdef PRINT_ERRORS
            Log_Report("GenericHostSockets : Host name error %d.\n", GetLastError(), Log_Error);
        #endif
		
        CloseListenSocket();
        return FALSE;
    }
#else
  #ifdef VXWORKS
    if(pHostEnt == -1)
    {
        #ifdef PRINT_ERRORS
            Log_Report("GenericHostSockets : Host name error %d.\n", GetLastError(), Log_Error);
        #endif
		
        CloseListenSocket();
        return FALSE;
    }
  #endif
#endif

#ifdef WIN32
    memcpy((char *)& (soiAddressServer.sin_addr), pHostEnt->h_addr, 
           pHostEnt->h_length); 
#else
  #ifdef VXWORKS
    // pHostEnt has the network address as an int in host byte ordering.
    memcpy(&soiAddressServer.sin_addr, &pHostEnt, sizeof(int));
  #endif
#endif
*/
	soiAddressServer.sin_port           = htons(m_ushPort);
	soiAddressServer.sin_family         = PF_INET;
	soiAddressServer.sin_addr.s_addr    = htonl(INADDR_ANY);

#ifdef WIN32
    memset(soiAddressServer.sin_zero, 0, 8);
#else
  #ifdef VXWORKS
    bzero( soiAddressServer.sin_zero, 8 );
  #endif
#endif


	// Create the listening socket.
	m_socketListen = socket(PF_INET, SOCK_STREAM, 0);

	if(m_socketListen == INVALID_SOCKET)
	{
		// Create failed.
		#ifdef PRINT_ERRORS
			Log_Report("GenericHostSockets : Socket creation failed.\n", Log_Error);
        #endif

		return FALSE;
	}
/*
	// Set the listen socket to non-blocking.
#ifdef WIN32
	if(ioctlsocket(m_socketListen, FIONBIO, (unsigned long*)&dwNoBlocking))
#else 
    #ifdef VXWORKS
	  // Blocking on vxWorks
	  //	  if (ioctlsocket(m_socketListen, FIONBIO, &dwNoBlocking))
	  if (0) 
    #else
	  #error "VXWORKS not defined"
    #endif 
#endif
	{
		// Non-blocking failed.
		#ifdef PRINT_ERRORS
			Log_Report("GenericHostSockets : Non-blocking error %d.\n", 
                   GetLastError(), Log_Error);
        #endif

		CloseListenSocket();
		return FALSE;
	}
*/
    if (setsockopt(m_socketListen, SOL_SOCKET, SO_REUSEADDR, 
                    (char *)&dwNoBlocking, sizeof(int)))
    {
		#ifdef PRINT_ERRORS
			Log_Report("GenericHostSockets : SO_REUSEADDR error %d.\n", GetLastError(), Log_Error);
        #endif  

		CloseListenSocket();
		return FALSE;
    }

	// Set the listen socket to not use the Nagle algorithm.
    // This was done since typically the client and Host are directly
    // connected.
	if(setsockopt(m_socketListen, IPPROTO_TCP, TCP_NODELAY, 
                  (char *)&dwNoBlocking, sizeof(int)))
	{
		#ifdef PRINT_ERRORS
			Log_Report("GenericHostSockets : TCP_NODELAY error %d.\n", GetLastError(), Log_Error);
        #endif  

		CloseListenSocket();
		return FALSE;
	}

    linger lng;
    lng.l_onoff     = 1;
    lng.l_linger    = 0;
    if (setsockopt(m_socketListen, SOL_SOCKET, SO_LINGER, 
                   (char *)&lng, sizeof(lng)))
    {
		// Send buffer could not be changed
		#ifdef PRINT_ERRORS
			Log_Report("GenericHostSockets : Linger error %d.\n", 
                   GetLastError(), Log_Error);
        #endif

		CloseListenSocket();
		return FALSE;
    }

    if (setsockopt(m_socketListen, SOL_SOCKET, SO_SNDBUF, 
                   (char *)&size, sizeof(size)))
    {
		// Send buffer could not be changed
		#ifdef PRINT_ERRORS
			Log_Report("GenericHostSockets : Send buffer change error %d.\n", 
                   GetLastError(), Log_Error);
        #endif

		CloseListenSocket();
		return FALSE;
    }

    size = m_ulReceiveCache * m_ulMaxReceiveBuffer;
    if (setsockopt(m_socketListen, SOL_SOCKET, SO_RCVBUF, 
                   (char *)&size, sizeof(size)))
    {
		// Send buffer could not be changed
		#ifdef PRINT_ERRORS
			Log_Report("GenericHostSockets : Receive buffer change error %d.\n", 
                   GetLastError(), Log_Error);
        #endif

		CloseListenSocket();
		return FALSE;
    }

	// Bind the listening socket to the machine IP address and TCP port number.
	if(bind(m_socketListen, 
            (struct sockaddr *)& soiAddressServer, 
            sizeof(soiAddressServer)) == SOCKET_ERROR)
	{
		// Bind failed.
		#ifdef PRINT_ERRORS
			Log_Report("GenericHostSockets : Binding error %d.\n", GetLastError(), Log_Error);
        #endif

		CloseListenSocket();
		return FALSE;
	}

	// Start the listen socket.
	if(listen(m_socketListen, 0) == INVALID_SOCKET)
	{
		// Listen failed.
		#ifdef PRINT_ERRORS
			Log_Report("GenericHostSockets : Listen error %d.\n", GetLastError(), Log_Error);
        #endif

		CloseListenSocket();
		return FALSE;
	}

	// Socket created.
	return TRUE;
}

/////////////////////////////////////////////////////////////////////////////
//
// CGenericHostSockets::IsValidAcceptSocket
//
// Inputs           : None.
//
// Return Values    : whether the current Accept socket is valid.
//
// Date             : 06 May 1999
//
// Engineer         : Billy Baker
//
// Description      : IsValidAcceptSocket determines whether the current Accept
//                    socket is valid.
/////////////////////////////////////////////////////////////////////////////
BOOL CGenericHostSockets::IsValidAcceptSocket()
{
	return (m_socketAccept != INVALID_SOCKET);
}

/////////////////////////////////////////////////////////////////////////////
//
// CGenericHostSockets::IsValidListenSocket
//
// Inputs           : None.
//
// Return Values    : whether the current listen socket is valid.
//
// Date             : 06 May 1999
//
// Engineer         : Billy Baker
//
// Description      : IsValidListenSocket determines whether the current listen
//                    socket is valid.
/////////////////////////////////////////////////////////////////////////////
BOOL CGenericHostSockets::IsValidListenSocket()
{
	return (m_socketListen != INVALID_SOCKET);
}

/////////////////////////////////////////////////////////////////////////////
//
// CGenericHostSockets::ReceiveData
//
// Inputs           : None.
//
// Return Values    : None.
//
// Date             : 06 May 1999
//
// Engineer         : Billy Baker
//
// Description      : ReceiveData will loop until it has read 
//                    MAX_RECEIVE_BUFFER from the accept socket.  If an error
//                    occurs, then the accept socket is closed.  If no data
//                    is available, then the method simply returns.  If no
//                    data is read, the accept socket is closed.  If more bytes
//                    than requested are received, the accept socket is closed.
//                    ReceiveData should be called after the listen and accept
//                    sockets are created.  It should be called each pass
//                    before SendData is called.  After SendData, any received
//                    data should be processed.
/////////////////////////////////////////////////////////////////////////////
void CGenericHostSockets::ReceiveData()
{
    int nStartReceivePos    = 0;
    int nBytesToReceive     = m_ulMaxReceiveBuffer;
    int nBytesReceived      = 0;

    memset(m_cReceiveBuffer, 0, m_ulMaxReceiveBuffer);

    // *****ADDED BY T.E.D.
    // Verify that we won't have to block if we call recv
#ifdef WIN32
    if (ioctlsocket(m_socketAccept, FIONREAD, (unsigned long *)&nBytesReceived)) 
#else 
    if (ioctlsocket(m_socketAccept, FIONREAD, (int)&nBytesReceived))
#endif
    {
#ifdef PRINT_ERRORS

      int nError = GetLastError();

      Log_Report ("GenericHostSockets : ioctlsocket FIONREAD error %d", nError, Log_Error);
#endif
      return;
    }
    if (nBytesReceived < nBytesToReceive) {return;}
    // *****END ADDED BY T.E.D.

    while(nBytesToReceive > 0)
	{
		// Receive data.
		if((nBytesReceived = recv(m_socketAccept, 
                                  &m_cReceiveBuffer[nStartReceivePos], 
                                  nBytesToReceive, 0)) == SOCKET_ERROR)
		{
			// Receive failed, see why.
			int	nError = GetLastError();

			if(nError != WSAEWOULDBLOCK)
			{
				// Error during receive.
				switch(nError)
				{
					case WSAESHUTDOWN:
					{
						#ifdef PRINT_ERRORS
							Log_Report("GenericHostSockets : client disconnected.\n", Log_Warning);
                        #endif
					}
					break;

					case WSAECONNRESET:
					{
						#ifdef PRINT_ERRORS
							Log_Report("GenericHostSockets : client disconnected.\n", Log_Warning);
                        #endif
					}
					break;

					default:
					{
						#ifdef PRINT_ERRORS
							Log_Report("GenericHostSockets : Receive error %d.\n", 
                                   nError, Log_Error);
                        #endif
					}
					break;
				}

				// Close the socket.
                CloseAcceptSocket();
			}

			 // Either an error occured or the socket would block, break out
			 // of the receive loop.
			break;
		}
		else if(nBytesReceived == 0)
		{
			// client disconnected, close the socket.
			CloseAcceptSocket();

			#ifdef PRINT_ERRORS
				Log_Report("GenericHostSockets : client disconnected.\n", Log_Warning);
            #endif

			// Break out of the receive loop.
			break;
		}
		else
		{
			// Update the receive parameters.
			if(nBytesReceived > nBytesToReceive)
			{
				// Error, for some reason the socket received more than it 
                // was asked for.
                CloseAcceptSocket();

				#ifdef PRINT_ERRORS
                    Log_Report("GenericHostSockets : Received %d bytes from socket, "
                           "only asked for %d.\n", nBytesReceived, nBytesToReceive, Log_Warning);
                #endif

				// Break out of the receive loop.
				break;
			}
			else
			{
				// Received no more bytes from socket than was asked for.
                #ifdef PRINT_STATISTICS
                    Log_Report("GenericHostSockets : bytes received = %d\n", 
                           nBytesReceived, Log_Warning);
                #endif

                nStartReceivePos += nBytesReceived;
                nBytesToReceive  -= nBytesReceived;
			}
		}
	}

	// End.
	return;
}

/////////////////////////////////////////////////////////////////////////////
//
// CGenericHostSockets::SendData()
//
// Inputs           : None.
//
// Return Values    : whether the current listen socket is valid.
//
// Date             : 06 May 1999
//
// Engineer         : Billy Baker
//
// Description      : SendData will loop until it has read MAX_SEND_BUFFER from
//                    the accept socket.  If an error occurs, then the accept 
//                    socket is closed.  MAX_SEND_BUFFER bytes will try to be
//                    sent on each pass even if no data is in the send buffer.
//                    If an error other than blocking occurs, the accept socket
//                    will be closed.  If blocking were to occur, no data is 
//                    sent.  If the number of bytes sent is larger than
//                    MAX_SEND_BUFFER, the accept socket is closed.
//
//                    SendData should be called after the listen and accept
//                    sockets are created.  It should be called each pass
//                    after ReceiveData is called.  It should also be called
//                    to send an error packet back to the client.
/////////////////////////////////////////////////////////////////////////////
void CGenericHostSockets::SendData()
{
    int nStartSendPos             = 0;
    int	nBytesToSend              = m_ulMaxSendBuffer;
    int	nBytesSent                = 0;
#ifdef PRINT_ERRORS
    static int count              = 1;
#endif

    // Start to send until everything has been sent.
    while(nBytesToSend > 0)
	{
		if((nBytesSent = send(m_socketAccept, 
                              &m_cSendBuffer[nStartSendPos], 
                              nBytesToSend, 0)) == SOCKET_ERROR)
		{
			// Send failed, see why.
			int		nError;

			if((nError = GetLastError()) != WSAEWOULDBLOCK)
			{
				// Error, close the socket.
				CloseAcceptSocket();
			}

			#ifdef PRINT_ERRORS
                                Log_Report("GenericHostSockets : Send error %d %d.", nError, count, Log_Error);
                                count++;
                        #endif

			// Either an error occured or the socket would block, break out
			// of the send loop.
			break;
		}
		else
		{
			// No error or blocking, update the send parameters.
			if(nBytesSent > nBytesToSend)
			{
				// Error, for some reason the socket sent more than it was 
                // asked to. 
				CloseAcceptSocket();

				#ifdef PRINT_ERRORS
					Log_Report("GenericHostSockets : Socket sent %d bytes, "
                           "only asked to send %d.\n", 
                           nBytesSent, nBytesToSend, Log_Warning);
                #endif

				// Break out of the send loop.
				break;
			}
			else
			{
				// Send was successful, update the send parameters.
    			nStartSendPos += nBytesSent;
                nBytesToSend  -= nBytesSent;
			}
		}
	}

	return;
}

/////////////////////////////////////////////////////////////////////////////
//
// CGenericHostSockets::ReverseS
//
// Inputs           : short s.
//
// Return Values    : a byte reverse version of s.
//
// Date             : 06 May 1999
//
// Engineer         : Billy Baker
//
// Description      : ReverseD reverses the byte ordering for the short 
//                    value passed.
/////////////////////////////////////////////////////////////////////////////
short CGenericHostSockets::ReverseS(short s)
{
    char cN[2] = {0,0};
    char* cH = (char *)&s;

    cN[0] = cH[1];
    cN[1] = cH[0];

    return *(short *)cN;
}

/////////////////////////////////////////////////////////////////////////////
//
// CGenericHostSockets::ReverseL
//
// Inputs           : long l.
//
// Return Values    : a byte reverse version of l.
//
// Date             : 06 May 1999
//
// Engineer         : Billy Baker
//
// Description      : ReverseL reverses the byte ordering for the 32-bit long 
//                    value passed.
/////////////////////////////////////////////////////////////////////////////
long CGenericHostSockets::ReverseL(long l)
{
    char cN[4] = {0,0,0,0};
    char *cH = (char *)&l;

    cN[0] = cH[3];
    cN[1] = cH[2];
    cN[2] = cH[1];
    cN[3] = cH[0];

    return *(long *)cN;
}

/////////////////////////////////////////////////////////////////////////////
//
// CGenericHostSockets::ReverseD
//
// Inputs           : double d.
//
// Return Values    : a byte reverse version of d.
//
// Date             : 06 May 1999
//
// Engineer         : Billy Baker
//
// Description      : ReverseD reverses the byte ordering for the double 
//                    value passed.
/////////////////////////////////////////////////////////////////////////////
double CGenericHostSockets::ReverseD(double d)
{
    char cN[8] = {0,0,0,0,0,0,0,0};
    char *cH = (char *)&d;

    cN[0] = cH[7];
    cN[1] = cH[6];
    cN[2] = cH[5];
    cN[3] = cH[4];
    cN[4] = cH[3];
    cN[5] = cH[2];
    cN[6] = cH[1];
    cN[7] = cH[0];

    return *(double *)cN;
}

/////////////////////////////////////////////////////////////////////////////
//
// CGenericHostSockets::NoChangeS
//
// Inputs           : short s.
//
// Return Values    : the value passed in.
//
// Date             : 06 May 1999
//
// Engineer         : Billy Baker
//
// Description      : NoChangeS returns the passed value. No conversion 
//                    needed because the data packet is assumed to be in the
//                    same byte ordering as the host.
/////////////////////////////////////////////////////////////////////////////
short CGenericHostSockets::NoChangeS(short s)
{
    return s;
}

/////////////////////////////////////////////////////////////////////////////
//
// CGenericHostSockets::NoChangeL
//
// Inputs           : long l.
//
// Return Values    : the value passed in.
//
// Date             : 06 May 1999
//
// Engineer         : Billy Baker
//
// Description      : NoChangeL returns the passed value. No conversion 
//                    needed because the data packet is assumed to be in the
//                    same byte ordering as the host.
/////////////////////////////////////////////////////////////////////////////
long CGenericHostSockets::NoChangeL(long l)
{
    return l;
}

/////////////////////////////////////////////////////////////////////////////
//
// CGenericHostSockets::NoChangeD
//
// Inputs           : double d.
//
// Return Values    : the value passed in.
//
// Date             : 06 May 1999
//
// Engineer         : Billy Baker
//
// Description      : NoChangeD returns the passed value. No conversion 
//                    needed because the data packet is assumed to be in the
//                    same byte ordering as the host.
/////////////////////////////////////////////////////////////////////////////
double CGenericHostSockets::NoChangeD(double d)
{
    return d;
}

