/////////////////////////////////////////////////////////////////////////////
//
//           F L I G H T S A F E T Y   I N T E R N A T I O N A L
//                     Simulation Systems Division
//                      2700 North Hemlock Circle
//                     Broken Arrow, Oklahoma 74012
//                          (918) 259-4000
/////////////////////////////////////////////////////////////////////////////
//
// DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
// Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
// technical or operational data or information from automatic dissemination
// under the International Exchange Program or by other means.  This protection
// covers information required solely for administrative or operational
// purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
//
// WARNING:  This document contains technical data whose export is restricted
// by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
// Executive Order 12470.  Violation of these export control laws is subject
// to severe criminal penalties.  Dissemination of this document is controlled
// under DOD Directive 5230.25
//
/////////////////////////////////////////////////////////////////////////////
//
//
// Filename         : TimePlotWidget.cpp
//
// Date             : 01 February 2000
//
// Engineer         : Billy Baker
//
// Revision         : $Revision: 1.4 $
//
// Description      : TimePlotWidget.cpp contains the implementation of the 
//                    CTimePlotWidget class.  This class draws a plot based
//                    on the values of a variable (for y) and time.  A maximum
//                    of six plots may be drawn.  Each plot may have a 
//                    different color, and differt y limits.  The grids can be 
//                    toggled on and off.  The number of grid divisions can be 
//                    set between 2 and 10.  The current x,y value is shown as 
//                    is the name of the host and rate of data from the host.  
//                    Inspection of values is possible.  Three plotting modes
//                    exist--stop, wrap, erase.  Stop will stop when the right
//                    side maximum is reached.  But, if the time on the x-axis
//                    grows before the plot has stopped, then the plot will
//                    continue.  Erase will erase the plotting region when the 
//                    right hand side of the region is reached.  Wrap mode will
//                    continue to plot values until the maximum size is reached.
//                    The maximum size is based on the rate of data from the 
//                    host and the constant nMINUTES_OF_DATA.
//
// Classification   : UNCLASSIFIED
//
// Requirements     : None.
//
// Components Used  : General::CPlotWidget, General::CPlotData, 
//                    Core::CXMLWidget, Comms::CCommsAction, 
//                    Core::CVariant, _FSI_STL::string, CString, 
//                    Core::CWidget, Core::CChangeValue, and
//                    General::COpenGLWidget.
//
// Operational 
//    Restrictions  : Machine dependencies/restrictions
//                        None.
//                    Design dependencies/restrictions
//                        None.
//                    Operations containing dependencies/restrictions
//                        None.
//                    Compiler dependencies/restrictions
//                        None.
//                    Other conditions for proper execution
//                        None.
//
// Environment      : Operating system(s) - Microsoft Windows NT 4.0 with
//                                              NT service pack 3, 4, or 5
//                                          Microsoft Windows NT 2000
//
//                    Compiler(s) - Visual C++ 6.0
//
//                    Architechure(s) - Intel Pentium, Pentium II
/////////////////////////////////////////////////////////////////////////////

/////////////////////////////////////////////////////////////////////////////
//
//                              R e v i s i o n   H i s t o r y
//
/////////////////////////////////////////////////////////////////////////////
// $Log: TimePlotWidget.cpp $                                                                   
// Revision 1.4  2000/03/07 20:38:32  billyb                                                                   
// Changed calls to AddElementValue.  Fixed crash in wrap mode.                                                                   
// Changed function signatures to eliminate warnings.                                                                   
// Revision 1.3  2000/02/07 07:49:28  billyb                                                                   
// Changed code so that MSVC would not think that general.dll                                                                   
// depends on msvcp60.dll.                                                                   
// Revision 1.2  2000/02/01 19:58:59  billyb                                                                   
// New file with comments, independent grid toggling, drawing of                                                                   
// inspection values, the host name, the data rate, and current                                                                   
// y values, and support for 10 minutes of data.                                                                   
/////////////////////////////////////////////////////////////////////////////
#include "..\core\stdafx.h"
#include <gl\gl.h>
#include <gl\glu.h>

#include <map>
#include <string>

#include "OpenGLWidget.h"

#include "TimePlotWidget.h"

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif

/////////////////////////////////////////////////////////////////////////////
//
// CTimePlotWidget::CTimePlotWidget()
//
// Inputs           : None.
//
// Return Values    : None.
//
// Date             : 21 January 2000
//
// Engineer         : Billy Baker
//
// Description      : Default contructor.  Adds variables for additional 
//                    plots, a plot mode, plot min and maxes for the y-axis,
//                    y variable names, colors for each plot, and whether to
//                    show each plot.  
//
/////////////////////////////////////////////////////////////////////////////
CTimePlotWidget::CTimePlotWidget()
{
    m_stlStrWidgetName          = _FSI_STL::string("Time_Plot");

    m_listGraphicalElementVars.push_back("Plot Mode");

    m_listGraphicalElementVars.push_back("Plot Y Var 2 Name");
    m_listGraphicalElementVars.push_back("Plot Y Var 3 Name");
    m_listGraphicalElementVars.push_back("Plot Y Var 4 Name");
    m_listGraphicalElementVars.push_back("Plot Y Var 5 Name");
    m_listGraphicalElementVars.push_back("Plot Y Var 6 Name");

    m_mapCommsActions["Plot Y Var 2 Name"] = NULL;
    m_mapCommsActions["Plot Y Var 3 Name"] = NULL;
    m_mapCommsActions["Plot Y Var 4 Name"] = NULL;
    m_mapCommsActions["Plot Y Var 5 Name"] = NULL;
    m_mapCommsActions["Plot Y Var 6 Name"] = NULL;

    m_listGraphicalElementVars.push_back("Show Plot 1");
    m_listGraphicalElementVars.push_back("Show Plot 2");
    m_listGraphicalElementVars.push_back("Show Plot 3");
    m_listGraphicalElementVars.push_back("Show Plot 4");
    m_listGraphicalElementVars.push_back("Show Plot 5");
    m_listGraphicalElementVars.push_back("Show Plot 6");

    m_listGraphicalElementVars.push_back("Color Y Var 1");
    m_listGraphicalElementVars.push_back("Color Y Var 2");
    m_listGraphicalElementVars.push_back("Color Y Var 3");
    m_listGraphicalElementVars.push_back("Color Y Var 4");
    m_listGraphicalElementVars.push_back("Color Y Var 5");
    m_listGraphicalElementVars.push_back("Color Y Var 6");

    m_listGraphicalElementVars.push_back("Y Var 2 Min");
    m_listGraphicalElementVars.push_back("Y Var 2 Max");
    m_listGraphicalElementVars.push_back("Y Var 3 Min");
    m_listGraphicalElementVars.push_back("Y Var 3 Max");
    m_listGraphicalElementVars.push_back("Y Var 4 Min");
    m_listGraphicalElementVars.push_back("Y Var 4 Max");
    m_listGraphicalElementVars.push_back("Y Var 5 Min");
    m_listGraphicalElementVars.push_back("Y Var 5 Max");
    m_listGraphicalElementVars.push_back("Y Var 6 Min");
    m_listGraphicalElementVars.push_back("Y Var 6 Max");

    m_listGraphicalElementVars.push_back("Seconds Per Division");

    CTimePlotWidget::InitReinit();
}

void CTimePlotWidget::InitReinit()
{
    CPlotWidget::InitReinit();

    m_ucPlotMode                = WRAP_MODE;
    m_ucPlotMode_cv             = m_ucPlotMode;

    m_fSecondsPerDivision       = 3.0;
    m_fSecondsPerDivision_cv    = m_fSecondsPerDivision;

    m_fPlotMinX                 = 0.0f;
    m_fPlotMaxX                 = m_fPlotMinX + 
                                  m_fSecondsPerDivision * m_ulXDivisions;
    m_fPlotMaxX                *= m_fRate;

    m_fCurrentMaxDatapointTime  = 0.0f;
}

/////////////////////////////////////////////////////////////////////////////
//
// CTimePlotWidget::~CTimePlotWidget()
//
// Inputs           : None.
//
// Return Values    : None.
//
// Date             : 21 January 2000
//
// Engineer         : Billy Baker
//
// Description      : Default destructor.
//
/////////////////////////////////////////////////////////////////////////////
CTimePlotWidget::~CTimePlotWidget()
{
}

BEGIN_MESSAGE_MAP(CTimePlotWidget, CButtonWidget)
	//{{AFX_MSG_MAP(CTimePlotWidget)
  	ON_WM_WINDOWPOSCHANGED()
	ON_WM_MOUSEMOVE()
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
//
// CWidget* CTimePlotWidget::CreateObject()
//
// Inputs           : None.
//
// Return Values    : A pointer to a new instance of CTimePlotWidget.
//
// Date             : 21 January 2000
//
// Engineer         : Billy Baker
//
// Description      : CTimePlotWidget::CreateObject() is a common 
//                    framework method used to create a new instance of 
//                    CTimePlotWidget.  It is a static member.  The 
//                    address of this method is stored in a map of 
//                    widget name to CreateObject address in the 
//                    component interface for the DLL that contains the 
//                    CTimePlotWidget.  Using this data structure, a 
//                    new instance of CTimePlotWidget can be created 
//                    when one is encountered in the XML page 
//                    description file.
//
/////////////////////////////////////////////////////////////////////////////
CWidget* CTimePlotWidget::CreateObject()
{
    return new CTimePlotWidget();
}

/////////////////////////////////////////////////////////////////////////////
//
// void CTimePlotWidget::Initialize()
//
// Inputs           : CXMLWidget*& rpXMLWidget - reference to a pointer 
//                                               that holds the data to 
//                                               create a CTimePlotWidget.
//                    CWnd* pWnd               - the parent CWnd object.
//                    const long lId           - identifier.
//                    bool bEditMode           - whether editing is taking 
//                                               place.
//
// Return Values    : None.
//
// Date             : 21 January 2000
//
// Engineer         : Billy Baker
//
// Description      : CTimePlotWidget::Initialize() is a common 
//                    framework method.  It is called after an instance 
//                    of CTimePlotWidget has been created.  It is 
//                    responsible for initializing most of the internal 
//                    data members with data from the XMLWidget.  This 
//                    is done using the call to ResetProperties.  If 
//                    the XMLWidget pointer is NULL, a new XMLWidget 
//                    instance is created with default settings.  
//                    Generic help text is set here and any CWnd 
//                    derived Create() calls are made to create the 
//                    Windows object that allows for the use of the MFC 
//                    messages.  The HWND for the CWnd derived object 
//                    that is created is then added to a map in CWidget 
//                    that keeps track of what CWidget derived pointers 
//                    are valid.  This is to make sure that all of the 
//                    threads do not try to dereference bad pointers.
//
/////////////////////////////////////////////////////////////////////////////
void CTimePlotWidget::Initialize(CXMLWidget*& rpXMLWidget, CWnd* pWnd, 
                               const long lId, bool bEditMode)
{
    CWidget::Initialize(rpXMLWidget,pWnd, lId, bEditMode);

    CTimePlotWidget::InitReinit();

    if (rpXMLWidget != NULL)
    {
        ResetProperties();
    }

    if (GetSafeHwnd() == NULL)
    {
        CreateEx(WS_EX_TRANSPARENT, "BUTTON","",
            WS_CHILD | WS_VISIBLE  | CS_OWNDC | BS_OWNERDRAW | WS_CLIPCHILDREN | WS_CLIPSIBLINGS, 
                 CRect(m_exPtUpperLeft, m_exPtLowerRight),
                (CWnd*)pWnd, lId);
    }
    else
    {
        SetParent(pWnd);
        MoveWindow(m_exPtUpperLeft.X(), m_exPtUpperLeft.Y(),
                   m_exPtLowerRight.X() - m_exPtUpperLeft.X(),
                   m_exPtLowerRight.Y() - m_exPtUpperLeft.Y(), FALSE);
        ShowWindow(SW_NORMAL);
    }

    m_pWnd = this;
    m_mapValidAddresses[this] = m_pWnd->GetSafeHwnd();

    // Do OpenGL setup 
    Setup();

    // Create a comms action to get a default XMLWidget
    // instance that can be modified as the user wants
    // different varaibles.
    CCommsAction* pCommsAction = new CCommsAction();
    if (pCommsAction != NULL)
    {
        CXMLWidget* pXMLWidget = NULL;
        pCommsAction->Initialize(pXMLWidget, this, 1, false);
        if (pXMLWidget != NULL)
        {
            m_xmlCommsSampleWidget = *pXMLWidget;
        }

        delete pXMLWidget;
        pCommsAction->Deleting(true);

        // Set the read mode to always and the write mode to never.
        CXMLElement* pXMLElement = NULL;
        POSITION pos = NULL;
        if (m_xmlCommsSampleWidget.FindElement(pXMLElement, pos, _FSI_STL::string("VARIABLE")) == true)
        {
            pXMLElement->AddElementValue(_FSI_STL::string("None"));

            STRING2STRING_MAP::iterator s2sIt;
            CString strValue;
            if (pXMLElement->FindAttribute(s2sIt,_FSI_STL::string("READ_MODE")) == true)
            {
                CString strValue;
                strValue.Format("%d",READ_ALWAYS);
                (*s2sIt).second = _FSI_STL::string((LPCTSTR)strValue);
            }

            if (pXMLElement->FindAttribute(s2sIt,_FSI_STL::string("WRITE_MODE")) == true)
            {
                (*s2sIt).second = _FSI_STL::string("NO");
            }
        }
    }
}

/////////////////////////////////////////////////////////////////////////////
//
// void CXYPlotWidget::Render()
//
// Inputs           : None.
//
// Return Values    : None.
//
// Date             : 21 January 2000
//
// Engineer         : Billy Baker
//
// Description      : Common method among the OpenGL related classes.  Render()
//                    makes OpenGL calls to draw.  This method is invoked in
//                    COpenGLWidget::DrawItem().
//
/////////////////////////////////////////////////////////////////////////////
void CTimePlotWidget::Render()
{
    CPlotWidget::Render();

    DrawXLabels();

    _FSI_STL::map<_FSI_STL::string, CPlotData>::iterator mIt;
    _FSI_STL::map<_FSI_STL::string, CPlotData>::iterator mendIt;
    mIt     = m_mapVarName2PlotData.begin();
    mendIt  = m_mapVarName2PlotData.end();
    float fPlotCount = 0.0f;
    COpenGLtext oglTextLastValue;
    CString     strLastValue; 

    while (mIt != mendIt)
    {
        // Draw last value.
        glPushMatrix();
        {
            glTranslatef(fPlotCount / 8.0f * m_width, m_height + 13.0f, 0.0f);
            oglTextLastValue.SetForegroundColor((*mIt).second.m_color.Red(), 
                                                (*mIt).second.m_color.Green(), 
                                                (*mIt).second.m_color.Blue());
            if ((*mIt).second.m_pfPlotData != NULL)
            {
                float fLastValue = (*mIt).second.m_pfPlotData[((*mIt).second.m_ulSizeOfData - 1) * 2 + 1];
                strLastValue.Format("%d: %.2f", (int)fPlotCount + 1, fLastValue);
                if (fLastValue < (*mIt).second.m_fPlotMinY ||
                    fLastValue > (*mIt).second.m_fPlotMaxY)
                {
                    // In case something special is needed for out of bounds
                    // data.
                    oglTextLastValue.Draw(strLastValue, true, 0.0f, 175.0f);
                }
                else
                {
                    oglTextLastValue.Draw(strLastValue, true, 0.0f, 175.0f);
                }
            }
            else
            {
                strLastValue.Format("%d: N/A", (int)fPlotCount);
                oglTextLastValue.Draw(strLastValue, true, 0.0f, 175.0f);
            }
        }
        glPopMatrix();

        fPlotCount += 1.0f;

        DrawYLabels((*mIt).first);
        mIt++;
    }

    glPushMatrix();
    if (m_bPlot == true || m_bClearPlot == false)
    {
        glStencilFunc(GL_NOTEQUAL, 1, 1);
        glStencilOp(GL_KEEP, GL_KEEP, GL_KEEP);

        // Graph the lines 
        mIt = m_mapVarName2PlotData.begin();
        double dXOffset = (fXBORDER + fSMALLOFFSET)/(m_width)*(m_fPlotMaxX - m_fPlotMinX);
        while (mIt != mendIt)
        {
            if ((*mIt).second.m_pfPlotData != NULL)
            {
                if ((*mIt).second.m_bShow == true)
                {
                    glColor3ub((*mIt).second.m_color.Red(), 
                               (*mIt).second.m_color.Green(), 
                               (*mIt).second.m_color.Blue());

                    glVertexPointer(2, GL_FLOAT, 0, (void*)(*mIt).second.m_pfPlotData);
                    double dYOffset = (fYBORDER + fSMALLOFFSET)/m_height*((*mIt).second.m_fPlotMaxY - (*mIt).second.m_fPlotMinY);
                    if (m_ucPlotMode == WRAP_MODE)
                    {
                        float fDiffX    = m_fPlotMaxX - m_fPlotMinX;
                        float fMaxTime  = (*mIt).second.m_pfPlotData[0];
                        float fStartTime = fMaxTime;
                        float fPlotMinX = 0.0f;
                        float fPlotMaxX = fDiffX;
                        while (!(fPlotMinX <=  fMaxTime && fPlotMaxX >= fMaxTime))
                        {
                            fPlotMinX += fDiffX;
                            fPlotMaxX += fDiffX;
                        }
                        fMaxTime = fPlotMinX;

                        int   nFirst    = 0;
                        int   nToDraw   = 0;
                        int   nCount    = (int)(fMaxTime + fDiffX - fStartTime);
                        int   nSize     = (*mIt).second.m_ulSizeOfData;
                        while (fMaxTime < m_fCurrentMaxDatapointTime)
                        {
                            glLoadIdentity();
                            gluOrtho2D(fMaxTime - dXOffset,
                                       fMaxTime + fDiffX + dXOffset,
                                       (*mIt).second.m_fPlotMinY - dYOffset, 
                                       (*mIt).second.m_fPlotMaxY + dYOffset);

                            try
                            {
                            if (fDiffX < (m_fCurrentMaxDatapointTime - fMaxTime))
                            {
                                glDrawArrays(GL_LINE_STRIP, nFirst, nCount + 1);
                                nFirst += nCount;
                                fStartTime  = (*mIt).second.m_pfPlotData[2*nFirst];
                                nCount      = (int)(fMaxTime + 2.0f*fDiffX - fStartTime);
                            }
                            else
                            {
                                nToDraw = nSize - nFirst;

                                glDrawArrays(GL_LINE_STRIP, nFirst, nToDraw);
                            }
                            }
                            catch(...)
                            {
                                int i;
                                i = 1;
                            }

                            fMaxTime   += fDiffX;
                        }
                    }
                    else
                    {
                        glLoadIdentity();
                        gluOrtho2D(m_fPlotMinX - dXOffset,
                                   m_fPlotMaxX  + dXOffset,
                                   (*mIt).second.m_fPlotMinY - dYOffset, 
                                   (*mIt).second.m_fPlotMaxY + dYOffset);

                        unsigned long int ulCount = 
                                (unsigned long int)(m_fCurrentMaxDatapointTime - m_fPlotMinX);
                        if (ulCount > (*mIt).second.m_ulSizeOfData)
                        {
                            ulCount = (*mIt).second.m_ulSizeOfData;
                        }

                        int nFirst  = (*mIt).second.m_ulSizeOfData - ulCount;

                        glDrawArrays(GL_LINE_STRIP, nFirst, ulCount);
                    }
                }
            }

            mIt++;
        }
    }
    glPopMatrix();

    if (m_bInspectValues == true)
    {
        // Draw a box with values from the current 
        // min/max time.
        DrawInspection();
    }

    glStencilFunc(GL_ALWAYS, 0, 0);
    glStencilOp(GL_KEEP, GL_KEEP, GL_KEEP);

    m_width_old     = m_width;
    m_height_old    = m_height;
    m_bGridChanged  = false;
}

/////////////////////////////////////////////////////////////////////////////
//
// void CTimePlotWidget::UpdatePlotData()
//
// Inputs           : _FSI_STL::string& rstlStrVariable - internal name of 
//                                                        variable
//                    CVariant* pVariant                - y data
//
// Return Values    : None.
//
// Date             : 21 January 2000
//
// Engineer         : Billy Baker
//
// Description      : CTimePlotWidget::UpdatePlotData() is called from 
//                    ChangeValue() whenever new y data arrives.
//
/////////////////////////////////////////////////////////////////////////////
void CTimePlotWidget::UpdatePlotData(const _FSI_STL::string& rstlStrVariable, 
                                     CVariant* pVariant)
{
    CPlotData& rPlotData = m_mapVarName2PlotData_cv[rstlStrVariable];

    if (rPlotData.m_pfPlotData == NULL)
    {
        rPlotData.m_pfPlotData      = new float[2 * (int)(m_fRate * 60 * nMINUTES_OF_DATA)];
        rPlotData.m_pfPlotData[0]   = m_fCurrentMaxDatapointTime;
        rPlotData.m_pfPlotData[1]   = (float)(*pVariant);
        rPlotData.m_ulSizeOfData    = 1;
    }
    else
    {
        // If the max size has been reached, then move the data down.
        if (rPlotData.m_ulSizeOfData == (int)(m_fRate * 60 * nMINUTES_OF_DATA))
        {
            int nLast   = ((int)(m_fRate * 60 * nMINUTES_OF_DATA - 1)) * 2;
            int i       = 0;
            while (i < nLast)
            {
                rPlotData.m_pfPlotData[i]   = rPlotData.m_pfPlotData[i+2];
                rPlotData.m_pfPlotData[i+1] = rPlotData.m_pfPlotData[i+3];

                i += 2;
            }

            rPlotData.m_ulSizeOfData--;
        }

        // Get the last time and increment it.
        float fTime = rPlotData.m_pfPlotData[(rPlotData.m_ulSizeOfData - 1) * 2];
        fTime = (((float)(unsigned long int)(fTime + 0.5f)) + 1.0f);

        // If the current time is greater than the current max time, update
        // current max time.
        if (fTime > m_fCurrentMaxDatapointTime)
        {
            m_fCurrentMaxDatapointTime = fTime;
        }

        // Add the data to the array.
        rPlotData.m_pfPlotData[rPlotData.m_ulSizeOfData * 2]        = fTime;
        rPlotData.m_pfPlotData[rPlotData.m_ulSizeOfData * 2 + 1]    = (float)(*pVariant);
        rPlotData.m_ulSizeOfData++;

        if (fTime > m_fPlotMaxX_cv)
        {
            if (m_ucPlotMode == STOP_MODE)
            {
                // The current time is one greater than the stop time.  Decrease it.
                m_fCurrentMaxDatapointTime--;

                // The size of the data is one larger than it needs to be to stop.
                // Decrease the size.
                rPlotData.m_ulSizeOfData--;
                m_bStopped = true;
            }
            else
            {
                // Change the time limits for the wrap and erase modes.
                m_fPlotMinX_cv = m_fPlotMaxX_cv;
                m_fPlotMaxX_cv = m_fPlotMaxX_cv + m_fPlotMaxX_cv - m_fPlotMinX_cv;
            }
        }

    }
}

/////////////////////////////////////////////////////////////////////////////
//
// void CTimePlotWidget::DrawXLabels()
//
// Inputs           : None.
//
// Return Values    : None.
//
// Date             : 01 October 1999
//
// Engineer         : Billy Baker
//
// Description      : CTimePlotWidget::DrawXLabels() is called from Render().
//                    This method will check to see if the width or height 
//                    of the widget has changed since the last pass.  If it 
//                    has, then the map of x label position to label will
//                    be recreated.  For the time based plot, the map
//                    would need to regenerate the labels if the width and
//                    height didn't change but the graph either wrapped or
//                    erased.
//
/////////////////////////////////////////////////////////////////////////////
void CTimePlotWidget::DrawXLabels()
{
    if (m_width != m_width_old || m_height != m_height_old || 
        m_bGridChanged == true)
    {
        m_mapXLabels.clear();

        CString strLabel;
        COpenGLtext oglText;
        float fLabel;
        float fWidthDivDivisions    = m_width  / (float)m_ulXDivisions;
        float fMinDivRate           = (m_fPlotMinX / m_fRate);
        unsigned long int ul = 0; 
        while (ul <= m_ulXDivisions)
        {
            fLabel = (float)ul * m_fSecondsPerDivision + fMinDivRate;
            strLabel.Format("%.2f",fLabel);
            oglText.Text(strLabel);
            m_mapXLabels[(float)ul * fWidthDivDivisions] = oglText;
            ul++;
        }
    }

    // Draw all of the x labels.
    _FSI_STL::map<float, COpenGLtext>::iterator mIt = m_mapXLabels.begin();
    _FSI_STL::map<float, COpenGLtext>::iterator mendIt = m_mapXLabels.end();
    while (mIt != mendIt)
    {
        glPushMatrix();
        {
            glTranslatef((*mIt).first, -13.0f, 0.0f);
            (*mIt).second.Draw((*mIt).second.Text(), true, 0.0f, 175.0f);
        }
        glPopMatrix();
        mIt++;
    }
}

/////////////////////////////////////////////////////////////////////////////
//
// void CTimePlotWidget::DrawYLabels()
//
// Inputs           : None.
//
// Return Values    : None.
//
// Date             : 31 January 2000
//
// Engineer         : Billy Baker
//
// Description      : CTimePlotWidget::DrawYLabels() is called from Render().
//                    This method will check to see if the width or height 
//                    of the widget has changed since the last pass.  If it 
//                    has, then the map of y label position to label will
//                    be recreated.
//
/////////////////////////////////////////////////////////////////////////////
void CTimePlotWidget::DrawYLabels(const _FSI_STL::string& rstlStrVarName)
{
    CPlotData& rPlotData = m_mapVarName2PlotData[rstlStrVarName];
    
    CString strNumber(rstlStrVarName.c_str());
    strNumber = strNumber.Mid(strNumber.ReverseFind(' ') + 1);
    int nVariableNumber = atoi(strNumber);

    if (m_width != m_width_old || m_height != m_height_old ||
        rPlotData.m_bRecomputeYLabels == true || m_bGridChanged == true)
    {
        m_mapVarName2YLabels[rstlStrVarName].clear();

        CString strLabel;
        COpenGLtext oglText;
        float fLabel = rPlotData.m_fPlotMaxY - rPlotData.m_fPlotMinY;
        fLabel /= (float)m_ulYDivisions;

        float fHeightDivDivisions    = m_height  / (float)m_ulYDivisions;
        float fHeightDiv5Divisions  = m_height / (10.0f * 5.0f);
        unsigned long int ul = 0; 
        while (ul <= m_ulYDivisions)
        {
            strLabel.Format("%.2f",(float)ul * fLabel + rPlotData.m_fPlotMinY);
            oglText.Text(strLabel);
            oglText.SetForegroundColor(rPlotData.m_color.Red(), 
                                       rPlotData.m_color.Green(), 
                                       rPlotData.m_color.Blue());

            if (ul == 0)
            {
                int nRemainder = nVariableNumber % 3;
                if (nRemainder == 0)
                {
                    (m_mapVarName2YLabels[rstlStrVarName])[(float)ul * fHeightDivDivisions + 2.0f * fHeightDiv5Divisions] = oglText;
                }
                else if (nRemainder == 1)
                {
                    (m_mapVarName2YLabels[rstlStrVarName])[(float)ul * fHeightDivDivisions ] = oglText;
                }
                else
                {
                    (m_mapVarName2YLabels[rstlStrVarName])[(float)ul * fHeightDivDivisions + 1.0f * fHeightDiv5Divisions] = oglText;
                }
            }
            else if (ul == m_ulYDivisions)
            {
                int nRemainder = nVariableNumber % 3;
                if (nRemainder == 0)
                {
                    (m_mapVarName2YLabels[rstlStrVarName])[(float)ul * fHeightDivDivisions] = oglText;
                }
                else if (nRemainder == 1)
                {
                    (m_mapVarName2YLabels[rstlStrVarName])[(float)ul * fHeightDivDivisions - 2.0f * fHeightDiv5Divisions] = oglText;
                }
                else
                {
                    (m_mapVarName2YLabels[rstlStrVarName])[(float)ul * fHeightDivDivisions - 1.0f * fHeightDiv5Divisions] = oglText;
                }
            }
            else
            {
                int nRemainder = nVariableNumber % 3;
                if (nRemainder == 0)
                {
                    (m_mapVarName2YLabels[rstlStrVarName])[(float)ul * fHeightDivDivisions + 1.0f * fHeightDiv5Divisions] = oglText;
                }
                else if (nRemainder == 1)
                {
                    (m_mapVarName2YLabels[rstlStrVarName])[(float)ul * fHeightDivDivisions - 1.0f * fHeightDiv5Divisions] = oglText;
                }
                else
                {
                    (m_mapVarName2YLabels[rstlStrVarName])[(float)ul * fHeightDivDivisions] = oglText;
                }
            }

            ul++;
        }

        rPlotData.m_bRecomputeYLabels = false;
    }

    if (rPlotData.m_bShow == true)
    {
        // Draw all of the y labels.
        _FSI_STL::map<float, COpenGLtext>::iterator mIt = 
                                  m_mapVarName2YLabels[rstlStrVarName].begin();
        _FSI_STL::map<float, COpenGLtext>::iterator mendIt = 
                                  m_mapVarName2YLabels[rstlStrVarName].end();
        if (nVariableNumber < 4)
        {
            while (mIt != mendIt)
            {
                glPushMatrix();
                {
                    glTranslatef(-49.0f, (*mIt).first, 0.0f);
                    (*mIt).second.Draw((*mIt).second.Text(), true, 0.0f, 175.0f);
                }
                glPopMatrix();

                mIt++;
            }
        }
        else
        {
            while (mIt != mendIt)
            {
                glPushMatrix();
                {
                    glTranslatef(m_width + 49.0f, (*mIt).first, 0.0f);
                    (*mIt).second.Draw((*mIt).second.Text(), true, 0.0f, 175.0f);
                }
                glPopMatrix();

                mIt++;
            }
        }
    }
}

/////////////////////////////////////////////////////////////////////////////
//
// void CTImePlotWidget::DrawInspection()
//
// Inputs           : None.
//
// Return Values    : None.
//
// Date             : 21 January 2000
//
// Engineer         : Billy Baker
//
// Description      : CTImePlotWidget::DrawInspection() is called from Render().
//                    This method will draw information related to the data
//                    that would be intersect by a line passing throught the
//                    mouse's current position and perpendicular to the 
//                    bounding box of the plot area.
//
/////////////////////////////////////////////////////////////////////////////
void CTimePlotWidget::DrawInspection()
{
    // Translate the mouse position to a position within the plotting region.
    float fTransX = (m_width + 2 * fXBORDER + 2 * fSMALLOFFSET) / 
                    m_width * (float)m_ptMouse.x - 
                    fXBORDER - fSMALLOFFSET;
    float fTransY = (m_height + 2 * fYBORDER + 2 * fSMALLOFFSET) / 
                    m_height * (m_height - (float)m_ptMouse.y) - 
                    fYBORDER - fSMALLOFFSET;

    if (fTransX < 0.0f || fTransX > m_width || fTransY < 0.0f || fTransY > m_height)
    {
        return;
    }

    COpenGLtext oglInspection[6];
    oglInspection[0].SetBackgroundColor(true, 0, 0, 0);
    oglInspection[1].SetBackgroundColor(true, 0, 0, 0);
    oglInspection[2].SetBackgroundColor(true, 0, 0, 0);
    oglInspection[3].SetBackgroundColor(true, 0, 0, 0);
    oglInspection[4].SetBackgroundColor(true, 0, 0, 0);
    oglInspection[5].SetBackgroundColor(true, 0, 0, 0);
    CString strValues[6]   = {"", "", "", "", "", ""};

    _FSI_STL::map<_FSI_STL::string, CPlotData>::iterator mIt;
    _FSI_STL::map<_FSI_STL::string, CPlotData>::iterator mendIt;
    mIt     = m_mapVarName2PlotData.begin();
    mendIt  = m_mapVarName2PlotData.end();
    float fMouseTime            = m_fPlotMinX + fTransX * (m_fPlotMaxX - m_fPlotMinX) / m_width;
    float fHeightDiv5Divisions  = m_height / (10.0f * 5.0f);
    int nCount  = 0;
    int nLabels = 0;

    while (mIt != mendIt)
    {
        if ((*mIt).second.m_ulSizeOfData > 0)
        {
            float fFirstTime    = (*mIt).second.m_pfPlotData[0];
            float fLastTime     = (*mIt).second.m_pfPlotData[((*mIt).second.m_ulSizeOfData - 1) * 2];
            if (fFirstTime <= fMouseTime && fMouseTime <= fLastTime)
            {
                nLabels++;
                strValues[nCount].Format("%6.3f , %12.3f", (*mIt).second.m_pfPlotData[(int)(fMouseTime - fFirstTime) * 2] / m_fRate,
                                                            (*mIt).second.m_pfPlotData[(int)(fMouseTime - fFirstTime) * 2 + 1]);
                oglInspection[nCount].SetForegroundColor((*mIt).second.m_color.Red(), 
                                                 (*mIt).second.m_color.Green(),
                                                 (*mIt).second.m_color.Blue());
            }

        }

        nCount++;
        mIt++;
    }

    nCount = 0;

    if (nLabels > 0)
    {
        glPushMatrix();

        // Draw line from top to bottom at current mouse position.
        glBegin(GL_LINES);
          glColor3ub(255, 255, 255);
          glVertex2f(fTransX, 0.0f);
          glVertex2f(fTransX, m_height);
        glEnd();

        // Make sure the entire box of data will fit in the plotting region.
        if (fTransX < m_width / 9.0f)
        {
            fTransX = m_width / 9.0f ;
        }
        else if (fTransX > 8.0f * m_width / 9.0f)
        {
            fTransX = 8.0f * m_width / 9.0f;
        }

        if (fTransY < (nLabels + 1) * fHeightDiv5Divisions)
        {
            fTransY = (nLabels + 1) * fHeightDiv5Divisions;
        }
        else if (fTransY > m_height - 2 * fHeightDiv5Divisions)
        {
            fTransY = m_height - 2 * fHeightDiv5Divisions;
        }

        glTranslatef(fTransX, fTransY, 0.0f);

        // Draw a black background for the data.
        glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
        glBegin(GL_QUADS);
            glColor3ub(0, 0, 0);
            glVertex2f(-m_width/10.0f, fHeightDiv5Divisions);
            glVertex2f( m_width/10.0f, fHeightDiv5Divisions);
            glVertex2f( m_width/10.0f, -(float)nLabels * fHeightDiv5Divisions);
            glVertex2f(-m_width/10.0f, -(float)nLabels * fHeightDiv5Divisions);
        glEnd();

        // Surround the black background with a white border.
        glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
        glBegin(GL_LINE_STRIP);
            glColor3ub(255, 255, 255);
            glVertex2f(-m_width/10.0f, fHeightDiv5Divisions);
            glVertex2f( m_width/10.0f, fHeightDiv5Divisions);
            glVertex2f( m_width/10.0f, -(float)nLabels * fHeightDiv5Divisions);
            glVertex2f(-m_width/10.0f, -(float)nLabels * fHeightDiv5Divisions);
            glVertex2f(-m_width/10.0f, fHeightDiv5Divisions);
        glEnd();

        // Draw the data.
        glPushMatrix();
        while (nCount < 6)
        {
            if (strValues[nCount] != "")
            {
                oglInspection[nCount].Draw(strValues[nCount], true, 0.0f, 175.0f);
                glTranslatef(0.0f, -fHeightDiv5Divisions, 0.0f);
            }

            nCount++;
        }
        glPopMatrix();

        glPopMatrix();
    }

}

/////////////////////////////////////////////////////////////////////////////
//
// Methods called from Comms thread 
//
/////////////////////////////////////////////////////////////////////////////

/////////////////////////////////////////////////////////////////////////////
//
// void CTimePlotWidget::ChangeValue()
//
// Inputs           : CString& rstrElementVar - the internal name of a 
//                                              variable that needs to be 
//                                              updated.
//                    CChangeValue* pCV       - pointer to data used to 
//                                              update an internal 
//                                              variable.
//
// Return Values    : None.
//
// Date             : 21 January 2000
//
// Engineer         : Billy Baker
//
// Description      : CTimePlotWidget::ChangeValue() is a common framework 
//                    method that is used by the Comms thread to send 
//                    new data to the graphical elements.  Since it is 
//                    called from another thread, the special internal 
//                    variables are used to store the updates.  Before 
//                    the graphical element is redrawn, 
//                    UpdateRenderVariables is called to copy the 
//                    updated values to the values used for rendering.  
//                    This causes more memory to be used but prevents 
//                    the use of synchronization objects that may stall 
//                    the Comms thread.
//
/////////////////////////////////////////////////////////////////////////////
void CTimePlotWidget::ChangeValue(const CString& rstrElementVar, CChangeValue* pCV)
{
    if (pCV == NULL)
    {
        return;
    }

    if (rstrElementVar   == "Start/Stop")
    {
        // Change the state of plotting.
        m_bPlot_cv = !m_bPlot;
        if (m_bPlot_cv == true)
        {
            m_fPlotMinX_cv = 0.0f;
            m_fCurrentMaxDatapointTime = 0.0f;

            // Erase all of the x,y data.
            _FSI_STL::map<_FSI_STL::string, CPlotData>::iterator mIt;
            _FSI_STL::map<_FSI_STL::string, CPlotData>::iterator mendIt;
            mIt = m_mapVarName2PlotData_cv.begin();
            mendIt = m_mapVarName2PlotData_cv.end();
            while (mIt != mendIt)
            {
                if ((*mIt).second.m_pfPlotData != NULL)
                {
                    delete [] (*mIt).second.m_pfPlotData;
                    (*mIt).second.m_pfPlotData      = NULL;
                    (*mIt).second.m_ulSizeOfData    = 0;
                }

                mIt++;
            }
        }
    }
    else if (rstrElementVar   == "Clear")
    {
        m_fPlotMinX_cv = 0.0f;
        m_fCurrentMaxDatapointTime = 0.0f;

        // Erase all of the x,y data.
        _FSI_STL::map<_FSI_STL::string, CPlotData>::iterator mIt;
        _FSI_STL::map<_FSI_STL::string, CPlotData>::iterator mendIt;
        mIt = m_mapVarName2PlotData_cv.begin();
        mendIt = m_mapVarName2PlotData_cv.end();
        while (mIt != mendIt)
        {
            if ((*mIt).second.m_pfPlotData != NULL)
            {
                delete [] (*mIt).second.m_pfPlotData;
                (*mIt).second.m_pfPlotData      = NULL;
                (*mIt).second.m_ulSizeOfData    = 0;
            }

            mIt++;
        }
    }
    else if (rstrElementVar   == "Plot Mode")
    {
        m_ucPlotMode_cv = (long)*pCV->Variant();
    }
    else if (rstrElementVar   == "Plot Y Var 2 Name")
    {
        CXMLElement* pXMLElement = NULL;
        POSITION pos = NULL;
        if (m_xmlCommsSampleWidget.FindElement(pXMLElement, pos, _FSI_STL::string("VARIABLE")) == true)
        {
            pXMLElement->AddElementValue((_FSI_STL::string)*pCV->Variant());

            STRING2STRING_MAP::iterator s2sIt;

            if (pXMLElement->FindAttribute(s2sIt,_FSI_STL::string("ELEMENT_VAR")) == true)
            {
                (*s2sIt).second = _FSI_STL::string("Plot Y Var 2");
            }
        }

        _FSI_STL::map<_FSI_STL::string, CPlotData>::iterator mIt = 
                                            m_mapVarName2PlotData_cv.find("Plot Y Var 2");
        if (mIt == m_mapVarName2PlotData_cv.end())
        {
            m_mapVarName2PlotData_cv["Plot Y Var 2"];
            mIt = m_mapVarName2PlotData_cv.find("Plot Y Var 2");
        }

        (*mIt).second.m_xmlCommsWidget = m_xmlCommsSampleWidget;

        if ((*mIt).second.m_pfPlotData != NULL)
        {
            delete (*mIt).second.m_pfPlotData;
            (*mIt).second.m_pfPlotData = NULL;
            (*mIt).second.m_ulSizeOfData = 0;
        }

        if (m_mapCommsActions["Plot Y Var 2"] != NULL)
        {
            (m_mapCommsActions["Plot Y Var 2"])->Deleting(true);
            m_mapCommsActions["Plot Y Var 2"] = NULL;
        }
    }
    else if (rstrElementVar   == "Plot Y Var 3 Name")
    {
        CXMLElement* pXMLElement = NULL;
        POSITION pos = NULL;
        if (m_xmlCommsSampleWidget.FindElement(pXMLElement, pos, _FSI_STL::string("VARIABLE")) == true)
        {
            pXMLElement->AddElementValue((_FSI_STL::string)*pCV->Variant());

            STRING2STRING_MAP::iterator s2sIt;

            if (pXMLElement->FindAttribute(s2sIt,_FSI_STL::string("ELEMENT_VAR")) == true)
            {
                (*s2sIt).second = _FSI_STL::string("Plot Y Var 3");
            }
        }

        _FSI_STL::map<_FSI_STL::string, CPlotData>::iterator mIt = 
                                            m_mapVarName2PlotData_cv.find("Plot Y Var 3");
        if (mIt == m_mapVarName2PlotData_cv.end())
        {
            m_mapVarName2PlotData_cv["Plot Y Var 3"];
            mIt = m_mapVarName2PlotData_cv.find("Plot Y Var 3");
        }

        (*mIt).second.m_xmlCommsWidget = m_xmlCommsSampleWidget;

        if ((*mIt).second.m_pfPlotData != NULL)
        {
            delete (*mIt).second.m_pfPlotData;
            (*mIt).second.m_pfPlotData = NULL;
            (*mIt).second.m_ulSizeOfData = 0;
        }

        if (m_mapCommsActions["Plot Y Var 3"] != NULL)
        {
            (m_mapCommsActions["Plot Y Var 3"])->Deleting(true);
            m_mapCommsActions["Plot Y Var 3"] = NULL;
        }
    }
    else if (rstrElementVar   == "Plot Y Var 4 Name")
    {
        CXMLElement* pXMLElement = NULL;
        POSITION pos = NULL;
        if (m_xmlCommsSampleWidget.FindElement(pXMLElement, pos, _FSI_STL::string("VARIABLE")) == true)
        {
            pXMLElement->AddElementValue((_FSI_STL::string)*pCV->Variant());

            STRING2STRING_MAP::iterator s2sIt;

            if (pXMLElement->FindAttribute(s2sIt,_FSI_STL::string("ELEMENT_VAR")) == true)
            {
                (*s2sIt).second = _FSI_STL::string("Plot Y Var 4");
            }
        }

        _FSI_STL::map<_FSI_STL::string, CPlotData>::iterator mIt = 
                                            m_mapVarName2PlotData_cv.find("Plot Y Var 4");
        if (mIt == m_mapVarName2PlotData_cv.end())
        {
            m_mapVarName2PlotData_cv["Plot Y Var 4"];
            mIt = m_mapVarName2PlotData_cv.find("Plot Y Var 4");
        }

        (*mIt).second.m_xmlCommsWidget = m_xmlCommsSampleWidget;

        if ((*mIt).second.m_pfPlotData != NULL)
        {
            delete (*mIt).second.m_pfPlotData;
            (*mIt).second.m_pfPlotData = NULL;
            (*mIt).second.m_ulSizeOfData = 0;
        }

        if (m_mapCommsActions["Plot Y Var 4"] != NULL)
        {
            (m_mapCommsActions["Plot Y Var 4"])->Deleting(true);
            m_mapCommsActions["Plot Y Var 4"] = NULL;
        }
    }
    else if (rstrElementVar   == "Plot Y Var 5 Name")
    {
        CXMLElement* pXMLElement = NULL;
        POSITION pos = NULL;
        if (m_xmlCommsSampleWidget.FindElement(pXMLElement, pos, _FSI_STL::string("VARIABLE")) == true)
        {
            pXMLElement->AddElementValue((_FSI_STL::string)*pCV->Variant());

            STRING2STRING_MAP::iterator s2sIt;

            if (pXMLElement->FindAttribute(s2sIt,_FSI_STL::string("ELEMENT_VAR")) == true)
            {
                (*s2sIt).second = _FSI_STL::string("Plot Y Var 5");
            }
        }

        _FSI_STL::map<_FSI_STL::string, CPlotData>::iterator mIt = 
                                            m_mapVarName2PlotData_cv.find("Plot Y Var 5");
        if (mIt == m_mapVarName2PlotData_cv.end())
        {
            m_mapVarName2PlotData_cv["Plot Y Var 5"];
            mIt = m_mapVarName2PlotData_cv.find("Plot Y Var 5");
        }

        (*mIt).second.m_xmlCommsWidget = m_xmlCommsSampleWidget;

        if ((*mIt).second.m_pfPlotData != NULL)
        {
            delete (*mIt).second.m_pfPlotData;
            (*mIt).second.m_pfPlotData = NULL;
            (*mIt).second.m_ulSizeOfData = 0;
        }

        if (m_mapCommsActions["Plot Y Var 5"] != NULL)
        {
            (m_mapCommsActions["Plot Y Var 5"])->Deleting(true);
            m_mapCommsActions["Plot Y Var 5"] = NULL;
        }
    }
    else if (rstrElementVar   == "Plot Y Var 6 Name")
    {
        CXMLElement* pXMLElement = NULL;
        POSITION pos = NULL;
        if (m_xmlCommsSampleWidget.FindElement(pXMLElement, pos, _FSI_STL::string("VARIABLE")) == true)
        {
            pXMLElement->AddElementValue((_FSI_STL::string)*pCV->Variant());

            STRING2STRING_MAP::iterator s2sIt;

            if (pXMLElement->FindAttribute(s2sIt,_FSI_STL::string("ELEMENT_VAR")) == true)
            {
                (*s2sIt).second = _FSI_STL::string("Plot Y Var 6");
            }
        }

        _FSI_STL::map<_FSI_STL::string, CPlotData>::iterator mIt = 
                                            m_mapVarName2PlotData_cv.find("Plot Y Var 6");
        if (mIt == m_mapVarName2PlotData_cv.end())
        {
            m_mapVarName2PlotData_cv["Plot Y Var 6"];
            mIt = m_mapVarName2PlotData_cv.find("Plot Y Var 6");
        }

        (*mIt).second.m_xmlCommsWidget = m_xmlCommsSampleWidget;

        if ((*mIt).second.m_pfPlotData != NULL)
        {
            delete (*mIt).second.m_pfPlotData;
            (*mIt).second.m_pfPlotData = NULL;
            (*mIt).second.m_ulSizeOfData = 0;
        }

        if (m_mapCommsActions["Plot Y Var 6"] != NULL)
        {
            (m_mapCommsActions["Plot Y Var 6"])->Deleting(true);
            m_mapCommsActions["Plot Y Var 6"] = NULL;
        }
    }
    else if (rstrElementVar   == "Plot Y Var 2")
    {
        if (m_bPlot_cv == true && m_bStopped == false)
        {
            UpdatePlotData(_FSI_STL::string("Plot Y Var 2"), pCV->Variant());
        }
    }
    else if (rstrElementVar   == "Plot Y Var 3")
    {
        if (m_bPlot_cv == true && m_bStopped == false)
        {
            UpdatePlotData(_FSI_STL::string("Plot Y Var 3"), pCV->Variant());
        }
    }
    else if (rstrElementVar   == "Plot Y Var 4")
    {
        if (m_bPlot_cv == true && m_bStopped == false)
        {
            UpdatePlotData(_FSI_STL::string("Plot Y Var 4"), pCV->Variant());
        }
    }
    else if (rstrElementVar   == "Plot Y Var 5")
    {
        if (m_bPlot_cv == true && m_bStopped == false)
        {
            UpdatePlotData(_FSI_STL::string("Plot Y Var 5"), pCV->Variant());
        }
    }
    else if (rstrElementVar   == "Plot Y Var 6")
    {
        if (m_bPlot_cv == true && m_bStopped == false)
        {
            UpdatePlotData(_FSI_STL::string("Plot Y Var 6"), pCV->Variant());
        }
    }
    else if (rstrElementVar == "Seconds Per Division")
    {
        m_fSecondsPerDivision_cv = *pCV->Variant();
        if (m_fSecondsPerDivision_cv <= 0.0f)
        {
            m_fSecondsPerDivision_cv = 1.0f;
        }
    }
    else if (rstrElementVar == "Y Var 2 Min")
    {
        if (m_mapVarName2PlotData_cv["Plot Y Var 2"].m_fPlotMinY != (float)*pCV->Variant())
        {
            m_mapVarName2PlotData_cv["Plot Y Var 2"].m_fPlotMinY = *pCV->Variant();
            m_mapVarName2PlotData_cv["Plot Y Var 2"].m_bRecomputeYLabels = true;
        }
    }
    else if (rstrElementVar == "Y Var 2 Max")
    {
        if (m_mapVarName2PlotData_cv["Plot Y Var 2"].m_fPlotMaxY != (float)*pCV->Variant())
        {
            m_mapVarName2PlotData_cv["Plot Y Var 2"].m_fPlotMaxY = *pCV->Variant();
            m_mapVarName2PlotData_cv["Plot Y Var 2"].m_bRecomputeYLabels = true;
        }
    }
    else if (rstrElementVar == "Y Var 3 Min")
    {
        if (m_mapVarName2PlotData_cv["Plot Y Var 3"].m_fPlotMinY != (float)*pCV->Variant())
        {
            m_mapVarName2PlotData_cv["Plot Y Var 3"].m_fPlotMinY = *pCV->Variant();
            m_mapVarName2PlotData_cv["Plot Y Var 3"].m_bRecomputeYLabels = true;
        }
    }
    else if (rstrElementVar == "Y Var 3 Max")
    {
        if (m_mapVarName2PlotData_cv["Plot Y Var 3"].m_fPlotMaxY != (float)*pCV->Variant())
        {
            m_mapVarName2PlotData_cv["Plot Y Var 3"].m_fPlotMaxY = *pCV->Variant();
            m_mapVarName2PlotData_cv["Plot Y Var 3"].m_bRecomputeYLabels = true;
        }
    }
    else if (rstrElementVar == "Y Var 4 Min")
    {
        if (m_mapVarName2PlotData_cv["Plot Y Var 4"].m_fPlotMinY != (float)*pCV->Variant())
        {
            m_mapVarName2PlotData_cv["Plot Y Var 4"].m_fPlotMinY = *pCV->Variant();
            m_mapVarName2PlotData_cv["Plot Y Var 4"].m_bRecomputeYLabels = true;
        }
    }
    else if (rstrElementVar == "Y Var 4 Max")
    {
        if (m_mapVarName2PlotData_cv["Plot Y Var 4"].m_fPlotMaxY != (float)*pCV->Variant())
        {
            m_mapVarName2PlotData_cv["Plot Y Var 4"].m_fPlotMaxY = *pCV->Variant();
            m_mapVarName2PlotData_cv["Plot Y Var 4"].m_bRecomputeYLabels = true;
        }
    }
    else if (rstrElementVar == "Y Var 5 Min")
    {
        if (m_mapVarName2PlotData_cv["Plot Y Var 5"].m_fPlotMinY != (float)*pCV->Variant())
        {
            m_mapVarName2PlotData_cv["Plot Y Var 5"].m_fPlotMinY = *pCV->Variant();
            m_mapVarName2PlotData_cv["Plot Y Var 5"].m_bRecomputeYLabels = true;
        }
    }
    else if (rstrElementVar == "Y Var 5 Max")
    {
        if (m_mapVarName2PlotData_cv["Plot Y Var 5"].m_fPlotMaxY != (float)*pCV->Variant())
        {
            m_mapVarName2PlotData_cv["Plot Y Var 5"].m_fPlotMaxY = *pCV->Variant();
            m_mapVarName2PlotData_cv["Plot Y Var 5"].m_bRecomputeYLabels = true;
        }
    }
    else if (rstrElementVar == "Y Var 6 Min")
    {
        if (m_mapVarName2PlotData_cv["Plot Y Var 6"].m_fPlotMinY != (float)*pCV->Variant())
        {
            m_mapVarName2PlotData_cv["Plot Y Var 6"].m_fPlotMinY = *pCV->Variant();
            m_mapVarName2PlotData_cv["Plot Y Var 6"].m_bRecomputeYLabels = true;
        }
    }
    else if (rstrElementVar == "Y Var 6 Max")
    {
        if (m_mapVarName2PlotData_cv["Plot Y Var 6"].m_fPlotMaxY != (float)*pCV->Variant())
        {
            m_mapVarName2PlotData_cv["Plot Y Var 6"].m_fPlotMaxY = *pCV->Variant();
            m_mapVarName2PlotData_cv["Plot Y Var 6"].m_bRecomputeYLabels = true;
        }
    }
    else if (rstrElementVar == "Show Plot 1")
    {
        m_mapVarName2PlotData_cv["Plot Y Var 1"].m_bShow = (bool)*pCV->Variant();
    }
    else if (rstrElementVar == "Show Plot 2")
    {
        m_mapVarName2PlotData_cv["Plot Y Var 2"].m_bShow = (bool)*pCV->Variant();
    }
    else if (rstrElementVar == "Show Plot 3")
    {
        m_mapVarName2PlotData_cv["Plot Y Var 3"].m_bShow = (bool)*pCV->Variant();
    }
    else if (rstrElementVar == "Show Plot 4")
    {
        m_mapVarName2PlotData_cv["Plot Y Var 4"].m_bShow = (bool)*pCV->Variant();
    }
    else if (rstrElementVar == "Show Plot 5")
    {
        m_mapVarName2PlotData_cv["Plot Y Var 5"].m_bShow = (bool)*pCV->Variant();
    }
    else if (rstrElementVar == "Show Plot 6")
    {
        m_mapVarName2PlotData_cv["Plot Y Var 6"].m_bShow = (bool)*pCV->Variant();
    }
    else if (rstrElementVar == "Color Y Var 1")
    {
        CColor& rColor = m_mapColors[pCV->Text()];
        m_mapVarName2PlotData_cv["Plot Y Var 1"].m_color = rColor;
    }
    else if (rstrElementVar == "Color Y Var 2")
    {
        CColor& rColor = m_mapColors[pCV->Text()];
        m_mapVarName2PlotData_cv["Plot Y Var 2"].m_color = rColor;
    }
    else if (rstrElementVar == "Color Y Var 3")
    {
        CColor& rColor = m_mapColors[pCV->Text()];
        m_mapVarName2PlotData_cv["Plot Y Var 3"].m_color = rColor;
    }
    else if (rstrElementVar == "Color Y Var 4")
    {
        CColor& rColor = m_mapColors[pCV->Text()];
        m_mapVarName2PlotData_cv["Plot Y Var 4"].m_color = rColor;
    }
    else if (rstrElementVar == "Color Y Var 5")
    {
        CColor& rColor = m_mapColors[pCV->Text()];
        m_mapVarName2PlotData_cv["Plot Y Var 5"].m_color = rColor;
    }
    else if (rstrElementVar == "Color Y Var 6")
    {
        CColor& rColor = m_mapColors[pCV->Text()];
        m_mapVarName2PlotData_cv["Plot Y Var 6"].m_color = rColor;
    }
    else
    {
        CPlotWidget::ChangeValue(rstrElementVar, pCV);
    }
}

/////////////////////////////////////////////////////////////////////////////
//
// bool CTimePlotWidget::UpdateRenderVariables()
//
// Inputs           : None.
//
// Return Values    : true  - all of the render variables were updated.
//                    false - a problem occurred updating the render 
//                            variables.
//
// Date             : 21 January 2000
//
// Engineer         : Billy Baker
//
// Description      : CTimePlotWidget::UpdateRenderVariables is a 
//                    common framework method that is called by the 
//                    Comms thread to make sure that all of the 
//                    variables used for rendering have the most up to 
//                    date data.
//
/////////////////////////////////////////////////////////////////////////////
bool CTimePlotWidget::UpdateRenderVariables()
{
    bool bRetVal = CPlotWidget::UpdateRenderVariables();

    if (bRetVal == true)
    {
        if (m_fSecondsPerDivision != m_fSecondsPerDivision_cv)
        {
            m_bGridChanged          = true;
        }

        m_fSecondsPerDivision   = m_fSecondsPerDivision_cv;

        m_fPlotMaxX_cv          = m_fPlotMinX_cv + 
                                  m_fSecondsPerDivision * m_ulXDivisions * m_fRate;

        m_fPlotMinX             = m_fPlotMinX_cv;
        m_fPlotMaxX             = m_fPlotMaxX_cv;

        float fTimeDiff = m_fPlotMaxX - m_fPlotMinX;
        if (m_fCurrentMaxDatapointTime <= fTimeDiff)
        {
            m_fPlotMinX    = 0.0f;
            m_fPlotMaxX    = fTimeDiff;
            m_fPlotMinX_cv = 0.0f;
            m_fPlotMaxX_cv = fTimeDiff;
        }
        else
        {
            m_fPlotMinX = 0.0f;
            m_fPlotMaxX = fTimeDiff;
            while (!(m_fPlotMinX <  m_fCurrentMaxDatapointTime &&
                     m_fPlotMaxX >= m_fCurrentMaxDatapointTime))
            {
                m_fPlotMinX += fTimeDiff;
                m_fPlotMaxX += fTimeDiff;
            }

            m_fPlotMinX_cv = m_fPlotMinX;
            m_fPlotMaxX_cv = m_fPlotMaxX;
        }

        m_ucPlotMode            = m_ucPlotMode_cv;
    }

    return bRetVal;
}
