/////////////////////////////////////////////////////////////////////////////
//
//           F L I G H T S A F E T Y   I N T E R N A T I O N A L
//                     Simulation Systems Division
//                      2700 North Hemlock Circle
//                     Broken Arrow, Oklahoma 74012
//                          (918) 259-4000
/////////////////////////////////////////////////////////////////////////////
//
// DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
// Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
// technical or operational data or information from automatic dissemination
// under the International Exchange Program or by other means.  This protection
// covers information required solely for administrative or operational
// purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
//
// WARNING:  This document contains technical data whose export is restricted
// by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
// Executive Order 12470.  Violation of these export control laws is subject
// to severe criminal penalties.  Dissemination of this document is controlled
// under DOD Directive 5230.25
//
/////////////////////////////////////////////////////////////////////////////
//
//
// Filename         : SliderWidget.cpp
//
// Date             : 03 October 1999
//
// Engineer         : Billy Baker
//
// Revision         : $Revision: 1.9 $
//
// Description      : SliderWidget.cpp contains the implementation of 
//                    the CSliderWidget class.  This class is used to 
//                    draw a slider that looks similar to the common 
//                    Windows slider control.  The biggest change is 
//                    that the slider will resize as the size of the 
//                    graphical element changes.  Tics are available.
//
// Classification   : UNCLASSIFIED
//
// Requirements     : None.
//
// Components Used  : Core::CWidget, _FSI_STL::string, _FSI_STL::list, 
//                    Core::CVariant, CString, Core::CExtentsPoint.
//
// Operational 
//    Restrictions  : Machine dependencies/restrictions
//                        None.
//                    Design dependencies/restrictions
//                        None.
//                    Operations containing dependencies/restrictions
//                        None.
//                    Compiler dependencies/restrictions
//                        None.
//                    Other conditions for proper execution
//                        None.
//
// Environment      : Operating system(s) - Microsoft Windows NT 4.0 with
//                                              NT service pack 3, 4, or 5
//                                          Microsoft Windows NT 2000
//
//                    Compiler(s) - Visual C++ 6.0
//
//                    Architechure(s) - Intel Pentium, Pentium II
/////////////////////////////////////////////////////////////////////////////

/////////////////////////////////////////////////////////////////////////////
//
//                              R e v i s i o n   H i s t o r y
//
/////////////////////////////////////////////////////////////////////////////
// $Log: SliderWidget.cpp $                                                                   
// Revision 1.9  2000/03/07 20:47:32  billyb                                                                   
// Changed function signatures to eliminate warnings.                                                                   
// Revision 1.8  1999/12/08 17:50:14  billyb                                                                   
// Added CS_OWNDC.                                                                   
// Revision 1.7  1999/11/09 21:43:12  billyb                                                                   
// Changed FillRect to FillSolidRect or PatBlt.  Removed unneeded                                                                   
// message map entries.  Optimized loops.                                                                    
// Revision 1.6  1999/10/20 18:47:59  billyb                                                                   
// Changed access to range data to new map of                                                                   
// variable name to range data.  The variable is                                                                   
// assumed to be the default variable.                                                                   
// Revision 1.5  1999/10/20 07:37:46  billyb                                                                   
// Changed a number of loops to perform less calculations.                                                                   
// Revision 1.4  1999/10/04 17:00:47  billyb                                                                   
// Fixed problems with OnMouseMove and OnMouseLeave                                                                   
// where the pointer may have been moved to a bogus position                                                                   
// if it was not dragged.  Also added support for doing nothing                                                                   
// when the widget is disabled.                                                                   
// Revision 1.3  1999/10/04 03:22:42  billyb                                                                   
// Added comments.  Removed usage of m_variant_cv.                                                                   
/////////////////////////////////////////////////////////////////////////////
#include "..\core\stdafx.h"
#include "SliderWidget.h"

#include "SliderPage.h"
#include "..\core\IOAction.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif


/////////////////////////////////////////////////////////////////////////////
//
// CSliderWidget::CSliderWidget()
//
// Inputs           : None.
//
// Return Values    : None.
//
// Date             : 03 October 1999
//
// Engineer         : Billy Baker
//
// Description      : Default constructor.  Set the name of the widget which
//                    is used in the XML data and under the icons in the 
//                    editor.  Initialize other member variables.
//
/////////////////////////////////////////////////////////////////////////////
CSliderWidget::CSliderWidget()
{
    m_stlStrWidgetName      = _FSI_STL::string("Slider");

    m_listWidgetPropPages.push_back(CSliderPage::CreateObject);

    m_listGraphicalElementVars.push_back("Current Value");

    CSliderWidget::InitReinit();

    m_exPtUpperLeft         =  CExtentsPoint(CPoint(0,0));
    m_exPtLowerRight        =  CExtentsPoint(CPoint(200,128));

}

void CSliderWidget::InitReinit()
{
    m_lOptions              = HORIZONTAL | BOTH;

    m_varSliding.Value((float)0.0);
    m_varSliding_cv         = m_varSliding;

    m_ucNumberOfTics        = 6;

    m_bShowTics             = true;
    m_bDarkTics             = true;
}

/////////////////////////////////////////////////////////////////////////////
//
// CSliderWidget::~CSliderWidget()
//
// Inputs           : None.
//
// Return Values    : None.
//
// Date             : 03 October 1999
//
// Engineer         : Billy Baker
//
// Description      : Default destructor.
//
/////////////////////////////////////////////////////////////////////////////
CSliderWidget::~CSliderWidget()
{
}


BEGIN_MESSAGE_MAP(CSliderWidget, CButtonWidget)
	//{{AFX_MSG_MAP(CSliderWidget)
	ON_WM_LBUTTONDOWN()
	ON_WM_LBUTTONUP()
	ON_WM_MOUSEMOVE()
	//}}AFX_MSG_MAP
    ON_MESSAGE(WM_MOUSELEAVE, OnMouseLeave)
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
//
// CWidget* CSliderWidget::CreateObject()
//
// Inputs           : None.
//
// Return Values    : A pointer to a new instance of CSliderWidget.
//
// Date             : 03 October 1999
//
// Engineer         : Billy Baker
//
// Description      : CSliderWidget::CreateObject() is a common 
//                    framework method used to create a new instance of 
//                    CSliderWidget.  It is a static member.  The 
//                    address of this method is stored in a map of 
//                    widget name to CreateObject address in the 
//                    component interface for the DLL that contains the 
//                    CSliderWidget.  Using this data structure, a 
//                    new instance of CSliderWidget can be created 
//                    when one is encountered in the XML page 
//                    description file.
//
/////////////////////////////////////////////////////////////////////////////
CWidget* CSliderWidget::CreateObject()
{
    return new CSliderWidget();
}

/////////////////////////////////////////////////////////////////////////////
//
// void CSliderWidget::Initialize()
//
// Inputs           : CXMLWidget*& rpXMLWidget - reference to a pointer 
//                                               that holds the data to 
//                                               create a CSliderWidget.
//                    CWnd* pWnd               - the parent CWnd object.
//                    const long lId           - identifier.
//                    bool bEditMode           - whether editing is taking 
//                                               place.
//
// Return Values    : None.
//
// Date             : 03 October 1999
//
// Engineer         : Billy Baker
//
// Description      : CSliderWidget::Initialize() is a common 
//                    framework method.  It is called after an instance 
//                    of CSliderWidget has been created.  It is 
//                    responsible for initializing most of the internal 
//                    data members with data from the XMLWidget.  This 
//                    is done using the call to ResetProperties.  If 
//                    the XMLWidget pointer is NULL, a new XMLWidget 
//                    instance is created with default settings.  
//                    Generic help text is set here and any CWnd 
//                    derived Create() calls are made to create the 
//                    Windows object that allows for the use of the MFC 
//                    messages.  The HWND for the CWnd derived object 
//                    that is created is then added to a map in CWidget 
//                    that keeps track of what CWidget derived pointers 
//                    are valid.  This is to make sure that all of the 
//                    threads do not try to dereference bad pointers.
//
/////////////////////////////////////////////////////////////////////////////
void CSliderWidget::Initialize(CXMLWidget*& rpXMLWidget, CWnd* pWnd, 
                               const long lId, bool bEditMode)
{
    CWidget::Initialize(rpXMLWidget,pWnd, lId, bEditMode);

    CSliderWidget::InitReinit();

    if (rpXMLWidget != NULL)
    {
        ResetProperties();
    }

    if (GetSafeHwnd() == NULL)
    {
        Create("",WS_CHILD | WS_VISIBLE | WS_CLIPCHILDREN | BS_OWNERDRAW | CS_OWNDC, 
               CRect(m_exPtUpperLeft, m_exPtLowerRight),
               (CWnd*)pWnd, lId);
    }
    else
    {
        SetParent(pWnd);
        MoveWindow(m_exPtUpperLeft.X(), m_exPtUpperLeft.Y(),
                   m_exPtLowerRight.X() - m_exPtUpperLeft.X(),
                   m_exPtLowerRight.Y() - m_exPtUpperLeft.Y(), FALSE);
        ShowWindow(SW_NORMAL);
    }

    m_pWnd = this;

    // For use with CWidget::IsValidAddress().
    m_mapValidAddresses[this] = m_pWnd->GetSafeHwnd();
}

/////////////////////////////////////////////////////////////////////////////
//
// void CSliderWidget::ResetProperties()
//
// Inputs           : None.
//
// Return Values    : None.
//
// Date             : 03 October 1999
//
// Engineer         : Billy Baker
//
// Description      : CSliderWidget::ResetProperties() is a common 
//                    framework method.  It is called whenever the XML 
//                    in the CXMLWidget needs to be parsed for data 
//                    used by the SliderWidget.
//
/////////////////////////////////////////////////////////////////////////////
void CSliderWidget::ResetProperties()
{
    CButtonWidget::ResetProperties();

    // Find the OPTIONS element and get the options data.
    CXMLElement* pXMLElement = NULL;
    POSITION pos = NULL;
    if (m_pXMLWidget->FindElement(pXMLElement, pos, 
                                  _FSI_STL::string("OPTIONS")) == true)
    {
        m_lOptions = atoi(pXMLElement->ElementValue().c_str());

        STRING2STRING_MAP::iterator s2sIt;
        if (pXMLElement->FindAttribute(s2sIt,
                                       _FSI_STL::string("SIZE")) == true)
        {
            m_ucNumberOfTics = atoi((*s2sIt).second.c_str());
        }
        else
        {
            CString strValue;
            strValue.Format("%d",m_ucNumberOfTics);
            pXMLElement->AddAttribute(_FSI_STL::string("SIZE"),
                                      _FSI_STL::string((LPCTSTR)strValue));
        }

        if (pXMLElement->FindAttribute(s2sIt,
                                       _FSI_STL::string("SHOW_TICS")) == true)
        {
            CString strValue ((*s2sIt).second.c_str());
            strValue.MakeUpper();
            if (strValue == "YES")
            {
                m_bShowTics = true;
            }
            else
            {
                m_bShowTics = false;
            }
        }
        else
        {
            m_bShowTics = true;
            pXMLElement->AddAttribute(_FSI_STL::string("SHOW_TICS"),
                                      _FSI_STL::string("YES"));
        }

        if (pXMLElement->FindAttribute(s2sIt,
                                       _FSI_STL::string("DARK_TICS")) == true)
        {
            CString strValue ((*s2sIt).second.c_str());
            strValue.MakeUpper();
            if (strValue == "YES")
            {
                m_bDarkTics = true;
            }
            else
            {
                m_bDarkTics = false;
            }
        }
        else
        {
            m_bDarkTics = true;
            pXMLElement->AddAttribute(_FSI_STL::string("DARK_TICS"),
                                      _FSI_STL::string("YES"));
        }
    }
    else
    {
        m_bShowTics = true;
        m_ucNumberOfTics = 6;
        m_bDarkTics = true;

        // No element was found.  Add an OPTIONS element and set it to
        // the default options.
        CString strOptions;
        strOptions.Format("%d",m_lOptions);
        m_pXMLWidget->AddElement(_FSI_STL::string("OPTIONS"), 
                                 _FSI_STL::string((LPCTSTR)strOptions),
                                 NULL);

        pXMLElement = NULL;
        if (m_pXMLWidget->FindElement(pXMLElement, pos, 
                                      _FSI_STL::string("OPTIONS")) == true)
        {
            pXMLElement->AddAttribute(_FSI_STL::string("SHOW_TICS"),
                                      _FSI_STL::string("YES"));

            pXMLElement->AddAttribute(_FSI_STL::string("SIZE"),
                                      _FSI_STL::string("6"));

            pXMLElement->AddAttribute(_FSI_STL::string("DARK_TICS"),
                                      _FSI_STL::string("YES"));
        }
    }
}

/////////////////////////////////////////////////////////////////////////////
//
// void CSliderWidget::OnLButtonDown()
//
// Inputs           : UINT nFlags  - flags that tell what other keys may 
//                                   have been pressed when the click 
//                                   occurred.
//                    CPoint point - the position of the click.
//
// Return Values    : None.
//
// Date             : 03 October 1999
//
// Engineer         : Billy Baker
//
// Description      : CSliderWidget::OnLButtonDown() is an override of 
//                    the standard MFC method for CWnd derived classes. 
//                    Here, it is used to showing a tracking rectangle in
//                    the editor or to show any focus or push down states
//                    in the runtime.
//
/////////////////////////////////////////////////////////////////////////////
void CSliderWidget::OnLButtonDown(UINT nFlags, CPoint point) 
{
    if (m_bEnabled == false)
    {
        return;
    }

    if (m_bEditing == true)
    {
        CWidget::Track(point);
    }
    else
    {
        m_bRedraw = false;

//        if (m_rectPointer.PtInRect(point) == TRUE || 
//            m_rectChannel.PtInRect(point) == TRUE)
        {
	        CButtonWidget::OnLButtonDown(nFlags, point);
        }
/*
        else
        {
            if (m_bFocus != false || m_bLButtonDown != false)
            {
                m_bFocus = false;
                m_bLButtonDown = false;
                m_bRedraw = true;
            }
        }
*/
        if (m_bRedraw == true)
        {
            Invalidate(FALSE);
        }
    }
}

/////////////////////////////////////////////////////////////////////////////
//
// void CSliderWidget::OnLButtonUp()
//
// Inputs           : UINT nFlags  - flags that tell what other keys may 
//                                   have been pressed when the click 
//                                   occurred.
//                    CPoint point - the position of the click.
//
// Return Values    : None.
//
// Date             : 03 October 1999
//
// Engineer         : Billy Baker
//
// Description      : CSliderWidget::OnLButtonUp() is an override of 
//                    the standard MFC method for CWnd derived classes. 
//                    Here, it is used to get the state of the 
//                    Slider and to tell all comms actions to update 
//                    via the call to CButtonWidget::OnLButtonUp().
//
/////////////////////////////////////////////////////////////////////////////
void CSliderWidget::OnLButtonUp(UINT nFlags, CPoint point) 
{
    if (m_bEnabled == false)
    {
        return;
    }

    if (m_bEditing == true)
    {
    }
    else
    {
        // Create a filled region.
        m_bRedraw = false;

//        if (m_rectPointer.PtInRect(point) == TRUE ||
//            m_rectChannel.PtInRect(point) == TRUE)
        {
            float fRelativePos;
            if ((m_lOptions & VERTICAL) != 0)
            {
                fRelativePos = (1.0 * (m_rectChannel.bottom - point.y)) / 
                                                       m_rectChannel.Height();

                if (fRelativePos < 0.0f) 
                {
                    fRelativePos = 0.0f;
                }
                else if (fRelativePos > 1.0f)
                {
                    fRelativePos = 1.0f;
                }
            }
            else
            {
                fRelativePos = (1.0 * (point.x - m_rectChannel.left)) / 
                                                        m_rectChannel.Width();

                if (fRelativePos < 0.0f) 
                {
                    fRelativePos = 0.0f;
                }
                else if (fRelativePos > 1.0f)
                {
                    fRelativePos = 1.0f;
                }
            }

            RangeData rdDefault = m_mapRangeData["Default"];
            float fChannelPos = fRelativePos * 
                                 ((float)rdDefault.s_varHigh - (float)rdDefault.s_varLow) + 
                                 (float)rdDefault.s_varLow;

            m_variant.Value(fChannelPos);

  	        CButtonWidget::OnLButtonUp(nFlags, point);
        }
/*
        else
        {
            CButton::OnLButtonUp(nFlags, point);
            if (m_bFocus != false || m_bLButtonDown != false)
            {
                m_bFocus = false;
                m_bLButtonDown = false;
                m_bRedraw = true;
            }
        }
*/
        if (m_bRedraw == true)
        {
            Invalidate(FALSE);
        }
    }
}

/////////////////////////////////////////////////////////////////////////////
//
// void CSliderWidget::DrawItem()
//
// Inputs           : LPDRAWITEMSTRUCT lpDrawItemStruct - pointer to a 
//                                                        Win32 structure
//                    .
//
// Return Values    : None.
//
// Date             : 03 October 1999
//
// Engineer         : Billy Baker
//
// Description      : CSliderWidget::DrawItem() is an override of a 
//                    standard MFC method for use with owner drawn 
//                    controls.  It is overriden here to allow the 
//                    background to be whatever the user wants and make the
//                    channel and pointer resize as the graphical element's
//                    size changes.
//
/////////////////////////////////////////////////////////////////////////////
void CSliderWidget::DrawItem(LPDRAWITEMSTRUCT lpDrawItemStruct) 
{
    CRect rect(&(lpDrawItemStruct->rcItem));

    CDC *pDC = CDC::FromHandle(lpDrawItemStruct->hDC);

    CDC* pDCur = pDC;
    pDC = m_pDC;
    pDC->CreateCompatibleDC(pDCur);

    long int lSavedDC = pDC->SaveDC();

    m_pBMP->CreateCompatibleBitmap(pDCur, m_exPtLowerRight.X() - m_exPtUpperLeft.X(),
                                    m_exPtLowerRight.Y() - m_exPtUpperLeft.Y());
    pDC->SelectObject(m_pBMP);

    CColor colorBackground = Background();
    pDC->FillSolidRect(rect, (COLORREF)colorBackground);

    if ((m_lOptions & VERTICAL) != 0)
    {
        DrawVerticalSlider(pDC, rect);
    }
    else
    {
        DrawHorizontalSlider(pDC, rect);
    }

    pDCur->BitBlt(0, 0, rect.Width(), rect.Height(), pDC, 0, 0, SRCCOPY);
    pDC->RestoreDC(lSavedDC);

    m_pDC->DeleteDC();
    m_pBMP->DeleteObject();

    // Move and invalidate the children for the different states.
    if (m_bLButtonDown != m_bLButtonDownLastPass)
    {
        m_bLButtonDownLastPass = m_bLButtonDown;

        // Make sure that only the base widget updates all of the children.
        if (m_bTraverseUpdate == false)
        {
            TraverseBaseWidgets((CWidget*)this, true);
        }
    }
    else
    {
        if (m_bTraverseUpdate == false)
        {
            TraverseBaseWidgets((CWidget*)this, false);
        }
    }

    m_bTraverseUpdate = false;
}

/////////////////////////////////////////////////////////////////////////////
//
// void CSliderWidget::DrawHorizontalSlider()
//
// Inputs           : CDC* pDC    - pointer to the device context in which 
//                                  to draw.
//                    CRect& rect - the area in which to draw.
//
// Return Values    : None.
//
// Date             : 03 October 1999
//
// Engineer         : Billy Baker
//
// Description      : CSliderWidget::DrawHorizontalSlider() is a helper 
//                    method for DrawItem.  It draws a Horizontal slider 
//                    in the area that is passed to this method.
//
/////////////////////////////////////////////////////////////////////////////
void CSliderWidget::DrawHorizontalSlider(CDC* pDC, CRect& rect)
{
    CRect rectChannel = rect;

    // Draw upper channel highlight
    CPen pen1(PS_SOLID, 1, Background().DarkColor());
    pDC->SelectObject(&pen1);
    pDC->MoveTo(CPoint(rect.left, rect.bottom));
    pDC->LineTo(CPoint(rect.left, rect.top));
    pDC->LineTo(CPoint(rect.right, rect.top));

    // Draw black inner channel highlight
    CPen pen2(PS_SOLID, 1, GetSysColor(COLOR_3DDKSHADOW));
    pDC->SelectObject(&pen2);
    pDC->MoveTo(CPoint(rect.left+1, rect.bottom));
    pDC->LineTo(CPoint(rect.left+1, rect.top+1));
    pDC->LineTo(CPoint(rect.right-2, rect.top+1));

    // Draw lower channel highlight
    CPen pen3(PS_SOLID, 1, Background().HighlightColor());
    pDC->SelectObject(&pen3);
    pDC->MoveTo(CPoint(rect.left, rect.bottom-1));
    pDC->LineTo(CPoint(rect.right-1, rect.bottom-1));
    pDC->MoveTo(CPoint(rect.right-1, rect.bottom-1));
    pDC->LineTo(CPoint(rect.right-1, rect.top-1));

    rectChannel.top = rect.top + (rect.Height()*(6.0/16.0) + 0.5);
    rectChannel.bottom = rectChannel.top + (rect.Height()*(4.0/16.0) + 0.5);
    rectChannel.left = rect.left + (rect.Width()/10.0 + 0.5);
    rectChannel.right = rect.right - (rect.Width()/10.0 + 0.5);
    m_rectChannel = rectChannel;

    if (!m_bEnabled)
    {
        pDC->FillSolidRect(rectChannel, (COLORREF)(Background().HighlightColor()));
    }
    else
    {

        pDC->FillSolidRect(rectChannel, (COLORREF)Background());

   }

    // Draw upper channel highlight
    pDC->SelectObject(&pen1);
    pDC->MoveTo(CPoint(rectChannel.left, rectChannel.bottom));
    pDC->LineTo(CPoint(rectChannel.left, rectChannel.top));
    pDC->LineTo(CPoint(rectChannel.right, rectChannel.top));

    // Draw black inner channel highlight
    pDC->SelectObject(&pen2);
    pDC->MoveTo(CPoint(rectChannel.left+1, rectChannel.top+1));
    pDC->LineTo(CPoint(rectChannel.right-1, rectChannel.top+1));

    // Draw lower channel highlight
    pDC->SelectObject(&pen3);
    pDC->MoveTo(CPoint(rectChannel.left, rectChannel.bottom));
    pDC->LineTo(CPoint(rectChannel.right, rectChannel.bottom));
    pDC->MoveTo(CPoint(rectChannel.right, rectChannel.bottom));
    pDC->LineTo(CPoint(rectChannel.right, rectChannel.top-1));

    CRect rectPointer = rectChannel;
    
    if (m_bShowTics == true)
    {
        if (m_bDarkTics == true)
        {
            pDC->SelectObject(&pen1);
        }
        else
        {
            pDC->SelectObject(&pen3);
        }
    
        int nOffsetFourth       = rect.Height() * (4.0/16.0) + 0.5;
        int nOffsetSixteenth    = rect.Height() * (1.0/16.0) + 0.5;
        int starty              = rect.top + nOffsetSixteenth;
        int startx              = rectChannel.left;
        int nStartXOffset       = (1.0 * rectChannel.Width()) / 
                                  (1.0 * (m_ucNumberOfTics-1));
        int i                   = 0;
        while (i < m_ucNumberOfTics)
        {
            if ((m_lOptions & BOTH) != 0 || (m_lOptions & LEFT_UP) != 0)
            {
                pDC->MoveTo(startx, starty);
                pDC->LineTo(startx, starty + nOffsetFourth);
            }

            if ((m_lOptions & BOTH) != 0 || (m_lOptions & RIGHT_DOWN) != 0)
            {
                pDC->MoveTo(startx, rectChannel.bottom + nOffsetSixteenth);
                pDC->LineTo(startx, rectChannel.bottom + nOffsetSixteenth + 
                                    nOffsetFourth);
            }

            startx += nStartXOffset;
            i++;
        }
    }

    RangeData rdDefault = m_mapRangeData["Default"];
    float fPosRatio = ((float)m_varSliding - (float)rdDefault.s_varLow) /
                      ((float)rdDefault.s_varHigh - (float)rdDefault.s_varLow);

    long int lAspectSize = 0;
    if (rectChannel.Width() <= rectChannel.Height())
    {
        lAspectSize = (rectChannel.Width() * .75 + 0.5);
    }
    else
    {
        lAspectSize = (rectChannel.Height() * .75 + 0.5);
    }

    if ((lAspectSize & 0x01) != 0)
    {
        lAspectSize++;
    }

    if ((m_lOptions & BOTH) != 0)
    {
        long int lPos = rectChannel.Width() * fPosRatio;
        rectPointer.left = lPos + rectChannel.left - (2.0*lAspectSize/3.0 + 0.5);
        rectPointer.right = rectPointer.left + (4.0*lAspectSize/3.0 + 0.5);
        rectPointer.top = rectChannel.top - lAspectSize;
        rectPointer.bottom = rectChannel.bottom + lAspectSize;

        pDC->FillSolidRect(rectPointer, (COLORREF)Background());

        // Draw upper channel highlight
        pDC->SelectObject(&pen3);
        pDC->MoveTo(CPoint(rectPointer.left, rectPointer.bottom));
        pDC->LineTo(CPoint(rectPointer.left, rectPointer.top));
        pDC->LineTo(CPoint(rectPointer.right, rectPointer.top));

        // Draw black inner Pointer highlight
        pDC->SelectObject(&pen1);
        pDC->MoveTo(CPoint(rectPointer.left+1, rectPointer.bottom-1));
        pDC->LineTo(CPoint(rectPointer.right-1, rectPointer.bottom-1));
        pDC->MoveTo(CPoint(rectPointer.right-1, rectPointer.bottom-1));
        pDC->LineTo(CPoint(rectPointer.right-1, rectPointer.top));

        // Draw lower Pointer highlight
        pDC->SelectObject(&pen2);
        pDC->MoveTo(CPoint(rectPointer.left, rectPointer.bottom));
        pDC->LineTo(CPoint(rectPointer.right, rectPointer.bottom));
        pDC->MoveTo(CPoint(rectPointer.right, rectPointer.bottom));
        pDC->LineTo(CPoint(rectPointer.right, rectPointer.top-1));
    }
    else if ((m_lOptions & LEFT_UP) != 0)
    {
        long int lPos = rectChannel.Width() * fPosRatio;
        rectPointer.left = lPos + rectChannel.left - (2.0*lAspectSize/3.0 + 0.5);
        rectPointer.right = rectPointer.left + (4.0*lAspectSize/3.0 + 0.5);
        rectPointer.top = rectChannel.top - lAspectSize - (rectPointer.Width()/2.0 + 0.5);
        rectPointer.bottom = rectChannel.bottom + lAspectSize;

        pDC->FillSolidRect(rectPointer, (COLORREF)Background());

        // Draw upper channel highlight
        pDC->SelectObject(&pen3);
        pDC->MoveTo(rectPointer.left, rectPointer.bottom);
        pDC->LineTo(rectPointer.left, rectPointer.top + (rectPointer.Width()/2.0 + 0.5));
        pDC->LineTo(rectPointer.left + (rectPointer.Width()/2.0 + 0.5), rectPointer.top);

        // Draw black inner Pointer highlight
        pDC->SelectObject(&pen1);
        pDC->MoveTo(rectPointer.left+1, rectPointer.bottom-1);
        pDC->LineTo(rectPointer.right-1, rectPointer.bottom-1);
        pDC->MoveTo(rectPointer.right-1, rectPointer.bottom-1);
        pDC->LineTo(rectPointer.right-1, rectPointer.top + (rectPointer.Width()/2.0 + 0.5));
        pDC->LineTo(rectPointer.left + (rectPointer.Width()/2.0 + 0.5) - 1, rectPointer.top);

        // Draw lower Pointer highlight
        pDC->SelectObject(&pen2);
        pDC->MoveTo(rectPointer.left, rectPointer.bottom);
        pDC->LineTo(rectPointer.right, rectPointer.bottom);
        pDC->MoveTo(rectPointer.right, rectPointer.bottom);
        pDC->LineTo(rectPointer.right, rectPointer.top + (rectPointer.Width()/2.0 + 0.5));
        pDC->LineTo(rectPointer.left + (rectPointer.Width()/2.0 + 0.5) - 1, rectPointer.top-1);
    }
    else 
    {
        long int lPos = rectChannel.Width() * fPosRatio;
        rectPointer.left = lPos + rectChannel.left - (2.0*lAspectSize/3.0 + 0.5);
        rectPointer.right = rectPointer.left + (4.0*lAspectSize/3.0 + 0.5);
        rectPointer.top = rectChannel.top - lAspectSize;
        rectPointer.bottom = rectChannel.bottom + lAspectSize + (rectPointer.Width()/2.0 + 0.5);

          pDC->FillSolidRect(rectPointer, (COLORREF)Background());

        // Draw upper channel highlight
        pDC->SelectObject(&pen3);
        pDC->MoveTo(rectPointer.left + (rectPointer.Width()/2.0 + 0.5), rectPointer.bottom);
        pDC->LineTo(rectPointer.left, rectPointer.bottom - (rectPointer.Width()/2.0 + 0.5) + 1);
        pDC->LineTo(rectPointer.left, rectPointer.top);
        pDC->LineTo(rectPointer.right, rectPointer.top);

        // Draw black inner Pointer highlight
        pDC->SelectObject(&pen1);
        pDC->MoveTo(rectPointer.left + (rectPointer.Width()/2.0 + 0.5), rectPointer.bottom - 1);
        pDC->LineTo(rectPointer.right-1, rectPointer.bottom - (rectPointer.Width()/2.0 + 0.5) + 1);
        pDC->MoveTo(rectPointer.right-1, rectPointer.bottom - (rectPointer.Width()/2.0 + 0.5) + 1);
        pDC->LineTo(rectPointer.right-1, rectPointer.top);

        // Draw lower Pointer highlight
        pDC->SelectObject(&pen2);
        pDC->MoveTo(rectPointer.left + (rectPointer.Width()/2.0 + 0.5), rectPointer.bottom);
        pDC->LineTo(rectPointer.right, rectPointer.bottom - (rectPointer.Width()/2.0 + 0.5) + 1);
        pDC->MoveTo(rectPointer.right, rectPointer.bottom - (rectPointer.Width()/2.0 + 0.5) + 1);
        pDC->LineTo(rectPointer.right, rectPointer.top - 1);
    }

    m_rectPointer = rectPointer;
}

/////////////////////////////////////////////////////////////////////////////
//
// void CSliderWidget::DrawVerticalSlider()
//
// Inputs           : CDC* pDC    - pointer to the device context in which 
//                                  to draw.
//                    CRect& rect - the area in which to draw.
//
// Return Values    : None.
//
// Date             : 03 October 1999
//
// Engineer         : Billy Baker
//
// Description      : CSliderWidget::DrawVerticalSlider() is a helper 
//                    method for DrawItem.  It draws a vertical slider 
//                    in the area that is passed to this method.
//
/////////////////////////////////////////////////////////////////////////////
void CSliderWidget::DrawVerticalSlider(CDC* pDC, CRect& rect)
{
    CRect rectChannel = rect;
    CColor colorBackground = Background();

    // Draw upper highlight
    CPen pen1(PS_SOLID, 1, colorBackground.DarkColor());
    pDC->SelectObject(&pen1);
    CPoint pt(rect.left, rect.bottom);
    pDC->MoveTo(pt);
    pt.y = rect.top;
    pDC->LineTo(pt);
    pt.x = rect.right;
    pDC->LineTo(CPoint(rect.right, rect.top));

    // Draw black inner highlight
    CPen pen2(PS_SOLID, 1, GetSysColor(COLOR_3DDKSHADOW));
    pDC->SelectObject(&pen2);
    pt.x = rect.left + 1;
    pt.y = rect.bottom - 2;
    pDC->MoveTo(pt);
    pt.y = rect.top + 1;
    pDC->LineTo(pt);
    pt.x = rect.right - 2;
    pDC->LineTo(pt);

    // Draw lower highlight
    CPen pen3(PS_SOLID, 1, colorBackground.HighlightColor());
    pDC->SelectObject(&pen3);
    pt.x = rect.left;
    pt.y = rect.bottom - 1;
    pDC->MoveTo(pt);
    pt.x = rect.right;
    pDC->LineTo(pt);
    pt.x = rect.right - 1;
    pDC->LineTo(pt);
    pt.y = rect.top - 1;
    pDC->LineTo(pt);


    rectChannel.top = rect.top + (rect.Height()/10.0f + 0.5f);
    rectChannel.bottom = rect.bottom - (rect.Height()/10.0f + 0.5f);
    rectChannel.left = rect.left + (rect.Width()*0.375f + 0.5f);
    rectChannel.right = rectChannel.left + (rect.Width()*0.25f + 0.5f);
    m_rectChannel = rectChannel;

    if (!m_bEnabled)
    {
        pDC->FillSolidRect(rectChannel, (COLORREF)(colorBackground.HighlightColor()));
    }
    else
    {
        pDC->FillSolidRect(rectChannel, (COLORREF)colorBackground);
    }

    // Draw upper channel highlight
    pDC->SelectObject(&pen1);
    pt.x = rectChannel.left;
    pt.y = rectChannel.bottom;
    pDC->MoveTo(pt);
    pt.y = rectChannel.top;
    pDC->LineTo(pt);
    pt.x = rectChannel.right;
    pDC->LineTo(pt);

    // Draw black inner channel highlight
    pDC->SelectObject(&pen2);
    pt.x = rectChannel.left + 1;
    pt.y = rectChannel.bottom - 2;
    pDC->MoveTo(pt);
    pt.y = rectChannel.top;
    pDC->LineTo(pt);

    // Draw lower channel highlight
    pDC->SelectObject(&pen3);
    pt.x = rectChannel.left;
    pt.y = rectChannel.bottom;
    pDC->MoveTo(pt);
    pt.x = rectChannel.right;
    pDC->LineTo(pt);
    pDC->LineTo(pt);
    pt.y = rectChannel.top - 1;
    pDC->LineTo(pt);

    if (m_bShowTics == true)
    {
        if (m_bDarkTics == true)
        {
            pDC->SelectObject(&pen1);
        }
        else
        {
            pDC->SelectObject(&pen3);
        }

        int nOffsetFourth       = rect.Width() * .25f + 0.5f;
        int nOffsetSixteenth    = rect.Width() * .0625f + 0.5f;
        int starty              = rectChannel.top;
        int startx              = rect.left + nOffsetSixteenth;
        int nStartYOffset       = (1.0 * rectChannel.Height()) / 
                                  (1.0 * (m_ucNumberOfTics-1));
        int i                   = 0;
        while (i < m_ucNumberOfTics)
        {
            if ((m_lOptions & BOTH) != 0 || (m_lOptions & LEFT_UP) != 0)
            {
                pDC->MoveTo(startx, starty);
                pDC->LineTo(startx + nOffsetFourth, starty);
            }

            if ((m_lOptions & BOTH) != 0 || (m_lOptions & RIGHT_DOWN) != 0)
            {
                pDC->MoveTo(rectChannel.right + nOffsetSixteenth, starty);
                pDC->LineTo(rectChannel.right + nOffsetSixteenth + nOffsetFourth,
                            starty);
            }

            starty += nStartYOffset;
            i++;
        }
    }

    CRect rectPointer = rectChannel;
    RangeData rdDefault = m_mapRangeData["Default"];
    float fPosRatio = ((float)m_varSliding - (float)rdDefault.s_varLow) /
                      ((float)rdDefault.s_varHigh - (float)rdDefault.s_varLow);

    long int lAspectSize = 0;
    if (rectChannel.Width() <= rectChannel.Height())
    {
        lAspectSize = rectChannel.Width() * .75f + 0.5f;
    }
    else
    {
        lAspectSize = rectChannel.Height() * .75f + 0.5f;
    }

    if ((lAspectSize & 0x01) != 0)
    {
        lAspectSize++;
    }

    if ((m_lOptions & BOTH) != 0)
    {
        long int lPos = rectChannel.Height() * fPosRatio;
        rectPointer.left = rectChannel.left - lAspectSize;
        rectPointer.right = rectChannel.right + lAspectSize;
        rectPointer.top = rectChannel.bottom - lPos - (2.0f/3.0f * lAspectSize + 0.5f);
        rectPointer.bottom = rectPointer.top + (4.0f/3.0f * lAspectSize + 0.5f);

        pDC->FillSolidRect(rectPointer, (COLORREF)colorBackground);

        // Draw upper channel highlight
        pDC->SelectObject(&pen3);
        pt.x = rectPointer.left;
        pt.y = rectPointer.bottom;
        pDC->MoveTo(pt);
        pt.y = rectPointer.top;
        pDC->LineTo(pt);
        pt.x = rectPointer.right;
        pDC->LineTo(pt);

        // Draw black inner Pointer highlight
        pDC->SelectObject(&pen1);
        pt.x = rectPointer.left + 1;
        pt.y = rectPointer.bottom - 1;
        pDC->MoveTo(pt);
        pt.x = rectPointer.right - 1;
        pDC->LineTo(pt);
        pDC->MoveTo(pt);
        pt.y = rectPointer.top;
        pDC->LineTo(pt);

        // Draw lower Pointer highlight
        pDC->SelectObject(&pen2);
        pt.x = rectPointer.left;
        pt.y = rectPointer.bottom;
        pDC->MoveTo(pt);
        pt.x = rectPointer.right;
        pDC->LineTo(pt);
        pDC->MoveTo(pt);
        pt.y = rectPointer.top - 1;
        pDC->LineTo(pt);
    }
    else if ((m_lOptions & LEFT_UP) != 0)
    {
        long int lPos = rectChannel.Height() * fPosRatio;
        rectPointer.top = rectChannel.bottom - lPos - (2.0f/3.0f * lAspectSize + 0.5f);
        rectPointer.bottom = rectPointer.top + (4.0f/3.0f * lAspectSize + 0.5f);
        rectPointer.left = rectChannel.left - lAspectSize - (lAspectSize * 0.5f + 0.5f);
        rectPointer.right = rectChannel.right + lAspectSize;

        pDC->FillSolidRect(rectPointer, (COLORREF)colorBackground);

        // Draw upper Pointer highlight
        pDC->SelectObject(&pen3);
        pDC->MoveTo(rectPointer.left, rectPointer.top + (rectPointer.Height() * 0.5f + 0.5f));
        pDC->LineTo(rectPointer.left + (rectPointer.Height() * 0.5f + 0.5f), rectPointer.top);
        pDC->LineTo(rectPointer.right, rectPointer.top);

        // Draw inner Pointer highlight
        pDC->SelectObject(&pen1);
        pDC->MoveTo(rectPointer.left + 1, rectPointer.top + (rectPointer.Height() * 0.5f + 0.5f));
        pDC->LineTo(rectPointer.left + (rectPointer.Height() * 0.5f + 0.5f), rectPointer.bottom - 1);
        pDC->LineTo(rectPointer.right - 1, rectPointer.bottom - 1);
        pDC->LineTo(rectPointer.right - 1, rectPointer.top);

        // Draw lower black Pointer highlight
        pDC->SelectObject(&pen2);
        pDC->MoveTo(rectPointer.left, rectPointer.top + (rectPointer.Height() * 0.5f + 0.5f));
        pDC->LineTo(rectPointer.left + (rectPointer.Height() * 0.5f + 0.5f), rectPointer.bottom);
        pDC->LineTo(rectPointer.right, rectPointer.bottom);
        pDC->LineTo(rectPointer.right, rectPointer.top - 1);
    }
    else 
    {
        long int lPos = rectChannel.Height() * fPosRatio;
        rectPointer.top = rectChannel.bottom - lPos - (2.0f/3.0f * lAspectSize + 0.5f);
        rectPointer.bottom = rectPointer.top + (4.0f/3.0f * lAspectSize + 0.5f);
        rectPointer.left = rectChannel.left - lAspectSize;
        rectPointer.right = rectChannel.right + lAspectSize + (lAspectSize * 0.5f + 0.5f);

        pDC->FillSolidRect(rectPointer, (COLORREF)colorBackground);

        // Draw upper Pointer highlight
        pDC->SelectObject(&pen3);
        pDC->MoveTo(rectPointer.left, rectPointer.bottom);
        pDC->LineTo(rectPointer.left, rectPointer.top);
        pDC->LineTo(rectPointer.right - (rectPointer.Height() * 0.5f + 0.5f) + 1, rectPointer.top);
        pDC->LineTo(rectPointer.right, rectPointer.top + (rectPointer.Height() * 0.5f + 0.5f));

        // Draw inner Pointer highlight
        pDC->SelectObject(&pen1);
        pDC->MoveTo(rectPointer.left, rectPointer.bottom - 1);
        pDC->LineTo(rectPointer.right - (rectPointer.Height()/2.0 + 0.5), rectPointer.bottom - 1);
        pDC->LineTo(rectPointer.right-1, rectPointer.bottom - (rectPointer.Height()/2.0 + 0.5));

        // Draw lower black Pointer highlight
        pDC->SelectObject(&pen2);
        pDC->MoveTo(rectPointer.left, rectPointer.bottom);
        pDC->LineTo(rectPointer.right - (rectPointer.Height()/2.0 + 0.5), rectPointer.bottom);
        pDC->LineTo(rectPointer.right, rectPointer.bottom - (rectPointer.Height()/2.0 + 0.5));
    }

    m_rectPointer = rectPointer;
}

/////////////////////////////////////////////////////////////////////////////
//
// void CSliderWidget::OnMouseMove()
//
// Inputs           : UINT nFlags  - flags that tell what other keys may 
//                                   have been pressed when the click 
//                                   occurred.
//                    CPoint point - the position of the click.
//
// Return Values    : None.
//
// Date             : 03 October 1999
//
// Engineer         : Billy Baker
//
// Description      : CSliderWidget::OnMouseMove() is an override of 
//                    the standard MFC method for CWnd derived classes. 
//                    Here, it is used to move the pointer in the slider
//                    with the mouse as it moves if the left mouse button
//                    is down.
//
/////////////////////////////////////////////////////////////////////////////
void CSliderWidget::OnMouseMove(UINT nFlags, CPoint point) 
{
    if (m_bEnabled == false)
    {
        return;
    }

    m_bRedraw = false;
    bool bInValidRange = false;

    if ((m_lOptions & VERTICAL) != 0)
    {
//        if (point.y >= m_rectChannel.top &&
//            point.y <= m_rectChannel.bottom &&
//            point.x >= m_rectPointer.left &&
//            point.x <= m_rectPointer.right)
        {
            bInValidRange = true;
        }
    }
    else
    {
//        if (point.x >= m_rectChannel.left &&
//            point.x <= m_rectChannel.right &&
//            point.y >= m_rectPointer.top &&
//            point.y <= m_rectPointer.bottom)
        {
            bInValidRange = true;
        }
    }

    if (bInValidRange)
    {
        CButtonWidget::OnMouseMove(nFlags, point);

        if (m_bLButtonDown == true)
        {
            float fRelativePos;
            if ((m_lOptions & VERTICAL) != 0)
            {
                fRelativePos = (1.0 * (m_rectChannel.bottom - point.y)) / 
                                                       m_rectChannel.Height();

                if (fRelativePos < 0.0f) 
                {
                    fRelativePos = 0.0f;
                }
                else if (fRelativePos > 1.0f)
                {
                    fRelativePos = 1.0f;
                }
            }
            else
            {
                fRelativePos = (1.0 * (point.x - m_rectChannel.left)) / 
                                                        m_rectChannel.Width();

                if (fRelativePos < 0.0f) 
                {
                    fRelativePos = 0.0f;
                }
                else if (fRelativePos > 1.0f)
                {
                    fRelativePos = 1.0f;
                }
            }

            RangeData rdDefault = m_mapRangeData["Default"];
            float fChannelPos = fRelativePos * 
                                 ((float)rdDefault.s_varHigh - (float)rdDefault.s_varLow) + 
                                 (float)rdDefault.s_varLow;

            m_variant.Value(fChannelPos);

            if (!(m_varSliding == m_variant))
            {
                m_varSliding = m_variant;
                Invalidate(FALSE);
            }

            // Update other widgets...
            _FSI_STL::list<CAction*>::iterator lIt = m_listActions.begin();
            for (; lIt != m_listActions.end(); lIt++)
            {
               CIOAction* pIOAction = dynamic_cast<CIOAction*>(*lIt);
               if (pIOAction != NULL)
               {
                  CString str(pIOAction->GraphicalElementName().c_str());
                  if (str  == "Current Value")
                  {
                     // Force a "click", which in turn will cause GetValue() to be called, which in turn will update the above variables.
                     pIOAction->OnLButtonUp();
                  }
               }
            }
        }
    }
    else
    {
        if (m_bFocus != false || m_bLButtonDown != false)
        {
            m_bFocus = false;
            m_bLButtonDown = false;
            m_bRedraw = true;
        }
    }

    if (m_bRedraw == true)
    {
        Invalidate(FALSE);
    }
}

/////////////////////////////////////////////////////////////////////////////
//
// long CSliderWidget::OnMouseLeave()
//
// Inputs           : UINT nFlags  - flags that tell what other keys may 
//                                   have been pressed when the click 
//                                   occurred.
//                    CPoint point - the position of the click.
//
// Return Values    : 0 - not used.
//
// Date             : 03 October 1999
//
// Engineer         : Billy Baker
//
// Description      : CSliderWidget::OnMouseLeave() is an override of 
//                    the standard MFC method for CWnd derived classes. 
//                    Here, it is used to keep track of the last value
//                    where the mouse was when it left the slider channel.
//
/////////////////////////////////////////////////////////////////////////////
long CSliderWidget::OnMouseLeave(WPARAM wParam, LPARAM lParam)
{
    if (m_bEnabled == false)
    {
        return 0;
    }

    CPoint ptCursor;
    GetCursorPos(&ptCursor);
    ScreenToClient(&ptCursor);
    CRect rect(m_exPtUpperLeft, m_exPtLowerRight);

//    if (m_rectPointer.PtInRect(ptCursor) != TRUE ||
//        m_rectChannel.PtInRect(ptCursor) != TRUE)
    {

        if (m_bLButtonDown)
        {
            // Update other widgets...
            _FSI_STL::list<CAction*>::iterator lIt = m_listActions.begin();
            for (; lIt != m_listActions.end(); lIt++)
            {
               CIOAction* pIOAction = dynamic_cast<CIOAction*>(*lIt);
               if (pIOAction != NULL)
               {
                  CString str(pIOAction->GraphicalElementName().c_str());
                  if (str  == "Current Value"   || str   == "Default")
                  {
                     // Force a "click", which in turn will cause GetValue() to be called, which in turn will update the above variables.
                     pIOAction->OnLButtonUp();
                  }
               }
            }
        }

        m_varSliding = m_varLastPosition;
        CButtonWidget::OnMouseLeave(wParam, lParam);
    }
/*
    else
    {
        TRACKMOUSEEVENT tme;  

        tme.cbSize      = sizeof(TRACKMOUSEEVENT); 
        tme.dwFlags     = TME_LEAVE; 
        tme.hwndTrack   = GetSafeHwnd();  
        TrackMouseEvent(&tme); 
    }
*/
    return 0;
}

/////////////////////////////////////////////////////////////////////////////
//
// CVariant* CSliderWidget::GetValue()
//
// Inputs           : CString& rstrValue - the name of the internal 
//                                         variable whose value is needed
//                    .
//
// Return Values    : A pointer to CVariant instance that holds the 
//                    value for the requested internal variable.
//
// Date             : 03 October 1999
//
// Engineer         : Billy Baker
//
// Description      : CSliderWidget::GetValue() is a common framework 
//                    method.  It is used when ever a user performs an 
//                    action like a button click or pressing a 
//                    return/enter key. These actions are performed on 
//                    the graphical element and signal that data needs 
//                    to go from the graphical element to somewhere 
//                    else.  Here, the somewhere else would either be a 
//                    local comms variable or a host variable.  Here, the
//                    value being sent is the status of the Slider.
//
/////////////////////////////////////////////////////////////////////////////
CVariant* CSliderWidget::GetValue(const CString &rstrValue)
{
   if (rstrValue  == "Current Value")
   {
      return   &m_variant;
   }
   else
      return   &m_variant;
}

//////////////////////////////////////////////////////////////////////
// Methods called from Comms thread
//////////////////////////////////////////////////////////////////////

/////////////////////////////////////////////////////////////////////////////
//
// void CSliderWidget::ChangeValue()
//
// Inputs           : CString& rstrElementVar - the internal name of a 
//                                              variable that needs to be 
//                                              updated.
//                    CChangeValue* pCV       - pointer to data used to 
//                                              update an internal 
//                                              variable.
//
// Return Values    : None.
//
// Date             : 03 October 1999
//
// Engineer         : Billy Baker
//
// Description      : CSlider::ChangeValue() is a common framework 
//                    method that is used by the Comms thread to send 
//                    new data to the graphical elements.  Since it is 
//                    called from another thread, the special internal 
//                    variables are used to store the updates.  Before 
//                    the graphical element is redrawn, 
//                    UpdateRenderVariables is called to copy the 
//                    updated values to the values used for rendering.  
//                    This causes more memory to be used but prevents 
//                    the use of synchronization objects that may stall 
//                    the Comms thread.
//
/////////////////////////////////////////////////////////////////////////////
void CSliderWidget::ChangeValue(const CString& rstrElementVar, CChangeValue* pCV)
{
    if (pCV == NULL)
    {
        return;
    }

    if (rstrElementVar == "Default")
    {
       if (!m_bLButtonDown)
           m_varSliding_cv.Value((float)*(pCV->Variant()));
    }
    else if (rstrElementVar  == CString("Current Value"))
    {
        m_varSliding_cv.Value((float)*(pCV->Variant()));
    }
    else
    {
        CWidget::ChangeValue(rstrElementVar, pCV);
    }
}

/////////////////////////////////////////////////////////////////////////////
//
// bool CSliderWidget::UpdateRenderVariables()
//
// Inputs           : None.
//
// Return Values    : true  - all of the render variables were updated.
//                    false - a problem occurred updating the render 
//                            variables.
//
// Date             : 03 October 1999
//
// Engineer         : Billy Baker
//
// Description      : CSliderWidget::UpdateRenderVariables is a 
//                    common framework method that is called by the 
//                    Comms thread to make sure that all of the 
//                    variables used for rendering have the most up to 
//                    date data.
//
/////////////////////////////////////////////////////////////////////////////
bool CSliderWidget::UpdateRenderVariables()
{

    m_varSliding = m_varSliding_cv;

    RangeData rdDefault = m_mapRangeData["Default"];

    if (m_varSliding > rdDefault.s_varHigh)
    {
        m_varSliding = rdDefault.s_varHigh;
    }
    else if (m_varSliding < rdDefault.s_varLow)
    {
        m_varSliding = rdDefault.s_varLow;
    }

    m_varLastPosition = m_varSliding;

    return CWidget::UpdateRenderVariables();
}