/////////////////////////////////////////////////////////////////////////////
//
//           F L I G H T S A F E T Y   I N T E R N A T I O N A L
//                     Simulation Systems Division
//                      2700 North Hemlock Circle
//                     Broken Arrow, Oklahoma 74012
//                          (918) 259-4000
/////////////////////////////////////////////////////////////////////////////
//
// DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
// Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
// technical or operational data or information from automatic dissemination
// under the International Exchange Program or by other means.  This protection
// covers information required solely for administrative or operational
// purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
//
// WARNING:  This document contains technical data whose export is restricted
// by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
// Executive Order 12470.  Violation of these export control laws is subject
// to severe criminal penalties.  Dissemination of this document is controlled
// under DOD Directive 5230.25
//
/////////////////////////////////////////////////////////////////////////////
//
//
// Filename         : plotwidget.cpp
//
// Date             : 21 January 2000
//
// Engineer         : Billy Baker
//
// Revision         : $Revision: 1.6 $
//
// Description      : PlotWidget.cpp contains the implementation of the 
//                    CPlotWidget class.  This class is used as a starting
//                    point for both the XY and Time Based Plot Widgets.
//                    As such, CPlotWidget contains the minimum code
//                    for a plot.  This includes the number of x and y
//                    divisions, the min and max for the x axis, etc.
//                    A number of virtual methods are defined which 
//                    deal with drawing the the tics and labels on each
//                    axis and drawing the inspection tool.
//
//                    PlotWidget.h also contains the implementation of the
//                    CPlotData class.  This class holds information that
//                    can be unique for each plot.
//
// Classification   : UNCLASSIFIED
//
// Requirements     : None.
//
// Components Used  : _FSI_STL::map, _FSI_STL::string, Core::COpenGLtext,
//                    Comms::CCommsAction, Core::CXMLWidget, _FSI_STL::list.
//
// Operational 
//    Restrictions  : Machine dependencies/restrictions
//                        None.
//                    Design dependencies/restrictions
//                        None.
//                    Operations containing dependencies/restrictions
//                        None.
//                    Compiler dependencies/restrictions
//                        None.
//                    Other conditions for proper execution
//                        None.
//
// Environment      : Operating system(s) - Microsoft Windows NT 4.0 with
//                                              NT service pack 3, 4, or 5
//                                          Microsoft Windows NT 2000
//
//                    Compiler(s) - Visual C++ 6.0
//
//                    Architechure(s) - Intel Pentium, Pentium II
/////////////////////////////////////////////////////////////////////////////

/////////////////////////////////////////////////////////////////////////////
//
//                              R e v i s i o n   H i s t o r y
//
/////////////////////////////////////////////////////////////////////////////
// $Log: plotwidget.cpp $                                                                   
// Revision 1.6  2000/03/07 20:45:16  billyb                                                                   
// Changed function signatures to eliminate warnings.                                                                   
// Changed calls to AddElementValue.  Added casts to eliminate                                                                   
// warnings.                                                                   
// Revision 1.5  2000/02/01 19:57:30  billyb                                                                   
// Added comments.  Stripped out time based specific code.  Add                                                                   
// virtual methods for drawing labels, grid lines, inspection data. tick                                                                   
// marks, and updating the plot data.  Stripped out CWidget creation                                                                   
// code since a CPlotWidget should never be created.                                                                   
/////////////////////////////////////////////////////////////////////////////
#include "..\core\stdafx.h"
#include <gl\gl.h>
#include <gl\glu.h>

#include <map>
#include <string>

#include "OpenGLWidget.h"

#include "PlotWidget.h"

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif

/////////////////////////////////////////////////////////////////////////////
//
//  CPlotData
//
/////////////////////////////////////////////////////////////////////////////


/////////////////////////////////////////////////////////////////////////////
//
// CPlotData::CPlotData()
//
// Inputs           : None.
//
// Return Values    : None.
//
// Date             : 21 January 2000
//
// Engineer         : Billy Baker
//
// Description      : Default constructor.
//
/////////////////////////////////////////////////////////////////////////////
CPlotData::CPlotData()
{ 
    m_fPlotMinY                 = 0.0f;
    m_fPlotMaxY                 = 1.0f;
    m_pfPlotData                = NULL;
    m_ulSizeOfData              = 0;
    m_color                     = CColor(255, 0, 0);
    m_bShow                     = false;
    m_bRecomputeYLabels         = true;
}

/////////////////////////////////////////////////////////////////////////////
//
// CPlotData::CPlotData()
//
// Inputs           : const CPlotData& rPlotData - reference to a instance
//                                                 of CPlotData being copied.
//
// Return Values    : None.
//
// Date             : 21 January 2000
//
// Engineer         : Billy Baker
//
// Description      : Copy constructor.
//
/////////////////////////////////////////////////////////////////////////////
CPlotData::CPlotData(const CPlotData& rPlotData)
{
    m_fPlotMinY                 = 0.0f;
    m_fPlotMaxY                 = 1.0f;
    m_pfPlotData                = NULL;
    m_ulSizeOfData              = 0;
    m_color                     = CColor(255, 0, 0);
    m_bShow                     = false;
    m_bRecomputeYLabels         = true;

    *this = rPlotData;
}

/////////////////////////////////////////////////////////////////////////////
//
// CPlotData::~CPlotData()
//
// Inputs           : None.
//
// Return Values    : None.
//
// Date             : 21 January 2000
//
// Engineer         : Billy Baker
//
// Description      : Default destructor.  Deallocates any memory for actual
//                    plot data.
//
/////////////////////////////////////////////////////////////////////////////
CPlotData::~CPlotData()
{
    if (m_pfPlotData != NULL)
    {
        delete [] m_pfPlotData;
        m_pfPlotData                = NULL;
        m_ulSizeOfData              = 0;
    }
}

/////////////////////////////////////////////////////////////////////////////
//
// CPlotData& CPlotData::operator=()
//
// Inputs           : const CPlotData& rPlotData - reference to a instance
//                                                 of CPlotData being assigned
//                                                 to another instance.
//
// Return Values    : reference to the object that received the data from
//                    the passed reference.
//
// Date             : 21 January 2000
//
// Engineer         : Billy Baker
//
// Description      : Override of operator= to make a copy of the actual plot 
//                    data.
//
/////////////////////////////////////////////////////////////////////////////
CPlotData& CPlotData::operator=(const CPlotData& rPlotData)
{
    // not complete
    if (this != &rPlotData)
    {
        if (m_pfPlotData != NULL)
        {
            delete [] m_pfPlotData;
            m_pfPlotData                = NULL;
            m_ulSizeOfData              = 0;
        }

        m_pfPlotData        = NULL;
        m_fPlotMinY         = rPlotData.m_fPlotMinY;
        m_fPlotMaxY         = rPlotData.m_fPlotMaxY;
        m_ulSizeOfData      = rPlotData.m_ulSizeOfData;
        m_color             = rPlotData.m_color;
        m_bShow             = rPlotData.m_bShow;
        m_bRecomputeYLabels = rPlotData.m_bRecomputeYLabels;
        m_xmlCommsWidget    = rPlotData.m_xmlCommsWidget;

        if (rPlotData.m_pfPlotData != NULL)
        {
            int nSize = m_ulSizeOfData * 2;
            m_pfPlotData    = new float[nSize];
            int nCount = 0;
            while (nCount < nSize)
            {
                *(m_pfPlotData + nCount) = *(rPlotData.m_pfPlotData + nCount);
                nCount ++;
            }
        }
    }

    return *this;
}
        
/////////////////////////////////////////////////////////////////////////////
//
//  CPlotWidget
//
/////////////////////////////////////////////////////////////////////////////

_FSI_STL::map<_FSI_STL::string, CColor> CPlotWidget::m_mapColors;

/////////////////////////////////////////////////////////////////////////////
//
// CPlotWidget::CPlotWidget()
//
// Inputs           : None.
//
// Return Values    : None.
//
// Date             : 21 January 2000
//
// Engineer         : Billy Baker
//
// Description      : Default contructor.  Reads the registry for the rate
//                    of communication with the host and the host's name.
//                    Sets up variables for a minimal graph--one y variable,
//                    start/stop, clear, inspect, x divisions, y divisions, 
//                    colors, etc.
//
/////////////////////////////////////////////////////////////////////////////
CPlotWidget::CPlotWidget()
{
    m_fRate             = 30.f;
    m_stlStrHost        = _FSI_STL::string("");

    // Open the branch of the registry for Comms to get the rate and hostname.
    HKEY hKey;
    RegOpenKeyEx(HKEY_LOCAL_MACHINE, _T("Software\\FlightSafety\\FSISuite\\Comms"), 0, KEY_READ, &hKey);

    if (hKey != NULL)
    {
		CString strValue = "30";
		DWORD dwType, dwCount;

        // Find the rate.
		LONG lResult = RegQueryValueEx(hKey, _T("HostRate"), 
                                       NULL, &dwType, NULL, &dwCount);
		if (lResult == ERROR_SUCCESS)
		{
			ASSERT(dwType == REG_SZ);
			lResult = RegQueryValueEx(hKey, _T("HostRate"), NULL, &dwType,
				            (LPBYTE)strValue.GetBuffer(dwCount/sizeof(TCHAR)),
                                      &dwCount);

			strValue.ReleaseBuffer();
		}

        m_fRate = atof((LPCSTR)strValue);

        strValue = "";

        // Find the hostname.
		lResult = RegQueryValueEx(hKey, _T("MachineName"), 
                                       NULL, &dwType, NULL, &dwCount);
		if (lResult == ERROR_SUCCESS)
		{
			ASSERT(dwType == REG_SZ);
			lResult = RegQueryValueEx(hKey, _T("MachineName"), NULL, &dwType,
				            (LPBYTE)strValue.GetBuffer(dwCount/sizeof(TCHAR)),
                                      &dwCount);

			strValue.ReleaseBuffer();
		}

        m_stlStrHost = _FSI_STL::string((LPCTSTR)strValue);

        RegCloseKey(hKey);
    }

    // Must be set once for InitReinit() to work correctly.
    m_gridX             = NULL;
    m_gridY             = NULL;
    m_small_ticks       = NULL;
    m_large_ticks       = NULL;

    // Every plot should have these.
    m_listGraphicalElementVars.clear();
    m_listGraphicalElementVars.push_back("Start/Stop");
    m_listGraphicalElementVars.push_back("Clear");
    m_listGraphicalElementVars.push_back("Show Plot X Grid");
    m_listGraphicalElementVars.push_back("Plot X Divisions");
    m_listGraphicalElementVars.push_back("Show Plot Y Grid");
    m_listGraphicalElementVars.push_back("Plot Y Divisions");
    m_listGraphicalElementVars.push_back("Inspect Plot Values");

    // Every plot should have at least one y variable
    m_listGraphicalElementVars.push_back("Plot Y Var 1 Name");

    m_mapCommsActions["Plot Y Var 1 Name"] = NULL;

    m_listGraphicalElementVars.push_back("Y Var 1 Min");
    m_listGraphicalElementVars.push_back("Y Var 1 Max");

    if (m_mapColors.empty())
    {
        // Those shared colors.
        m_mapColors["Red"]      = CColor(255, 192, 203);
        m_mapColors["Green"]    = CColor(  0, 255,   0);
        m_mapColors["Blue"]     = CColor(  0, 255, 255);
        m_mapColors["Yellow"]   = CColor(255, 255,   0);
        m_mapColors["Orange"]   = CColor(255, 165,   0);
        m_mapColors["Purple"]   = CColor(238, 130, 238);
    }

    CPlotWidget::InitReinit();
}

void CPlotWidget::InitReinit()
{
    // Delete any comms actions that were created.
    _FSI_STL::map<_FSI_STL::string, CCommsAction*>::iterator mcaIt;
    _FSI_STL::map<_FSI_STL::string, CCommsAction*>::iterator mcaendIt;
    mcaIt       = m_mapCommsActions.begin();
    mcaendIt    = m_mapCommsActions.end();
    CWidget* pWidget = NULL;
    while (mcaIt != mcaendIt)
    {
        pWidget = dynamic_cast<CWidget*>((*mcaIt).second);

        if (pWidget != NULL)
        {
            if (IsValidAddress(pWidget) != INVALID)
            {
                pWidget->Deleting(true);
            }
        }

        (*mcaIt).second = NULL;

        mcaIt++;
    }

    // Delete any memory for grid lines and tic marks.
    if (m_gridX != NULL)
    {
        delete [] m_gridX;
    }

    if (m_gridY != NULL)
    {
        delete [] m_gridY;
    }

    if (m_small_ticks != NULL)
    {
        delete [] m_small_ticks;
    }

    if (m_large_ticks != NULL)
    {
        delete [] m_large_ticks;
    }

    m_bClearPlot        = false;
    m_bClearPlot_cv     = m_bClearPlot;
    m_bGridChanged      = true;
    m_bGridLinesX       = true;
    m_bGridLinesX_cv    = m_bGridLinesX;
    m_bGridLinesY       = true;
    m_bGridLinesY_cv    = m_bGridLinesY;
    m_bInspectValues    = true;
    m_bInspectValues_cv = m_bInspectValues;
    m_bPlot             = false;
    m_bPlot_cv          = m_bPlot;
    m_bStopped          = false;

    m_fPlotMinX         = 0.0f;
    m_fPlotMinX_cv      = m_fPlotMinX;
    m_fPlotMaxX         = 1.0f;
    m_fPlotMaxX_cv      = m_fPlotMaxX;

    m_gridX             = NULL;
    m_gridY             = NULL;

    m_height_old        = 0.0f;

    m_large_ticks       = NULL;

    m_mapVarName2PlotData.clear();
    m_mapVarName2PlotData_cv.clear();

    _FSI_STL::map<_FSI_STL::string, _FSI_STL::map<float, COpenGLtext> >::iterator myIt;
    _FSI_STL::map<_FSI_STL::string, _FSI_STL::map<float, COpenGLtext> >::iterator myendIt;
    myIt    = m_mapVarName2YLabels.begin();
    myendIt = m_mapVarName2YLabels.end();

    while (myIt != myendIt)
    {
        (*myIt).second.clear();
        myIt++;
    }

    m_mapVarName2YLabels.clear();

    m_mapXLabels.clear();

    m_small_ticks       = NULL;

    m_ulXDivisions      = 10;
    m_ulXDivisions_cv   = m_ulXDivisions;
    m_ulYDivisions      = 10;
    m_ulYDivisions_cv   = m_ulYDivisions;

    m_width_old         = 0.0f;
}

/////////////////////////////////////////////////////////////////////////////
//
// CPlotWidget::~CPlotWidget()
//
// Inputs           : None.
//
// Return Values    : None.
//
// Date             : 21 January 2000
//
// Engineer         : Billy Baker
//
// Description      : Default destructor.  Deallocates any memory grid data
//                    and comms actions.
//
/////////////////////////////////////////////////////////////////////////////
CPlotWidget::~CPlotWidget()
{
    // Delete any comms actions that were created.
    _FSI_STL::map<_FSI_STL::string, CCommsAction*>::iterator mcaIt;
    _FSI_STL::map<_FSI_STL::string, CCommsAction*>::iterator mcaendIt;
    mcaIt       = m_mapCommsActions.begin();
    mcaendIt    = m_mapCommsActions.end();
    CWidget* pWidget = NULL;
    while (mcaIt != mcaendIt)
    {
        pWidget = dynamic_cast<CWidget*>((*mcaIt).second);
        if (pWidget != NULL)
        {
            if (IsValidAddress(pWidget) != INVALID)
            {
                pWidget->Deleting(true);
            }
        }
        mcaIt++;
    }

    // Delete any memory for grid lines and tic marks.
    if (m_gridX != NULL)
    {
        delete [] m_gridX;
    }

    if (m_gridY != NULL)
    {
        delete [] m_gridY;
    }

    if (m_small_ticks != NULL)
    {
        delete [] m_small_ticks;
    }

    if (m_large_ticks != NULL)
    {
        delete [] m_large_ticks;
    }
}

void CPlotWidget::Deleting(bool bDeleting)
{
    if (bDeleting == true)
    {
        // Delete any comms actions that were created.
        _FSI_STL::map<_FSI_STL::string, CCommsAction*>::iterator mcaIt;
        _FSI_STL::map<_FSI_STL::string, CCommsAction*>::iterator mcaendIt;
        mcaIt       = m_mapCommsActions.begin();
        mcaendIt    = m_mapCommsActions.end();
        CWidget* pWidget = NULL;
        while (mcaIt != mcaendIt)
        {
            pWidget = dynamic_cast<CWidget*>((*mcaIt).second);

            if (pWidget != NULL)
            {
                if (IsValidAddress(pWidget) != INVALID)
                {
                    pWidget->Deleting(true);
                }
            }

            (*mcaIt).second = NULL;

            mcaIt++;
        }
    }

    CWidget::Deleting(bDeleting);
}

BEGIN_MESSAGE_MAP(CPlotWidget, CButtonWidget)
	//{{AFX_MSG_MAP(CPlotWidget)
  	ON_WM_WINDOWPOSCHANGED()
	ON_WM_MOUSEMOVE()
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
//
// void CPlotWidget::OnMouseMove()
//
// Inputs           : UINT nFlags   - status of keyboard such as CTRL.
//                    CPoint point  - the position of the mouse.
//
// Return Values    : None.
//
// Date             : 21 January 2000
//
// Engineer         : Billy Baker
//
// Description      : OnMouseMove makes sure that the class knows where
//                    the mouse is so that inspection data can be displayed
//                    for the mouse's current position.
//
/////////////////////////////////////////////////////////////////////////////
void CPlotWidget::OnMouseMove(UINT nFlags, CPoint point)
{
    m_ptMouse   = point;
}

/////////////////////////////////////////////////////////////////////////////
//
// void CPlotWidget::Setup()
//
// Inputs           : None.
//
// Return Values    : None.
//
// Date             : 21 January 2000
//
// Engineer         : Billy Baker
//
// Description      : Common method among the OpenGL related classes.  Setup()
//                    get the width and height of where the OpenGL code will
//                    draw and gets the screen coordinates of where the code
//                    will draw.  These values are used to set the viewport
//                    and scissor areas.
//
/////////////////////////////////////////////////////////////////////////////
void CPlotWidget::Setup()
{
    if (GetSafeHwnd() == NULL)
    {
        return;
    }

    CRect rectangle;
    GetWindowRect(&rectangle);

    m_width  =  (float)rectangle.Width();
    m_height =  (float)rectangle.Height();

    rectViewCoord = rectangle;
    rectView = CRect(0,0,1152,800);

    // Get the size of the view.
    CFrameWnd* pFrame = GetParentFrame();
    if (pFrame != NULL)
    {
       CView* pView = pFrame->GetActiveView();
       if (pView != NULL)
       {
           pView->GetWindowRect(&rectView);
           pView->ScreenToClient(&rectViewCoord);
       }
    }

    glClearStencil(1);
    glClear(GL_STENCIL_BUFFER_BIT);
}

/////////////////////////////////////////////////////////////////////////////
//
// void CPlotWidget::Render()
//
// Inputs           : None.
//
// Return Values    : None.
//
// Date             : 21 January 2000
//
// Engineer         : Billy Baker
//
// Description      : Common method among the OpenGL related classes.  Render()
//                    makes OpenGL calls to draw.  This method is invoked in
//                    COpenGLWidget::DrawItem().
//
/////////////////////////////////////////////////////////////////////////////
void CPlotWidget::Render()
{
    glViewport(rectViewCoord.left, rectView.Height() - rectViewCoord.bottom, 
               (int)m_width, (int)m_height);
    glScissor(rectViewCoord.left, rectView.Height() - rectViewCoord.bottom, 
              (int)m_width, (int)m_height);

    glLoadIdentity();

    // Clear to the background color.
    glClearColor(Background().Red()/255.0f, Background().Green()/255.0f, 
                 Background().Blue()/255.0f, 1.0f);  
    glClear(GL_COLOR_BUFFER_BIT);

    glStencilFunc(GL_ALWAYS, 0, 0);
    glStencilOp(GL_REPLACE, GL_REPLACE, GL_REPLACE);

    glLoadIdentity();

    gluOrtho2D(0.0      - fXBORDER - fSMALLOFFSET, 
               m_width  + fXBORDER + fSMALLOFFSET, 
               0.0      - fYBORDER - fSMALLOFFSET, 
               m_height + fYBORDER + fSMALLOFFSET);

    // Fill the graphing area with a color different
    // than the background color.
    glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
    glBegin(GL_QUADS);
        glColor3f(0.0f, 0.0f, 0.0f);
        glVertex2f(0.0, 0.0);
        glVertex2f(m_width, 0.0);
        glVertex2f(m_width, m_height);
        glVertex2f(0.0, m_height);
    glEnd();

    // Disable/enable the stencil test.  The Appian Jeronimo 2000
    // PCI had problems without these lines.
    glDisable(GL_STENCIL_TEST);

    // If grid lines are desired, then put them in.
    DrawGrid();

    // Bound the graphing area with a color different
    // than the background color.
    glBegin(GL_LINE_LOOP);
        glColor3f(1.0f, 1.0f, 1.0f);
        glVertex2f(0.0, 0.0);
        glVertex2f(m_width, 0.0);
        glVertex2f(m_width, m_height);
        glVertex2f(0.0, m_height);
    glEnd();

    glEnable(GL_STENCIL_TEST);

    glStencilFunc(GL_ALWAYS, 1, 1);
    glStencilOp(GL_REPLACE, GL_REPLACE, GL_REPLACE);

    // Put the small tick marks in.
    DrawSmallTicks();

    // Put the large tick marks in.
    DrawLargeTicks();

    // Put plot rate and host name in.
    COpenGLtext oglTextLastValue;
    CString     strLastValue; 

    glPushMatrix();
    {
        glTranslatef(7.0f * m_width / 8.0f, m_height + 13.0f, 0.0f);
        if (m_bInspectValues == true)
        {
            strLastValue.Format("INSP RATE: %.1f HZ  HOST: %-24s", 
                                m_fRate, m_stlStrHost.c_str());
        }
        else
        {
            strLastValue.Format("     RATE: %.1f HZ  HOST: %-24s", 
                                m_fRate, m_stlStrHost.c_str());
        }
        strLastValue.MakeUpper();
        oglTextLastValue.SetForegroundColor(255,255,255);
        oglTextLastValue.Draw(strLastValue, true, 0.0f, 175.0f);
    }
    glPopMatrix();

}

/////////////////////////////////////////////////////////////////////////////
//
// void CPlotWidget::UpdatePlotData()
//
// Inputs           : _FSI_STL::string& rstlStrVariable - internal name of 
//                                                        variable
//                    CVariant* pVariant                - y data
//
// Return Values    : None.
//
// Date             : 21 January 2000
//
// Engineer         : Billy Baker
//
// Description      : CPlotWidget::UpdatePlotData() is called from 
//                    ChangeValue() whenever new y data arrives.
//
/////////////////////////////////////////////////////////////////////////////
void CPlotWidget::UpdatePlotData(const _FSI_STL::string& rstlStrVariable, 
                                 CVariant* pVariant)
{
}

/////////////////////////////////////////////////////////////////////////////
//
// void CPlotWidget::DrawGrid()
//
// Inputs           : None.
//
// Return Values    : None.
//
// Date             : 01 October 1999
//
// Engineer         : Billy Baker
//
// Description      : CPlotWidget::DrawGrid() is called from Render() 
//                    when m_bGridLines is true.  This method will 
//                    check to see if the width or height of the widget 
//                    has changed since the last pass.  If it has, then 
//                    the array of vertices for drawing the grid lines 
//                    will be recreated. The arrays will always be drawn
//                    unless the user decides not to show the grid lines
//                    or the user wants to inspect values.
//
/////////////////////////////////////////////////////////////////////////////
void CPlotWidget::DrawGrid()
{
    if (m_bInspectValues == true)
    {
        return;
    }

//  ulVertices    = 2 * m_ulXDivisions - 2 + 2 * m_ulYDivisions - 2;
    unsigned long int ulVerticesX   = 2 * m_ulXDivisions - 2;
    unsigned long int ulVerticesY   = 2 * m_ulYDivisions - 2;

    if (m_width != m_width_old || m_height != m_height_old || m_bGridChanged == true ||
        m_gridX == NULL || m_gridY == NULL)
    {
        // Number of vertices times the number of components that
        // make up the vertice.
        unsigned long int ulArraySizeX   = 2 * ulVerticesX;
        unsigned long int ulArraySizeY   = 2 * ulVerticesY;

        // Delete any old data.
        if (m_gridX != NULL)
        {
            delete [] m_gridX;
        }

        if (m_gridY != NULL)
        {
            delete [] m_gridY;
        }

        m_gridX = new GLfloat[ulArraySizeX];
        m_gridY = new GLfloat[ulArraySizeY];

        unsigned long int ulIndex = 0;

        if (m_gridX != NULL)
        {

            // Vertical lines.
            float fWidthDivDivisions    = m_width  / (float)m_ulXDivisions;
            unsigned long int ul = 1; 
            while (ul < ulVerticesX)
            {
                m_gridX[2 * (ul-1)] = (ul - ulIndex) * fWidthDivDivisions;
                m_gridX[2 * ul - 1] = 0.0f;

                m_gridX[2 * ul]     = (ul - ulIndex) * fWidthDivDivisions;
                m_gridX[2 * ul + 1] = m_height;

                ulIndex = ul - ulIndex;
                ul += 2;
            }
        }

        if (m_gridY != NULL)
        {
            ulIndex = 0;

            // Horizontal lines.
            float fHeightDivDivisions   = m_height / (float)m_ulYDivisions;
            unsigned long int ul = 1; 
            while (ul < ulVerticesY)
            {
                m_gridY[2 * (ul-1)] = 0.0f;
                m_gridY[2 * ul - 1] = (ul - ulIndex) * fHeightDivDivisions;

                m_gridY[2 * ul]     = m_width;
                m_gridY[2 * ul + 1] = (ul - ulIndex) * fHeightDivDivisions;

                ulIndex = ul - ulIndex;
                ul += 2;
            }
        }
    }


    // EnableClientState is not called since it is used just about everywhere
    // and should be enabled from the very beginning.
    if (m_bGridLinesX == true && m_gridX != NULL)
    {
        glColor3f(0.502f, 0.502f, 0.502f);
        glVertexPointer(2, GL_FLOAT, 0, (const void*)m_gridX);
        glDrawArrays(GL_LINES, 0, ulVerticesX);
    }

    if (m_bGridLinesY == true && m_gridY != NULL)
    {
        glColor3f(0.502f, 0.502f, 0.502f);
        glVertexPointer(2, GL_FLOAT, 0, (const void*)m_gridY);
        glDrawArrays(GL_LINES, 0, ulVerticesY);
    }
}

/////////////////////////////////////////////////////////////////////////////
//
// void CPlotWidget::DrawSmallTicks()
//
// Inputs           : None.
//
// Return Values    : None.
//
// Date             : 01 October 1999
//
// Engineer         : Billy Baker
//
// Description      : CPlotWidget::DrawSmallTicks() is called from Render().
//                    This method will check to see if the width or height 
//                    of the widget has changed since the last pass.  If it 
//                    has, then the array of vertices for drawing the Small 
//                    ticks will be recreated. The array will always be drawn.
//
/////////////////////////////////////////////////////////////////////////////
void CPlotWidget::DrawSmallTicks()
{
//  ulVertices    = 2 * (5 * m_ulXDivisions + 1) + 
//                  2 * (2 * (5 * m_ulYDivisions + 1));
    unsigned long int ulVertices    = 10 * m_ulXDivisions + 
                                      20 * m_ulYDivisions + 6;

    if (m_width != m_width_old || m_height != m_height_old || m_bGridChanged == true)
    {
        // The width or height changed, recreate the array of vertices.

        // Number of vertices times the number of components that
        // make up the vertice.
        unsigned long int ulArraySize   = 2 * ulVertices;

        // Delete any old data.
        if (m_small_ticks != NULL)
        {
            delete [] m_small_ticks;
        }

        m_small_ticks = new GLfloat[ulArraySize];

        if (m_small_ticks != NULL)
        {
            // x tick lines.
            float fWidthDiv5Divisions       = m_width  / (m_ulXDivisions * 5.0f);
            float fHalfWidthDiv5Divisions   = fWidthDiv5Divisions / 2.0f;
            float fHeightDiv5Divisions      = m_height / (m_ulYDivisions * 5.0f);
            float fFourthHeightDiv5Divisions = fHeightDiv5Divisions / 4.0f;
            int n10XDivisions               = 10 * m_ulXDivisions;

            int i = 0; 
            while (i <= n10XDivisions)
            {
                m_small_ticks[2 * i]     = (float)i * fHalfWidthDiv5Divisions;
                m_small_ticks[2 * i + 1] = -3.0f;

                m_small_ticks[2 * i + 2] = (float)i * fHalfWidthDiv5Divisions;
                m_small_ticks[2 * i + 3] = 0.0f;

                i += 2;
            }

//          ulArrayOffset = (10 * m_ulXDivisions / 2 + 1) * 4;
            unsigned long int ulArrayOffset = 20 * m_ulXDivisions + 4;

            // y tick lines
            int n20YDivisions = 20 * m_ulYDivisions;
            i = 0; 
            while (i <= n20YDivisions)
            {
                m_small_ticks[2 * i + ulArrayOffset]     = -3.0f;
                m_small_ticks[2 * i + 1 + ulArrayOffset] = (float)i *
                                                fFourthHeightDiv5Divisions;

                m_small_ticks[2 * i + 2 + ulArrayOffset] = 0.0f;
                m_small_ticks[2 * i + 3 + ulArrayOffset] = (float)i *
                                                fFourthHeightDiv5Divisions;

                m_small_ticks[2 * i + 4 + ulArrayOffset] = m_width + 
                                                           fSMALLOFFSET;
                m_small_ticks[2 * i + 5 + ulArrayOffset] = (float)i * 
                                                fFourthHeightDiv5Divisions;

                m_small_ticks[2 * i + 6 + ulArrayOffset] = m_width + 
                                                           fSMALLOFFSET + 
                                                           3.0f;
                m_small_ticks[2 * i + 7 + ulArrayOffset] = (float)i * 
                                                fFourthHeightDiv5Divisions;

                i += 4;
            }
        }
    }


    // EnableClientState is not called since it is used just about everywhere
    // and should be enabled from the very beginning.
    glColor3f(1.0f, 1.0f, 1.0f);
    glVertexPointer(2, GL_FLOAT, 0, (const void*)m_small_ticks);
    glDrawArrays(GL_LINES, 0, ulVertices);
}

/////////////////////////////////////////////////////////////////////////////
//
// void CPlotWidget::DrawLargeTicks()
//
// Inputs           : None.
//
// Return Values    : None.
//
// Date             : 01 October 1999
//
// Engineer         : Billy Baker
//
// Description      : CPlotWidget::DrawLargeTicks() is called from Render().
//                    This method will check to see if the width or height 
//                    of the widget has changed since the last pass.  If it 
//                    has, then the array of vertices for drawing the large 
//                    ticks will be recreated. The array will always be drawn.
//
/////////////////////////////////////////////////////////////////////////////
void CPlotWidget::DrawLargeTicks()
{
//  ulVertices    = 2 * m_ulXDivisions + 2 + 4 * m_ulYDivisions + 4;
    unsigned long int ulVertices    = 2 * m_ulXDivisions + 
                                      4 * m_ulYDivisions + 6;

    if (m_width != m_width_old || m_height != m_height_old || m_bGridChanged == true)
    {
        // The width or height changed, recreate the array of vertices.

        // Number of vertices times the number of components that
        // make up the vertice.
        unsigned long int ulArraySize   = 2 * ulVertices;

        // Delete any old data.
        if (m_large_ticks != NULL)
        {
            delete [] m_large_ticks;
        }

        m_large_ticks = new GLfloat[ulArraySize];

        if (m_large_ticks != NULL)
        {
            // large x ticks.
            float fWidthDivDivisions        = m_width  / (float)m_ulXDivisions;
            float fHalfWidthDivDivisions    = fWidthDivDivisions / 2.0f;
            float fHeightDivDivisions       = m_height / (float)m_ulYDivisions;
            float fFourthHeightDivDivisions = fHeightDivDivisions / 4.0f;
            int n2XDivisions                = 2 * m_ulXDivisions;
            int i = 0; 
            while (i <= n2XDivisions)
            {
                m_large_ticks[2 * i]     = (float)i * fHalfWidthDivDivisions;
                m_large_ticks[2 * i + 1] = -6.0f;

                m_large_ticks[2 * i + 2] = (float)i * fHalfWidthDivDivisions;
                m_large_ticks[2 * i + 3] = 0.0f;

                i += 2;
            }

//          ulArrayOffset = (2 * m_ulXDivisions / 2 + 1) * 4;
            unsigned long int ulArrayOffset = (m_ulXDivisions + 1) * 4;

            // large y ticks.
            int n4YDivisions            = 4 * m_ulYDivisions;

            i = 0; 
            while (i <= n4YDivisions)
            {
                m_large_ticks[2 * i + ulArrayOffset] = -6.0f;
                m_large_ticks[2 * i + 1 + ulArrayOffset] = (float)i *
                                                fFourthHeightDivDivisions;

                m_large_ticks[2 * i + 2 + ulArrayOffset] = 0.0f;
                m_large_ticks[2 * i + 3 + ulArrayOffset] = 
                                                (i / 4) * fHeightDivDivisions;

                m_large_ticks[2 * i + 4 + ulArrayOffset] = m_width;
                m_large_ticks[2 * i + 5 + ulArrayOffset] = (float)i *
                                                fFourthHeightDivDivisions;

                m_large_ticks[2 * i + 6 + ulArrayOffset] = m_width + 
                                                           fSMALLOFFSET + 
                                                           6.0f;
                m_large_ticks[2 * i + 7 + ulArrayOffset] = (float)i *
                                                fFourthHeightDivDivisions;

                i += 4;
            }
        }
    }


    // EnableClientState is not called since it is used just about everywhere
    // and should be enabled from the very beginning.
    glColor3f(1.0f, 1.0f, 1.0f);
    glVertexPointer(2, GL_FLOAT, 0, (const void*)m_large_ticks);
    glDrawArrays(GL_LINES, 0, ulVertices);
}

/////////////////////////////////////////////////////////////////////////////
//
// void CPlotWidget::DrawXLabels()
//
// Inputs           : None.
//
// Return Values    : None.
//
// Date             : 01 October 1999
//
// Engineer         : Billy Baker
//
// Description      : CPlotWidget::DrawXLabels() is called from Render().
//                    This method will check to see if the width or height 
//                    of the widget has changed since the last pass.  If it 
//                    has, then the map of x label position to label will
//                    be recreated.  For the time based plot, the map
//                    would need to regenerate the labels if the width and
//                    height didn't change but the graph either wrapped or
//                    erased.
//
/////////////////////////////////////////////////////////////////////////////
void CPlotWidget::DrawXLabels()
{
}

/////////////////////////////////////////////////////////////////////////////
//
// void CPlotWidget::DrawYLabels()
//
// Inputs           : None.
//
// Return Values    : None.
//
// Date             : 01 October 1999
//
// Engineer         : Billy Baker
//
// Description      : CPlotWidget::DrawYLabels() is called from Render().
//                    This method will check to see if the width or height 
//                    of the widget has changed since the last pass.  If it 
//                    has, then the map of y label position to label will
//                    be recreated.  
//
/////////////////////////////////////////////////////////////////////////////
void CPlotWidget::DrawYLabels(const _FSI_STL::string& rstlStrVarName)
{
}

/////////////////////////////////////////////////////////////////////////////
//
// void CPlotWidget::DrawInspection()
//
// Inputs           : None.
//
// Return Values    : None.
//
// Date             : 21 January 2000
//
// Engineer         : Billy Baker
//
// Description      : CPlotWidget::DrawInspection() is called from Render().
//                    This method will draw information related to the data
//                    that would be intersect by a line passing throught the
//                    mouse's current position and perpendicular to the 
//                    bounding box of the plot area.
//
/////////////////////////////////////////////////////////////////////////////
void CPlotWidget::DrawInspection()
{
}

/////////////////////////////////////////////////////////////////////////////
//
// Methods called from Comms thread 
//
/////////////////////////////////////////////////////////////////////////////

/////////////////////////////////////////////////////////////////////////////
//
// void CPlotWidget::ChangeValue()
//
// Inputs           : CString& rstrElementVar - the internal name of a 
//                                              variable that needs to be 
//                                              updated.
//                    CChangeValue* pCV       - pointer to data used to 
//                                              update an internal 
//                                              variable.
//
// Return Values    : None.
//
// Date             : 21 January 2000
//
// Engineer         : Billy Baker
//
// Description      : CPlotWidget::ChangeValue() is a common framework 
//                    method that is used by the Comms thread to send 
//                    new data to the graphical elements.  Since it is 
//                    called from another thread, the special internal 
//                    variables are used to store the updates.  Before 
//                    the graphical element is redrawn, 
//                    UpdateRenderVariables is called to copy the 
//                    updated values to the values used for rendering.  
//                    This causes more memory to be used but prevents 
//                    the use of synchronization objects that may stall 
//                    the Comms thread.
//
/////////////////////////////////////////////////////////////////////////////
void CPlotWidget::ChangeValue(const CString& rstrElementVar, CChangeValue* pCV)
{
    if (pCV == NULL)
    {
        return;
    }

    if (rstrElementVar == "Show Plot X Grid")
    {
        m_bGridLinesX_cv = (bool)*pCV->Variant();
    }
    else if (rstrElementVar == "Plot X Divisions")
    {
        m_ulXDivisions_cv = *pCV->Variant();
    }
    else if (rstrElementVar == "Show Plot Y Grid")
    {
        m_bGridLinesY_cv = (bool)*pCV->Variant();
    }
    else if (rstrElementVar == "Plot Y Divisions")
    {
        m_ulYDivisions_cv = *pCV->Variant();
    }
    else if (rstrElementVar == "Inspect Plot Values")
    {
        m_bInspectValues_cv = !m_bInspectValues;
    }
    else if (rstrElementVar   == "Plot Y Var 1 Name")
    {
        CXMLElement* pXMLElement = NULL;
        POSITION pos = NULL;
        if (m_xmlCommsSampleWidget.FindElement(pXMLElement, pos, _FSI_STL::string("VARIABLE")) == true)
        {
            pXMLElement->AddElementValue((_FSI_STL::string)*pCV->Variant());

            STRING2STRING_MAP::iterator s2sIt;

            if (pXMLElement->FindAttribute(s2sIt,_FSI_STL::string("ELEMENT_VAR")) == true)
            {
                (*s2sIt).second = _FSI_STL::string("Plot Y Var 1");
            }
        }

        m_mapVarName2PlotData_cv["Plot Y Var 1"].m_xmlCommsWidget = m_xmlCommsSampleWidget;
        if (m_mapVarName2PlotData_cv["Plot Y Var 1"].m_pfPlotData != NULL)
        {
            delete m_mapVarName2PlotData_cv["Plot Y Var 1"].m_pfPlotData;
            m_mapVarName2PlotData_cv["Plot Y Var 1"].m_pfPlotData = NULL;
            m_mapVarName2PlotData_cv["Plot Y Var 1"].m_ulSizeOfData = 0;
        }

        if (m_mapCommsActions["Plot Y Var 1"] != NULL)
        {
            (m_mapCommsActions["Plot Y Var 1"])->Deleting(true);
            m_mapCommsActions["Plot Y Var 1"] = NULL;
        }
    }
    else if (rstrElementVar   == "Plot Y Var 1")
    {
        if (m_bPlot_cv == true && m_bStopped == false)
        {
            UpdatePlotData(_FSI_STL::string("Plot Y Var 1"), pCV->Variant());
        }
    }
    else if (rstrElementVar == "Y Var 1 Min")
    {
        if (m_mapVarName2PlotData_cv["Plot Y Var 1"].m_fPlotMinY != (float)*pCV->Variant())
        {
            m_mapVarName2PlotData_cv["Plot Y Var 1"].m_fPlotMinY = *pCV->Variant();
            m_mapVarName2PlotData_cv["Plot Y Var 1"].m_bRecomputeYLabels = true;
        }
    }
    else if (rstrElementVar == "Y Var 1 Max")
    {
        if (m_mapVarName2PlotData_cv["Plot Y Var 1"].m_fPlotMaxY != (float)*pCV->Variant())
        {
            m_mapVarName2PlotData_cv["Plot Y Var 1"].m_fPlotMaxY = *pCV->Variant();
            m_mapVarName2PlotData_cv["Plot Y Var 1"].m_bRecomputeYLabels = true;
        }
    }
    else
    {
        CWidget::ChangeValue(rstrElementVar, pCV);
    }
}

/////////////////////////////////////////////////////////////////////////////
//
// bool CPlotWidget::UpdateRenderVariables()
//
// Inputs           : None.
//
// Return Values    : true  - all of the render variables were updated.
//                    false - a problem occurred updating the render 
//                            variables.
//
// Date             : 21 January 2000
//
// Engineer         : Billy Baker
//
// Description      : CPlotWidget::UpdateRenderVariables is a 
//                    common framework method that is called by the 
//                    Comms thread to make sure that all of the 
//                    variables used for rendering have the most up to 
//                    date data.
//
/////////////////////////////////////////////////////////////////////////////
bool CPlotWidget::UpdateRenderVariables()
{
    bool bRetVal = CWidget::UpdateRenderVariables();

    if (bRetVal == true)
    {
        if (m_fPlotMaxX != m_fPlotMaxX_cv                       || 
            m_fPlotMinX != m_fPlotMinX_cv                       ||
            m_ulXDivisions != m_ulXDivisions_cv                 ||
            m_ulYDivisions != m_ulYDivisions_cv)
        {
            m_bGridChanged          = true;
        }

        m_ulXDivisions          = m_ulXDivisions_cv;
        m_ulYDivisions          = m_ulYDivisions_cv;

        m_bInspectValues        = m_bInspectValues_cv;
        m_bClearPlot            = m_bClearPlot_cv;

        m_mapVarName2PlotData   = m_mapVarName2PlotData_cv;

        m_bGridLinesX           = m_bGridLinesX_cv;
        m_bGridLinesY           = m_bGridLinesY_cv;

        // Create any comms actions that need to be created.
        _FSI_STL::map<_FSI_STL::string, CPlotData>::iterator mIt;
        _FSI_STL::map<_FSI_STL::string, CPlotData>::iterator mendIt;
        mIt = m_mapVarName2PlotData.begin();
        mendIt = m_mapVarName2PlotData.end();
        while (mIt != mendIt)
        {
            if (m_mapCommsActions[(*mIt).first] == NULL && 
                (*mIt).second.m_bShow == true)
            {
                m_mapCommsActions[(*mIt).first] = new CCommsAction();

                if (m_mapCommsActions[(*mIt).first] != NULL)
                {
                    CXMLWidget* pXMLWidget = &((*mIt).second.m_xmlCommsWidget);
                    m_mapCommsActions[(*mIt).first]->ParentWidget(this);
                    m_mapCommsActions[(*mIt).first]->Initialize(pXMLWidget,
                                                                this, 1, false);
                    m_mapCommsActions[(*mIt).first]->BaseWidget(BaseWidget());
                }
            }

            mIt++;
        }

        if (m_bStopped == true)
        {
            m_bPlot_cv = false;
            m_bPlot    = m_bPlot_cv;
            m_bStopped = false;
        }
        else
        {
            m_bPlot    = m_bPlot_cv;
        }

        m_bStopped = false;
    }

    return bRetVal;
}
