// Variant.h: interface for the CVariant class.
//
//////////////////////////////////////////////////////////////////////

#if !defined(_VARIANT_H_)
#define _VARIANT_H_

#ifdef CORE
#define DLLAPI __declspec( dllexport )
#else
#define DLLAPI __declspec( dllimport )
#endif

#include <string>
#include <deque>
#include <map>

#if _MSC_VER > 1000
#pragma once
#endif // _MSC_VER > 1000

enum
{
    VAR_UNKNOWN                 = -1,
    VAR_CHAR                    =  0,
    VAR_UNSIGNED_CHAR           =  1,
    VAR_SHORT_INT               =  2,
    VAR_UNSIGNED_SHORT_INT      =  3,
    VAR_LONG_INT                =  4,
    VAR_UNSIGNED_LONG_INT       =  5,
    VAR_FLOAT                   =  6,
    VAR_DOUBLE                  =  7,
    VAR_BOOL                    =  8,
    VAR_STRING                  =  9,
    VAR_TOTAL_TYPES             = 10
};

union VariantTypeData
{
    char                u_cValue;
    unsigned char       u_ucValue;
    short int           u_shValue;
    unsigned short int  u_ushValue;
    long int            u_lValue;
    unsigned long int   u_ulValue;
    float               u_fValue;
    double              u_dValue;
    bool                u_bValue;
};

struct VariantListData
{
    char *              s_pcValue;
    unsigned int        s_unLength;
    long int            s_lType;
};

class DLLAPI CVariant  
{
private:
    // The type of data
    long int                m_lType;

    // The maximum size that a list of historical data
    // can get.
    unsigned long int       m_ulMaxList;

    // The list of historical data.
    _FSI_STL::deque<VariantListData>* m_pdequeValues;

    // Pointer to array of bools that tell whether
    // data has changed.
    bool*                   m_pbDataChanged;

    // Length of data--1 to n if any data has been
    // assigned to the variant.
    unsigned int            m_unLength;

    // Pointer to the actual data
    char*                   m_pcValue;

    // A union of common types so that memory is not
    // constantly being created and freed for single
    // element arrays.
    VariantTypeData         m_varTypeData;

    // Whether any piece of data has been changed
    bool                    m_bAnyPieceChanged;

    // Whether the variant has a list.
    bool                    m_bIsList;

    // Whether the variant should have its floating point
    // representation limited in the number of digits after
    // the decimal.
    bool                    m_bLimitFloating;

    // The number of digits after the decimal.
    unsigned char           m_ucPrecision;

    static unsigned char    m_ucSizes[VAR_TOTAL_TYPES];

    void                    ToString() const;
    void                    ToString(_FSI_STL::string& str) const;
    _FSI_STL::string        m_stlStrValue;

    static _FSI_STL::map<long, long>    m_mapReferenceListCount;

public:
	CVariant();
	virtual ~CVariant();
	CVariant(const CVariant& rVariant);

    // Assignment overload
    CVariant& operator=(const CVariant& rVariant);

    CVariant  operator[](unsigned int unIndex) const;
    
    inline operator char() const;
    inline operator unsigned char() const;
    inline operator short() const;
    inline operator unsigned short() const;
    inline operator long() const;
    inline operator unsigned long() const;
    inline operator float() const;
    inline operator double() const;
    inline operator bool() const;
    inline operator _FSI_STL::string() const;

    // Conditional will always return false if the type is string!!!!
    inline bool operator <(const CVariant& rvar1);
    inline bool operator <=(const CVariant& rvar1);
    inline bool operator ==(const CVariant& rvar1);
    inline bool operator !=(const CVariant& rvar1);
    inline bool operator >=(const CVariant& rvar1);
    inline bool operator >(const CVariant& rvar1);

    // Mutators
    inline void Value(char cValue, unsigned int unLength = 1);
    inline void Value(unsigned char ucValue, unsigned int unLength = 1);
    inline void Value(short int shValue, unsigned int unLength = 1);
    inline void Value(unsigned short int ushValue, unsigned int unLength = 1);
    inline void Value(long int lValue, unsigned int unLength = 1);
    inline void Value(unsigned long int ulValue, unsigned int unLength = 1);
    inline void Value(float fValue, unsigned int unLength = 1);
    inline void Value(double dValue, unsigned int unLength = 1);
    inline void Value(bool bValue, unsigned int unLength = 1);
    inline void Value(const _FSI_STL::string& strValue);
    inline void Value(char *pcValue, long int lType, unsigned int unLength);

    // Accessors
    inline bool List();
    inline unsigned long int MaxSize();
    inline unsigned char Precision();
    inline bool LimitFloating();
    inline long int Type();
    inline unsigned int Length();
    inline bool WasChanged(const int nIndex = -1);

    // Mutators
    inline void List(bool bList);
    inline void MaxSize(unsigned long int ulMaxSize);
    inline void Precision(const unsigned char ucPrecision);
    inline void LimitFloating(const bool bLimitFloating);
    inline void WasChanged(const int nIndex, const bool bWasChanged);

    static unsigned char Size(long int lVarType);
    inline void* CreateVar();
    inline _FSI_STL::deque<VariantListData>* DataList();

    inline void AddValue(CVariant* pVariant);

    inline void ChangeType(long int lVarType);
};

#endif // !defined(_VARIANT_H_)
