/////////////////////////////////////////////////////////////////////////////
//
//           F L I G H T S A F E T Y   I N T E R N A T I O N A L
//                     Simulation Systems Division
//                      2700 North Hemlock Circle
//                     Broken Arrow, Oklahoma 74012
//                          (918) 259-4000
/////////////////////////////////////////////////////////////////////////////
//
// DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
// Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
// technical or operational data or information from automatic dissemination
// under the International Exchange Program or by other means.  This protection
// covers information required solely for administrative or operational
// purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
//
// WARNING:  This document contains technical data whose export is restricted
// by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
// Executive Order 12470.  Violation of these export control laws is subject
// to severe criminal penalties.  Dissemination of this document is controlled
// under DOD Directive 5230.25
//
/////////////////////////////////////////////////////////////////////////////
//
//
// Filename         : Undo.h
//
// Date             : 20 January 2000
//
// Engineer         : Billy Baker
//
// Revision         : $Revision: 1.3 $
//
// Description      : Undo.h contains a template definition for an undo
//                    system.  This system stores undo states.  Each undo
//                    state must be a pointer--i.e Type must be a pointer
//                    type.  When the maximum size, as configured by the 
//                    user, is reached, states are popped off the front of the
//                    list of undo states.  The last state that was added
//                    using ok_push is kept so that, if a cancel is needed
//                    of states that have been applied but not okayed, the
//                    change can be rolled back.  When a CUndo<Type> instance
//                    is deleted it will delete the memory for each undo
//                    state.
//
// Classification   : UNCLASSIFIED
//
// Requirements     : None.
//
// Components Used  : _FSI_STL::list, _FSI_STL::list<Type>::iterator.
//
// Operational 
//    Restrictions  : Machine dependencies/restrictions
//                        None.
//                    Design dependencies/restrictions
//                        None.
//                    Operations containing dependencies/restrictions
//                        None.
//                    Compiler dependencies/restrictions
//                        None.
//                    Other conditions for proper execution
//                        None.
//
// Environment      : Operating system(s) - Microsoft Windows NT 4.0 with
//                                              NT service pack 3, 4, or 5
//                                          Microsoft Windows NT 2000
//
//                    Compiler(s) - Visual C++ 6.0
//
//                    Architechure(s) - Intel Pentium, Pentium II
/////////////////////////////////////////////////////////////////////////////

/////////////////////////////////////////////////////////////////////////////
//
//                              R e v i s i o n   H i s t o r y
//
/////////////////////////////////////////////////////////////////////////////
// $Log: Undo.h $                                                                   
// Revision 1.3  2000/01/20 19:13:26  billyb                                                                   
// Added comments.  Fixed adding a state when the max size                                                                   
// is zero or less.  Changed undo to return top() if canUndo was                                                                   
// true.                                                                   
/////////////////////////////////////////////////////////////////////////////
#if !defined(_UNDO_H_)
#define _UNDO_H_

#include <list>

#if _MSC_VER > 1000
#pragma once
#endif // _MSC_VER > 1000

template <class Type>
class CUndo  
{
private:
    // The maximum size of the list.  The user can specify this
    // value.
    unsigned long int                   m_ulMaxSize;

    // List to keep track of the undo states.  An undo state
    // might be the name of a file with a previous version,
    // a pointer to an object that has a previous version, etc.
    // Type must be a pointer type.
    _FSI_STL::list<Type>                m_listUndoStates;

    // The last state that the user applied, okayed, or undid.
    _FSI_STL::list<Type>::iterator      m_itState;

    // The last committed state that the user accepted via an 
    // ok type of action.
    _FSI_STL::list<Type>::iterator      m_itPreviousState;

public:
    /////////////////////////////////////////////////////////////////////////
    //
    // CUndo<Type>::CUndo<Type>()
    //
    // Inputs           : None.
    //
    // Return Values    : None.
    //
    // Date             : 20 January 2000
    //
    // Engineer         : Billy Baker
    //
    // Description      : CUndo<Type>::CUndo<Type>() is the default 
    //                    constructor.  It will initialize the the
    //                    CUndo<Type> instance to a state in which
    //                    no undo states have been added and none could
    //                    added since the maximum size of the undo list
    //                    will be zero.
    //
    /////////////////////////////////////////////////////////////////////////
    CUndo() : m_itState(NULL), m_itPreviousState(NULL), m_ulMaxSize(0)
    {
    }

    /////////////////////////////////////////////////////////////////////////
    //
    // CUndo<Type>::~CUndo<Type>()
    //
    // Inputs           : None.
    //
    // Return Values    : None.
    //
    // Date             : 20 January 2000
    //
    // Engineer         : Billy Baker
    //
    // Description      : CUndo<Type>::~CUndo<Type>() is the default 
    //                    destructor.  It will not delete any memory
    //                    that was allocated for each undo state if
    //                    the undo states are pointers.  It will simply
    //                    clear the list.
    //
    /////////////////////////////////////////////////////////////////////////
    ~CUndo() 
    {
        clear();
    }

    /////////////////////////////////////////////////////////////////////////
    //
    // void CUndo<Type>::maxSize()
    //
    // Inputs           : unsigned long int ulSize - the maximum number of
    //                                               undo states.
    //
    // Return Values    : None.
    //
    // Date             : 20 January 2000
    //
    // Engineer         : Billy Baker
    //
    // Description      : CUndo<Type>::maxSize is a mutator method to set
    //                    the maximum number of undo states as set by the
    //                    user.  This method should be called after an
    //                    instance of CUndo<Type> is created.
    //
    /////////////////////////////////////////////////////////////////////////
    void maxSize(unsigned long int ulSize)
    {
        m_ulMaxSize = ulSize;
    }
    
    /////////////////////////////////////////////////////////////////////////
    //
    // Type CUndo void<Type>::top()
    //
    // Inputs           : None.
    //
    // Return Values    : The last state added to the undo list.
    //                    If no states have been added, then NULL is returned.
    //
    // Date             : 20 January 2000
    //
    // Engineer         : Billy Baker
    //
    // Description      : If an undo state has been added to the undo list, 
    //                    then CUndo<Type>::top will return the last state.
    //                    If no states have been added or the list size is
    //                    zero, which would not allow any states to be added,
    //                    then NULL is returned.
    //
    /////////////////////////////////////////////////////////////////////////
    Type top()
    {
        _FSI_STL::list<Type>::iterator nullIt = NULL;
        if (m_itState != nullIt)
        {
            return (*m_itState);
        }

        return NULL;
    }

    /////////////////////////////////////////////////////////////////////////
    //
    // void apply_push()
    //
    // Inputs           : const Type& t - a reference to a state that may 
    //                                    be added.
    //
    // Return Values    : None.
    //
    // Date             : 20 January 2000
    //
    // Engineer         : Billy Baker
    //
    // Description      : CUndo<Type>::apply_push is invoked to attempt to
    //                    add a new state to the list of undo states.  The
    //                    maximum size of the list must be greater than zero
    //                    for this to occur.  Likewise, the new state must
    //                    not match the current state.  If a redo command(s)
    //                    have been issued before apply_push is invoked, then
    //                    the redo states beyond the current iterator will
    //                    be deleted and the new state to be added will
    //                    become the new current state.
    //
    /////////////////////////////////////////////////////////////////////////
    void apply_push(const Type& t)
    {
        // Make sure that the undo list can have undo states.
        if (m_ulMaxSize <= 0)
        {
            return;
        }

        // If some states have been added, then checks are need for
        // reaching the maximum size of the list, if the state to be added
        // matches the last state, and if there are any states beyond the
        // current state which will be lost if the new state is added to
        // the list.
        if (size() > 0)
        {
            // If states have been added (size() > 0) but the last current
            // state is NULL, then a problem has probably occurred.  Don't
            // do anything.
            if (top() != NULL)
            {
                // See if the new state is the same as the current state.
                // If it is, it will be deleted.
                if (!( *t == *(top()) ))
                {
                    // Remove any states beyond the current state.  This
                    // can occur if states have been undone and the current
                    // iterator is not at the back of the list.  Adding
                    // a new state when remove states beyond the current
                    // state and make the state to be added the new current
                    // state.
                    while (m_listUndoStates.back() != *m_itState && size() > 1)
                    {
                        delete m_listUndoStates.back();
                        m_listUndoStates.pop_back();
                    }

                    if (size() == m_ulMaxSize)
                    {
                        // The maxium number of states have been added.
                        // Thus, the front of the list needs to be popped
                        // and the current and last committed iterators need
                        // to be updated if they are at the beginning of the
                        // list.
                        bool bSetPreviousBegin  = false;
                        bool bSetCurrentBegin   = false;
                        if (m_itPreviousState == m_listUndoStates.begin())
                        {
                            // The last committed iterator needs to be
                            // reset once the front is popped.
                            bSetPreviousBegin = true;
                        }

                        if (m_itState == m_listUndoStates.begin())
                        {
                            // The current iterator needs to be
                            // reset once the front is popped.
                            bSetCurrentBegin = true;
                        }

                        // Delete the front state and remove it from the
                        // list.
                        delete m_listUndoStates.front();
                        m_listUndoStates.pop_front();

                        if (bSetPreviousBegin == true)
                        {
                            // Update the last committed iterator to
                            // the new beginning of the list.
                            m_itPreviousState = m_listUndoStates.begin();
                        }

                        if (bSetCurrentBegin == true)
                        {
                            // Update the current iterator to
                            // the new beginning of the list.
                            m_itState = m_listUndoStates.begin();
                        }
                    }

                    // Put the new state into the list and increment
                    // the current iterator.  If there had been states
                    // beyond the current iterator when this method was
                    // invoked, they would have been deleted in the while
                    // loop above making the current iterator the last
                    // state in the list.
                    m_listUndoStates.push_back(t);
                    m_itState++;
                }
                else
                {
                    // Since they are the same, get rid of the new copy.
                    delete t;
                }
            }
        }
        else
        {
            // Since the list of undo states does not have any states 
            // added yet, add the state that was passed and update
            // the current state iterator.
            m_listUndoStates.push_back(t);
            m_itState = m_listUndoStates.begin();
        }
    }

    /////////////////////////////////////////////////////////////////////////
    //
    // void CUndo<Type>::ok_push()
    //
    // Inputs           : const Type& t - a state that may need to be added
    //                                    to the list of undo states.
    //
    // Return Values    : None.
    //
    // Date             : 20 January 2000
    //
    // Engineer         : Billy Baker
    //
    // Description      : CUndo<Type>::ok_push() should be called in response
    //                    to an action that commits a change(s).  The 
    //                    apply_push method is invoked to try and add the new
    //                    state to the list and then the last committed state
    //                    iterator is updated.
    //
    /////////////////////////////////////////////////////////////////////////
    void ok_push(const Type& t)
    {
        apply_push(t);
        m_itPreviousState = m_itState;
    }

    /////////////////////////////////////////////////////////////////////////
    //
    // unsigned long int CUndo<Type>::size()
    //
    // Inputs           : None.
    //
    // Return Values    : The number of undo states in the list.
    //
    // Date             : 20 January 2000
    //
    // Engineer         : Billy Baker
    //
    // Description      : CUndo<Type>::size is an accessor method to get the
    //                    current number of undo states in the list.
    //
    /////////////////////////////////////////////////////////////////////////
    unsigned long int size()
    {
        return m_listUndoStates.size();
    }

    /////////////////////////////////////////////////////////////////////////
    //
    // void CUndo<Type>::clear()
    //
    // Inputs           : None.
    //
    // Return Values    : None.
    //
    // Date             : 20 January 2000
    //
    // Engineer         : Billy Baker
    //
    // Description      : CUndo<Type>::clear is called in the destructor to 
    //                    delete all of the undo states.   Since Type has
    //                    been assumed to be a pointer, then the memory 
    //                    pointed to is freed with a call to delete.  If
    //                    Type is not a pointer type, problems could occur.
    //
    /////////////////////////////////////////////////////////////////////////
    void clear()
    {
        while (size() > 0)
        {
            delete m_listUndoStates.back();
            m_listUndoStates.pop_back();
        }

        m_itState = NULL;
        m_itPreviousState = NULL;
    }

    /////////////////////////////////////////////////////////////////////////
    //
    // Type CUndo<Type>::undo()
    //
    // Inputs           : None.
    //
    // Return Values    : The new state or NULL.
    //
    // Date             : 20 January 2000
    //
    // Engineer         : Billy Baker
    //
    // Description      : CUndo<Type>::undo is invoked to attempt to revert to 
    //                    a previous state that has been added to the list of
    //                    update states.  If no such states exist, then NULL 
    //                    is returned.  Two types of undo are possible.  The
    //                    first, bUndoAll is true, will revert to the last
    //                    committed state.  If any states have been applied
    //                    but not committed, then they are deleted.  The 
    //                    second, bUndoAll is false, will move back one state
    //                    and commit that state.
    //
    /////////////////////////////////////////////////////////////////////////
    Type undo(bool bUndoAll = false)
    {
        if (canUndo())
        {
            if (bUndoAll == true)
            {
                // Delete applied but not committed states.
                while (m_listUndoStates.back() != *m_itPreviousState)
                {
                    delete m_listUndoStates.back();
                    m_listUndoStates.pop_back();
                }

                // Set the current state to the last committed state.
                m_itState = m_itPreviousState;
            }
            else
            {
                // Move the current state back one and make the
                // last committed state the same as the current state.
                m_itState--;
                m_itPreviousState = m_itState;
            }

            return top();
        }

        return NULL;
    }

    /////////////////////////////////////////////////////////////////////////
    //
    // Type CUndo<Type>::redo()
    //
    // Inputs           : None.
    //
    // Return Values    : The new state or NULL.
    //
    // Date             : 20 January 2000
    //
    // Engineer         : Billy Baker
    //
    // Description      : CUndo<Type>::redo is invoked to attempt to apply 
    //                    a state that has been added once buy then undone.
    //                    If no such states exist, then NULL is returned.
    //
    /////////////////////////////////////////////////////////////////////////
    Type redo()
    {
        if (canRedo())
        {
            // Move to the next state and make it the committed state.
            m_itState++;
            m_itPreviousState = m_itState;
            return top();
        }

        return NULL;
    }

    /////////////////////////////////////////////////////////////////////////
    //
    // bool CUndo<Type>::canUndo()
    //
    // Inputs           : None.
    //
    // Return Values    : Whether the undo list has a state(s) that can
    //                    be undone.
    //
    // Date             : 20 January 2000
    //
    // Engineer         : Billy Baker
    //
    // Description      : CUndo<Type>::canUndo is used to determine if the
    //                    undo list has states before the current state.  If
    //                    so, then the current state can be undo by moving back
    //                    to one of the previous states using an undo command.
    //
    /////////////////////////////////////////////////////////////////////////
    bool canUndo()
    {
        _FSI_STL::list<Type>::iterator nullIt = NULL;

        // If the current state is not NULL and size() is greater than
        // 1 which would mean that undo states may exist before m_itState and
        // the current state is not the first state in the list,
        // then there must be an undo state before m_itState.
        if (m_itState != m_listUndoStates.begin() && size() > 1 &&
            m_itState != nullIt)
        {
            return true;
        }

        return false;
    }

    /////////////////////////////////////////////////////////////////////////
    //
    // bool CUndo<Type>::canRedo()
    //
    // Inputs           : None.
    //
    // Return Values    : Whether the undo list has a state(s) that can
    //                    be applied again.
    //
    // Date             : 20 January 2000
    //
    // Engineer         : Billy Baker
    //
    // Description      : CUndo<Type>::canRedo is used to determine if the
    //                    undo list has states beyond the current state.  If
    //                    so, then these states can be applied again via a
    //                    redo command.
    //
    /////////////////////////////////////////////////////////////////////////
    bool canRedo()
    {
        _FSI_STL::list<Type>::iterator nullIt = NULL;

        // If the current state is not NULL and size() is greater than
        // 1 which would mean that redo states may exist beyond m_itState,
        // then check to see if there are redo states beyond m_itState.
        if (m_itState != nullIt && size() > 1)
        {
            _FSI_STL::list<Type>::iterator lIt = m_itState;

            // Increment an iterator that matches m_itState.
            // If the increment results in moving to the end, then
            // there were not any redo states beyond m_itState.
            // If the increment results in moving to another state,
            // then at least one redo state exists.
            lIt++;
            if (lIt != m_listUndoStates.end())
            {
                return true;
            }
            else
            {
                return false;
            }
        }

        return false;
    }
};

#endif // !defined(_UNDO_H_)
