/////////////////////////////////////////////////////////////////////////////
//
//           F L I G H T S A F E T Y   I N T E R N A T I O N A L
//                     Simulation Systems Division
//                      2700 North Hemlock Circle
//                     Broken Arrow, Oklahoma 74012
//                          (918) 259-4000
/////////////////////////////////////////////////////////////////////////////
//
// DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
// Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
// technical or operational data or information from automatic dissemination
// under the International Exchange Program or by other means.  This protection
// covers information required solely for administrative or operational
// purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
//
// WARNING:  This document contains technical data whose export is restricted
// by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
// Executive Order 12470.  Violation of these export control laws is subject
// to severe criminal penalties.  Dissemination of this document is controlled
// under DOD Directive 5230.25
//
/////////////////////////////////////////////////////////////////////////////
//
//
// Filename         : Range.cpp
//
// Date             : 08 February 2000
//
// Engineer         : Billy Baker
//
// Revision         : $Revision: 1.5 $
//
// Description      : Range.cpp contains the implementation of the 
//                    CRange class.  This class store information about 
//                    possible values that a variable may have.  The 
//                    possible values are represented as conditional 
//                    statements.  For an enumeration, a variable might 
//                    have several ranges each one checking if the 
//                    current value of the varible is equal to both the 
//                    lower and upper conditional value which would be 
//                    the same.
//                    
//                    For each conditional range, true and false color 
//                    and text values are possible.  Also, an action 
//                    may be performed when the range is true.  Each 
//                    range may also have a custom message that will be 
//                    logged if the range is true.
//
// Classification   : UNCLASSIFIED
//
// Requirements     : None.
//
// Components Used  : _FSI_STL::string, CWinApp, CWnd, Core::CVariant, Co
//                    re::CColor.
//
// Operational 
//    Restrictions  : Machine dependencies/restrictions
//                        None.
//                    Design dependencies/restrictions
//                        None.
//                    Operations containing dependencies/restrictions
//                        None.
//                    Compiler dependencies/restrictions
//                        None.
//                    Other conditions for proper execution
//                        None.
//
// Environment      : Operating system(s) - Microsoft Windows NT 4.0 with
//                                              NT service pack 3, 4, or 5
//                                          Microsoft Windows NT 2000
//
//                    Compiler(s) - Visual C++ 6.0
//
//                    Architechure(s) - Intel Pentium, Pentium II
/////////////////////////////////////////////////////////////////////////////

/////////////////////////////////////////////////////////////////////////////
//
//                              R e v i s i o n   H i s t o r y
//
/////////////////////////////////////////////////////////////////////////////
// $Log: Range.cpp $                                                                   
// Revision 1.5  2000/05/04 16:56:53  billyb                                                                   
// Added code to determine if the value sent to Evaluate has                                                                   
// crossed the bounds of the range since the last call to Evaluate.                                                                   
// Prevents extra messages in the message log when a value                                                                   
// changes but stays in the same range.                                                                   
// Revision 1.4  2000/03/07 07:19:53  billyb                                                                   
// Removed unused methods.                                                                   
// Revision 1.3  2000/02/08 19:39:22  billyb                                                                   
// Added comments.  Changed enum values.  Added static                                                                   
// members for Windows message IDs.  Removed methods                                                                   
// protected methods that were only called from Evaluate.                                                                   
/////////////////////////////////////////////////////////////////////////////
#include "stdafx.h"
#include "Range.h"

const UINT   CRange::m_WM_DISPLAY_AUTO_OVERLAY = RegisterWindowMessage("WM_DISPLAY_AUTO_OVERLAY");;
const UINT   CRange::m_WM_CLOSE_AUTO_OVERLAY = RegisterWindowMessage("WM_CLOSE_AUTO_OVERLAY");;

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif

/////////////////////////////////////////////////////////////////////////////
//
// CRange::CRange()
//
// Inputs           : None.
//
// Return Values    : None.
//
// Date             : 08 February 2000
//
// Engineer         : Billy Baker
//
// Description      : Default constructor.
//
/////////////////////////////////////////////////////////////////////////////
CRange::CRange() :
    m_lID               (-1),
    m_enumActionType    (NO_ACTION),
    m_ucLowerConditional(LESS_THAN),
    m_ucUpperConditional(LESS_THAN),
    m_bNewValidRange    (false),
    m_bLowerLast        (false),
    m_bUpperLast        (false)
{
}

/////////////////////////////////////////////////////////////////////////////
//
// void CRange::PerformAction()
//
// Inputs           : None.
//
// Return Values    : None.
//
// Date             : 08 February 2000
//
// Engineer         : Billy Baker
//
// Description      : PerformAction() is invoked whenever a range has an 
//                    action specified that is not equal to NO_ACTION.
//                    When invoked it will post a message to the main 
//                    application window which will actually perform any
//                    work that is needed.  One case for this method's use
//                    is for showing popups automatically when data changes
//                    to inform the user of such things as systems timing out
//                    or an airplane crashing.
//
/////////////////////////////////////////////////////////////////////////////
void CRange::PerformAction()
{
    CWinApp* pApp   = NULL;
    CWnd* pWnd      = NULL;
    HWND hWnd       = NULL;

    // Get the HWND for the main window to send a message
    // about an action to perform.
    pApp    = AfxGetApp();
    if (pApp == NULL)
        return;

    pWnd    = pApp->GetMainWnd();
    if (pWnd == NULL)
        return;

    hWnd    = pWnd->GetSafeHwnd();
    if (hWnd == NULL)
        return;

    if (m_enumActionType != NO_ACTION)
    {
        switch(m_enumActionType)
        {
        case DISPLAY_AUTO_OVERLAY:
            {
                // Tell the main frame to add an automatic popup overlay
                ::PostMessage(hWnd, m_WM_DISPLAY_AUTO_OVERLAY, 0, (long)&m_stlStrActionData);
            }
            break;
        case CLOSE_AUTO_OVERLAY:
            {
                // Tell the main frame to close an automatic popup overlay
                ::PostMessage(hWnd, m_WM_CLOSE_AUTO_OVERLAY, 0, (long)&m_stlStrActionData);
            }
            break;
        };
    }
}

/////////////////////////////////////////////////////////////////////////////
//
// void CRange::Evaluate()
//
// Inputs           : CVariant& rvarValue.
//
// Return Values    : None.
//
// Date             : 08 February 2000
//
// Engineer         : Billy Baker
//
// Description      : Evaluate() is invoked whenever a value needs to be
//                    checked against the conditional stored in the CRange
//                    instance.  
//
/////////////////////////////////////////////////////////////////////////////
bool CRange::Evaluate( CVariant& rvarValue)
{
    bool bLower = false;
    bool bUpper = false;

    switch (m_ucLowerConditional)
    {
    case LESS_THAN:
        bLower = (m_varLowerValue < rvarValue);
        break;

    case LESS_THAN_EQUAL_TO:
        bLower = (m_varLowerValue <= rvarValue);
        break;

    case EQUAL_TO:
        bLower = (m_varLowerValue == rvarValue);
        break;

    case NOT_EQUAL_TO:
        bLower = (m_varLowerValue != rvarValue);
        break;

    case GREATER_THAN_EQUAL_TO:
        bLower = (m_varLowerValue >= rvarValue);
        break;

    case GREATER_THAN:
        bLower = (m_varLowerValue > rvarValue);
        break;
    }

    // Early exit
    if (bLower == false)
    {
        m_bNewValidRange = false;

        m_bLowerLast = bLower;

        return false;
    }

    switch (m_ucUpperConditional)
    {
    case LESS_THAN:
        bUpper = (rvarValue < m_varUpperValue);
        break;

    case LESS_THAN_EQUAL_TO:
        bUpper = (rvarValue <= m_varUpperValue);
        break;

    case EQUAL_TO:
        bUpper = (rvarValue == m_varUpperValue);
        break;

    case NOT_EQUAL_TO:
        bUpper = (rvarValue != m_varUpperValue);
        break;

    case GREATER_THAN_EQUAL_TO:
        bUpper = (rvarValue >= m_varUpperValue);
        break;

    case GREATER_THAN:
        bUpper = (rvarValue > m_varUpperValue);
        break;
    }

    if (bLower && bUpper && !m_bNewValidRange && ((bLower != m_bLowerLast) || (bUpper != m_bUpperLast)))
        m_bNewValidRange = true;
    else
        m_bNewValidRange = false;

    m_bLowerLast = bLower;
    m_bUpperLast = bUpper;

    return (bLower && bUpper);
}

//////////////////////////////////////////////////////////////////////
// Accessors
//////////////////////////////////////////////////////////////////////
CVariant CRange::Lower()
{
    return m_varLowerValue;
}

CVariant CRange::Upper()
{
    return m_varUpperValue;
}

unsigned char CRange::LowerConditional()
{
    return m_ucLowerConditional;
}

unsigned char CRange::UpperConditional()
{
    return m_ucUpperConditional;
}

CColor CRange::BackgroundTrue()
{
    return m_colorBackgroundTrue;
}

CColor CRange::ForegroundTrue()
{
    return m_colorForegroundTrue;
}

CColor CRange::BackgroundFalse()
{
    return m_colorBackgroundFalse;
}

CColor CRange::ForegroundFalse()
{
    return m_colorForegroundFalse;
}

_FSI_STL::string CRange::TextTrue()
{
    return m_stlStrTextTrue;
}

_FSI_STL::string CRange::TextFalse()
{
    return m_stlStrTextFalse;
}

_FSI_STL::string CRange::LogMessage()
{
    return m_stlStrLogMessage;
}

long int CRange::ID()
{
    return m_lID;
}

_FSI_STL::string CRange::Conditional()
{
    _FSI_STL::string strConditional = (_FSI_STL::string)m_varLowerValue;

    switch (m_ucLowerConditional)
    {
    case LESS_THAN:
        strConditional += " < ";
        break;

    case LESS_THAN_EQUAL_TO:
        strConditional += " <= ";
        break;

    case EQUAL_TO:
        strConditional += " == ";
        break;

    case NOT_EQUAL_TO:
        strConditional += " != ";
        break;

    case GREATER_THAN_EQUAL_TO:
        strConditional += " >= ";
        break;

    case GREATER_THAN:
        strConditional += " > ";
        break;
    }

    strConditional += " value ";

    switch (m_ucUpperConditional)
    {
    case LESS_THAN:
        strConditional += "< ";
        break;

    case LESS_THAN_EQUAL_TO:
        strConditional += "<= ";
        break;

    case EQUAL_TO:
        strConditional += "== ";
        break;

    case NOT_EQUAL_TO:
        strConditional += "!= ";
        break;

    case GREATER_THAN_EQUAL_TO:
        strConditional += ">= ";
        break;

    case GREATER_THAN:
        strConditional += "> ";
        break;
    }

    strConditional += (_FSI_STL::string)m_varUpperValue;

    return strConditional;
}

enum RangeActionTypes CRange::ActionType()
{
    return m_enumActionType;
}

_FSI_STL::string CRange::ActionData()
{
    return m_stlStrActionData;
}

bool CRange::NewValidRange()
{
    return m_bNewValidRange;
}

//////////////////////////////////////////////////////////////////////
// Mutators
//////////////////////////////////////////////////////////////////////
void CRange::Lower(const CVariant& rvarLower)
{
    m_varLowerValue = rvarLower;
}

void CRange::Upper(const CVariant& rvarUpper)
{
    m_varUpperValue = rvarUpper;
}

void CRange::LowerConditional(const unsigned char ucConditional)
{
    m_ucLowerConditional = ucConditional;
}

void CRange::UpperConditional(const unsigned char ucConditional)
{
    m_ucUpperConditional = ucConditional;
}

void CRange::BackgroundTrue(const CColor& rcolorValue)
{
    m_colorBackgroundTrue = rcolorValue;
}

void CRange::ForegroundTrue(const CColor& rcolorValue)
{
    m_colorForegroundTrue = rcolorValue;
}

void CRange::BackgroundFalse(const CColor& rcolorValue)
{
    m_colorBackgroundFalse = rcolorValue;
}

void CRange::ForegroundFalse(const CColor& rcolorValue)
{
    m_colorForegroundFalse = rcolorValue;
}

void CRange::TextTrue(const _FSI_STL::string& rstlStrText)
{
    m_stlStrTextTrue = rstlStrText;
}

void CRange::TextFalse(const _FSI_STL::string& rstlStrText)
{
    m_stlStrTextFalse = rstlStrText;
}

void CRange::LogMessage(const _FSI_STL::string& rstlStrLogMessage)
{
    m_stlStrLogMessage = rstlStrLogMessage;
}

void CRange::ID(const long int lID)
{
    m_lID = lID;
}

void CRange::ActionType(unsigned char ucActionType)
{
    m_enumActionType = static_cast<enum RangeActionTypes>(ucActionType);
}

void CRange::ActionData(const _FSI_STL::string& rstlStrActionData)
{
    m_stlStrActionData = rstlStrActionData;
}