/////////////////////////////////////////////////////////////////////////////
//
//           F L I G H T S A F E T Y   I N T E R N A T I O N A L
//                     Simulation Systems Division
//                      2700 North Hemlock Circle
//                     Broken Arrow, Oklahoma 74012
//                          (918) 259-4000
/////////////////////////////////////////////////////////////////////////////
//
// DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
// Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
// technical or operational data or information from automatic dissemination
// under the International Exchange Program or by other means.  This protection
// covers information required solely for administrative or operational
// purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
//
// WARNING:  This document contains technical data whose export is restricted
// by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
// Executive Order 12470.  Violation of these export control laws is subject
// to severe criminal penalties.  Dissemination of this document is controlled
// under DOD Directive 5230.25
//
/////////////////////////////////////////////////////////////////////////////
//
//
// Filename         : OpenGLCircle.cpp
//
// Date             : 04 August 1999
//
// Engineer         : Billy Baker
//
// Revision         : $Revision: 1.4 $
//
// Description      : OpenGLCircle.cpp contains the implementation of 
//                    the COpenGLCircle class.  This class is used to 
//                    draw a circle in the xy plane.  The user of this 
//                    class is responsible for putting the circle in 
//                    the correct position and for scaling the radius.
//
// Classification   : UNCLASSIFIED
//
// Requirements     : OpenGL.
//
// Components Used  : OpenGL vertex arrays, math funcitons.
//
// Operational 
//    Restrictions  : Machine dependencies/restrictions
//                        None.
//                    Design dependencies/restrictions
//                        None.
//                    Operations containing dependencies/restrictions
//                        None.
//                    Compiler dependencies/restrictions
//                        None.
//                    Other conditions for proper execution
//                        None.
//
// Environment      : Operating system(s) - Microsoft Windows NT 4.0 with
//                                              NT service pack 3, 4, or 5
//                                          Microsoft Windows NT 2000
//
//                    Compiler(s) - Visual C++ 6.0
//
//                    Architechure(s) - Intel Pentium, Pentium II
/////////////////////////////////////////////////////////////////////////////

/////////////////////////////////////////////////////////////////////////////
//
//                              R e v i s i o n   H i s t o r y
//
/////////////////////////////////////////////////////////////////////////////
// $Log: openglcircle.cpp $                                                                   //
// Revision 1.4  2000/01/27 20:04:31  billyb                                                                   //
// Changed byte by byte memset to 4 byte by 4 byte memset.                                                                   //
// Revision 1.3  2000/01/27 00:36:17  billyb                                                                   //
// Corrected possible memory leaks/memory corruption with                                                                   //
// copy constructor and assignment operator.                                                                   //
// Revision 1.2  1999/10/20 07:38:14  billyb                                                                   //
// Changed a number of loops to perform less calculations.                                                                   //
/////////////////////////////////////////////////////////////////////////////
#include "stdafx.h"
#include "OpenGLCircle.h"
#include "dataconversion.h"

/////////////////////////////////////////////////////////////////////////////
//
// COpenGLCircle::COpenGLCircle()
//
// Inputs           : None.
//
// Return Values    : None.
//
// Date             : 04 August 1999
//
// Engineer         : Billy Baker
//
// Description      : Default constructor.  Initializes the array of 
//                    vertices to NULL, the primitive type to 
//                    GL_LINE_STRIP, and the number of vertices to zero.
//
/////////////////////////////////////////////////////////////////////////////
COpenGLCircle::COpenGLCircle()
{
    m_vertices      = NULL;
    m_primitive     = GL_LINE_STRIP;
    m_ulArraySize   = 0;
}

/////////////////////////////////////////////////////////////////////////////
//
// COpenGLCircle::COpenGLCircle(const COpenGLCircle& rOGLCircle)
//
// Inputs           : const COpenGLCircle& rOGLCircle - a COpenGLCircle to
//                                                   to copy to this instance.
//
// Return Values    : None.
//
// Date             : 04 August 1999
//
// Engineer         : Billy Baker
//
// Description      : Copy constructor.
//
/////////////////////////////////////////////////////////////////////////////
COpenGLCircle::COpenGLCircle(const COpenGLCircle& rOGLCircle)
{
    m_vertices      = NULL;
    m_primitive     = GL_LINE_STRIP;
    m_ulArraySize   = 0;

    *this = rOGLCircle;
}

/////////////////////////////////////////////////////////////////////////////
//
// COpenGLCircle::~COpenGLCircle()
//
// Inputs           : None.
//
// Return Values    : None.
//
// Date             : 04 August 1999
//
// Engineer         : Billy Baker
//
// Description      : Default destructor.  Deletes any data for the 
//                    array of vertices.
//
/////////////////////////////////////////////////////////////////////////////
COpenGLCircle::~COpenGLCircle()
{
    if (m_vertices != NULL)
    {
        delete [] m_vertices;

        m_vertices = NULL;
    }
}

/////////////////////////////////////////////////////////////////////////////
//
// COpenGLCircle& COpenGLCircle::operator=(const COpenGLCircle& rOGLCircle)
//
// Inputs           : const COpenGLCircle& rOGLCircle - a COpenGLCircle to
//                                                to assign to this instance.
//
// Return Values    : a reference to this instance.
//
// Date             : 04 August 1999
//
// Engineer         : Billy Baker
//
// Description      : Override of the assignment operator.
//
/////////////////////////////////////////////////////////////////////////////
COpenGLCircle& COpenGLCircle::operator=(const COpenGLCircle& rOGLCircle)
{
	// no work if they are the same
	if (this != &rOGLCircle)
	{
        if (m_vertices != NULL)
        {
            delete [] m_vertices;

            m_vertices = NULL;
        }

        // Don't share the m_vertices value because one could be destroyed
        // without the other knowing.
        Primitive(rOGLCircle.m_primitive);
        ArraySize(rOGLCircle.m_ulArraySize);
	}

	return *this;
}

/////////////////////////////////////////////////////////////////////////////
//
// bool COpenGLCircle::operator=(const COpenGLCircle& rOGLCircle)
//
// Inputs           : const COpenGLCircle& rOGLCircle - a COpenGLCircle to
//                                               to compare to this instance.
//
// Return Values    : whether the two instances being compared are equal.
//
// Date             : 04 August 1999
//
// Engineer         : Billy Baker
//
// Description      : Override of the equals operator.
//
/////////////////////////////////////////////////////////////////////////////
bool COpenGLCircle::operator==(const COpenGLCircle& rOGLCircle)
{
    bool bRetVal = true;

    if (m_primitive != rOGLCircle.m_primitive)
    {
        bRetVal = false;
    }

    if (m_ulArraySize != rOGLCircle.m_ulArraySize)
    {
        bRetVal = false;
    }

    return bRetVal;
}

/////////////////////////////////////////////////////////////////////////////
//
// void COpenGLCircle::Draw()
//
// Inputs           : None.
//
// Return Values    : None.
//
// Date             : 04 August 1999
//
// Engineer         : Billy Baker
//
// Description      : COpenGLCircle::Draw() draws the entire array of 
//                    vertices which should be a full circle.
//
/////////////////////////////////////////////////////////////////////////////
void COpenGLCircle::Draw()
{
    glVertexPointer(3, GL_FLOAT, 0, (const void*)m_vertices);

    // GL_LINE_STRIP has shown better ICD support than GL_LINE_LOOP.
    if (m_primitive == GL_LINE_STRIP)
    {
        glDrawArrays(m_primitive, 0, m_ulArraySize + 1);
    }
    else
    {
        glDrawArrays(m_primitive, 0, m_ulArraySize);
    }
}

/////////////////////////////////////////////////////////////////////////////
//
// void COpenGLCircle::DrawArc()
//
// Inputs           : const double dStartAngle - the start of the arc.
//                    const double dEndAngle   - the end of the arc.
//
// Return Values    : None.
//
// Date             : 04 August 1999
//
// Engineer         : Billy Baker
//
// Description      : COpenGLCircle::DrawArc() is used to draw a 
//                    portion of the entire circle.  If the start and 
//                    end angles resolve to the beginning and end of 
//                    the arrays, the COpenGLCircle::Draw is called.  
//                    Otherwise, the start must be less than the end and 
//                    they must both be less than 360 for a partial arc 
//                    to be drawn.
//
/////////////////////////////////////////////////////////////////////////////
void COpenGLCircle::DrawArc(const double dStartAngle, const double dEndAngle)
{
    if (fabs(dEndAngle - dStartAngle) >= 360.0)
    {
        // Draw the full circle.
        Draw();
    }
    else
    {
        if (dStartAngle >= dEndAngle    || 
            dStartAngle > 360.0         || 
            dEndAngle > 360.0)
        {
            // Nothing to draw or bounds that are not within 0 to 360.
            return;
        }

        // Compute the indices for the supplied angles.
        double dDegPerIndex = 360.0 / (double)m_ulArraySize;
        unsigned long int nBeginIndex = (int)(dStartAngle * dDegPerIndex - 0.5);

        unsigned long int nEndIndex   = (int)(dEndAngle   * dDegPerIndex + 0.5);

        if (nEndIndex > m_ulArraySize)
        {
            nEndIndex = m_ulArraySize;
        }

        // Draw the partial arc.
        glVertexPointer(3, GL_FLOAT, 0, (const void*)m_vertices);
        glDrawArrays(m_primitive, nBeginIndex, nEndIndex - nBeginIndex + 1);
    }
}

/////////////////////////////////////////////////////////////////////////////
//
// void COpenGLCircle::InitializeArray()
//
// Inputs           : None.
//
// Return Values    : None.
//
// Date             : 04 August 1999
//
// Engineer         : Billy Baker
//
// Description      : COpenGLCircle::InitializeArray() is a protected 
//                    member method that computes the geometry data for 
//                    a circle.  The last point is always the same as 
//                    the first for use for GL_LINE_STIP.
//
/////////////////////////////////////////////////////////////////////////////
void COpenGLCircle::InitializeArray()
{
    if (m_vertices != NULL)
    {
        double d2PiDivArraySize = 2.0 * PI / (double)m_ulArraySize;
        for (unsigned long int i = 0; i < m_ulArraySize; i++)
        {
             m_vertices[3*i]     = (float)cos((double)i * d2PiDivArraySize);

             m_vertices[3*i + 1] = (float)sin((double)i * d2PiDivArraySize);
        }

        // Assign the first to the last.
        m_vertices[3*m_ulArraySize]      = m_vertices[0];
        m_vertices[3*m_ulArraySize + 1]  = m_vertices[1];
    }
}

//////////////////////////////////////////////////////////////////////
// Accessors
//////////////////////////////////////////////////////////////////////

/////////////////////////////////////////////////////////////////////////////
//
// GLenum COpenGLCircle::Primitive()
//
// Inputs           : None.
//
// Return Values    : The type of primitive used to draw the circle.  
//                    The valid types are GL_POINTS, GL_LINE_LOOP, and
//                    GL_LINE_STRIP.
//
// Date             : 04 August 1999
//
// Engineer         : Billy Baker
//
// Description      : COpenGLCircle::Primitive() is an accessor method 
//                    for the type of primitive used to render the circle.
//
/////////////////////////////////////////////////////////////////////////////
GLenum COpenGLCircle::Primitive()
{
    return m_primitive;
}

/////////////////////////////////////////////////////////////////////////////
//
// unsigned long int COpenGLCircle::ArraySize()
//
// Inputs           : None.
//
// Return Values    : The number of vertices used to draw a full circle.
//
// Date             : 04 August 1999
//
// Engineer         : Billy Baker
//
// Description      : COpenGLCircle::ArraySize() is an accessor method 
//                    for the number of vertices used to draw a full circle.
//
/////////////////////////////////////////////////////////////////////////////
unsigned long int COpenGLCircle::ArraySize()
{
    return m_ulArraySize;
}

//////////////////////////////////////////////////////////////////////
// Mutators
//////////////////////////////////////////////////////////////////////

/////////////////////////////////////////////////////////////////////////////
//
// void COpenGLCircle::Primitive()
//
// Inputs           : const GLenum primitive - an OpenGL primitive type.
//
// Return Values    : None.
//
// Date             : 04 August 1999
//
// Engineer         : Billy Baker
//
// Description      : COpenGLCircle::Primitive() is a mutator method 
//                    used to set the OpenGL primitive type to use when 
//                    rendering the circle.  This should be either 
//                    GL_POINTS, GL_LINE_LOOP, or GL_LINE_STRIP.  
//                    GL_LINE_STRIP has shown to be better supported in 
//                    vendor ICDs than GL_LINE_LOOP.
//
/////////////////////////////////////////////////////////////////////////////
void COpenGLCircle::Primitive(const GLenum primitive)
{
    if (primitive == GL_POINTS      ||
        primitive == GL_LINE_STRIP  ||
        primitive == GL_LINE_LOOP)
    {
        m_primitive = primitive;
    }
    else
    {
        m_primitive = GL_LINE_STRIP;
    }
}

/////////////////////////////////////////////////////////////////////////////
//
// void COpenGLCircle::ArraySize()
//
// Inputs           : const unsigned long int ulArraySize - the number 
//                    of                          vertices to use when  
//                                                                   
//                    constructing the circle.
//
// Return Values    : None.
//
// Date             : 04 August 1999
//
// Engineer         : Billy Baker
//
// Description      : COpenGLCircle::ArraySize() is a mutator method 
//                    used to resize the array of geometry data.  Any 
//                    old data is deleted and the new array is 
//                    initialized using the COpenGLCircle::InitializeArray 
//                    method.
//
/////////////////////////////////////////////////////////////////////////////
void COpenGLCircle::ArraySize(const unsigned long int ulArraySize)
{
    if (m_vertices != NULL)
    {
        delete m_vertices;
    }

    // Make the array one vertices larger than the size requested to append
    // the first vertice to the end for use with GL_LINE_STRIP.
    m_vertices = new GLfloat[(ulArraySize + 1) * 3];

    // Set everyone to 0.0.
    unsigned long ul = 0;
    unsigned long ulSize = 3 * (ulArraySize + 1);
    while ( ul < ulSize)
    {
        m_vertices[ul]      = 0.0;
        m_vertices[ul + 1]  = 0.0;
        m_vertices[ul + 2]  = 0.0;

        ul += 3;
    }

    m_ulArraySize = ulArraySize;

    // Generate the geometry data.
    InitializeArray();
}
