/////////////////////////////////////////////////////////////////////////////
//
//           F L I G H T S A F E T Y   I N T E R N A T I O N A L
//                     Simulation Systems Division
//                      2700 North Hemlock Circle
//                     Broken Arrow, Oklahoma 74012
//                          (918) 259-4000
/////////////////////////////////////////////////////////////////////////////
//
// DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
// Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
// technical or operational data or information from automatic dissemination
// under the International Exchange Program or by other means.  This protection
// covers information required solely for administrative or operational
// purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
//
// WARNING:  This document contains technical data whose export is restricted
// by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
// Executive Order 12470.  Violation of these export control laws is subject
// to severe criminal penalties.  Dissemination of this document is controlled
// under DOD Directive 5230.25
//
/////////////////////////////////////////////////////////////////////////////
//
//
// Filename         : DataConversion.h
//
// Date             : 01 September 1999
//
// Engineer         : Billy Baker
//
// Revision         : $Revision: 1.6 $
//
// Description      : DataConversion.h contains the definition of an 
//                    abstract class with a number of static members.  
//                    The purpose of this class is to provide a central 
//                    area for performing conversions from one format 
//                    to another.  For a given format, 
//                    PossibleConversions will provide a mapping of the 
//                    other formats to which the data can be converted 
//                    using the Convert() method.  An enumeration is 
//                    provided for all of the different data types as 
//                    is FormatName() to provide a string name for a 
//                    given format.
//
// Classification   : UNCLASSIFIED
//
// Requirements     : None.
//
// Components Used  : _FSI_STL::map, _FSI_STL::string, Core::CVariant.
//
// Operational 
//    Restrictions  : Machine dependencies/restrictions
//                        None.
//                    Design dependencies/restrictions
//                        None.
//                    Operations containing dependencies/restrictions
//                        None.
//                    Compiler dependencies/restrictions
//                        None.
//                    Other conditions for proper execution
//                        None.
//
// Environment      : Operating system(s) - Microsoft Windows NT 4.0 with
//                                              NT service pack 3, 4, or 5
//                                          Microsoft Windows NT 2000
//
//                    Compiler(s) - Visual C++ 6.0
//
//                    Architechure(s) - Intel Pentium, Pentium II
/////////////////////////////////////////////////////////////////////////////

/////////////////////////////////////////////////////////////////////////////
//
//                              R e v i s i o n   H i s t o r y
//
/////////////////////////////////////////////////////////////////////////////
// $Log: DataConversion.h $                                                                   //
// Revision 1.6  2000/03/30 07:54:57  billyb                                                                   //
// Added conversions for inshesHg to millibars, miles                                                                   //
// to kilometers, and signed angles to unsigned angles.                                                                   //
// Revision 1.5  2000/03/07 07:23:37  billyb                                                                   //
// Removed unused methods.                                                                   //
// Revision 1.4  2000/02/28 19:26:40  billyb                                                                   //
// Added temperature conversions.  Changed signatures to                                                                   //
// decrease the number of copies.                                                                   //
// Revision 1.3  1999/10/11 18:07:18  billyb                                                                   //
// Added the rest of mymath.h.                                                                   //
/////////////////////////////////////////////////////////////////////////////
#if !defined(_DATACONVERSION_H_)
#define _DATACONVERSION_H_

#if _MSC_VER > 1000
#pragma once
#endif // _MSC_VER > 1000

#ifdef CORE
#define DLLAPI __declspec( dllexport )
#else
#define DLLAPI __declspec( dllimport )
#endif

#include <math.h>
#include <map>
#include <string>

#include "Variant.h"

enum DATA_FORMATS
{
    Generic_No_Conversions,
    Decimal_Degrees,
    Degrees_Minutes_Seconds_Latitude,
    Degrees_Minutes_Seconds_Longitude,
    BAMS,
    Celsius,
    Fahrenheit,
    Kilometers,
    StatuteMiles,
    Millibars,
    InchesHg,
    NormalizedAngle,
    SignedAngle,
    Seconds,
    Minutes,
    TimeFormat,
    IntX1000,
    Float2T,
    IntX100,
    Total_Data_Formats
};

// Bams to Degrees conversion factor
//   180.0 / 2^(n-1) where n=32 (# of bits in a long integer)
//   approx. 8.3819031715393e-008
static const double dBamsToDegs = 180.0 / (unsigned long)(1<<31);

// Standard value of PI (Radians in 180 Degrees).
static const double PI          = 3.14159265358979323846264338327950288;

// Standard value of PI * 2 (Radians in 360 Degrees).
static const double TWO_PI      = 2.0   *  PI;

// Multiply an angle in Degrees by DEG_TO_RAD to convert the angle to Radians.
static const double DEG_TO_RAD  = PI    /  180.0;

// Multiply an angle in Radians by RAD_TO_DEG to convert the angle to Degrees.
static const double RAD_TO_DEG  = 180.0 /  PI;

// Multiply statute miles by MILES_TO_KM to convert a distance to kilometers;
static const float  MILES_TO_KM = 1.609344f;
static const float  KM_TO_MILES = 1.0f / MILES_TO_KM;

// Multiply inches HG by INHG_TO_MB to convert a pressure to millibars.
static const float  INHG_TO_MB = 33.86388f;
static const float  MB_TO_INHG = 1.0f / INHG_TO_MB;

// ROUND macro:
// Round to the nearest 1/reciprical(th). If positive round up, if negative round down.
// For example if value =  1.234 and reciprical = 100.0 then result would be  1.23  or 
//             if value = -4.567 and reciprical = 100.0 then result would be -4.57
#define  ROUND(value, reciprical)                                                                  \
         (                                                                                         \
            (value >= 0.00)   ?                                                                    \
               ((double)   ((int)((value  *  (double)reciprical)  +  0.5)  /  (double)reciprical)) \
            :                                                                                      \
               ((double)   ((int)((value  *  (double)reciprical)  -  0.5)  /  (double)reciprical)) \
         )
// End of ROUND macro.


// BamsToDegrees
// Convert from Bams to Degrees
// Entry   : Bams
// Returns : Degrees
inline double BamsToDegrees(long lBams)
{
	double dDegrees = lBams * dBamsToDegs;
	return (dDegrees >= 180) ? dDegrees - 360 : dDegrees;
}

// DegreesToBams
// Convert from Degrees to Bams
// Entry   : Degrees
// Returns : Bams
inline long DegreesToBams(double dDegrees)
{
	return (long)(dDegrees / dBamsToDegs);
}


typedef CVariant (*ConversionMethod)(const CVariant& rvarWhat);

class DLLAPI CDataConversion  
{
protected:
    static _FSI_STL::map<DATA_FORMATS, _FSI_STL::map<DATA_FORMATS, ConversionMethod> >  m_mapPossibleConversions;
    static _FSI_STL::map<DATA_FORMATS, _FSI_STL::string>               m_mapFormatNames;

    static bool             m_bInitialized;

    static void InitializeData();

    inline static CVariant DecimalDegreesToDMSLat(const CVariant& rvarWhat);
    inline static CVariant DecimalDegreesToDMSLon(const CVariant& rvarWhat);
    inline static CVariant DecimalDegreesToBAMS(const CVariant& rvarWhat);

    inline static CVariant DMSLatToDecimalDegrees(const CVariant& rvarWhat);
    inline static CVariant DMSLatToBAMS(const CVariant& rvarWhat);

    inline static CVariant DMSLonToDecimalDegrees(const CVariant& rvarWhat);
    inline static CVariant DMSLonToBAMS(const CVariant& rvarWhat);

    inline static CVariant BAMSToDecimalDegrees(const CVariant& rvarWhat);
    inline static CVariant BAMSToDMSLat(const CVariant& rvarWhat);
    inline static CVariant BAMSToDMSLon(const CVariant& rvarWhat);

    inline static CVariant FahrenheitToCelsius(const CVariant& rvarWhat);
    inline static CVariant CelsiusToFahrenheit(const CVariant& rvarWhat);

    inline static CVariant StatuteMilesToKilometers(const CVariant& rvarWhat);
    inline static CVariant KilometersToStatuteMiles(const CVariant& rvarWhat);

    inline static CVariant InchesHgToMillibars(const CVariant& rvarWhat);
    inline static CVariant MillibarsToInchesHg(const CVariant& rvarWhat);

    inline static CVariant SignedToNormalizedAngle(const CVariant& rvarWhat);
    inline static CVariant NormalizedToSignedAngle(const CVariant& rvarWhat);

    inline static CVariant SecondsToTimeFormat(const CVariant& rvarWhat);
    inline static CVariant MinutesToTimeFormat(const CVariant& rvarWhat);

    inline static CVariant IntX1000ToFloat2T(const CVariant& rvarWhat);

    inline static CVariant IntX100ToFloat2T(const CVariant& rvarWhat);

	CDataConversion();

public:
    static _FSI_STL::map<DATA_FORMATS, ConversionMethod> PossibleConversions(DATA_FORMATS eDataFormat);

    static _FSI_STL::string FormatName(DATA_FORMATS eDataFormat);

    static CVariant         Convert(const CVariant& rvarWhat, 
                                    DATA_FORMATS eFromType, 
                                    DATA_FORMATS eToType);
};

#endif // !defined(_DATACONVERSION_H_)
