/////////////////////////////////////////////////////////////////////////////
//
//           F L I G H T S A F E T Y   I N T E R N A T I O N A L
//                     Simulation Systems Division
//                      2700 North Hemlock Circle
//                     Broken Arrow, Oklahoma 74012
//                          (918) 259-4000
/////////////////////////////////////////////////////////////////////////////
//
// DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
// Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
// technical or operational data or information from automatic dissemination
// under the International Exchange Program or by other means.  This protection
// covers information required solely for administrative or operational
// purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
//
// WARNING:  This document contains technical data whose export is restricted
// by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
// Executive Order 12470.  Violation of these export control laws is subject
// to severe criminal penalties.  Dissemination of this document is controlled
// under DOD Directive 5230.25
//
/////////////////////////////////////////////////////////////////////////////
//
//
// Filename         : DataConversion.cpp
//
// Date             : 01 September 1999
//
// Engineer         : Billy Baker
//
// Revision         : $Revision: 1.7 $
//
// Description      : DataConversion.cpp contains the implementation of an 
//                    abstract class with a number of static members.  
//                    The purpose of this class is to provide a central 
//                    area for performing conversions from one format 
//                    to another.  For a given format, 
//                    PossibleConversions will provide a mapping of the 
//                    other formats to which the data can be converted 
//                    using the Convert() method.  An enumeration is 
//                    provided for all of the different data types as 
//                    is FormatName() to provide a string name for a 
//                    given format.
//
// Classification   : UNCLASSIFIED
//
// Requirements     : None.
//
// Components Used  : _FSI_STL::map, _FSI_STL::string, Core::CVariant.
//
// Operational 
//    Restrictions  : Machine dependencies/restrictions
//                        None.
//                    Design dependencies/restrictions
//                        None.
//                    Operations containing dependencies/restrictions
//                        None.
//                    Compiler dependencies/restrictions
//                        None.
//                    Other conditions for proper execution
//                        None.
//
// Environment      : Operating system(s) - Microsoft Windows NT 4.0 with
//                                              NT service pack 3, 4, or 5
//                                          Microsoft Windows NT 2000
//
//                    Compiler(s) - Visual C++ 6.0
//
//                    Architechure(s) - Intel Pentium, Pentium II
/////////////////////////////////////////////////////////////////////////////

/////////////////////////////////////////////////////////////////////////////
//
//                              R e v i s i o n   H i s t o r y
//
/////////////////////////////////////////////////////////////////////////////
// $Log: DataConversion.cpp $                                                                   //
// Revision 1.7  2000/03/30 07:54:57  billyb                                                                   //
// Added conversions for inshesHg to millibars, miles                                                                   //
// to kilometers, and signed angles to unsigned angles.                                                                   //
// Revision 1.6  2000/03/07 07:23:53  billyb                                                                   //
// Removed unused methods and variables..                                                                   //
// Revision 1.5  2000/02/28 19:26:19  billyb                                                                   //
// Added temperature conversions.  Changed signatures to                                                                   //
// decrease the number of copies.                                                                   //
// Revision 1.4  2000/01/27 19:50:17  billyb                                                                   //
// Corrected possible stack overflow problem.                                                                   //
// Revision 1.3  1999/09/19 23:38:07  billyb                                                                   //
// Optimized conversion routines.                                                                   //
// Revision 1.2  1999/09/02 16:50:56  billyb                                                                   //
/////////////////////////////////////////////////////////////////////////////
#include "stdafx.h"
#include "DataConversion.h"

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif

_FSI_STL::map<DATA_FORMATS, _FSI_STL::map<DATA_FORMATS, ConversionMethod> >  
                                      CDataConversion::m_mapPossibleConversions;
_FSI_STL::map<DATA_FORMATS, _FSI_STL::string>               
                                      CDataConversion::m_mapFormatNames;
bool                                  CDataConversion::m_bInitialized = false;

/////////////////////////////////////////////////////////////////////////////
//
// CDataConversion::CDataConversion()
//
// Inputs           : None.
//
// Return Values    : None.
//
// Date             : 01 September 1999
//
// Engineer         : Billy Baker
//
// Description      : The default constructor is protected by default 
//                    which means that an instance of CDataConversion 
//                    can not be created.  Derived classes may use the 
//                    constructor to initialize any default conversions.
//
/////////////////////////////////////////////////////////////////////////////
CDataConversion::CDataConversion()
{
    InitializeData();
}

/////////////////////////////////////////////////////////////////////////////
//
// void CDataConversion::InitializeData()
//
// Inputs           : None.
//
// Return Values    : None.
//
// Date             : 01 September 1999
//
// Engineer         : Billy Baker
//
// Description      : InitializeData will initialize the data structure 
//                    for mapping the DATA_FORMAT value to its string 
//                    name.  And, it will initialize the data structure 
//                    mapping a DATA_FORMAT to a map of formats and 
//                    their conversion methods.  The formats in the map 
//                    of formats to their conversion methods are the 
//                    possible conversions for a given DATA_FORMAT.
//
/////////////////////////////////////////////////////////////////////////////
void CDataConversion::InitializeData()
{
    m_mapFormatNames.clear();
    m_mapPossibleConversions.clear();

    m_mapFormatNames[Generic_No_Conversions] = _FSI_STL::string("Generic_No_Conversions");

    m_mapFormatNames[Decimal_Degrees] = _FSI_STL::string("Decimal_Degrees");
    (m_mapPossibleConversions[Decimal_Degrees])[Degrees_Minutes_Seconds_Latitude] = DecimalDegreesToDMSLat;
    (m_mapPossibleConversions[Decimal_Degrees])[Degrees_Minutes_Seconds_Longitude] = DecimalDegreesToDMSLon;
    (m_mapPossibleConversions[Decimal_Degrees])[BAMS] = DecimalDegreesToBAMS;

    m_mapFormatNames[Degrees_Minutes_Seconds_Latitude] = _FSI_STL::string("Degrees_Minutes_Seconds_Latitude");
    (m_mapPossibleConversions[Degrees_Minutes_Seconds_Latitude])[Decimal_Degrees] = DMSLatToDecimalDegrees;
    (m_mapPossibleConversions[Degrees_Minutes_Seconds_Latitude])[BAMS] = DMSLatToBAMS;

    m_mapFormatNames[Degrees_Minutes_Seconds_Longitude] = _FSI_STL::string("Degrees_Minutes_Seconds_Longitude");
    (m_mapPossibleConversions[Degrees_Minutes_Seconds_Longitude])[Decimal_Degrees] = DMSLonToDecimalDegrees;
    (m_mapPossibleConversions[Degrees_Minutes_Seconds_Longitude])[BAMS] = DMSLonToBAMS;

    m_mapFormatNames[BAMS] = _FSI_STL::string("BAMS");
    (m_mapPossibleConversions[BAMS])[Decimal_Degrees] = BAMSToDecimalDegrees;
    (m_mapPossibleConversions[BAMS])[Degrees_Minutes_Seconds_Latitude] = BAMSToDMSLat;
    (m_mapPossibleConversions[BAMS])[Degrees_Minutes_Seconds_Longitude] = BAMSToDMSLon;

    m_mapFormatNames[Celsius] = _FSI_STL::string("Celsius");
    (m_mapPossibleConversions[Celsius])[Fahrenheit] = CelsiusToFahrenheit;

    m_mapFormatNames[Fahrenheit] = _FSI_STL::string("Fahrenheit");
    (m_mapPossibleConversions[Fahrenheit])[Celsius] = FahrenheitToCelsius;

    m_mapFormatNames[Kilometers] = _FSI_STL::string("Kilometers");
    (m_mapPossibleConversions[Kilometers])[StatuteMiles] = KilometersToStatuteMiles;

    m_mapFormatNames[StatuteMiles] = _FSI_STL::string("StatuteMiles");
    (m_mapPossibleConversions[StatuteMiles])[Kilometers] = StatuteMilesToKilometers;

    m_mapFormatNames[Millibars] = _FSI_STL::string("Millibars");
    (m_mapPossibleConversions[Millibars])[InchesHg] = MillibarsToInchesHg;

    m_mapFormatNames[InchesHg] = _FSI_STL::string("InchesHg");
    (m_mapPossibleConversions[InchesHg])[Millibars] = InchesHgToMillibars;

    m_mapFormatNames[NormalizedAngle] = _FSI_STL::string("Normalized_Angle");
    (m_mapPossibleConversions[NormalizedAngle])[SignedAngle] = NormalizedToSignedAngle;

    m_mapFormatNames[SignedAngle] = _FSI_STL::string("Signed_Angle");
    (m_mapPossibleConversions[SignedAngle])[NormalizedAngle] = SignedToNormalizedAngle;

    m_mapFormatNames[TimeFormat] = _FSI_STL::string("TimeFormat");

    m_mapFormatNames[Seconds] = _FSI_STL::string("Seconds");
    (m_mapPossibleConversions[Seconds])[TimeFormat] = SecondsToTimeFormat;

    m_mapFormatNames[Minutes] = _FSI_STL::string("Minutes");
    (m_mapPossibleConversions[Minutes])[TimeFormat] = MinutesToTimeFormat;


    m_mapFormatNames[Float2T]  = _FSI_STL::string("Float truncated to .xx precision");

    m_mapFormatNames[IntX1000] = _FSI_STL::string("Integer * 1000");
    (m_mapPossibleConversions[IntX1000])[Float2T]  =  IntX1000ToFloat2T;

    m_mapFormatNames[IntX100] = _FSI_STL::string("Integer * 100");
    (m_mapPossibleConversions[IntX100])[Float2T]  =  IntX100ToFloat2T;
    

    m_bInitialized = true;
}

/////////////////////////////////////////////////////////////////////////////
//
// CVariant CDataConversion::Convert()
//
// Inputs           : const CVariant& rvarWhat - the value to convert
//                    DATA_FORMATS eFromType   - the initial data format
//                    DATA_FORMATS eToType     - the format to convert to
//                    
//
// Return Values    : A CVariant with the converted value.
//
// Date             : 01 September 1999
//
// Engineer         : Billy Baker
//
// Description      : Convert is used to convert a value from one 
//                    format to another.  Because of the data 
//                    structures in this class, this method should not 
//                    have to change as new conversions are added.
//
/////////////////////////////////////////////////////////////////////////////
CVariant CDataConversion::Convert(const CVariant& rvarWhat, 
                                  enum DATA_FORMATS eFromType, 
                                  enum DATA_FORMATS eToType)
{
    if (m_bInitialized == false)
    {
        InitializeData();
    }

    _FSI_STL::map<DATA_FORMATS, _FSI_STL::map<DATA_FORMATS, ConversionMethod> >::iterator mIt = 
                                                            m_mapPossibleConversions.find(eFromType);

    if (mIt != m_mapPossibleConversions.end())
    {
        if ((*mIt).second.find(eToType) != (*mIt).second.end())
        {
            return (*mIt).second[eToType](rvarWhat);
        }
    }

    return rvarWhat;
}

/////////////////////////////////////////////////////////////////////////////
//
// _FSI_STL::map<DATA_FORMATS, ConversionMethod> 
//                             CDataConversion::PossibleConversions()
//
// Inputs           : DATA_FORMATS eDataFormat - a DATA_FORMAT for 
//                    which any possible conversions are desired.
//
// Return Values    : A map of the DATA_FORMATs to which the original 
//                    data may be converted and the conversions methods.
//
// Date             : 01 September 1999
//
// Engineer         : Billy Baker
//
// Description      : PossibleConversions is used mainly in the editor 
//                    to get the possible conversions for a particular 
//                    piece of data.  The user can then select the 
//                    conversion desired.  
//
/////////////////////////////////////////////////////////////////////////////
_FSI_STL::map<DATA_FORMATS, ConversionMethod> CDataConversion::PossibleConversions(enum DATA_FORMATS eDataFormat)
{
    if (m_bInitialized == false)
    {
        InitializeData();
    }

    if (m_mapPossibleConversions.find(eDataFormat) != m_mapPossibleConversions.end())
    {
        return m_mapPossibleConversions[eDataFormat];
    }

    return _FSI_STL::map<DATA_FORMATS, ConversionMethod>();
}

/////////////////////////////////////////////////////////////////////////////
//
// _FSI_STL::string CDataConversion::FormatName()
//
// Inputs           : DATA_FORMATS eDataFormat - the DATA_FORMAT whose 
//                    string name is desired.
//
// Return Values    : The string name of the passed DATA_FORMAT.
//
// Date             : 02 September 1999
//
// Engineer         : Billy Baker
//
// Description      : FormatName simply returns the name of a 
//                    DATA_FORMAT as a string.
//
/////////////////////////////////////////////////////////////////////////////
_FSI_STL::string CDataConversion::FormatName(enum DATA_FORMATS eDataFormat)
{
    if (m_bInitialized == false)
    {
        InitializeData();
    }

    if (m_mapFormatNames.find(eDataFormat) != m_mapFormatNames.end())
    {
        return m_mapFormatNames[eDataFormat];
    }

    return _FSI_STL::string("");
}

//////////////////////////////////////////////////////////////////////
// Conversion Methods
//////////////////////////////////////////////////////////////////////
CVariant CDataConversion::DecimalDegreesToDMSLat(const CVariant& rvarWhat)
{
    CVariant var(rvarWhat);
    double  dDegrees = (double)var;
    char    cDMS[16] = "               ";

    double dMinutesSeconds;
    int nDegrees;

    if (dDegrees >= 0.0)
    {
        nDegrees        =  (int)dDegrees;
        dMinutesSeconds =  (dDegrees - nDegrees) * 60.0;
    }
    else
    {
        nDegrees        =  -(int)dDegrees;
        dMinutesSeconds =  -(dDegrees + nDegrees) * 60.0;
    }

//    int nMinutes =  nMinutesSeconds   /  60;
//    int nSeconds =  nMinutesSeconds   %  60;

    // Fix funky floating point mis-representations which cause things like 
    // 90.00 to be 8960'00"
//    if (nSeconds   == 60)
//    {
//        nSeconds =  0;
//        nMinutes++;
//    }

//    if (nMinutes   == 60)
//    {
//        nMinutes =  0;
//        nDegrees++;
//    }

    if (dDegrees <  0.001   && dDegrees >=  -90.0)
    {
        cDMS[0] = 'S';
    }
    else if (dDegrees > 0.001 && dDegrees <= 90)
    {
        cDMS[0] = 'N';
    }

    int nMinutes        = (int)dMinutesSeconds;
    int nFracSeconds    = (int)((dMinutesSeconds - nMinutes) * 100.0);
    sprintf(&cDMS[2], "%02d %02d.%02d", nDegrees, nMinutes, nFracSeconds);
//    cDMS[11]         = '\0';

    var.Value(cDMS, VAR_STRING, 11);
    return var;
}
 
CVariant CDataConversion::DecimalDegreesToDMSLon(const CVariant& rvarWhat)
{
    CVariant var(rvarWhat);
    double  dDegrees = (double)var;
    char    cDMS[16] = "               ";

    double  dMinutesSeconds;
    int     nDegrees;

    if (dDegrees >= 0.0)
    {
        nDegrees        =  (int)dDegrees;
        dMinutesSeconds =  (dDegrees - nDegrees) * 60.0;
    }
    else
    {
        nDegrees        =  -(int)dDegrees;
        dMinutesSeconds =  -(dDegrees + nDegrees) * 60.0;
    }

//    int nMinutes =  nMinutesSeconds   /  60;
//    int nSeconds =  nMinutesSeconds   %  60;

    // Fix funky floating point mis-representations which cause things like 
    // 90.00 to be 8960'00"
//    if (nSeconds   == 60)
//    {
//        nSeconds =  0;
//        nMinutes++;
//    }

//    if (nMinutes   == 60)
//    {
//        nMinutes =  0;
//        nDegrees++;
//    }

    if (dDegrees <  0.001   && dDegrees >  -180.0)
    {
        cDMS[0] = 'W';
    }
    else if (dDegrees > 0.001 && dDegrees < 180.0)
    {
        cDMS[0] = 'E';
    }

    int nMinutes        = (int)dMinutesSeconds;
    int nFracSeconds    = (int)((dMinutesSeconds - nMinutes) * 100.0);
    sprintf(&cDMS[2], "%03d %02d.%02d", nDegrees, nMinutes, nFracSeconds);
//    cDMS[12]         = '\0';

    var.Value(cDMS, VAR_STRING, 12);
    return var;
}

CVariant CDataConversion::DecimalDegreesToBAMS(const CVariant& rvarWhat)
{
    CVariant var(rvarWhat);
    double dDegrees = (double)var;
    long lBams = (long)(dDegrees / dBamsToDegs);

    var.Value(lBams);
	return var;
}

CVariant CDataConversion::DMSLatToDecimalDegrees(const CVariant& rvarWhat)
{
    CVariant var(rvarWhat);

    CString  str = ((_FSI_STL::string)var).c_str();

    if (str.GetLength() >= 10)
    {
        double dDegrees = (double)atoi(str.Mid(2,2));

        var.Value(_FSI_STL::string((LPCTSTR)(str.Mid(5,5))));
        var.Value(dDegrees + ((double)var)/60.0);

        if (str[0] == 'S' || str[0] == 's')
        {
            var.Value((double)var * -1.0);
        }
    }

    return var;
}

CVariant CDataConversion::DMSLatToBAMS(const CVariant& rvarWhat)
{
    CVariant var = DMSLatToDecimalDegrees(rvarWhat);
    var = DecimalDegreesToBAMS(var);

    return var;
}

CVariant CDataConversion::DMSLonToDecimalDegrees(const CVariant& rvarWhat)
{
    CVariant var(rvarWhat);

    CString  str = ((_FSI_STL::string)var).c_str();
    if (str.GetLength() >= 11)
    {
        double dDegrees = (double)atoi(str.Mid(2,3));

        var.Value(_FSI_STL::string((LPCTSTR)(str.Mid(6,5))));
        var.Value(dDegrees + ((double)var)/60.0);

        if (str[0] == 'W' || str[0] == 'w')
        {
            var.Value((double)var * -1.0);
        }
    }

    return var;
}

CVariant CDataConversion::DMSLonToBAMS(const CVariant& rvarWhat)
{
    CVariant var = DMSLonToDecimalDegrees(rvarWhat);
    var = DecimalDegreesToBAMS(var);

    return var;
}

CVariant CDataConversion::BAMSToDecimalDegrees(const CVariant& rvarWhat)
{
    CVariant var(rvarWhat);
	double dDegrees = (long)var * dBamsToDegs;

    if (dDegrees >= 180.0)
    {
        var.Value(dDegrees - 360.0);
        return var;
    }

    var.Value(dDegrees);
    return var;
}

CVariant CDataConversion::BAMSToDMSLat(const CVariant& rvarWhat)
{
    CVariant var = BAMSToDecimalDegrees(rvarWhat);
    var = DecimalDegreesToDMSLat(var);

    return var;
}

CVariant CDataConversion::BAMSToDMSLon(const CVariant& rvarWhat)
{
    CVariant var = BAMSToDecimalDegrees(rvarWhat);
    var = DecimalDegreesToDMSLon(var);

    return var;
}

CVariant CDataConversion::FahrenheitToCelsius(const CVariant& rvarWhat)
{
    CVariant var(rvarWhat);
    var.Value(((float)var - 32.0f) * 5.0f / 9.0f);

    return var;
}

CVariant CDataConversion::CelsiusToFahrenheit(const CVariant& rvarWhat)
{
    CVariant var(rvarWhat);
    var.Value((float)var * 9.0f / 5.0f + 32.0f);
    return var;
}

CVariant CDataConversion::StatuteMilesToKilometers(const CVariant& rvarWhat)
{

    CVariant var(rvarWhat);
    var.Value((float)var * MILES_TO_KM);
    return var;
}

CVariant CDataConversion::KilometersToStatuteMiles(const CVariant& rvarWhat)
{
    CVariant var(rvarWhat);
    var.Value((float)var * KM_TO_MILES);
    return var;
}

CVariant CDataConversion::InchesHgToMillibars(const CVariant& rvarWhat)
{

    CVariant var(rvarWhat);
    var.Value((float)var * INHG_TO_MB);
    return var;
}

CVariant CDataConversion::MillibarsToInchesHg(const CVariant& rvarWhat)
{

    CVariant var(rvarWhat);
    var.Value((float)var * MB_TO_INHG);
    return var;
}

CVariant CDataConversion::SignedToNormalizedAngle(const CVariant& rvarWhat)
{

    // SignedToNormalizedAngle limits an angle in the range of 0..360 degrees.
    CVariant var(rvarWhat);
    double angle = (double)var;

    if (angle >= 360.0 || angle <= -360.0)
        angle = fmod(angle, 360.0);        // limit to -360..+360

    if (angle < 0.0)                       // limit to 0..360
        angle = angle + 360.0;

    var.Value(angle);
    return var;
}


CVariant CDataConversion::NormalizedToSignedAngle(const CVariant& rvarWhat)
{
    // NormalizedToSignedAngle limits an angle in the range of -180..+180 deg.
    CVariant var(rvarWhat);
    double angle = (double)var;

    if (angle >= 360.0 || angle <= -360.0)  
        angle = fmod(angle, 360.0);         // limit to -360..+360

    if (angle > 180.0)
        angle = angle - 360.0;              // limit max to 180.0

    else if (angle < -180.0)
        angle = angle + 360.0;              // limit min to -180.0

    var.Value(angle);
    return var;
}



CVariant CDataConversion::SecondsToTimeFormat(const CVariant& rvarWhat)
{
   unsigned long ulHours, ulMinutes, ulSeconds;

   CVariant var;
   var.ChangeType(9);

   CString strCaption;
   
   unsigned long ulAllSeconds = (unsigned long)rvarWhat;

   ulHours = ulAllSeconds / 3600;
   ulMinutes = (ulAllSeconds % 3600) / 60;
   ulSeconds = (ulAllSeconds % 60);
   strCaption.Format("%02d:%02d:%02d",ulHours,
                                     ulMinutes,
                                     ulSeconds);
   _FSI_STL::string stlStrCaption((LPCTSTR)strCaption);
   var.Value(stlStrCaption);

   return var;
}


CVariant CDataConversion::MinutesToTimeFormat(const CVariant& rvarWhat)
{
   unsigned long ulHours, ulMinutes, ulSeconds;

   CVariant var;
   var.ChangeType(9);

   CString strCaption;

   float fAllSeconds =  (float)rvarWhat   *  60.0;

   ulHours     =  fAllSeconds / 3600;
   ulMinutes   =  fmod(fAllSeconds, 3600.0)  /  60;
   ulSeconds   =  fmod(fAllSeconds, 60.0);

   strCaption.Format("%02d:%02d:%02d", ulHours, ulMinutes, ulSeconds);
   _FSI_STL::string stlStrCaption((LPCTSTR)strCaption);
   var.Value(stlStrCaption);

   return var;
}


CVariant CDataConversion::IntX1000ToFloat2T(const CVariant& rvarWhat)
{
   long     requested_int_x1000  =  (long)rvarWhat;
   float    desired_float        =  (float)(requested_int_x1000)/1000.0f;
   CString  desired_output("---.--");

   if (desired_float)
   {
      // The format %5.4lf would show four places after the decimal, but %5.2lf would
      // cause the value to be rounded before showing two places after the decimal.
      // The following code truncates the value to two places after the decimal without rounding.
      CString  tmpvalue;
      tmpvalue.Format("%5.4lf",  desired_float);
      desired_output.Format("%.6s", tmpvalue);
   }

   CVariant var;
   var.ChangeType(9);
   _FSI_STL::string stl_desired_output((LPCTSTR)desired_output);
   var.Value(stl_desired_output);

   return   var;
}


CVariant CDataConversion::IntX100ToFloat2T(const CVariant& rvarWhat)
{
   long     requested_int_x100  =  (long)rvarWhat;
   float    desired_float        =  (float)(requested_int_x100)/100.0f;
   CString  desired_output("---.--");

   if (desired_float)
   {
      // The format %5.4lf would show four places after the decimal, but %5.2lf would
      // cause the value to be rounded before showing two places after the decimal.
      // The following code truncates the value to two places after the decimal without rounding.
      CString  tmpvalue;
      tmpvalue.Format("%5.4lf",  desired_float);
      desired_output.Format("%.6s", tmpvalue);
   }

   CVariant var;
   var.ChangeType(9);
   _FSI_STL::string stl_desired_output((LPCTSTR)desired_output);
   var.Value(stl_desired_output);

   return   var;
}