/////////////////////////////////////////////////////////////////////////////
//
//           F L I G H T S A F E T Y   I N T E R N A T I O N A L
//                     Simulation Systems Division
//                      2700 North Hemlock Circle
//                     Broken Arrow, Oklahoma 74012
//                          (918) 259-4000
/////////////////////////////////////////////////////////////////////////////
//
// DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
// Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
// technical or operational data or information from automatic dissemination
// under the International Exchange Program or by other means.  This protection
// covers information required solely for administrative or operational
// purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
//
// WARNING:  This document contains technical data whose export is restricted
// by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
// Executive Order 12470.  Violation of these export control laws is subject
// to severe criminal penalties.  Dissemination of this document is controlled
// under DOD Directive 5230.25
//
/////////////////////////////////////////////////////////////////////////////
//
//
// Filename         : ColorPage.cpp
//
// Date             : 11 November 1998
//
// Engineer         : Billy Baker
//
// Revision         : $Revision: 1.3 $
//
// Description      : ColorPage.cpp contains the implementation of both 
//                    the CColorStatic and CColorPage classes.  
//                    Normally, two classes would not occupy the same 
//                    files.  However, the CColorStatic class is 
//                    special and should only be used with the 
//                    CColorPage class.  To use the CColorStatic class 
//                    with another class, the OnClicked method would 
//                    most likely need to be changed.
//                    
//                    The CColorStatic class is derived from CStatic 
//                    and is used to show a color and changes 
//                    custom/new color in the CColorPage class.
//                    
//                    The CColorPage class is derived from CBasePage is 
//                    serves as the color property page for any widget 
//                    that needs to change a color of some aspect of 
//                    the widget.  
//
// Classification   : UNCLASSIFIED
//
// Requirements     : CBasePage is required for CColorPage.  Both 
//                    CBasePage and CStatic, the base of CColorStatic, 
//                    are either a part of MFC or rely on MFC.  
//                    Likewise, CColorStatic and CColorPage make use of 
//                    the CColor class.
//
// Components Used  : _FSI_STL::string, _FSI_STL::map, Core::CColor,
//                    Core::CBasePage, 
//
// Operational 
//    Restrictions  : Machine dependencies/restrictions
//                        None.
//                    Design dependencies/restrictions
//                        None.
//                    Operations containing dependencies/restrictions
//                        CColorStatic::OnClicked relies on CColorPage.
//                    Compiler dependencies/restrictions
//                        None.
//                    Other conditions for proper execution
//                        None.
//
// Environment      : Operating system(s) - Microsoft Windows NT 4.0 with
//                                              NT service pack 3
//                                          Microsoft Windows NT 5.0
//
//                    Compiler(s) - Visual C++ 5.0 with VisC++ service pack 3
//                                  Visual C++ 6.0
//
//                    Architechure(s) - Intel Pentium, Pentium II
/////////////////////////////////////////////////////////////////////////////

/////////////////////////////////////////////////////////////////////////////
//
//                              R e v i s i o n   H i s t o r y
//
/////////////////////////////////////////////////////////////////////////////
// $Log: ColorPage.cpp $
// Revision 1.3  2000/03/07 07:21:02  billyb
// Removed unused methods.  Changed calls to 
// AddElementValue.
// Revision 1.2  1999/09/07 19:20:17  billyb
// Added comments. Changed a couple of casts to use either
// dynamic_cast or reinterpret_cast.  Got rid of code 
// dealing only with front and back color attributes 
// and made it generic for any attribute that the
// parent widget defines.
/////////////////////////////////////////////////////////////////////////////
#include "stdafx.h"
#include "ColorPage.h"

#include "Widget.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

/////////////////////////////////////////////////////////////////////////////
// CColorStatic helper
/////////////////////////////////////////////////////////////////////////////
//
// CColorStatic::CColorStatic
//
// Inputs           : None.
//
// Return Values    : None.
//
// Date             : 11 November 1998
//
// Engineer         : Billy Baker
//
// Description      : Default constructor for creating a CColorStatic 
//                    object that is derived from CStatic.
//
/////////////////////////////////////////////////////////////////////////////
CColorStatic::CColorStatic() : CStatic()
{
    m_brush.CreateSolidBrush(m_color);
}

/////////////////////////////////////////////////////////////////////////////
//
// CColorStatic::~CColorStatic()
//
// Inputs           : None.
//
// Return Values    : None.
//
// Date             : 11 November 1998
//
// Engineer         : Billy Baker
//
// Description      : Default destructor that deletes the resources for 
//                    the solid brush.
//
/////////////////////////////////////////////////////////////////////////////
CColorStatic::~CColorStatic()
{
    m_brush.DeleteObject();
}

/////////////////////////////////////////////////////////////////////////////
//
// CColor CColorStatic::Color()
//
// Inputs           : None.
//
// Return Values    : A CColor object with the current background color 
//                    for the CColorStatic object.
//
// Date             : 11 November 1998
//
// Engineer         : Billy Baker
//
// Description      : Accessor method for getting the current 
//                    background color of the CColorStatic object.
//
/////////////////////////////////////////////////////////////////////////////
CColor CColorStatic::Color()
{
    return m_color;
}

/////////////////////////////////////////////////////////////////////////////
//
// void CColorStatic::Color()
//
// Inputs           : const CColor& rColor - a constant reference to a 
//                    new background color.
//
// Return Values    : None.
//
// Date             : 11 November 1998
//
// Engineer         : Billy Baker
//
// Description      : Mutator method for changing the background color 
//                    of the CColorStatic object.  The CBrush member 
//                    variable must be deleted and then created again 
//                    with the new background color.
//
/////////////////////////////////////////////////////////////////////////////
void CColorStatic::Color(const CColor& rColor)
{
    m_color = rColor;
    m_brush.DeleteObject();
    m_brush.CreateSolidBrush(m_color);
}

BEGIN_MESSAGE_MAP(CColorStatic, CStatic)
	//{{AFX_MSG_MAP(CColorStatic)
	ON_CONTROL_REFLECT(BN_CLICKED, OnClicked)
	ON_WM_CTLCOLOR_REFLECT()
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
//
// void CColorStatic::OnClicked()
//
// Inputs           : None.
//
// Return Values    : None.
//
// Date             : 11 November 1998
//
// Engineer         : Billy Baker
//
// Description      : Handles the BN_CLICKED message for a CColorStatic 
//                    button.  Each CColorStatic object must have 
//                    Notify set when it is created otherwise the 
//                    parent will handle all messages.  The behavior 
//                    here is specific to the CColorPage--it sets the 
//                    custom/new color in the CColorPage object.
//
/////////////////////////////////////////////////////////////////////////////
void CColorStatic::OnClicked() 
{
    CWnd* pWnd = GetParent();
    CColorPage* pPage = dynamic_cast<CColorPage*>(pWnd);

    if (pPage != NULL)
    {
        pPage->CustomColor(m_color);
    }
}

/////////////////////////////////////////////////////////////////////////////
//
// HBRUSH CColorStatic::CtlColor()
//
// Inputs           : CDC* pDC - the current drawing context.
//                    UINT nCtlColor - which type of control.
//
// Return Values    : A handle to a brush that will be used as the new 
//                    background color.
//
// Date             : 11 November 1998
//
// Engineer         : Billy Baker
//
// Description      : Overridden method to change the background color 
//                    of a CColorStatic object to the stored background 
//                    color in m_color.
//
/////////////////////////////////////////////////////////////////////////////
HBRUSH CColorStatic::CtlColor(CDC* pDC, UINT nCtlColor) 
{
    pDC->SetBkColor(m_color);
    pDC->SetTextColor(m_color);
	
	// TODO: Return a non-NULL brush if the parent's handler should not 
    // be called
	return m_brush;
}

/////////////////////////////////////////////////////////////////////////////
// CColorPage property page

/////////////////////////////////////////////////////////////////////////////
//
// CColorPage::ColorPage()
//
// Inputs           : None.
//
// Return Values    : None.
//
// Date             : 11 November 1998
//
// Engineer         : Billy Baker
//
// Description      : Default constructor for the CColorPage class.  
//                    Besides control initialization, the constructor 
//                    sets the name of the property page 
//                    (m_stlStrPageName in CBasePage).
//
/////////////////////////////////////////////////////////////////////////////
CColorPage::CColorPage() : CBasePage(CColorPage::IDD)
{
	//{{AFX_DATA_INIT(CColorPage)
	m_strRedValue = _T("");
	m_strGreenValue = _T("");
	m_strBlueValue = _T("");
	//}}AFX_DATA_INIT
    m_stlStrPageName = _FSI_STL::string("Color");
}

void CColorPage::DoDataExchange(CDataExchange* pDX)
{
	CBasePage::DoDataExchange(pDX);
	//{{AFX_DATA_MAP(CColorPage)
	DDX_Control(pDX, IDC_GLOBAL_COLORS, m_comboGlobals);
	DDX_Control(pDX, IDC_COLOR_FACE_COMBO, m_ctrlFaceCombo);
	DDX_Control(pDX, IDC_RED_SCROLLBAR, m_ctrlRedBar);
	DDX_Control(pDX, IDC_GREEN_SCROLLBAR, m_ctrlGreenBar);
	DDX_Control(pDX, IDC_BLUE_SCROLLBAR, m_ctrlBlueBar);
	DDX_Text(pDX, IDC_RED_VALUE, m_strRedValue);
	DDX_Text(pDX, IDC_GREEN_VALUE, m_strGreenValue);
	DDX_Text(pDX, IDC_BLUE_VALUE, m_strBlueValue);
	//}}AFX_DATA_MAP

    // Extra code for setting the ranges on the scroll bars.
    // WorldCom MFC library inspired this addition.
    if (pDX->m_bSaveAndValidate == FALSE)
    {
        m_ctrlRedBar.SetScrollRange(0, 255, FALSE);
        m_ctrlGreenBar.SetScrollRange(0, 255, FALSE);
        m_ctrlBlueBar.SetScrollRange(0, 255, FALSE);
    }
}


BEGIN_MESSAGE_MAP(CColorPage, CBasePage)
	//{{AFX_MSG_MAP(CColorPage)
	ON_WM_HSCROLL()
	ON_CBN_SELCHANGE(IDC_COLOR_FACE_COMBO, OnSelchangeColorFace)
	ON_CBN_SELCHANGE(IDC_GLOBAL_COLORS, OnSelchangeGlobalColors)
	ON_CBN_SETFOCUS(IDC_COLOR_FACE_COMBO, OnSetfocusColorFaceCombo)
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CColorPage message handlers
/////////////////////////////////////////////////////////////////////////////
//
// CBasePage* CColorPage::CreateObject()
//
// Inputs           : None.
//
// Return Values    : A CBasePage pointer to a newly created CColorPage.
//
// Date             : 11 November 1998
//
// Engineer         : Billy Baker
//
// Description      : Creates a new CColorPage and returns a base 
//                    pointer to the new CColorPage.  Each widget has a 
//                    list of CreateObject pointers for each property 
//                    page that the widget needs.  The main framework 
//                    traverses the list to call each CreateObject 
//                    method and create the property pages.  Each 
//                    CreateObject method is static and thus the same for
//                    every instance of CColorPage.
//
/////////////////////////////////////////////////////////////////////////////
CBasePage* CColorPage::CreateObject()
{
    return new CColorPage;
}

/////////////////////////////////////////////////////////////////////////////
//
// void CColorPage::SetProperties()
//
// Inputs           : CWidget* pWidget - a pointer to the CWidget 
//                    instance for which property display is  requested.
//
// Return Values    : None.
//
// Date             : 11 November 1998
//
// Engineer         : Billy Baker
//
// Description      : SetProperties is common to all Widget property 
//                    pages.  It is used to initialize the member 
//                    variables of the property page with values from 
//                    the CWidget instance.
//
/////////////////////////////////////////////////////////////////////////////
void CColorPage::SetProperties(void* pWidget)
{
    CWidget* pWidgetLocal = reinterpret_cast<CWidget*>(pWidget);
    if (pWidgetLocal == NULL)
    {
        return;
    }

    CXMLWidget* pXMLWidget = pWidgetLocal->XMLWidget();
    if (pXMLWidget != NULL)
    {
        // XML traversal
        CXMLElement* pXMLElement = NULL;
        POSITION pos = NULL;

        // For gatting data from the XML data structures.
        CString strValue;
        STRING2STRING_MAP::iterator s2sIt;
        CString strGlobal;

        while (pXMLWidget->FindElement(pXMLElement, 
                                       pos, 
                                       _FSI_STL::string("COLOR")) == true)
        {
            // If no GLOBALCOLOR, then colorGlobal should have its default
            // settings.
            CColor colorGlobal;

            strValue    = pXMLElement->ElementValue().c_str();

            strGlobal   = "Not Used";

            // Check for the GLOBALCOLOR attribute
            if (pXMLElement->FindAttribute(s2sIt,
                                      _FSI_STL::string("GLOBALCOLOR")) == true)
            {
                strGlobal   = (*s2sIt).second.c_str();
                colorGlobal = CColor::GlobalColor(strGlobal);
            }

            // Check for the PLANE attribute which is the face or some other
            // aspect of the parent widget that can have custom colors.
            if (pXMLElement->FindAttribute(s2sIt,
                                           _FSI_STL::string("PLANE")) == true)
            {
                if (strGlobal != "Not Used")
                {
                    m_mapColorAttributes[(*s2sIt).second].s_colorCurrent = 
                                                                   colorGlobal;

                    m_mapColorAttributes[(*s2sIt).second].s_colorCustom = 
                                                                   colorGlobal;

                    m_mapColorAttributes[(*s2sIt).second].s_colorGlobal = 
                                                                   colorGlobal;
                }
                else
                {
                    m_mapColorAttributes[(*s2sIt).second].s_colorCurrent = 
                                                              CColor(strValue);

                    m_mapColorAttributes[(*s2sIt).second].s_colorCustom = 
                                                              CColor(strValue);

                    m_mapColorAttributes[(*s2sIt).second].s_colorGlobal = 
                                                              CColor(strValue);
                }

                m_mapColorAttributes[(*s2sIt).second].s_stlStrGlobal = 
                                          _FSI_STL::string((LPCTSTR)strGlobal);
            }
        }
    }
}

/////////////////////////////////////////////////////////////////////////////
//
// void CColorPage::UpdateProperties()
//
// Inputs           : CWidget* pWidget - a pointer to the CWidget 
//                    instance for which property updating is  requested.
//
// Return Values    : None.
//
// Date             : 12 November 1998
//
// Engineer         : Billy Baker
//
// Description      : UpdateProperties is common to all Widget property 
//                    pages.  It is used to change the values of 
//                    the CWidget instance.
//
/////////////////////////////////////////////////////////////////////////////
void CColorPage::UpdateProperties(void* pWidget)
{
    CWidget* pWidgetLocal = reinterpret_cast<CWidget*>(pWidget);
    if (pWidgetLocal == NULL)
    {
        return;
    }

    CXMLWidget* pXMLWidget = pWidgetLocal->XMLWidget();
    if (pXMLWidget != NULL && GetSafeHwnd() != NULL)
    {
        UpdateData(TRUE);

        CString strAttribute;
        if (m_ctrlFaceCombo.GetCurSel() != CB_ERR)
        {
	        m_ctrlFaceCombo.GetLBText(m_ctrlFaceCombo.GetCurSel(), 
                                      strAttribute);
        }

        // Update what is shown on the screen.
        m_ctrlColorCurrent.Color(m_ctrlColorCustom.Color());
        m_ctrlColorCurrent.Invalidate();

        // Update the data structure containing data for all attributes.
        _FSI_STL::string stlStrAttribute = 
                                    _FSI_STL::string((LPCTSTR)strAttribute);

        if (m_mapColorAttributes.find(stlStrAttribute) !=
                                                    m_mapColorAttributes.end())
        {
            // Update the previous selection's colors.
            m_mapColorAttributes[stlStrAttribute].s_colorCurrent = 
                                                    m_ctrlColorCurrent.Color();

            m_mapColorAttributes[stlStrAttribute].s_colorCustom =
                                                    m_ctrlColorCustom.Color();
        }

        // XML Traversal
        CXMLElement*                pXMLElement     = NULL;
        POSITION                    pos             = NULL;

        // for writing data to the XML data structures.
        CString                     strValue;
        CString                     strUpr;
        _FSI_STL::string            stlStrUpr;
        STRING2STRING_MAP::iterator s2sIt;
        CColor                      color;

        _FSI_STL::list<_FSI_STL::string>& listColorSelectable = 
                                    ((CWidget*)m_pWidget)->ColorSelectable();
        _FSI_STL::list<_FSI_STL::string>::iterator lIt = 
                                                 listColorSelectable.begin();

        // Walk through all of the colors that the parent widget might have
        // and give them colors in the XML data.
        for (; lIt != listColorSelectable.end(); lIt++)
        {
            if (pXMLWidget->FindElement(pXMLElement, 
                                        pos, 
                                        _FSI_STL::string("COLOR")) == true)
            {
                // Make sure that the attribute/face name is in all caps.
                strUpr      = (*lIt).c_str();
                strUpr.MakeUpper();

                stlStrUpr   = _FSI_STL::string((LPCTSTR)strUpr);
                color       = m_mapColorAttributes[stlStrUpr].s_colorCustom;

                // Get the color as rrrgggbbb
                strValue.Format("%03d%03d%03d", color.Red(), 
                                                color.Green(), 
                                                color.Blue());

                pXMLElement->AddElementValue(_FSI_STL::string((LPCTSTR)strValue));

                if (pXMLElement->FindAttribute(s2sIt,
                                            _FSI_STL::string("PLANE")) == true)
                {
                    (*s2sIt).second = stlStrUpr;
                }

                if (pXMLElement->FindAttribute(s2sIt,
                                      _FSI_STL::string("GLOBALCOLOR")) == true)
                {
                    (*s2sIt).second = m_mapColorAttributes[stlStrUpr].s_stlStrGlobal;
                }
            }
        }
    }
}

/////////////////////////////////////////////////////////////////////////////
//
// BOOL CColorPage::OnInitDialog()
//
// Inputs           : None.
//
// Return Values    : Success of OnInitDialog code.
//
// Date             : 11 November 1998
//
// Engineer         : Billy Baker
//
// Description      : CColorPage::OnInitDialog() is an override to 
//                    initialize parts of the CColorPage property page. 
//                    A CColorStatic object is attached to each CStatic 
//                    item from the dialog template using 
//                    SubclassDlgItem and the color of the CColorStatic 
//                    items are set.  Also, CColorStatic objects are attached
//                    to the current color display and the new/custom color
//                    display.
//
/////////////////////////////////////////////////////////////////////////////
BOOL CColorPage::OnInitDialog() 
{
	CBasePage::OnInitDialog();

    unsigned char ucStep    = 0;
    CWnd* pColorStatic   = NULL;
    CRect rectFirstBox;

    // Size the first CStatic to the right size.  The dialog template has its
    // upper left corner in the correct position but the size is slightly off.
    // Using a CWnd* is okay because the function calls are the same for a
    // CStatic.
    pColorStatic = GetDlgItem(IDC_COLOR_BOX_1);
    if (pColorStatic != NULL)
    {
        pColorStatic->SetWindowPos(NULL,1,1,20,17,SWP_NOMOVE | SWP_NOZORDER);
        pColorStatic->GetWindowRect(rectFirstBox);
        ScreenToClient(&rectFirstBox);

        int nOffsetWidth    = rectFirstBox.Width() + 5;
        int nOffsetHeight   = rectFirstBox.Height() + 5;

        // Step through all of the buttons and attach a CColorStatic object to
        // each and set its color.
        for (; ucStep < NUMBUTTONS; ucStep++)
        {
            pColorStatic = GetDlgItem(m_unIDs[ucStep]);
            if (pColorStatic != NULL)
            {
                // Place the boxes in the correct positions.  Note, the 
                // (ucStep/8)*8 must remain as is.  ucStep/8 with be 0 for
                // the first row, 1 for the second, 2 for the third, etc.  
                // Thus, 8 will be subtracted from each on the second row, 
                // 16 for each on the third, and so on.
                pColorStatic->SetWindowPos(NULL, 
                             rectFirstBox.left + 
                                  (ucStep - (ucStep / 8) * 8) * nOffsetWidth,
                             rectFirstBox.top + (ucStep / 8) * nOffsetHeight,
                             20, 17, SWP_NOZORDER);

                m_ctrlColorBoxes[ucStep].SubclassDlgItem(m_unIDs[ucStep], this);
                m_ctrlColorBoxes[ucStep].Color(CColor(m_clrefColors[ucStep]));
                m_ctrlColorBoxes[ucStep].Invalidate(FALSE);
            }
        }
    }


    m_ctrlColorCurrent.SubclassDlgItem(IDC_CURRENT_COLOR, this);
    m_ctrlColorCustom.SubclassDlgItem(IDC_CUSTOM_COLOR, this);

    m_ctrlFaceCombo.ResetContent();
    if (m_pWidget != NULL)
    {
        _FSI_STL::list<_FSI_STL::string>& listColorSelectable = 
                                    ((CWidget*)m_pWidget)->ColorSelectable();
        _FSI_STL::list<_FSI_STL::string>::iterator lIt = 
                                                 listColorSelectable.begin();

        CString strUpr;
        _FSI_STL::string stlStrUpr;
        for (; lIt != listColorSelectable.end(); lIt++)
        {
            // Make sure all of the attribute/face names are in caps.
            strUpr      = (*lIt).c_str();
            strUpr.MakeUpper();

            m_ctrlFaceCombo.AddString(strUpr);
            stlStrUpr   = _FSI_STL::string((LPCTSTR)strUpr);

            if (m_mapColorAttributes.find(stlStrUpr) == m_mapColorAttributes.end())
            {
                m_mapColorAttributes[stlStrUpr].s_stlStrGlobal = 
                                                  _FSI_STL::string("Not Used");
            }
        }

        if (m_mapColorAttributes.size() > 0)
        {
            m_ctrlFaceCombo.SelectString(-1,(*(m_mapColorAttributes.begin())).first.c_str());
        }
    }

    if (m_mapColorAttributes.size() > 0)
    {
        m_ctrlColorCurrent.Color((*(m_mapColorAttributes.begin())).second.s_colorCurrent);
        m_ctrlColorCustom.Color((*(m_mapColorAttributes.begin())).second.s_colorCustom);
    }

    CustomColor(m_ctrlColorCustom.Color());

    // Initialize the global color combo control
    m_comboGlobals.ResetContent();

    HKEY hKey;
    RegOpenKeyEx(HKEY_LOCAL_MACHINE, _T("Software\\FlightSafety\\FSISuite\\Settings"), 0, KEY_READ, &hKey);

    if (hKey != NULL)
    {
        long int lGlobalColorCount = 0;
        long int lGlobalColorStep = 1;
        unsigned long int ulData = 0;
        DWORD dwSize = sizeof (unsigned long int);
        RegQueryValueEx(hKey, _T("GlobalColorCount"), NULL, NULL, 
                        (BYTE*)&ulData, &dwSize);
        lGlobalColorCount = ulData;

        CString strGlobalColorXXX;
        DWORD dwType, dwCount;
        CString strValue = "";
        LONG lResult;

        for (; lGlobalColorStep <= lGlobalColorCount; lGlobalColorStep++)
        {
            strGlobalColorXXX.Format("GlobalColor%d",lGlobalColorStep);

            // Get the name of the GlobalColor.
		    strValue = "";
		    lResult = RegQueryValueEx(hKey, (LPCTSTR)strGlobalColorXXX, 
                                      NULL, &dwType, NULL, &dwCount);
		    if (lResult == ERROR_SUCCESS)
		    {
			    ASSERT(dwType == REG_SZ);
			    lResult = RegQueryValueEx(hKey, (LPCTSTR)strGlobalColorXXX, 
                                          NULL, &dwType,
				             (LPBYTE)strValue.GetBuffer(dwCount/sizeof(TCHAR)),
                                          &dwCount);

			    strValue.ReleaseBuffer();
		    }

            m_comboGlobals.AddString(strValue);
        }

        RegCloseKey(hKey);
    }

    m_comboGlobals.AddString("Not Used");
    if (m_mapColorAttributes.size() > 0)
    {
        CString strGlobal = 
               (*(m_mapColorAttributes.begin())).second.s_stlStrGlobal.c_str();

        m_comboGlobals.SelectString(-1, strGlobal);
    }
    else
    {
        m_comboGlobals.SelectString(-1, "Not Used");
    }

	return TRUE;  // return TRUE unless you set the focus to a control
	              // EXCEPTION: OCX Property Pages should return FALSE
}

/////////////////////////////////////////////////////////////////////////////
//
// void CColorPage::CurrentColor()
//
// Inputs           : const CColor& rColor - the new color for the 
//                    current color.
//
// Return Values    : None.
//
// Date             : 11 November 1998
//
// Engineer         : Billy Baker
//
// Description      : Mutator method that changes the color of the 
//                    display of the current color.  The scroll bars 
//                    and the text representing the values of the 
//                    scroll bars are also updated.
//
/////////////////////////////////////////////////////////////////////////////
void CColorPage::CurrentColor(const CColor& rColor)
{
    // Set the current color display
    m_ctrlColorCurrent.Color(rColor);
    m_ctrlColorCurrent.Invalidate(FALSE);

    // Set the sliders
    m_ctrlRedBar.SetScrollPos(m_ctrlColorCurrent.Color().Red());
    m_ctrlGreenBar.SetScrollPos(m_ctrlColorCurrent.Color().Green());
    m_ctrlBlueBar.SetScrollPos(m_ctrlColorCurrent.Color().Blue());

    // Set the text under the sliders
    m_strRedValue.Format("%d",m_ctrlColorCurrent.Color().Red());
    m_strGreenValue.Format("%d",m_ctrlColorCurrent.Color().Green());
    m_strBlueValue.Format("%d",m_ctrlColorCurrent.Color().Blue());

    UpdateData(FALSE);
}

/////////////////////////////////////////////////////////////////////////////
//
// void CColorPage::CustomColor()
//
// Inputs           : const CColor& rColor - the new color for the 
//                    custom color.
//
// Return Values    : None.
//
// Date             : 11 November 1998
//
// Engineer         : Billy Baker
//
// Description      : Mutator method that changes the color of the 
//                    display of the custom color.  The scroll bars 
//                    and the text representing the values of the 
//                    scroll bars are also updated.
//
/////////////////////////////////////////////////////////////////////////////
void CColorPage::CustomColor(const CColor& rColor)
{
    CString strAttribute;
    if (m_ctrlFaceCombo.GetCurSel() != CB_ERR)
    {
        m_ctrlFaceCombo.GetLBText(m_ctrlFaceCombo.GetCurSel(), strAttribute);

        // Set the custom color display
        m_ctrlColorCustom.Color(rColor);
        m_ctrlColorCustom.Invalidate(FALSE);

        // Set the sliders
        m_ctrlRedBar.SetScrollPos(m_ctrlColorCustom.Color().Red());
        m_ctrlGreenBar.SetScrollPos(m_ctrlColorCustom.Color().Green());
        m_ctrlBlueBar.SetScrollPos(m_ctrlColorCustom.Color().Blue());

        // Set the text under the sliders
        m_strRedValue.Format("%d",m_ctrlColorCustom.Color().Red());
        m_strGreenValue.Format("%d",m_ctrlColorCustom.Color().Green());
        m_strBlueValue.Format("%d",m_ctrlColorCustom.Color().Blue());

    
        // If the color is not the same as the global color for the
        // given face, then reset the global color string to Not Used.
        _FSI_STL::string stlStrAttribute = 
                                    _FSI_STL::string((LPCTSTR)strAttribute);

        // Make sure that the new selection is valid.
        if (m_mapColorAttributes.find(stlStrAttribute) !=
                                                  m_mapColorAttributes.end())
        {
            if (!(m_mapColorAttributes[stlStrAttribute].s_colorGlobal == 
                                                                       rColor))
            {
                m_mapColorAttributes[stlStrAttribute].s_stlStrGlobal = 
                                                  _FSI_STL::string("Not Used");

                m_comboGlobals.SelectString(-1, "Not Used");
            }
        }
    }
    
    UpdateData(FALSE);
}

/////////////////////////////////////////////////////////////////////////////
//
// void CColorPage::OnHScroll()
//
// Inputs           : UINT nSBCode - the type of movement that occurred.
//                    UINT nPos - current position of the slider when 
//                         nSBCode is SB_THUMBTRACK or SB_THUMBPOSITION.
//                    CScrollBar* pScrollBar - a pointer to the 
//                         scrollbar that was just moved.
//
// Return Values    : None.
//
// Date             : 11 November 1998
//
// Engineer         : Billy Baker
//
// Description      : Message handler for all three of the color 
//                    component scrollbars.  The sliders are moved 
//                    within 0 - 255 and then custom color is updated.
//
/////////////////////////////////////////////////////////////////////////////
void CColorPage::OnHScroll(UINT nSBCode, UINT nPos, CScrollBar* pScrollBar) 
{
    if (pScrollBar != NULL)
    {
        int pos = pScrollBar->GetScrollPos();
        switch(nSBCode)
        {
            // clicked in the between the slider and the end arrows
        case SB_PAGELEFT:
            {
                if ((pos - 32) < 0)
                {
                    pScrollBar->SetScrollPos(0);
                }
                else
                {
                    pScrollBar->SetScrollPos(pos-32);
                }
            }
            break;

            // clicked in the between the slider and the end arrows
        case SB_PAGERIGHT:
            {
                if ((pos + 32) > 255)
                {
                    pScrollBar->SetScrollPos(255);
                }
                else
                {
                    pScrollBar->SetScrollPos(pos+32);
                }
            }
            break;

            // clicked on the left arrow
        case SB_LINELEFT:
            {
                if (pos > 0)
                {
                    pScrollBar->SetScrollPos(--pos);
                }
            }
            break;

            // clicked on the right arrow
        case SB_LINERIGHT:
            {
                if (pos < 255)
                {
                    pScrollBar->SetScrollPos(++pos);
                }
            }
            break;

            // moving the slider
        case SB_THUMBTRACK:
            {
                pScrollBar->SetScrollPos(nPos);
            }
            break;
        }

        // Reset the custom color display
        CustomColor(CColor(RGB(m_ctrlRedBar.GetScrollPos(), 
                               m_ctrlGreenBar.GetScrollPos(),
                               m_ctrlBlueBar.GetScrollPos())));
    }
    else
    {
	    CBasePage::OnHScroll(nSBCode, nPos, pScrollBar);
    }
}

/////////////////////////////////////////////////////////////////////////////
//
// void CColorPage::OnSelchangeColorFace()
//
// Inputs           : None.
//
// Return Values    : None.
//
// Date             : 06 September 1999
//
// Engineer         : Billy Baker
//
// Description      : OnSelchangeColorFace is a callback for when a 
//                    new selection is made in the color face/attribute
//                    combobox.
//
/////////////////////////////////////////////////////////////////////////////
void CColorPage::OnSelchangeColorFace() 
{
    UpdateData(TRUE);
    CString strAttribute;
    if (m_ctrlFaceCombo.GetCurSel() != CB_ERR)
    {
	    m_ctrlFaceCombo.GetLBText(m_ctrlFaceCombo.GetCurSel(), strAttribute);
    }

    // Update the data structure containing data for all attributes.
    _FSI_STL::string stlStrAttribute = _FSI_STL::string((LPCTSTR)strAttribute);

    _FSI_STL::string stlStrPrevAttribute = 
                                     _FSI_STL::string((LPCTSTR)m_strFocusFace);

    if (m_mapColorAttributes.find(stlStrPrevAttribute) !=
                                                  m_mapColorAttributes.end())
    {
        // Update the previous selection's colors.
        m_mapColorAttributes[stlStrPrevAttribute].s_colorCurrent = 
                                                m_ctrlColorCurrent.Color(); 

        m_mapColorAttributes[stlStrPrevAttribute].s_colorCustom =
                                                m_ctrlColorCustom.Color();

        if (m_comboGlobals.GetCurSel() != CB_ERR)
        {
            CString strPrevGlobal;
            m_comboGlobals.GetLBText(m_comboGlobals.GetCurSel(), 
                                     strPrevGlobal);

            m_mapColorAttributes[stlStrPrevAttribute].s_stlStrGlobal = 
                                  _FSI_STL::string((LPCTSTR)strPrevGlobal);
        }
    }

    // Make sure that the new selection is valid.
    if (m_mapColorAttributes.find(stlStrAttribute) !=
                                              m_mapColorAttributes.end())
    {
        // Update the controls with the new selection's colors.
        m_ctrlColorCurrent.Color(m_mapColorAttributes[stlStrAttribute].s_colorCurrent);
        m_ctrlColorCustom.Color(m_mapColorAttributes[stlStrAttribute].s_colorCustom);

        // Update the global combo with the global selection for the new 
        // attribute's global color setting.
        m_comboGlobals.SelectString(-1, m_mapColorAttributes[stlStrAttribute].s_stlStrGlobal.c_str());
    }

    m_ctrlColorCurrent.Invalidate();

    m_strFocusFace = strAttribute;

    CustomColor(m_ctrlColorCustom.Color());
}

/////////////////////////////////////////////////////////////////////////////
//
// void CColorPage::OnSelchangeGlobalColors()
//
// Inputs           : None.
//
// Return Values    : None.
//
// Date             : 06 September 1999
//
// Engineer         : Billy Baker
//
// Description      : OnSelchangeGlobalColors is a callback for when a 
//                    new selection is made in the global colors combobox.
//
/////////////////////////////////////////////////////////////////////////////
void CColorPage::OnSelchangeGlobalColors() 
{
    UpdateData(TRUE);		

    // Get the global color now used.
    CString strSelection;
    if (m_comboGlobals.GetCurSel() != CB_ERR)
    {
	    m_comboGlobals.GetLBText(m_comboGlobals.GetCurSel(), strSelection);
    }

    // Get the attribute to which the new global color needs to be applied.
    CString strAttribute;
    if (m_ctrlFaceCombo.GetCurSel() != CB_ERR)
    {
	    m_ctrlFaceCombo.GetLBText(m_ctrlFaceCombo.GetCurSel(), strAttribute);

        // Reset the color displays for this attribute.
        CColor colorGlobal = CColor::GlobalColor(strSelection);

        m_ctrlColorCurrent.Color(colorGlobal);
        m_ctrlColorCurrent.Invalidate();
        m_ctrlColorCustom.Color(colorGlobal);

        // Update the data structure containing data for all attributes.
        _FSI_STL::string stlStrAttribute = 
                                       _FSI_STL::string((LPCTSTR)strAttribute);

        if (m_mapColorAttributes.find(stlStrAttribute) !=
                                                    m_mapColorAttributes.end())
        {
            m_mapColorAttributes[stlStrAttribute].s_stlStrGlobal = 
                                       _FSI_STL::string((LPCTSTR)strSelection);

            m_mapColorAttributes[stlStrAttribute].s_colorGlobal = colorGlobal;

        }

        CustomColor(m_ctrlColorCustom.Color());
    }
}

/////////////////////////////////////////////////////////////////////////////
//
// void CColorPage::OnSetfocusColorFaceCombo()
//
// Inputs           : None.
//
// Return Values    : None.
//
// Date             : 07 September 1999
//
// Engineer         : Billy Baker
//
// Description      : OnSetfocusColorFaceCombo is a callback for when 
//                    the combobox with the list of attributes/faces 
//                    recieves focus.  This makes sure that the 
//                    m_strFocusFace member always has the last value 
//                    in the combobox.
//
/////////////////////////////////////////////////////////////////////////////
void CColorPage::OnSetfocusColorFaceCombo() 
{
    if (m_ctrlFaceCombo.GetCurSel() != CB_ERR)
    {
        m_ctrlFaceCombo.GetLBText(m_ctrlFaceCombo.GetCurSel(), m_strFocusFace);
    }
}
