/////////////////////////////////////////////////////////////////////////////
//
//           F L I G H T S A F E T Y   I N T E R N A T I O N A L
//                     Simulation Systems Division
//                      2700 North Hemlock Circle
//                     Broken Arrow, Oklahoma 74012
//                          (918) 259-4000
/////////////////////////////////////////////////////////////////////////////
//
// Copyright 1998 FlightSafety International
//
// The information contained herein is the property of FLIGHTSAFETY
// INTERNATIONAL Simulation Systems Division and shall not be copied, in
// part or in whole, or disclosed to others in any manner without the
// express written authorization of the FLIGHTSAFETY INTERNATIONAL
// Simulation Systems Division.
//
/////////////////////////////////////////////////////////////////////////////
//
//
// Filename         : Color.h
//
// Date             : 29 August 1998
//
// Author           : Billy Baker
//
// Description      : CColor is a simple class that represents any 
//                    color values for an object in the .  The 
//                    private data members m_ucRed, m_ucGreen, and 
//                    m_ucBlue are accessible/mutable using Red(), 
//                    Green(), Blue(), Red(const unsigned char ucRed), 
//                    Green(const unsigned char ucGreen), and 
//                    Blue(const unsigned char ucBlue).
//                    
//                    Four constructors are avaiable--one to set 
//                    everything to black (default) or to the passed in 
//                    components.  The second to parse a CString for 
//                    rgb components.  This second constructor should 
//                    be passed and should parse a string in the color 
//                    format specified in the DTD.  The third is a 
//                    conversion of a Microsoft COLORREF to a CColor.  
//                    And finally, the fourth is a copy constructor.
//                    
//                    An assignment operator is provided as is a conversion
//                    method to allow a CColor to be used where a COLORREF 
//                    is expected.  Two friend I/O operators are also 
//                    provided for sending a CColor to a CArchive and 
//                    for reading a CColor from a CArchive.  The input 
//                    operator will most likely never be used since an XML 
//                    parse will be used to read in a file.
//
// Classification   : UNCLASSIFIED
//
// Requirements     : None.
//
// Components Used  : None.
//
// Operational 
//    Restrictions  : Machine dependencies/restrictions
//                        Since the COLORREF structure and the CArchive class
//                        are used, a Win32 platform is expected.
//                    Design dependencies/restrictions
//                        The constructor accepting a CString is dependent on
//                        the format of a color specified in the DTD.
//                    Operations containing dependencies/restrictions
//                        None.
//                    Compiler dependencies/restrictions
//                        A Standard C++ compiler is required.
//                    Other condtions for proper execution
//                        None.
//
// Envrionment      : Operating system(s) - Microsoft Windows NT 4.0 with
//                                              NT service packs 3 or 4
//                                          Microsoft Windows NT 5.0
//
//                    Compiler(s) - Visual C++ 5.0
//                                  Visual C++ 5.0 with VisC++ service pack 3
//                                  Visual C++ 6.0
//
//                    Architechure(s) - Intel Pentium, Pentium II
/////////////////////////////////////////////////////////////////////////////

/////////////////////////////////////////////////////////////////////////////
//
//                              Revision History
//
/////////////////////////////////////////////////////////////////////////////
//
// Date             : 29 August 1998
//
// Author           : Billy Baker
//
// Description      : This revision was made to update the structure of the 
//                    class to match the C++ coding document.  Changes 
//                    included reordering some of the body of the class, 
//                    changing the Visual C++ #if !defined() line to fit 
//                    into 80 columns, name changes, and the use of namespace
//                    and explicit.
//
/////////////////////////////////////////////////////////////////////////////
//
// Date             : 29 August 1998
//
// Author           : Billy Baker
//
// Description      : The addition of the namespace FSISuite caused 
//                    problems with the friend functions/methods.  
//                    Further research showed that the implementation
//                    of the friend functions/methods needed to be
//                    surrounded by the same namespace 'string' {}
//                    syntax as the class containing the prototypes.
//
/////////////////////////////////////////////////////////////////////////////
//
// Date             : 31 August 1998
//
// Author           : Billy Baker
//
// Description      : Deleted the explicit keyword from the copy 
//                    constructor.  Changed the copy constructor to use 
//                    the assingment operator for copying data.
//
/////////////////////////////////////////////////////////////////////////////

#include "stdafx.h"

#include "Color.h"

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif

_FSI_STL::map<_FSI_STL::string, _FSI_STL::string> CColor::m_mapGlobalColors;

/////////////////////////////////////////////////////////////////////////////
//
// CArchive& operator<<()
//
// Inputs           : CArchive& rArchive - the archive to dump data to
//                    CColor& rColor - a CColor object to dump
//
// Return Values    : rArchive - this is returned so that the << operator
//                               can be used in a sequence of dumps to
//                               a CArchive.
//
// Date             : 29 August 1998
//
// Author           : Billy Baker
//
// Description      : Friend method for saving CColor data to an 
//                    archive.  This method is implemented so that the 
//                    serialization of widgets will not have to know what 
//                    the color  format is.  This class could have 
//                    inherited from CObject and then could have 
//                    implemented its own Serialize method, but by not 
//                    doing this, the code should be cleaner.
//
/////////////////////////////////////////////////////////////////////////////
CArchive& operator<<(CArchive& rArchive, const CColor& rColor)
{
	CString strFormattedOutput;
	strFormattedOutput.Format("<Color>%d%d%d</Color>\r\n",
                             rColor.m_ucRed, rColor.m_ucGreen,
                             rColor.m_ucBlue);

	rArchive.WriteString(strFormattedOutput);
	return (rArchive);
}

/////////////////////////////////////////////////////////////////////////////
//
// CArchive& operator>>()
//
// Inputs           : CArchive& rArchive - the archive to dump data to
//                    CColor& rColor - the CColor object being read in
//
// Return Values    : rArchive - this is returned so that the >> operator
//                               can be used in a sequence of reads from
//                               a CArchive.
//
// Date             : 29 August 1998
//
// Author           : Billy Baker
//
// Description      : Friend method for reading CColor data from an 
//                    archive.  This method is implemented so that the 
//                    serialization of widgets will not have to know 
//                    what the color format is.  This class could have 
//                    inherited from CObject and then could have 
//                    implemented its own Serialize method, but by not 
//                    doing this, the code should be cleaner.
//                    
//                    Note:  this method may never be used.  Since XML 
//                    is being used to define the file format, an XML 
//                    parser/validator/well-formedness checker will be 
//                    used to read files.
//
/////////////////////////////////////////////////////////////////////////////
CArchive& operator>>(CArchive& rArchive, CColor& rColor)
{
	CString strColor;

	// Make sure a read was successful.  If it wasn't, then leave
	// the red, green, and blue values in classColor alone.
	if (rArchive.ReadString(strColor) != FALSE)
	{
		int nStartElementEnd, nEndElementBegin;
		nStartElementEnd = strColor.Find(">");

		// If the end of the <Color> tag can be found, then
		// continue on parsing the classColor as defined in the DTD.
		// Else, the red, green, and blue values in classColor are left
		// alone.
		if (nStartElementEnd > -1)
		{
			strColor = strColor.Mid(nStartElementEnd + 1);
			nEndElementBegin = strColor.Find("<");

			// If the beginning of the </Color> tag can be found,
			// then continue on parsing the classColor as defined in the DTD.
			// Else, the red, green, and blue values in classColor are left
			// alone.
			if (nEndElementBegin > -1)
			{
				strColor = strColor.Left(nEndElementBegin);
				rColor = CColor(strColor);
			}
		}
	}
	
	return (rArchive);
}

/////////////////////////////////////////////////////////////////////////////
//
// CColor::CColor()
//
// Inputs           : const unsigned char ucRed - a red rgb component
//                    const unsigned char ucGreen - a green rgb component
//                    const unsigned char ucBlue - a blue rgb component
//
// Return Values    : None.
//
// Date             : 29 August 1998
//
// Author           : Billy Baker
//
// Description      : Default constructor with the components of a color
//                    passed as parameters.  This creates a CColor object
//                    with a color of (ucRed, ucGreen, ucBlue).  If ucRed,
//                    ucGreen, and/or ucBlue are not specified, their
//                    default value is 0.  Thus, CColor colorBackground;
//                    will initialize the background color object to
//                    black (0,0,0).
//
/////////////////////////////////////////////////////////////////////////////
CColor::CColor(const unsigned char ucRed, const unsigned char ucGreen,
               const unsigned char ucBlue)
{
    m_ucRed = ucRed;
    m_ucGreen = ucGreen;
    m_ucBlue = ucBlue;
}

/////////////////////////////////////////////////////////////////////////////
//
// CColor::CColor()
//
// Inputs           : const CString& rStrColor - an RGB string in DTD color
//                                               format
//
// Return Values    : None.
//
// Date             : 29 August 1998
//
// Author           : Billy Baker
//
// Description      : Since the color is currently defined in the DTD to
//                    be a rrrgggbbb value and since that value will be
//                    read in as a string, a constructor is provided to
//                    break down that string into its components.
//                    
//                    Assumption: strColor is in the format specified in
//                                the DTD for color.  Currently this is
//                                rrrgggbbb.
//                    
//                    If atoi can't convert a substring to an integer,
//                    then it initializes the color componet to 0.
//
/////////////////////////////////////////////////////////////////////////////
CColor::CColor(const CString& rStrColor)
{
	m_ucRed     = (unsigned char)atoi(rStrColor.Left(3));
	m_ucGreen   = (unsigned char)atoi(rStrColor.Mid(3,3));
	m_ucBlue    = (unsigned char)atoi(rStrColor.Right(3));
}

/////////////////////////////////////////////////////////////////////////////
//
// CColor::CColor()
//
// Inputs           : const COLORREF& rColorRef - a Microsoft COLORREF
//                                                structure
//
// Return Values    : None.
//
// Date             : 29 August 1998
//
// Author           : Billy Baker
//
// Description      : Constructor for initializing a CColor object with
//                    the values stored in a COLORREF structure.
//
/////////////////////////////////////////////////////////////////////////////
CColor::CColor(const COLORREF& rColorRef)
{
	m_ucRed = GetRValue(rColorRef);
	m_ucGreen = GetGValue(rColorRef);
	m_ucBlue = GetBValue(rColorRef);
}

/////////////////////////////////////////////////////////////////////////////
//
// bool CColor::operator==()
//
// Inputs           : const CColor& rColor - a CColor object to compare
//
// Return Values    : true - if all color components are equal
//                    false - if any color component is different
//
// Date             : 29 August 1998
//
// Author           : Billy Baker
//
// Description      : Overload of the equals operator to allow one CColor
//                    object to be compared to another CColor object.
//
/////////////////////////////////////////////////////////////////////////////
bool CColor::operator==(const CColor& rColor)
{
    bool bRetVal = true;

    if (m_ucRed != rColor.m_ucRed ||
        m_ucGreen != rColor.m_ucGreen ||
        m_ucBlue != rColor.m_ucBlue)
    {
        bRetVal = false;
    }

	return bRetVal;
}

/////////////////////////////////////////////////////////////////////////////
//
// CColor::operator COLORREF()
//
// Inputs           : None.
//
// Return Values    : a COLORREF structure
//
// Date             : 29 August 1998
//
// Author           : Billy Baker
//
// Description      : Conversion operator that allows a CColor object
//                    to be converted to a Microsoft COLORREF.  Thus,
//                    a CColor object can be passed where a COLORREF is
//                    expected.
//
/////////////////////////////////////////////////////////////////////////////
CColor::operator COLORREF()
{
	return RGB(m_ucRed,m_ucGreen,m_ucBlue);
}

/////////////////////////////////////////////////////////////////////////////
//
// unsigned char CColor::Red()
//
// Inputs           : None.
//
// Return Values    : m_ucRed - the red color component
//
// Date             : 29 August 1998
//
// Author           : Billy Baker
//
// Description      : Accessor method for getting the value of m_ucRed.
//
/////////////////////////////////////////////////////////////////////////////
unsigned char CColor::Red()
{
	return m_ucRed;
}

/////////////////////////////////////////////////////////////////////////////
//
// unsigned char CColor::Green()
//
// Inputs           : None.
//
// Return Values    : m_ucGreen - the green color component
//
// Date             : 29 August 1998
//
// Author           : Billy Baker
//
// Description      : Accessor method for getting the value of m_ucGreen.
//
/////////////////////////////////////////////////////////////////////////////
unsigned char CColor::Green()
{
	return m_ucGreen;
}

/////////////////////////////////////////////////////////////////////////////
//
// unsigned char CColor::Blue()
//
// Inputs           : None.
//
// Return Values    : m_ucBlue - the blue color component
//
// Date             : 29 August 1998
//
// Author           : Billy Baker
//
// Description      : Accessor method for getting the value of m_ucBlue.
//
/////////////////////////////////////////////////////////////////////////////
unsigned char CColor::Blue()
{
	return m_ucBlue;
}

/////////////////////////////////////////////////////////////////////////////
//
// void CColor::Red()
//
// Inputs           : const unsigned char ucRed - a new red component value
//
// Return Values    : None.
//
// Date             : 29 August 1998
//
// Author           : Billy Baker
//
// Description      : Mutator method for setting the value of m_ucRed.
//
/////////////////////////////////////////////////////////////////////////////
void CColor::Red(const unsigned char ucRed)
{
	m_ucRed = ucRed;
}

/////////////////////////////////////////////////////////////////////////////
//
// void CColor::Green()
//
// Inputs           : const unsigned char ucGreen - a new green component
//                                                 value
//
// Return Values    : None.
//
// Date             : 29 August 1998
//
// Author           : Billy Baker
//
// Description      : Mutator method for setting the value of m_ucGreen.
//
/////////////////////////////////////////////////////////////////////////////
void CColor::Green(const unsigned char ucGreen)
{
	m_ucGreen = ucGreen;
}

/////////////////////////////////////////////////////////////////////////////
//
// void CColor::Blue()
//
// Inputs           : const unsigned char ucBlue - a new blue component
//                                                 value
//
// Return Values    : None.
//
// Date             : 29 August 1998
//
// Author           : Billy Baker
//
// Description      : Mutator method for setting the value of m_ucBlue.
//
/////////////////////////////////////////////////////////////////////////////
void CColor::Blue(const unsigned char ucBlue)
{
	m_ucBlue = ucBlue;
}

/////////////////////////////////////////////////////////////////////////////
//
// void CColor::RGBtoHLS()
//
// Inputs           : unsigned char& rucH = hue
//                    unsigned char& rucL = luminance
//                    unsigned char& rucS = saturation
//
// Return Values    : None.
//
// Date             : 2 February 1999
//
// Author           : Billy Baker
//
// Description      : Conversion method to get the HLS values for an
//                    RGB color.  This method came from the MSDN library.
//
/////////////////////////////////////////////////////////////////////////////
void CColor::RGBtoHLS(unsigned char& rucH, unsigned char& rucL, 
               unsigned char& rucS)
{ 
   WORD R,G,B;          /* input RGB values */
   BYTE cMax,cMin;      /* max and min RGB values */
   WORD  Rdelta,Gdelta,Bdelta; /* intermediate value: % of spread from max */

   /* get R, G, and B out of DWORD */
   R = m_ucRed;
   G = m_ucGreen;
   B = m_ucBlue;

   /* calculate lightness */
   cMax = (unsigned char)__max(__max(R,G), B);
   cMin = (unsigned char)__min(__min(R,G), B);
   rucL = (unsigned char)(( ((cMax+cMin)*HLSMAX) + RGBMAX )/(2*RGBMAX));

   if (cMax == cMin) {           /* r=g=b --> achromatic case */
      rucS = 0;                     /* saturation */
      rucH = UNDEFINED;             /* hue */
   }
   else {                        /* chromatic case */
      /* saturation */
      if (rucL <= (HLSMAX/2))
      {
         rucS = (unsigned char)(( ((cMax-cMin)*HLSMAX) + ((cMax+cMin)/2) ) / (cMax+cMin));
      }
      else
      {
         rucS = (unsigned char)(( ((cMax-cMin)*HLSMAX) + ((2*RGBMAX-cMax-cMin)/2) )
                                / (2*RGBMAX-cMax-cMin));
      }

      /* hue */
      Rdelta = (unsigned short)(( ((cMax-R)*(HLSMAX/6)) + ((cMax-cMin)/2) ) / (cMax-cMin));
      Gdelta = (unsigned short)(( ((cMax-G)*(HLSMAX/6)) + ((cMax-cMin)/2) ) / (cMax-cMin));
      Bdelta = (unsigned short)(( ((cMax-B)*(HLSMAX/6)) + ((cMax-cMin)/2) ) / (cMax-cMin));

      if (R == cMax)
      {
         rucH = (unsigned char)(Bdelta - Gdelta);
      }
      else if (G == cMax)
      {
         rucH = (unsigned char)((HLSMAX/3) + Rdelta - Bdelta);
      }
      else /* B == cMax */
      {
         rucH = (unsigned char)(((2*HLSMAX)/3) + Gdelta - Rdelta);
      }

      if (rucH < 0)
      {
         rucH += HLSMAX;
      }

      if (rucH > HLSMAX)
      {
         rucH -= HLSMAX;
      }
   }
}

/* utility routine for HLStoRGB */
WORD CColor::HueToRGB(WORD n1, WORD n2, WORD hue)
{ 

   /* range check: note values passed add/subtract thirds of range */
   if (hue < 0)
      hue += HLSMAX;

   if (hue > HLSMAX)
      hue -= HLSMAX;

   /* return r,g, or b value from this tridrant */
   if (hue < (HLSMAX/6))
   {
      return (unsigned short)( n1 + (((n2-n1)*hue+(HLSMAX/12))/(HLSMAX/6)) );
   }

   if (hue < (HLSMAX/2))
   {
      return (unsigned short)( n2 );
   }

   if (hue < ((HLSMAX*2)/3))
   {
      return (unsigned short)( n1 + (((n2-n1)*(((HLSMAX*2)/3)-hue)+(HLSMAX/12))/(HLSMAX/6)));
   }
   else
   {
      return (unsigned short)( n1 );
   }
} 

CColor CColor::HLStoRGB(WORD hue, WORD lum, WORD sat)
{ 

   WORD R,G,B;                /* RGB component values */
   WORD  Magic1,Magic2;       /* calculated magic numbers (really!) */

   if (sat == 0) 
   {            /* achromatic case */
      R = G = B = (unsigned short)((lum*RGBMAX)/HLSMAX);
      if (hue != UNDEFINED)
      {
         /* ERROR */
      }
   }
   else
   {                    /* chromatic case */
      /* set up magic numbers */
      if (lum <= (HLSMAX/2))
      {
         Magic2 = (unsigned short)((lum*(HLSMAX + sat) + (HLSMAX/2))/HLSMAX);
      }
      else
      {
         Magic2 = (unsigned short)(lum + sat - ((lum*sat) + (HLSMAX/2))/HLSMAX);
      }

      Magic1 = (unsigned short)(2*lum-Magic2);

      /* get RGB, change units from HLSMAX to RGBMAX */
      R = (unsigned short)( ( HueToRGB(Magic1, Magic2, hue + (HLSMAX/3) ) * RGBMAX +
                            (HLSMAX/2) ) / HLSMAX); 
      G = (unsigned short)( ( HueToRGB(Magic1, Magic2, hue) * RGBMAX + 
                            (HLSMAX / 2) ) / HLSMAX);
      B = (unsigned short)( ( HueToRGB(Magic1, Magic2, hue - (HLSMAX/3) ) * RGBMAX +
                            (HLSMAX / 2) ) / HLSMAX); 
   }

   return CColor((unsigned char)R,(unsigned char)G,(unsigned char)B);
} 

CColor CColor::DarkColor()
{
    unsigned char H;
    unsigned char L;
    unsigned char S;

    RGBtoHLS(H, L, S);

    L = (unsigned char)(L * 666/1000);

    return HLStoRGB(H, L, S);
}

CColor CColor::HighlightColor()
{
    unsigned char H;
    unsigned char L;
    unsigned char S;

    RGBtoHLS(H, L, S);

    L = (unsigned char)(((1000-500) * L + 500*241) / 1000);

    return HLStoRGB(H, L, S);
}

CColor CColor::GlobalColor(const CString& rstrGlobalColor)
{
    if (true == m_mapGlobalColors.empty())
    {
        HKEY hKey;
        RegOpenKeyEx(HKEY_LOCAL_MACHINE, _T("Software\\FlightSafety\\FSISuite\\Settings"), 0, KEY_READ, &hKey);
        if (hKey != NULL)
        {
            long int lGlobalColorCount = 0;
            long int lGlobalColorStep = 1;
            unsigned long int ulData = 0;
            DWORD dwSize = sizeof (unsigned long int);
            RegQueryValueEx(hKey, _T("GlobalColorCount"), NULL, NULL, (BYTE*)&ulData, 
                            &dwSize);
            lGlobalColorCount = ulData;

            CString strGlobalColorXXX;
            CString strGlobalColorRGB;
            CString strGlobalColorName;

            for (; lGlobalColorStep <= lGlobalColorCount; lGlobalColorStep++)
            {
                strGlobalColorXXX.Format("GlobalColor%d",lGlobalColorStep);
 
                // Get the name of the GlobalColor.
		        CString strValue;
		        DWORD dwType, dwCount;
		        LONG lResult = RegQueryValueEx(hKey, (LPCTSTR)strGlobalColorXXX, 
                                               NULL, &dwType, NULL, &dwCount);
		        if (lResult == ERROR_SUCCESS)
		        {
			        ASSERT(dwType == REG_SZ);
			        lResult = RegQueryValueEx(hKey, (LPCTSTR)strGlobalColorXXX, 
                                              NULL, &dwType,
				                 (LPBYTE)strValue.GetBuffer(dwCount/sizeof(TCHAR)),
                                              &dwCount);

			        strValue.ReleaseBuffer();
		        }

                strGlobalColorName = strValue;

                // Get the RGB for the GlobalColor
                if (strGlobalColorName != "")
                {
                    // Code copied from GetProfileString() in CWinApp 
                    // implementation.
		            lResult = RegQueryValueEx(hKey, (LPCTSTR)strGlobalColorName, 
                                                   NULL, &dwType, NULL, &dwCount);
		            if (lResult == ERROR_SUCCESS)
		            {
			            ASSERT(dwType == REG_SZ);
			            lResult = RegQueryValueEx(hKey, (LPCTSTR)strGlobalColorName, 
                                                  NULL, &dwType,
				                 (LPBYTE)strValue.GetBuffer(dwCount/sizeof(TCHAR)),
                                                  &dwCount);

			            strValue.ReleaseBuffer();
		            }

                    strGlobalColorRGB = strValue;

                    m_mapGlobalColors[(LPCTSTR)strGlobalColorName] = 
                                            _FSI_STL::string((LPCTSTR)strGlobalColorRGB);
                }
            }

            RegCloseKey(hKey);
        }
    }

    CString strColor = "";

    if (m_mapGlobalColors.find(_FSI_STL::string((LPCTSTR)rstrGlobalColor)) != m_mapGlobalColors.end())
    {
        strColor = m_mapGlobalColors[(LPCTSTR)rstrGlobalColor].c_str();
        CString str = "";
        CString strR = "";
        CString strG = "";
        CString strB = "";
        if (strColor.Find(",") > -1)
        {
            if (strColor.Find(",") > -1)
            {
                str = strColor.Mid(strColor.Find(",") + 1);
                strR.Format("%03s",strColor.Left(strColor.Find(",")));
            }
            if (str.Find(",") > -1)
            {
                strG.Format("%03s", str.Left(str.Find(",")));
                strB.Format("%03s", str.Mid(str.Find(",") + 1));
                strColor = strR + strG + strB;
            }
        }
    }

    if (strColor == "")
    {
        return CColor();
    }
    else
    {
        return CColor(strColor);
    }
}

void CColor::ResetColors()
{
    m_mapGlobalColors.clear();
}