/////////////////////////////////////////////////////////////////////////////
//
//           F L I G H T S A F E T Y   I N T E R N A T I O N A L
//                     Simulation Systems Division
//                      2700 North Hemlock Circle
//                     Broken Arrow, Oklahoma 74012
//                          (918) 259-4000
/////////////////////////////////////////////////////////////////////////////
//
// DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
// Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
// technical or operational data or information from automatic dissemination
// under the International Exchange Program or by other means.  This protection
// covers information required solely for administrative or operational
// purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
//
// WARNING:  This document contains technical data whose export is restricted
// by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
// Executive Order 12470.  Violation of these export control laws is subject
// to severe criminal penalties.  Dissemination of this document is controlled
// under DOD Directive 5230.25
//
/////////////////////////////////////////////////////////////////////////////
//
//
// Filename         : WTXCommsSystemInterface.cpp
//
// Date             : 10 February 1999
//
// Engineer         : Billy Baker
//
// Revision         : $Revision: 1.3 $
//
// Description      : WTXCommsSystemInterface.cpp contains the 
//                    implementation of the CWTXCommsSystemInterface 
//                    class.  This class is derived from 
//                    CFSISuiteCommsSystemInterface instead of 
//                    CCommsSystemInterface because 
//                    CFSISuiteCommsSystemInterface already implements 
//                    the LoadVariables code needed for WTX...  This 
//                    class is meant to be used on for initial testing 
//                    and demos of IOS to VxWorks simulation 
//                    communications.  The USE_WTX switch, when set, 
//                    will set all of the necessary symbols and add the 
//                    wtx library(ies) for linkage.  No mention of the 
//                    WTX library should exist any where else except 
//                    for at the initialization of the m_mapCommsSystem 
//                    data structure.  
//
// Classification   : UNCLASSIFIED
//
// Requirements     : None.
//
// Components Used  : CVariant, CFSISuiteCommsSystemInterface, 
//                    CCommsSystemInterface, CCriticalSection, map,
//                    iterator, CVariableData.  
//
// Operational 
//    Restrictions  : Machine dependencies/restrictions
//                        None.
//                    Design dependencies/restrictions
//                        If USE_WTX is set, then the WTX libraries are 
//                        needed.  Of course, the licenses should also be paid.
//                    Operations containing dependencies/restrictions
//                        None.
//                    Compiler dependencies/restrictions
//                        Microsoft Visual C++ pragma used to localize the 
//                        WTX symbols.
//                    Other conditions for proper execution
//                        None.
//
// Environment      : Operating system(s) - Microsoft Windows NT 4.0 with
//                                              NT service pack 3
//                                          Microsoft Windows NT 5.0
//
//                    Compiler(s) - Visual C++ 5.0 with VisC++ service pack 3
//                                  Visual C++ 6.0
//
//                    Architechure(s) - Intel Pentium, Pentium II
/////////////////////////////////////////////////////////////////////////////

/////////////////////////////////////////////////////////////////////////////
//
//                              R e v i s i o n   H i s t o r y
//
/////////////////////////////////////////////////////////////////////////////
// $Log: WTXCommsSystemInterface.cpp $
// Revision 1.3  2000/03/06 16:07:22  billyb
// Changed function signature and moved unused variables to
// eliminate warnings.
// Revision 1.2  2000/01/27 07:55:53  billyb
// Added comments about array support not being implemented.
// Revision 1.1  1999/08/24 18:25:03  billyb
// Initial Revision
//
#include "..\core\stdafx.h"
#include "WTXCommsSystemInterface.h"

#ifdef USE_WTX
HWTX  CWTXCommsSystemInterface::m_hWtx;
#endif
//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

/////////////////////////////////////////////////////////////////////////////
//
// CWTXCommsSystemInterface::CWTXCommsSystemInterface()
//
// Inputs           : None.
//
// Return Values    : None.
//
// Date             : 09 February 1999
//
// Engineer         : Billy Baker
//
// Description      : Default constructor.
//
/////////////////////////////////////////////////////////////////////////////
CWTXCommsSystemInterface::CWTXCommsSystemInterface() : 
                            CFSISuiteCommsSystemInterface()
{

}

/////////////////////////////////////////////////////////////////////////////
//
// CWTXCommsSystemInterface::~CWTXCommsSystemInterface()
//
// Inputs           : None.
//
// Return Values    : None.
//
// Date             : 09 February 1999
//
// Engineer         : Billy Baker
//
// Description      : Default destructor.
//
/////////////////////////////////////////////////////////////////////////////
CWTXCommsSystemInterface::~CWTXCommsSystemInterface()
{
}

/////////////////////////////////////////////////////////////////////////////
//
// CCommsSystemInterface* CWTXCommsSystemInterface::CreateComms()
//
// Inputs           : None.
//
// Return Values    : address of a new instance of 
//                    CFISSuiteCommsSystemInterface.
//
// Date             : 09 February 1999
//
// Engineer         : Billy Baker
//
// Description      : CreateComms() creates a new instance of
//                    CWTXCommsSystemInterface.  Each CCommsSystemInterface
//                    derived class must implement CreateComms so that the
//                    comms protocol can be dynamically changed during 
//                    operation.  A name, like WTX, is associated with
//                    a pointer to this method.  By storing pointers to 
//                    CreateComms methods, a large switch statement is not
//                    needed.
//
/////////////////////////////////////////////////////////////////////////////
CCommsSystemInterface* CWTXCommsSystemInterface::CreateComms()
{
    return new CWTXCommsSystemInterface;
}

UINT CWTXCommsSystemInterface::ThreadFunc(LPVOID lParam)
{
    DWORD dwData = 0;
    DWORD dwBytesRead;
    DWORD dwBytesLeft;
    DWORD dwBytesLeftMsg;

    unsigned long int ulRate = (unsigned long int)lParam;

    while (1)
    {
        // Check the pipe to see if a shutdown has been sent.
        PeekNamedPipe(m_hPipeRead, &dwData, sizeof(dwData), &dwBytesRead, 
                      &dwBytesLeft, &dwBytesLeftMsg);

        if (dwData == 1)
        {
            CloseHandle(m_hPipeRead);
            break;
        }

        ReadData();
        Sleep(1000 / ulRate);
    }

    return 0;
}

/////////////////////////////////////////////////////////////////////////////
//
// void CWTXCommsSystemInterface::ReadData()
//
// Inputs           : HWND hWnd - timer callback specific
//                    UINT uiMsg - timer callback specific
//                    UINT idEvent - timer callback specific
//                    DWORD dwTime - timer callback specific
//
// Return Values    : None.
//
// Date             : 09 February 1999
//
// Engineer         : Billy Baker
//
// Description      : ReadData() is a generic name used for a timer callback
//                    function.  Each comms system interface derived class
//                    should have a method used for passing data from 
//                    simulation on to the CVariants.  Then UpdateComms should
//                    be called to get the CCommsActions to update the
//                    graphical elements.  Be sure to use the CCriticalSection
//                    m_sync with Lock and Unlock to prevent a page change from
//                    deleting memory that needs to be written to. For WTX,
//                    the VxWorks/Tornado WTX communications library is used
//                    with variables requested one at a time.
//
/////////////////////////////////////////////////////////////////////////////
void CWTXCommsSystemInterface::ReadData()
{
    // Read all of the data.
    m_sync.Lock();

    // Add any comms actions that have been marked for Adding.
    if (m_listCommsActionsToAdd.size() > 0)
    {
        m_syncCommsAdd.Lock();

        _FSI_STL::list<CCommsAction*>::iterator lIt = 
                                            m_listCommsActionsToAdd.begin();
        for (; lIt != m_listCommsActionsToAdd.end(); lIt++)
        {
            AddComms((CIOAction*)(*lIt));
        }

        m_listCommsActionsToAdd.clear();

        m_syncCommsAdd.Unlock();
    }

    // Remove any comms actions that have been marked for deleting.
    if (m_listCommsActionsToDelete.size() > 0)
    {
        m_syncCommsDelete.Lock();

        _FSI_STL::list<CCommsAction*>::iterator lIt = 
                                            m_listCommsActionsToDelete.begin();
        for (; lIt != m_listCommsActionsToDelete.end(); lIt++)
        {
            DeleteComms((CIOAction*)(*lIt));
        }

        m_listCommsActionsToDelete.clear();

        m_syncCommsDelete.Unlock();
    }

#ifdef USE_WTX
    char* pPos = m_pcDebriefData;
    CVariableData* pVarData = NULL;
    CVariant variant;

    TGT_ADDR_T          symValue    = 0;
    BOOL32              exactName   = FALSE;
    UINT8               symType     = 0; 
    UINT8               typeMask    = 0;
    WTX_SYMBOL          *pSymbol    = NULL;


    _FSI_STL::map<_FSI_STL::string, CVariant*>::iterator mvIt = NULL;
    _FSI_STL::map<_FSI_STL::string, CVariableData*>::iterator mIt = NULL;

    // Step through only the CVariants and give the value
    // from simulation to the CVariant.
    mvIt = m_mapVarName2Variant.begin();
    for (; mvIt != m_mapVarName2Variant.end(); mvIt++)
    {
        pVarData = m_mapVarName2Data[(*mvIt).first];
        // Make sure that the variant is not null.
        if ((*mvIt).second != NULL)
        {
            pSymbol = NULL;
            if (pVarData->AddressValid() == false)
            {
                //  Read a value from simulation.
                pSymbol = wtxSymFind(m_hWtx, (*mvIt).first.c_str(), symValue, 
                                     exactName, symType, typeMask);
                if (pSymbol != NULL)
                {
                    pVarData->Address(pSymbol->value);
                    pVarData->AddressValid(true);
                    wtxResultFree(m_hWtx, pSymbol);
                }
            }

            // Create just enough memory for the value.
            unsigned char ucSize = (*mvIt).second->Size(pVarData->Type());

            // May no longer work with changes to CVariant. !!!!!
            void *pValue = (*mvIt).second->CreateVar();

            if (pValue != NULL)
            {
                // Read the value and assign it to the
                // variant;
                wtxMemRead(m_hWtx, pVarData->Address(), pValue, ucSize);

                // Set the value of the variant.
                (*mvIt).second->Value(pVarData->Type(), pValue);

                if ((*mvIt).second->List() == true)
                {
                    (*mvIt).second->AddValue((*mvIt).second);
                }
            }
        }
    }

    // If this is a pass that should record debrief data,
    // then step though the list of debrief variables and 
    // place each simulation variable into the data block to
    // be recorded.
    if (m_ucDebriefUpdatePass >= m_ucDebriefRatio && 
        m_mapVarName2Debrief.size() > 0)
    {
        mIt = m_mapVarName2Debrief.begin();
        for (; mIt != m_mapVarName2Debrief.end(); mIt++)
        {
            pVarData = (*mIt).second;

            //  Read a value from simulation.
            pSymbol = NULL;
            if (pVarData->AddressValid() == false)
            {
                pSymbol = wtxSymFind(m_hWtx, 
                                     (*mvIt).first.c_str(), 
                                     symValue, exactName, symType, 
                                     typeMask);

                if (pSymbol != NULL)
                {
                    pVarData->Address(pSymbol->value);
                    pVarData->AddressValid(true);
                    wtxResultFree(m_hWtx, pSymbol);
                }

                // Create just enough memory for the value.
                unsigned char ucSize = variant.Size(pVarData->Type());
                void *pValue = variant.CreateVar();

                if (pValue != NULL)
                {
                    // Read the value and assign it to the
                    // variant;
                    wtxMemRead(m_hWtx, pVarData->Address(), 
                               pValue, ucSize);

                    // Set the value of the variant.
                    variant.Value(pVarData->Type(), pValue);
                }
            }

            // May no longer work with changes to CVariant. !!!!!
            memcpy(pPos, variant.CreateVar(), variant.Size(pVarData->Type()));
            pPos += variant.Size(pVarData->Type());
        }

        if (m_fileDebrief.m_hFile != CFile::hFileNull)
        {
            m_fileDebrief.Write(m_pcDebriefData, m_ulDebriefBufferSize);
        }

        m_ucDebriefUpdatePass = 1;
    }
    else
    {
        m_ucDebriefUpdatePass++;
    }
#endif

    UpdateComms();
    m_sync.Unlock();
}

/////////////////////////////////////////////////////////////////////////////
//
// void CWTXCommsSystemInterface::WriteData()
//
// Inputs           : CVariant* pVariant - the value to write to simulation.
//                    CVariableData* pVarData - the variable data from the
//                                              comms data file.
//
// Return Values    : None.
//
// Date             : 09 February 1999
//
// Engineer         : Billy Baker
//
// Description      : WriteData() will be called from 
//                    CCommsAction::OnLButtonUp() which is called whenever a
//                    user action is performed on a graphical element--mouse
//                    click, space bar press, enter key press, etc.
//
/////////////////////////////////////////////////////////////////////////////
void CWTXCommsSystemInterface::WriteData(CVariant* pVariant, 
                                CVariableData* pVarData)
{
    if (pVariant == NULL || pVarData == NULL)
    {
        return;
    }

#ifdef USE_WTX
    TGT_ADDR_T          symValue    = 0;
    BOOL32              exactName   = FALSE;
    UINT8               symType     = 0;
    UINT8               typeMask    = 0;
    WTX_SYMBOL          *pSymbol    = NULL;

    if (pVarData->AddressValid() == false)
    {
        pSymbol = wtxSymFind(m_hWtx, 
                             pVarData->SimVarName().c_str(), 
                             symValue, exactName, symType, typeMask);
        if (pSymbol != NULL)
        {
            pVarData->Address(pSymbol->value);
            pVarData->AddressValid(true);
            wtxResultFree(m_hWtx, pSymbol);
        }
    }

    unsigned char ucSize = 
                   pVariant->Size(pVarData->Type());

    // May no longer work with changes to CVariant. !!!!!
    wtxMemWrite(m_hWtx, pVariant->CreateVar(), pVarData->Address(), 
                ucSize);
#endif
}

/////////////////////////////////////////////////////////////////////////////
//
// void CWTXCommsSystemInterface::Start()
//
// Inputs           : None.
//
// Return Values    : None.
//
// Date             : 09 February 1999
//
// Engineer         : Billy Baker
//
// Description      : Start() can be called at any time but must be 
//                    called after the loading of variables.  The call will
//                    be made when the first CCommsAction is created or prior
//                    to that when the COMMS DLL is loaded and initialized.
//                    Start() is responsible for setting up the mechanism for
//                    reading and writing.  Here, this means initializing the
//                    WTX library, attaching to the other machine, and 
//                    registering events.
//
/////////////////////////////////////////////////////////////////////////////
void CWTXCommsSystemInterface::Start()
{

    CCommsSystemInterface::Start();

    // Setup a timer for testing.
#ifdef USE_WTX
    if (wtxInitialize(&m_hWtx) != OKAY)
    {
        CString msg;
        msg.Format("Error: Cannot initialize WTX API.");
        printf(msg);
        return;
    }

    if (wtxToolAttach(m_hWtx, m_stlStrMachineName.c_str(), "TornadoTest") != OKAY)
    {
        CString msg;
        msg.Format("Error: Cannot attach to %s.",m_stlStrMachineName.c_str());
        printf(msg);
        return;
    }

    if (wtxRegisterForEvent(m_hWtx, ".*") != OKAY)
    {
        CString msg;
        msg.Format("Error: Cannot register for events.");
            printf(msg);

        wtxToolDetach(m_hWtx);
        wtxTerminate(m_hWtx);
        return;
    }
#endif

    m_pCommsThread = AfxBeginThread(ThreadFunc, 
                                    (LPVOID)m_ulRate,
                                    THREAD_PRIORITY_TIME_CRITICAL,
                                    0, CREATE_SUSPENDED);

    CreatePipe(&m_hPipeRead, &m_hPipeWrite, NULL, 0);

    if (m_pCommsThread != NULL)
    {
        m_pCommsThread->m_bAutoDelete = false;

        m_pCommsThread->ResumeThread();

        m_bStarted = true;
    }
}

/////////////////////////////////////////////////////////////////////////////
//
// void CWTXCommsSystemInterface::Stop()
//
// Inputs           : None.
//
// Return Values    : None.
//
// Date             : 09 February 1999
//
// Engineer         : Billy Baker
//
// Description      : Stop() can be called at any time but must be 
//                    called before the current instance of a comms 
//                    system interface derived class is deleted. The base
//                    class must be called after making sure that no addtional
//                    data will be read and have to be processed.  WTX 
//                    cleanup is also needed here.
//
/////////////////////////////////////////////////////////////////////////////
void CWTXCommsSystemInterface::Stop()
{
    // Stop listening/reading.
    if (m_pCommsThread != NULL)
    {

        // The comms thread will not auto delete itself because it should
        // run until the comms library is unloaded.  Thus, the comms thread
        // must be deleted.
        DWORD dwData = 1;
        DWORD dwBytesWritten;
        WriteFile(m_hPipeWrite, &dwData, sizeof(dwData), &dwBytesWritten, NULL);
        CloseHandle(m_hPipeWrite);

        WaitForSingleObject(m_pCommsThread->m_hThread, INFINITE);
        delete m_pCommsThread;
    }

    CCommsSystemInterface::Stop();

#ifdef USE_WTX
    wtxToolDetach(m_hWtx);
    wtxTerminate(m_hWtx);
#endif

    m_bStarted = false;
}

