/////////////////////////////////////////////////////////////////////////////
//
//           F L I G H T S A F E T Y   I N T E R N A T I O N A L
//                     Simulation Systems Division
//                      2700 North Hemlock Circle
//                     Broken Arrow, Oklahoma 74012
//                          (918) 259-4000
/////////////////////////////////////////////////////////////////////////////
//
// DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
// Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
// technical or operational data or information from automatic dissemination
// under the International Exchange Program or by other means.  This protection
// covers information required solely for administrative or operational
// purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
//
// WARNING:  This document contains technical data whose export is restricted
// by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
// Executive Order 12470.  Violation of these export control laws is subject
// to severe criminal penalties.  Dissemination of this document is controlled
// under DOD Directive 5230.25
//
/////////////////////////////////////////////////////////////////////////////
//
//
// Filename         : MessageLoggingSystemInterface.cpp
//
// Date             : 25 August 1999
//
// Engineer         : Billy Baker
//
// Revision         : $Revision: 1.8 $
//
// Description      : MessageLoggingSystemInterface.cpp contains the 
//                    implementation of the 
//                    CMessageLoggingSystemInterface class.  This class 
//                    is used about the same way as the comms system 
//                    interface classes in that each of the classes has 
//                    data either from variables on the host, a data 
//                    file, or local variables or from a logs.  Both 
//                    types of classes also manage the actions that can 
//                    access this data and then distribute the data to 
//                    the graphical elements.
//
// Classification   : UNCLASSIFIED
//
// Requirements     : None.
//
// Components Used  : CMessageLogAction, _FSI_STL::string, 
//                    _FSI_STL::map, _FSI_STL::list, CAction, CIOAction.
//
// Operational 
//    Restrictions  : Machine dependencies/restrictions
//                        None.
//                    Design dependencies/restrictions
//                        None.
//                    Operations containing dependencies/restrictions
//                        None.
//                    Compiler dependencies/restrictions
//                        None.
//                    Other conditions for proper execution
//                        None.
//
// Environment      : Operating system(s) - Microsoft Windows NT 4.0 with
//                                              NT service pack 3, 4, or 5
//                                          Microsoft Windows NT 2000
//
//                    Compiler(s) - Visual C++ 6.0
//
//                    Architechure(s) - Intel Pentium, Pentium II
/////////////////////////////////////////////////////////////////////////////

/////////////////////////////////////////////////////////////////////////////
//
//                              R e v i s i o n   H i s t o r y
//
/////////////////////////////////////////////////////////////////////////////
// $Log: MessageLoggingSystemInterface.cpp $                                                                   //
// Revision 1.8  2000/05/03 20:49:12  billyb                                                                   //
// Changed day display format.                                                                   //
// Revision 1.7  2000/03/31 22:46:49  billyb                                                                   //
// Added UpdateComms with code from AddMessage to make                                                                   //
// updating display of logs more in line with screen updates rather                                                                   //
// than every time a message is added.                                                                   //
// Revision 1.6  2000/03/06 16:08:38  billyb                                                                   //
// Changed function signatures to eliminate warnings.                                                                   //
// Revision 1.5  2000/01/21 10:11:46  billyb                                                                   //
// Added accessor for data structure of all logs.  Changed                                                                    //
// "Simulation" specific code to generic using a message                                                                   //
// log action's variable member.                                                                   //
// Revision 1.4  1999/11/06 22:40:14  billyb                                                                   //
// Included <algorithm> for find function.                                                                   //
// Revision 1.3  1999/11/04 19:57:52  billyb                                                                   //
// Changed for loop to while and added variable for map end()                                                                   //
// call to decrease calls to end().  Changed size() call to empty()                                                                   //
// since size() is a calculation and empty() is an accessor for                                                                   //
// a member variable.                                                                   //
// Revision 1.2  1999/08/25 20:01:31  billyb                                                                   //
// Added comments.  Corrected problem with deleting actions from the local data structures that caused a crash.                                                                   //
/////////////////////////////////////////////////////////////////////////////
#include "..\core\stdafx.h"
#include <algorithm>
#include "MessageLoggingSystemInterface.h"
#include "MessageLogAction.h"

static const int nLogSize = 16;

const char cLogDir[]    = "c:\\temp\\";
const char cLogFile[]   = "msglog.txt";

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

/////////////////////////////////////////////////////////////////////////////
//
// CMessageLoggingSystemInterface::CMessageLoggingSystemInterface()
//
// Inputs           : None.
//
// Return Values    : None.
//
// Date             : 21 October 2000
//
// Engineer         : Billy Baker
//
// Description      : Default constructor.  An attempt is made to open
//                    a file on the hard disk for storing all of the 
//                    messages. 
//
/////////////////////////////////////////////////////////////////////////////
CMessageLoggingSystemInterface::CMessageLoggingSystemInterface() : fp(NULL)
{
    fp   =  fopen(CString(cLogDir) + CString(cLogFile), "w+");
}

/////////////////////////////////////////////////////////////////////////////
//
// CMessageLoggingSystemInterface::~CMessageLoggingSys
//                    temInterface()
//
// Inputs           : None.
//
// Return Values    : None.
//
// Date             : 21 October 2000
//
// Engineer         : Billy Baker
//
// Description      : Default destructor.  If a file was opened in the 
//                    constructor, then it is closed here.  Any variants for
//                    storing a small set of the log messages are deleted 
//                    here. The file of log messages is copied with a time
//                    value in its name so that restarted the log will
//                    preserve the old messages.
//
/////////////////////////////////////////////////////////////////////////////
CMessageLoggingSystemInterface::~CMessageLoggingSystemInterface()
{
    // Close the file and copy it with a timestamp in its name.
    if (fp)
    {
        fclose(fp);
        struct tm* sTime;
        time_t now;
        time(&now);
        sTime=localtime(&now);

        char cTimeFormat[32];
        strftime(cTimeFormat, 32, "%d_%b_%Y_%H_%M_%S", sTime);

        cTimeFormat[31] = '\0';

//        CopyFile(CString(cLogDir) + cLogFile, CString(cLogDir) + CString(cTimeFormat) + "_" + cLogFile, FALSE);
    }

    // Clear all of the actions for each log.
    _FSI_STL::map<_FSI_STL::string, _FSI_STL::list<CMessageLogAction*> >::iterator mlIt = NULL;
    _FSI_STL::map<_FSI_STL::string, _FSI_STL::list<CMessageLogAction*> >::iterator mlendIt = NULL;

    mlIt    = m_mapLog2Actions.begin();
    mlendIt = m_mapLog2Actions.end();

    while (mlIt != mlendIt)
    {
        (*mlIt).second.clear();

        mlIt++;
    }

    m_mapLog2Actions.clear();

    // Delete the variants for each log.
    _FSI_STL::map<_FSI_STL::string, CVariant*>::iterator mlvIt = NULL;
    _FSI_STL::map<_FSI_STL::string, CVariant*>::iterator mlvendIt = NULL;

    mlvIt       = m_mapLog2Variant.begin();
    mlvendIt    = m_mapLog2Variant.end();

    while (mlvIt != mlvendIt)
    {
        delete (*mlvIt).second;

        mlvIt++;
    }

    m_mapLog2Variant.clear();
}

/////////////////////////////////////////////////////////////////////////////
//
// void CMessageLoggingSystemInterface::AddAction()
//
// Inputs           : CIOAction* pIOAction - a pointer to a 
//                                           CMessageLogAction that needs 
//                                           to be added to the data 
//                                           structures of this system 
//                                           interface so that the system 
//                                           interface knows which actions 
//                                           need to be updated when new 
//                                           data is available.
//
// Return Values    : None.
//
// Date             : 25 August 1999
//
// Engineer         : Billy Baker
//
// Description      : AddAction is called to add all CMessageLogActions 
//                    that have been created to the data structures in 
//                    this system interface.  These data structures are 
//                    used to determine which actions need to be 
//                    updated when new data arrives.
//
/////////////////////////////////////////////////////////////////////////////
void CMessageLoggingSystemInterface::AddAction(CIOAction* pIOAction)
{
    CMessageLogAction* pAction = (CMessageLogAction*)pIOAction;

    // Make sure that we got a pointer.
    if (pAction == NULL)
        return;

    // If a variant has not be created for the particular log in question,
    // then create it.
    _FSI_STL::string stlStrLogName = pAction->Variable();
    if (m_mapLog2Variant.find(stlStrLogName) == m_mapLog2Variant.end())
    {
        m_mapLog2Variant[stlStrLogName] = new CVariant();

        m_mapLog2Variant[stlStrLogName]->List(true);
        m_mapLog2Variant[stlStrLogName]->MaxSize(nLogSize);
    }

    // Tell the action which variant has the log messages.
    pAction->Variant(m_mapLog2Variant[stlStrLogName]);

    // Add the action to the log list of actions to update when
    // new message arrive.
    m_mapLog2Actions[stlStrLogName].push_back(pAction);
}

/////////////////////////////////////////////////////////////////////////////
//
// void CMessageLoggingSystemInterface::DeleteAction()
//
// Inputs           : CIOAction* pIOAction - a pointer to a 
//                                           CMessageLogAction that needs 
//                                           to be deleted from the data 
//                                           structures of this system 
//                                           interface.
//
// Return Values    : None.
//
// Date             : 25 August 1999
//
// Engineer         : Billy Baker
//
// Description      : DeleteAction is called to delete a CMessageLogActions 
//                    from the data structures in this system interface.  
//                    These data structures are used to determine which 
//                    actions need to be updated when new data arrives.
//
/////////////////////////////////////////////////////////////////////////////
void CMessageLoggingSystemInterface::DeleteAction(CIOAction* pIOAction)
{
    CMessageLogAction* pAction = (CMessageLogAction*)pIOAction;

    // Make sure that we got a pointer.
    if (pAction == NULL)
        return;

    // See if the log in question is still valid.
    _FSI_STL::string stlStrLogName = pAction->Variable();
    if (m_mapLog2Actions.find(stlStrLogName) != m_mapLog2Actions.end())
    {
        _FSI_STL::list<CMessageLogAction*>::iterator lIt = NULL;
        lIt = _FSI_STL::find(m_mapLog2Actions[stlStrLogName].begin(),
                             m_mapLog2Actions[stlStrLogName].end(),
                             pAction);

        // See if the action in question is in the list of actions
        // for this log.
        if (lIt !=  m_mapLog2Actions[stlStrLogName].end())
        {
            _FSI_STL::map<_FSI_STL::string, CVariant*>::iterator mIt = NULL;
            mIt = m_mapLog2Variant.find(stlStrLogName);

            // If the action is the last in the list,
            // then delete the variant for the log.
            if (m_mapLog2Actions[stlStrLogName].size() == 1)
            {
                if (mIt != m_mapLog2Variant.end())
                {
                    delete m_mapLog2Variant[stlStrLogName];

                    m_mapLog2Variant.erase(mIt);
                }

                // Remove the action from the list for the log.
                m_mapLog2Actions[stlStrLogName].erase(lIt);
            }
            else if (m_mapLog2Actions[stlStrLogName].size() > 1)
            {
                // Else, just remove the action from the list
                // for the log.
                m_mapLog2Actions[stlStrLogName].erase(lIt);
            }
        }
    }

    if (CWidget::IsValidAddress( ((CWidget*)pIOAction->Widget()) ) != INVALID)
        ((CWidget*)pIOAction->Widget())->Deleting(true);
}

/////////////////////////////////////////////////////////////////////////////
//
// void CMessageLoggingSystemInterface::AddMessage()
//
// Inputs           : _FSI_STL::string& stlStrLog - the name of the log to 
//                                      which the message should be added.
//                    _FSI_STL::string& stlStrMessage - the message to add.
//
// Return Values    : None.
//
// Date             : 25 August 1999
//
// Engineer         : Billy Baker
//
// Description      : AddMessage is called any time a local, host, or 
//                    debriefing comms system interface determines that 
//                    a message needs to be put into the log.
//
/////////////////////////////////////////////////////////////////////////////
void CMessageLoggingSystemInterface::AddMessage(const _FSI_STL::string& stlStrLog,
                                                const _FSI_STL::string& stlStrMessage)
{
    _FSI_STL::string stlStrFormattedMessage;

    // Get the current time.
    struct tm* sTime;
    time_t now;
    time(&now);
    sTime=localtime(&now);

    char cTimeFormat[32];
    strftime(cTimeFormat, 32, "%d %b %Y %H:%M:%S", sTime);
    cTimeFormat[31] = '\0';

    // Add the timestamp to the message.
    stlStrFormattedMessage = _FSI_STL::string(cTimeFormat) + "     " + stlStrMessage;

    // Add the new message to the log file.
    if (fp)
    {
       fwrite(stlStrFormattedMessage.c_str(), sizeof(char), 
              stlStrFormattedMessage.size(), fp);
       fwrite("\n", sizeof(char), 1, fp);
       fflush(fp);
    }

    m_sync.Lock();

    // Make sure that the log exists (has a variant or has actions).
    if (m_mapLog2Variant.find(stlStrLog) == m_mapLog2Variant.end())
    {
        m_mapLog2Variant[stlStrLog] = new CVariant();

        m_mapLog2Variant[stlStrLog]->List(true);
        m_mapLog2Variant[stlStrLog]->MaxSize(nLogSize);
    }

    // Set the current value of the variant to the new message and
    // add the message to the list in the variant.
    m_mapLog2Variant[stlStrLog]->Value(stlStrFormattedMessage);
    m_mapLog2Variant[stlStrLog]->AddValue(m_mapLog2Variant[stlStrLog]);

    m_mapLogChanged[stlStrLog] = true;

    m_sync.Unlock();
}

/////////////////////////////////////////////////////////////////////////////
//
// void CMessageLoggingSystemInterface::ToDelete()
//
// Inputs           : CIOAction* pIOAction - a pointer to a CMessageAction 
//                                         that needs to be deleted from the
//                                         CMessageLoggingSystemInterface
//                    .
//
// Return Values    : None.
//
// Date             : 25 August 1999
//
// Engineer         : Billy Baker
//
// Description      : ToDelete is used to add a pointer to a list of 
//                    CMessageLogActions that need to be deleted to this 
//                    system interface.  Because of the multiple 
//                    thread, the actions are not directly deleted.  
//                    Instead, when the system interface does an 
//                    update, it adds all of the actions that need to 
//                    be added and then deletes all of the actions that 
//                    need to be deleted.
//
/////////////////////////////////////////////////////////////////////////////
void CMessageLoggingSystemInterface::ToDelete(CIOAction* pIOAction)
{
    m_syncDelete.Lock();

    if (_FSI_STL::find(m_listActionsToDelete.begin(), 
                       m_listActionsToDelete.end(),
                       pIOAction) == m_listActionsToDelete.end())
    {
        m_listActionsToDelete.push_back(pIOAction);
    }

    m_syncDelete.Unlock();
}

/////////////////////////////////////////////////////////////////////////////
//
// void CMessageLoggingSystemInterface::ToAdd()
//
// Inputs           : CIOAction* pIOAction - a pointer to a CMessageAction 
//                                         that needs to be added to the
//                                         CMessageLoggingSystemInterface
//                    .
//
// Return Values    : None.
//
// Date             : 25 August 1999
//
// Engineer         : Billy Baker
//
// Description      : ToAdd is used to add a pointer to a list of 
//                    CMessageLogActions that need to be added to this 
//                    system interface.  Because of the multiple 
//                    thread, the actions are not directly added.  
//                    Instead, when the system interface does an 
//                    update, it adds all of the actions that need to 
//                    be added and then deletes all of the actions that 
//                    need to be deleted.
//
/////////////////////////////////////////////////////////////////////////////
void CMessageLoggingSystemInterface::ToAdd(CIOAction* pIOAction)
{
    m_syncAdd.Lock();

    if (_FSI_STL::find(m_listActionsToAdd.begin(), 
                       m_listActionsToAdd.end(),
                       pIOAction) == m_listActionsToAdd.end())
    {
        m_listActionsToAdd.push_back(pIOAction);
    }

    m_syncAdd.Unlock();
}

/////////////////////////////////////////////////////////////////////////////
//
// void CMessageLoggingSystemInterface::Stop()
//
// Inputs           : None.
//
// Return Values    : None.
//
// Date             : 25 August 1999
//
// Engineer         : Billy Baker
//
// Description      : Stop() is called before the class is destroyed in 
//                    order to free data for the log messages and the 
//                    actions that access the messages.  This is 
//                    usually done when the DLL is about to be unloaded 
//                    from memory.
//
/////////////////////////////////////////////////////////////////////////////
void CMessageLoggingSystemInterface::Stop()
{
    m_sync.Lock();

    // Remove any comms actions that have been marked for deleting.
    if (!m_listActionsToAdd.empty())
    {
        m_syncAdd.Lock();

        _FSI_STL::list<CIOAction*>::iterator lIt = 
                                            m_listActionsToAdd.begin();
        _FSI_STL::list<CIOAction*>::iterator lendIt = 
                                            m_listActionsToAdd.end();
        while (lIt != lendIt)
        {
            AddAction((CIOAction*)(*lIt));

            lIt++;
        }

        m_listActionsToAdd.clear();

        m_syncAdd.Unlock();
    }

    // Remove any comms actions that have been marked for deleting.
    if (!m_listActionsToDelete.empty())
    {
        m_syncDelete.Lock();

        _FSI_STL::list<CIOAction*>::iterator lIt = 
                                            m_listActionsToDelete.begin();
        _FSI_STL::list<CIOAction*>::iterator lendIt = 
                                            m_listActionsToDelete.end();
        while (lIt != lendIt)
        {
            DeleteAction((CIOAction*)(*lIt));

            lIt++;
        }

        m_listActionsToDelete.clear();

        m_syncDelete.Unlock();
    }

    // Free memory for the CVariant instances that
    // were created in a call to AddComms.
    _FSI_STL::map<_FSI_STL::string, CVariant*>::iterator mvIt = NULL;
    mvIt = m_mapLog2Variant.begin();
    _FSI_STL::map<_FSI_STL::string, CVariant*>::iterator mvendIt = NULL;
    mvendIt = m_mapLog2Variant.end();
    while (mvIt != mvendIt)
    {
        delete (*mvIt).second;
        mvIt++;
    }
    m_mapLog2Variant.clear();

    // Cleanup all of the actions.
    _FSI_STL::map<_FSI_STL::string, _FSI_STL::list<CMessageLogAction*> >::iterator mlIt = NULL;
    mlIt = m_mapLog2Actions.begin();
    _FSI_STL::map<_FSI_STL::string, _FSI_STL::list<CMessageLogAction*> >::iterator mlendIt = NULL;
    mlendIt = m_mapLog2Actions.end();

    _FSI_STL::list<CMessageLogAction*>::iterator lIt = NULL;

    while (mlIt != mlendIt)
    {
        lIt     = (*mlIt).second.begin();

        while (lIt != (*mlIt).second.end())
        {
            CMessageLogAction* pAction = (*lIt);
            DeleteAction((CIOAction*)(*lIt));
            pAction->Deleting(true);

            lIt = (*mlIt).second.begin();
        }
        (*mlIt).second.clear();

        mlIt++;
    }
    m_mapLog2Actions.clear();

    m_sync.Unlock();

    m_bStarted = false;
}

/////////////////////////////////////////////////////////////////////////////
//
// void CMessageLoggingSystemInterface::UpdateComms()
//
// Inputs           : None.
//
// Return Values    : None.
//
// Date             : 21 October 2000
//
// Engineer         : Billy Baker
//
// Description      : UpdateComms is called from 
//                    CCommsSystemInterface::UpdateComms at the same rate
//                    that screen refreshes occur so as to decrease the
//                    amount of traversals of the log data structures.
//
/////////////////////////////////////////////////////////////////////////////
void CMessageLoggingSystemInterface::UpdateComms()
{
    m_sync.Lock();

    // Add any  actions that have been marked for Adding.
    if (!m_listActionsToAdd.empty())
    {
        m_syncAdd.Lock();

        _FSI_STL::list<CIOAction*>::iterator lIt = 
                                            m_listActionsToAdd.begin();
        _FSI_STL::list<CIOAction*>::iterator lendIt = 
                                            m_listActionsToAdd.end();
        while (lIt != lendIt)
        {
            m_mapLogChanged[(*lIt)->Variable()] = true;
            AddAction((CIOAction*)(*lIt));

            lIt++;
        }

        m_listActionsToAdd.clear();

        m_syncAdd.Unlock();
    }

    // Remove any  actions that have been marked for deleting.
    if (!m_listActionsToDelete.empty())
    {
        m_syncDelete.Lock();

        _FSI_STL::list<CIOAction*>::iterator lIt = 
                                            m_listActionsToDelete.begin();
        _FSI_STL::list<CIOAction*>::iterator lendIt = 
                                            m_listActionsToDelete.end();
        while (lIt != lendIt)
        {
            DeleteAction((CIOAction*)(*lIt));

            lIt++;
        }

        m_listActionsToDelete.clear();

        m_syncDelete.Unlock();
    }

    _FSI_STL::map<_FSI_STL::string, _FSI_STL::list<CMessageLogAction*> >::iterator mlIt = NULL;
    _FSI_STL::map<_FSI_STL::string, _FSI_STL::list<CMessageLogAction*> >::iterator mlendIt = NULL;

    mlIt    = m_mapLog2Actions.begin();
    mlendIt = m_mapLog2Actions.end();

    _FSI_STL::list<CMessageLogAction*>::iterator lIt    = NULL;
    _FSI_STL::list<CMessageLogAction*>::iterator lendIt = NULL;

    // For each log, if the log changed, call OnUpdate for each
    // action attached to that log.
    while (mlIt != mlendIt)
    {
        if (true == m_mapLogChanged[(*mlIt).first])
        {
            m_mapLogChanged[(*mlIt).first] = false;

            // Update all of the actions.
            lIt     = (*mlIt).second.begin();
            lendIt  = (*mlIt).second.end();

            while (lIt != lendIt)
            {
                (*lIt)->OnUpdate();

                lIt++;
            }
        }

        mlIt++;
    }

    m_sync.Unlock();
}

//////////////////////////////////////////////////////////////////////
// Accessors
//////////////////////////////////////////////////////////////////////
_FSI_STL::map<_FSI_STL::string, CVariant*> CMessageLoggingSystemInterface::Logs()
{
    _FSI_STL::map<_FSI_STL::string, CVariant*> mapLogs;
    m_sync.Lock();
    
    mapLogs = m_mapLog2Variant;

    m_sync.Unlock();

    return mapLogs;
}