// iosComms.cpp : implementation file
//
// IOS Runtime Comms class.
//
// Copyright (c) 1997 by ZCT Systems Group, Inc.
// All Rights Reserved.
//
// Notes :
//
//	This file contains the IOS Runtime comms class.  This class is designed to
// perform all communications with the host via TCP/IP sockets.  The code in this
// class is extremely sensitive to changes in that it is multi-threaded, employs
// critical sections and events for multi-thread access to common memory, and uses
// sockets.  If you are not experienced in coding in this environment, ZCT strongly
// urges that you do not make changes to this class.
//

#include "..\core\stdafx.h"
#include "iosComms.h"
#include "iosCommsSocket.h"
#include "iosCommsObject.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

/////////////////////////////////////////////////////////////////////////////
// CIosComms

//
// CIosComms::CIosComms
// CIosComms constructor
// Entry   : nothing
// Returns : nothing
//

CIosComms::CIosComms()
{
	// Initialize variables.

	m_nCommandQueueSendPosition = 0;
	m_nSendReadCommandDelay = 0;
	m_nReadReturnTag = -1;
	m_nSetReadPointersTag = 0;
	m_pSocket = NULL;
	m_pThreadReceive = NULL;
	m_pThreadSend = NULL;
	m_bUpdateReadPointers = FALSE;
	m_bUpdateWrite = FALSE;
	m_bUpdateWritePointers = FALSE;
	m_bDisplayDialog = false;
	m_bTryToConnect = true;

    m_strMachineName = "";
    m_ulPort = 3000;

	// Initialize the critical section variable.

	InitializeCriticalSection(& m_csCommandsLock);

	// Initialize the comms commands.

	CommandsInitialize();
}

//
// CIosComms::~CIosComms
// CIosComms destructor
// Entry   : nothing
// Returns : nothing
//

CIosComms::~CIosComms()
{
	// Delete comms.

	Delete();
    DeleteCriticalSection(&m_csCommandsLock);
}

/////////////////////////////////////////////////////////////////////////////
// CIosComms functions.

//
// CIosComms::Close
// Close comms
// Entry   : nothing
// Returns : nothing
// Notes   :
//
//	Call this function to close comms.  Note, however, the comms commands
// remain intact.  In case the host temporarily disconnects, the commands
// will be ready when the host reconnects.  Call Delete() to close comms
// AND delete the comms commands.
//

void	CIosComms::Close()
{
	// Tell m_pSocket to abort so that ThreadReceive() and ThreadSend()
	// will terminate.

	if(m_pSocket != NULL)
		m_pSocket->SocketAbort();

	// Wait for ThreadReceive() to terminate.

	if(m_pThreadReceive)
	{
		// Wait for the thread to terminate, which is waiting for m_pSocket to abort.

		if(m_pThreadReceive)
		{
			HANDLE	hThread = m_pThreadReceive->m_hThread;
			::WaitForSingleObject(hThread, 5000);
			m_pThreadReceive = NULL;
		}
	}

	// Wait for ThreadSend() to terminate.

	if(m_pThreadSend)
	{
		// Wait for the thread to terminate, which is waiting for m_pSocket to abort.

		m_evThreadSend.SetEvent();

		if(m_pThreadSend)
		{
			HANDLE	hThread = m_pThreadSend->m_hThread;
			::WaitForSingleObject(hThread, 5000);
			m_pThreadSend = NULL;
		}

		// Flush the send buffer.

		CommandBufferFlush();
	}

	// Now that the threads are gone, delete the socket.

	if(m_pSocket != NULL)
	{
		delete m_pSocket;
		m_pSocket = NULL;
	}
}

//
// CIosComms::CommandBufferAdd
// Add a command to the comms send command buffer
// Entry   : pointer to command
// Returns : TRUE if added to comms send buffer, FALSE if not
// Notes   :
//
//	This function adds a command to the ThreadSend function recirculating
// buffer m_pCommandQueueSend.  Note that commands are sent in first in
// first out order, since they are added to the tail and removed from the
// head of m_pCommandQueueSend .
//

BOOL	CIosComms::CommandBufferAdd(iosRtCommand* piosRtCommand)
{
	// Block thread command access.

	CommandsLock();

	// Check for a send thread.

	BOOL	bReturn = FALSE;

	if(m_pThreadSend)
	{
		// Update() has created a send thread, check for an available buffer.

		if(m_pCommandQueueSend.GetCount() < MAX_QUEUED_COMMANDS)
		{
			// Buffer is available, get the next buffer space from the command queue memory pool.
			//
			// Note the memory pool for the buffers is allocated at compile time in the file
			// iosComms.h by the statement:
			//
			//		iosRtCommand	m_cmdCommandQueueSend[MAX_QUEUED_COMMANDS];
			//
			// The address of the next available buffer space is obtained by the statements:
			//
			//	iosRtCommand* pNextCommand = & m_cmdCommandQueueSend[m_nCommandQueueSendPosition ++];
			//
			//	and:
			//
			//  if(m_nCommandQueueSendPosition >= MAX_QUEUED_COMMANDS)
			//		m_nCommandQueueSendPosition = 0;
			//
			// where the first statement obtains the address, and the second statement recirculates
			// through the buffers.
			//
			// In order to guarantee that comms objects do not alter data that is being sent to
			// the host by ThreadSend(), the commands passed to this function are copied to these
			// buffers.  The reason for using an array buffer storage is to avoid "new" or "malloc"
			// calls that can create runtime overhead and memory fragmentation.
			//
			// To increase the size of the memory pool (and thus number of buffers), increase the
			// value of MAX_QUEUED_COMMANDS in iosComms.h and recompile.

			iosRtCommand*	pNextCommand = & m_cmdCommandQueueSend[m_nCommandQueueSendPosition ++];
			if(m_nCommandQueueSendPosition >= MAX_QUEUED_COMMANDS)
				m_nCommandQueueSendPosition = 0;

			// Copy the command to the buffer.

			memcpy(pNextCommand, piosRtCommand, ntohl(piosRtCommand->nCommandSize));

			// Add the pointer to the next command to the comms ThreadSend() queue.

			m_pCommandQueueSend.AddTail(pNextCommand);

			// Wake up ThreadSend().

			m_evThreadSend.SetEvent();

			// End.

			bReturn = TRUE;
		}
	}

	// Restore thread command access.

	CommandsUnlock();

	// End.

	return	bReturn;
}

//
// CIosComms::CommandBufferFlush
// Flush the comms command buffer
// Entry   : nothing
// Returns : nothing
//

void	CIosComms::CommandBufferFlush()
{
	// Block thread command access.

	CommandsLock();

	// Flush the command buffer.

	m_pCommandQueueSend.RemoveAll();

	// Restore thread command access.

	CommandsUnlock();
}

//
// CIosComms::CommandsFlush
// Flush any pending comms commands
// Entry   : nothing
// Returns : nothing
//

void	CIosComms::CommandsFlush()
{
	// Call update to send any pending comms commands to the host.

	Update();

	// Detach all comms objects currently attached to comms.

	POSITION	pos = m_commsReadObjects.GetHeadPosition();
	while(pos != NULL)
	{
		CIosCommsObject*	pComms = m_commsReadObjects.GetNext(pos);
		pComms->m_pReadPointer = NULL;
	}

	// Initialize the comms commands.

	CommandsInitialize();
}

//
// CIosComms::CommandsInitialize
// Commands initialization
// Entry   : nothing
// Returns : nothing
//

void	CIosComms::CommandsInitialize()
{
	// Block thread command access.

	CommandsLock();

	// Empty the comms object list.

	m_commsReadObjects.RemoveAll();

	// Empty the comms plot buffers.

	m_nHostPlots = 0;

	// Initialize the null command.

	m_cmdCommandNull.nCommand = htonl(IOSRT_COMMAND_NULL);
	m_cmdCommandNull.nCommandSize = htonl(IOSRT_COMMAND_HEADER_SIZE);

	// Initialize the read command.

	m_cmdCommandRead.nCommand = htonl(IOSRT_COMMAND_READ);
	m_cmdCommandRead.nCommandSize = htonl(IOSRT_COMMAND_HEADER_SIZE + IOSRT_COMMAND_READ_HEADER_SIZE);
	m_cmdCommandRead.Command.Read.nCount = 0;

	// Initialize the read return command.

	m_nCommandReadReturnPosition = 0;

	// Initialize the set read pointers command.

	m_cmdCommandSetReadPointers.nCommand = htonl(IOSRT_COMMAND_SET_READ_POINTERS);
	m_cmdCommandSetReadPointers.nCommandSize = htonl(IOSRT_COMMAND_HEADER_SIZE + IOSRT_COMMAND_SET_READ_POINTERS_HEADER_SIZE);
	m_cmdCommandSetReadPointers.Command.SetReadPointers.nPointers = 0;
	m_cmdCommandSetReadPointers.Command.SetReadPointers.nTag = htonl(m_nSetReadPointersTag);
	m_nCommandSetReadPointersPosition = 0;

	// Initialize the set write pointers command.

	m_cmdCommandSetWritePointers.nCommand = htonl(IOSRT_COMMAND_SET_WRITE_POINTERS);
	m_cmdCommandSetWritePointers.nCommandSize = htonl(IOSRT_COMMAND_HEADER_SIZE + IOSRT_COMMAND_SET_WRITE_POINTERS_HEADER_SIZE);
	m_cmdCommandSetWritePointers.Command.SetWritePointers.nPointers = 0;
	m_nCommandSetWritePointersPosition = 0;

	// Initialize the write command.

	m_cmdCommandWrite.nCommand = htonl(IOSRT_COMMAND_WRITE);
	m_cmdCommandWrite.nCommandSize = htonl(IOSRT_COMMAND_HEADER_SIZE + IOSRT_COMMAND_WRITE_HEADER_SIZE);
	m_cmdCommandWrite.Command.Write.nCount = 0;
	m_nCommandWritePosition = 0;

	// Restore thread command access.

	CommandsUnlock();
}

//
// CIosComms::CommandsLock
// Lock access to the comms commands
// Entry   : nothing
// Returns : nothing
//

void	CIosComms::CommandsLock()
{
	// Lock commands.

	EnterCriticalSection(& m_csCommandsLock);
}

//
// CIosComms::CommandsUnlock
// Unlock access to the comms commands
// Entry   : nothing
// Returns : nothing
//

void	CIosComms::CommandsUnlock()
{
	// Unlock commands.

	LeaveCriticalSection(& m_csCommandsLock);
}

//
// CIosComms::CommsObjectAttachRead
// Attach a comms object read to comms
// Entry   : pointer to comms object
// Returns : nothing
//

BOOL	CIosComms::CommsObjectAttachRead(CIosCommsObject* pComms)
{
	// Set the pointers.

	if(((m_nCommandSetReadPointersPosition + pComms->m_nDWords) < MAX_COMMAND_DATA) &&
	   ((m_nCommandReadReturnPosition + pComms->m_nDWords) < MAX_COMMAND_DATA))
	{
		// Space is available in both the set read pointers command and the read return
		// command.

		m_cmdCommandSetReadPointers.Command.SetReadPointers.pPointers[m_nCommandSetReadPointersPosition].nPointerType = htonl(pComms->m_nLinkType);
		m_cmdCommandSetReadPointers.Command.SetReadPointers.pPointers[m_nCommandSetReadPointersPosition].pPointer = htonl((DWORD)pComms->m_pLinkOffset);
		m_cmdCommandSetReadPointers.Command.SetReadPointers.nPointers = htonl(ntohl(m_cmdCommandSetReadPointers.Command.SetReadPointers.nPointers) + 1);
		m_cmdCommandSetReadPointers.nCommandSize = htonl(ntohl(m_cmdCommandSetReadPointers.nCommandSize) + sizeof(m_cmdCommandSetReadPointers.Command.SetReadPointers.pPointers[0]));

		// Tell the comms object where to read its data, then allocate and initialize the
		// data storage.

		if(pComms->m_nLinkType & PLOT_BIT)
		{
			if(m_nHostPlots < MAX_PLOT_BUFFERS)
			{
				// Obtain storage for the plot variable.

				HostPlot*	pNextPlot = & m_hpHostPlots[m_nHostPlots];

				// Initialize storage for the plot variable.

				pNextPlot->pAddress = NULL;
				pNextPlot->nCount = 0;
				pNextPlot->nTypeDWords = pComms->m_nDWords;

				// Change the read pointer type for the plot variable to incorporate
				// the plot buffer number.  Both iosRt and iosHost use this number as
				// a buffer number of the plot buffer for this variable plot.

				pNextPlot->nType = pComms->m_nLinkType | ((m_nHostPlots & PLOT_BUFFER_NUMBER_BITS) << PLOT_BUFFER_NUMBER_LSB);
				m_cmdCommandSetReadPointers.Command.SetReadPointers.pPointers[m_nCommandSetReadPointersPosition].nPointerType = htonl(pNextPlot->nType);
				
				// Count host plots.

				m_nHostPlots ++;

				// Count reads.

				m_cmdCommandRead.Command.Read.nCount = htonl(ntohl(m_cmdCommandRead.Command.Read.nCount) + pComms->m_nDWords);

				// Tell the comms object where to read its data from.

				pComms->m_pReadPointer = pNextPlot;
			}
		}
		else
		{
			// Initialize for a data variable.

			pComms->m_pReadPointer = & m_cmdCommandReadReturn.Command.ReadReturn.dwData[m_nCommandReadReturnPosition];

			int		nCount;

			for(nCount = 0; nCount < pComms->m_nDWords; nCount ++)
				m_cmdCommandReadReturn.Command.ReadReturn.dwData[m_nCommandReadReturnPosition ++] = 0;
			
			m_cmdCommandRead.Command.Read.nCount = htonl(ntohl(m_cmdCommandRead.Command.Read.nCount) + pComms->m_nDWords);
		}

		// Count set read pointers.

		m_nCommandSetReadPointersPosition ++;

		// Update the read pointers tag.

		m_nSetReadPointersTag += 1;

		// A read pointer update will be required.

		m_bUpdateReadPointers = TRUE;

		// Add the comms object to the comms read object list.

		m_commsReadObjects.AddTail(pComms);

		// Return success.

		 return TRUE;
	}
	else
	{
		// Return failure.

//		csLastErrorMessage = "CIosComms : data read count exceeded.";
//		CIosrtApp *pApp = (CIosrtApp*)AfxGetApp();
//		pApp->LogMessage(CString("CIosComms : data read count exceeded\t" \
//									" \t" + csCurrentDocumentName),
//							IDB_ERROR);
		return FALSE;
	}
}

//
// CIosComms::CommsObjectAttachWrite
// Attach a comms object write to comms
// Entry   : pointer to comms object
// Returns : nothing
//

BOOL	CIosComms::CommsObjectAttachWrite(CIosCommsObject* pComms)
{
	// Set the pointers.

	if(((m_nCommandSetWritePointersPosition + pComms->m_nDWords) < MAX_COMMAND_DATA) &&
	   ((m_nCommandWritePosition + pComms->m_nDWords) < MAX_COMMAND_DATA))
	{
		// Space is available in both the set write pointers command and the write command.

		m_cmdCommandSetWritePointers.Command.SetWritePointers.pPointers[m_nCommandSetWritePointersPosition].pPointer = htonl((DWORD)pComms->m_pLinkOffset);
		m_cmdCommandSetWritePointers.Command.SetWritePointers.pPointers[m_nCommandSetWritePointersPosition].nPointerType = htonl(pComms->m_nLinkType);
		m_cmdCommandSetWritePointers.Command.SetWritePointers.nPointers = htonl(ntohl(m_cmdCommandSetWritePointers.Command.SetWritePointers.nPointers) + 1);
		m_cmdCommandSetWritePointers.nCommandSize = htonl(ntohl(m_cmdCommandSetWritePointers.nCommandSize) + sizeof(m_cmdCommandSetWritePointers.Command.SetWritePointers.pPointers[0]));
		m_nCommandSetWritePointersPosition ++;

		// Tell the comms object where to write its data.

		pComms->m_pWritePointer = & m_cmdCommandWrite.Command.Write.wrData[m_nCommandWritePosition];

		// Allocate and initialize the proper number of DWORDS required for the data type.

		int		nCount;

		for(nCount = 0; nCount < pComms->m_nDWords; nCount ++)
			m_cmdCommandWrite.Command.Write.wrData[m_nCommandWritePosition ++] = 0;
		
		m_cmdCommandWrite.nCommandSize = htonl(ntohl(m_cmdCommandWrite.nCommandSize) + (pComms->m_nDWords << 2));
		m_cmdCommandWrite.Command.Write.nCount = htonl(ntohl(m_cmdCommandWrite.Command.Write.nCount) + pComms->m_nDWords);

		// A write pointer update will be required.

		m_bUpdateWritePointers = TRUE;

		// Return success.

		return TRUE;
	}
	else
	{
		// Return failure.

//		csLastErrorMessage = "CIosComms : data write count exceeded.";
//		CIosrtApp *pApp = (CIosrtApp*)AfxGetApp();
//		pApp->LogMessage(CString("CIosComms : data write count exceeded\t" \
//									" \t" + csCurrentDocumentName),
//							IDB_ERROR);
		return FALSE;
	}
}

//
// CIosComms::Delete
// Delete comms
// Entry   : nothing
// Returns : nothing
// Notes   :
//
//	Call this function to delete comms.  Note the comms commands are also
// deleted.  Call Close() to close comms without deleting the comms commands.
//

void	CIosComms::Delete()
{
	// Close comms.

	Close();

	// Initialize the comms commands.

	CommandsInitialize();
}

//
// CIosComms::Update
// Update the comms
// Entry   : nothing
// Returns : nothing
// Notes   :
//
//	This function controls all communications with the host.  The function
// is designed to be called at regular intervals in order to establish,
// maintain and reestablish communications with the host.
//

void	CIosComms::Update()
{
//	CIosrtApp* pApp = (CIosrtApp*)AfxGetApp();
//	if (m_bTryToConnect != false)
	{
		if(m_pSocket == NULL)
		{
			// Do not have a socket, see if a host name exists.

//			if(csHostName.GetLength())
			{
				// Do not have a socket, have a host name, create a socket.

				m_pSocket = new CIosCommsSocket();

				if(m_pSocket->SocketCreate() == FALSE)
				{
					// Socket create failed, closs comms.

					Close();
//					pApp->m_bSymbolsLoaded = false;
					m_bDisplayDialog = true;
				}
				else
				{
					// Socket created, now attempt connection to the host.

					m_pSocket->Connect(m_strMachineName, m_ulPort);
				}
			}
		}

		else if(m_pSocket->SocketHasError())
		{
			// Socket detected an error, close comms.

			Close();
//			pApp->m_bSymbolsLoaded = false;
			m_bDisplayDialog = true;
		}

		else if(m_pSocket->SocketIsClosed())
		{
			// Socket at other end closed, close comms.

			Close();
//			pApp->m_bSymbolsLoaded = false;
			m_bDisplayDialog = true;
		}

		else if(m_pSocket->SocketIsConnected())
		{
			// A socket has been created and is connected to the host.
			// Start the receive thread if not already started.
#if 0
			if (!pApp->m_bSymbolsLoaded)
			{
				if (m_bDisplayDialog == true)
				{
					m_bDisplayDialog = false;
					m_bTryToConnect = false;

					/// display wait dialog, replaces AfxMessageBox so button can be bigger, PCT

					CIosWaitDlg dlgWait;
					dlgWait.DoModal();

					{
						pApp->m_symSymbols.Load((char *)pApp->m_strSymbols.operator LPCTSTR());
						m_bTryToConnect = true;
						AfxGetMainWnd()->PostMessage(WM_COMMAND,ID_FILE_RESETERRORLOG,BN_CLICKED);	
						AfxGetMainWnd()->PostMessage(WM_NEW_PAGE, 0, (LPARAM)& csCurrentDocumentName);
					}
				}
			}
#endif
			if(m_pThreadReceive == NULL)
			{
				m_pThreadReceive = AfxBeginThread(ThreadReceive, this);
			}

			// Start the send thread if not already started.

			if(m_pThreadSend == NULL)
			{
				CommandBufferFlush();
				m_pThreadSend = AfxBeginThread(ThreadSend, this);
				m_bSendReadCommand = TRUE;
			}

			// If the send thread is started, send commands to the host.

			if(m_pThreadSend != NULL)
			{
				// Send set read pointers command to host.

				if(m_bUpdateReadPointers)
				{
					m_cmdCommandSetReadPointers.Command.SetReadPointers.nTag = htonl(m_nSetReadPointersTag);

					if(CommandBufferAdd(& m_cmdCommandSetReadPointers))
					{
						m_bUpdateReadPointers = FALSE;
					}
				}

				// Send the read data command to the host.  Note that the read
				// command is sent even though a page may not require any read
				// data thus the read command will request a read from the host
				// of 0 items.  This is intentional in that it acts as a "keep
				// alive" indicator for comms.

				if(m_bSendReadCommand)
				{
					// Send another read.

					CommandBufferAdd(& m_cmdCommandRead);

					// Read command has been sent.

					m_bSendReadCommand = FALSE;

					// Reset the send read delay counter.

					m_nSendReadCommandDelay = 0; //2000 / 100;
				}
				else
				{
					// No send read command yet, check the send read delay counter.

					if(m_nSendReadCommandDelay)
					{
						// Send read delay counter is non zero, count it down.

						m_nSendReadCommandDelay --;
					}
					else
					{
						// Send read delay counter is zero, force a send read command next time.

						m_bSendReadCommand = TRUE;
					}
				}

				// Send set write pointers command to host.

				if(m_bUpdateWritePointers)
				{
					if(CommandBufferAdd(& m_cmdCommandSetWritePointers))
					{
						// Command sent, reset the command.

						m_cmdCommandSetWritePointers.nCommand = htonl(IOSRT_COMMAND_SET_WRITE_POINTERS);
						m_cmdCommandSetWritePointers.nCommandSize = htonl(IOSRT_COMMAND_HEADER_SIZE + IOSRT_COMMAND_SET_WRITE_POINTERS_HEADER_SIZE);
						m_cmdCommandSetWritePointers.Command.SetWritePointers.nPointers = 0;
						m_nCommandSetWritePointersPosition = 0;

						m_bUpdateWritePointers = FALSE;
					}
				}

				// Send write data command to host.

				if((m_bUpdateWrite) & (! m_bUpdateWritePointers))
				{
					if(CommandBufferAdd(& m_cmdCommandWrite))
					{
						// Command sent, reset command.

						m_cmdCommandWrite.nCommand = htonl(IOSRT_COMMAND_WRITE);
						m_cmdCommandWrite.nCommandSize = htonl(IOSRT_COMMAND_HEADER_SIZE + IOSRT_COMMAND_WRITE_HEADER_SIZE);
						m_cmdCommandWrite.Command.Write.nCount = 0;
						m_nCommandWritePosition = 0;

						m_bUpdateWrite = FALSE;
					}
				}
			}

		}
	}
}

//
// CIosComms::VarIntTypeToOleVarType(int nVarType)
// Convert a var int type to an ole var type
// Entry   : the enum value
// Returns : the value of the ole var
// Notes   :
//
int	CIosComms::VarIntTypeToOleVarType(int nVarType)
{
	switch (nVarType)
	{
		case IOSRT_CHAR:
			return VT_BSTR;
		break;

		case IOSRT_SHORT:
			return VT_I2;
		break;

		case IOSRT_USHORT:
			return VT_UI2;
		break;

		case IOSRT_INT:
			return VT_I4;
		break;

		case IOSRT_UINT:
			return VT_UI4;
		break;

		case IOSRT_LOGICAL:
			return VT_BOOL;
		break;

		case IOSRT_FLOAT:
			return VT_R4;
		break;

		case IOSRT_DOUBLE:
			return VT_R8;
		break;

		default:
			return VT_EMPTY;
		break;
	};
}

//
// CIosComms::VarIntTypeToString(int nVarType)
// Convert a comms var int type to string
// Entry   : the value from the enum
// Returns : the string name of the enum
// Notes   :
//
CString	CIosComms::VarIntTypeToString(int nVarType)
{
	switch(nVarType)
	{
		case IOSRT_CHAR:
			return CString("IOSRT_CHAR");
		break;

		case IOSRT_SHORT:
			return CString("IOSRT_SHORT");
		break;

		case IOSRT_USHORT:
			return CString("IOSRT_USHORT");
		break;

		case IOSRT_INT:
			return CString("IOSRT_INT");
		break;

		case IOSRT_UINT:
			return CString("IOSRT_UINT");
		break;

		case IOSRT_LOGICAL:
			return CString("IOSRT_LOGICAL");
		break;

		case IOSRT_FLOAT:
			return CString("IOSRT_FLOAT");
		break;

		case IOSRT_DOUBLE:
			return CString("IOSRT_DOUBLE");
		break;

		default:
			return CString("IOSRT_UNKNOWN");
		break;
	};
}

//
// CIosComms::VarStringToVarIntType(CString& strVarTypeName)
// Convert a string comms var name to int
// Entry   : the name of the enum value
// Returns : the value from the enum
// Notes   :
//
int	CIosComms::VarStringToVarIntType(const CString& strVarTypeName)
{
	if (strstr(strVarTypeName,"IOSRT_CHAR"))
	{
		return IOSRT_CHAR;
	}
	else if (strstr(strVarTypeName,"IOSRT_SHORT"))
	{
		return IOSRT_SHORT;
	}
	else if (strstr(strVarTypeName,"IOSRT_USHORT"))
	{
		return IOSRT_USHORT;
	}
	else if (strstr(strVarTypeName,"IOSRT_INT"))
	{
		return IOSRT_INT;
	}
	else if (strstr(strVarTypeName,"IOSRT_UINT"))
	{
		return IOSRT_UINT;
	}
	else if (strstr(strVarTypeName,"IOSRT_LOGICAL"))
	{
		return IOSRT_LOGICAL;
	}
	else if (strstr(strVarTypeName,"IOSRT_FLOAT"))
	{
		return IOSRT_FLOAT;
	}
	else if (strstr(strVarTypeName,"IOSRT_DOUBLE"))
	{
		return IOSRT_DOUBLE;
	}
	else // IOSRT_UNKNOWN
	{
		return -1;
	}

}

/////////////////////////////////////////////////////////////////////////////
// CIosComms threads.

//
// CIosComms::ThreadReceive
// Thread to receive commands from the host
// Entry   : pointer to CMainFrame
// Returns : 0
//

UINT	CIosComms::ThreadReceive(LPVOID pParam)
{
	// Obtain a pointer to the comms object.

	CIosComms*	pComms = (CIosComms *)pParam;

	// Obtain a pointer to the main frame object.

//	CWnd*	pMainFrame = AfxGetMainWnd();

	// Receive data from host.

	while(TRUE)
	{
		// Read the command header.

		HostCommand			hcCommandIn;

		if((pComms->m_pSocket->SocketRead((char *)& hcCommandIn, HOST_COMMAND_HEADER_SIZE)) == FALSE)
		{
			// Read error, end the thread.

			AfxEndThread(0);
		}

		// Validate the command header.

		int		nCommand = ntohl(hcCommandIn.nCommand);
		int		nCommandSize = ntohl(hcCommandIn.nCommandSize);

		if( nCommand == 0 )  // <paul & scott>
			continue;

		if((nCommand < HOST_FIRST_COMMAND) ||
		   (nCommand > HOST_LAST_COMMAND) ||
		   (nCommandSize > HOST_COMMAND_MAX_SIZE))
		{
			// Header error, end the thread.

			CString	csError;
			
			if((nCommand < HOST_FIRST_COMMAND) ||
			   (nCommand > HOST_LAST_COMMAND))
				csError.Format("CIosComms :  received invalid host command '%d'.", nCommand);
			else
				csError.Format("CIosComms :  received invalid host command size %d (max is %d).", nCommandSize, HOST_COMMAND_MAX_SIZE);

//			pMainFrame->SendMessage(WM_IOSRT_MESSAGE, MESSAGE_ERROR, (LPARAM)csError.operator LPCTSTR());
			AfxEndThread(0);
		}

		// Header is valid, read the command.

		switch(nCommand)
		{
			case HOST_COMMAND_NULL:
			{
				// HOST_COMMAND_NULL command.  Nothing more to do.

			}
			break;

			case HOST_COMMAND_NO_READ_POINTERS:
			{
				// HOST_COMMAND_NO_READ_POINTERS command.  Send the current
				// read pointers to the host.

				pComms->m_bUpdateReadPointers = TRUE;
			}
			break;

			case HOST_COMMAND_NO_WRITE_POINTERS:
			{
				// HOST_COMMAND_NO_WRITE_POINTERS command.  Send the current
				// write pointers to the host.

				pComms->m_bUpdateWritePointers = TRUE;
			}
			break;

			case HOST_COMMAND_READ_OUT_OF_RANGE:
			{
				// HOST_COMMAND_READ_OUT_OF_RANGE command.  Send the current
				// read pointers to the host.

				pComms->m_bUpdateReadPointers = TRUE;
			}
			break;

			case HOST_COMMAND_READ_RETURN:
			{
				// HOST_COMMAND_READ_RETURN command, read the read return
				// parameters.

				if((pComms->m_pSocket->SocketRead((char *)& hcCommandIn.Command.ReadReturn, HOST_COMMAND_READ_RETURN_HEADER_SIZE)) == FALSE)
				{
					// Read error, end the thread.

					AfxEndThread(0);
				}

				int		nCount = ntohl(hcCommandIn.Command.ReadReturn.nCount);

				// Validate the read return parameters.

				if(nCount > MAX_COMMAND_DATA)
				{
					// Error, too much data.

//					pMainFrame->SendMessage(WM_IOSRT_MESSAGE, (WPARAM)MESSAGE_ERROR, (LPARAM)"CIosComms :  host command read returned too much data.");
					AfxEndThread(0);
				}

				// Now read the read return data.

				if((pComms->m_pSocket->SocketRead((char *)hcCommandIn.Command.ReadReturn.dwData, nCount * sizeof(hcCommandIn.Command.ReadReturn.dwData[0]))) == FALSE)
				{
					// Read error, end the thread.

					AfxEndThread(0);
				}

				// Block main thread command access.

				pComms->CommandsLock();

				// Copy the read data to the read return command.

				memcpy((char *)& pComms->m_cmdCommandReadReturn, (char *)& hcCommandIn, ntohl(hcCommandIn.nCommandSize));

				// Update the read return tag number.

				pComms->m_nReadReturnTag = ntohl(hcCommandIn.Command.ReadReturn.nTag);
				pComms->m_bSendReadCommand = TRUE;

				// Restore main thread command access.

				pComms->CommandsUnlock();
			}
			break;

			case HOST_COMMAND_WRITE_OUT_OF_RANGE:
			{
				// HOST_COMMAND_WRITE_OUT_OF_RANGE command.  Send the current
				// write pointers to the host.

				pComms->m_bUpdateWritePointers = TRUE;
			}
			break;

			case HOST_COMMAND_PLOT:
			{
				// HOST_COMMAND_PLOT command, read the plot parameters.

				if((pComms->m_pSocket->SocketRead((char *)& hcCommandIn.Command.Plot, HOST_COMMAND_PLOT_HEADER_SIZE)) == FALSE)
				{
					// Read error, end the thread.

					AfxEndThread(0);
				}

				// Validate the plot parameters.

				int		nCount = ntohl(hcCommandIn.Command.Plot.nCount);

				if(nCount > MAX_PLOT_DATA)
				{
					// Error, too much data.

//					pMainFrame->SendMessage(WM_IOSRT_MESSAGE, (WPARAM)MESSAGE_ERROR, (LPARAM)"CIosComms :  host command plot 1 returned too much data.");
					AfxEndThread(0);
				}

				// Now read the plot data.

				nCount *= (ntohl(hcCommandIn.Command.Plot.nTypeDWords) << 2);

				if((pComms->m_pSocket->SocketRead((char *)hcCommandIn.Command.Plot.nArray, nCount)) == FALSE)
				{
					// Read error, end the thread.

					AfxEndThread(0);
				}

				// Copy the plot array.

				nCount = (ntohl(hcCommandIn.Command.Plot.nType) >> PLOT_BUFFER_NUMBER_LSB) & PLOT_BUFFER_NUMBER_BITS;

				pComms->CommandsLock();

				pComms->m_hpHostPlots[nCount].nCount = ntohl(hcCommandIn.Command.Plot.nCount);
				pComms->m_hpHostPlots[nCount].uHostTick = ntohl(hcCommandIn.Command.Plot.uHostTick);

				memcpy((char *)pComms->m_hpHostPlots[nCount].nArray,
					   (char *)hcCommandIn.Command.Plot.nArray,
					   ntohl(hcCommandIn.Command.Plot.nCount) * (ntohl(hcCommandIn.Command.Plot.nTypeDWords) << 2));

				pComms->CommandsUnlock();
			}
			break;
		}
	}

	// End the thread.

	AfxEndThread(0);
	return 0L;
}

//
// CIosComms::ThreadSend
// Thread to send commands to the host
// Entry   : pointer CIosComms
// Returns : 0
// Notes   :
//
//	This function removes the commands placed in the recirculating buffer
// m_pCommandQueueSend by CommandBufferAdd and sends them to the host.
// Note that commands are sent in first in first out order, since they
// are added to the tail and removed from the head of m_pCommandQueueSend .
//

UINT	CIosComms::ThreadSend(LPVOID pParam)
{
	// Obtain a pointer to the comms object.

	CIosComms*	pComms = (CIosComms*)pParam;

	// Obtain a pointer to the main frame object.

//	CWnd* pMainFrame = AfxGetMainWnd();

	// Send commands to the host.

	while(pComms->m_pSocket->SocketIsAborted() == FALSE)
	{
		// Block main thread command access.

		pComms->CommandsLock();

		// Get the pointer to the next command.  Note the pointer is not
		// REMOVED.  This keeps CommandBufferAdd() from using this buffer
		// until ThreadSend is finished with it.  It will be removed after
		// it is sent.

		iosRtCommand*	chNextCommand = NULL;	
		if(!(pComms->m_pCommandQueueSend.IsEmpty()))
			chNextCommand = pComms->m_pCommandQueueSend.GetHead();

		// Restore main thread command access.

		pComms->CommandsUnlock();

		// If there is a next command, send it.

		if(chNextCommand)
		{
			// There is, validate the header.

			int		nCommand = ntohl(chNextCommand->nCommand);
			int		nCommandSize = ntohl(chNextCommand->nCommandSize);

			if((nCommand < IOSRT_FIRST_COMMAND) ||
			   (nCommand > IOSRT_LAST_COMMAND) ||
			   (nCommandSize > IOSRT_COMMAND_MAX_SIZE))
			{
				// Header error, end the thread.

				CString	csError;

				if((nCommand < IOSRT_FIRST_COMMAND) ||
				   (nCommand > IOSRT_LAST_COMMAND))
				   csError.Format("CIosComms :  invalid IosRt command '%d'.", nCommand);
				else
					csError.Format("CIosComms :  invalid IosRt command size %d (max is %d).", nCommandSize, IOSRT_COMMAND_MAX_SIZE);

//				pMainFrame->SendMessage(WM_IOSRT_MESSAGE, MESSAGE_ERROR, (LPARAM)csError.operator LPCTSTR());
				AfxEndThread(0);
			}

			// Header valid, send command.

			if((pComms->m_pSocket->SocketSend((char *)chNextCommand, nCommandSize)) == FALSE)
			{
				// Error on send, end the thread.

				AfxEndThread(0);
			}

			// Block main thread command access.

			pComms->CommandsLock();

			// Since the command is now sent, remove the command pointer.
			// This frees the buffer that contained the command for use
			// by CommandBufferAdd().

			chNextCommand = pComms->m_pCommandQueueSend.RemoveHead();

			// Restore main thread command access.

			pComms->CommandsUnlock();
		}
		else
		{
			// There were no more commands, suspend thread execution until an
			// event notification is sent.

			pComms->m_evThreadSend.Lock();
		}
	}

	// Socket is aborted, end the thread.

	AfxEndThread(0);
	return 0L;
}

