/////////////////////////////////////////////////////////////////////////////
//
//           F L I G H T S A F E T Y   I N T E R N A T I O N A L
//                     Simulation Systems Division
//                      2700 North Hemlock Circle
//                     Broken Arrow, Oklahoma 74012
//                          (918) 259-4000
/////////////////////////////////////////////////////////////////////////////
//
// DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
// Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
// technical or operational data or information from automatic dissemination
// under the International Exchange Program or by other means.  This protection
// covers information required solely for administrative or operational
// purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
//
// WARNING:  This document contains technical data whose export is restricted
// by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
// Executive Order 12470.  Violation of these export control laws is subject
// to severe criminal penalties.  Dissemination of this document is controlled
// under DOD Directive 5230.25
//
/////////////////////////////////////////////////////////////////////////////
//
//
// Filename         : FailureSummarySystemInterface.cpp
//
// Date             : 16 February 2000
//
// Engineer         : Billy Baker
//
// Revision         : $Revision: 1.5 $
//
// Description      : FailureSummarySystemInterface.cpp contains the 
//                    implementation of the 
//                    CFailureSummarySystemInterface class.
//
// Classification   : UNCLASSIFIED
//
// Requirements     : None.
//
// Components Used  : CFailureSummaryAction, _FSI_STL::string, 
//                    _FSI_STL::map, _FSI_STL::list, CAction, CIOAction.
//
// Operational 
//    Restrictions  : Machine dependencies/restrictions
//                        None.
//                    Design dependencies/restrictions
//                        None.
//                    Operations containing dependencies/restrictions
//                        None.
//                    Compiler dependencies/restrictions
//                        None.
//                    Other conditions for proper execution
//                        None.
//
// Environment      : Operating system(s) - Microsoft Windows NT 4.0 with
//                                              NT service pack 3, 4, or 5
//                                          Microsoft Windows NT 2000
//
//                    Compiler(s) - Visual C++ 6.0
//
//                    Architechure(s) - Intel Pentium, Pentium II
/////////////////////////////////////////////////////////////////////////////

/////////////////////////////////////////////////////////////////////////////
//
//                              R e v i s i o n   H i s t o r y
//
/////////////////////////////////////////////////////////////////////////////
// $Log: FailureSummarySystemInterface.cpp $                                                                     
// Revision 1.5  2000/06/08 06:50:19  billyb                                                                     
// Added ToFailureString.                                                                     
// Revision 1.4  2000/05/26 16:28:57  billyb                                                                     
// Made adding and deleting failures cause the failure summary                                                                     
// actions to update.                                                                     
// Revision 1.3  2000/05/18 20:53:45  billyb                                                                     
// Added methods for working with failure variables as                                                                     
// specified in the variable definition file.                                                                     
// Revision 1.2  2000/03/06 16:23:07  billyb                                                                     
// Changed function signatures to eliminate warnings.                                                                     
/////////////////////////////////////////////////////////////////////////////
#include "..\core\stdafx.h"
#include <fstream>
#include "FailureSummaryAction.h"
#include "FailureSummarySystemInterface.h"

_FSI_STL::map<enum eFAILURE_ITEMS, _FSI_STL::map<CVariant*, _FSI_STL::string> > CFailureSummarySystemInterface::m_mapFailureVariables;

_FSI_STL::string CFailureSummarySystemInterface::m_stlStrFile = "c:\\temp\\FSISuite_Failure_Summary_Log.txt";

CFailureSummarySystemInterface::CFailureSummarySystemInterface()
{
    m_nFailedCBs                = 0;
    m_nFailedMalfs_Active       = 0;
    m_nFailedMalfs_Armed        = 0;
    m_nFailedNavaids            = 0;
    m_bScreenUpdateNeeded       = true;
    m_ulOrder                   = 0;
}

CFailureSummarySystemInterface::~CFailureSummarySystemInterface()
{
    _FSI_STL::map<long, DescWithOrder*>::iterator mIt    = m_mapFailuresInOrder.begin();
    _FSI_STL::map<long, DescWithOrder*>::iterator mendIt = m_mapFailuresInOrder.end();

    while (mIt != mendIt)
    {
        delete (*mIt).second;

        mIt++;
    }
}

void CFailureSummarySystemInterface::AddFailure(enum eFAILURE_ITEMS eFailureItem,
                                                const _FSI_STL::string& stlStrItem, 
                                                const _FSI_STL::string& stlStrDescription)
{
    m_sync.Lock();

    // If the item is already failed, then return without
    // changing anything.
    if (m_mapFailedItems[eFailureItem].find(stlStrItem) !=
        m_mapFailedItems[eFailureItem].end())
    {
        m_sync.Unlock();
        return;
    }

    // Get the current time.
    struct tm* sTime;
    time_t now;
    time(&now);
    sTime=localtime(&now);

    char cTimeFormat[32];
    strftime(cTimeFormat, 32, "%d %b %Y %H:%M:%S", sTime);
    cTimeFormat[31] = '\0';

    // Fill out a DESC_WITH_ORDER structure that will
    // have the description, the order of the failure,
    // and the time.
    DescWithOrder* pDWO = new DescWithOrder;
    if (pDWO == NULL)
    {
        m_sync.Unlock();
        return;
    }

    pDWO->s_stlStrDesc  = stlStrDescription;
    pDWO->s_stlStrItem  = stlStrItem;
    pDWO->s_stlStrTime  = _FSI_STL::string(cTimeFormat);
    pDWO->s_ulOrder     = m_ulOrder;
    ++m_ulOrder;

    m_mapFailedItems[eFailureItem][stlStrItem] = pDWO;
    m_mapFailuresInOrder[pDWO->s_ulOrder] = pDWO;

    m_sync.Unlock();

    switch(eFailureItem)
    {
    case CB:
        m_nFailedCBs++;
        break;
    case MALFUNCTION_ACTIVE:
        m_nFailedMalfs_Active++;
        break;
    case MALFUNCTION_ARMED:
        m_nFailedMalfs_Armed++;
        break;
    case NAVAID_KILLED:
        m_nFailedNavaids++;
        break;
    }

    m_bScreenUpdateNeeded = true;
}

void CFailureSummarySystemInterface::DeleteFailure(enum eFAILURE_ITEMS eFailureItem,
                                                   const _FSI_STL::string& stlStrItem,
                                                   const _FSI_STL::string& stlStrDescription)
{
    _FSI_STL::map<_FSI_STL::string, DescWithOrder*>::iterator mIt;

    m_sync.Lock();

    mIt = m_mapFailedItems[eFailureItem].find(stlStrItem);
    if (mIt == m_mapFailedItems[eFailureItem].end())
    {
        m_sync.Unlock();
        return;
    }

    DescWithOrder* pDWO = (*mIt).second;
    if (pDWO->s_stlStrDesc != stlStrDescription)
    {
        m_sync.Unlock();
        return;
    }

    _FSI_STL::map<long, DescWithOrder*>::iterator mFIOIt;
    mFIOIt = m_mapFailuresInOrder.find(pDWO->s_ulOrder);
    if (mFIOIt != m_mapFailuresInOrder.end())
        m_mapFailuresInOrder.erase(mFIOIt);

    m_mapFailedItems[eFailureItem].erase(mIt);

    delete pDWO;

    m_sync.Unlock();

    switch(eFailureItem)
    {
    case CB:
        m_nFailedCBs--;
        break;
    case MALFUNCTION_ACTIVE:
        m_nFailedMalfs_Active--;
        break;
    case MALFUNCTION_ARMED:
        m_nFailedMalfs_Armed--;
        break;
    case NAVAID_KILLED:
        m_nFailedNavaids--;
        break;
    }

    m_bScreenUpdateNeeded = true;
}

void CFailureSummarySystemInterface::DeleteFailuresOfType(enum eFAILURE_ITEMS eFailureItem)
{
    m_sync.Lock();

    if (m_mapFailedItems.find(eFailureItem) != m_mapFailedItems.end())
    {
        // Failure of eFailureItem type were in the map.
        // Delete all of the DescWithOrder's and
        // remove them from m_mapFailuresInOrder;
        _FSI_STL::map<_FSI_STL::string, DescWithOrder*>::iterator mIt;
        _FSI_STL::map<_FSI_STL::string, DescWithOrder*>::iterator mendIt;
        mIt     = m_mapFailedItems[eFailureItem].begin();
        mendIt  = m_mapFailedItems[eFailureItem].end();

        DescWithOrder* pDWO = NULL;
        _FSI_STL::map<long, DescWithOrder*>::iterator mFIOIt;

        while(mIt != mendIt)
        {
            pDWO    = (*mIt).second;
            mFIOIt  = m_mapFailuresInOrder.find(pDWO->s_ulOrder);

            if (mFIOIt != m_mapFailuresInOrder.end())
                m_mapFailuresInOrder.erase(mFIOIt);

            delete pDWO;

            ++mIt;
        }

        m_mapFailedItems.erase(m_mapFailedItems.find(eFailureItem));
    }

    m_sync.Unlock();

    switch(eFailureItem)
    {
    case CB:
        m_nFailedCBs = 0;
        break;
    case MALFUNCTION_ACTIVE:
        m_nFailedMalfs_Active = 0;
        break;
    case MALFUNCTION_ARMED:
        m_nFailedMalfs_Armed = 0;
        break;
    case NAVAID_KILLED:
        m_nFailedNavaids = 0;
        break;
    }

    m_bScreenUpdateNeeded = true;
}

_FSI_STL::string CFailureSummarySystemInterface::Summary()
{
    CString str;

    str.Format("CBs  %d    Active Malfs  %d    Armed Malfs  %d    Navaids  %d", 
               m_nFailedCBs,
               m_nFailedMalfs_Active,
               m_nFailedMalfs_Armed,
               m_nFailedNavaids);

    return _FSI_STL::string((LPCTSTR)str);
}


long int CFailureSummarySystemInterface::ToFailureType(const _FSI_STL::string& rstlStrType)
{
    if (rstlStrType == "CB")
    {
        return CB;
    }

    if (rstlStrType == "MALFUNCTION_ACTIVE")
    {
        return MALFUNCTION_ACTIVE;
    }

    if (rstlStrType == "MALFUNCTION_ARMED")
    {
        return MALFUNCTION_ARMED;
    }

    if (rstlStrType == "NAVAID_KILLED")
    {
        return NAVAID_KILLED;
    }

    return UNKNOWN_FAILURE;
}

_FSI_STL::string CFailureSummarySystemInterface::ToFailureString(const long int lType)
{
    if (lType == CB)
    {
        return _FSI_STL::string("CB");
    }

    if (lType == MALFUNCTION_ACTIVE)
    {
        return _FSI_STL::string("MALFUNCTION_ACTIVE");
    }

    if (lType == MALFUNCTION_ARMED)
    {
        return _FSI_STL::string("MALFUNCTION_ARMED");
    }

    if (lType == NAVAID_KILLED)
    {
        return _FSI_STL::string("NAVAID_KILLED");
    }

    return _FSI_STL::string("");
}

void CFailureSummarySystemInterface::EvaluateFailure(enum eFAILURE_ITEMS eFailure, 
                                                     CVariant* pVariant, 
                                                     _FSI_STL::string& rstlStrData)
{
    if (pVariant == NULL)
        return;

    if (!pVariant->WasChanged())
        return;

    switch (eFailure)
    {
    case CB:
        {
            m_bScreenUpdateNeeded = true;

            if ((bool)*pVariant)
            {
                DeleteFailure(CB, rstlStrData, " popped.");
            }
            else
            {
                AddFailure(CB, rstlStrData, " popped.");
            }
        }
        break;
    case MALFUNCTION_ACTIVE:
        break;
    case MALFUNCTION_ARMED:
        break;
    case NAVAID_KILLED:
        {
            CVariant* pVariantCount = CCommsShared::FindVariant("navaid_kill_count");
            if (pVariantCount != NULL)
            {
                m_bScreenUpdateNeeded = true;

                DeleteFailuresOfType(NAVAID_KILLED);

                long lStep  = 0;
                long lCount = (long)(*pVariantCount);
                char *pData = (char *)(pVariant->CreateVar());
                CString strIdent;

                lCount = __min(lCount, pVariant->Length() / 4);

                while (lStep < lCount)
                {
                    strIdent.Format("%.4s", &(pData[lStep*4]));
                    strIdent.MakeUpper();
                    AddFailure(NAVAID_KILLED, (LPCTSTR)strIdent, " killed.");
                    lStep ++;
                }
            }
        }
        break;
    default:
        break;
    }
}

void CFailureSummarySystemInterface::UpdateComms()
{
    _FSI_STL::map<enum eFAILURE_ITEMS, _FSI_STL::map<CVariant*, _FSI_STL::string> >::iterator mIt;
    _FSI_STL::map<enum eFAILURE_ITEMS, _FSI_STL::map<CVariant*, _FSI_STL::string> >::iterator mendIt;

    mIt     = m_mapFailureVariables.begin();
    mendIt  = m_mapFailureVariables.end();

    _FSI_STL::map<CVariant*, _FSI_STL::string>::iterator mvIt;
    _FSI_STL::map<CVariant*, _FSI_STL::string>::iterator mvendIt;
    
    while (mIt != mendIt)
    {
        mvIt     = (*mIt).second.begin();
        mvendIt  = (*mIt).second.end();
        
        while (mvIt != mvendIt)
        {
            EvaluateFailure((*mIt).first, (*mvIt).first, (*mvIt).second);

            mvIt++;
        }

        mIt++;
    }


    if (true == m_bScreenUpdateNeeded)
    {
        DumpAllFailuresToFile();

        m_bScreenUpdateNeeded = false;
        m_sync.Lock();

        _FSI_STL::list<CAction*>::iterator lIt = m_listActions.begin();
        _FSI_STL::list<CAction*>::iterator lendIt = m_listActions.end();
        while (lIt != lendIt)
        {
            (*lIt)->OnUpdate();

            lIt++;
        }

        m_sync.Unlock();
    }
}

/////////////////////////////////////////////////////////////////////////////
//
// void CFailureSummarySystemInterface::DumpAllFailuresToFile()
//
// Inputs           : None.
//
// Return Values    : None.
//
// Date             : 19 October 2000
//
// Engineer         : Billy Baker
//
// Description      : DumpAllFailuresToFile is called in UpdateComms
//                    whenever a screen update is needed.  A screen
//                    update is needed whenever a fialure has been added
//                    or deleted.
//
/////////////////////////////////////////////////////////////////////////////
void CFailureSummarySystemInterface::DumpAllFailuresToFile()
{
    _FSI_STL::map<long, DescWithOrder*>::iterator mIt    =  m_mapFailuresInOrder.begin();
    _FSI_STL::map<long, DescWithOrder*>::iterator mendIt =  m_mapFailuresInOrder.end();

    // The map has at least one item to dump.
    _FSI_STL::ofstream failureFile;
    failureFile.open(m_stlStrFile.c_str());
    if (!failureFile.fail())
    {
        // The file could be opened.
        DescWithOrder* pDWO;
        while (mIt != mendIt)
        {
            pDWO = (*mIt).second;

            failureFile << pDWO->s_stlStrTime.c_str();
            failureFile << "     ";
            failureFile << pDWO->s_stlStrItem.c_str();
            failureFile << pDWO->s_stlStrDesc.c_str();
            failureFile << "\n";

            ++mIt;
        }

        failureFile.close();
    }
}

void CFailureSummarySystemInterface::AddFailureVariable(_FSI_STL::string& rstlStrType,
                                                        _FSI_STL::string& rstlStrData,
                                                        CVariant* pVariant)
{
    enum eFAILURE_ITEMS eFailure = (enum eFAILURE_ITEMS)ToFailureType(rstlStrType);

    if (eFailure != UNKNOWN_FAILURE)
    {
        if (m_mapFailureVariables[eFailure].find(pVariant) == m_mapFailureVariables[eFailure].end())
        {
            m_mapFailureVariables[eFailure][pVariant] = rstlStrData;
        }
    }
}