/////////////////////////////////////////////////////////////////////////////
//
//           F L I G H T S A F E T Y   I N T E R N A T I O N A L
//                     Simulation Systems Division
//                      2700 North Hemlock Circle
//                     Broken Arrow, Oklahoma 74012
//                          (918) 259-4000
/////////////////////////////////////////////////////////////////////////////
//
// DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
// Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
// technical or operational data or information from automatic dissemination
// under the International Exchange Program or by other means.  This protection
// covers information required solely for administrative or operational
// purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
//
// WARNING:  This document contains technical data whose export is restricted
// by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
// Executive Order 12470.  Violation of these export control laws is subject
// to severe criminal penalties.  Dissemination of this document is controlled
// under DOD Directive 5230.25
//
/////////////////////////////////////////////////////////////////////////////
//
//
// Filename         : CommsSystemInterface.cpp
//
// Date             : 09 February 1999
//
// Engineer         : Billy Baker
//
// Revision         : $Revision: 1.21 $
//
// Description      : CommsSystemInterface.cpp contains the implementation of 
//                    the CCommsSystemInterface class.  This class 
//                    serves as the base class for all classes that 
//                    seek to communicate with simulation.  The base 
//                    class defines a number of data structures for 
//                    managing the distribution of values from 
//                    simulation to the graphical elements that need 
//                    the value.  Also, the base class keeps track of 
//                    all derived classes for display in the comms 
//                    configuration page.  Several virtual methods are 
//                    defined which must be implemented by the derived 
//                    class in order to communicate.
//
//                    A typical algorithm is as follows:  when the comms
//                    DLL is loaded, the name of the protocol is read from
//                    the registry and an instance of the correct comms system
//                    interface derived class is created.  Then, the variables
//                    file is read.  During reading, the m_mapVarName2Data map
//                    should be filled in.  At this point, nothing should
//                    happen until the first instance of a CCommsAction is 
//                    created.  Each instance will know its VarName in the
//                    m_mapVarName2Data map.The CCommsAction instance is 
//                    then added to the list that goes with its VarName, 
//                    m_mapVarName2Comms.  The first addition to the list will 
//                    also create a CVariant instance that is added to the 
//                    m_mapVarName2Variant map.  When a page is changed, each
//                    CCommsAction will delete itself from the map of VarName to
//                    comms.  The last deletion will delete the CVariant that
//                    was in VarName to Variant.  When the application shuts down, 
//                    stop must be called before deleting the instance of the
//                    comms system interface derived class.  Stop will delete
//                    the memory in m_mapVarName2Data as well as any CVariants
//                    that may still be in m_mapVarName2Variant.  The deletion of
//                    CVariant is more for dynamically changing the protocol.
//                    When stop is called in the normal fashion, all of the
//                    CVariants should already be deleted.  Stop will also
//                    clean up the debriefing memory and write the debriefing
//                    data.
//
//                    Malfunctions may be an exception to the above scenario.
//
// Classification   : UNCLASSIFIED
//
// Requirements     : None.
//
// Components Used  : CVariant, map, string, CCommsAction, CXMLPage, CVar
//                    iabledData.
//
// Operational 
//    Restrictions  : Machine dependencies/restrictions
//                        None.
//                    Design dependencies/restrictions
//                        None.
//                    Operations containing dependencies/restrictions
//                        None.
//                    Compiler dependencies/restrictions
//                        None.
//                    Other conditions for proper execution
//                        None.
//
// Environment      : Operating system(s) - Microsoft Windows NT 4.0 with
//                                              NT service pack 3
//                                          Microsoft Windows NT 5.0
//
//                    Compiler(s) - Visual C++ 5.0 with VisC++ service pack 3
//                                  Visual C++ 6.0
//
//                    Architechure(s) - Intel Pentium, Pentium II
/////////////////////////////////////////////////////////////////////////////

/////////////////////////////////////////////////////////////////////////////
//
//                              R e v i s i o n   H i s t o r y
//
/////////////////////////////////////////////////////////////////////////////
// $Log: CommsSystemInterface.cpp $
// Revision 1.21  2000/06/16 20:46:04  billyb
// Recorded missing data to debrief file.
// Revision 1.20  2000/06/08 06:49:35  billyb
// Added InitComplete to tell a protocol that all of the DLLs have
// been loaded and the protocol can proceed.  Called from 
// CCommsComponentInterface::InitComplete.
// Revision 1.19  2000/05/18 20:53:04  billyb
// Added Failure summary updating.
// Revision 1.18  2000/05/03 21:31:46  billyb
// Removed code to delete a CVariant in DeleteComms.
// Added calls for update callbacks.
// Removed overlays from statistics.
// Changed mechanism for updating screen to fix freeze on
// areamap.
// Revision 1.17  2000/04/07 05:19:05  billyb
// Changed FPS display.
// Revision 1.16  2000/03/31 22:44:44  billyb
// Made screen updates for log based on screen update rate.
// Fixed crashes from using bad iterator.  Removed Debugging 
// code.  Made list access more thread safe. 
// Revision 1.15  2000/03/06 21:32:54  billyb
// latest
// Revision 1.13.1.1  2000/03/06 21:31:37  billyb
// Changed function signatures to elimiinate warnings.
// Added methods for dynamically adding and deleting data
// from the host.  Added casts to eliminate warnings.
// Revision 1.13  2000/02/01 19:52:27  billyb
// Added initialization of map of IOS alias to host var name.
// Revision 1.12  2000/01/27 07:59:32  billyb
// Added missing data to debriefing file for data format and
// length.
// Revision 1.11  2000/01/21 10:07:12  billyb
// Changed definition of the update list to be in depth order.
// Revision 1.10  1999/12/31 16:42:22  billyb
// Will now added widgets that need to be redrawn because of
// WM_PAINT messages.
// Revision 1.9  1999/11/26 18:48:50  billyb
// Changed algorithm for updating screen.  It no longer dynamically
// allocates a list for each window each pass.  The new 
// Core::CUpdateList class is used.
// Revision 1.8  1999/11/16 07:26:36  billyb
// Changed conditional that check for a host error (HostError())
// to check for equal to zero as opposed to greater than -1.
// Revision 1.7  1999/11/15 18:31:45  billyb
// Initialized new static rate variables.  Changed UpdateComms
// to add FPS for each parent wnd to the map of statistics.  Also,
// changed algorithm to use new maps to better determine which
// parent wnds really needs to be updated.
// Revision 1.6  1999/11/04 19:55:07  billyb
// Changed for loop to while and added variable for map end()
// call to decrease calls to end().  Changed size() call to empty()
// since size() is a calculation and empty() is an accessor for
// a member variable.
// Revision 1.5  1999/09/28 06:07:01  billyb
// Small change to set a window to redraw status here
// rather than when the message is received to update.
// If the message took a long time to get to the update
// method, then two redraws may have been posted.
// Revision 1.4  1999/09/02 17:10:40  billyb
// Initialized a comms action's initial data format value when 
// the comms action is added in AddComms.
// Revision 1.3  1999/08/31 17:05:43  billyb
// Added to conditional in AddComms to take care of problem where leaving page
// and coming back in some cases would not create a new variant for use
// with comms.
// Revision 1.2  1999/08/25 20:03:48  billyb
// Changed the way that Stop() cleans up the data structures to match the changes in CMessageLoggingSystemInterface.  Before these changes a crash could occur when using Ctrl+T to change options for the whole application.
//
#include "..\core\stdafx.h"

#include <algorithm>

#include "CommsSystemInterface.h"
#include "CommsAction.h"
#include "FailureSummaryAction.h"

#include "CommsStatsPage.h"

#include "..\core\UpdateList.h"

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif

typedef CCommsSystemInterface* (*CreateComms)();

_FSI_STL::map<_FSI_STL::string, _FSI_STL::list<CCommsAction*> > CCommsSystemInterface::m_mapVarName2Comms;

_FSI_STL::map<_FSI_STL::string, CVariant*>            CCommsSystemInterface::m_mapVarName2Variant;

_FSI_STL::map<_FSI_STL::string, CVariableData*>       CCommsSystemInterface::m_mapVarName2Data;
_FSI_STL::map<_FSI_STL::string, CreateComms>          CCommsSystemInterface::m_mapCommsSystems;
_FSI_STL::map<_FSI_STL::string, CVariableData*>       CCommsSystemInterface::m_mapVarName2Debrief;
_FSI_STL::map<_FSI_STL::string, _FSI_STL::list<CRange*> >       CCommsSystemInterface::m_mapVarName2Ranges;
_FSI_STL::map<_FSI_STL::string, _FSI_STL::string>    CCommsSystemInterface::m_mapStatistics;
_FSI_STL::map<_FSI_STL::string, _FSI_STL::string> CCommsSystemInterface::m_mapHostToIOSAlias;

CCriticalSection                  CCommsSystemInterface::m_sync;
CCriticalSection                  CCommsSystemInterface::m_syncCommsDelete;
CCriticalSection                  CCommsSystemInterface::m_syncCommsAdd;
CCriticalSection                  CCommsSystemInterface::m_syncRefresh;
CCriticalSection                  CCommsSystemInterface::m_syncUpdateCallback;

unsigned char                     CCommsSystemInterface::m_ucDebriefRatio = 2;
unsigned char                     CCommsSystemInterface::m_ucDebriefUpdatePass = 2;
unsigned char                     CCommsSystemInterface::m_ucScreenUpdateRatio = 1;
_FSI_STL::map<HWND, unsigned char> CCommsSystemInterface::m_mapScreenUpdatePass;
_FSI_STL::map<HWND, unsigned long> CCommsSystemInterface::m_mapFPS;
char*                             CCommsSystemInterface::m_pcDebriefData = NULL;
CFile                             CCommsSystemInterface::m_fileDebrief;
unsigned long int                 CCommsSystemInterface::m_ulDebriefBufferSize = 0;
HANDLE                            CCommsSystemInterface::m_hPipeWrite;
HANDLE                            CCommsSystemInterface::m_hPipeRead;
bool                              CCommsSystemInterface::m_bConnected = true;
bool                              CCommsSystemInterface::m_bWritePending;
_FSI_STL::list<CCommsAction*>          CCommsSystemInterface::m_listCommsActionsToDelete;
_FSI_STL::list<CCommsAction*>          CCommsSystemInterface::m_listCommsActionsToAdd;
_FSI_STL::map<HWND, _FSI_STL::map<CWidget*, bool> >         CCommsSystemInterface::m_mapWidgetsToRefresh;
_FSI_STL::map<HWND, _FSI_STL::map<CWidget*, bool> >  CCommsSystemInterface::m_mapLocalsToRefresh;
_FSI_STL::list<UpdateCallback>    CCommsSystemInterface::m_listUpdateCallbacks;
float                             CCommsSystemInterface::m_fRate               = 30.f;
float                             CCommsSystemInterface::m_fDebriefRate        = 15.f;
float                             CCommsSystemInterface::m_fScreenUpdateRate   = 30.f;
unsigned long int                 CCommsSystemInterface::m_ulFPSCounter         = 0;
bool                              CCommsSystemInterface::m_bInitComplete        = false;
//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

/////////////////////////////////////////////////////////////////////////////
//
// CCommsSystemInterface::CCommsSystemInterface()
//
// Inputs           : None.
//
// Return Values    : None.
//
// Date             : 09 February 1999
//
// Engineer         : Billy Baker
//
// Description      : Default constructor.  Set the started state to 
//                    false or stopped.  Set the default timer rate to 
//                    30 hz.
//
/////////////////////////////////////////////////////////////////////////////
CCommsSystemInterface::CCommsSystemInterface()
{
    m_pXMLVariablesPage     = NULL;
    m_bRecordDebrief        = false;
    m_bConnected            = false;
    m_ulPort                = 3000;
    m_stlStrSecondaryFile   = _FSI_STL::string("");

    m_pCommsThread          = NULL;
}

/////////////////////////////////////////////////////////////////////////////
//
// CCommsSystemInterface::~CCommsSystemInterface()
//
// Inputs           : None.
//
// Return Values    : None.
//
// Date             : 09 February 1999
//
// Engineer         : Billy Baker
//
// Description      : Default destructor.  Delete any memory for a CXMLPage.
//
/////////////////////////////////////////////////////////////////////////////
CCommsSystemInterface::~CCommsSystemInterface()
{
    if (m_pXMLVariablesPage != NULL)
    {
        delete m_pXMLVariablesPage;
    }

    m_mapHostToIOSAlias.clear();
}

//////////////////////////////////////////////////////////////////////
// Accessors
//////////////////////////////////////////////////////////////////////

/////////////////////////////////////////////////////////////////////////////
//
// string CCommsSystemInterface::MachineName()
//
// Inputs           : None.
//
// Return Values    : the name of the machine to connect to.
//
// Date             : 09 February 1999
//
// Engineer         : Billy Baker
//
// Description      : MachineName() is an accessor method to get the name 
//                    of the machine to connect to.
//
/////////////////////////////////////////////////////////////////////////////
_FSI_STL::string CCommsSystemInterface::MachineName()
{
    return m_stlStrMachineName;
}

/////////////////////////////////////////////////////////////////////////////
//
// bool CCommsSystemInterface::Debrief()
//
// Inputs           : None.
//
// Return Values    : true - debrief data is recorded
//                    false - debrief data is not recorded
//
// Date             : 12 February 1999
//
// Engineer         : Billy Baker
//
// Description      : Debrief() is an accessor method to get whether to
//                    record debrief data.
//
/////////////////////////////////////////////////////////////////////////////
bool CCommsSystemInterface::Debrief()
{
    return m_bRecordDebrief;
}

/////////////////////////////////////////////////////////////////////////////
//
// unsigned long int CCommsSystemInterface::Port()
//
// Inputs           : None.
//
// Return Values    : the port number for the connection.
//
// Date             : 12 February 1999
//
// Engineer         : Billy Baker
//
// Description      : Port() is an accessor method to get the port number
//                    for the comms connection.
//
/////////////////////////////////////////////////////////////////////////////
unsigned long int CCommsSystemInterface::Port()
{
    return m_ulPort;
}

/////////////////////////////////////////////////////////////////////////////
//
// float CCommsSystemInterface::Rate()
//
// Inputs           : None.
//
// Return Values    : the Rate for the connection.
//
// Date             : 7 June 1999
//
// Engineer         : Billy Baker
//
// Description      : Rate() is an accessor method to get the Rate 
//                    for the comms connection.
//
/////////////////////////////////////////////////////////////////////////////
float CCommsSystemInterface::Rate()
{
    return m_fRate;
}

/////////////////////////////////////////////////////////////////////////////
//
// float CCommsSystemInterface::DebriefRate()
//
// Inputs           : None.
//
// Return Values    : the DebriefRate for saving debriefing data.
//
// Date             : 7 June 1999
//
// Engineer         : Billy Baker
//
// Description      : DebriefRate() is an accessor method to get the DebriefRate
//                    which tells how often data will be saved for debriefing.
//
/////////////////////////////////////////////////////////////////////////////
float CCommsSystemInterface::DebriefRate()
{
    return m_fDebriefRate;
}

/////////////////////////////////////////////////////////////////////////////
//
// float CCommsSystemInterface::ScreenUpdateRate()
//
// Inputs           : None.
//
// Return Values    : the ScreenUpdateRate updating the screen.
//
// Date             : 7 June 1999
//
// Engineer         : Billy Baker
//
// Description      : ScreenUpdateRate() is an accessor method to get the ScreenUpdateRate
//                    which tells how often the screen will be updated.
//
/////////////////////////////////////////////////////////////////////////////
float CCommsSystemInterface::ScreenUpdateRate()
{
    return m_fScreenUpdateRate;
}

//////////////////////////////////////////////////////////////////////
// Mutators
//////////////////////////////////////////////////////////////////////

/////////////////////////////////////////////////////////////////////////////
//
// void CCommsSystemInterface::MachineName()
//
// Inputs           : string& rstlStrMachineName - a new name for the machine
//                                                 to connect to.
//
// Return Values    : None.
//
// Date             : 09 February 1999
//
// Engineer         : Billy Baker
//
// Description      : MachineName() is a mutator method to set the name of the
//                    machine to connect to.
//
/////////////////////////////////////////////////////////////////////////////
void CCommsSystemInterface::MachineName(const _FSI_STL::string& rstlStrMachineName)
{
    m_stlStrMachineName = rstlStrMachineName;
}


/////////////////////////////////////////////////////////////////////////////
//
// void CCommsSystemInterface::Debrief()
//
// Inputs           : bool bRecordDebrief - whether to record debrief data.
//
// Return Values    : None.
//
// Date             : 12 February 1999
//
// Engineer         : Billy Baker
//
// Description      : Debrief() is a mutator method to set whether to
//                    record debrief data.
//
/////////////////////////////////////////////////////////////////////////////
void CCommsSystemInterface::Debrief(bool bRecordDebrief)
{
    m_bRecordDebrief = bRecordDebrief;
}

/////////////////////////////////////////////////////////////////////////////
//
// void CCommsSystemInterface::Port(unsigned long int ulPort)
//
// Inputs           : the port number for the connection.
//
// Return Values    : None.
//
// Date             : 12 February 1999
//
// Engineer         : Billy Baker
//
// Description      : Port() is a mutator method to set the port number
//                    for the comms connection.
//
/////////////////////////////////////////////////////////////////////////////
void CCommsSystemInterface::Port(unsigned long int ulPort)
{
    m_ulPort = ulPort;
}

/////////////////////////////////////////////////////////////////////////////
//
// void CCommsSystemInterface::Rate(unsigned int unRate)
//
// Inputs           : the Rate for the connection.
//
// Return Values    : None.
//
// Date             : 7 June 1999
//
// Engineer         : Billy Baker
//
// Description      : Rate() is a mutator method to set the Rate
//                    for the comms connection.
//
/////////////////////////////////////////////////////////////////////////////
void CCommsSystemInterface::Rate(float fRate)
{
    m_fRate = fRate;

    // Set the debriefing ratio
    m_ucDebriefRatio = (unsigned char)(m_fRate / m_fDebriefRate);

    // Make sure that something will be recorded.
    if (m_fRate < m_fDebriefRate)
    {
        m_ucDebriefRatio = 1;
    }

    // Make sure we get the first set of data.
    m_ucDebriefUpdatePass = m_ucDebriefRatio;

    // Set the screen update ratio
    m_ucScreenUpdateRatio = (unsigned char)(m_fRate / m_fScreenUpdateRate);

    // Make sure that something will be recorded.
    if (m_fRate < m_fScreenUpdateRate)
    {
        m_ucScreenUpdateRatio = 1;
    }
}

/////////////////////////////////////////////////////////////////////////////
//
// void CCommsSystemInterface::DebriefRate(unsigned int unDebriefRate)
//
// Inputs           : the DebriefRate for recording debriefing data.
//
// Return Values    : None.
//
// Date             : 7 June 1999
//
// Engineer         : Billy Baker
//
// Description      : DebriefRate() is a mutator method to set the DebriefRate
//                    which tells how often to record the debriefing data.
//
/////////////////////////////////////////////////////////////////////////////
void CCommsSystemInterface::DebriefRate(float fDebriefRate)
{
    m_fDebriefRate     = fDebriefRate;
    m_ucDebriefRatio    = (unsigned char)(m_fRate / m_fDebriefRate);

    // Make sure that something will be recorded.
    if (m_fRate < m_fDebriefRate)
    {
        m_ucDebriefRatio = 1;
    }

    // Make sure we get the first set of data.
    m_ucDebriefUpdatePass = m_ucDebriefRatio;
}

/////////////////////////////////////////////////////////////////////////////
//
// void CCommsSystemInterface::ScreenUpdateRate(unsigned int unScreenUpdateRate)
//
// Inputs           : the ScreenUpdateRate for updating the screen.
//
// Return Values    : None.
//
// Date             : 7 June 1999
//
// Engineer         : Billy Baker
//
// Description      : ScreenUpdateRate() is a mutator method to set the ScreenUpdateRate
//                    which tells how often the screen will be updated.
//
/////////////////////////////////////////////////////////////////////////////
void CCommsSystemInterface::ScreenUpdateRate(float fScreenUpdateRate)
{
    m_fScreenUpdateRate    = fScreenUpdateRate;
    m_ucScreenUpdateRatio   = (unsigned char)(m_fRate / m_fScreenUpdateRate);

    // Make sure that something will be recorded.
    if (m_fRate < m_fScreenUpdateRate)
    {
        m_ucScreenUpdateRatio = 1;
    }
}

/////////////////////////////////////////////////////////////////////////////
//
// void CCommsSystemInterface::AddComms()
//
// Inputs           : CIOAction* pIOAction - an instance of
//                                                 CCommsAction to add to
//                                                 the data structures.
//
// Return Values    : None.
//
// Date             : 09 February 1999
//
// Engineer         : Billy Baker
//
// Description      : AddComms() should be called whenever a 
//                    CCommsAction instance is initialized.  When 
//                    called, the variable that the CCommsAction is 
//                    supposed to watch will be checked against the 
//                    names of the variables in m_mapVarName2Data. If 
//                    VarName is already in m_mapVarName2Comms, then at 
//                    least one CommsAction has already been added for 
//                    VarName.  If this is the case then add the new 
//                    instance to the list and set the new instance's 
//                    variant pointer to the same variant pointer as 
//                    the other CommsActions attached to VarName.
//                    
//                    If VarName is not already in m_mapVarName2Comms, then 
//                    add it.  Create a new CVariant and add it to 
//                    m_mapVarName2Variant.  Be sure to set the 
//                    CommsAction's variant to the new CVariant.
//
/////////////////////////////////////////////////////////////////////////////
void CCommsSystemInterface::AddComms(CIOAction* pIOAction)
{
    CCommsAction* pCommsAction = (CCommsAction*)pIOAction;
    // Make sure that we got a CCommsAction pointer.
    if (pCommsAction == NULL)
    {
        return;
    }

    const _FSI_STL::string& stlStrName = pCommsAction->Variable();

    // Make sure pCommsAction's variables are in the list of varaibles.
    if (m_mapVarName2Data.find(stlStrName) != m_mapVarName2Data.end())
    {
        if (m_mapVarName2Ranges.find(stlStrName) != m_mapVarName2Ranges.end())
        {
            _FSI_STL::list<CRange*>::iterator lIt = 
                                        m_mapVarName2Ranges[stlStrName].begin();
            _FSI_STL::list<CRange*>::iterator lendIt = 
                                        m_mapVarName2Ranges[stlStrName].end();

            while (lIt != lendIt)
            {
                pCommsAction->AddRange((*lIt));
                lIt++;
            }
        }

        pCommsAction->InitialDataFormat(m_mapVarName2Data[stlStrName]->InitialDataFormat());

        // If the variable for pCommsAction has been added once, then
        // assign the same memory for the variant that stores the value
        // for Variable to pCommsAction.  Else, add pCommsAction to the 
        // map of variable IDs to CCommsAction pointers.  Then create a 
        // new variant for Variable and assign it to pCommsAction's variant
        // pointer.
        if (m_mapVarName2Comms.find(stlStrName) != m_mapVarName2Comms.end() &&
            !(*m_mapVarName2Comms.find(stlStrName)).second.empty())
        {
            // find the passed commsaction in the list to see if it has
            // been added once.
            _FSI_STL::list<CCommsAction*>::iterator lIt = 
                              _FSI_STL::find(m_mapVarName2Comms[stlStrName].begin(),
                                        m_mapVarName2Comms[stlStrName].end(), 
                                        pCommsAction);

            if (lIt == m_mapVarName2Comms[stlStrName].end())
            {
                // get the variant for this variable and give a pointer
                // to it to the passed comms action.
                CVariant* pVariant = m_mapVarName2Variant[stlStrName];

                pCommsAction->Variant(pVariant);

                m_mapVarName2Comms[stlStrName].push_back(pCommsAction);
            }
        }
        else
        {
            m_mapVarName2Comms[stlStrName].push_back(pCommsAction);

            // Need to store a variant in a list (??) for sending to the
            // host. Create the variant here and assign it to pCommsAction.
            if (m_mapVarName2Variant.find(stlStrName) == m_mapVarName2Variant.end())
            {
                m_mapVarName2Variant[stlStrName] = new CVariant;
            }

            pCommsAction->Variant(m_mapVarName2Variant[stlStrName]);
        }
    }
}

/////////////////////////////////////////////////////////////////////////////
//
// void CCommsSystemInterface::DeleteComms()
//
// Inputs           : CCommsAction* pCommsAction - an instance of CCommsAction
//                                                 to delete from the data
//                                                 structures.
//
// Return Values    : None.
//
// Date             : 09 February 1999
//
// Engineer         : Billy Baker
//
// Description      : DeleteComms() should be called whevener a 
//                    CCommsAction is deleted.  Its purpose is to 
//                    remove the deleted CCommsAction from the data 
//                    structures so that an attempt will not be made to 
//                    call its OnUpdate method after it is deleted.
//
//                    A check is first performed to see if the variable
//                    that the CommsAction is supposed to watch is in
//                    m_mapVarName2Data.  If it isn't, then the AddComms
//                    call should not have added anything.
//
//                    A search for VarName in m_mapVarName2Comms is 
//                    performed.  If it isn't found, then the CommsAction does
//                    not need to be deleted from the data structures.  If it
//                    is found, then a search of the list of CommsActions
//                    attached to the VarName is performed for the CommsAction that
//                    was passed.  If it is found, it is deleted from the list.
//                    If it is the last CommsAction in the list, then the 
//                    CVariant for that VarName is also deleted and VarName is 
//                    removed from m_mapVarName2Comms.
//
/////////////////////////////////////////////////////////////////////////////
void CCommsSystemInterface::DeleteComms(CIOAction* pIOAction)
{
    // Make sure that we got a CCommsAction pointer.
    CCommsAction* pCommsAction = (CCommsAction*)pIOAction;
    if (pCommsAction == NULL)
    {
        return;
    }

    const _FSI_STL::string& stlStrName = pCommsAction->Variable();

    // Make sure pCommsAction's variable is in the list of varaibles.
    if (m_mapVarName2Data.find(stlStrName) != m_mapVarName2Data.end())
    {
        // Valid variable.  Get ID for variable.
        CVariableData* pVarData = m_mapVarName2Data[stlStrName];

        if (pVarData == NULL)
        {
            return;
        }

        if (m_mapVarName2Comms.find(stlStrName) != m_mapVarName2Comms.end())
        {
            // VarName is in the map of VarName2Comms which means that it has 
            // commsactions attached to it.
            // Remove the comms object from the list of comms objects.
            _FSI_STL::map<_FSI_STL::string, _FSI_STL::list<CCommsAction*> >::iterator mIt;
            mIt = m_mapVarName2Comms.find(stlStrName);

            // find the passed commsaction in the list for ulID
            _FSI_STL::list<CCommsAction*>::iterator lIt = 
                                               _FSI_STL::find((*mIt).second.begin(),
                                                         (*mIt).second.end(), 
                                                         pCommsAction);
            if (lIt != (*mIt).second.end())
            {
                (*mIt).second.erase(lIt);
            }
        }
    }

    if (CWidget::IsValidAddress( ((CWidget*)pIOAction->Widget()) ) != INVALID)
    {
        ((CWidget*)pIOAction->Widget())->Deleting(true);
    }
}

/////////////////////////////////////////////////////////////////////////////
//
// void CCommsSystemInterface::UpdateComms()
//
// Inputs           : None.
//
// Return Values    : None.
//
// Date             : 09 February 1999
//
// Engineer         : Billy Baker
//
// Description      : UpdateComms() must be called from each derived 
//                    class's method for reading data from simulation.  
//                    It serves to tell each instance of CCommsAction 
//                    to update its parent graphical element.
//
/////////////////////////////////////////////////////////////////////////////
void CCommsSystemInterface::UpdateComms()
{
    static unsigned char ucLogUpdate        = m_ucScreenUpdateRatio;

    // Add any comms actions that have been marked for Adding.
    if (!m_listCommsActionsToAdd.empty())
    {
        m_syncCommsAdd.Lock();

        _FSI_STL::list<CCommsAction*>::iterator lIt = 
                                            m_listCommsActionsToAdd.begin();
        _FSI_STL::list<CCommsAction*>::iterator lendIt = 
                                            m_listCommsActionsToAdd.end();
        while (lIt != lendIt)
        {
            AddComms((CIOAction*)(*lIt));
            lIt++;
        }

        m_listCommsActionsToAdd.clear();

        m_syncCommsAdd.Unlock();
    }

    // Remove any comms actions that have been marked for deleting.
    if (!m_listCommsActionsToDelete.empty())
    {
        m_syncCommsDelete.Lock();

        _FSI_STL::list<CCommsAction*>::iterator lIt = 
                                            m_listCommsActionsToDelete.begin();
        _FSI_STL::list<CCommsAction*>::iterator lendIt = 
                                            m_listCommsActionsToDelete.end();
        while (lIt != lendIt)
        {
            DeleteComms((CIOAction*)(*lIt));
            lIt++;
        }

        m_listCommsActionsToDelete.clear();

        m_syncCommsDelete.Unlock();
    }

    // Send the read data on its way.
    _FSI_STL::map<_FSI_STL::string, _FSI_STL::list<CCommsAction*> >::iterator caIt;
    _FSI_STL::map<_FSI_STL::string, _FSI_STL::list<CCommsAction*> >::iterator caendIt;

    _FSI_STL::list<CCommsAction*>::iterator lIt;
    _FSI_STL::list<CCommsAction*>::iterator lendIt;

    caIt = m_mapVarName2Comms.begin();
    caendIt = m_mapVarName2Comms.end();

    // For each ID, run through the list of CCommsAction instances.
    while (caIt !=  caendIt)
    {
        lIt     = (*caIt).second.begin();
        lendIt  = (*caIt).second.end();

        // For each CCommsAction instance, update the parent graphical
        // element.
        while (lIt != lendIt)
        {
            if (*lIt != NULL)
            {
                (*lIt)->OnUpdate();
            }

            lIt++;
        }

        caIt++;
    }

    if (ucLogUpdate >= m_ucScreenUpdateRatio)
    {
        if (CCommsShared::m_pLogSystemInterface != NULL)
        {
            CCommsShared::m_pLogSystemInterface->UpdateComms();
        }

        ucLogUpdate = 1;
    }
    else
    {
        ucLogUpdate++;
    }

    if (CCommsShared::m_pFailureSummarySystemInterface != NULL)
    {
        CCommsShared::m_pFailureSummarySystemInterface->UpdateComms();
    }

    m_ulFPSCounter++;

    _FSI_STL::map<CWidget*, bool>::iterator                         mwIt    = NULL;
    _FSI_STL::map<CWidget*, bool>::iterator                         mwendIt = NULL;
    _FSI_STL::map<HWND, _FSI_STL::map<CWidget*, bool> >::iterator   mIt     = NULL;
    _FSI_STL::map<HWND, _FSI_STL::map<CWidget*, bool> >::iterator   mendIt  = NULL;
    _FSI_STL::map<HWND, _FSI_STL::map<CWidget*, bool> >             mapPaintWidgets;

    m_syncRefresh.Lock();

    mIt     = m_mapLocalsToRefresh.begin();
    mendIt  = m_mapLocalsToRefresh.end();

    // Copy the widgets that need an update from local comms to
    // the map of all widgets that need to be updated.
    while (mIt != mendIt)
    {
        mwIt    = (*mIt).second.begin();
        mwendIt = (*mIt).second.end();

        while (mwIt != mwendIt)
        {
            (m_mapWidgetsToRefresh[(*mIt).first])[(*mwIt).first] = true;
            mwIt++;
        }

        m_mapLocalsToRefresh[(*mIt).first].clear();

        mIt++;
    }

    m_mapLocalsToRefresh.clear();

    mapPaintWidgets = CUpdateList::GetPaintMap();
    mIt             = mapPaintWidgets.begin();
    mendIt          = mapPaintWidgets.end();

    // Copy the widgets that need to be painted to
    // the map of all widgets that need to be updated.
    while (mIt != mendIt)
    {
        mwIt    = (*mIt).second.begin();
        mwendIt = (*mIt).second.end();

        while (mwIt != mwendIt)
        {
            (m_mapWidgetsToRefresh[(*mIt).first])[(*mwIt).first] = true;
            mwIt++;
        }

        mIt++;
    }

    m_syncRefresh.Unlock();

    mIt     = m_mapWidgetsToRefresh.begin();
    mendIt  = m_mapWidgetsToRefresh.end();

    while (mIt != mendIt)
    {
        // Update statistics about frames per second as needed.
        if (m_ulFPSCounter == (int)m_fRate)
        {
            CString strFPS;
            CString str;
            strFPS.Format("%d", m_mapFPS[(*mIt).first]);
            str.Format("FPS - %s", CUpdateList::UpdateWindowText((*mIt).first).c_str());

            if (::IsWindow((*mIt).first) == FALSE)
            {
                if (m_mapStatistics.find(_FSI_STL::string((LPCTSTR)str)) !=
                    m_mapStatistics.end())
                {
                    m_mapStatistics.erase(m_mapStatistics.find(_FSI_STL::string((LPCTSTR)str)));
                }

                if (m_mapFPS.find((*mIt).first) != m_mapFPS.end())
                {
                    m_mapFPS.erase(m_mapFPS.find((*mIt).first));
                }
            }
            else
            {
                m_mapStatistics[_FSI_STL::string((LPCTSTR)str)] = _FSI_STL::string((LPCTSTR)strFPS);
                m_mapFPS[(*mIt).first] = 0;
            }
        }

        // If an HWND doesn't have a counter for the update screen pass,
        // then give it one.  The initial value is equal to the update
        // rate from the host so that the child widgets of HWND will be
        // drawn the first time through here.
        if (m_mapScreenUpdatePass.find((*mIt).first) == m_mapScreenUpdatePass.end())
        {
            m_mapScreenUpdatePass[(*mIt).first] = m_ucScreenUpdateRatio;
        }

        // See if the HWND needs to be told to redraw some of its children.
        if (m_mapScreenUpdatePass[(*mIt).first] >= m_ucScreenUpdateRatio)
        {
            // Make sure that the parent wnd is still there
            // and it is not currently redrawing.
            if (::IsWindow((*mIt).first) == TRUE &&
                CWidget::ScreenRedrawing((*mIt).first) == false)
            {
                _FSI_STL::map<long, CWidget*> listWidgetsToSendToFrame;

                mwIt    = (*mIt).second.begin();
                mwendIt = (*mIt).second.end();

                // No rendering is currently occuring, thus we can try to update.
                while (mwIt != mwendIt)
                {
                    // Make sure that the widget is valid.
                    if (CWidget::IsValidAddress((*mwIt).first) == VALID)
                    {
                        // Make sure that the render variables could be
                        // updated.  If the widget is deleted while in
                        // the UpdateRenderVariables code, then it will
                        // return false.
                        if ((*mwIt).first->UpdateRenderVariables() == true)
                        {
                            // Make sure that the parent is still there.
                            // The list will be deleted in the destructor of
                            // the parent wnd.  However, the Windows window will
                            // be destroyed before the list.  Thus, the push back
                            // should always occur before the list is deleted.
                            if (::IsWindow((*mIt).first) == TRUE)
                            {
                                listWidgetsToSendToFrame[((*mwIt).first)->ID()] = (*mwIt).first;
                            }
                        }
                    }

                    mwIt++;
                }

                // Clear the map of widgets for the main frame currently being
                // processed.
                (*mIt).second.clear();

                // Make sure that the main frame is still there and tell it to 
                // redraw.
                if (::IsWindow((*mIt).first) == TRUE)
                {
                    if (!listWidgetsToSendToFrame.empty())
                    {
                        CWidget::ScreenRedrawing((*mIt).first, true);
                        CUpdateList::SetList((*mIt).first, listWidgetsToSendToFrame);
                        ::PostMessage((*mIt).first, WM_UPDATE_SCREEN, 0, 0);

                    }
                }

                m_mapScreenUpdatePass[(*mIt).first] = 1;
                m_mapFPS[(*mIt).first] = m_mapFPS[(*mIt).first] + 1;
            }
            else if (::IsWindow((*mIt).first) == FALSE)
            {
                CString str;
                str.Format("FPS - %s", CUpdateList::UpdateWindowText((*mIt).first).c_str());

                if (m_mapStatistics.find(_FSI_STL::string((LPCTSTR)str)) !=
                    m_mapStatistics.end())
                {
                    m_mapStatistics.erase(m_mapStatistics.find(_FSI_STL::string((LPCTSTR)str)));
                }

                if (m_mapFPS.find((*mIt).first) != m_mapFPS.end())
                {
                    m_mapFPS.erase(m_mapFPS.find((*mIt).first));
                }
            }
        }
        else
        {
            m_mapScreenUpdatePass[(*mIt).first] = m_mapScreenUpdatePass[(*mIt).first] + 1;
        }

        mIt++;
    }

    if (m_ulFPSCounter == (int)m_fRate)
    {
        m_ulFPSCounter = 0;
    }
}

void CCommsSystemInterface::Start()
{
    // Clean out the widgets that were needing to be updated.
    _FSI_STL::map<HWND, _FSI_STL::map<CWidget*, bool> >::iterator mIt = 
                                                 m_mapWidgetsToRefresh.begin();
    _FSI_STL::map<HWND, _FSI_STL::map<CWidget*, bool> >::iterator mendIt = 
                                                 m_mapWidgetsToRefresh.end();

    while (mIt != mendIt)
    {
        (*mIt).second.clear();
        mIt++;
    }

    m_mapWidgetsToRefresh.clear();
}

/////////////////////////////////////////////////////////////////////////////
//
// void CCommsSystemInterface::Stop()
//
// Inputs           : None.
//
// Return Values    : None.
//
// Date             : 09 February 1999
//
// Engineer         : Billy Baker
//
// Description      : Stop() can be called at any time but must be 
//                    called before the current instance of a comms 
//                    system interface derived class is deleted.  Its 
//                    purpose is to delete memory stored in static data 
//                    structures that would be inaccessible in the 
//                    destructor.  Derived classes should call this 
//                    base method as well as implement their own 
//                    cleanup--killing timers, disconnecting, etc.
//
/////////////////////////////////////////////////////////////////////////////
void CCommsSystemInterface::Stop()
{
    m_sync.Lock();
    // Write the FML description of the block of data.
    if (m_fileDebrief.m_hFile != CFile::hFileNull)
    {
        // Header
        CString strDescription =  "\n<?xml version=\"1.0\" encoding=\"ISO-8859-1\" "
                                      "standalone=\"no\"?>\n"
                                  "<!DOCTYPE COMMS_VARIABLE_DATA>\n"
                                  "<!-- Types\n"
                                  "     VAR_CHAR                    =  0,\n"
                                  "     VAR_UNSIGNED_CHAR           =  1,\n"
                                  "     VAR_SHORT_INT               =  2,\n"
                                  "     VAR_UNSIGNED_SHORT_INT      =  3,\n"
                                  "     VAR_LONG_INT                =  4,\n"
                                  "     VAR_UNSIGNED_LONG_INT       =  5,\n"
                                  "     VAR_FLOAT                   =  6,\n"
                                  "     VAR_DOUBLE                  =  7,\n"
                                  "     VAR_BOOL                    =  8,\n"
                                  "     VAR_STRING                  =  9\n"
                                  "-->\n"
                                  "<COMMS_VARIABLE_DATA>\n";

        // Write comms variables XML data.
        _FSI_STL::map<_FSI_STL::string, CVariableData*>::iterator mIt = 
                                                m_mapVarName2Debrief.begin();
        _FSI_STL::map<_FSI_STL::string, CVariableData*>::iterator mendIt = 
                                                m_mapVarName2Debrief.end();

        while (mIt != mendIt)
        {
            if (m_mapVarName2Data.find((*mIt).first) != 
                m_mapVarName2Data.end())
            {
                if (m_mapVarName2Data[(*mIt).first] != NULL)
                {
//                    if (m_mapVarName2Data[(*mIt).first]->HostError() == 0)
                    {
                        CString strEntry;
                        strEntry.Format("<COMMS_VARIABLE ID=\"%d\" SIM_VAR_NAME=\"%s\" "
                                        "TYPE=\"%d\" DEBRIEF=\"YES\" " //PERSIST=\"%s\" "
                                        "READ_MODE=\"%d\" SIZE=\"%d\" FORMAT=\"%d\" "
                                        "LENGTH=\"%d\" FAILURE_TYPE=\"%s\" FAILURE_DATA=\"%s\">%s</COMMS_VARIABLE>\n",
                                              (*mIt).second->ID(),
                                              (*mIt).second->SimVarName().c_str(),
                                              (*mIt).second->Type(),
//                                              (*mIt).second->Persist().c_str(),
                                              (*mIt).second->ReadMode(),
                                              (*mIt).second->ListSize(),
                                              (*mIt).second->InitialDataFormat(),
                                              (*mIt).second->ArrayLength(),
                                              CFailureSummarySystemInterface::ToFailureString((*mIt).second->FailureType()).c_str(),
                                              (*mIt).second->FailureData().c_str(),
                                              (*mIt).first.c_str());

                        strDescription += strEntry;

                        // Add range data in XML format.
                        _FSI_STL::map<_FSI_STL::string, _FSI_STL::list<CRange*> >::iterator mrIt;
                        _FSI_STL::list<CRange*>::iterator lrIt      = NULL;
                        _FSI_STL::list<CRange*>::iterator lrendIt   = NULL;

                        // Check to see if the variable has ranges.
                        mrIt = m_mapVarName2Ranges.find((*mIt).first);

                        if (mrIt != m_mapVarName2Ranges.end())
                        {
                            // It does, so add them to the string to dump.
                            lrIt    = (*mrIt).second.begin();
                            lrendIt = (*mrIt).second.end();
                            while (lrIt != lrendIt)
                            {
/*
                                CString strEntry;
                                strEntry.Format("    <RANGE VAR=\"%s\" ID=\"%d\" LOW=\"%s\" "
                                                    "HIGH=\"%s\" FORE_TRUE=\"%03d%03d%03d\" "
                                                    "BACK_TRUE=\"%03d%03d%03d\" TEXT_TRUE=\"%s\" "
                                                    "FORE_FALSE=\"%03d%03d%03d\" "
                                                    "BACK_FALSE=\"%03d%03d%03d\" TEXT_FALSE=\"%s\" "
                                                    "LOWER_COND=\"%d\" "
                                                    "UPPER_COND=\"%d\" "
                                                    "ACTION_TYPE=\"%d\" "
                                                    "ACTION_DATA=\"%s\">%s</RANGE>\n",
                                                      (*mrIt).first.c_str(),
                                                      (*lrIt)->ID(),
                                                      ((_FSI_STL::string)(*lrIt)->Lower()).c_str(),
                                                      ((_FSI_STL::string)(*lrIt)->Upper()).c_str(),
                                                      (*lrIt)->ForegroundTrue().Red(),
                                                      (*lrIt)->ForegroundTrue().Green(),
                                                      (*lrIt)->ForegroundTrue().Blue(),
                                                      (*lrIt)->BackgroundTrue().Red(),
                                                      (*lrIt)->BackgroundTrue().Green(),
                                                      (*lrIt)->BackgroundTrue().Blue(),
                                                      (*lrIt)->TextTrue().c_str(),
                                                      (*lrIt)->ForegroundFalse().Red(),
                                                      (*lrIt)->ForegroundFalse().Green(),
                                                      (*lrIt)->ForegroundFalse().Blue(),
                                                      (*lrIt)->BackgroundFalse().Red(),
                                                      (*lrIt)->BackgroundFalse().Green(),
                                                      (*lrIt)->BackgroundFalse().Blue(),
                                                      (*lrIt)->TextFalse().c_str(),
                                                      (*lrIt)->LowerConditional(),
                                                      (*lrIt)->UpperConditional(),
                                                      (*lrIt)->ActionType(),
                                                      (*lrIt)->ActionData(),
                                                      (*lrIt)->LogMessage().c_str());

                                strDescription += strEntry;

                                lrIt++;
*/
                                CString   strEntry;

                                strEntry.Format("    <RANGE VAR=\"%s\" ID=\"%d\" LOW=\"%s\" "
                                                    "HIGH=\"%s\" FORE_TRUE=\"%03d%03d%03d\" "
                                                    "BACK_TRUE=\"%03d%03d%03d\" TEXT_TRUE=\"%s\" "
                                                    "FORE_FALSE=\"%03d%03d%03d\" "
                                                    "BACK_FALSE=\"%03d%03d%03d\" TEXT_FALSE=\"%s\" "
                                                    "LOWER_COND=\"%d\" "
                                                    "UPPER_COND=\"%d\" "
                                                    "ACTION_TYPE=\"%d\" "
                                                    "ACTION_DATA=\"%s\"",
                                                      (*mrIt).first.c_str(),
                                                      (*lrIt)->ID(),
                                                      ((_FSI_STL::string)(*lrIt)->Lower()).c_str(),
                                                      ((_FSI_STL::string)(*lrIt)->Upper()).c_str(),
                                                      (*lrIt)->ForegroundTrue().Red(),
                                                      (*lrIt)->ForegroundTrue().Green(),
                                                      (*lrIt)->ForegroundTrue().Blue(),
                                                      (*lrIt)->BackgroundTrue().Red(),
                                                      (*lrIt)->BackgroundTrue().Green(),
                                                      (*lrIt)->BackgroundTrue().Blue(),
                                                      (*lrIt)->TextTrue().c_str(),
                                                      (*lrIt)->ForegroundFalse().Red(),
                                                      (*lrIt)->ForegroundFalse().Green(),
                                                      (*lrIt)->ForegroundFalse().Blue(),
                                                      (*lrIt)->BackgroundFalse().Red(),
                                                      (*lrIt)->BackgroundFalse().Green(),
                                                      (*lrIt)->BackgroundFalse().Blue(),
                                                      (*lrIt)->TextFalse().c_str(),
                                                      (*lrIt)->LowerConditional(),
                                                      (*lrIt)->UpperConditional(),
                                                      (*lrIt)->ActionType(),
                                                      (*lrIt)->ActionData());

                                CString   strMessage;
                                strMessage.Format(">%s</RANGE>\n", (*lrIt)->LogMessage().c_str());
                                strEntry        += strMessage;
                                strDescription  += strEntry;

                                lrIt++;
                            }
                        }
                    }
                }
            }

            mIt++;
        }

        strDescription += "</COMMS_VARIABLE_DATA>";


        unsigned long int ulLength = strDescription.GetLength();
        m_fileDebrief.Write(strDescription.GetBuffer(ulLength), ulLength);
        m_fileDebrief.Write(&m_ulDebriefBufferSize, sizeof(unsigned long int));
        m_fileDebrief.Write(&ulLength, sizeof(unsigned long int));
        m_fileDebrief.Write(&m_fDebriefRate, sizeof(float));

        m_fileDebrief.Close();
    }

    // Remove any comms actions that have been marked for deleting.
    if (!m_listCommsActionsToAdd.empty())
    {
        m_syncCommsAdd.Lock();

        _FSI_STL::list<CCommsAction*>::iterator lIt = 
                                            m_listCommsActionsToAdd.begin();
        _FSI_STL::list<CCommsAction*>::iterator lendIt = 
                                            m_listCommsActionsToAdd.end();
        while (lIt != lendIt)
        {
            AddComms((CIOAction*)(*lIt));
            lIt++;
        }

        m_listCommsActionsToAdd.clear();

        m_syncCommsAdd.Unlock();
    }

    // Remove any comms actions that have been marked for deleting.
    if (!m_listCommsActionsToDelete.empty())
    {
        m_syncCommsDelete.Lock();

        _FSI_STL::list<CCommsAction*>::iterator lIt = 
                                            m_listCommsActionsToDelete.begin();
        _FSI_STL::list<CCommsAction*>::iterator lendIt = 
                                            m_listCommsActionsToDelete.end();

        while (lIt != lendIt)
        {
            DeleteComms((CIOAction*)(*lIt));
            lIt++;
        }

        m_listCommsActionsToDelete.clear();

        m_syncCommsDelete.Unlock();
    }

    // Clear the locals to refresh.
    if (!m_mapLocalsToRefresh.empty())
    {
        _FSI_STL::map<HWND, _FSI_STL::map<CWidget*, bool> >::iterator mIt = NULL;
        _FSI_STL::map<HWND, _FSI_STL::map<CWidget*, bool> >::iterator mendIt = NULL;

        m_syncRefresh.Lock();

        mIt     = m_mapLocalsToRefresh.begin();
        mendIt  = m_mapLocalsToRefresh.end();

        while (mIt != mendIt)
        {
            (*mIt).second.clear();
            mIt++;
        }

        m_mapLocalsToRefresh.clear();

        m_syncRefresh.Unlock();
    }


    _FSI_STL::map<_FSI_STL::string, _FSI_STL::list<CCommsAction*> >::iterator caIt = 
                                                                    m_mapVarName2Comms.begin();
    _FSI_STL::map<_FSI_STL::string, _FSI_STL::list<CCommsAction*> >::iterator caendIt = 
                                                                    m_mapVarName2Comms.end();
    CCommsAction* pCommsAction = NULL;
    while (caIt != caendIt)
    {
        _FSI_STL::list<CCommsAction*>::iterator lIt = (*caIt).second.begin();
        while (lIt != (*caIt).second.end())
        {
            pCommsAction = (*lIt);
            DeleteComms((CIOAction*)(*lIt));
            pCommsAction->Deleting(true);

            lIt = (*caIt).second.begin();
        }

        (*caIt).second.clear();

        caIt++;
    }
    m_mapVarName2Comms.clear();

    _FSI_STL::map<_FSI_STL::string, CVariableData*>::iterator mIt = 
                                                    m_mapVarName2Data.begin();
    _FSI_STL::map<_FSI_STL::string, CVariableData*>::iterator mendIt = 
                                                    m_mapVarName2Data.end();
    while (mIt != mendIt)
    {
        delete (*mIt).second;
        mIt++;
    }
    m_mapVarName2Data.clear();

    // Free memory for the CVariant instances that
    // were created in a call to AddComms.
    _FSI_STL::map<_FSI_STL::string, CVariant*>::iterator mvIt = 
                                                m_mapVarName2Variant.begin();
    _FSI_STL::map<_FSI_STL::string, CVariant*>::iterator mvendIt = 
                                                m_mapVarName2Variant.end();
    while (mvIt != mvendIt)
    {
        delete (*mvIt).second;
        mvIt++;
    }
    m_mapVarName2Variant.clear();

    delete [] m_pcDebriefData;
    m_pcDebriefData = NULL;

    // Make sure that no variables are thought to already exist.
    // for debriefing.  Don't delete the pointers because they
    // were to CCommsActions that have already been deleted.
    m_mapVarName2Debrief.clear();

    _FSI_STL::map<_FSI_STL::string, _FSI_STL::list<CRange*> >::iterator mrIt = 
                                                    m_mapVarName2Ranges.begin();
    _FSI_STL::map<_FSI_STL::string, _FSI_STL::list<CRange*> >::iterator mrendIt = 
                                                    m_mapVarName2Ranges.end();
    _FSI_STL::list<CRange*>::iterator lrIt = NULL;
    _FSI_STL::list<CRange*>::iterator lrendIt = NULL;
    while (mrIt != mrendIt)
    {
        lrIt    = (*mrIt).second.begin();
        lrendIt = (*mrIt).second.end();
        while (lrIt != lrendIt)
        {
            delete (*lrIt);
            lrIt++;
        }

        (*mrIt).second.clear();

        mrIt++;
    }
    m_mapVarName2Ranges.clear();

    _FSI_STL::list<CXMLWidget*>::iterator lIt = m_listXMLWidgetsPersist.begin();
    _FSI_STL::list<CXMLWidget*>::iterator lendIt = m_listXMLWidgetsPersist.end();
    while (lIt != lendIt)
    {
        delete (*lIt);
        lIt++;
    }
    m_listXMLWidgetsPersist.clear();

    m_sync.Unlock();
}

void CCommsSystemInterface::ToDelete(CCommsAction* pCommsAction)
{
    m_syncCommsDelete.Lock();

    if (_FSI_STL::find(m_listCommsActionsToDelete.begin(), 
                       m_listCommsActionsToDelete.end(),
                       pCommsAction) == m_listCommsActionsToDelete.end())
    {
        m_listCommsActionsToDelete.push_back(pCommsAction);
    }

    m_syncCommsDelete.Unlock();
}

void CCommsSystemInterface::ToAdd(CCommsAction* pCommsAction)
{
    m_syncCommsAdd.Lock();

    if (_FSI_STL::find(m_listCommsActionsToAdd.begin(), 
                       m_listCommsActionsToAdd.end(),
                       pCommsAction) == m_listCommsActionsToAdd.end())
    {
        m_listCommsActionsToAdd.push_back(pCommsAction);
    }

    m_syncCommsAdd.Unlock();
}

void CCommsSystemInterface::ToRefresh(CWidget* pWidget)
{
    CWnd* pWnd = pWidget->FrameworkParent();

    if (pWnd != NULL)
    {
        m_syncRefresh.Lock();

        (m_mapLocalsToRefresh[pWnd->GetSafeHwnd()])[pWidget] = true;

        m_syncRefresh.Unlock();
    }
}

void CCommsSystemInterface::AddData(CVariableData* pVarData)
{
}

void CCommsSystemInterface::DeleteData()
{
}

void CCommsSystemInterface::AddVariable(const _FSI_STL::string& stlStrVariable, 
                                        CVariableData& varData,
                                        _FSI_STL::list<CRange>& listRanges)
{
}

void CCommsSystemInterface::InitComplete()
{
    m_bInitComplete = true;
}

bool CCommsSystemInterface::Connected()
{
    return m_bConnected;
}