-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                 JPATS T-6A Texan-II Flight Training Device
--
--
--  Engineer:  Keith H. Rehm
--
--  Revision:  (Number and date inserted by Clearcase)
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------

with Jpats_Hydraulics.Container;

with Jpats_Secondary_Flight_Controls;
with Jpats_Secondary_Flight_Controls_Types;
with Jpats_Landing_Gear;
with Jpats_Powerplant;
with Jpats_Electrical;
with Jpats_Fuel;
with Jpats_Electrical_Types;
with Jpats_Hydraulics_Types;
with Length_Types;
with Jpats_Aircraft_Body;
with Normalized_Types;
with Force_Types;
with Log;

package body Jpats_Hydraulics.Controller is

   procedure Initialize
   is
      package Ctnr renames Container;
   begin
      Ctnr.System.Initialize (Ctnr.This_Subsystem.The_Primary_System);
      Ctnr.System.Initialize (Ctnr.This_Subsystem.The_Emergency_System);
      Ctnr.Reservoir.Initialize (Ctnr.This_Subsystem.The_Reservoir);
      Ctnr.Accumulator.Initialize (Ctnr.This_Subsystem.The_Accumulator);
      Ctnr.Fluid_Low_Level_Switch.Initialize (Ctnr.This_Subsystem.The_Reservoir_Fluid_Low_Level_Switch);
      Ctnr.Pressure_Low_Level_Switch.Initialize (Ctnr.This_Subsystem.The_Low_Pressure_Switch);
      Ctnr.Pump.Initialize(Ctnr.This_Subsystem.The_Pump);
   exception
      when others =>
         Log.Report("Jpats_Hydraulics.Controller.Initialize()");
         raise;
   end Initialize;


   High_Px_Malf_Px : Float renames Container.This_Subsystem.HC_High_Px_Malf_Px;
   Low_Qty_Timer   : Float renames Container.This_Subsystem.HC_Low_Qty_Timer;
   V_Hyd_Accum     : Length_Types.Gallons renames Container.This_Subsystem.HC_V_Hyd_Accum;
   P_Hyd_Temp      : Float;

   procedure Update
     (Iconst : in Float)
   is

      package Ctnr      renames Container;
      package Sov       renames Ctnr.Shutoff_Valve;
      package Fuel      renames Jpats_Fuel;
      package Gear      renames Jpats_Landing_Gear;
      package Sfc       renames Jpats_Secondary_Flight_Controls;
      package Sfc_T     renames Jpats_Secondary_Flight_Controls_Types;
      package Ac_Body   renames Jpats_Aircraft_Body;
      package Hyd_Sys   renames Ctnr.System;
      package Accum     renames Ctnr.Accumulator;
      package Reservoir renames Ctnr.Reservoir;
      package Pump      renames Ctnr.Pump;
      package Ele       renames Jpats_Electrical;
      package Ele_T     renames Jpats_Electrical_Types;

      The_Primary_Sys : Hyd_Sys.Instance   renames Ctnr.This_Subsystem.The_Primary_System;
      The_Emerg_Sys   : Hyd_Sys.Instance   renames Ctnr.This_Subsystem.The_Emergency_System;
      The_Accum       : Accum.Instance     renames Ctnr.This_Subsystem.The_Accumulator;
      The_Reservoir   : Reservoir.Instance renames Ctnr.This_Subsystem.The_Reservoir;
      The_Pump        : Pump.Instance      renames Ctnr.This_Subsystem.The_Pump;

      The_Hyd_Sov : Sov.Instance renames Ctnr.This_Subsystem.The_Shutoff_Valve;

      Ios : Ctnr.Ios_Interface_Instance renames Ctnr.This_Ios_Interface;

      Host : Ctnr.Host_Interface_Instance renames Ctnr.This_Host_Interface;

      Hyd_Shaft_Rpm : constant Float := Float(Jpats_Powerplant.Hyd_Shaft_Rpm);
      Ng : constant Float := Hyd_Shaft_Rpm/(0.2041*374.86);

      P_Hyd_Reg           : Force_Types.Psi;
      P_Hyd_Reg_Corrected : Force_Types.Psi;
      V_Hyd_Max_Reg       : Length_Types.Gpm;
      P_Hyd               : Force_Types.Psi;

      P_Hyd_Emerg         : Force_Types.Psi;
      K_Hyd               : Jpats_Hydraulics_Types.Normalized_Pressure_Type;
      K_Hyd_Emerg         : Jpats_Hydraulics_Types.Normalized_Pressure_Type;
      Reservoir_Quantity  : Length_Types.Gallons;
      Quantity_Normalized : Float;

      V_Hyd : Length_Types.Gpm;

      Flap_Pos : constant float := Float(Sfc.Mean_Flap_Position);
      Gear_Pos : constant Float := Float(Gear.Mean_Gear_Position);

   begin

      ------------------------------------
      -- HYDRAULIC SYSTEM SHUTOFF VALVE --
      ------------------------------------

      Sov.Update (The_Hyd_Sov, Fuel.Emer_Firewall_Shutoff);

      -----------
      -- V HYD --
      -----------

      if Gear.Emergency_Gear_Handle_Extended then

         V_Hyd := (Gear.Gear_Hyd_Load       +
                   Gear.Gear_Doors_Hyd_Load +
                   Sfc.Flaps_Hydraulic_Load +
                   Ac_Body.Get_Nose_Steering_Hydraulic_Flow_Rate);

      else

         V_Hyd := (Gear.Gear_Hyd_Load            +
                   Gear.Gear_Doors_Hyd_Load      +
                   Sfc.Flaps_Hydraulic_Load      +
                   Sfc.Speedbrake_Hydraulic_Load +
                   Ac_Body.Get_Nose_Steering_Hydraulic_Flow_Rate);
      end if;

      --------------
      -- The Pump --
      --------------

      Pump.Update
        (An_Instance => The_Pump,
         Load        => V_Hyd,
         Engine_Rpm  => 0.0); -- not using

      ------------------------
      -- RESERVOIR QUANTITY --
      ------------------------

      if Ios.Low_Hyd_Quantity_Malf then
         Reservoir_Quantity := Reservoir.Quantity (The_Reservoir) - Iconst;
         if Reservoir_Quantity < 0.0 then Reservoir_Quantity := 0.0; end if;
      else
         Reservoir_Quantity := 0.6;
      end if;

      Quantity_Normalized := Reservoir_Quantity / 0.6;

      Reservoir.Set_Quantity (The_Reservoir, Reservoir_Quantity);

      -----------------------------
      -- PRIMARY SYSTEM PRESSURE --
      -----------------------------

      P_Hyd_Reg := 3015.0;

      if Ios.Hyd_Pump_Fail_Malf then
         if Low_Qty_Timer < 2.0 then
            Low_Qty_Timer := Low_Qty_Timer + Iconst;
         else
            Low_Qty_Timer := 2.0;
         end if;
         P_Hyd_Reg_Corrected := Float'Min(P_Hyd_Reg,125.0*Ng) - Float'Min(P_Hyd_Reg,125.0*Ng) * (Low_Qty_Timer/2.0);
         High_Px_Malf_Px := P_Hyd_Reg_Corrected;
      elsif Ios.Hyd_Pressure_High_Malf then
         High_Px_Malf_Px := High_Px_Malf_Px + 10.0 * Iconst;
         if High_Px_Malf_Px > 3500.0 then High_Px_Malf_Px := 3500.0; end if;
         P_Hyd_Reg_Corrected := Float'Min(3500.0, High_Px_Malf_Px) * Quantity_Normalized;
      elsif Ios.Low_Hyd_Quantity_Malf then
         Low_Qty_Timer := Low_Qty_Timer + Iconst;
         if Low_Qty_Timer >= 60.0 then Low_Qty_Timer := 60.0; end if;
         P_Hyd_Reg_Corrected := Float'Min(P_Hyd_Reg,125.0*Ng) - Float'Min(P_Hyd_Reg,125.0*Ng) * (Low_Qty_Timer/60.0);
         High_Px_Malf_Px := P_Hyd_Reg_Corrected;
      else
         P_Hyd_Reg_Corrected := Float'Min(P_Hyd_Reg,125.0*Ng) * Quantity_Normalized;
         High_Px_Malf_Px := P_Hyd_Reg_Corrected;
         Low_Qty_Timer := 0.0;
      end if;

      V_Hyd_Max_Reg := Float'Max(0.1, (0.0524*Ng-0.00018*P_Hyd_Reg_Corrected));


--      if V_hyd < V_Hyd_Max_Reg then

         P_Hyd_Temp := Float'Max(0.0,(Float(P_Hyd_Reg_Corrected)-86.0*Float(V_Hyd)/Float(V_Hyd_Max_Reg)));
         --            P_Hyd_Temp := Float'Max(0.0,(Float(P_Hyd_Reg_Corrected)-340.0*Float(V_Hyd)/Float(V_Hyd_Max_Reg)));
         P_Hyd := Force_Types.Psi(P_Hyd_Temp);

--      else

--         P_Hyd_Temp := Float'Max(0.0,(Float(P_Hyd_Reg_Corrected)-5522.0*(Float(V_Hyd)-Float(V_Hyd_Max_Reg))));
--         P_Hyd := Force_Types.Psi(P_Hyd_Temp);

--      end if;


      ------------------------------------------------------
      -- EMERGENCY SYSTEM PRESSURE AND ACCUMULATOR VOLUME --
      ------------------------------------------------------

      if not Gear.Emergency_Gear_Handle_Extended then


         if Ios.Low_Emerg_Hyd_Pressure_Malf then
            V_Hyd_Accum := Accum.Quantity(The_Accum) - Length_Types.Gallons(Iconst* 1000.0);
            P_Hyd_Emerg := Hyd_Sys.Pressure(The_Emerg_Sys) - Force_Types.Psi(Iconst* 1000.0);
            if V_Hyd_Accum < 0.0 then V_Hyd_Accum := 0.0; end if;
            if P_Hyd_Emerg < 0.0 then P_Hyd_Emerg := 0.0; end if;
         else
            V_Hyd_Accum := 0.21;
            if P_Hyd > Hyd_Sys.Pressure(The_Emerg_Sys) then
               P_Hyd_Emerg := P_Hyd;
            else
               P_Hyd_Emerg := Hyd_Sys.Pressure(The_Emerg_Sys);
            end if;
            if P_Hyd_Emerg > 3500.0 then P_Hyd_Emerg := 3500.0; end if;
         end if;

      else

         if Ios.Low_Emerg_Hyd_Pressure_Malf then
            V_Hyd_Accum := Accum.Quantity(The_Accum) - Length_Types.Gallons(Iconst * 1000.0);
            if V_Hyd_Accum < 0.0 then V_Hyd_Accum := 0.0; end if;
            P_Hyd_Emerg := Hyd_Sys.Pressure(The_Emerg_Sys) - Force_Types.Psi(Iconst * 1000.0);
            if P_Hyd_Emerg < 0.0 then P_Hyd_Emerg := 0.0; end if;
         else
            V_Hyd_Accum := Length_Types.Gallons (Float'Max(0.0,(Float(Accum.Quantity(The_Accum))
                                                                - Float(V_Hyd) * Iconst/60.0)));


            if V_Hyd_Accum > 0.0 then
--               P_Hyd_Emerg := Force_Types.Psi (3000.0 - 4452.0*(0.21 - Float(Accum.Quantity(The_Accum))));
--               P_Hyd_Emerg := Force_Types.Psi (3000.0 - 6952.0*(0.21 - Float(Accum.Quantity(The_Accum))));
--               P_Hyd_Emerg := Force_Types.Psi(3015.0 - 1015.0 * Flap_Pos/50.0 * Gear_Pos);
               P_Hyd_Emerg := Force_Types.Psi (3000.0 - 5500.0*(0.21 - Float(Accum.Quantity(The_Accum))));

            else
               P_Hyd_Emerg := 0.0;
            end if;
            if V_Hyd_Accum < 0.0 then V_Hyd_Accum := 0.0; end if;
            if P_Hyd_Emerg < 0.0 then P_Hyd_Emerg := 0.0; end if;
         end if;
      end if;

      if Ios.Emerg_Hyd_Px_Reset or Host.Emerg_Px_Reset_Command then
         P_Hyd_Emerg := 3015.0;
         V_Hyd_Accum := 0.21;
         Ios.Emerg_Hyd_Px_Reset := False;
         Host.Emerg_Px_Reset_Command := False;
      end if;

      Accum.Update (The_Accum,P_Hyd_Emerg,V_Hyd_Accum);


      --------------------------------
      -- NORMALIZED PRESSURE VALUES --
      --------------------------------

      if P_Hyd_Emerg >= 3015.0 then
         K_Hyd_Emerg := 1.0;
      else
         K_Hyd_Emerg := 0.85 * P_Hyd_Emerg / 3015.0;
      end if;

      if P_Hyd >= 1790.0 and Sov.Is_Open(The_Hyd_Sov) then

         if P_Hyd > 3015.0 then
            K_Hyd := 1.0;
         else
            K_Hyd := P_Hyd / 3015.0;
         end if;

      else

         K_Hyd := 0.0;

      end if;

      if Ios.Hyd_Pressure then --ios commanded full pressure

         Hyd_Sys.Update
           (An_Instance         => The_Primary_Sys,
            Quantity            => 1.055,
            Pressure            => 3015.0,
            Normalized_Quantity => 1.0,
            Normalized_Pressure => 1.0);

         Hyd_Sys.Update
           (An_Instance         => The_Emerg_Sys,
            Quantity            => 0.194,
            Pressure            => 3015.0,
            Normalized_Quantity => 1.0,
            Normalized_Pressure => 1.0);

      else -- normal

         Hyd_Sys.Update
           (An_Instance         => The_Primary_Sys,
            Quantity            => 1.055,
            Pressure            => P_Hyd,
            Normalized_Quantity => 1.0,
            Normalized_Pressure => K_Hyd);

         Hyd_Sys.Update
           (An_Instance         => The_Emerg_Sys,
            Quantity            => 0.194,
            Pressure            => P_Hyd_Emerg,
            Normalized_Quantity => 1.0,
            Normalized_Pressure => K_Hyd_Emerg);

      end if;


      -- ios
      Ios.Primary_Hyd_Px := Float (Hyd_Sys.Pressure(The_Primary_Sys));

      --------------------------------
      -- RESERVOIR LOW LEVEL SWITCH --
      --------------------------------

      if Ele.Is_Powered(Ele_T.Hyd_Sys_Cb) then

         Ctnr.Fluid_Low_Level_Switch.Update
           (An_Instance       => Ctnr.This_Subsystem.The_Reservoir_Fluid_Low_Level_Switch,
            Quantity_Level    => Reservoir.Quantity (The_Reservoir),
            Malfunction_Close => False);

      else

         Ctnr.Fluid_Low_Level_Switch.Update
           (An_Instance       => Ctnr.This_Subsystem.The_Reservoir_Fluid_Low_Level_Switch,
            Quantity_Level    => 0.0,
            Malfunction_Close => False);

      end if;

      ----------------------------------------------------
      -- EMERGENCY HYDRAULIC SYSTEM LOW PRESSURE SWITCH --
      ----------------------------------------------------

      Ctnr.Pressure_Low_Level_Switch.Update
        (An_Instance       => Ctnr.This_Subsystem.The_Low_Pressure_Switch,
         Quantity_Level    => P_Hyd_Emerg,
         Malfunction_Close => False);

   exception
      when others =>
         Log.Report("Jpats_Hydraulics.Controller.Update()");
         raise;
   end Update;


end Jpats_Hydraulics.Controller;



