-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                 JPATS T-6A Texan-II Flight Training Device
--
--
--  Engineer:  Keith H. Rehm
--
--  Revision:  (Number and date inserted by Clearcase)
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------
with Interpolation_Table.Singly_Indexed;
with Tank_Mass_Property;
with Simulation_Dictionary;
with Jpats_Fuel.Container;
with Jpats_Electrical;
with Jpats_Powerplant;
with Tank;
with Tank.Wing;
with Tank.Collector;
with Jpats_Electrical_Types;
with Length_Types;
with Jpats_Simulated_Aircraft;
with Jpats_Fuel_Types;
with Jpats_Engine_Data_Manager;
with JPATS_Auto_test;
with Log;

package body Jpats_Fuel.Controller is
   package It renames Interpolation_Table;
   package Jsa renames Jpats_Simulated_Aircraft;

   AFUELBL_T : aliased IT.Singly_Indexed.Instance;
   FUELFS_T  : aliased IT.Singly_Indexed.Instance;
   FUELKXX_T : aliased IT.Singly_Indexed.Instance;
   FUELKYY_T : aliased IT.Singly_Indexed.Instance;
   FUELKZZ_T : aliased IT.Singly_Indexed.Instance;
   FUELIXZ_T : aliased IT.Singly_Indexed.Instance;
   File_Path : constant String := Simulation_Dictionary.Lookup("Fuel_Dir");

   procedure Initialize is
   begin
      Tank.Wing.Initialize (Container.This_Subsystem.The_Left_Wing_Tank,530.0);
      Tank.Wing.Initialize (Container.This_Subsystem.The_Right_Wing_Tank,530.0);
      Tank.Collector.Initialize (Container.This_Subsystem.The_Collector_Tank,40.0);

      --      Initialize Fuel Mass Properties

      IT.Read(File_Path & "afuelbl.ito",AFUELBL_T);
      IT.Read(File_Path & "fuelfs.ito", FUELFS_T);
      IT.Read(File_Path & "fuelkxx.ito",FUELKXX_T);
      IT.Read(File_Path & "fuelkyy.ito",FUELKYY_T);
      IT.Read(File_Path & "fuelkzz.ito",FUELKZZ_T);
      IT.Read(File_Path & "fuelixz.ito",FUELIXZ_T);

      Tank_Mass_Property.Assign_Cg((14.16,-7.08,6.455),
                                   An_Instance => Container.This_Subsystem.The_Left_Mass_Property);

--      Tank_Mass_Property.Assign_moment((7504.8,-3752.4,3421.15),
--                                       An_Instance => Container.This_Subsystem.The_Left_Mass_Property);

      Tank_Mass_Property.Assign_MOI ((71.1377,10.1823,77.8321,0.6583),
                                     An_Instance => Container.This_Subsystem.The_Left_Mass_Property);

      Tank_Mass_Property.Assign_Cg((14.16,7.08,6.455),
                                   An_Instance => Container.This_Subsystem.The_Right_Mass_Property);

--      Tank_Mass_Property.Assign_moment((7504.8,3752.4,3421.15),
--                                       An_Instance => Container.This_Subsystem.The_Right_Mass_Property);

      Tank_Mass_Property.Assign_MOI ((71.1377,10.1823,77.8321,0.6583),
                                     An_Instance => Container.This_Subsystem.The_Right_Mass_Property);

      Tank_Mass_Property.Assign_Cg((13.74,-1.13,5.988),
                                   An_Instance => Container.This_Subsystem.The_Collector_Mass_Property);

--      Tank_Mass_Property.Assign_moment((-549.6,-45.2,239.52),
--                                       An_Instance => Container.This_Subsystem.The_Collector_Mass_Property);

      Tank_Mass_Property.Assign_MOI ((0.1519,0.4673,0.4585,0.0049),
                                     An_Instance => Container.This_Subsystem.The_Collector_Mass_Property);

      --end Initialize_Mass_Property;
      Container.Boost_Pump.Initialize (Container.This_Subsystem.The_Fuel_Boost_Pump);

   exception
      when others =>
         Log.Report("Jpats_Fuel.Controller.Initialize()");
         raise;
   end Initialize;

   Fuel_Avail_To_Engines       : Boolean         renames Container.This_Subsystem.FC_Fuel_Avail_To_Engines;
   Fuel_Px_To_Engines          : Force_Types.Psi renames Container.This_Subsystem.FC_Fuel_Px_To_Engines;
   Fuel_Flo_To_Engines         : Mass_Types.Pph  renames Container.This_Subsystem.FC_Fuel_Flo_To_Engines;
   Central_Fuel_Line_Leak_Flow : Mass_Types.Pph  renames Container.This_Subsystem.FC_Central_Fuel_Line_Leak_Flow;

   function Fuel_Available_To_Engines
     return Boolean
   is
   begin
      return Fuel_Avail_To_Engines;
   exception
      when others =>
         Log.Report("Jpats_Fuel.Controller.Fuel_Available_To_Engines()");
         raise;
   end Fuel_Available_To_Engines;

   function Fuel_Pressure_To_Engines
     return Force_Types.Psi
   is
   begin
      return Fuel_Px_To_Engines;
   exception
      when others =>
         Log.Report("Jpats_Fuel.Controller.Fuel_Pressure_To_Engines()");
         raise;
   end Fuel_Pressure_To_Engines;

   function Fuel_Flow_To_Engines
     return Mass_Types.Pph
   is
   begin
      return Fuel_Flo_To_Engines;
   exception
      when others =>
         Log.Report("Jpats_Fuel.Controller.Fuel_Flow_To_Engines()");
         raise;
   end Fuel_Flow_To_Engines;

   --  ********************************************************************

   In_Inverted_Flight_Timer : Float renames Container.This_Subsystem.FC_In_Inverted_Flight_Timer;
   Not_In_Inverted_Flight_Timer : Float renames Container.This_Subsystem.FC_Not_In_Inverted_Flight_Timer;
   Central_Fuel_Line_Leak_Malf_Lp : Boolean renames Container.This_Subsystem.FC_Central_Fuel_Line_Leak_Malf_Lp;
--   Pressure_Capacitor renames Container.This_Subsystem.FC_Pressure_Capacitor;
--   Pressure_Capacitor : array (1..3) of Force_Types.Psi renames Container.This_Subsystem.FC_Pressure_Capacitor;
   Pressure_Capacitor : Container.Pressure_Capacitor renames Container.This_Subsystem.FC_Pressure_Capacitor;

   Left_Tank_Fill_Command  : Boolean renames Container.This_Subsystem.FC_Left_Tank_Fill_Command;
   Right_Tank_Fill_Command : Boolean renames Container.This_Subsystem.FC_Right_Tank_Fill_Command;
   Collector_Tank_Fill_Command : Boolean renames Container.This_Subsystem.FC_Collector_Tank_Fill_Command;

   Sov_Timer : Float renames Container.This_Subsystem.FC_Sov_Timer;

   procedure Update (Iconst : in Float) is

      package Ctnr                renames Container;
      package Ele                 renames Jpats_Electrical;
      package Ele_T               renames Jpats_Electrical_Types;
      package Boost               renames Ctnr.Boost_Pump;
      package Low_Pressure_Switch renames Ctnr.Low_Pressure_Switch;
      package Sov                 renames Ctnr.Shutoff_Valve;
      package Xfer_Valve          renames Ctnr.Transfer_Valve;
      package Pwrplant            renames Jpats_Powerplant;
      package Fuel_Probe          renames Ctnr.Fuel_Probe;
      package Edm                 renames Jpats_Engine_Data_Manager;
      package Sim_Ac              renames JPATS_Simulated_Aircraft;
      package Relay               renames Ctnr.Relay;

      Io    : Ctnr.Io_Interface_Instance renames Ctnr.This_Io_Interface;
      Ios   : Ctnr.Ios_Interface_Instance renames Ctnr.This_Ios_Interface;
      Sound : Ctnr.Sound_Interface_Instance renames Ctnr.This_Sound_Interface;

      The_Left_Wing_Tank      : Tank.Wing.Instance           renames Ctnr.This_Subsystem.The_Left_Wing_Tank;
      The_Right_Wing_Tank     : Tank.Wing.Instance           renames Ctnr.This_Subsystem.The_Right_Wing_Tank;
      The_Collector_Tank      : Tank.Collector.Instance      renames Ctnr.This_Subsystem.The_Collector_Tank;
      The_Boost_Pump          : Boost.Instance               renames Ctnr.This_Subsystem.The_Fuel_Boost_Pump;
      The_Firewall_Sov        : Ctnr.Shutoff_Valve.Instance  renames Ctnr.This_Subsystem.The_Firewall_Fuel_Shutoff_Valve;
      The_Low_Pressure_Switch : Low_Pressure_Switch.Instance renames Ctnr.This_Subsystem.The_Low_Pressure_Switch;
      The_Left_Xfer_Valve     : Xfer_Valve.Instance          renames Ctnr.This_Subsystem.The_Left_Wing_Transfer_Valve;
      The_Right_Xfer_Valve    : Xfer_Valve.Instance          renames Ctnr.This_Subsystem.The_Right_Wing_Transfer_Valve;

      Left_Wing_Tank_Low_Level_Sensor  : Ctnr.Fuel_Low_Level_Sensor.Instance renames
        Ctnr.This_Subsystem.The_Left_Wing_Tank_Low_Level_Sensor;
      Right_Wing_Tank_Low_Level_Sensor : Ctnr.Fuel_Low_Level_Sensor.Instance renames
        Ctnr.This_Subsystem.The_Right_Wing_Tank_Low_Level_Sensor;

      Emergency_Sov_Handle_Pulled : constant Boolean := Io.Emer_Firewall_Shutoff;
      Fuel_Bal_Cb    : constant Boolean:= Ele.Is_Powered (Ele_T.Fuel_Bal_Cb);
      Boost_Pump_Cb  : constant Boolean:= Ele.Is_Powered (Ele_T.Boost_Pump_Cb);
      Fuel_Qty_Lo_Cb : constant Boolean:= Ele.Is_Powered (Ele_T.Fuel_Qty_Lo_Cb);
      Auto_Test      : constant Boolean:= JPATS_Auto_test.Test_Number = 921 or
                                          JPATS_Auto_test.Test_Number = 922;

      Fuel_Pump_Relay                           : Boolean := False;
      Pwr_To_Boost_Pump                         : Boolean := False;

      L_Xfer_Valve_Pwr                          : Boolean := False;
      R_Xfer_Valve_Pwr                          : Boolean := False;

      Motive_Available_From_Engines             : Boolean := False;
      Motive_Flow_To_Left_Wing_Tank             : Boolean := False;
      Motive_Flow_From_Left_Wing_Tank           : Boolean := False;
      Motive_Flow_To_Right_Wing_Tank            : Boolean := False;
      Motive_Flow_From_Right_Wing_Tank          : Boolean := False;
      Motive_Flow_To_Collector_Tank             : Boolean := False;
      Motive_Flow_From_Collector_Tank           : Boolean := False;

      Pump_Flow_From_Collector_Tank             : Boolean := False;

      Gravity_Flow_From_Left_Wing_Tank          : Boolean := False;
      Gravity_Flow_From_Right_Wing_Tank         : Boolean := False;
      Gravity_Flow_To_Collector_Tank            : Boolean := False;

      Engine_Fuel_Flow          : constant Mass_Types.Pph := Pwrplant.Fuel_Flow;


      --       Left_Tank_Quantity : Mass_Types.Lbm;
      --       Right_Tank_Quantity : Mass_Types.Lbm;
      --       Collector_Tank_Quantity : Mass_Types.Lbm;

      Left_Fuel_Tank_Target                     : Mass_Types.Lbm := 0.0;
      Right_Fuel_Tank_Target                    : Mass_Types.Lbm := 0.0;
      Left_Fuel_Tank_Delta                      : Mass_Types.Lbm := 0.0;
      Right_Fuel_Tank_Delta                     : Mass_Types.Lbm := 0.0;

      Left_Tank_Total_Flow                      : Mass_Types.Lbm := 0.0;
      Right_Tank_Total_Flow                     : Mass_Types.Lbm := 0.0;
      Collector_Tank_Total_Flow                 : Mass_Types.Lbm := 0.0;

      Fuel_Flow_Leaving_Left_Wing_Tank          : Mass_Types.Lbm := 0.0;
      Fuel_Flow_Leaving_Right_Wing_Tank         : Mass_Types.Lbm := 0.0;
      Fuel_Flow_Leaving_Collector               : Mass_Types.Lbm := 0.0;
      Fuel_Flow_Entering_Collector              : Mass_Types.Lbm := 0.0;

      L_Inside_Probe_Reading                    : Mass_Types.Lbm := 0.0;
      L_Middle_Probe_Reading                    : Mass_Types.Lbm := 0.0;
      L_Outside_Probe_Reading                   : Mass_Types.Lbm := 0.0;
      R_Inside_Probe_Reading                    : Mass_Types.Lbm := 0.0;
      R_Middle_Probe_Reading                    : Mass_Types.Lbm := 0.0;
      R_Outside_Probe_Reading                   : Mass_Types.Lbm := 0.0;

      -- FLIGHT
      Sign                     : Float;
      Left_Gal                 : Length_types.Gallons;
      Right_Gal                : Length_types.Gallons;
      Coll_Gal                 : Length_types.Gallons;
      Left_lbs                 : Mass_Types.Lbm;
      Right_lbs                : Mass_Types.Lbm;
      coll_lbs                 : Mass_Types.Lbm;
      Tot_Lbs                  : Mass_Types.Lbm;
      coll_Cg                  : Coordinate_Types.Reference_System;
      left_Cg                  : Coordinate_Types.Reference_System;
      right_Cg                 : Coordinate_Types.Reference_System;
      Ac_Cg                    : Coordinate_Types.Reference_System;

      Pcl_Below_Idle_Sw : constant Boolean := Pwrplant.Boost_Pump_Below_Idle_Switch;
      In_Inverted_Flight : constant Boolean := Sim_Ac.Load_Factor.Z > 0.0;
      Flame_Out_Time : constant Float := 120.0;

      C_To_L_Excess_Flow : Mass_Types.Lbm;
      C_To_R_Excess_Flow : Mass_Types.Lbm;

      Fuel_Pressure_Form_Engines : constant Force_Types.Psi := Jpats_Powerplant.Intermediate_Fuel_Pump_Pressure;
      Fuel_Pressure_At_Low_Pressure_Switch : Force_Types.Psi := 0.0;
      Low_Px_Switch : Force_Types.Psi := 0.0;

   begin

      ---------------------------------
      -- FIREWALL FUEL SHUTOFF VALVE --
      ---------------------------------

      Sov.Update (The_Firewall_Sov, Emergency_Sov_Handle_Pulled);

      if not Sov.Is_Open(The_Firewall_Sov) then
         Sov_Timer := Sov_Timer + Iconst;
         if Sov_Timer > 10.0 then
            Sov_Timer := 10.0;
         end if;
      else
         Sov_Timer := 0.0;
      end if;


      -- repeater
      Ios.Emergency_Shutoff_Valve := Emergency_Sov_Handle_Pulled;

      -------------------------
      -- LOW PRESSURE SWITCH --
      -------------------------

--       if Jpats_Powerplant.Gas_Generator_Speed_Sensor < 100 then
--          Fuel_Pressure_At_Low_Pressure_Switch := 0.0;
--       elsif
--      null;
--       end if;

      Pressure_Capacitor(3) := Pressure_Capacitor(2);
      Pressure_Capacitor(2) := Pressure_Capacitor(1);
      Pressure_Capacitor(1) := Fuel_Pressure_Form_Engines;

      if Ios.Boost_Pump_Auto_Fail_Malf or Ios.Cycling_Malf then
         Fuel_Pressure_At_Low_Pressure_Switch := Pressure_Capacitor(1);
         Fuel_Pressure_At_Low_Pressure_Switch := Force_Types.Psi'Min(Fuel_Pressure_At_Low_Pressure_Switch,Pressure_Capacitor(2));
         Fuel_Pressure_At_Low_Pressure_Switch := Force_Types.Psi'Min(Fuel_Pressure_At_Low_Pressure_Switch,Pressure_Capacitor(3));
      else
         Fuel_Pressure_At_Low_Pressure_Switch := Pressure_Capacitor(1);
         Fuel_Pressure_At_Low_Pressure_Switch := Force_Types.Psi'Max(Fuel_Pressure_At_Low_Pressure_Switch,Pressure_Capacitor(2));
         Fuel_Pressure_At_Low_Pressure_Switch := Force_Types.Psi'Max(Fuel_Pressure_At_Low_Pressure_Switch,Pressure_Capacitor(3));
      end if;
      Low_Pressure_Switch.Update
        (An_Instance       => The_Low_Pressure_Switch,
         Quantity_Level    => Fuel_Pressure_At_Low_Pressure_Switch,
         Malfunction_Close => False);

      ----------------
      -- BOOST PUMP --
      ----------------

--       Fuel_Pump_Relay := ((Boost_Pump_Cb and
--                            ((Low_Pressure_Switch.Is_Closed (The_Low_Pressure_Switch) and
--                              not Io.Boost_Pump_Below_Idle_Sw) or
--                             (Ele.Is_Closed (Ele_T.Starter_On_Relay)))) and not Ios.Boost_Pump_Auto_Fail_Malf);

--       Fuel_Pump_Relay := ((Boost_Pump_Cb and
--                            ((Low_Pressure_Switch.Is_Closed (The_Low_Pressure_Switch) and
--                              not Pcl_Below_Idle_Sw) or
--                             (Ele.Is_Closed (Ele_T.Starter_On_Relay)))) and not Ios.Boost_Pump_Auto_Fail_Malf);


      Low_Px_Switch := Pressure_Capacitor(1);
      Low_Px_Switch := Force_Types.Psi'Min(Low_Px_Switch,Pressure_Capacitor(2));
      Low_Px_Switch := Force_Types.Psi'Min(Low_Px_Switch,Pressure_Capacitor(3));


--       Fuel_Pump_Relay := ((Boost_Pump_Cb
--                            and ((Low_Pressure_Switch.Is_Closed (The_Low_Pressure_Switch) and not Pcl_Below_Idle_Sw)
--                                 or Ele.Is_Closed (Ele_T.Starter_On_Relay)))
--                           and not Ios.Boost_Pump_Auto_Fail_Malf);
      Fuel_Pump_Relay := ((Boost_Pump_Cb
                           and ((Low_Px_Switch < 10.0 and not Pcl_Below_Idle_Sw)
                                or Ele.Is_Closed (Ele_T.Starter_On_Relay)))
                          and not Ios.Boost_Pump_Auto_Fail_Malf);

      Relay.Update
        (An_Instance => Ctnr.This_Subsystem.The_Fuel_Pump_Relay,
         Power_And_Ground => Fuel_Pump_Relay);

      Pwr_To_Boost_Pump := (Boost_Pump_Cb and
                            (Io.Boost_Pump_On_Sw or Fuel_Pump_Relay));


      Boost.Update (The_Boost_Pump, Pwr_To_Boost_Pump);

      ---------------------
      -- TRANSFER VALVES --
      ---------------------

      L_Xfer_Valve_Pwr := ((Io.Man_Fuel_Xfer_L_Sw and
                            Io.Fuel_Xfer_Man_Reset_Sw and
                            Fuel_Bal_Cb) or
                           (Edm.Left_Transfer_Valve_Out and not Ios.Stuck_Fuel_Malf));

      R_Xfer_Valve_Pwr := ((Io.Man_Fuel_Xfer_R_Sw and
                            Io.Fuel_Xfer_Man_Reset_Sw and
                            Fuel_Bal_Cb) or
                           (Edm.Right_Transfer_Valve_Out)) or
                           (Ios.Stuck_Fuel_Malf and not (Io.Man_Fuel_Xfer_L_Sw and
                            Io.Fuel_Xfer_Man_Reset_Sw and Fuel_Bal_Cb));

      Xfer_Valve.Update
        (An_Instance => The_Left_Xfer_Valve,
         Power_In    => L_Xfer_Valve_Pwr);

      Xfer_Valve.Update
        (An_Instance => The_Right_Xfer_Valve,
         Power_In    => R_Xfer_Valve_Pwr);

      -- repeater
      if Io.Man_Fuel_Xfer_L_Sw then
         Ios.Manual_Transfer_Sw := 1;
      elsif Io.Man_Fuel_Xfer_R_Sw then
         Ios.Manual_Transfer_Sw := 2;
      else
         Ios.Manual_Transfer_Sw := 0;
      end if;

      ---------------------
      -- INVERTED FLIGHT --
      ---------------------
      if not Sim_Ac.Get_Flight_Freeze then
         if In_Inverted_Flight then
            In_Inverted_Flight_Timer := Float'Min(120.0, In_Inverted_Flight_Timer + Iconst);
            Not_In_Inverted_Flight_Timer := 0.0;
         else
            Not_In_Inverted_Flight_Timer := Float'Min(5.0, Not_In_Inverted_Flight_Timer + Iconst);
            if Not_In_Inverted_Flight_Timer > 2.0 then
               In_Inverted_Flight_Timer := 0.0;
            end if;
         end if;
      end if;

      -----------------
      -- MOTIVE FLOW --
      -----------------

      Motive_Available_From_Engines := Jpats_Powerplant.Intermediate_Fuel_Pump_Pressure > 5.0;

      Motive_Flow_To_Left_Wing_Tank := (Motive_Available_From_Engines
                                        and Xfer_Valve.Is_Open (The_Left_Xfer_Valve));

      Motive_Flow_From_Left_Wing_Tank := (Motive_Flow_To_Left_Wing_Tank
                                          and Tank.Is_Not_Empty (The_Left_Wing_Tank)
                                          and In_Inverted_Flight_Timer < Flame_Out_Time);

      Motive_Flow_To_Right_Wing_Tank := (Motive_Available_From_Engines
                                         and Xfer_Valve.Is_Open (The_Right_Xfer_Valve));

      Motive_Flow_From_Right_Wing_Tank := (Motive_Flow_To_Right_Wing_Tank
                                           and Tank.Is_Not_Empty (The_Right_Wing_Tank)
                                           and In_Inverted_Flight_Timer < Flame_Out_Time);

      Motive_Flow_To_Collector_Tank := Motive_Available_From_Engines;

      Motive_Flow_From_Collector_Tank := (Motive_Flow_To_Collector_Tank
                                          and Tank.Is_Not_Empty (The_Collector_Tank)
                                          and In_Inverted_Flight_Timer < Flame_Out_Time);

      Tank.Collector.Set_Motive_Flow_Out (The_Collector_Tank,Motive_Flow_From_Collector_Tank);

      ------------------
      -- GRAVITY FLOW --
      ------------------

      Gravity_Flow_From_Left_Wing_Tank  := Tank.Is_Not_Empty (The_Left_Wing_Tank);

      Gravity_Flow_From_Right_Wing_Tank := Tank.Is_Not_Empty (The_Right_Wing_Tank);

      Gravity_Flow_To_Collector_Tank    :=
        Gravity_Flow_From_Left_Wing_Tank or Gravity_Flow_From_Right_Wing_Tank;

      ---------------
      -- PUMP FLOW --
      ---------------

      Pump_Flow_From_Collector_Tank := (Boost.Is_On (The_Boost_Pump)
                                        and Tank.Is_Not_Empty (The_Collector_Tank)
                                        and In_Inverted_Flight_Timer < Flame_Out_Time);


      ----------------
      -- FUEL FLOWS --
      ----------------

      if Pwrplant.Fuel_Flow < 450.0 and Motive_Flow_From_Collector_Tank then
         if Xfer_Valve.Is_Closed (The_Right_Xfer_Valve) then
            C_To_R_Excess_Flow := 8.75 * Iconst * (1.0/60.0);
         else
            C_To_R_Excess_Flow := 0.0;
         end if;

         if Xfer_Valve.Is_Closed (The_Left_Xfer_Valve) then
            C_To_L_Excess_Flow := 8.75 * Iconst * (1.0/60.0);
         else
            C_To_L_Excess_Flow := 0.0;
         end if;
      else
         C_To_R_Excess_Flow := 0.0;
         C_To_L_Excess_Flow := 0.0;
      end if;

      if Motive_Flow_From_Left_Wing_Tank and Motive_Flow_From_Right_Wing_Tank then
         Fuel_Flow_Leaving_Left_Wing_tank  := -((Engine_Fuel_Flow*Iconst/3600.0/2.0) +
                                                (Central_Fuel_Line_Leak_Flow*Iconst/3600.0/2.0) +
                                                 C_To_R_Excess_Flow);
         Fuel_Flow_Leaving_Right_Wing_Tank := -((Engine_Fuel_Flow*Iconst/3600.0/2.0) +
                                                (Central_Fuel_Line_Leak_Flow*Iconst/3600.0/2.0) +
                                                 C_To_L_Excess_Flow);
         Fuel_Flow_Leaving_Left_Wing_tank  := 1.05*Fuel_Flow_Leaving_Left_Wing_tank; --burn faster from left tank
         Fuel_Flow_Leaving_Right_Wing_Tank := 0.95*Fuel_Flow_Leaving_Right_Wing_Tank;

      elsif Motive_Flow_From_Left_Wing_Tank then
         Fuel_Flow_Leaving_Left_Wing_tank := -((Engine_Fuel_Flow*Iconst/3600.0) +
                                               (Central_Fuel_Line_Leak_Flow*Iconst/3600.0) +
                                                C_To_R_Excess_Flow);
      elsif Motive_Flow_From_Right_Wing_Tank then
         Fuel_Flow_Leaving_Right_Wing_tank := -((Engine_Fuel_Flow*Iconst/3600.0) +
                                                (Central_Fuel_Line_Leak_Flow*Iconst/3600.0) +
                                                 C_To_L_Excess_Flow);
      elsif Gravity_Flow_From_Left_Wing_Tank and Gravity_Flow_From_Right_Wing_Tank then
         Fuel_Flow_Leaving_Left_Wing_tank  := -((Engine_Fuel_Flow*Iconst/3600.0/2.0) +
                                                (Central_Fuel_Line_Leak_Flow*Iconst/3600.0/2.0));
         Fuel_Flow_Leaving_Right_Wing_Tank := -((Engine_Fuel_Flow*Iconst/3600.0/2.0) +
                                                (Central_Fuel_Line_Leak_Flow*Iconst/3600.0/2.0));
         Fuel_Flow_Leaving_Left_Wing_tank  := 1.05*Fuel_Flow_Leaving_Left_Wing_tank; --burn faster from left tank
         Fuel_Flow_Leaving_Right_Wing_Tank := 0.95*Fuel_Flow_Leaving_Right_Wing_Tank;


      elsif Gravity_Flow_From_Left_Wing_Tank then
         Fuel_Flow_Leaving_Left_Wing_tank := -((Engine_Fuel_Flow*Iconst/3600.0) +
                                               (Central_Fuel_Line_Leak_Flow*Iconst/3600.0));

      elsif Gravity_Flow_From_Right_Wing_Tank then
         Fuel_Flow_Leaving_Right_Wing_tank := -((Engine_Fuel_Flow*Iconst/3600.0) +
                                                (Central_Fuel_Line_Leak_Flow*Iconst/3600.0));

      end if;

      if Xfer_Valve.Is_Closed (The_Left_Xfer_Valve) and Tank.Is_Not_Empty (The_Right_Wing_Tank) then
         Fuel_Flow_Leaving_Left_Wing_Tank := 0.0;
      end if;

      if Xfer_Valve.Is_Closed (The_Right_Xfer_Valve) and Tank.Is_Not_Empty (The_Left_Wing_Tank) then
         Fuel_Flow_Leaving_Right_Wing_Tank := 0.0;
      end if;

      Left_Tank_Total_Flow := Fuel_Flow_Leaving_Left_Wing_Tank + C_To_L_Excess_Flow;
      Right_Tank_Total_Flow := Fuel_Flow_Leaving_Right_Wing_Tank + C_To_R_Excess_Flow;

      if In_Inverted_Flight_Timer < Flame_Out_Time then
         Fuel_Flow_Leaving_Collector := -((Engine_Fuel_Flow*Iconst/3600.0)+
                                          (Central_Fuel_Line_Leak_Flow*Iconst/3600.0));
      end if;

      Fuel_Flow_Entering_Collector := (-1.0*Fuel_Flow_Leaving_Left_Wing_Tank) + (-1.0*Fuel_Flow_Leaving_Right_Wing_Tank);

      Collector_Tank_Total_Flow := Fuel_Flow_Leaving_Collector + Fuel_Flow_Entering_Collector;


      ----------------
      -- FUEL TANKS --
      ----------------

      if Auto_test then

         if JPATS_Auto_test.Fuel < 100.0 then
            Tank.Update (The_Right_Wing_Tank, 0.0);
            Tank.Update (The_left_Wing_Tank, 0.0);
            Tank.Update (The_Collector_Tank, 0.0);
         else -- left and right equal less collector
            Tank.Update (The_Collector_Tank, 40.0);
            Left_Fuel_Tank_Target := (JPATS_Auto_test.Fuel -40.0)/ 2.0;
            Tank.Update (The_Right_Wing_Tank, left_fuel_tank_target);
            Tank.Update (The_left_Wing_Tank, left_fuel_tank_target);
         end if;

      else

         if Ios.Set_Left_Tank_Quantity and not Ios.Fill_Tanks then
            if Ios.Left_Tank_Requested_Quantity > 530.0 then
               Left_Fuel_Tank_Target := 530.0;
            elsif Ios.Left_Tank_Requested_Quantity < 0.0 then
               Left_Fuel_Tank_Target := 0.0;
            else
               Left_Fuel_Tank_Target := Ios.Left_Tank_Requested_Quantity;
            end if;
         end if;

         if Ios.Set_Right_Tank_Quantity and not Ios.Fill_Tanks then
            if Ios.Right_Tank_Requested_Quantity > 530.0 then
               Right_Fuel_Tank_Target := 530.0;
            elsif Ios.Right_Tank_Requested_Quantity < 0.0 then
               Right_Fuel_Tank_Target := 0.0;
            else
               Right_Fuel_Tank_Target := Ios.Right_Tank_Requested_Quantity;
            end if;
         end if;

         if Ios.Fill_Tanks then
            Left_Tank_Fill_Command  := True;
            Right_Tank_Fill_Command := True;
            Collector_Tank_Fill_Command := True;
            Ios.Fill_Tanks := False;
         end if;

         --------------------
         -- COLLECTOR TANK --
         --------------------

         if not Collector_Tank_Fill_Command and not Ios.Set_Left_Tank_Quantity and
                                                not Ios.Set_Right_Tank_Quantity then

            if not Sim_Ac.Get_Flight_Freeze and Tank.Quantity (The_Collector_Tank) > 0.0 then

               Tank.Update (The_Collector_Tank,
                            Tank.Quantity (The_Collector_Tank) + Collector_Tank_Total_Flow);

            end if;

            if Tank.Quantity (The_Collector_Tank) < 0.0 then

               Tank.Update (The_Collector_Tank, 0.0);

            elsif Tank.Quantity (The_Collector_Tank) > 40.0 then

               Tank.Update (The_Collector_Tank, 40.0);

            end if;

         else

            Tank.Update (The_Collector_Tank, 40.0);
            Collector_Tank_Fill_Command := False;

         end if;

         ---------------
         -- LEFT TANK --
         ---------------

         if not Left_Tank_Fill_Command and not Ios.Set_Left_Tank_Quantity then

            if not Sim_Ac.Get_Flight_Freeze and Tank.Quantity (The_Left_Wing_Tank) > 0.0 then

               Tank.Update (The_Left_Wing_Tank,
                            Tank.Quantity (The_Left_Wing_Tank) + Left_Tank_Total_Flow);

            end if;

         elsif Left_Tank_Fill_Command then

            Left_Fuel_Tank_Delta := 530.0 - Tank.Quantity (The_Left_Wing_Tank);

            if Left_Fuel_Tank_Delta > 200.0 then

               Tank.Update (The_Left_Wing_Tank, Tank.Quantity (The_Left_Wing_Tank) + 200.0);

            else
               Tank.Update (The_Left_Wing_Tank, 530.0);
               Left_Tank_Fill_Command := False;
            end if;

         elsif Ios.Set_Left_Tank_Quantity then

            Left_Fuel_Tank_Delta := Left_Fuel_Tank_Target - Tank.Quantity (The_Left_Wing_Tank);

            if Left_Fuel_Tank_Delta > 200.0 then

               Tank.Update (The_Left_Wing_Tank, Tank.Quantity (The_Left_Wing_Tank) + 200.0);

            elsif Left_Fuel_Tank_Delta < -200.0 then

               Tank.Update (The_Left_Wing_Tank, Tank.Quantity (The_Left_Wing_Tank) - 200.0);

            else
               Tank.Update (The_Left_Wing_Tank, Left_Fuel_Tank_Target);
               Ios.Set_Left_Tank_Quantity := False;
            end if;

         end if;

         if Tank.Quantity (The_Left_Wing_Tank) < 0.0 then

            Tank.Update (The_Left_Wing_Tank, 0.0);

         elsif Tank.Quantity (The_Left_Wing_Tank) > 530.0 then

            Tank.Update (The_Left_Wing_Tank, 530.0);

         end if;

         ----------------
         -- RIGHT TANK --
         ----------------

         if not Right_Tank_Fill_Command and not Ios.Set_Right_Tank_Quantity then

            if not Sim_Ac.Get_Flight_Freeze and Tank.Quantity (The_Right_Wing_Tank) > 0.0 then

               Tank.Update (The_Right_Wing_Tank,
                            Tank.Quantity (The_Right_Wing_Tank) + Right_Tank_Total_Flow);

            end if;

         elsif Right_Tank_Fill_Command then

            Right_Fuel_Tank_Delta := 530.0 - Tank.Quantity (The_Right_Wing_Tank);

            if Right_Fuel_Tank_Delta > 200.0 then

               Tank.Update (The_Right_Wing_Tank, Tank.Quantity (The_Right_Wing_Tank) + 200.0);

            else
               Tank.Update (The_Right_Wing_Tank, 530.0);
               Right_Tank_Fill_Command := False;
            end if;

         elsif Ios.Set_Right_Tank_Quantity then

            Right_Fuel_Tank_Delta := Right_Fuel_Tank_Target - Tank.Quantity (The_Right_Wing_Tank);

            if Right_Fuel_Tank_Delta > 200.0 then

               Tank.Update (The_Right_Wing_Tank, Tank.Quantity (The_Right_Wing_Tank) + 200.0);

            elsif Right_Fuel_Tank_Delta < -200.0 then

               Tank.Update (The_Right_Wing_Tank, Tank.Quantity (The_Right_Wing_Tank) - 200.0);

            else
               Tank.Update (The_Right_Wing_Tank, Right_Fuel_Tank_Target);
               Ios.Set_Right_Tank_Quantity := False;
            end if;

         end if;

         if Tank.Quantity (The_Right_Wing_Tank) < 0.0 then

            Tank.Update (The_Right_Wing_Tank, 0.0);

         elsif Tank.Quantity (The_Right_Wing_Tank) > 530.0 then

            Tank.Update (The_Right_Wing_Tank, 530.0);

         end if;

      end if;

   -- Update Fuel Mass Property;

      Left_Gal      := Tank.Volume(The_Left_Wing_Tank) + 2.985;
      Right_Gal     := Tank.Volume(The_Right_Wing_Tank) + 2.985;
      Coll_Gal      := Tank.Volume(The_Collector_Tank);
      Left_lbs      := Tank.Quantity(An_Instance => Container.This_Subsystem.The_Left_Wing_Tank) + 20.0;
      Right_lbs     := Tank.Quantity(An_Instance => Container.This_Subsystem.The_Right_Wing_Tank) + 20.0;
      coll_lbs := Tank.Quantity(An_Instance => Container.This_Subsystem.The_Collector_Tank);

      Sign := -1.0;   --| -ve for Left Wing Tank
      Tank_Mass_Property.Set_Wing_Cg
        (Sign             => Sign,
         Gal           => Left_Gal,
         AFUELBL            => AFUELBL_T,
         FUELFS             => FUELFS_T,
         An_Instance        => Container.This_Subsystem.The_Left_Mass_Property);

      Sign := +1.0;  --| +ve for Right Wing Tank
      Tank_Mass_Property.Set_Wing_Cg
        (Sign           => Sign,
         Gal            => Right_Gal,
         AFUELBL        => AFUELBL_T,
         FUELFS         => FUELFS_T,
         An_Instance    => Container.This_Subsystem.The_Right_Mass_Property);

      coll_Cg := (13.744, -1.13, (68.16 + 0.62 * Coll_Gal)/12.0);

      Tank_Mass_Property.Assign_Cg (coll_Cg,
                                    An_Instance => Container.This_Subsystem.The_Collector_Mass_Property);

      Tank_Mass_Property.Set_Wing_Inertia
        (Gal                => Left_Gal,
         Fuel_Wt            => Left_lbs,
         FUELKXX            => FUELKXX_T,
         FUELKYY            => FUELKYY_T,
         FUELKZZ            => FUELKZZ_T,
         FUELIXZ            => FUELIXZ_T,
         An_Instance        => Container.This_Subsystem.The_Left_Mass_Property);

      Tank_Mass_Property.Set_Wing_Inertia
        (Gal                => Right_Gal,
         Fuel_Wt            => Right_lbs,
         FUELKXX            => FUELKXX_T,
         FUELKYY            => FUELKYY_T,
         FUELKZZ            => FUELKZZ_T,
         FUELIXZ            => FUELIXZ_T,
         An_Instance        => Container.This_Subsystem.The_Right_Mass_Property);

      Tank_Mass_Property.Set_Collector_Inertia
        (Fuel_Wt => coll_lbs,
         An_Instance   => Container.This_Subsystem.The_Collector_Mass_Property);

      Ac_CG := (JSA.Get_x_CG, JSA.Get_y_CG, JSA.Get_Z_CG);

      Tank_Mass_Property.Set_MOI
        (Ac_CG,
         Left_lbs,
         An_Instance => Container.This_Subsystem.The_Left_Mass_Property);

      Tank_Mass_Property.Set_MOI
        (Ac_CG,
         Right_lbs,
         An_Instance => Container.This_Subsystem.The_Right_Mass_Property);

      Tank_Mass_Property.Set_MOI
        (Ac_cg,
         coll_lbs,
         An_Instance => Container.This_Subsystem.The_Collector_Mass_Property);

      Left_Cg := Tank_Mass_Property.Get_Cg
        (An_Instance => Container.This_subsystem.The_Left_mass_Property);
      right_Cg := Tank_Mass_Property.Get_Cg
                 (An_Instance => Container.This_subsystem.The_right_mass_Property);
      Tot_Lbs := Left_Lbs + Right_Lbs + Coll_Lbs;
      Container.This_Subsystem.The_Fuel_Cg.fs :=
          (Left_Cg.Fs * Left_Lbs + Right_Cg.Fs * Right_Lbs + Coll_Cg.Fs * Coll_Lbs)/Tot_Lbs;
      Container.This_Subsystem.The_Fuel_Cg.bl :=
        (Left_Cg.bl * Left_Lbs + Right_Cg.bl * Right_Lbs + Coll_Cg.bl * Coll_Lbs)/Tot_Lbs;
       Container.This_Subsystem.The_Fuel_Cg.wl :=
          (Left_Cg.wl * Left_Lbs + Right_Cg.wl * Right_Lbs + Coll_Cg.wl * Coll_Lbs)/Tot_Lbs;
      --end Update_Mass_Property;

      -----------------
      -- FUEL PROBES --
      -----------------
      L_Inside_Probe_Reading := Mass_Types.Lbm (Float'Max(0.0,Float'Min(310.0,Tank.Quantity (The_Left_Wing_Tank))));

      Fuel_Probe.Update
        (An_Instance  => Ctnr.This_Subsystem.The_Left_Wing_Tank_Inside_Fuel_Probe,
         Amount       => L_Inside_Probe_Reading,
         Fail_Command => False);

      L_Middle_Probe_Reading := Mass_Types.Lbm (Float'Max(0.0,Float'Min(140.0,Tank.Quantity (The_Left_Wing_Tank)-310.0)));

      Fuel_Probe.Update
        (An_Instance  => Ctnr.This_Subsystem.The_Left_Wing_Tank_Middle_Fuel_Probe,
         Amount       => L_Middle_Probe_Reading,
         Fail_Command => False);

      L_Outside_Probe_Reading := Mass_Types.Lbm (Float'Max(0.0,Float'Min(80.0,Tank.Quantity (The_Left_Wing_Tank)-450.0)));

      Fuel_Probe.Update
        (An_Instance  => Ctnr.This_Subsystem.The_Left_Wing_Tank_Outside_Fuel_Probe,
         Amount       => L_Outside_Probe_Reading,
         Fail_Command => Ios.Lo_Fuel_Probe_Fail_Malf);

      R_Inside_Probe_Reading := Mass_Types.Lbm (Float'Max(0.0,Float'Min(310.0,Tank.Quantity (The_Right_Wing_Tank))));

      Fuel_Probe.Update
        (An_Instance  => Ctnr.This_Subsystem.The_Right_Wing_Tank_Inside_Fuel_Probe,
         Amount       => R_Inside_Probe_Reading,
         Fail_Command => Ios.Ri_Fuel_Probe_Fail_Malf);

      R_Middle_Probe_Reading := Mass_Types.Lbm (Float'Max(0.0,Float'Min(140.0,Tank.Quantity (The_Right_Wing_Tank)-310.0)));

      Fuel_Probe.Update
        (An_Instance  => Ctnr.This_Subsystem.The_Right_Wing_Tank_Middle_Fuel_Probe,
         Amount       => R_Middle_Probe_Reading,
         Fail_Command => False);

      R_Outside_Probe_Reading := Mass_Types.Lbm (Float'Max(0.0,Float'Min(80.0,Tank.Quantity (The_Right_Wing_Tank)-450.0)));

      Fuel_Probe.Update
        (An_Instance  => Ctnr.This_Subsystem.The_Right_Wing_Tank_Outside_Fuel_Probe,
         Amount       => R_Outside_Probe_Reading,
         Fail_Command => False);

      Fuel_Probe.Update
        (An_Instance  => Ctnr.This_Subsystem.The_Collector_Tank_Fuel_Probe,
         Amount       => Tank.Quantity (The_Collector_Tank),
         Fail_Command => False);


      -- left_side = 3 left wing tank + 0.5 of collector
      -- right_side = 3 right wing tank + 0.5 of collector

      -- a malfunctioning probe will result in an erroneous value sent to indicator

      ----------------------------------------------------------------
      -- DEBUG VALUES.....(CIRCUMVENT EDM)
      ----------------------------------------------------------------

      IOS.LEFT_Fuel_Quantity_Sys  := Float((Fuel_Probe.Amount(Ctnr.This_Subsystem.The_Left_Wing_Tank_Inside_Fuel_Probe) +
                                            Fuel_Probe.Amount(Ctnr.This_Subsystem.The_Left_Wing_Tank_Middle_Fuel_Probe) +
                                            Fuel_Probe.Amount(Ctnr.This_Subsystem.The_Left_Wing_Tank_Outside_Fuel_Probe) +
                                            0.5*Fuel_Probe.Amount(Ctnr.This_Subsystem.The_Collector_Tank_Fuel_Probe) ));



      IOS.RIGHT_Fuel_Quantity_Sys := Float((Fuel_Probe.Amount(Ctnr.This_Subsystem.The_Right_Wing_Tank_Inside_Fuel_Probe) +
                                            Fuel_Probe.Amount(Ctnr.This_Subsystem.The_Right_Wing_Tank_Middle_Fuel_Probe) +
                                            Fuel_Probe.Amount(Ctnr.This_Subsystem.The_Right_Wing_Tank_Outside_Fuel_Probe) +
                                            0.5*Fuel_Probe.Amount(Ctnr.This_Subsystem.The_Collector_Tank_Fuel_Probe) ));



      Ios.Collector_Quantity := Float(Fuel_Probe.Amount(Ctnr.This_Subsystem.The_Collector_Tank_Fuel_Probe));


      -- If the IOS fuel slider widget is not being used to set the quantity, then
      -- reflect the current tank quantity back to the slider, so it can track the actual fuel quantity

      if not Ios.Set_Left_Tank_Quantity then
         Ios.Left_Tank_Requested_Quantity :=  Float( Tank.Quantity (The_Left_Wing_Tank));
      end if;

      if not Ios.Set_Right_Tank_Quantity then
         Ios.Right_Tank_Requested_Quantity := Float( Tank.Quantity (The_Right_Wing_Tank));
      end if;


      -------------------------------------------------------------------------------
      -- FUEL LOW LEVEL SENSORS
      -------------------------------------------------------------------------------
      --|Update wing tank low fuel sensors.


      if Fuel_Qty_Lo_Cb then
         Ctnr.Fuel_Low_Level_Sensor.Update
           (An_Instance       => Left_Wing_Tank_Low_Level_Sensor,
            Quantity_Level    => Tank.Quantity (The_Left_Wing_Tank),
            Malfunction_Close => False);
         Ctnr.Fuel_Low_Level_Sensor.Update
           (An_Instance       => Right_Wing_Tank_Low_Level_Sensor,
            Quantity_Level    => Tank.Quantity (The_Right_Wing_Tank),
            Malfunction_Close => False);
      else
         Ctnr.Fuel_Low_Level_Sensor.Update
           (An_Instance       => Left_Wing_Tank_Low_Level_Sensor,
            Quantity_Level    => 500.0,
            Malfunction_Close => False);
         Ctnr.Fuel_Low_Level_Sensor.Update
           (An_Instance       => Right_Wing_Tank_Low_Level_Sensor,
            Quantity_Level    => 500.0,
            Malfunction_Close => False);
      end if;


      ----------------------------------------------------------------
      -- * FUEL PRESSURE, FUEL FLOW, AND FUEL AVAILABLE FOR ENGINES --
      -- * SET LEAK FLOW RATE FOR CENTRAL FUEL LINE MALF            --
      ----------------------------------------------------------------

      if Ios.Fuel_Available then
         Fuel_Px_To_Engines          := 27.08;
         Fuel_Flo_To_Engines         := 1900.0;
         Central_Fuel_Line_Leak_Flow := 0.0;
         Fuel_Avail_To_Engines       := True;
      elsif In_Inverted_Flight_Timer < Flame_Out_Time then
         if Tank.Quantity (The_Collector_Tank) > 0.0 then
            if Ctnr.Shutoff_Valve.Is_Open (The_Firewall_Sov) or Sov_Timer < 10.0 then
               if Tank.Collector.Motive_Flow_Out (The_Collector_Tank) and not Ctnr.Boost_Pump.Is_On (The_Boost_Pump) then
                  if Ios.Central_Fuel_Line_Leak_Malf then
                     Fuel_Px_To_Engines          := 7.758/3.0;
                     Fuel_Flo_To_Engines         := 2454.298/3.0;
                     Central_Fuel_Line_Leak_Flow := 25.0 * 2454.298 * (2.0/3.0);
                     Fuel_Avail_To_Engines       := True;
                  else -- normal with motive and no pump
                     Fuel_Px_To_Engines          := 7.758;      -- avg from table K fuel model
                     Fuel_Flo_To_Engines         := 2454.298;   --
                     Central_Fuel_Line_Leak_Flow := 0.0;
                     Fuel_Avail_To_Engines       := True;
                  end if;
               elsif Ctnr.Boost_Pump.Is_On (The_Boost_Pump) then
                  if Ios.Central_Fuel_Line_Leak_Malf then
                     Fuel_Px_To_Engines          := 27.08/3.0;
                     Fuel_Flo_To_Engines         := 1900.0/3.0;
                     Central_Fuel_Line_Leak_Flow := 25.0 * 1900.0 * (2.0/3.0);
                     Fuel_Avail_To_Engines       := True;
                  else -- normal with pump on
                     Fuel_Px_To_Engines          := 27.08;  -- avg from table V fuel model
                     Fuel_Flo_To_Engines         := 1900.0; --
                     Central_Fuel_Line_Leak_Flow := 0.0;
                     Fuel_Avail_To_Engines       := True;
                  end if;
               else
                  Fuel_Px_To_Engines          := 0.0;
                  Fuel_Flo_To_Engines         := 0.0;
                  Central_Fuel_Line_Leak_Flow := 0.0;
                  Fuel_Avail_To_Engines       := False;
               end if;
            else
               Fuel_Px_To_Engines          := 0.0;
               Fuel_Flo_To_Engines         := 0.0;
               Central_Fuel_Line_Leak_Flow := 0.0;
               Fuel_Avail_To_Engines       := False;
            end if;
         else
            Fuel_Px_To_Engines          := 0.0;
            Fuel_Flo_To_Engines         := 0.0;
            Central_Fuel_Line_Leak_Flow := 0.0;
            Fuel_Avail_To_Engines       := False;
         end if;

      else
         Fuel_Px_To_Engines          := 0.0;
         Fuel_Flo_To_Engines         := 0.0;
         Central_Fuel_Line_Leak_Flow := 0.0;
         Fuel_Avail_To_Engines       := False;
      end if;
      Ios.Fuel_Available_Sys  := Fuel_Avail_To_Engines;
      Ios.Fuel_Px_To_Engines  := Fuel_Px_To_Engines;
      Ios.Fuel_Flo_To_Engines := Fuel_Flo_To_Engines;


      -----------
      -- SOUND --
      -----------

      if Boost.Is_On (The_Boost_Pump) then
         Sound.Boost_Pump_On := 1.0;
      else
         Sound.Boost_Pump_On := 0.0;
      end if;


      -------------------------
      --                     --
      -------------------------

      if not Ios.Central_Fuel_Line_Leak_Malf and Central_Fuel_Line_Leak_Malf_Lp then
         null; --Ios.Fill_Tanks := True;
      end if;
      Central_Fuel_Line_Leak_Malf_Lp := Ios.Central_Fuel_Line_Leak_Malf;


   exception
      when others =>
         Log.Report("Jpats_Fuel.Controller.Update()");
         raise;
   end Update;



end Jpats_Fuel.Controller;

