-------------------------------------------------------------------------------
--|
--|            FlightSafety International Simulation Systems Division
--|                     Broken Arrow, OK  USA  918-259-4000
--|
--|                  JPATS T-6A Texan-II Flight Training Device
--|
--|
--|   Engineer:  Howard Landmann
--|
--|   Revision:  (Number and date inserted by Clearcase)
--|
--|
--|  DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
--|  Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
--|  technical or operational data or information from automatic dissemination
--|  under the International Exchange Program or by other means.  This protection
--|  covers information required solely for administrative or operational
--|  purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--|
--|  WARNING:  This document contains technical data whose export is restricted
--|  by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
--|  Executive Order 12470.  Violation of these export control laws is subject
--|  to severe criminal penalties.  Dissemination of this document is controlled
--|  under DOD Directive 5230.25
--|
-------------------------------------------------------------------------------
--| Reference: Kimball, D., "Flight Simulation Mathematical Model of
--|            the Beech MkII Joint Primary Aircraft Training System
--|            (JPATS)", Document 133E794 Rev. B, 11 November, 1998.
--|            Flight Manual Dated April 30,1998.
--|                          AIR FORCE TO 1T-6A-1,
--|                          NAVY (NAVAIR) A1-T6AAA-NFM-100
--|            Standards from Flight Dynamics Group
--|            Standards from Auto-Flight Group
--|            Flight Safety International, Inc., System Simulation Division
--|            Broken Arrow, OK 74012
-------------------------------------------------------------------------------
--|
with Ada.Numerics.Elementary_Functions;         use Ada.Numerics.Elementary_Functions;
with Ada.Numerics.Float_Random;                 use Ada.Numerics.Float_Random;

with Log;

with JPATS_Aircraft_Body;
with JPATS_Atmosphere;
with JPATS_Secondary_Flight_Controls;
with JPATS_Simulated_Aircraft;
with JPATS_Simulated_Aircraft_Types;
with Jpats_Radio_Db_If;
with Jpats_Landing_Gear;
with Jpats_Landing_Gear_Types;
with JPATS_Flight_Instruments.Container;
with Jpats_Flight_Instruments_Types;
with Jpats_Electrical;
with Jpats_Electrical_Types;
with Interpolation_Table;
with Interpolation_Table.singly_Indexed;
with Interpolation_Table.Doubly_Indexed;
with Interpolation_Table.Multiply_Indexed;
with Arinc_429_Types;
with Angle_Types;
with Temperature_Types;
with Air_Data_Computer;
with Turn_Indicator;
with Flight_Instruments_Types;    use Flight_Instruments_Types;
with Flight_Instruments_Arinc_Labels;
with Jpats_Electrical;
with Jpats_io;
with Jpats_Io_Types;
with Io_Types;

package body JPATS_Flight_Instruments.Controller is

   package It renames Interpolation_Table;
   package Jsa renames Jpats_Simulated_Aircraft;
   package Cnt renames Container;
   package A429 renames Flight_Instruments_Arinc_Labels;
   package Jiot renames Jpats_Io_Types;
   package Iot renames Io_Types;
   A429_out : Container.Arinc_Output_instance renames Container.This_Arinc_output;

   --| Specifying the path in which the DATA files for lookup tables are stored in.
   A_File_Path : String renames JPATS_Flight_Instruments_Types.A_File_Path;

   DHPADC_T           : aliased IT.Multiply_Indexed.Instance;
   Flap_effect_t      : aliased IT.Multiply_Indexed.Instance;
   VKIASADC_T         : aliased IT.Doubly_Indexed.Instance;
   Cas_T              : aliased IT.Doubly_Indexed.Instance;
   Mach_T             : aliased IT.Doubly_Indexed.Instance;
   AOACAL_T           : aliased IT.Doubly_Indexed.Instance;
   Sb_Ias_T           : aliased IT.singly_Indexed.Instance;
   Sb_Alt_T           : aliased IT.singly_Indexed.Instance;
   Rad2deg            : constant Float := 57.3;


   procedure Register_Ios_Variables is separate;
   procedure Register_Io_Variables is separate;

   procedure Initialize is
   begin
      Register_Ios_Variables;
      Register_Io_Variables;

      IT.Read(A_File_Path & "dhpadc.ito", DHPADC_T);
      IT.Read(A_File_Path & "vkiasadc.ito", VKIASADC_T);
      IT.Read(A_File_Path & "aoacal.ito",AOACAL_T);
      IT.Read(A_File_Path & "ias_sa.ito",Flap_effect_t);
      IT.Read(A_File_Path & "cas_ai.ito",CAS_T);
      IT.Read(A_File_Path & "tm_cai.ito",Mach_T);
      IT.Read(A_File_Path & "sb_ias.ito",Sb_ias_T);
      IT.Read(A_File_Path & "sb_alt.ito",Sb_alt_T);

   end Initialize;

   procedure Update(Dt       :in Float) is
      Adc_off                 : Boolean;
      Alt_Volts               : Float;
      Ay                      : Float;
      Az                      : Float;
      Num_G                   : Float;
      R_dot                   : Angle_Types.Radians_per_Sq_Sec;
      roll                    : Angle_Types.Radians;
      Q_dot                   : Float; -- pitch accel
      Pitch_Rate              : Angle_Types.Radians_per_Sec;
      Yaw_Rate                : Angle_Types.Radians_per_Sec;
      pressure                : Force_Types.Pounds_per_Sq_Feet;
      Density                 : Mass_Types.Slugs_per_Cubic_Feet;
      dyn_pressure            : Force_Types.Pounds_per_Sq_Feet;
      Blockage                : Float := 0.0;
      Sl_pressure             : Force_Types.Pounds_per_Sq_Feet;
      Flap_Pos                : Angle_Types.Degrees;
      Oat                     : Temperature_Types.Rankine;
      Beta                    : Angle_Types.Radians;
      Temp                    : Integer := 0;
      Word_Out                : Arinc_429_Types.Message_type;
      Temp1                   : Float;
      Temp_Bool               : Boolean;
      Gear_Up                 : Boolean;
   begin

      Ay           := JSA.Load_factor.Y * 32.2;
      Az           := - JSA.Load_factor.Z * 32.2;
      Num_G        := - JSA.Load_factor.Z;
      R_dot        := JSA.Get_Angular_Acceleration_Body_Axis.Yaw;
      roll         := JSA.Get_Roll_Angle;
      Q_dot        := JSA.Get_Angular_Acceleration_Body_Axis.Pitch;
      Sl_Pressure  := Jpats_Atmosphere.Sl_Pressure;
      Yaw_Rate     := Jsa.Get_Yaw_Rate;
      pitch_Rate   := Jsa.Get_pitch_Rate;
      Beta         :=  Jsa.Get_Side_Slip_Angle;
      Density        := Jpats_Atmosphere.Density;
      Pressure       := Jpats_Atmosphere.Pressure;
      dyn_Pressure   := JSA.Get_Dynamic_Pressure;
      Oat            := Jpats_Atmosphere.Temperature;
      Flap_Pos       := JPATS_Secondary_Flight_Controls.Mean_Flap_Position;
      Gear_Up := JPATS_Landing_Gear.Is_closed(Jpats_Landing_Gear_Types.Left_Gear_Up_And_Locked_Switch ) and
        JPATS_Landing_Gear.Is_closed(Jpats_Landing_Gear_Types.Right_Gear_Up_And_Locked_Switch) and
        JPATS_Landing_Gear.Is_closed(Jpats_Landing_Gear_Types.Nose_Gear_Up_And_Locked_Switch );

      if JPATS_Electrical.Is_Powered(Jpats_Electrical_Types.Trn_Rt_Cb) then
         Turn_Indicator.Set_Turn_Rate
           (roll         ,
            Pitch_rate   ,
            Yaw_rate     ,
            An_Instance => Cnt.This_Subsystem.The_Standby_Turn_Indicator);
         -- off flag is retracted
         Cnt.This_IO_Interface.Stdby_Turn_Slip_Flag := True;
      else
         -- set flag, do nothing to indicator
         -- display off flag
         Cnt.This_IO_Interface.Stdby_Turn_Slip_Flag := False;
      end if;




      -- scale turn rate to 10 volts (2 min turn = 3 deg/sec = .0523 rad/sec)
      temp1 := turn_Indicator.Get_Turn_Rate
        (Cnt.This_Subsystem.The_Standby_Turn_Indicator)*60.0*180.0/Ada.Numerics.Pi/360.0*13.0*0.85 ;
      if Temp1 > 10.0 then
         Temp1 := 10.0;
      elsif Temp1 < -10.0 then
         Temp1 := -10.0;
      end if;
      Cnt.This_IO_Interface.Stdby_Turn_Rate := Temp1;




      -- slip ball position computed in volts
      Container.EADI_Slip_Indicator.Update
        (Ay            ,
         Az            ,
         R_dot         ,
         Dt            ,
         An_Instance => Cnt.This_Subsystem.The_EADI_Slip_Indicator);
      Container.Standby_Slip_Indicator.Update
        (Ay            ,
         Az            ,
         R_dot         ,
         Dt            ,
         An_Instance => Cnt.This_Subsystem.The_Standby_Slip_Indicator);

      Cnt.This_IO_Interface.Eadi_Slip_Ball :=
        Container.EADI_Slip_Indicator.Position
        (Cnt.This_Subsystem.The_EADI_Slip_Indicator)/0.18; -- 0.18 inches/ball

      Cnt.This_IO_Interface.Stdby_Slip_Ball :=
        Container.Standby_Slip_Indicator.Position
        (Cnt.This_Subsystem.The_Standby_Slip_Indicator)/0.34;  -- 0.34 inches/ball


      if Num_g < -3.0 then
         Num_g := -3.0;
      elsif Num_g > 10.0 then
         Num_g := 10.0;
      end if;

      --  per Bob Davis 12/4 scaling will be done on IO side
      Cnt.This_IO_Interface.Load_factor := num_g;

      -- Compute pnematic instrument indications
      -- note; no ice buildup on static source, no malfunctions on standby
      -- compute pitot and static "sensed" pressures

      --  Primary Static port -- subject to malfunction
      if Cnt.This_IOS_Interface.Static_Block then
         Blockage := 1.0;
      else
         Blockage := 0.0;
      end if;
      Air_data_Computer.Compute_Sensed_static_Pressure
        (Pressure,
         Blockage,
         An_Instance => Cnt.This_Subsystem.The_Primary_Adc);

      --  primary pitot probe -- subject to malfunction and/or ice
      if Cnt.This_IOS_Interface.pitot_Block then
         Blockage := 1.0;
      else
         Blockage := 0.0;
      end if;

      blockage := Jpats_Atmosphere.Primary_Pitot_Ice + Blockage;
      if Blockage > 1.0 then
         Blockage := 1.0;
      end if;
      Air_data_Computer.Compute_Sensed_pitot_Pressure
        (Dyn_Pressure,
         Pressure ,
         Blockage,
         An_Instance => Cnt.This_Subsystem.The_Primary_Adc);

      -- standby static port has no malfunction or ice blockage
      Air_data_Computer.Compute_Sensed_static_Pressure
        (Pressure,
         0.0,
         An_Instance => Cnt.This_Subsystem.The_Standby_Adc);

      -- Standby pitot -- subject only to ice
      Blockage := Jpats_Atmosphere.standby_Pitot_Ice;
      Air_data_Computer.Compute_Sensed_Pitot_Pressure
        (Dyn_Pressure,
         Pressure,
         Blockage,
         An_Instance => Cnt.This_Subsystem.The_standby_Adc);

      Air_Data_Computer.Compute_Pressure_Altitude
        (Pressure   ,
         An_Instance => Cnt.This_Subsystem.The_Primary_adc);

      --  includes malfunctions and ice
      Air_Data_Computer.Compute_Sensed_Altitude
        (An_Instance => Cnt.This_Subsystem.The_Primary_adc);

      Air_Data_Computer.Compute_Sensed_Altitude
        (An_Instance => Cnt.This_Subsystem.The_standby_adc);

      --Includes Position error
      Air_Data_Computer.Compute_Primary_ind_Alt
        (Flap_Pos  ,
         DHPADC_T  ,
         An_Instance => Cnt.This_Subsystem.The_Primary_adc);

      Air_Data_Computer.Compute_standby_ind_Alt
        (Flap_Pos     ,
         Flap_effect_T,
         An_Instance => Cnt.This_Subsystem.The_standby_adc);

      -- Standby altimeter sin/cos drive
      -- first get what altimeter is currently indicating
      Alt_Volts := Cnt.This_Io_Interface.Altimeter_Volts;
      Air_Data_Computer.Stdby_Alt_Drive
        (Alt_Volts,
         Sb_Alt_T,
         An_Instance => Cnt.This_Subsystem.The_standby_adc);

      -- get sin/cos (-1 > +1) and output to IO
      Cnt.This_Io_Interface.Stdby_Alt_Sin :=
        Air_Data_Computer.Stdby_sin(Cnt.This_Subsystem.The_standby_adc);
      Cnt.This_Io_Interface.Stdby_Alt_cos :=
        Air_Data_Computer.Stdby_cos(Cnt.This_Subsystem.The_standby_adc);

      -- indicated airspeed
      Air_Data_Computer.calc_Primary_Ias
        (Flap_Pos  ,
         Gear_Up   ,
         Density   ,
         OAT       ,
         Yaw_Rate  ,
         Jsa.Get_Angle_Of_Attack * 57.3,
         VKIASADC_T,
         An_Instance => Cnt.This_Subsystem.The_Primary_adc);

      Air_Data_Computer.calc_Standby_Ias
        (Flap_Pos     ,
         Gear_Up      ,
         Beta         ,
         Density      ,
         Yaw_Rate     ,
         CAS_T        ,
         Mach_T       ,
         An_Instance => Cnt.This_Subsystem.The_standby_adc);

      cnt.This_IO_Interface.Stdby_Airspeed :=
        Air_Data_Computer.Indicated_Airspeed(CNT.This_Subsystem.The_Standby_ADC);
      cnt.This_IO_Interface.Stdby_Max_os   :=
        Air_Data_computer.Maximum_Operating_Airspeed(Cnt.This_Subsystem.The_standby_adc);

      -- Rate of Climb Indicator
      if not Jsa.Get_Flight_Freeze then
         Air_Data_Computer.Calc_Vvi
           (Dt, An_Instance => Cnt.This_Subsystem.The_Primary_adc);
      end if;

      --   check for adc powered or fail malfunction,
      --   fail message on if malfunction or if no power
      if not JPATS_Electrical.Is_Powered(Jpats_Electrical_Types.Adc_Cb) then
         Adc_Off := True;
      elsif Container.This_Ios_Interface.Adc_Fail then
         Adc_Off := True;
      else
         Adc_Off := False;
      end if;

      for Channel in a429.FI_Bnr_Sdi_Type range a429.Channel_A .. a429.Channel_B  loop

         Air_Data_Computer.Calc_Label_203
           (Word_Out,
            Channel,
            Adc_off,
            An_Instance => Cnt.This_Subsystem.The_Primary_adc);
         Container.This_Arinc_Output.A429_Tx_Adc_203_Alt(Channel) := Word_Out;

         Air_Data_Computer.Calc_Label_205
           (Word_Out,
            Channel,
            Adc_off,
            An_Instance => Cnt.This_Subsystem.The_Primary_adc);
         Container.This_Arinc_Output.A429_Tx_Adc_205_mach(Channel) := Word_Out;

         Air_Data_Computer.Calc_Label_206
           (Word_Out,
            Channel,
            Adc_off,
            An_Instance => Cnt.This_Subsystem.The_Primary_adc);
         Container.This_Arinc_Output.A429_Tx_Adc_206_ias(Channel) := Word_Out;

         Air_Data_Computer.Calc_Label_207
           (Word_Out,
            Channel,
            Adc_off,
            An_Instance => Cnt.This_Subsystem.The_Primary_adc);
         Container.This_Arinc_Output.A429_Tx_Adc_207_Max_as(Channel) := Word_Out;

         Air_Data_Computer.Calc_Label_212
           (Word_Out,
            Channel,
            Adc_off,
            An_Instance => Cnt.This_Subsystem.The_Primary_adc);
         Container.This_Arinc_Output.A429_Tx_Adc_212_vvi(Channel) := Word_Out;

      end loop;

      -- repeater instruments on IOS
      cnt.This_IOS_Interface.True_Hdg       := Jsa.Get_Hdg_Angle * rad2deg;
      cnt.This_IOS_Interface.Mag_Hdg        := Jsa.Get_Hdg_Angle * Rad2deg - Jpats_Radio_Db_If.Magnetic_Variation;
      cnt.This_IOS_Interface.Ind_Mach       := Air_Data_computer.Indicated_Mach_number(Cnt.This_Subsystem.The_Primary_adc);
      cnt.This_IOS_Interface.Ias_Lim        := Air_Data_Computer.Maximum_Operating_Airspeed(CNT.This_Subsystem.The_Primary_adc);
      cnt.This_IOS_Interface.Nz             := -Jsa.Load_Factor.Z;
      cnt.This_IOS_Interface.Baro_Set       := Jpats_Atmosphere.Sl_Pressure;
      cnt.This_IOS_Interface.Baro_Set_mb    := cnt.This_IOS_Interface.Baro_Set * 25.401;
      cnt.This_IOS_Interface.Rate_Of_Climb  :=  Air_Data_computer.Get_vvi(Cnt.This_Subsystem.The_Primary_adc);
      cnt.This_IOS_Interface.Oat_Deg        := Temperature_Types.Rankine_To_Celsius(Oat);
      cnt.This_IOS_Interface.EADI_Slip_Ball := Container.EADI_Slip_Indicator.Position(Cnt.This_Subsystem.The_EADI_Slip_Indicator)/0.18;
      cnt.This_IOS_Interface.Ias            := Air_Data_computer.Indicated_airspeed(Cnt.This_Subsystem.The_Primary_adc);
      cnt.This_IOS_Interface.Vsi_Power      := JPATS_Electrical.Is_Powered(Jpats_Electrical_Types.vvi_Cb);
      cnt.This_IOS_Interface.Asi_Power      := JPATS_Electrical.Is_Powered(Jpats_Electrical_Types.asi_Cb);
      cnt.This_IOS_Interface.Alt_Power      := JPATS_Electrical.Is_Powered(Jpats_Electrical_Types.Altm_Cb_Fwd_Avi_gen_bus);


      -- note: IOS Instrument Repeater Altimeter displays geometric altitude for now.
      -- This is due to the lack of barometric pressure input from the cockpit altimeter.
      if not Cnt.This_IOS_Interface.Static_Block then
         cnt.This_IOS_Interface.Ind_altitude := Jsa.Get_Aircraft_Geometric_Altitude;
      end if;

      -- ***************************************************************************************

      --   Power To Light Up  glass instruments
      Cnt.This_IO_Interface.Vvi_Pwr         := JPATS_Electrical.Is_Powered(Jpats_Electrical_Types.vvi_Cb);
      Cnt.This_IO_Interface.asi_Pwr         := JPATS_Electrical.Is_Powered(Jpats_Electrical_Types.asi_Cb);
      Cnt.This_IO_Interface.alt_Pwr         := JPATS_Electrical.Is_Powered(Jpats_Electrical_Types.Altm_Cb_Fwd_Avi_gen_bus);
      Cnt.This_Ios_Interface.Parameter_Frz_On:=Cnt.This_Io_Interface.Parameter_Freeze_On;

      -- overspeed warning
      Temp_bool := Cnt.This_IOS_Interface.Ias > Jpats_Flight_Instruments.Max_Operating_Airspeed - 2.0;
      Cnt.This_Subsystem.Overspeed_Warn :=
        (Temp_Bool or Cnt.This_Io_Interface.Overspeed_Test)
        and JPATS_Electrical.Is_Powered(Jpats_Electrical_Types.Aural_Warn_Cb);

   end Update;

   end JPATS_Flight_Instruments.Controller;







