
-------------------------------------------------------------------------------
--|
--|            FlightSafety International Simulation Systems Division
--|                     Broken Arrow, OK  USA  918-259-4000
--|
--|                  JPATS T-6A Texan-II Flight Training Device
--|
--|
--|   Engineer:  Asep Rahmat
--|
--|   Revision:  (Number and date inserted by Clearcase)
--|
--|
--|  DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
--|  Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
--|  technical or operational data or information from automatic dissemination
--|  under the International Exchange Program or by other means.  This protection
--|  covers information required solely for administrative or operational
--|  purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--|
--|  WARNING:  This document contains technical data whose export is restricted
--|  by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
--|  Executive Order 12470.  Violation of these export control laws is subject
--|  to severe criminal penalties.  Dissemination of this document is controlled
--|  under DOD Directive 5230.25
--|
-------------------------------------------------------------------------------
--| Reference: Kimball, D., "Flight Simulation Mathematical Model of
--|            the Beech MkII Joint Primary Aircraft Training System
--|            (JPATS)", Document 133E794 Rev. B, 11 November, 1998.
--|            Flight Manual Dated April 30,1998.
--|                          AIR FORCE TO 1T-6A-1,
--|                          NAVY (NAVAIR) A1-T6AAA-NFM-100
--|            Standards from Flight Dynamics Group
--|            Standards from Auto-Flight Group
--|            Flight Safety International, Inc., System Simulation Division
--|            Broken Arrow, OK 74012
-------------------------------------------------------------------------------
--|

with Ada.Numerics.Elementary_Functions;             use Ada.Numerics.Elementary_Functions;

with JPATS_Aircraft_Body;
with JPATS_Atmosphere;
with JPATS_Electrical;
with Jpats_Landing_Gear;
with JPATS_Secondary_Flight_Controls;
with JPATS_Simulated_Aircraft;
with JPATS_Electrical_Types;
with Jpats_Landing_Gear_Types;
with Angle_Types;
with JPATS_Flight_Instruments.Container;
with JPATS_Flight_Instruments_Types;
with Interpolation_Table.Singly_Indexed;
with Interpolation_Table.Doubly_Indexed;


package body JPATS_Flight_Instruments.AOA_System_Controller is
   package It renames Interpolation_Table;
   package Fit renames Flight_Instruments_Types;
   package Cnt renames Container;

   --| the path where DATA tables are stored
   A_File_Path : String := JPATS_Flight_Instruments_Types.A_File_Path;

   ASHAKE_T             : aliased It.Singly_Indexed.Instance;
   AOACAL_T             : aliased It.Doubly_Indexed.Instance;


   filt_Aoa             : Float renames CNT.This_Subsystem.Filt_Aoa;
   Ind_aoa              : Float renames CNT.This_Subsystem.Ind_Aoa;
   Old_Units            : Float renames CNT.This_Subsystem.Old_Units;

   procedure Initialize is
   begin

      It.Read(A_File_Path & "aoacal.ito",AOACAL_T);
      It.Read(A_File_Path & "ashake.ito",ASHAKE_T);

   end Initialize;

   procedure Update(Dt :in Float) is
      Gear_Down_And_Locked : Boolean;
      Power                : Boolean;
      Shaker_AoA           : Float := 0.0;
      Aoa_Indexer          : Integer := 0;
      Flap_Pos             : Float := 0.0;
      AOA                  : Angle_Types.Degrees;
      WOW                  : Boolean := false;
      units                : Float;
      Delt                 : Float;
   begin
      Gear_Down_And_Locked :=
        JPATS_Landing_Gear.Is_closed(Jpats_Landing_Gear_Types.Left_Gear_Down_And_Locked_Switch ) and
        JPATS_Landing_Gear.Is_closed(Jpats_Landing_Gear_Types.Right_Gear_Down_And_Locked_Switch) and
        JPATS_Landing_Gear.Is_closed(Jpats_Landing_Gear_Types.Nose_Gear_Down_And_Locked_Switch );
      Power := JPATS_Electrical.Is_Powered(Jpats_Electrical_Types.Aoa_Cb);
      Flap_Pos  := JPATS_Secondary_Flight_Controls.Mean_Flap_Position;
      WOW       := JPATS_Aircraft_Body.Get_WOW;

      if power then

         --   compute indicated AOA
         if (JPATS_Atmosphere.AoA_Vane_Ice/100.0) < 0.8 then
            -- normal operation; else indicator frozen

            AOA := Angle_Types.Radians_To_Degrees(JPATS_Simulated_Aircraft.Get_Angle_of_Attack);

            filt_aoa := It.Doubly_Indexed.Interpolate(aoa,Flap_Pos,AOACAL_T'access);
            if Wow then
               if (CNT.This_IO_Interface.AOA_Test_hi) then
                  Filt_Aoa := 18.0;
               elsif (CNT.This_IO_Interface.AOA_Test_lo) then
                  Filt_Aoa := 10.5;
               else
                  Filt_Aoa := 0.0;
               end if;
            end if;
         end if;

         ind_aoa := ind_Aoa + (filt_aoa - ind_aoa)* Dt;
         if Ind_Aoa < -7.5 then
            Ind_Aoa := -7.5;
         end if;

         -- During AOA tests, AOA indexer will follow the  gage (ind_aoa).
         -- DR 72744
         if (CNT.This_IO_Interface.AOA_Test_hi)or (CNT.This_IO_Interface.AOA_Test_lo) then
            Filt_Aoa := -Old_Units/12.0;
         end if;


         --  compute stick shaker on
         shaker_AoA := It.Singly_Indexed.Interpolate (Flap_Pos,ASHAKE_T'access);

         if ((CNT.This_IO_Interface.AOA_Test_hi and Wow ) or
             CNT.This_IOs_Interface.Stick_Shaker_On or
             (Ind_Aoa > shaker_Aoa  and not Wow)) and
           (not Jpats_Simulated_Aircraft.Get_Flight_Freeze) then
            CNT.This_IO_Interface.Shaker_on  := True;
         else
            CNT.This_IO_Interface.Shaker_on  := False;
         end if;

         -- AOA Indexer

         if Gear_Down_And_Locked  and
           Jpats_Electrical.Is_Powered(Jpats_Electrical_Types.Ldg_Gr_Cont_Cb) then
            if Filt_Aoa > 12.0 then
               CNT.This_IO_Interface.AOA_Index_HI  := True;
               CNT.This_IO_Interface.AOA_Index_OK  := False;
               CNT.This_IO_Interface.AOA_Index_LO  := False;
               Aoa_Indexer := 4;
            elsif Filt_Aoa > 11.0 then
               CNT.This_IO_Interface.AOA_Index_HI  := True;
               CNT.This_IO_Interface.AOA_Index_OK  := true;
               CNT.This_IO_Interface.AOA_Index_LO  := False;
               Aoa_Indexer := 6;
            elsif Filt_Aoa > 10.0 then
               CNT.This_IO_Interface.AOA_Index_HI  := false;
               CNT.This_IO_Interface.AOA_Index_OK  := true;
               CNT.This_IO_Interface.AOA_Index_LO  := False;
               Aoa_Indexer := 2;
            elsif Filt_Aoa > 9.0 then
               CNT.This_IO_Interface.AOA_Index_HI  := false;
               CNT.This_IO_Interface.AOA_Index_OK  := true;
               CNT.This_IO_Interface.AOA_Index_LO  := true;
               Aoa_Indexer := 3;
            else
               CNT.This_IO_Interface.AOA_Index_HI  := false;
               CNT.This_IO_Interface.AOA_Index_OK  := false;
               CNT.This_IO_Interface.AOA_Index_LO  := true;
               Aoa_Indexer := 1;
            end if;
         else
            CNT.This_IO_Interface.AOA_Index_HI  := false;
            CNT.This_IO_Interface.AOA_Index_OK  := false;
            CNT.This_IO_Interface.AOA_Index_LO  := false;
            Aoa_Indexer := 0;
         end if;

      else  -- system not powered
         Ind_Aoa := -4.0;
         CNT.This_IO_Interface.Shaker_On     := False;
         CNT.This_IO_Interface.AOA_Index_HI  := false;
         CNT.This_IO_Interface.AOA_Index_OK  := false;
         CNT.This_IO_Interface.AOA_Index_LO  := false;
         Aoa_Indexer := 0;
      end if;

      cnt.This_IOS_Interface.Aoa_Indexer := Aoa_Indexer; -- IOS repeater

      --| Outputs to cockpit and IOS
      --| for Angle of Attack Indicator
      --|  limit rate to 3 deg/sec
      units := -12.0 * Ind_Aoa;
      Delt := Units - Old_Units;
      if Delt > 2.4 then
         Delt := 2.4;
      elsif delt < -2.4 then
         Delt :=  - 2.4;
      end if;
      Old_Units := Old_Units + delt;
      CNT.This_IO_Interface.AOA_Sin := sin(old_units,360.0) * 10.0;  -- in volts
      CNT.This_IO_Interface.AOA_Cos := Cos(old_units,360.0) * 10.0;
      cnt.This_IOS_Interface.Aoa_Units := Ind_AOA;

      if (CNT.This_IO_Interface.AOA_Test_hi) then
         CNT.This_IOs_Interface.AOA_test_sw  := 2;
      elsif (CNT.This_IO_Interface.AOA_Test_lo) then
         CNT.This_IOs_Interface.AOA_test_sw  := 1;
      else
         CNT.This_IOs_Interface.AOA_test_sw  := 0;
      end if;

   end Update;
end JPATS_Flight_Instruments.AOA_System_Controller;




