-------------------------------------------------------------------------------
--|
--|            FlightSafety International Simulation Systems Division
--|                     Broken Arrow, OK  USA  918-259-4000
--|
--|                  JPATS T-6A Texan-II Flight Training Device
--|
--|
--|   Engineer:  Howard Landmann
--|
--|   Revision:  (Number and date inserted by Clearcase)
--|
--|
--|  DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
--|  Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
--|  technical or operational data or information from automatic dissemination
--|  under the International Exchange Program or by other means.  This protection
--|  covers information required solely for administrative or operational
--|  purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--|
--|  WARNING:  This document contains technical data whose export is restricted
--|  by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
--|  Executive Order 12470.  Violation of these export control laws is subject
--|  to severe criminal penalties.  Dissemination of this document is controlled
--|  under DOD Directive 5230.25
--|
-------------------------------------------------------------------------------
--|
with Ada.Numerics.Elementary_Functions;        use Ada.Numerics.Elementary_Functions;
with Jpats_Simulated_Aircraft;
with Jpats_Avionics; --for debug purposes only
with Save_Restore;
with JPATS_Flight_Instruments.Container;
with Jpats_Atmosphere;
with JPATS_Auto_Test;


package body Air_Data_Computer is

   package Cnt renames JPATS_Flight_Instruments.Container;

   No_Discretes : Arinc_429_Types.No_Discretes_Type;
   SSM          : Arinc_429_Types.Bnr_Ssm_Type;

   function Flimit(VALUE : float;
                   MIN   : float;
                   MAX   : float) return Float;

    procedure Set_Baro_Correction
        (altimeter_set    :in     Force_Types.Inches_of_Hg;
         An_Instance      :in out Instance) is
    begin
        An_Instance.The_Baro_Correction := 929.7 * (29.92 - altimeter_set);
    end Set_Baro_Correction;

    function Get_Baro_Correction
          (An_Instance :in Instance) return Length_Types.Feet is
    begin
        return An_Instance.The_Baro_Correction;
    end Get_Baro_Correction;

    procedure Compute_Pressure_Altitude
        (Pressure        : in  Force_Types.Pounds_per_Sq_Feet;
         An_Instance     : in out Instance) is
    begin
             An_Instance.The_Pressure_Altitude :=
               (1.0 - (Pressure/2116.2) ** 0.19018) * 145447.0;

    end Compute_Pressure_Altitude;

    procedure Compute_sensed_Altitude
        (An_Instance     : in out Instance) is
    begin
       An_Instance.The_Old_sensed_Altitude := An_Instance.The_sensed_Altitude;
       An_Instance.The_sensed_Altitude :=
         (1.0 - (An_Instance.Sensed_static_Press/2116.2)
                                         ** 0.19018) * 145447.0;
    end Compute_sensed_Altitude;

    procedure Compute_primary_Ind_alt
        (Flap_Pos            :in     Angle_Types.Degrees;
         DHPADC_Table        :in out IT.Multiply_Indexed.Instance;
         An_Instance         :in out Instance) is
       Temp : IT.Index_Vector := (0.0,0.0,0.0);
       Phase : Integer;
    begin

       -- Temporary work around until ECP to add baro press input from cockpit
       -- is completed. Check for autotest in progress to match aircraft
       -- altimeter correction, otherwise use geometric altitude.
       -- Autotest is off when phase is zero and trimming when phase is nine.
       --
       Phase := JPATS_Auto_Test.At_Phase;

       if Phase > 0 and Phase < 9 then  -- Autotest active
          Temp := (Flap_Pos ,
                   An_Instance.The_Indicated_Airspeed,
                   An_Instance.The_sensed_altitude);
          An_Instance.The_Indicated_Altitude :=
             An_Instance.The_sensed_altitude + IT.Multiply_Indexed.Interpolate
             (Temp,DHPADC_Table'access);
       else                             -- Autotest inactive
          if not Cnt.This_IOS_Interface.Static_Block then
             An_Instance.The_Indicated_Altitude :=
                Jpats_Simulated_Aircraft.Get_Aircraft_Geometric_Altitude +
                (910.0 * (29.92 - float(Jpats_Atmosphere.Sl_Pressure)));
          end if;
       end if;

       if An_Instance.The_Indicated_Altitude > 53000.0 then
          An_Instance.The_Indicated_Altitude := 53000.0;
       elsif An_Instance.The_Indicated_Altitude < -1000.0 then
          An_Instance.The_Indicated_Altitude := -1000.0;
       end if;

    end Compute_primary_Ind_alt;

    procedure Compute_Standby_Ind_alt
        (Flap_Pos         :in     Angle_Types.Degrees;
         Flap_Effect_T    :in out IT.Multiply_Indexed.Instance;
         An_Instance      :in out Instance) is
       Temp  : IT.Index_Vector := (0.0,0.0,0.0);
       Phase : Integer;
    begin

       -- Temporary work around until ECP to add baro press input from cockpit
       -- is completed. Check for autotest in progress to match aircraft
       -- altimeter correction, otherwise use geometric altitude.
       -- Autotest is off when phase is zero and trimming when phase is nine.
       --
       Phase := JPATS_Auto_Test.At_Phase;

       if Phase > 0 and Phase < 9 then  -- Autotest active
          Temp :=  (Flap_Pos ,
                    An_Instance.The_sensed_Altitude,
                    An_Instance.The_Indicated_Airspeed);
          An_Instance.The_Indicated_Altitude :=
             An_Instance.The_Sensed_altitude - IT.Multiply_Indexed.Interpolate
             (Temp,Flap_Effect_T'access);
       else                             -- Autotest inactive
          An_Instance.The_Indicated_Altitude :=
             Jpats_Simulated_Aircraft.Get_Aircraft_Geometric_Altitude +
             (910.0 * (29.92 - float(Jpats_Atmosphere.Sl_Pressure)));
       end if;

       if An_Instance.The_Indicated_Altitude > 53000.0 then
          An_Instance.The_Indicated_Altitude := 53000.0;
       elsif An_Instance.The_Indicated_Altitude < -1000.0 then
          An_Instance.The_Indicated_Altitude := -1000.0;
       end if;

    end Compute_Standby_Ind_alt;

    function primary_Ind_Alt
         (An_Instance :in Instance) return Length_Types.Feet is
    begin
        return An_Instance.The_Indicated_altitude;
    end primary_Ind_Alt;

    function standby_Ind_Alt
         (An_Instance :in Instance) return Length_Types.Feet is
    begin
        return An_Instance.The_Indicated_altitude;
    end standby_Ind_Alt;

    function pressure_Altitude
         (An_Instance :in Instance) return Length_Types.Feet is
    begin
        return An_Instance.The_pressure_altitude;
    end pressure_Altitude;

    procedure calc_Vvi (Dt :in     Float;
               An_Instance :in out Instance) is
    begin

      An_Instance.The_vvi := ( An_Instance.The_sensed_altitude
                          -  An_Instance.The_old_sensed_altitude) / dt * 60.0;
    end calc_VVI;

    function Get_VVI (An_Instance :in Instance)
                   return Length_Types.Feet_per_Min is
    begin
        return An_Instance.The_VVI;
    end Get_VVI;

   --| Raytheon's model for Mach, Ias, Maximum_Operating_speed
    procedure calc_Primary_Ias
        (Flap_Pos           :in     Angle_Types.Degrees;
         Gear_Up            :in     Boolean;
         air_dens           :in     Mass_Types.Slugs_per_Cubic_Feet;
         OAT                :in     Temperature_Types.Rankine;
         Yaw_Rate           :in     Angle_Types.Radians_Per_Sec;
         Alpha              :in     Float;
         VKIASADC_T         :in out IT.Doubly_Indexed.Instance;
         An_Instance        :in out Instance) is

         Sensed_CAS  : Length_Types.Knots        := 0.0;
         Sensed_Tas  : Length_Types.Feet_per_Sec := 0.0;
         Sensed_Mach : Length_Types.Mach         := 0.0;
         Max_Cas     : Length_Types.knots := 0.0;
         Max_Ias     : Length_Types.knots := 0.0;
    begin
        Sensed_Tas  := Sqrt(2.0 * An_Instance.Sensed_Pitot_Press
                            / Float'Max(air_dens,1.0e-4)) - Yaw_Rate * 6.0; -- pitot probe near L wingtip
        -- Change per D. Kimball to change the moment arm of pitot from 13.5 to 6.0
        -- This is to match test 483 & 484

        Sensed_Mach  := Sensed_Tas / Sqrt(2402.6 * oat); -- gamma*R*T

        Sensed_Cas := Sensed_Tas * Sqrt(Air_Dens/0.002378) * 0.5925;
        An_Instance.The_Indicated_Airspeed := IT.Doubly_Indexed.Interpolate
                                       (sensed_CAS,Flap_Pos,VKIASADC_T'access);
        -- hidh alpha correction
        An_Instance.The_Indicated_Airspeed := An_Instance.The_Indicated_Airspeed +
             1.5 * Float'Max(0.0,Float'Min(Alpha -20.0,10.0));

        An_Instance.The_Indicated_Mach_Number := Sqrt(5.0 *
              (((((An_Instance.The_Indicated_Airspeed / 1479.1)**2
                + 1.0) ** 3.5 -1.0)
              * (2116.22 / Float'Max(An_Instance.Sensed_static_press,1.0))
                + 1.0) ** (1.0/3.5) - 1.0 ));

        Max_Cas := 1479.1 * Sqrt(
                 (1.0 + An_Instance.Sensed_Static_Press / 2116.22 * 0.36285)** 0.2857 - 1.0);

        Max_Ias := IT.Doubly_Indexed.Interpolate(max_CAS,Flap_Pos,VKIASADC_T'access);
        An_Instance.The_Max_Operating_Airspeed:= Float'Min(316.0,Max_Ias);

    end Calc_Primary_Ias;

    procedure calc_Standby_Ias
        (Flap_Pos            :in     Angle_Types.Degrees;
         Gear_Up             :in     Boolean;
         Side_Slip_Angle     :in     Angle_Types.Radians;
         Air_Dens            :in     Mass_Types.Slugs_per_Cubic_Feet;
         Yaw_Rate            :in     Angle_Types.Radians_Per_Sec;
         CAS_T               :in out IT.Doubly_Indexed.Instance;
         Mach_T              :in out IT.Doubly_Indexed.Instance;
         An_Instance         :in out Instance) is

         sensed_CAS  : Length_Types.knots := 0.0;
         Sensed_Tas  : Length_Types.knots := 0.0;
         Max_Cas     : Length_Types.knots := 0.0;
         Max_Ias     : Length_Types.knots := 0.0;
    begin

        Sensed_Tas  := Sqrt(2.0 * An_Instance.Sensed_Pitot_Press
                            / Float'Max(air_dens,1.0e-4)) + Yaw_Rate * 6.0; -- pitot probe near R wingtip
        -- Change per D. Kimball to change the moment arm of pitot from 13.5 to 6.0
        -- This is to match test 483 & 484
        --     Cas := JPATS_Simulated_Aircraft.Get_Calibrated_Airspeed;
        Sensed_Cas := Sensed_Tas * Sqrt(Air_Dens/0.002378) * 0.5925;
        An_Instance.ADC_Mach_Number := IT.Doubly_Indexed.Interpolate
                  (An_Instance.The_sensed_altitude,sensed_cas,Mach_T'access);
        --| Based on Mach v/s Calibrated airspeed from RAC Flight Manual Pg. A1-14.
        An_Instance.The_Indicated_Airspeed := sensed_Cas -
          IT.Doubly_Indexed.Interpolate(Flap_Pos,Sensed_CAS,CAS_T'access);

        An_Instance.The_Indicated_Mach_Number :=
          An_Instance.adc_Mach_Number - 0.01;
        Max_Cas := 1479.1 * Sqrt
            ((1.0 + An_Instance.Sensed_Static_Press / 2116.22 * 0.36285)** 0.2857 - 1.0);


        Max_Ias := max_Cas - IT.Doubly_Indexed.Interpolate(Flap_Pos,max_cas,CAS_T'access);
        An_Instance.The_Max_Operating_Airspeed := Float'Min(316.0,Max_Ias);

   end calc_Standby_ias;

   procedure Compute_Sensed_Pitot_Pressure
                      -- (true_mach   : in     Float;
                     (Dyn_Press   : in     Force_Types.Pounds_per_Sq_Feet;
                      True_Static : in     Force_Types.Pounds_per_Sq_Feet;
                      blockage    : in     Float;
                      An_Instance : in out Instance) is
      True_Total_Pressure : Float := 0.0;
   begin
      --  True_Total_Pressure := True_Static * (1.0 + 0.2 * True_Mach * True_Mach) ** 3.5;
      True_Total_Pressure := True_Static + dyn_press;
      An_Instance.Sensed_Total_press := An_Instance.Sensed_Total_press + (1.0 - Blockage) *
                                                   (True_Total_Pressure - An_Instance.Sensed_Total_press);
      An_Instance.Sensed_pitot_Press := An_Instance.Sensed_Total_press - An_Instance.Sensed_Static_Press;
      --   prevent negative pressure which can happen with pitot blockage
      --   will cause exception in ind airspeed calc (sqrt neg number)
      if An_Instance.Sensed_Pitot_Press < 0.0 then
         An_Instance.Sensed_Pitot_Press := 0.01;
      end if;
   end Compute_Sensed_Pitot_Pressure;

    procedure Compute_Sensed_static_Pressure
                     (Pressure    : in     Force_Types.Pounds_per_Sq_Feet;
                      Lag         : in     Float;
                      An_Instance : in out Instance) is
   begin
      An_Instance.Sensed_static_Press :=
           An_Instance.Sensed_static_Press + (1.0 - Lag)
               * (Pressure - An_Instance.Sensed_static_Press);
   end Compute_Sensed_static_Pressure;

   function Indicated_Airspeed (An_Instance :in Instance) return Length_Types.Knots is
   begin
       return An_Instance.The_Indicated_Airspeed;
   end Indicated_Airspeed;

   function Maximum_Operating_Airspeed
             (An_Instance :in Instance) return Length_Types.Knots is
   begin
       return An_Instance.The_Max_Operating_Airspeed;
   end Maximum_Operating_Airspeed;

   function Indicated_Mach_Number(An_Instance :in Instance)
                     return Length_Types.Mach is
   begin
       return An_Instance.The_Indicated_Mach_Number;
   end Indicated_Mach_Number;

   function Overspeed_Warning (An_Instance :in Instance) return Boolean is
   begin
       return An_Instance.The_Overspeed_Aural_Warning;
   end Overspeed_Warning;

   procedure Calc_Label_203(Tx_203_Alt  :     out Arinc_429_Types.Message_type;
                            Channel     : in      Flight_instruments_Arinc_Labels.Fi_Bnr_Sdi_Type;
                            Adc_off     : in      Boolean;
                            An_Instance : in      instance) is
      Altitude : Float;
      Output   : Float;
   begin
         Altitude := An_Instance.The_Indicated_altitude;

         if Altitude  >= 65536.0 then
           Output := 65536.0;
         elsif Altitude <= -2000.0 then
           Output := -2000.0;
         else
           Output := altitude;
         end if;

         -- set SSM bits
         if  Adc_off then
           Ssm := Arinc_429_Types.Failure;
         elsif Altitude < -2000.0 or Altitude > 65536.0 then
           Ssm := Arinc_429_Types.No_Computed_Data;
         else
           Ssm := Arinc_429_Types.Normal;
         end if;

         TX_203_Alt :=
           Flight_Instruments_Arinc_labels.Label_203.Pack
           ( A_Value           => Output,
             A_Sdi             => Channel,
             A_Ssm             => Ssm,
             A_Discretes_Value => No_Discretes);

   end Calc_Label_203;

   procedure Calc_Label_205(Tx_205_mach    :     out Arinc_429_Types.Message_type;
                            Channel        : in     Flight_instruments_Arinc_Labels.Fi_Bnr_Sdi_Type;
                            Adc_off        : in     Boolean;
                            An_Instance    : in     instance) is
          Mach : Float;
      Output   : Float;
   begin
         Mach := An_Instance.The_Indicated_Mach_number;

         if Mach  >= 2.048 then
           Output := 2.048;
         elsif Mach <= 0.0 then
           Output := 0.0;
         else
           Output := mach;
         end if;

         -- set SSM bits
         if  Adc_off then
           Ssm := Arinc_429_Types.Failure;
         elsif Mach > 2.048 then
           Ssm := Arinc_429_Types.No_Computed_Data;
         else
           Ssm := Arinc_429_Types.Normal;
         end if;

         TX_205_mach :=
           Flight_Instruments_Arinc_labels.Label_205.Pack
           ( A_Value           => Output,
             A_Sdi             => Channel,
             A_Ssm             => Ssm,
             A_Discretes_Value => No_Discretes);

   end Calc_Label_205;

   procedure Calc_Label_206(Tx_206_ias     :    out Arinc_429_Types.Message_type;
                            Channel        : in     Flight_instruments_Arinc_Labels.Fi_Bnr_Sdi_Type;
                            Adc_off        : in     Boolean;
                            An_Instance    : in     instance) is
      Ias      : Float;
      Output   : Float;
   begin
         Ias := An_Instance.The_Indicated_airspeed;

         if Ias  >= 512.0 then
           Output := 512.08;
         elsif IAS <= 0.0 then
           Output := 0.0;
         else
           Output := IAS;
         end if;

         -- set SSM bits
         if  Adc_off then
           Ssm := Arinc_429_Types.Failure;
         elsif IAS > 450.0 then
           Ssm := Arinc_429_Types.No_Computed_Data;
         else
           Ssm := Arinc_429_Types.Normal;
         end if;

         TX_206_ias :=
           Flight_Instruments_Arinc_labels.Label_206.Pack
           ( A_Value           => Output,
             A_Sdi             => Channel,
             A_Ssm             => Ssm,
             A_Discretes_Value => No_Discretes);

   end Calc_Label_206;

   procedure Calc_Label_207(Tx_207_Max_as  :     out Arinc_429_Types.Message_type;
                            Channel        : in     Flight_instruments_Arinc_Labels.Fi_Bnr_Sdi_Type;
                            Adc_off        : in     Boolean;
                            An_Instance    : in     instance) is
      Max_AS : Float;
      Output   : Float;
   begin
         Max_As := An_Instance.The_Max_Operating_airspeed;

         if Max_as  >= 512.0 then
            Output := 512.0;
         elsif Max_as <= 0.0 then
            Output := 0.0;
         else
           Output := Max_as;
         end if;

         -- set SSM bits
         if  Adc_off then
           Ssm := Arinc_429_Types.Failure;
         elsif Max_as  > 360.0 then
           Ssm := Arinc_429_Types.No_Computed_Data;
         else
           Ssm := Arinc_429_Types.Normal;
         end if;

         TX_207_Max_as :=
           Flight_Instruments_Arinc_labels.Label_207.Pack
           ( A_Value           => Output,
             A_Sdi             => Channel,
             A_Ssm             => Ssm,
             A_Discretes_Value => No_Discretes);

   end Calc_Label_207;

   procedure Calc_Label_212(Tx_212_vvi     :    out Arinc_429_Types.Message_type;
                            Channel        : in     Flight_instruments_Arinc_Labels.Fi_Bnr_Sdi_Type;
                            Adc_off        : in     Boolean;
                            An_Instance    : in     instance)is
      Vvi       : Float;
      Output    : Float;
   begin
         vvi := An_Instance.The_vvi;

         if vvi  >= 5900.0 then
           Output := 5900.0;
         elsif vvi <= -5900.0 then
           Output := -5900.0;
         else
           Output := vvi;
         end if;
         -- set SSM bits
         if  Adc_off then
           Ssm := Arinc_429_Types.Failure;
         elsif vvi < -25000.0 or vvi > 25000.0 then
           Ssm := Arinc_429_Types.No_Computed_Data;
         else
           Ssm := Arinc_429_Types.Normal;
         end if;

        TX_212_vvi :=
           Flight_Instruments_Arinc_labels.Label_212.Pack
           ( A_Value           => Output,
             A_Sdi             => Channel,
             A_Ssm             => Ssm,
             A_Discretes_Value => No_Discretes);

   end Calc_Label_212;

   procedure Scale_airspeed(Sb_Ias_t     :in out IT.singly_Indexed.Instance;
                            An_Instance  :in out Instance) is
       begin
        An_Instance.The_Scaled_Ias  := IT.singly_Indexed.Interpolate
          (An_Instance.The_Indicated_airspeed,Sb_Ias_t'access);
        An_Instance.The_Scaled_Max_As := IT.singly_Indexed.Interpolate
          (An_Instance.The_Max_Operating_Airspeed,Sb_Ias_t'access);

   end Scale_airspeed;


   function scaled_As (An_Instance :in Instance) return float is
   begin
       return An_Instance.The_Scaled_ias;
   end Scaled_as;

   function Scaled_Max_AS
             (An_Instance :in Instance) return float is
   begin
       return An_Instance.The_Scaled_Max_As;
   end Scaled_Max_as;


   function Flimit(VALUE : float;
                   MIN   : float;
                   MAX   : float) return float is

      Answer : float;
   begin

      if VALUE < MIN then      Answer := MIN;
      elsif VALUE > MAX then   Answer := MAX;
      else                     Answer := VALUE;
      end if;
      return Answer;
   end Flimit;

   -- The Standby altitude indicator is driven using a closed
   -- loop process, ie. a feedback from the instrument (Volts argument)
   -- is needed to check that the device followed the commanded altitude.
   -- Problem arises when record playback is running, for in that situation
   -- we only have the recorded feedback and commanded altitude,  and not
   -- the actual ones.

   -- Hence, we are forced to use the following local variable and
   -- drive the instrument in open loop.

   Record_Playback_Command_Ind : Float := 0.0;

   procedure Stdby_Alt_Drive(Volts           :in     Float;
                             Sb_Alt_T        :in out IT.singly_Indexed.Instance;
                             An_Instance     :in out Instance) is
      Delta_Alt   : Float;
      Fine_Alt    : Float;
      Present_Ind : Float;

   begin
      -- look up current standby altimeter indication
      -- by converting feedback volts from altimeter into feet
      Present_Ind := IT.singly_Indexed.Interpolate(volts,Sb_alt_t'access);
      Delta_Alt := Present_Ind - An_Instance.Command_Ind;

      if Jpats_Avionics.Nav_Bool1 then  -- debug, run open loop, ignore feedback.
        An_Instance.Command_Ind := An_Instance.Command_Ind
           + Flimit(An_Instance.The_Indicated_Altitude
                    - An_Instance.Command_Ind,
                    -5.0,5.0);
        Jpats_Avionics.Set_Nav_Float1(An_Instance.Command_Ind); -- plot commanded alt
        Jpats_Avionics.Set_Nav_Float2(Present_Ind);             -- vs. indicated alt
        Jpats_Avionics.Set_Nav_Float3(Delta_Alt);               -- difference of commanded and present alt
        Jpats_Avionics.Set_Nav_Float4(Volts);                   -- voltage from SimIO
      elsif
        not Save_Restore.Replay_In_Progress and
        not Save_Restore.Replay_Trimming and
        abs(Delta_Alt) > 500.0 then

         An_Instance.Command_Ind := Present_Ind;

      elsif
        not Save_Restore.Replay_In_Progress and
        not Save_Restore.Replay_Trimming then

         -- Problems arise if a large jump in altitude occurs,
         -- so command_ind has to be kept in small delta's

         An_Instance.Command_Ind := An_Instance.Command_Ind
           + Flimit(An_Instance.The_Indicated_Altitude
                    - An_Instance.Command_Ind,
                    -5.0,5.0);
         Record_Playback_Command_Ind := An_Instance.The_Indicated_Altitude;

      else
         -- case in record playback mode.

         Record_Playback_Command_Ind := Record_Playback_Command_Ind
           + Flimit(An_Instance.The_Indicated_Altitude
                    - Record_Playback_Command_Ind,
                    -5.0,5.0);

         An_Instance.Command_Ind := Record_Playback_Command_Ind;

      end if;

      Fine_Alt := 0.360 * Float(Integer(An_Instance.Command_Ind)rem 1000);


      An_Instance.Stdby_sin := Sin(Fine_Alt,360.0) * 10.0;
      An_Instance.Stdby_Cos := Cos(Fine_Alt,360.0) * 10.0;

   end Stdby_Alt_Drive;

   function Stdby_sin
             (An_Instance :in Instance) return float is
   begin
       return An_Instance.Stdby_sin;
   end Stdby_sin;

   function Stdby_cos
             (An_Instance :in Instance) return float is
   begin
       return An_Instance.Stdby_cos;
   end Stdby_cos;

end Air_Data_Computer;



