-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                 JPATS T-6A Texan-II Flight Training Device
--
--
--  Engineer:  Keith H. Rehm
--
--  Revision:  (Number and date inserted by Clearcase)
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------

with Log;

package body Obogs_Concentrator_G is

--    State_Lp : State_Type := Off;

   procedure Update
     (An_Instance      : in out Instance;
      Iconst           : in     Float;
      Power_Available  : in     Boolean;
      Bit_In           : in     Boolean;
      Cockpit_Altitude : in     Altitude_Type;
      Air_Available    : in     Boolean;
      Left_Wow_Sw      : in     Boolean)
   is
      Fail_Pub    : constant Boolean := False;
      Fail_Sensor : constant Boolean := False;
      Fail_Ibit   : constant Boolean := False;
      Fail_Mbit_1 : constant Boolean := False;
      Fail_Mbit_2 : constant Boolean := False;

      Is_Powered : Boolean;
      Is_Powered_Last_Pass : Boolean;

      State : State_Type renames An_Instance.State;
      State_Lp : State_Type renames An_Instance.State_Lp;

      BIT_In_Last_Pass : Boolean renames An_Instance.BIT_In_Last_Pass;

      PUB_Test_Clock       : Float renames An_Instance.PUB_Test_Clock;
      O2_Sensor_Test_Clock : Float renames An_Instance.O2_Sensor_Test_Clock;
      BIT_In_Clock         : Float renames An_Instance.BIT_In_Clock;
      Ibit_Test_Clock      : Float renames An_Instance.Ibit_Test_Clock;
      Mbit_Test_1_Clock    : Float renames An_Instance.Mbit_Test_1_Clock;
      Mbit_Test_2_Clock    : Float renames An_Instance.Mbit_Test_2_Clock;

      Bit_Out              : Boolean renames An_Instance.Bit_Out;
      O2_Sel_Out           : Boolean renames An_Instance.O2_Sel_Out;

      Obogs_Fail_Warn : Boolean renames An_Instance.Obogs_Fail_Warn;

      PUB_Test_Status       : Boolean renames An_Instance.PUB_Test_Status;
      O2_Sensor_Test_Status : Boolean renames An_Instance.O2_Sensor_Test_Status;
      Ibit_Test_Status      : Boolean renames An_Instance.Ibit_Test_Status;

      Mbit_Test_1_Status : Boolean renames An_Instance.Mbit_Test_1_Status;
      Mbit_Test_2_Status : Boolean renames An_Instance.Mbit_Test_2_Status;

      Load : Current_Type renames An_Instance.Load;

      Ckpt_Alt : Float := 0.0;
      -- look at bit during bit and following failed bit
      --   type State_Type is (Off,Power_Up,Normal,Ibit,Mbit);
   begin

      Is_Powered := Power_Available;
      if Is_Powered then Load := 3.0; else Load := 0.0; end if;
      Bit_Out    := Is_Powered;
      O2_Sel_Out := Is_Powered;

      case State is

         when Normal =>

            Ibit_Test_Clock := BIT_In_Clock;
            Mbit_Test_1_Clock := BIT_In_Clock;

            if not BIT_In and BIT_In_Last_Pass and BIT_In_Clock < 5.0 then
               State := Ibit;
            elsif not BIT_In and BIT_In_Last_Pass and BIT_In_Clock >= 20.0 and Left_Wow_Sw then
               State := Mbit;
            end if;

            if BIT_In then
               BIT_In_Clock := BIT_In_Clock + Iconst;
            else
               BIT_In_Clock := 0.0;
            end if;

            if not Is_Powered then State := Off; end if;

         when Off =>

            PUB_Test_Clock := 0.0;
            O2_Sensor_Test_Clock := 0.0;
            if Is_Powered then State := Power_Up; end if;

         when Power_Up =>

            PUB_Test_Clock := PUB_Test_Clock + Iconst;

            PUB_Test_Status := PUB_Test_Clock <= 2.0 or not Fail_Pub; -- pwr up BIT

            if PUB_Test_Status and Air_Available then
               O2_Sensor_Test_Clock := O2_Sensor_Test_Clock + Iconst;
               O2_Sensor_Test_Status := O2_Sensor_Test_Clock <= 180.0 or not Fail_Sensor; -- o2 sensor accuracy BIT
               if O2_Sensor_Test_Clock > 180.0 then -- normal mode
                    State := Normal;
               end if;
            elsif not PUB_Test_Status and not BIT_In and BIT_In_Last_Pass then
               PUB_Test_Clock := 0.0;
            end if;

            if not Is_Powered then
               PUB_Test_Status := True;
               O2_Sensor_Test_Status := True;
               State := Off;
            end if;

         when Ibit =>

            Ibit_Test_Clock := Ibit_Test_Clock + Iconst;

            if Ibit_Test_Clock < 20.0 then
               Ibit_Test_Status := True;
            elsif Ibit_Test_Clock > 20.0 and Ibit_Test_Clock <= 120.0 then
               Ibit_Test_Status := False;
            elsif Ibit_Test_Clock > 120.0 and not Ibit_Test_Status then
               Ibit_Test_Status := not Fail_Ibit;
            elsif Ibit_Test_Clock > 120.0 and Ibit_Test_Status then
               State := Normal;
            end if;

            if not Ibit_Test_Status and not BIT_In and BIT_In_Last_Pass and Ibit_Test_Clock > 120.0 then
               Ibit_Test_Clock := 0.0;
            end if;

            if not Is_Powered then
               Ibit_Test_Status := True;
               Ibit_Test_Clock := 0.0;
               State := Off;
            end if;

         when Mbit =>

            Mbit_Test_1_Clock := Mbit_Test_1_Clock + Iconst;

            if Mbit_Test_1_Clock < 20.0 then
               Mbit_Test_1_Status := True;
            elsif Mbit_Test_1_Clock > 20.0 and Mbit_Test_1_Clock <= 180.0 then
               Mbit_Test_1_Status := False;
            elsif Mbit_Test_1_Clock > 180.0 and not Mbit_Test_1_Status then
               Mbit_Test_1_Status := not Fail_Mbit_1;
            end if;

            if not Mbit_Test_1_Status and not BIT_In and BIT_In_Last_Pass and Mbit_Test_1_Clock > 180.0 then
               Mbit_Test_1_Clock := 0.0;
            end if;

            if Mbit_Test_1_Status and Mbit_Test_1_Clock > 180.0 then

               Mbit_Test_2_Clock := Mbit_Test_2_Clock + Iconst;

               if Mbit_Test_2_Clock < 120.0 then
                  Mbit_Test_2_Status := True;
               elsif Mbit_Test_2_Clock > 120.0 then
                  Mbit_Test_2_Status := not Fail_Mbit_2;
               end if;

               if not Mbit_Test_2_Status and not BIT_In and BIT_In_Last_Pass and Mbit_Test_2_Clock > 120.0 then
                  Mbit_Test_1_Clock := 0.0;
               end if;

               if Mbit_Test_2_Status and Mbit_Test_2_Clock > 120.0 then
                  State := Normal;
               end if;

            end if;

            if not Is_Powered then
               Mbit_Test_1_Status := True;
               Mbit_Test_2_Status := True;
               Mbit_Test_1_Clock := 0.0;
               Mbit_Test_2_Clock := 0.0;
               State := Off;
            end if;

      end case;



      Obogs_Fail_Warn := (not PUB_Test_Status or
                          not O2_Sensor_Test_Status or
                          not Ibit_Test_Status or
                          not Mbit_Test_1_Status or
                          not Mbit_Test_2_Status or
                          State = Off);

      Is_Powered_Last_Pass := Is_Powered;
      --|set pressure.
      if Cockpit_Altitude < 0.0 then
         Ckpt_Alt := 0.0;
      elsif Cockpit_Altitude > 3000.0 then
         Ckpt_Alt := 3000.0;
      else
         Ckpt_Alt := Float(Cockpit_Altitude);
      end if;

      An_Instance.Oxygen_Concentration :=
        Oxygen_Concentration_Type'Min (Oxygen_Concentration_Type'Last,
                                       Oxygen_Concentration_Type'Max (Oxygen_Concentration_Type'First,
                                                                      Oxygen_Concentration_Type((Ckpt_Alt/3000.0)*100.0)));


      BIT_In_Last_Pass := Bit_In;


 --       if State_Lp /= State then
 --          Log.Report("State = " & State_Type'Image(State));
 --       end if;
      State_Lp := State;


   exception
      when others =>
         Log.Report(Package_Name & "(.Update()");
         raise;
   end Update;

   procedure Initialize
     (An_Instance : in out Instance)
   is
   begin
      An_Instance.Is_Powered           := False;
      An_Instance.Is_Powered_Last_Pass := False;
      An_Instance.PUB_Test_Clock       := 0.0;
      An_Instance.Oxygen_Concentration := 0.0;
      An_Instance.Pressure             := 0.0;
      An_Instance.PUB_Test_Status      := False;
      An_Instance.Obogs_Fail_Warn      := False;
      An_Instance.State                := Off;

      An_Instance.O2_Sensor_Test_Clock := 0.0;
      An_Instance.BIT_In_Clock         := 0.0;
      An_Instance.Ibit_Test_Clock      := 0.0;
      An_Instance.Mbit_Test_1_Clock    := 0.0;
      An_Instance.Mbit_Test_2_Clock    := 0.0;

      An_Instance.BIT_In_Last_Pass := False;

      An_Instance.PUB_Test_Status       := True;
      An_Instance.O2_Sensor_Test_Status := True;
      An_Instance.Ibit_Test_Status      := True;
      An_Instance.Mbit_Test_1_Status    := True;
      An_Instance.Mbit_Test_2_Status    := True;

      An_Instance.Bit_Out    := False;
      An_Instance.O2_Sel_Out := False;

      An_Instance.Load := 0.0;
   exception
      when others =>
         Log.Report(Package_Name & "(.Initialize()");
         raise;
   end Initialize;

   function Oxygen_Concentration
     (An_Instance : in Instance)
     return Oxygen_Concentration_Type
   is
   begin
      return An_Instance.Oxygen_Concentration;
   exception
      when others =>
         Log.Report(Package_Name & "(.Oxygen_Concentration()");
         raise;
   end Oxygen_Concentration;

   function Pressure
     (An_Instance : in Instance)
     return Pressure_Type
   is
   begin
      return An_Instance.Pressure;
   exception
      when others =>
         Log.Report(Package_Name & "(.Pressure()");
         raise;
   end Pressure;

   function Obogs_Fail_Warn
     (An_Instance : in Instance)
     return Boolean
   is
   begin
      return An_Instance.Obogs_Fail_Warn;
   exception
      when others =>
         Log.Report(Package_Name & "(.Obogs_Fail_Warn()");
         raise;
   end Obogs_Fail_Warn;

   function Bit_Out
     (An_Instance : in Instance)
     return Boolean
   is
   begin
      return An_Instance.Bit_Out;
   exception
      when others =>
         Log.Report(Package_Name & "(.Bit_Out()");
         raise;
   end Bit_Out;

   function O2_Sel_Out
     (An_Instance : in Instance)
     return Boolean
   is
   begin
      return An_Instance.O2_Sel_Out;
   exception
      when others =>
         Log.Report(Package_Name & "(.O2_Sel_Out()");
         raise;
   end O2_Sel_Out;

   function Load
     (An_Instance : in Instance)
     return Current_Type
   is
   begin
     return An_Instance.Load;
   exception
      when others =>
         Log.Report(Package_Name & "(.Load()");
         raise;
   end Load;

end Obogs_Concentrator_G;
