-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                 JPATS T-6A Texan-II Flight Training Device
--
--
--  Engineer:  Keith H. Rehm
--
--  Revision:  (Number and date inserted by Clearcase)
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------

with Jpats_Environmental.Container;

with Jpats_Electrical;
with Jpats_Simulated_Aircraft;
with Ada.Characters.Latin_1;
with Jpats_Electrical_Types;
with Length_Types;
with Angle_Types;
with Interpolation_Table.Doubly_Indexed;
with Simulation_Dictionary;
with Log;
with Ada.Numerics.Elementary_Functions; use Ada.Numerics.Elementary_Functions;

--with Scheduler.Executive;


package body Jpats_Environmental.Seat_Controller is

   Velocity_Bank_Angle_Table  : aliased Interpolation_Table.Doubly_Indexed.Instance;
   Velocity_Sink_Rate_Table   : aliased Interpolation_Table.Doubly_Indexed.Instance;
   Velocity_Dive_Angle_Table  : aliased Interpolation_Table.Doubly_Indexed.Instance;

   procedure Read_Tables
   is
   begin
      Interpolation_Table.Read
        (File_Name => Simulation_Dictionary.Lookup ("Environmental_Dir") & "ejvelbnk.ito",
         Table     => Velocity_Bank_Angle_Table);
      Interpolation_Table.Read
        (File_Name => Simulation_Dictionary.Lookup ("Environmental_Dir") & "ejvelsnk.ito",
         Table     => Velocity_Sink_Rate_Table);
      Interpolation_Table.Read
        (File_Name => Simulation_Dictionary.Lookup ("Environmental_Dir") & "ejveldiv.ito",
         Table     => Velocity_Dive_Angle_Table);
   exception
      when others =>
         Log.Report("Jpats_Environmental.Seat_Controller.Read_Tables()");
         raise;
   end Read_Tables;


   procedure Initialize
   is
      package Ctnr renames Container;
      package Seat renames Ctnr.Ejection_Seat;
      The_Seat : Seat.Instance renames Ctnr.This_Subsystem.The_Ejection_Seat;
   begin
      Seat.Initialize (The_Seat);
      Read_Tables;
   exception
      when others =>
         Log.Report("Jpats_Environmental.Seat_Controller.Initialize()");
         raise;
   end Initialize;

   -- using this ejection_seat_last_pass variable to find when
   -- the pilot has pulled the ejection pin to eject.
   Eject_Handle_Last_Pass : Boolean renames Container.This_Subsystem.SC_Eject_Handle_Last_Pass;
   Ejection_Status        : Boolean renames Container.This_Subsystem.SC_Ejection_Status;
   Pilot_Is_Ejected       : Boolean renames Container.This_Subsystem.SC_Pilot_Is_Ejected;
   Pilot_Is_Ejected_Last_Pass : Boolean renames Container.This_Subsystem.SC_Pilot_Is_Ejected_Last_Pass;

   Freeze_Request : Boolean renames Container.This_Subsystem.SC_Freeze_Request;

   function Request_Freeze
     return Boolean
   is
   begin
      return Freeze_Request;
   end Request_Freeze;

   procedure Update
     (Iconst : in Float)
   is

      package Ctnr   renames Container;
      package Seat   renames Ctnr.Ejection_Seat;
      package Ele    renames Jpats_Electrical;
      package Ele_T  renames Jpats_Electrical_Types;
      package Double renames Interpolation_Table.Doubly_Indexed;

      Io  : Ctnr.Io_Interface_Instance renames Ctnr.This_Io_Interface;
      Ios : Ctnr.Ios_Interface_Instance renames Ctnr.This_Ios_Interface;
      Host : Ctnr.Host_Interface_Instance renames Ctnr.This_Host_Interface;

      The_Seat : Seat.Instance renames Ctnr.This_Subsystem.The_Ejection_Seat;

      Altitude   : constant Length_Types.Feet := -1.0 * Jpats_Simulated_Aircraft.Get_Aircraft_Height_Above_Local_Terrain;
      Velocity   : constant Float := Float (Length_Types.Feet_Per_Sec_To_Knots (Jpats_Simulated_Aircraft.Get_True_Airspeed));
      Dive_Angle : constant Float := -1.0 * Float (Angle_Types.Radians_To_Degrees (Jpats_Simulated_Aircraft.Get_Pitch_Angle));
      Bank_Angle : constant Float := abs(Float (Angle_Types.Radians_To_Degrees (Jpats_Simulated_Aircraft.Get_Roll_Angle)));
      Sink_Rate  : constant Float := -1.0 * Float (Jpats_Simulated_Aircraft.Get_Rate_of_Climb);
      Pilot_Ejected : constant Boolean := Io.Eject_Handle and not Eject_Handle_Last_Pass;
      Eject_Reset   : constant Boolean := Ios.Eject_Reset or Host.Ejection_Reset_Command;
      Envelope_Height : Length_Types.Feet := 0.0;


      CRLF : String := Ada.Characters.Latin_1.CR & Ada.Characters.Latin_1.LF;

   begin

      if jpats_Simulated_Aircraft.Get_flight_Freeze and Freeze_Request then
         Freeze_Request := False;
      end if;

      --------------
      -- EJECTION --
      --------------

      if Pilot_Ejected then

         Seat.Eject(The_Seat);

         Pilot_Is_Ejected := True;

         Envelope_Height :=
           Length_Types.Feet (Double.Interpolate (Velocity, Bank_Angle, Velocity_Bank_Angle_Table'Access) +
                              Double.Interpolate (Velocity, Sink_Rate, Velocity_Sink_Rate_Table'Access)   +
                              Double.Interpolate (Velocity, Dive_Angle, Velocity_Dive_Angle_Table'Access));

         Ejection_Status := Altitude >= Envelope_Height;

         IOS.Ejection_Velocity        := Velocity;
         IOS.Ejection_Bank_Angle      := Bank_Angle;
         IOS.Ejection_Sink_Rate       := Sink_Rate;
         IOS.Ejection_Dive_Angle      := Dive_Angle;
         IOS.Ejection_Envelope_Height := Float (Envelope_Height);
         IOS.Ejection_Altitude        := Float (Altitude);


         Ios.Ejection_Status := Ejection_Status;

         Log.Report("    velocity   = " & Float'Image(Velocity) & CRLF &
                    "    bank angle = " & Float'Image(Bank_Angle) & CRLF &
                    "    sink rate  = " & Float'Image(Sink_Rate) & CRLF &
                    "    dive angle = " & Float'Image(Dive_Angle) & CRLF &
                    "    envelope height = " & Float'Image(Float(Envelope_Height)) & CRLF &
                    "    altitude        = " & Float'Image(Float(Altitude)) & CRLF);

      else

         if Eject_Reset and not Io.Eject_Handle then
            Seat.Reset(The_Seat);
            Pilot_Is_Ejected := False;
            Ios.Eject_Reset := False;
            Host.Ejection_Reset_Command := False;
         end if;

      end if;


--       if Pilot_Is_Ejected and not Pilot_Is_Ejected_Last_Pass then
-- --         Log.Report("altitude = " & Float'Image(Altitude));
-- --         Log.Report("envelope height = " & Float'Image(Envelope_Height));
--          --         Scheduler.Executive.Freeze;
--          Freeze_Request := True;
--       end if;
      if Pilot_Is_Ejected  then
--         Log.Report("altitude = " & Float'Image(Altitude));
--         Log.Report("envelope height = " & Float'Image(Envelope_Height));
         --         Scheduler.Executive.Freeze;
         Freeze_Request := True;
      end if;

      Eject_Handle_Last_Pass := Io.Eject_Handle;
      Pilot_Is_Ejected_Last_Pass := Pilot_Is_Ejected;

      Ios.Is_Ejected := Seat.Is_Ejected(The_Seat);

   exception
      when others =>
         Log.Report("Jpats_Environmental.Seat_Controller.Update()");
         raise;
   end Update;


end Jpats_Environmental.Seat_Controller;
