-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                 JPATS T-6A Texan-II Flight Training Device
--
--
--  Engineer:  Keith H. Rehm
--
--  Revision:  (Number and date inserted by Clearcase)
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------

with Jpats_Environmental.Container;

with Jpats_Electrical;
with Jpats_Powerplant;
with Jpats_Landing_Gear;
with Jpats_Landing_Gear_Types;
with Jpats_Electrical_Types;
with Force_Types;
with Temperature_Types;
with Log;

package body Jpats_Environmental.Obogs_Controller is

   procedure Initialize
   is
      package Ctnr renames Container;
      package Concentrator renames Ctnr.Concentrator;
      package Regulator    renames Ctnr.Obogs_Regulator;
      package High_Temp_Sw renames Ctnr.Obogs_High_Temperature_Switch;
      package Low_Press_Sw renames Ctnr.Obogs_Low_Pressure_Switch;
      package Sov          renames Ctnr.Obogs_Shutoff_Valve;

      The_Concentrator : Concentrator.Instance renames Ctnr.This_Subsystem.The_Obogs_Concentrator;
      The_Regulator    : Regulator.Instance    renames Ctnr.This_Subsystem.The_Obogs_Regulator;
      The_Low_Press_Sw : Low_Press_Sw.Instance renames Ctnr.This_Subsystem.The_Obogs_Low_Pressure_Switch;
      The_High_Temp_Sw : High_Temp_Sw.Instance renames Ctnr.This_Subsystem.The_Obogs_High_Temperature_Switch;
      The_Sov          : Sov.Instance          renames Ctnr.This_Subsystem.The_Obogs_Shutoff_Valve;
   begin
      Concentrator.Initialize (The_Concentrator);
      Regulator.Initialize (The_Regulator);
      High_Temp_Sw.Initialize (The_High_Temp_Sw);
      Low_Press_Sw.Initialize (The_Low_Press_Sw);
      Sov.Initialize (The_Sov);
   exception
      when others =>
         Log.Report("Jpats_Environmental.Obogs_Controller.Initialize()");
         raise;
   end Initialize;

   Duct_Temp : Temperature_Types.Fahrenheit := 0.0;

   procedure Update
     (Iconst : in Float)
   is
      package Ctnr         renames Container;
      package Ele          renames Jpats_Electrical;
      package Concentrator renames Ctnr.Concentrator;
      package Regulator    renames Ctnr.Obogs_Regulator;
      package High_Temp_Sw renames Ctnr.Obogs_High_Temperature_Switch;
      package Low_Press_Sw renames Ctnr.Obogs_Low_Pressure_Switch;
      package Gear         renames Jpats_Landing_Gear;
      package Gear_T       renames Jpats_Landing_Gear_Types;
      package Cockpit      renames Ctnr.Cockpit_Pressure_Vessel;
      package Sov          renames Ctnr.Obogs_Shutoff_Valve;
      package Temp         renames Temperature_Types;

      The_Cockpit      : Cockpit.Instance      renames Ctnr.This_Subsystem.The_Cockpit_Pressure_Vessel;
      The_Concentrator : Concentrator.Instance renames Ctnr.This_Subsystem.The_Obogs_Concentrator;
      The_Regulator    : Regulator.Instance    renames Ctnr.This_Subsystem.The_Obogs_Regulator;
      The_Low_Press_Sw : Low_Press_Sw.Instance renames Ctnr.This_Subsystem.The_Obogs_Low_Pressure_Switch;
      The_High_Temp_Sw : High_Temp_Sw.Instance renames Ctnr.This_Subsystem.The_Obogs_High_Temperature_Switch;
      The_Sov          : Sov.Instance          renames Ctnr.This_Subsystem.The_Obogs_Shutoff_Valve;

      Obogs_Cb : constant Boolean := Ele.Is_Powered (Jpats_Electrical_Types.Obogs_Cb);
      Batt_Rly : constant Boolean := Ele.Is_Closed (Jpats_Electrical_Types.Battery_Relay);

      Io : Ctnr.Io_Interface_Instance renames Ctnr.This_Io_Interface;
      Ios : Ctnr.Ios_Interface_Instance renames Ctnr.This_Ios_Interface;

      Press_Lever_Emerg : constant Boolean := Io.Emer_Oxygen_Sw;
      Press_Lever_Test_Mask : constant Boolean := Io.Test_Oxygen_Sw;
      Press_Lever_Normal : constant Boolean := not Press_Lever_Emerg and not Press_Lever_Test_Mask;

      Concentration_Lever_Max : constant Boolean := Io.Oxygen_Max_Sw;
      Concentration_Lever_Normal : constant Boolean := not Io.Oxygen_Max_Sw;

      Supply_Lever_On : constant Boolean := Io.Oxygen_Supply_On_Sw;
      Supply_Lever_Off : constant Boolean := not Io.Oxygen_Supply_On_Sw;

      Bit_Sw_In : constant Boolean := Io.Bit_Sw;

      Duct_Press : Force_Types.Psi;

   begin

      ----------------------------------
      -- OBOGS MASTER SWITCH REPEATER --
      ----------------------------------

      if Io.Emer_Oxygen_Sw then
         Ios.Obogs_Sw_State := 2;
      elsif Io.Test_Oxygen_Sw then
         Ios.Obogs_Sw_State := 1;
      else
         Ios.Obogs_Sw_State := 0;
      end if;


      ----------------------------
      -- OXYGEN SWITCH REPEATER --
      ----------------------------

      if Io.Oxygen_Max_Sw then
         Ios.Oxygen_Sw_State := 1;
      else
         Ios.Oxygen_Sw_State := 0;
      end if;

      ----------------------------
      -- SUPPLY SWITCH REPEATER --
      ----------------------------

      if Io.Oxygen_Supply_On_Sw then
         Ios.Supply_Sw_State := 1;
      else
         Ios.Supply_Sw_State := 0;
      end if;


      ------------------------------------
      -- OBOGS DUCT LOW PRESSURE SWITCH --
      ------------------------------------
      -- thinking better estimate from RAC here...
      if Ios.Obogs_Fail_Malf then
         if Jpats_Powerplant.Power_Control_Lever_Angle < 30.0 then
            Duct_Press := 4.0;
         else
            Duct_Press := 40.0;
         end if;
      else -- 40%N1 P&W 100%=37468.0
         if Jpats_Powerplant.Gas_Generator_Speed_Sensor > 14987.2 then
            Duct_Press := 40.0;
         else
            Duct_Press := 4.0;
         end if;
--          Duct_Press := Jpats_Powerplant.Bleed_Pressure;
      end if;

      Low_Press_Sw.Update
        (An_Instance => The_Low_Press_Sw,
         Quantity_In => Duct_Press);


      ----------------------------------------
      -- OBOGS DUCT HIGH TEMPERATURE SWITCH --
      ----------------------------------------

      if Ios.Obogs_Heat_Exch_Malf then
--         Duct_Temp := Temp.Celsius_To_Fahrenheit (Jpats_Powerplant.Bleed_Temperature);
         if Jpats_Powerplant.Gas_Generator_Speed_Sensor > 1000.0 then
            if Supply_Lever_On then
               Duct_Temp := 350.0;
            elsif Duct_Temp > 70.0 then
               Duct_Temp := Duct_Temp - (30.0 * Iconst);
            end if;
         else
            Duct_Temp := 70.0;
         end if;
      else
--         Duct_Temp := Temp.Celsius_To_Fahrenheit (Jpats_Powerplant.Bleed_Temperature) * 0.05;
         Duct_Temp := 70.0;
      end if;

      High_Temp_Sw.Update
        (An_Instance => The_High_Temp_Sw,
         Quantity_In => Duct_Temp);

      -------------------------------
      -- OBOGS SYSTEM CONCENTRATOR --
      -------------------------------

      Concentrator.Update
        (An_Instance      => The_Concentrator,
         Iconst           => Iconst,
         Power_Available  => Regulator.Power_Out (The_Regulator),
         Bit_In           => Regulator.Bit_Out (The_Regulator),
         Cockpit_Altitude => Cockpit.Pressure_Altitude (The_Cockpit),
         Air_Available    => Low_Press_Sw.Is_Open (The_Low_Press_Sw),
         Left_Wow_Sw      => Gear.Is_Closed (Gear_T.Left_Gear_WOW_Switch));


      ----------------------------
      -- OBOGS SYSTEM REGULATOR --
      ----------------------------

      Regulator.Update
        (An_Instance              => The_Regulator,
         Pressure_In              => Concentrator.Pressure (The_Concentrator),
         Power_In                 => Obogs_Cb,
         Supply_Sw_On             => Supply_Lever_On,
         Pressure_Cntrl_Sw_Normal => Press_Lever_Normal,
         Pressure_Cntrl_Sw_Emerg  => Press_Lever_Emerg,
         Concentration_Sw_Normal  => Concentration_Lever_Normal,
         Bit_Sw_In                => Bit_Sw_In,
         Bit_In                   => Concentrator.Bit_Out (The_Concentrator),
         O2_Sel_In                => Concentrator.O2_Sel_Out (The_Concentrator),
         Max_Light_In             => True);--? (from fwd lighting term module)


      --------------------------------
      -- OBOGS SYSTEM SHUTOFF VALVE --
      --------------------------------

      Sov.Update
        (An_Instance  => The_Sov,
         Power_In     => Regulator.Power_Out (The_Regulator));


      --------------
      -- HARDWARE --
      --------------

      Io.Oxygen_Lt := Concentration_Lever_Max and Batt_Rly;

--       if Concentration_Lever_Max then
--          Io.Oxygen_Flow_Meter := True;
--       elsif Io.Anti_Suffocation_Sw then
--          Io.Oxygen_Flow_Meter := True;
--       else
--          Io.Oxygen_Flow_Meter := False;
--       end if;

      if Io.Anti_Suffocation_Sw then
         Io.Oxygen_Flow_Meter := True;
      else
         Io.Oxygen_Flow_Meter := False;
      end if;


   exception
      when others =>
         Log.Report("Jpats_Environmental.Obogs_Controller.Update()");
         raise;
   end Update;

end Jpats_Environmental.Obogs_Controller;
