-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                 JPATS T-6A Texan-II Flight Training Device
--
--
--  Engineer:  Keith H. Rehm
--
--  Revision:  (Number and date inserted by Clearcase)
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------

with Jpats_Environmental.Container;
with Jpats_Electrical;
with Jpats_Atmosphere;
with Jpats_Landing_Gear;
with Jpats_Landing_Gear_Types;
with Jpats_Electrical_Types;
with Length_Types;
with Force_Types;
with Temperature_Types;
with Interpolation_Table.Singly_Indexed;
with Simulation_Dictionary;
with Jpats_Flight_Instruments;
with Jpats_Powerplant;
with Jpats_Simulated_Aircraft;
with Ada.Numerics.Elementary_Functions; use Ada.Numerics.Elementary_Functions;
with Log;

package body Jpats_Environmental.Cockpit_Pressure_Controller is

   Pressure_To_Altitude_Table : aliased Interpolation_Table.Singly_Indexed.Instance;
   Altitude_To_Pressure_Table : aliased Interpolation_Table.Singly_Indexed.Instance;
   Roc_Correction_Table       : aliased Interpolation_Table.Singly_Indexed.Instance;
   Pressure_Ratio_2_Restriction_Table : aliased Interpolation_Table.Singly_Indexed.Instance;


   procedure Read_Tables
   is
   begin
      Interpolation_Table.Read
        (File_Name => Simulation_Dictionary.Lookup ("Environmental_Dir") & "prs2alt.ito",
         Table     => Pressure_To_Altitude_Table);
      Interpolation_Table.Read
        (File_Name => Simulation_Dictionary.Lookup ("Environmental_Dir") & "alt2prs.ito",
         Table     => Altitude_To_Pressure_Table);
      Interpolation_Table.Read
        (File_Name => Simulation_Dictionary.Lookup ("Environmental_Dir") & "roccorct.ito",
         Table     => Roc_Correction_Table);
      Interpolation_Table.Read
        (File_Name => Simulation_Dictionary.Lookup ("Environmental_Dir") & "prtio2n.ito",
         Table     => Pressure_Ratio_2_Restriction_Table);
   exception
      when others =>
         Log.Report("Jpats_Environmental.Cockpit_Pressure_Controller.Read_Tables()");
         raise;
   end Read_Tables;


   Procedure Initialize
   is
      package Ctnr  renames Container;
      package Alt_Sw renames Ctnr.Altitude_Switch;
      package Sol_Dump_Vlv renames Ctnr.Solenoid_Dump_Valve;
      package Control_Vlv renames Ctnr.Control_Valve;
      package Safety_Valve renames Ctnr.Safety_Valve;


      The_Alt_Sw : Alt_Sw.Instance renames Ctnr.This_Subsystem.The_Cockpit_Altitude_Switch;
      The_Sol_Dump_Vlv : Sol_Dump_Vlv.Instance renames Ctnr.This_Subsystem.The_Solenoid_Dump_Valve;
      The_Control_Vlv : Control_Vlv.Instance renames Ctnr.This_Subsystem.The_Control_Valve;
      The_Safety_Valve : Safety_Valve.Instance renames Ctnr.This_Subsystem.The_Safety_Valve;
   begin
      Sol_Dump_Vlv.Initialize (The_Sol_Dump_Vlv);
      Control_Vlv.Initialize (The_Control_Vlv);
      Safety_Valve.Initialize (The_Safety_Valve);
      Alt_Sw.Initialize (The_Alt_Sw);

      Read_Tables;
      Control_Vlv.Read_Tables;

   exception
      when others =>
         Log.Report("Jpats_Environmental.Cockpit_Pressure_Controller.Initialize()");
         raise;
   end Initialize;



-- *     _______________________________________________
-- *                         []
-- *           P1                         P2
-- *     ____________________[]_________________________
-- *                        AREA

-- *     Air flows across restriction are calcultaed using the formula
-- *     below.  The discharge coefficient is assumed to be 0.87 which is
-- *     the coefficient for a butterfly valve.  For reference see Chester
-- *     Smith's 'Calculation of Flow of Air and Diatomic Gases', and
-- *     Andersen's 'The Analysis and Design of Pneumatic Systems'.

-- *     W = (K * C * A * P1 * N)/SQRT(T)

-- *     W = Flow(lbm/min)
-- *     K = 31.82, dimensional constant for Deg R, psia, lbm/min units
-- *     C = Discharge Coefficient for valve or orifice(Assume 0.87)
-- *     A = Throat area through valve or orifice (SQ IN)
-- *     P1 = Upstream air pressure (psia)
-- *     P2 = Downstream air pressure (psia)
-- *     N = Restriction factor from published table based on P1/P2
-- *     T = Inlet temperature (Deg R = Deg F + 460.0)

-- *     Restriction Factor Table (Andersen Appendix A)
   function Flow
     (Upstream_Pressure   : in Float;
      Downstream_Pressure : in Float;
      Throat_Area         : in Float;
      Inlet_Temperature   : in Float)
     return Float
   is
      package Single renames Interpolation_Table.Singly_Indexed;
      N : Float;
      P2 : Float;
   begin
      if Downstream_Pressure < 1.0 then P2 := 1.0; else P2 := Downstream_Pressure; end if;
      N := Single.Interpolate (Upstream_Pressure/P2, Pressure_Ratio_2_Restriction_Table'Access);
      return (31.82 * 0.87 * Throat_Area * Upstream_Pressure * N) / Sqrt(Inlet_Temperature);
   exception
      when others =>
         Log.Report("Jpats_Environmental.Cockpit_Pressure_Controller.Flow()");
         raise;
   end Flow;

-- *     _______________________________________________
-- *                         []
-- *           P1                         P2
-- *     ____________________[]_________________________
-- *                        AREA

-- *     Air flows across restriction are calcultaed using the formula
-- *     below.  The discharge coefficient is assumed to be 0.87 which is
-- *     the coefficient for a butterfly valve.  For reference see Chester
-- *     Smith's 'Calculation of Flow of Air and Diatomic Gases', and
-- *     Andersen's 'The Analysis and Design of Pneumatic Systems'.

-- *     A = (W * SQRT(T))/(K * C * P1 * N)

-- *     A = Throat area through valve or orifice (SQ IN)
-- *     W = Flow(lbm/min)
-- *     K = 31.82, dimensional constant for Deg R, psia, lbm/min units
-- *     C = Discharge Coefficient for valve or orifice(Assume 0.87)
-- *     P1 = Upstream air pressure (psia)
-- *     P2 = Downstream air pressure (psia)
-- *     N = Restriction factor from published table based on P1/P2
-- *     T = Inlet temperature (Deg R = Deg F + 460.0)

-- *     Restriction Factor Table (Andersen Appendix A)
   function Throat_Area
     (Upstream_Pressure   : in Float;
      Downstream_Pressure : in Float;
      Flow_Rate           : in Float;
      Inlet_Temperature   : in Float)
     return Float
   is
      package Single renames Interpolation_Table.Singly_Indexed;
      N : Float;
      P2 : Float;
   begin
      if Downstream_Pressure < 1.0 then P2 := 1.0; else P2 := Downstream_Pressure; end if;
      N := Single.Interpolate (Upstream_Pressure/P2, Pressure_Ratio_2_Restriction_Table'Access);
      if N > 0.0 then
         return (Flow_Rate * Sqrt(Inlet_Temperature))/(31.82 * 0.87 * Upstream_Pressure * N);
      else
         return 3000000.0;
      end if;
   exception
      when others =>
         Log.Report("Jpats_Environmental.Cockpit_Pressure_Controller.Throat_Area()");
         raise;
   end Throat_Area;


   Mass_Init_Counter     : float renames Container.This_Subsystem.CPC_Mass_Init_Counter;
   Cockpit_Roc_Last_Pass : Length_Types.Feet renames Container.This_Subsystem.CPC_Cockpit_Roc_Last_Pass;
   Malfunction_Leak_Area : Length_Types.Sq_Inches renames Container.This_Subsystem.CPC_Malfunction_Leak_Area;
   Differential_Pressure_Last_Pass : Force_Types.Psi renames Container.This_Subsystem.CPC_Differential_Pressure_Last_Pass;


   procedure Update
     (Iconst : in Float)
   is

      package Ctnr         renames Container;
      package Ele          renames Jpats_Electrical;
      package Ele_T        renames Jpats_Electrical_Types;
      package Cockpit      renames Ctnr.Cockpit_Pressure_Vessel;
      package Single       renames Interpolation_Table.Singly_Indexed;
      package Alt_Sw       renames Ctnr.Altitude_Switch;
      package Gear         renames Jpats_Landing_Gear;
      package Gear_T       renames Jpats_Landing_Gear_Types;
      package Sol_Dump_Vlv renames Ctnr.Solenoid_Dump_Valve;
      package Control_Vlv  renames Ctnr.Control_Valve;
      package Safety_Valve renames Ctnr.Safety_Valve;
      package Canopy       renames Ctnr.Canopy;
      package SAC          renames Jpats_Simulated_Aircraft;

      Io  : Ctnr.Io_Interface_Instance  renames Ctnr.This_Io_Interface;
      Ios : Ctnr.Ios_Interface_Instance renames Ctnr.This_Ios_Interface;
      Sound : Ctnr.Sound_Interface_Instance renames Ctnr.This_Sound_Interface;


      The_Cockpit      : Cockpit.Instance      renames Ctnr.This_Subsystem.The_Cockpit_Pressure_Vessel;
      The_Alt_Sw       : Alt_Sw.Instance       renames Ctnr.This_Subsystem.The_Cockpit_Altitude_Switch;
      The_Sol_Dump_Vlv : Sol_Dump_Vlv.Instance renames Ctnr.This_Subsystem.The_Solenoid_Dump_Valve;
      The_Control_Vlv  : Control_Vlv.Instance  renames Ctnr.This_Subsystem.The_Control_Valve;
      The_Safety_Valve : Safety_Valve.Instance renames Ctnr.This_Subsystem.The_Safety_Valve;
      The_Canopy       : Canopy.Instance       renames Ctnr.This_Subsystem.The_Canopy;


      Aircraft_Pressure_Altitude : constant Length_Types.Feet := Jpats_Flight_Instruments.Pressure_Alt;
      Air_Flow_In                : constant Mass_Types.Ppm    := Cockpit.Flow_In (The_Cockpit);
      Ambient_Pressure           : constant Force_Types.Psi   := (1.0/144.0)*Jpats_Atmosphere.Pressure;
      Dump_Valve_Pwr             : constant Boolean           := (Gear.Is_Open (Gear_T.Right_Landing_Gear_A2_WOW_Relay) and
                                                                  Io.Ecs_Press_Norm_Sw and
                                                                  Ele.Is_Powered (Ele_T.Inflow_Sys_Cb));

      Cockpit_Air_Mass      : Mass_Types.Lbm := 0.0;
      Cockpit_Pressure      : Force_Types.Psi;
      Differential_Pressure : Force_Types.Psi;
      Roc                   : Length_Types.Feet_Per_Min;
      Cockpit_Altitude      : Length_Types.Feet;
      Selected_Altitude     : Length_Types.Feet := 8000.0;
      Altitude_Difference   : Length_Types.Feet;
      Roc_Correction        : Float;
      Selected_Roc          : Length_Types.Feet_Per_Min;
      Roc_Error             : Float;
      Natural_Leak_Area     : Length_Types.Sq_Inches := 0.0;
      Total_Leak_Area       : Length_Types.Sq_Inches := 0.0;
      Negative_Relief_Area  : Length_Types.Sq_Inches := 0.0;
      Cockpit_Temperature   : Temperature_Types.Rankine := Temperature_Types.Fahrenheit_To_Rankine(70.0);
      Air_Flow_Out          : Mass_Types.Ppm;
      Total_Flow            : Mass_Types.Ppm;


      Ambient_Pressure_Alt  : Length_Types.Feet := 0.0;

      D_P : Float := 0.0;

   begin

      ------------------------------
      -- PRESSURE SWITCH REPEATER --
      ------------------------------

      if Io.Ecs_Press_Ram_Dump_Sw then
         Ios.Pressure_Sw_State := 2;
      elsif Io.Ecs_Press_Norm_Sw then
         Ios.Pressure_Sw_State := 0;
      else
         Ios.Pressure_Sw_State := 1;
      end if;

      --------------------------------------------
      -- AUTO PRESSURE/COCKPIT ALTITUDE  SWITCH --
      --------------------------------------------

      Alt_Sw.Update (The_Alt_Sw,
                     Length_Types.Feet (Single.Interpolate (Float (Ambient_Pressure),
                                                            Pressure_To_Altitude_Table'Access)));

      -------------------------
      -- SOLENOID DUMP VALVE --
      -------------------------

      Sol_Dump_Vlv.Update (The_Sol_Dump_Vlv, Dump_Valve_Pwr);

      ----------------------
      -- COCKPIT PRESSURE --
      ----------------------
      -- PV = MRT
      -- P = cockpit pressure psi
      -- V = cockpit volume = 160704 in^3 (= 93 ft^3)
      -- R = 640.2 (in-lb/lbm deg R)
      -- T = cockpit temperature (assumed constant 70F=294.26K=529.67R in Rankine)
      -- M = lbm of air

      if Mass_Init_Counter < 4.0 then
         Cockpit_Air_Mass := 0.47392065 * Ambient_Pressure;
         Cockpit.Set_Air_Mass (The_Cockpit,Cockpit_Air_Mass);
         Mass_Init_Counter := Mass_Init_Counter + iconst;
      end if;

      Cockpit_Pressure := Cockpit.Air_Mass (The_Cockpit) * 2.11005783303; -- (RT/V = 2.11005783303)

      Cockpit.Set_Air_Pressure (The_Cockpit, Cockpit_Pressure);

      Ios.Ambient_Pressure_Psi := Ambient_Pressure;

      -----------------------------------
      -- COCKPIT DIFFERENTIAL PRESSURE --
      -----------------------------------

      Differential_Pressure := Cockpit.Air_Pressure(The_Cockpit) - Ambient_Pressure;

      Cockpit.Set_Differential_Pressure (The_Cockpit,Differential_Pressure);

      -- D_P is diff press truncated for ios repeater
      D_P := (Float'Truncation(Float(Differential_Pressure * 10.0))) / 10.0;
      Ios.Cockpit_Differential_Pressure := D_P;

      -------------------------------
      -- COCKPIT PRESSURE ALTITUDE --
      -------------------------------
      
      Cockpit_Altitude := Length_Types.Feet(Single.Interpolate (Float(Cockpit_Pressure), Pressure_To_Altitude_Table'Access));
      Cockpit.Set_Pressure_Altitude (The_Cockpit, Cockpit_Altitude);

      Ios.Cockpit_Pressure_Altitude := Float (Cockpit_Altitude);

      Ambient_Pressure_Alt := Length_Types.Feet(Single.Interpolate (Float(Ambient_Pressure), Pressure_To_Altitude_Table'Access));
      Ios.Ambient_Pressure_Alt := Ambient_Pressure_Alt;

      ---------------------------
      -- COCKPIT RATE OF CLIMB --
      ---------------------------

      Roc := (Cockpit_Altitude - Cockpit.Pressure_Altitude_Last_Pass (The_Cockpit))*(60.0/Iconst);

      Cockpit.Set_Rate_Of_Climb (The_Cockpit,Roc);

      if Cockpit.Rate_Of_Climb (The_Cockpit) > 20000.0 then
         Cockpit.Set_Rate_Of_Climb (The_Cockpit, 20000.0);
      end if;

      if Cockpit.Rate_Of_Climb (The_Cockpit) < -20000.0 then
         Cockpit.Set_Rate_Of_Climb (The_Cockpit, -20000.0);
      end if;

      Ios.Cockpit_Roc := Cockpit.Rate_Of_Climb (The_Cockpit);

      ---------------------------------------
      -- SELECTED COCKPIT ALTITUDE AND ROC --
      --------------------------------------

      -- Selected_Altitude = constant = 8000.0 for jpats (in declarative block)
      if Ambient_Pressure_Alt < 8000.0 then
         Selected_Altitude := Ambient_Pressure_Alt;
      end if;

      Selected_Roc := abs (SAC.Get_Rate_Of_Climb); -- ft per min

      if Selected_Roc < 14700.0 then
         Selected_Roc := 14700.0;
      end if;

      Ios.SAC_Roc           := Selected_Roc;
      Ios.Selected_Altitude := Selected_Altitude;

      -------------------------------------
      -- RATE OF CLIMB CORRECTION FACTOR --
      -------------------------------------

      Altitude_Difference := Selected_Altitude - Cockpit_Altitude;

      Ios.Aircraft_Pressure_Altitude := Aircraft_Pressure_Altitude;

      Roc_Correction := Single.Interpolate (Altitude_Difference, Roc_Correction_Table'Access);

      -------------------------
      -- RATE OF CLIMB ERROR --
      -------------------------

      Roc_Error := (Selected_Roc * Roc_Correction) - Cockpit.Rate_Of_Climb (The_Cockpit);

      ---------------------------
      -- OUTFLOW/CONTROL VALVE --
      ---------------------------

      -- dump valve open => control valve -> open
      if Ios.Over_Pressure_Malf then
         if Sol_Dump_Vlv.Is_Open (The_Sol_Dump_Vlv) then
            Control_Vlv.Open (The_Control_Vlv, Iconst);
         else
            Control_Vlv.Close (The_Control_Vlv, Iconst);
         end if;
      elsif Sol_Dump_Vlv.Is_Open (The_Sol_Dump_Vlv) then
         Control_Vlv.Open (The_Control_Vlv, Iconst);
      else -- normal flight
         Control_Vlv.Update (The_Control_Vlv, Iconst, Roc_Error);
      end if;

      Ios.Control_Valve_Position := Control_Vlv.Position(The_Control_Vlv);
      Ios.Control_Valve_Area     := Control_Vlv.Area(The_Control_Vlv);

      ------------------
      -- SAFETY VALVE --
      ------------------

      if Ios.Over_Pressure_Malf then
         if Differential_Pressure >= 4.3 then
            Safety_Valve.Open (The_Safety_Valve, Iconst);
         else
            Safety_Valve.Close (The_Safety_Valve, Iconst);
         end if;
      elsif Differential_Pressure > 3.65 then -- pneumatic signal from delta pressure valve
         Safety_Valve.Open (The_Safety_Valve, Iconst);
      else
         Safety_Valve.Close (The_Safety_Valve, Iconst);
      end if;

      Ios.Safety_Valve_Position := Safety_Valve.Position(The_Safety_Valve);
      Ios.Safety_Valve_Area     := Safety_Valve.Area(The_Safety_Valve);

      -----------------------------------
      -- COCKPIT PRESSURE VESSEL LEAKS --
      -----------------------------------

      --create leaks to include malfunctions
      Natural_Leak_Area := 0.03;

      if Ios.Rapid_Decompression_Malf then
         Malfunction_Leak_Area := 4.0; -- ? appropriate number
      elsif Ios.Slow_Decompression_Malf then
         Malfunction_Leak_Area := 3.0;
      elsif Jpats_Powerplant.Engine_Failure then
         -- Slightly speed up no bleed air loss of cabin pressure for training emphasis
         Malfunction_Leak_Area := 0.05;
      else
         Malfunction_Leak_Area := 0.0;
      end if;

      if Canopy.Is_Fractured (The_Canopy) then
         Natural_Leak_Area := 4.0;
      end if;

      --------------------------
      -- NEGATIVE RELIEF AREA --
      --------------------------

      if Ambient_Pressure-0.3 > Cockpit_Pressure then
         Negative_Relief_Area := (Ambient_Pressure - Cockpit_Pressure)*50.0;
      end if;

      ---------------------
      -- TOTAL LEAK AREA --
      ---------------------

      --add all leak areas to to get total
      Total_Leak_Area := (Control_Vlv.Area (The_Control_Vlv) +
                          Safety_Valve.Area (The_Safety_Valve) +
                          Natural_Leak_Area +
                          Malfunction_Leak_Area+
                          Negative_Relief_Area);


      Ios.Total_Leak_Area := Total_Leak_Area;

      ----------------------
      -- AIR FLOW BALANCE --
      ----------------------

      if Cockpit_Pressure >= Ambient_Pressure then
         Air_Flow_Out := Flow (Upstream_Pressure   => Cockpit_Pressure,      -- in Float;
                               Downstream_Pressure => Ambient_Pressure,      -- in Float;
                               Throat_Area         => Total_Leak_Area,       -- in Float;
                               Inlet_Temperature   => 529.67);--Cockpit_Temperature);  -- in Float)

      else
         Air_Flow_Out := Flow (Upstream_Pressure   => Ambient_Pressure,      -- in Float;
                               Downstream_Pressure => Cockpit_Pressure,      -- in Float;
                               Throat_Area         => Total_Leak_Area,       -- in Float;
                               Inlet_Temperature   => 529.67);--Cockpit_Temperature);  -- in Float)
         Air_Flow_Out := -1.0*Air_Flow_Out;
      end if;

      Ios.Flow_Into_Cockpit       := Air_Flow_In;
      Ios.Flow_Out_of_Cockpit     := Air_Flow_Out;


      Total_Flow := Air_Flow_In - Air_Flow_Out;

      Ios.Total_Flow_Ppm := Total_Flow;

      ----------------------
      -- COCKPIT AIR MASS --
      ----------------------

      Cockpit_Air_Mass := Cockpit.Air_Mass (The_Cockpit) + (Total_Flow * 0.0166667 * Iconst);

      Cockpit.Set_Air_Mass (The_Cockpit, Cockpit_Air_Mass);

      Ios.Cockpit_Air_Mass := Cockpit_Air_Mass;

      -----------
      -- SOUND --
      -----------

      if Ambient_Pressure_Alt >= 7500.0 then
         Sound.Pressure_System_Running := Ambient_Pressure_Alt / 8000.0;
         if Sound.Pressure_System_Running > 1.0 then Sound.Pressure_System_Running := 1.0; end if;
      else
         Sound.Pressure_System_Running := 0.0;
      end if;

      Sound.Differential_Pressure := Differential_Pressure / 3.9;

      if Differential_Pressure - Differential_Pressure_Last_Pass < -0.5 then
         Sound.Explosive_Decompression := Sound.Differential_Pressure;
      else
         Sound.Explosive_Decompression := 0.0;
      end if;

   exception
      when others =>
         Log.Report("Jpats_Environmental.Cockpit_Pressure_Controller.Update()");
         raise;
   end Update;

end Jpats_Environmental.Cockpit_Pressure_Controller;
