-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                 JPATS T-6A Texan-II Flight Training Device
--
--
--  Engineer:  Keith H. Rehm
--
--  Revision:  (Number and date inserted by Clearcase)
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------

with Jpats_Environmental.Container;

with Jpats_Electrical;
with Jpats_Simulated_Aircraft;
with Jpats_Powerplant;
with Jpats_Atmosphere;
with Force_Types;
with Jpats_Electrical_Types;
with Length_Types;
with Temperature_Types;
with Log;
with Interpolation_Table.Doubly_Indexed;
with Simulation_Dictionary;

package body Jpats_Environmental.Air_Conditioner_Controller is

   Heat_Exchanger_Capability_Table : aliased Interpolation_Table.Doubly_Indexed.Instance;

   procedure Read_Tables
   is
   begin
      Interpolation_Table.Read
        (File_Name => Simulation_Dictionary.Lookup ("Environmental_Dir") & "htexchgr.ito",
         Table     => Heat_Exchanger_Capability_Table);
   exception
      when others =>
         Log.Report("Jpats_Environmental.Air_Conditioner_Controller.Read_Tables()");
         raise;
   end Read_Tables;


   procedure Initialize
   is
      package Ctnr                                          renames Container;
      package Evap_Blwr                                     renames Ctnr.Evaporator_Blower;
      package Condensor_Blwr                                renames Ctnr.Condensor_Blower;
      package Heat_Exchanger_Blwr                           renames Ctnr.Heat_Exchanger_Blower;
      package Ram_Air_Valve                                 renames Ctnr.Ram_Air_Valve;
      package Bypass_Valve                                  renames Ctnr.Heat_Exchange_Bypass_Valve;
      package Bypass_Valve_30_Degrees_From_Full_Cold_Switch renames Ctnr.Bypass_Valve_30_Degrees_From_Full_Cold_Switch;
      package Cockpit_Temp_Controller                       renames Ctnr.Cockpit_Temperature_Controller;
      package Low_Press_Sw                                  renames Ctnr.Ac_Low_Pressure_Switch;
      package Over_Press_Sw                                 renames Ctnr.Ac_Over_Pressure_Switch;
      package Over_Temp_Sw                                  renames Ctnr.Ac_Over_Temp_Sw;

      The_Evap_Blwr               : Evap_Blwr.Instance               renames Ctnr.This_Subsystem.The_Evaporator_Blower;
      The_Condensor_Blwr          : Condensor_Blwr.Instance          renames Ctnr.This_Subsystem.The_Condensor_Blower;
      The_Heat_Exchanger_Blwr     : Heat_Exchanger_Blwr.Instance     renames Ctnr.This_Subsystem.The_Heat_Exchanger_Blower;
      The_Ram_Air_Valve           : Ram_Air_Valve.Instance           renames Ctnr.This_Subsystem.The_Ram_Air_Valve;
      The_Bypass_Valve            : Bypass_Valve.Instance            renames Ctnr.This_Subsystem.The_Heat_Exchange_Bypass_Valve;
      The_Cockpit_Temp_Controller : Cockpit_Temp_Controller.Instance renames Ctnr.This_Subsystem.The_Cockpit_Temperature_Controller;
      The_Low_Press_Sw            : Low_Press_Sw.Instance            renames Ctnr.This_Subsystem.The_Ac_Low_Pressure_Switch;
      The_Over_Press_Sw           : Over_Press_Sw.Instance           renames Ctnr.This_Subsystem.The_Ac_Over_Pressure_Switch;
      The_Over_Temp_Sw            : Over_Temp_Sw.Instance            renames Ctnr.This_Subsystem.The_Ac_Over_Temp_Switch;

      The_Bypass_Valve_30_Degrees_From_Full_Cold_Switch :
        Bypass_Valve_30_Degrees_From_Full_Cold_Switch.Instance renames
        Ctnr.This_Subsystem.The_Bypass_Valve_30_Degrees_From_Full_Cold_Switch;
   begin
      Evap_Blwr.Initialize (The_Evap_Blwr);
      Condensor_Blwr.Initialize (The_Condensor_Blwr);
      Heat_Exchanger_Blwr.Initialize (The_Heat_Exchanger_Blwr);
      Ram_Air_Valve.Initialize (The_Ram_Air_Valve);
      Bypass_Valve.Initialize (The_Bypass_Valve);
      Bypass_Valve_30_Degrees_From_Full_Cold_Switch.Initialize (The_Bypass_Valve_30_Degrees_From_Full_Cold_Switch);
      Cockpit_Temp_Controller.Initialize (The_Cockpit_Temp_Controller);
      Low_Press_Sw.Initialize (The_Low_Press_Sw);
      Over_Press_Sw.Initialize (The_Over_Press_Sw);
      Over_Temp_Sw.Initialize (The_Over_Temp_Sw);

      Read_Tables;
   exception
      when others =>
         Log.Report("Jpats_Environmental.Air_Conditioner_Controller.Initialize()");
         raise;
   end Initialize;

   Target_Temp : Temperature_Types.Fahrenheit renames Container.This_Subsystem.ACC_Target_Temp;
   Duct_Temp_Timer : Float renames Container.This_Subsystem.ACC_Duct_Temp_Timer;
   Duct_Temp_Target :Temperature_Types.Fahrenheit renames Container.This_Subsystem.ACC_Duct_Temp_Target;
   Duct_Temp_First_Pass : Boolean;

   procedure Update
     (Iconst : in Float)
   is
      package Ctnr                                          renames Container;
      package Ele                                           renames Jpats_Electrical;
      package Ele_T                                         renames Jpats_Electrical_Types;
      package Evap_Blwr                                     renames Ctnr.Evaporator_Blower;
      package Condensor_Blwr                                renames Ctnr.Condensor_Blower;
      package Heat_Exchanger_Blwr                           renames Ctnr.Heat_Exchanger_Blower;
      package Relays                                        renames Ctnr.Relay_Collection;
      package Ram_Air_Valve                                 renames Ctnr.Ram_Air_Valve;
      package Env_T                                         renames Jpats_Environmental_Types;
      package Bypass_Valve                                  renames Ctnr.Heat_Exchange_Bypass_Valve;
      package Bypass_Valve_30_Degrees_From_Full_Cold_Switch renames Ctnr.Bypass_Valve_30_Degrees_From_Full_Cold_Switch;
      package Cockpit_Temp_Controller                       renames Ctnr.Cockpit_Temperature_Controller;
      package Cockpit                                       renames Ctnr.Cockpit_Pressure_Vessel;
      package Low_Press_Sw                                  renames Ctnr.Ac_Low_Pressure_Switch;
      package Over_Press_Sw                                 renames Ctnr.Ac_Over_Pressure_Switch;
      package Over_Temp_Sw                                  renames Ctnr.Ac_Over_Temp_Sw;
      package Pwrplnt                                       renames Jpats_Powerplant;
      package Double                                        renames Interpolation_Table.Doubly_Indexed;
      package Temp                                          renames Temperature_Types;
      package Atmos                                         renames Jpats_Atmosphere;
      package Ac_Compressor                                 renames Ctnr.Ac_Compressor;
      package Thermal_Time_Delay                            renames Ctnr.Thermal_Time_Delay;
      package Oat_Switch                                    renames Ctnr.Oat_Switch;


      The_Relays                  : Relays.Instance                  renames Ctnr.This_Subsystem.The_Relays;
      The_Evap_Blwr               : Evap_Blwr.Instance               renames Ctnr.This_Subsystem.The_Evaporator_Blower;
      The_Condensor_Blwr          : Condensor_Blwr.Instance          renames Ctnr.This_Subsystem.The_Condensor_Blower;
      The_Heat_Exchanger_Blwr     : Heat_Exchanger_Blwr.Instance     renames Ctnr.This_Subsystem.The_Heat_Exchanger_Blower;
      The_Ram_Air_Valve           : Ram_Air_Valve.Instance           renames Ctnr.This_Subsystem.The_Ram_Air_Valve;
      The_Bypass_Valve            : Bypass_Valve.Instance            renames Ctnr.This_Subsystem.The_Heat_Exchange_Bypass_Valve;
      The_Cockpit_Temp_Controller : Cockpit_Temp_Controller.Instance renames Ctnr.This_Subsystem.The_Cockpit_Temperature_Controller;
      The_Cockpit                 : Cockpit.Instance                 renames Ctnr.This_Subsystem.The_Cockpit_Pressure_Vessel;
      The_Low_Press_Sw            : Low_Press_Sw.Instance            renames Ctnr.This_Subsystem.The_Ac_Low_Pressure_Switch;
      The_Over_Press_Sw           : Over_Press_Sw.Instance           renames Ctnr.This_Subsystem.The_Ac_Over_Pressure_Switch;
      The_Over_Temp_Sw            : Over_Temp_Sw.Instance            renames Ctnr.This_Subsystem.The_Ac_Over_Temp_Switch;
      The_Ac_Compressor           : Ac_Compressor.Instance           renames Ctnr.This_Subsystem.The_Compressor;
      The_Thermal_Time_Delay      : Thermal_Time_Delay.Instance      renames Ctnr.This_Subsystem.The_Thermal_Time_Delay;
      The_Oat_Switch              : Oat_Switch.Instance              renames Ctnr.This_Subsystem.The_Oat_Switch;

      The_Bypass_Valve_30_Degrees_From_Full_Cold_Switch :
        Bypass_Valve_30_Degrees_From_Full_Cold_Switch.Instance renames
        Ctnr.This_Subsystem.The_Bypass_Valve_30_Degrees_From_Full_Cold_Switch;

      Io : Ctnr.Io_Interface_Instance renames Ctnr.This_Io_Interface;
      Ios : Ctnr.Ios_Interface_Instance renames Ctnr.This_Ios_Interface;
      Sound : Ctnr.Sound_Interface_Instance renames Ctnr.This_Sound_Interface;

      Evap_Blwr_Cb            : constant Boolean := Ele.Is_Powered (Ele_T.Evap_Blower_Cb);
      Condensor_Blwr_Cb       : constant Boolean := Ele.Is_Powered (Ele_T.Cond_Blower_Cb);
      Heat_Exchanger_Blwr_Cb  : constant Boolean := Ele.Is_Powered (Ele_T.Heat_Exch_Blower_Cb);
      Ram_Air_Valve_Cb        : constant Boolean := Ele.Is_Powered (Ele_T.Ram_Air_Valve_Cb);
      Ckpt_Temp_Cb            : constant Boolean := Ele.Is_Powered (Ele_T.Ckpt_Temp_Cb);
      Condensor_Blwr_Rly      : constant Boolean := Relays.Is_Closed (The_Relays,Env_T.Condensor_Blower_Relay);
      Heat_Exchanger_Blwr_Rly : constant Boolean := Relays.Is_Closed (The_Relays,Env_T.Heat_Exchanger_Blower_Relay);
      Auto_Press_Rly          : constant Boolean := Relays.Is_Closed (The_Relays,Env_T.Auto_Pressure_Relay);
      Ram_Air_Valve_Rly       : constant Boolean := Relays.Is_Closed (The_Relays,Env_T.Ram_Air_Valve_Relay);

      Ram_Air_Valve_High_In : constant Boolean := ((Io.Ecs_Bleed_Air_Inflow_Hi_Sw and
                                                    not Pwrplnt.Engine_Failure    and
                                                    not Auto_Press_Rly            and
                                                    not Ram_Air_Valve_Rly         and
                                                    Ram_Air_Valve_Cb)                    or
                                                   (Ram_Air_Valve_Rly             and
                                                    Ram_Air_Valve_Cb));

      Ram_Air_Valve_Norm_In : constant Boolean := ((not Io.Ecs_Bleed_Air_Inflow_Hi_Sw   and
                                                    not Io.Ecs_Bleed_Air_Inflow_Off_Sw  and
                                                    not Pwrplnt.Engine_Failure)         and
                                                   not Auto_Press_Rly                   and
                                                   not Ram_Air_Valve_Rly                and
                                                   Ram_Air_Valve_Cb);

      Ram_Air_Valve_Close_In : constant Boolean := (((Io.Ecs_Bleed_Air_Inflow_Off_Sw     or
                                                     Pwrplnt.Engine_Failure)            and
                                                     not Auto_Press_Rly                 and
                                                     not Ram_Air_Valve_Rly              and
                                                     Ram_Air_Valve_Cb)                  or
                                                    (Auto_Press_Rly                     and
                                                     not Ram_Air_Valve_Rly              and
                                                     Ram_Air_Valve_Cb));

      Bypass_Vlv_Cold_Pin : Boolean;
      Bypass_Vlv_Hot_Pin  : Boolean;
      Outside_Air_Temperature : constant Temp.Fahrenheit :=
        Temp.Rankine_To_Fahrenheit (Atmos.Temperature);
      Bleed_Temp : Temperature_Types.Fahrenheit;
      Ac_Line_Press : Force_Types.Psi;
      Duct_Temp : Temperature_Types.Fahrenheit;
      Heat_Exchanger_Capability : Float;
      Auto_Temp_Commanded : Temp.Fahrenheit;
      Bleed_Temp_Aft_Bypass  : Temp.Fahrenheit;
      Cockpit_Temp : Temp.Fahrenheit := Cockpit.Temperature(The_Cockpit);
      Recirulated_Air_Temp : Temp.Fahrenheit;
      Xtemp : Temp.Fahrenheit;
      Temp_Rate : Temp.Fahrenheit;
      Cool_Air_Temp : Temp.Fahrenheit;
      Ac_On_High : Boolean;
      Ac_On_Low : Boolean;
      Bleed_Available : constant Boolean := Cockpit.Bleed_Pressure(The_Cockpit) > 0.0;
      Clutch_Pwr : Boolean;
      Bypass_K : Float;

   begin

      ---------------------------
      -- BLEED AIR FROM ENGINE --
      ---------------------------
      if Bleed_Available then
         Bleed_Temp := Temp.Kelvin_To_Fahrenheit (Pwrplnt.Bleed_Temperature);
      else
         Bleed_Temp := Outside_Air_Temperature;
      end if;

      ------------------------------------
      -- COCKPIT TEMPERATURE CONTROLLER --
      ------------------------------------
      -- auto mode selects temp = [60,90] (F)
      if Io.Ecs_Temp_Cont_Auto_Sw then
         Auto_Temp_Commanded :=  60.0 + 30.0 * ((Io.Ecs_Temp_Cont+10.0)/20.0);
      else
         Auto_Temp_Commanded :=  75.0;
      end if;

      Ios.Auto_Temp_Control_Sw := 60.0 + 30.0 * ((Io.Ecs_Temp_Cont+10.0)/20.0);

      Cockpit_Temp_Controller.Update
        (An_Instance       => The_Cockpit_Temp_Controller,
         Ckpt_Temp         => Cockpit.Temperature (The_Cockpit),
         Temp_Demanded     => Auto_Temp_Commanded,
         Voltage_In        => Ckpt_Temp_Cb,
         Auto_In           => Io.Ecs_Temp_Cont_Auto_Sw and Ckpt_Temp_Cb,
         Man_Cold          => Io.Ecs_Temp_Cont_Man_Cold_Sw,
         Man_Hot           => Io.Ecs_Temp_Cont_Man_Hot_Sw,
         Auto_Ovrheat_Malf => Ios.Auto_Mode_Over_Temp_Malf);

      ----------------------------------
      -- HEAT EXCHANGER BY-PASS VALVE --
      ----------------------------------
      Bypass_Vlv_Cold_Pin := ((Io.Ecs_Temp_Cont_Man_Cold_Sw or
                               Cockpit_Temp_Controller.Decrease (The_Cockpit_Temp_Controller)) and
                              Ckpt_Temp_Cb);

      Bypass_Vlv_Hot_Pin := ((Io.Ecs_Temp_Cont_Man_Hot_Sw or
                              Cockpit_Temp_Controller.Increase (The_Cockpit_Temp_Controller)) and
                             Ckpt_Temp_Cb);

      Bypass_Valve.Update
        (An_Instance     => The_Bypass_Valve,
         Iconst          => Iconst,
         Cold_Pin        => Bypass_Vlv_Cold_Pin,
         Hot_Pin         => Bypass_Vlv_Hot_Pin);

      -------------------------------------------------------------------
      -- HEAT EXCHANGER BY-PASS VALVE 30 DEGREES FROM FULL COLD SWITCH --
      -------------------------------------------------------------------

      Bypass_Valve_30_Degrees_From_Full_Cold_Switch.Update
        (An_instance    => The_Bypass_Valve_30_Degrees_From_Full_Cold_Switch,
         Valve_Position => Bypass_Valve.Position (The_Bypass_Valve));

      ---------------------------
      -- HEAT EXCHANGER BLOWER --
      ---------------------------

      Heat_Exchanger_Blwr.Update
        (An_Instance => The_Heat_Exchanger_Blwr,
         Power_In    => Heat_Exchanger_Blwr_Cb and then Heat_Exchanger_Blwr_Rly);

      -----------------------------------------------------
      -- HEAT EXCHANGER BY-PASS VALVE OUTPUT TEMPERATURE --
      -----------------------------------------------------

      if Heat_Exchanger_Blwr.Is_On (The_Heat_Exchanger_Blwr) then
         Heat_Exchanger_Capability := Double.Interpolate (50.0+Float(JPATS_Simulated_Aircraft.Get_True_Airspeed),
                                                          Float(Jpats_Atmosphere.Temperature),
                                                          Heat_Exchanger_Capability_Table'Access);
      else
         Heat_Exchanger_Capability := Double.Interpolate (Float(JPATS_Simulated_Aircraft.Get_True_Airspeed),
                                                          Float(Jpats_Atmosphere.Temperature),
                                                          Heat_Exchanger_Capability_Table'Access);
      end if;

      Cool_Air_Temp := (Bleed_Temp -
                                (Bleed_Temp *
                                 Heat_Exchanger_Capability *
                                 Bypass_Valve.Position(The_Bypass_Valve)/90.0));

      Bypass_K := (Bypass_Valve.Position(The_Bypass_Valve)/90.0)*0.5;

      Bleed_Temp_Aft_Bypass := Bypass_K*Bleed_Temp + (1.0-Bypass_K)*Cool_Air_Temp;

      ---------------------------------------------
      -- AIR CONDITIONER OVER TEMPERATURE SWITCH --
      ---------------------------------------------

      --  Duct_Temp := Bleed_Temp_Aft_Bypass;

      ---------------------------------------
      -- KLUDGE TO MAKE OVERTEMP MALF WORK --
      ---------------------------------------

      if not Ios.Auto_Mode_Over_Temp_Malf then
         Duct_Temp := Bleed_Temp_Aft_Bypass;
         Duct_Temp_Target := Duct_Temp;
         Duct_Temp_First_Pass := True;
      else

         if JPATS_POWERPLANT.Gas_Generator_Speed_Sensor > 1000.0
           and not (Io.Ecs_Bleed_Air_Inflow_Off_Sw or Pwrplnt.Engine_Failure) then

            if Io.Ecs_Temp_Cont_Auto_Sw or Duct_Temp_First_Pass then
               Duct_Temp_Target  := Duct_Temp_target + 100.0*Iconst;
               if Duct_Temp_Target > 400.0 then
                  Duct_Temp_Target := 400.0;
                  Duct_Temp_First_Pass := False;
               end if;
            elsif Io.Ecs_Temp_Cont_Man_Cold_Sw then
               Duct_Temp_Target := Duct_Temp_Target - 10.0*Iconst;
               if Duct_Temp_Target < Bleed_Temp_Aft_Bypass then
                  Duct_Temp_Target := Bleed_Temp_Aft_Bypass;
               end if;
            end if;

         else
            Duct_Temp_Target := Duct_Temp_Target - 10.0*Iconst;
            if Duct_Temp_Target < Bleed_Temp_Aft_Bypass then
               Duct_Temp_Target := Bleed_Temp_Aft_Bypass;
            end if;
         end if;

         Duct_Temp := Duct_Temp_Target;
      end if;

      Duct_Temp_Timer := Duct_Temp_Timer + Iconst;
      if Duct_Temp_Timer > 10.0 then
         Duct_Temp_Timer := 0.0;
--         Log.Report("Duct_Temp = " & Temperature_Types.Fahrenheit'Image(Duct_Temp));
      end if;

      -----------------------------
      -- Over Temperature Switch --
      -----------------------------

      Over_Temp_Sw.Update
        (An_Instance => The_Over_Temp_Sw,
         Quantity_In => Duct_Temp);

      -----------------------
      -- EVAPORATOR BLOWER --
      -----------------------

      Evap_Blwr.Update
        (An_Instance => The_Evap_Blwr,
         High_In     => Evap_Blwr_Cb and then Io.Ecs_Air_Cond_On_Sw,
         Low_In      => Evap_Blwr_Cb and then not Io.Ecs_Air_Cond_On_Sw);

      ----------------------
      -- CONDENSOR BLOWER --
      ----------------------

      Condensor_Blwr.Update
        (An_Instance => The_Condensor_Blwr,
         Power_In    => Condensor_Blwr_Cb and then Condensor_Blwr_Rly);

      ----------------
      -- OAT SWITCH --
      ----------------

      Oat_Switch.Update
        (An_Instance => The_Oat_Switch,
         Temperature => Outside_Air_Temperature);

      ------------------------
      -- THERMAL TIME DELAY --
      ------------------------

      Thermal_Time_Delay.Update
        (An_Instance => The_Thermal_Time_Delay,
         Value       => 0.0);

      --------------------------------
      -- AIR CONDITIONER COMPRESSOR --
      --------------------------------

      Clutch_Pwr :=  ((Oat_Switch.Is_Closed (The_Oat_Switch) and
                           Thermal_Time_Delay.Is_Closed (The_Thermal_Time_Delay) and
                           Over_Press_Sw.Is_open (The_Over_Press_Sw) and
                           Ele.Is_Powered (Name => Ele_T.Air_Cond_Cb)) and
                          (Relays.Is_Closed(The_Relays,Env_T.Defog_Select_Relay) or
                           Relays.Is_Closed(The_Relays,Env_T.Cool_Command_Relay)));
      Ac_Compressor.Update
        (An_Instance     => The_Ac_Compressor,
         Engage_demanded => Clutch_Pwr,
         Engine_Rpm      => Jpats_Powerplant.Gas_Generator_Speed_Pct);

      ------------------------------------------------------------------
      -- AIR CONDITIONER LOW PRESSURE SWITCH AND OVER PRESSURE SWITCH --
      ------------------------------------------------------------------

      Ac_Line_Press := Pwrplnt.Bleed_Pressure;

      Low_Press_Sw.Update
        (An_Instance => The_Low_Press_Sw,
         Quantity_In => Ac_Line_Press);

      Over_Press_Sw.Update
        (An_Instance => The_Over_Press_Sw,
         Quantity_In => Ac_Line_Press);

      -------------------
      -- RAM AIR VALVE --
      -------------------

      Ram_Air_Valve.Update
        (An_Instance => The_Ram_Air_Valve,
         High_In     => Ram_Air_Valve_High_In,
         Norm_In     => Ram_Air_Valve_Norm_In,
         Close_In    => Ram_Air_Valve_Close_In);

      -----------------------------------
      -- RAM AIR VALVE SWITCH REPEATER --
      -----------------------------------

      if Io.Ecs_Ram_Air_Flow_Hi_Sw then
         Ios.Ram_Air_Flow_Sw_State := 2;
      elsif Io.Ecs_Ram_Air_Flow_Off_Sw then
         Ios.Ram_Air_Flow_Sw_State := 0;
      else
         Ios.Ram_Air_Flow_Sw_State := 1;
      end if;

      -----------------------------------------
      -- COCKPIT PRESSURE VESSEL TEMPERATURE --
      -----------------------------------------

      Ac_On_High := (Ctnr.Ac_Compressor.Is_On(The_Ac_Compressor) and
                     Condensor_Blwr.Is_On (The_Condensor_Blwr) and
                     Evap_Blwr.Is_High (The_Evap_Blwr));

      Ac_On_Low := (Ctnr.Ac_Compressor.Is_On(The_Ac_Compressor) and
                    Condensor_Blwr.Is_On (The_Condensor_Blwr) and
                    Evap_Blwr.Is_Low (The_Evap_Blwr));

      if Ac_On_High then
         Recirulated_Air_Temp := 40.0;
         Temp_Rate            := 0.1*Iconst;
      elsif Ac_On_Low then
         Recirulated_Air_Temp := 50.0;
         Temp_Rate            := 0.67*Iconst;
      else
         Recirulated_Air_Temp := Duct_Temp;
         Temp_Rate            := 0.04*Iconst;
      end if;

      Xtemp := (Duct_Temp + Recirulated_Air_Temp) / 2.0;

      Target_Temp := Target_Temp + (Xtemp - Target_Temp) * 0.5 * Iconst;

      if Target_Temp > Cockpit_Temp then
         Cockpit_Temp := Cockpit_Temp + Temp_Rate;
      elsif Target_Temp < Cockpit_Temp then
         Cockpit_Temp := Cockpit_Temp - Temp_Rate;
      else
         null;
      end if;

      Cockpit.Set_Temperature(The_Cockpit,Cockpit_Temp);

      Ios.Cockpit_Temp_F := Cockpit_Temp;

      -----------
      -- SOUND --
      -----------

      if Ac_Compressor.Is_On (The_Ac_Compressor) then
         Sound.Air_Conditioning_System_Running := 1.0;
      else
         Sound.Air_Conditioning_System_Running := 1.0;
      end if;

      Sound.Windshield_Defrost_On := 0.0;

   exception
      when others =>
         Log.Report("Jpats_Environmental.Air_Conditioner_Controller.Update()");
         raise;
   end Update;

end Jpats_Environmental.Air_Conditioner_Controller;
