-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                      JPATS T-6A Flight Training Device
--
--
--  Engineer: Steven D. Roberts
--            Keith H. Rehm
--
--  Revision:
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------
-- Update Count    : 0
with Arinc_429_Bnr;
with Arinc_429_Types;
with Arinc_429_Discrete;
with Ada.Numerics.Elementary_Functions; use Ada.Numerics.Elementary_Functions;


package body Engine_Data_Manager is


  --|  ENGINE DATA MANAGER (EDM)
  --|  The engine data manager (EDM) is a computer unit which
  --|  monitors engine operating parameters and illuminates the
  --|  appropriate advisory, caution or warning annunciator as
  --|  necessary. The EDM also drives the primary and alternate
  --|  engine systems displays, and the engine fluids display. Non-engine
  --|  related functions performed by the EDM include fuel
  --|  balancing and fuel quantity indication. Redundant power for
  --|  the engine data manager is provided through a circuit
  --|  breaker placarded EDM, located on the battery bus circuit
  --|  breaker panel in the front cockpit, and through a circuit
  --|  breaker placarded EDM on the generator bus circuit breaker
  --|  panel in the front cockpit. In the event of battery or genera-tor
  --|  malfunction, the EDM continues to operate with power
  --|  supplied by the remaining system.
  --|  The EDM transmits engine data outputs over the data bus,
  --|  using EDM A and EDM B channels. If an output is bad, or
  --|  if either data bus channel fails, the affected electronic instru-ment
  --|  display (EID) will display EDM A or EDM B FAIL in
  --|  amber text, respectively. If the EDM fails, the affected EIDs
  --|  will display EDM FAIL in amber text.
  --|
  --|  Ref. AIR FORCE TO 1T-6A-1
  --|       NAVY (NAVAIR) A1-T6AAA-NFM-100
  --|       Page 1-4 - 1-6 CVV DRAFT DECEMBER 1998
  -------------------------------------------------------------------------------

  No_Discretes : Arinc_429_Types.No_Discretes_Type;

  -- Access A429_TX_EDM_A_101_Dc_Amps
  function  A429_TX_EDM_A_101_Dc_Amps
    (An_Instance : in Instance)
    return Arinc_429_Types.Message_Type is
  begin
    return An_Instance.A429_TX_EDM_101_Dc_Amps(Edm_T.Channel_A);
  end A429_TX_EDM_A_101_Dc_Amps;

  procedure Set_A429_TX_EDM_A_101_Dc_Amps
    (An_Instance : in out Instance) is
  begin
    null;
  end Set_A429_TX_EDM_A_101_Dc_Amps;

  -- Access A429_TX_EDM_A_103_Dc_Volts
  function  A429_TX_EDM_A_103_Dc_Volts
    (An_Instance : in Instance)
    return Arinc_429_Types.Message_Type is
  begin
    return An_Instance.A429_TX_EDM_103_Dc_Volts(Edm_T.Channel_A);
  end A429_TX_EDM_A_103_Dc_Volts;

  procedure Set_A429_TX_EDM_A_103_Dc_Volts
    (An_Instance : in out Instance) is
  begin
    null;
  end Set_A429_TX_EDM_A_103_Dc_Volts;

  -- Access A429_TX_EDM_A_130_Outside_Air_Temperature
  function  A429_TX_EDM_A_130_Outside_Air_Temperature
    (An_Instance : in Instance)
    return Arinc_429_Types.Message_Type is
  begin
    return An_Instance.A429_TX_EDM_130_Outside_Air_Temperature(Edm_T.Channel_A);
  end A429_TX_EDM_A_130_Outside_Air_Temperature;

  procedure Set_A429_TX_EDM_A_130_Outside_Air_Temperature
    (An_Instance : in out Instance) is
  begin
    null;
  end Set_A429_TX_EDM_A_130_Outside_Air_Temperature;

  -- Access A429_TX_EDM_A_174_Hydraulic_Pressure
  function  A429_TX_EDM_A_174_Hydraulic_Pressure
    (An_Instance : in Instance)
    return Arinc_429_Types.Message_Type is
  begin
    return An_Instance.A429_TX_EDM_174_Hydraulic_Pressure(Edm_T.Channel_A);
  end A429_TX_EDM_A_174_Hydraulic_Pressure;

  procedure Set_A429_TX_EDM_A_174_Hydraulic_Pressure
    (An_Instance : in out Instance) is
  begin
    null;
  end Set_A429_TX_EDM_A_174_Hydraulic_Pressure;

  -- Access A429_TX_EDM_A_203_Cockpit_Pressure_Altitude
  function  A429_TX_EDM_A_203_Cockpit_Pressure_Altitude
    (An_Instance : in Instance)
    return Arinc_429_Types.Message_Type is
  begin
    return An_Instance.A429_TX_EDM_203_Cockpit_Pressure_Altitude(Edm_T.Channel_A);
  end A429_TX_EDM_A_203_Cockpit_Pressure_Altitude;

  procedure Set_A429_TX_EDM_A_203_Cockpit_Pressure_Altitude
    (An_Instance : in out Instance) is
  begin
    null;
  end Set_A429_TX_EDM_A_203_Cockpit_Pressure_Altitude;

  -- Access A429_TX_EDM_A_243_Cockpit_Differential_Pressure
  function  A429_TX_EDM_A_243_Cockpit_Differential_Pressure
    (An_Instance : in Instance)
    return Arinc_429_Types.Message_Type is
  begin
    return An_Instance.A429_TX_EDM_243_Cockpit_Differential_Pressure(Edm_T.Channel_A);
  end A429_TX_EDM_A_243_Cockpit_Differential_Pressure;

  procedure Set_A429_TX_EDM_A_243_Cockpit_Differential_Pressure
    (An_Instance : in out Instance) is
  begin
    null;
  end Set_A429_TX_EDM_A_243_Cockpit_Differential_Pressure;

  -- Access A429_TX_EDM_A_244_Fuel_Flow
  function  A429_TX_EDM_A_244_Fuel_Flow
    (An_Instance : in Instance)
    return Arinc_429_Types.Message_Type is
  begin
    return An_Instance.A429_TX_EDM_244_Fuel_Flow(Edm_T.Channel_A);
  end A429_TX_EDM_A_244_Fuel_Flow;

  procedure Set_A429_TX_EDM_A_244_Fuel_Flow
    (An_Instance : in out Instance) is
  begin
    null;
  end Set_A429_TX_EDM_A_244_Fuel_Flow;

  -- Access A429_TX_EDM_A_256_Left_Fuel_Quantity
  function  A429_TX_EDM_A_256_Left_Fuel_Quantity
    (An_Instance : in Instance)
    return Arinc_429_Types.Message_Type is
  begin
    return An_Instance.A429_TX_EDM_256_Left_Fuel_Quantity(Edm_T.Channel_A);
  end A429_TX_EDM_A_256_Left_Fuel_Quantity;

  procedure Set_A429_TX_EDM_A_256_Left_Fuel_Quantity
    (An_Instance : in out Instance) is
  begin
    null;
  end Set_A429_TX_EDM_A_256_Left_Fuel_Quantity;

  -- Access A429_TX_EDM_A_257_Right_Fuel_Quantity
  function  A429_TX_EDM_A_257_Right_Fuel_Quantity
    (An_Instance : in Instance)
    return Arinc_429_Types.Message_Type is
  begin
    return An_Instance.A429_TX_EDM_257_Right_Fuel_Quantity(Edm_T.Channel_A);
  end A429_TX_EDM_A_257_Right_Fuel_Quantity;

  procedure Set_A429_TX_EDM_A_257_Right_Fuel_Quantity
    (An_Instance : in out Instance) is
  begin
    null;
  end Set_A429_TX_EDM_A_257_Right_Fuel_Quantity;

  -- Access A429_TX_EDM_A_270_Engine_Data_Discretes
  function  A429_TX_EDM_A_270_Engine_Data_Discretes
    (An_Instance : in Instance)
    return Arinc_429_Types.Message_Type is
  begin
    return An_Instance.A429_TX_EDM_270_Engine_Data_Discretes(Edm_T.Channel_A);
  end A429_TX_EDM_A_270_Engine_Data_Discretes;

  procedure Set_A429_TX_EDM_A_270_Engine_Data_Discretes
    (An_Instance : in out Instance) is
  begin
    null;
  end Set_A429_TX_EDM_A_270_Engine_Data_Discretes;

  -- Access A429_TX_EDM_A_301_ITT
  function  A429_TX_EDM_A_301_ITT
    (An_Instance : in Instance)
    return Arinc_429_Types.Message_Type is
  begin
    return An_Instance.A429_TX_EDM_301_ITT(Edm_T.Channel_A);
  end A429_TX_EDM_A_301_ITT;

  procedure Set_A429_TX_EDM_A_301_ITT
    (An_Instance : in out Instance) is
  begin
    null;
  end Set_A429_TX_EDM_A_301_ITT;

  -- Access A429_TX_EDM_A_302_N1
  function  A429_TX_EDM_A_302_N1
    (An_Instance : in Instance)
    return Arinc_429_Types.Message_Type is
  begin
    return An_Instance.A429_TX_EDM_302_N1(Edm_T.Channel_A);
  end A429_TX_EDM_A_302_N1;

  procedure Set_A429_TX_EDM_A_302_N1
    (An_Instance : in out Instance) is
  begin
    null;
  end Set_A429_TX_EDM_A_302_N1;

  -- Access A429_TX_EDM_A_316_Oil_Temperature
  function  A429_TX_EDM_A_316_Oil_Temperature
    (An_Instance : in Instance)
    return Arinc_429_Types.Message_Type is
  begin
    return An_Instance.A429_TX_EDM_316_Oil_Temperature(Edm_T.Channel_A);
  end A429_TX_EDM_A_316_Oil_Temperature;

  procedure Set_A429_TX_EDM_A_316_Oil_Temperature
    (An_Instance : in out Instance) is
  begin
    null;
  end Set_A429_TX_EDM_A_316_Oil_Temperature;

  -- Access A429_TX_EDM_A_317_Oil_Pressure
  function  A429_TX_EDM_A_317_Oil_Pressure
    (An_Instance : in Instance)
    return Arinc_429_Types.Message_Type is
  begin
    return An_Instance.A429_TX_EDM_317_Oil_Pressure(Edm_T.Channel_A);
  end A429_TX_EDM_A_317_Oil_Pressure;

  procedure Set_A429_TX_EDM_A_317_Oil_Pressure
    (An_Instance : in out Instance) is
  begin
    null;
  end Set_A429_TX_EDM_A_317_Oil_Pressure;

  -- Access A429_TX_EDM_A_336_Engine_Torque
  function  A429_TX_EDM_A_336_Engine_Torque
    (An_Instance : in Instance)
    return Arinc_429_Types.Message_Type is
  begin
    return An_Instance.A429_TX_EDM_336_Engine_Torque(Edm_T.Channel_A);
  end A429_TX_EDM_A_336_Engine_Torque;

  procedure Set_A429_TX_EDM_A_336_Engine_Torque
    (An_Instance : in out Instance) is
  begin
    null;
  end Set_A429_TX_EDM_A_336_Engine_Torque;

  -- Access A429_TX_EDM_A_344_Engine_Rpm
  function  A429_TX_EDM_A_344_Engine_Rpm
    (An_Instance : in Instance)
    return Arinc_429_Types.Message_Type is
  begin
    return An_Instance.A429_TX_EDM_344_Engine_Rpm(Edm_T.Channel_A);
  end A429_TX_EDM_A_344_Engine_Rpm;

  procedure Set_A429_TX_EDM_A_344_Engine_Rpm
    (An_Instance : in out Instance) is
  begin
    null;
  end Set_A429_TX_EDM_A_344_Engine_Rpm;

  -- Access A429_TX_EDM_A_345_Engine_ITT
  function  A429_TX_EDM_A_345_Engine_ITT
    (An_Instance : in Instance)
    return Arinc_429_Types.Message_Type is
  begin
    return An_Instance.A429_TX_EDM_345_Engine_ITT(Edm_T.Channel_A);
  end A429_TX_EDM_A_345_Engine_ITT;

  procedure Set_A429_TX_EDM_A_345_Engine_ITT
    (An_Instance : in out Instance) is
  begin
    null;
  end Set_A429_TX_EDM_A_345_Engine_ITT;

  -- Access A429_TX_EDM_A_346_Propeller_Rpm
  function  A429_TX_EDM_A_346_Propeller_Rpm
    (An_Instance : in Instance)
    return Arinc_429_Types.Message_Type is
  begin
    return An_Instance.A429_TX_EDM_346_Propeller_Rpm(Edm_T.Channel_A);
  end A429_TX_EDM_A_346_Propeller_Rpm;

  procedure Set_A429_TX_EDM_A_346_Propeller_Rpm
    (An_Instance : in out Instance) is
  begin
    null;
  end Set_A429_TX_EDM_A_346_Propeller_Rpm;

  -- Access A429_TX_EDM_A_350_Discrete_Diagnostics
  function  A429_TX_EDM_A_350_Discrete_Diagnostics
    (An_Instance : in Instance)
    return Arinc_429_Types.Message_Type is
  begin
    return An_Instance.A429_TX_EDM_350_Discrete_Diagnostics(Edm_T.Channel_A);
  end A429_TX_EDM_A_350_Discrete_Diagnostics;

  procedure Set_A429_TX_EDM_A_350_Discrete_Diagnostics
    (An_Instance : in out Instance) is
  begin
    null;
  end Set_A429_TX_EDM_A_350_Discrete_Diagnostics;

  -- Access A429_TX_EDM_B_101_Dc_Amps
  function  A429_TX_EDM_B_101_Dc_Amps
    (An_Instance : in Instance)
    return Arinc_429_Types.Message_Type is
  begin
    return An_Instance.A429_TX_EDM_101_Dc_Amps(Edm_T.Channel_B);
  end A429_TX_EDM_B_101_Dc_Amps;

  procedure Set_A429_TX_EDM_B_101_Dc_Amps
    (An_Instance : in out Instance) is
  begin
    null;
  end Set_A429_TX_EDM_B_101_Dc_Amps;

  -- Access A429_TX_EDM_B_103_Dc_Volts
  function  A429_TX_EDM_B_103_Dc_Volts
    (An_Instance : in Instance)
    return Arinc_429_Types.Message_Type is
  begin
    return An_Instance.A429_TX_EDM_103_Dc_Volts(Edm_T.Channel_B);
  end A429_TX_EDM_B_103_Dc_Volts;

  procedure Set_A429_TX_EDM_B_103_Dc_Volts
    (An_Instance : in out Instance) is
  begin
    null;
  end Set_A429_TX_EDM_B_103_Dc_Volts;

  -- Access A429_TX_EDM_B_130_Outside_Air_Temperature
  function  A429_TX_EDM_B_130_Outside_Air_Temperature
    (An_Instance : in Instance)
    return Arinc_429_Types.Message_Type is
  begin
    return An_Instance.A429_TX_EDM_130_Outside_Air_Temperature(Edm_T.Channel_B);
  end A429_TX_EDM_B_130_Outside_Air_Temperature;

  procedure Set_A429_TX_EDM_B_130_Outside_Air_Temperature
    (An_Instance : in out Instance) is
  begin
    null;
  end Set_A429_TX_EDM_B_130_Outside_Air_Temperature;

  -- Access A429_TX_EDM_B_174_Hydraulic_Pressure
  function  A429_TX_EDM_B_174_Hydraulic_Pressure
    (An_Instance : in Instance)
    return Arinc_429_Types.Message_Type is
  begin
    return An_Instance.A429_TX_EDM_174_Hydraulic_Pressure(Edm_T.Channel_B);
  end A429_TX_EDM_B_174_Hydraulic_Pressure;

  procedure Set_A429_TX_EDM_B_174_Hydraulic_Pressure
    (An_Instance : in out Instance) is
  begin
    null;
  end Set_A429_TX_EDM_B_174_Hydraulic_Pressure;

  -- Access A429_TX_EDM_B_203_Cockpit_Pressure_Altitude
  function  A429_TX_EDM_B_203_Cockpit_Pressure_Altitude
    (An_Instance : in Instance)
    return Arinc_429_Types.Message_Type is
  begin
    return An_Instance.A429_TX_EDM_203_Cockpit_Pressure_Altitude(Edm_T.Channel_B);
  end A429_TX_EDM_B_203_Cockpit_Pressure_Altitude;

  procedure Set_A429_TX_EDM_B_203_Cockpit_Pressure_Altitude
    (An_Instance : in out Instance) is
  begin
    null;
  end Set_A429_TX_EDM_B_203_Cockpit_Pressure_Altitude;

  -- Access A429_TX_EDM_B_243_Cockpit_Differential_Pressure
  function  A429_TX_EDM_B_243_Cockpit_Differential_Pressure
    (An_Instance : in Instance)
    return Arinc_429_Types.Message_Type is
  begin
    return An_Instance.A429_TX_EDM_243_Cockpit_Differential_Pressure(Edm_T.Channel_B);
  end A429_TX_EDM_B_243_Cockpit_Differential_Pressure;

  procedure Set_A429_TX_EDM_B_243_Cockpit_Differential_Pressure
    (An_Instance : in out Instance) is
  begin
    null;
  end Set_A429_TX_EDM_B_243_Cockpit_Differential_Pressure;

  -- Access A429_TX_EDM_B_244_Fuel_Flow
  function  A429_TX_EDM_B_244_Fuel_Flow
    (An_Instance : in Instance)
    return Arinc_429_Types.Message_Type is
  begin
    return An_Instance.A429_TX_EDM_244_Fuel_Flow(Edm_T.Channel_B);
  end A429_TX_EDM_B_244_Fuel_Flow;

  procedure Set_A429_TX_EDM_B_244_Fuel_Flow
    (An_Instance : in out Instance) is
  begin
    null;
  end Set_A429_TX_EDM_B_244_Fuel_Flow;

  -- Access A429_TX_EDM_B_256_Left_Fuel_Quantity
  function  A429_TX_EDM_B_256_Left_Fuel_Quantity
    (An_Instance : in Instance)
    return Arinc_429_Types.Message_Type is
  begin
    return An_Instance.A429_TX_EDM_256_Left_Fuel_Quantity(Edm_T.Channel_B);
  end A429_TX_EDM_B_256_Left_Fuel_Quantity;

  procedure Set_A429_TX_EDM_B_256_Left_Fuel_Quantity
    (An_Instance : in out Instance) is
  begin
    null;
  end Set_A429_TX_EDM_B_256_Left_Fuel_Quantity;

  -- Access A429_TX_EDM_B_257_Right_Fuel_Quantity
  function  A429_TX_EDM_B_257_Right_Fuel_Quantity
    (An_Instance : in Instance)
    return Arinc_429_Types.Message_Type is
  begin
    return An_Instance.A429_TX_EDM_257_Right_Fuel_Quantity(Edm_T.Channel_B);
  end A429_TX_EDM_B_257_Right_Fuel_Quantity;

  procedure Set_A429_TX_EDM_B_257_Right_Fuel_Quantity
    (An_Instance : in out Instance) is
  begin
    null;
  end Set_A429_TX_EDM_B_257_Right_Fuel_Quantity;

  -- Access A429_TX_EDM_B_270_Engine_Data_Discretes
  function  A429_TX_EDM_B_270_Engine_Data_Discretes
    (An_Instance : in Instance)
    return Arinc_429_Types.Message_Type is
  begin
    return An_Instance.A429_TX_EDM_270_Engine_Data_Discretes(Edm_T.Channel_B);
  end A429_TX_EDM_B_270_Engine_Data_Discretes;

  procedure Set_A429_TX_EDM_B_270_Engine_Data_Discretes
    (An_Instance : in out Instance) is
  begin
    null;
  end Set_A429_TX_EDM_B_270_Engine_Data_Discretes;

  -- Access A429_TX_EDM_B_301_ITT
  function  A429_TX_EDM_B_301_ITT
    (An_Instance : in Instance)
    return Arinc_429_Types.Message_Type is
  begin
    return An_Instance.A429_TX_EDM_301_ITT(Edm_T.Channel_B);
  end A429_TX_EDM_B_301_ITT;

  procedure Set_A429_TX_EDM_B_301_ITT
    (An_Instance : in out Instance) is
  begin
    null;
  end Set_A429_TX_EDM_B_301_ITT;

  -- Access A429_TX_EDM_B_302_N1
  function  A429_TX_EDM_B_302_N1
    (An_Instance : in Instance)
    return Arinc_429_Types.Message_Type is
  begin
    return An_Instance.A429_TX_EDM_302_N1(Edm_T.Channel_B);
  end A429_TX_EDM_B_302_N1;

  procedure Set_A429_TX_EDM_B_302_N1
    (An_Instance : in out Instance) is
  begin
    null;
  end Set_A429_TX_EDM_B_302_N1;

  -- Access A429_TX_EDM_B_316_Oil_Temperature
  function  A429_TX_EDM_B_316_Oil_Temperature
    (An_Instance : in Instance)
    return Arinc_429_Types.Message_Type is
  begin
    return An_Instance.A429_TX_EDM_316_Oil_Temperature(Edm_T.Channel_B);
  end A429_TX_EDM_B_316_Oil_Temperature;

  procedure Set_A429_TX_EDM_B_316_Oil_Temperature
    (An_Instance : in out Instance) is
  begin
    null;
  end Set_A429_TX_EDM_B_316_Oil_Temperature;

  -- Access A429_TX_EDM_B_317_Oil_Pressure
  function  A429_TX_EDM_B_317_Oil_Pressure
    (An_Instance : in Instance)
    return Arinc_429_Types.Message_Type is
  begin
    return An_Instance.A429_TX_EDM_317_Oil_Pressure(Edm_T.Channel_B);
  end A429_TX_EDM_B_317_Oil_Pressure;

  procedure Set_A429_TX_EDM_B_317_Oil_Pressure
    (An_Instance : in out Instance) is
  begin
    null;
  end Set_A429_TX_EDM_B_317_Oil_Pressure;

  -- Access A429_TX_EDM_B_336_Engine_Torque
  function  A429_TX_EDM_B_336_Engine_Torque
    (An_Instance : in Instance)
    return Arinc_429_Types.Message_Type is
  begin
    return An_Instance.A429_TX_EDM_336_Engine_Torque(Edm_T.Channel_B);
  end A429_TX_EDM_B_336_Engine_Torque;

  procedure Set_A429_TX_EDM_B_336_Engine_Torque
    (An_Instance : in out Instance) is
  begin
    null;
  end Set_A429_TX_EDM_B_336_Engine_Torque;

  -- Access A429_TX_EDM_B_344_Engine_Rpm
  function  A429_TX_EDM_B_344_Engine_Rpm
    (An_Instance : in Instance)
    return Arinc_429_Types.Message_Type is
  begin
    return An_Instance.A429_TX_EDM_344_Engine_Rpm(Edm_T.Channel_B);
  end A429_TX_EDM_B_344_Engine_Rpm;

  procedure Set_A429_TX_EDM_B_344_Engine_Rpm
    (An_Instance : in out Instance) is
  begin
    null;
  end Set_A429_TX_EDM_B_344_Engine_Rpm;

  -- Access A429_TX_EDM_B_345_Engine_ITT
  function  A429_TX_EDM_B_345_Engine_ITT
    (An_Instance : in Instance)
    return Arinc_429_Types.Message_Type is
  begin
    return An_Instance.A429_TX_EDM_345_Engine_ITT(Edm_T.Channel_B);
  end A429_TX_EDM_B_345_Engine_ITT;

  procedure Set_A429_TX_EDM_B_345_Engine_ITT
    (An_Instance : in out Instance) is
  begin
    null;
  end Set_A429_TX_EDM_B_345_Engine_ITT;

  -- Access A429_TX_EDM_B_346_Propeller_Rpm
  function  A429_TX_EDM_B_346_Propeller_Rpm
    (An_Instance : in Instance)
    return Arinc_429_Types.Message_Type is
  begin
    return An_Instance.A429_TX_EDM_346_Propeller_Rpm(Edm_T.Channel_B);
  end A429_TX_EDM_B_346_Propeller_Rpm;

  procedure Set_A429_TX_EDM_B_346_Propeller_Rpm
    (An_Instance : in out Instance) is
  begin
    null;
  end Set_A429_TX_EDM_B_346_Propeller_Rpm;

  -- Access A429_TX_EDM_B_350_Discrete_Diagnostics
  function  A429_TX_EDM_B_350_Discrete_Diagnostics
    (An_Instance : in Instance)
    return Arinc_429_Types.Message_Type is
  begin
    return An_Instance.A429_TX_EDM_350_Discrete_Diagnostics(Edm_T.Channel_B);
  end A429_TX_EDM_B_350_Discrete_Diagnostics;

  procedure Set_A429_TX_EDM_B_350_Discrete_Diagnostics
    (An_Instance : in out Instance) is
  begin
    null;
  end Set_A429_TX_EDM_B_350_Discrete_Diagnostics;

  -- Access Nose Wheel Steering Enable
  function Nws_Enable
    (An_Instance : in Instance)
    return Boolean is
  begin
    return An_Instance.Nws_Enable;
  end Nws_Enable;

  -- Access Oil_Press_Lo_Signal
  function  Oil_Press_Lo_Signal
    (An_Instance : in Instance)
    return Boolean is
  begin
    return An_Instance.Oil_Press_Lo_Signal;
  end Oil_Press_Lo_Signal;

  -- Access Cockpit_Alt_Hi_Output
  function  Cockpit_Alt_Hi_Output
    (An_Instance : in Instance)
    return Boolean is
  begin
    return An_Instance.Cockpit_Alt_Hi_Output;
  end Cockpit_Alt_Hi_Output;

  -- Access Cockpit_Press_Hi_Output
  function  Cockpit_Press_Hi_Output
    (An_Instance : in Instance)
    return Boolean is
  begin
    return An_Instance.Cockpit_Press_Hi_Output;
  end Cockpit_Press_Hi_Output;

  -- Access Hyd_Press_Lo_Output
  function  Hyd_Press_Lo_Output
    (An_Instance : in Instance)
    return Boolean is
  begin
    return An_Instance.Hyd_Press_Lo_Output;
  end Hyd_Press_Lo_Output;

  -- Access Auto_Balance_Fail_Output
  function  Auto_Balance_Fail_Output
    (An_Instance : in Instance)
    return Boolean is
  begin
    return An_Instance.Auto_Balance_Fail_Output;
  end Auto_Balance_Fail_Output;

  -- Access Left_Transfer_Valve_Out
  function  Left_Transfer_Valve_Out
    (An_Instance : in Instance)
    return Boolean is
  begin
    return An_Instance.Left_Transfer_Valve_Out;
  end Left_Transfer_Valve_Out;

  -- Access Right_Transfer_Valve_Out
  function  Right_Transfer_Valve_Out
    (An_Instance : in Instance)
    return Boolean is
  begin
    return An_Instance.Right_Transfer_Valve_Out;
  end Right_Transfer_Valve_Out;

  -- Access Collector_Fuel_Quantity
  function  Collector_Fuel_Quantity
    (An_Instance : in Instance)
    return Mass_Types.Lbm is
  begin
    return An_Instance.Collector_Fuel_Quantity;
  end Collector_Fuel_Quantity;

  -- Access Left_Fuel_Quantity
  function  Left_Fuel_Quantity
    (An_Instance : in Instance)
    return Mass_Types.Lbm is
  begin
    return An_Instance.Left_Fuel_Quantity;
  end Left_Fuel_Quantity;

  -- Access Right_Fuel_Quantity
  function  Right_Fuel_Quantity
    (An_Instance : in Instance)
    return Mass_Types.Lbm is
  begin
    return An_Instance.Right_Fuel_Quantity;
  end Right_Fuel_Quantity;


  -----------------------------------------------------------------------
  -- Power On Initialize
  --
  -- When input power is with the specified operating range during a
  -- power up cycle, the EDM shall initialize itself and conduct and
  -- required self testing automatically.  The intialization sequence
  -- shall include testing ( and resetting if necessary) of all random
  -- access memory (RAM), read only memory (ROM) from the various
  -- processor and momory devices, testing of interal bus cards.
  -- During this period the EDM status shall be evaluated fro global
  -- type failures.  This procedure must be completed within a time
  -- span of 2 seconds at which point, data shall be availale.
  -- Exception may include pneumatic instrument functions, and fuel
  -- quantity measurement which by their nature, require up to 5
  -- seconds to gather all data, and restore accurate data outputs.
  -- These parameters shall be transmitted as NCD ( ref 3.2.0 and
  -- Table III) until valid sensor data is again available.
  --
  -- Ref. Specification for Operation Requirement for Engine Data
  -- Manager for the Model 300 Aircraft.  Raytheon Doc.  BS24863.
  ------------------------------------------------------------------------


  procedure Power_On_Initialize
    (Next_Instance : in out Instance;
     This_Instance : in out Instance;
     Weight_On_Wheels : in  Boolean) is
  begin
    if Weight_On_Wheels then
      This_Instance.L_Wheel_Speed_30 := True;
      This_Instance.R_Wheel_Speed_30 := True;
      --      Label_350_Discrete.A_Ram_Self_Test_Failure       := False;
      --      Label_350_Discrete.A_Arinc_Self_Test_Failure     := False;
      --      Label_350_Discrete.A_Program_Integrity_Failure   := False;
      --      Label_350_Discrete.A_Power_Supply_Failure        := False;
      --      Label_350_Discrete.B_Ram_Self_Test_Failure       := False;
      --      Label_350_Discrete.B_Arinc_Self_Test_Failure     := False;
      --      Label_350_Discrete.B_Program_Integrity_Failure   := False;
      --      Label_350_Discrete.B_Power_Supply_Failure        := False;
    end if;
  end Power_On_Initialize;


  Fuel_20_Lb_Diff_Timer : Float := 0.0; -- timer for fuel balance
  Fuel_30_Lb_Diff_Timer : Float := 0.0; --  "     "    "    "

  procedure Update
    (Next_Instance                              : in out Instance;
     This_Instance                              : in out Instance;
     Iconst                                     : in     Float;
     Alternate_Itt_Input_Hi                     : in     Temperature_Types.Celsius;       -- pin   Label 301
     Alternate_N1_Input_Hi                      : in     Angle_Types.Rpm;                 -- pin   Label 302
     Anti_Ice_On                                : in     Boolean;                         -- pin
     Auto_Balance_Selected                      : in     Boolean;                         -- pin
     Bat_Bus_28_VDC                             : in     Boolean;                         -- pin
     Collector_Probe_0_Signal                   : in     Mass_Types.Lbm;                  -- pin
     Collector_Probe_0_Signal_Malfunction       : in     Boolean;
     DC_Amps                                    : in     Electrical_Units_Types.Amps;     -- pin
     Dc_Volts                                   : in     Electrical_Units_Types.Volts_DC; -- pin
     Fuel_Balance_28_VDC                        : in     Boolean;                         -- pin
     Fuel_Flow_Input_Hi                         : in     Mass_Types.Pph;                  -- pin
     Gen_Bus_28_Vdc                             : in     Boolean;                         -- pin
     Hyd_Press_Input_Hi                         : in     Force_Types.Psi;                 -- pin
     L_Fuel_Probe_1_Signal                      : in     Mass_Types.Lbm;                  -- pin
     L_Fuel_Probe_3_Signal                      : in     Mass_Types.Lbm;                  -- pin
     L_Fuel_Probe_5_Signal                      : in     Mass_Types.Lbm;                  -- pin
     L_Fuel_Probe_1_Signal_Malfunction          : in     Boolean;
     L_Fuel_Probe_3_Signal_Malfunction          : in     Boolean;
     L_Fuel_Probe_5_Signal_Malfunction          : in     Boolean;
     L_Wheel_Spd_Sig_Hi                         : in     Float;                           -- pin
     Oil_Press_Input_Hi                         : in     Force_Types.Psi;                 -- pin
     Oil_Temp_Signal_Hi                         : in     Temperature_Types.Celsius;       -- pin
     A429_RX_PMU_A_336_Engine_Torque            : in     Arinc_429_Types.Message_Type;    -- Label 336
     A429_RX_PMU_A_344_Engine_Rpm               : in     Arinc_429_Types.Message_Type;    -- Label 344
     A429_RX_PMU_A_345_Engine_ITT               : in     Arinc_429_Types.Message_Type;    -- Label 345
     A429_RX_PMU_A_346_Propeller_Rpm            : in     Arinc_429_Types.Message_Type;    -- Label 346
     A429_RX_PMU_A_130_Outside_Air_Temperature  : in     Arinc_429_Types.Message_Type;    -- Label 130
     A429_RX_PMU_A_134_Power_Lever_Angle        : in     Arinc_429_Types.Message_Type;    -- Label 134
     A429_RX_PMU_B_336_Engine_Torque            : in     Arinc_429_Types.Message_Type;    -- Label 336
     A429_RX_PMU_B_344_Engine_Rpm               : in     Arinc_429_Types.Message_Type;    -- Label 344
     A429_RX_PMU_B_345_Engine_ITT               : in     Arinc_429_Types.Message_Type;    -- Label 345
     A429_RX_PMU_B_346_Propeller_Rpm            : in     Arinc_429_Types.Message_Type;    -- Label 346
     A429_RX_PMU_B_130_Outside_Air_Temperature  : in     Arinc_429_Types.Message_Type;    -- Label 130
     A429_RX_PMU_B_134_Power_Lever_Angle        : in     Arinc_429_Types.Message_Type;    -- Label 134
     R_Fuel_Probe_2_Signal                      : in     Mass_Types.Lbm;                  -- pin
     R_Fuel_Probe_4_Signal                      : in     Mass_Types.Lbm;                  -- pin
     R_Fuel_Probe_6_Signal                      : in     Mass_Types.Lbm;                  -- pin
     R_Fuel_Probe_2_Signal_Malfunction          : in     Boolean;
     R_Fuel_Probe_4_Signal_Malfunction          : in     Boolean;
     R_Fuel_Probe_6_Signal_Malfunction          : in     Boolean;
     R_Wheel_Spd_Sig_Hi                         : in     Float;                           -- pin
     Weight_On_Wheels                           : in     Boolean;                         -- pin
     Cockpit_Pressure_Altitude                  : in     Length_Types.Feet;
     Cockpit_Differential_Pressure              : in     Force_Types.Psi;
     EDM_Test                                   : in     Boolean;
     EDM_Fail                                   : in     Boolean;
     Auto_Balance_Fail_Malfunction              : in     Boolean)

    --   Fuel_Quantity_Calibrate_Dry : in     Boolean;                                 -- pin
    --   Pmu_Input_hi                : in     Integer;                                 -- pin
    --   L_Wheel_Spd_Excitation      : in     Float;                                   -- pin
    --   R_Wheel_Spd_Excitation      : in     Float;                                   -- pin
    --   L_Dc_Ground                 : in     Electrical_Units_Types.Volts_Dc;         -- pin
    --   R_Dc_Ground                 : in     Electrical_Units_Types.Volts_Dc;         -- pin
    --   R_Wheel_Spd_Sig_Lo          : in     Float;                                   -- pin
    --   Chassis_Ground              : in     Boolean;                                 -- pin
    --   A_28_VDC_Power_Return       : in     Boolean;                                 -- pin
    --   Alternate_Itt_Input_Lo      : in     Temperature_Types.Celsius;               -- pin
    --   Alternate_N1_Input_Lo       : in     Angle_Types.Rpm;                         -- pin
    --   Ann_Discrete_Output_Return  : in     Boolean;                                 -- pin
    --   B_28_VDC_Power_Return       : in     Boolean;                                 -- pin
    --   DC_Amps                     : in     Electrical_Units_Types.Amps;             -- pin
    --   Fuel_Flow_Input_Lo          : in     Mass_Types.Pph;                          -- pin
    --   Hyd_Press_Input_Lo          : in     Force_Types.Psi;                         -- pin
    --   L_Wheel_Spd_Sig_Lo          : in     Float;                                   -- pin
    --   Oil_Press_Input_Lo          : in     Force_Types.Psi;                         -- pin
    --   Oil_Temp_Signal_Lo          : in     Temperature_Types.Celsius;               -- pin
    --   Pmu_Input_lo                : in     Integer;                                 -- pin
    --   Collector_Probe_0_Drive     : in     Mass_Types.Lbm;                          -- pin
    --   Collector_Probe_0_Return    : in     Mass_Types.Lbm;                          -- pin
    --   Dc_Volts_Low                : in     Electrical_Units_Types.Volts_Dc;         -- pin
    --   Discrete_Growth             : in     Boolean;                                 -- pin
    --   Discrete_Growth             : in     Boolean;                                 -- pin
    --   Discrete_Growth             : in     Boolean;                                 -- pin
    --   Discrete_Growth             : in     Boolean;                                 -- pin
    --   Discrete_Growth             : in     Boolean;                                 -- pin
    --   Factory_A_receive           : in     boolean;                                 -- pin
    --   Factory_B_Receive           : in     boolean;                                 -- pin
    --   L_Fuel_Probe_1_Drive        : in     Mass_Types.Lbm;                          -- pin
    --   L_Fuel_Probe_1_Return       : in     Mass_Types.Lbm;                          -- pin
    --   L_Fuel_Probe_3_Drive        : in     Mass_Types.Lbm;                          -- pin
    --   L_Fuel_Probe_3_Return       : in     Mass_Types.Lbm;                          -- pin
    --   L_Fuel_Probe_5_Drive        : in     Mass_Types.Lbm;                          -- pin
    --   L_Fuel_Probe_5_Return       : in     Mass_Types.Lbm;                          -- pin
    --   R_Fuel_Probe_2_Drive        : in     Mass_Types.Lbm;                          -- pin
    --   R_Fuel_Probe_2_Return       : in     Mass_Types.Lbm;                          -- pin
    --   R_Fuel_Probe_4_Drive        : in     Mass_Types.Lbm;                          -- pin
    --   R_Fuel_Probe_4_Return       : in     Mass_Types.Lbm;                          -- pin
    --   R_Fuel_Probe_6_Drive        : in     Mass_Types.Lbm;                          -- pin
    --   R_Fuel_Probe_6_Return       : in     Mass_Types.Lbm;                          -- pin
    --   Spare                       : in     Boolean;                                 -- pin
    --   Spare                       : in     Boolean;                                 -- pin
    --   Spare                       : in     Boolean;                                 -- pin
    --   Spare                       : in     Boolean;                                 -- pin
    --   Spare                       : in     Boolean;                                 -- pin
    --   Spare                       : in     Boolean;                                 -- pin
    --   Spare                       : in     Boolean;                                 -- pin
    --   Spare                       : in     Boolean;                                 -- pin
    --   Spare                       : in     Boolean;                                 -- pin
    --   Spare                       : in     Boolean;                                 -- pin
    --   Spare                       : in     Boolean;                                 -- pin
    --   Spare                       : in     Boolean;                                 -- pin
    --   Spare                       : in     Boolean;                                 -- pin
    --   Spare                       : in     Boolean);                                -- pin
  is

    Sdi                          : Edm_T.Edm_Bnr_Sdi_Type := Edm_T.Channel_A;
    Power_Lever_Angle            : Angle_Types.Degrees;
    Auto_Balance_Selected_Toggle : Boolean;
    Edm_Power                    : Boolean;
  begin
    -- Power interrupts.  During a peroid of momentrary input power
    -- loss, the EDM shall cease further bus transmisiions.  Power
    -- interrupts of less thatn 50 mS shall not require the EDM to
    -- re-initialize, and output data shall be available within 25 mS
    -- of the completion of the interrupt.  Power interrupts longer
    -- than 50ms may require the EDM to perform initialization.  All
    -- power interrupts are defined from a starting input level of
    -- 27.5 VDC.

    Edm_Power := Bat_Bus_28_Vdc or Gen_Bus_28_Vdc;
    --    Log.Report("Domain: The edm_power is " & Boolean'Image(Edm_Power));
    --    Log.Report("Domain: The battery cb is " & Boolean'Image(Bat_Bus_28_Vdc));
    --    Log.Report("Domain: The generator cb is " & Boolean'Image(Gen_Bus_28_Vdc));
    if Edm_Power then
      if This_Instance.Normalized_Capacitance = 0.0 then
        Power_On_Initialize
          (Next_Instance => Next_Instance,
           This_Instance => This_Instance,
           Weight_On_Wheels => Weight_On_Wheels);
        if Sdi = Edm_T.Channel_A then
          Sdi := Edm_T.Channel_B;
        elsif Sdi = Edm_T.Channel_B then
          Sdi := Edm_T.Channel_A;
        end if;
        Next_Instance.Normalized_Capacitance := This_Instance.Normalized_Capacitance + iconst;
        --        Next_Instance.Data_Available_Timer :=  Float'Min( 1.0, This_Instance.Data_Available_Timer + Iconst);
      else
        Next_Instance.Normalized_Capacitance := This_Instance.Normalized_Capacitance + iconst;
        --        Next_Instance.Data_Available_Timer :=  Float'Min( 1.0, This_Instance.Data_Available_Timer + Iconst);
        --        Next_Instance.Data_Available_Timer := 0.0;
      end if;

    else
      Next_Instance.Normalized_Capacitance := Float'Min(1.0, Float'Max(0.0, This_Instance.Normalized_Capacitance - iconst));
      --      Next_Instance.Data_Available_Timer := 0.0;
    end if;    -- power
    Next_Instance.Normalized_Capacitance := Float'Min(1.0, Float'Max(0.0, Next_Instance.Normalized_Capacitance));
    Next_Instance.Data_Available := Next_Instance.Normalized_Capacitance > 0.025;

    if Next_Instance.Data_Available then
      ----------------------------------------------------
      -- Fuel Quantity Control
      ----------------------------------------------------

      Next_Instance.Auto_Balance_Mode := Auto_Balance_Selected;

      -- The left inside probe is not failed until the left middle probe quantity is zero
      if L_Fuel_Probe_1_Signal_Malfunction and L_Fuel_Probe_3_Signal < 0.1 then
        Next_Instance.Left_Fuel_Quantity :=  0.0;
      else
        Next_Instance.Left_Fuel_Quantity :=  L_Fuel_Probe_1_Signal;
      end if;

      if not L_Fuel_Probe_3_Signal_Malfunction then
        Next_Instance.Left_Fuel_Quantity :=  Next_Instance.Left_Fuel_Quantity + L_Fuel_Probe_3_Signal;
      end if;

      if not L_Fuel_Probe_5_Signal_Malfunction then
        Next_Instance.Left_Fuel_Quantity :=  Next_Instance.Left_Fuel_Quantity + L_Fuel_Probe_5_Signal;
      end if;

      if not Collector_Probe_0_Signal_Malfunction and not (L_Fuel_Probe_1_Signal_Malfunction and L_Fuel_Probe_3_Signal_Malfunction and L_Fuel_Probe_5_Signal_Malfunction)  then
        Next_Instance.Left_Fuel_Quantity :=  Next_Instance.Left_Fuel_Quantity + Collector_Probe_0_Signal/2.0;
      end if;

      -- The right inside probe is not failed until the right middle probe quantity is zero
      if R_Fuel_Probe_2_Signal_Malfunction and R_Fuel_Probe_4_Signal < 0.1 then
        Next_Instance.Right_Fuel_Quantity :=  0.0;
      else
        Next_Instance.Right_Fuel_Quantity :=  R_Fuel_Probe_2_Signal;
      end if;

      if not R_Fuel_Probe_4_Signal_Malfunction then
        Next_Instance.Right_Fuel_Quantity :=  Next_Instance.Right_Fuel_Quantity + R_Fuel_Probe_4_Signal;
      end if;

      if not R_Fuel_Probe_6_Signal_Malfunction then
        Next_Instance.Right_Fuel_Quantity :=  Next_Instance.Right_Fuel_Quantity + R_Fuel_Probe_6_Signal;
      end if;

      if not Collector_Probe_0_Signal_Malfunction then
        Next_Instance.Right_Fuel_Quantity :=  Next_Instance.Right_Fuel_Quantity + Collector_Probe_0_Signal/2.0;
        Next_Instance.Collector_Fuel_Quantity := Collector_Probe_0_Signal;
      else
        Next_Instance.Collector_Fuel_Quantity := 0.0;
      end if;

      if abs(Next_Instance.Left_Fuel_Quantity - Next_Instance.Right_Fuel_Quantity) >= 20.0 then
        Fuel_20_Lb_Diff_Timer := Float'Min(30.0, Fuel_20_Lb_Diff_Timer + Iconst);
      else
        Fuel_20_Lb_Diff_Timer := 0.0;
      end if;

      if abs(Next_Instance.Left_Fuel_Quantity - Next_Instance.Right_Fuel_Quantity) >= 30.0 then
        Fuel_30_Lb_Diff_Timer := Float'Min(120.0, Fuel_30_Lb_Diff_Timer + Iconst);
      else
        Fuel_30_Lb_Diff_Timer := 0.0;
      end if;

      Auto_Balance_Selected_Toggle :=  Next_Instance.Auto_Balance_Mode
        and not This_Instance.Auto_Balance_Mode;

      if Auto_Balance_Selected_Toggle then
        Fuel_30_Lb_Diff_Timer  := 0.0;
      end if;

      Next_Instance.Auto_Balance_Fail_Output :=
        ((not Fuel_Balance_28_VDC)
         or Auto_Balance_Fail_Malfunction
         or L_Fuel_Probe_1_Signal_Malfunction
         or L_Fuel_Probe_3_Signal_Malfunction
         or L_Fuel_Probe_5_Signal_Malfunction
         or Collector_Probe_0_Signal_Malfunction
         or R_Fuel_Probe_2_Signal_Malfunction
         or R_Fuel_Probe_4_Signal_Malfunction
         or R_Fuel_Probe_6_Signal_Malfunction
         or Fuel_30_Lb_Diff_Timer >= 120.0
         or This_Instance.Auto_Balance_Fail_Output)
         and (not Auto_Balance_Selected_Toggle);

      if Next_Instance.Auto_Balance_Mode then
        if Fuel_20_Lb_Diff_Timer >= 30.0 then
          if Next_Instance.Left_Fuel_Quantity - Next_Instance.Right_Fuel_Quantity > 0.0 then
            Next_Instance.R_Xfr_Vlv_Close_Init := True;
          else
            Next_Instance.L_Xfr_Vlv_Close_Init := True;
          end if;
        else
          Next_Instance.L_Xfr_Vlv_Close_Init := False;
          Next_Instance.R_Xfr_Vlv_Close_Init := False;
        end if;
      else
        Next_Instance.L_Xfr_Vlv_Close_Init := False;
        Next_Instance.R_Xfr_Vlv_Close_Init := False;
      end if;

      if not Next_Instance.Auto_Balance_Fail_Output and then
        Next_Instance.Auto_Balance_Mode and then
        Fuel_Balance_28_VDC then
        Next_Instance.Left_Transfer_Valve_Out :=
          (Next_Instance.L_Xfr_Vlv_Close_Init or else
          (This_Instance.Left_Transfer_Valve_Out and then
          (Next_Instance.Right_Fuel_Quantity - Next_Instance.Left_Fuel_Quantity) > 0.0));

        Next_Instance.Right_Transfer_Valve_Out :=
          (Next_Instance.R_Xfr_Vlv_Close_Init or else
          (This_Instance.Right_Transfer_Valve_Out and then
          (Next_Instance.Left_Fuel_Quantity - Next_Instance.Right_Fuel_Quantity) > 0.0));
      else
        Next_Instance.Left_Transfer_Valve_Out  := False;
        Next_Instance.Right_Transfer_Valve_Out := Auto_Balance_Fail_Malfunction; --False;
      end if;

    end if;  -- data available


Label_134:
      -- Power Lever Angle from PMU Arinc Bus
    declare
      -- Label_134_Discrete : Edm_T.Label_134_Discrete_Type;
        Ssm                : Arinc_429_Types.Bnr_Ssm_Type;
        Channel            : Edm_T.Edm_Bnr_Sdi_Type;
      begin

        if Sdi = Edm_T.Channel_A then
          A429.Label_134.Unpack
            ( A_Message => A429_RX_PMU_A_134_Power_Lever_Angle,
              A_Value   => Power_Lever_Angle,
              A_Sdi     => Sdi,
              A_Ssm     => Ssm,
              A_Discretes_Value => No_Discretes);
        else

          A429.Label_134.Unpack
            ( A_Message => A429_RX_PMU_B_134_Power_Lever_Angle,
              A_Value   => Power_Lever_Angle,
              A_Sdi     => Sdi,
              A_Ssm     => Ssm,
              A_Discretes_Value => No_Discretes);
        end if;

      end Label_134;



      -- Outside Air Temperature from PMU Arinc Bus
      declare
        Output             : Float;
        Ssm                : Arinc_429_Types.Bnr_Ssm_Type;
        Channel            : Edm_T.Edm_Bnr_Sdi_Type := Edm_T.Channel_A;
      begin
        A429.Label_130.Unpack
          ( A_Message => A429_RX_PMU_A_130_Outside_Air_Temperature,
            A_Value   => Output,
            A_Sdi     => Sdi,
            A_Ssm     => Ssm,
            A_Discretes_Value => No_Discretes);
--      Edm_Power :=  Gen_Bus_28_Vdc;
        if  not Gen_Bus_28_Vdc or edm_fail then
          Ssm := Arinc_429_Types.Failure;
        elsif EDM_Test then
          Ssm := Arinc_429_Types.Functional_Test;
        elsif dc_volts >= 36.0 then
          Ssm := Arinc_429_Types.No_Computed_Data;
        else
          Ssm := Arinc_429_Types.Normal;
        end if;

        Next_Instance.A429_TX_EDM_130_Outside_Air_Temperature(Edm_T.Channel_A) :=
          A429.Label_130.Pack
          ( A_Value           => Output,
            A_Sdi             => Channel,
            A_Ssm             => Ssm,
            A_Discretes_Value => No_Discretes);
      end;

      declare
        Output             : Float;
        Ssm                : Arinc_429_Types.Bnr_Ssm_Type;
        Channel            : Edm_T.Edm_Bnr_Sdi_Type := Edm_T.Channel_B;
      begin
        A429.Label_130.Unpack
          ( A_Message => A429_RX_PMU_B_130_Outside_Air_Temperature,
            A_Value   => Output,
            A_Sdi     => Sdi,
            A_Ssm     => Ssm,
            A_Discretes_Value => No_Discretes);
        if  not Bat_Bus_28_Vdc or edm_fail then
          Ssm := Arinc_429_Types.Failure;
        elsif EDM_Test then
          Ssm := Arinc_429_Types.Functional_Test;
        elsif dc_volts >= 36.0 then
          Ssm := Arinc_429_Types.No_Computed_Data;
        else
          Ssm := Arinc_429_Types.Normal;
        end if;

        Next_Instance.A429_TX_EDM_130_Outside_Air_Temperature(Edm_T.Channel_B) :=
          A429.Label_130.Pack
          ( A_Value           => Output,
            A_Sdi             => Channel,
            A_Ssm             => Ssm,
            A_Discretes_Value => No_Discretes);
      end;

      -- Torque from PMU Arinc Bus
      declare
        Output      : Float;
        Ssm         : Arinc_429_Types.Bnr_Ssm_Type;
        Channel     : Edm_T.Edm_Bnr_Sdi_Type := Edm_T.Channel_A;
      begin
        A429.Label_336.Unpack
          ( A_Message => A429_RX_PMU_A_336_Engine_Torque,
            A_Value   => Output,
            A_Sdi     => Sdi,
            A_Ssm     => Ssm,
            A_Discretes_Value => No_Discretes);
        if  not Gen_Bus_28_Vdc or edm_fail then
          Ssm := Arinc_429_Types.Failure;
        elsif EDM_Test then
          Ssm := Arinc_429_Types.Functional_Test;
        elsif dc_volts >= 36.0 then
          Ssm := Arinc_429_Types.No_Computed_Data;
        else
          Ssm := Arinc_429_Types.Normal;
        end if;

        Next_Instance.A429_Tx_Edm_336_Engine_Torque(Edm_T.Channel_A) :=
          A429.Label_336.Pack
          ( A_Value           => Output,
            A_Sdi             => Channel,
            A_Ssm             => Ssm,
            A_Discretes_Value => No_Discretes);
      end;

      declare
        Output    : Float;
        Ssm       : Arinc_429_Types.Bnr_Ssm_Type;
        Channel   : Edm_T.Edm_Bnr_Sdi_Type := Edm_T.Channel_B;
      begin
        A429.Label_336.Unpack
          ( A_Message => A429_RX_PMU_B_336_Engine_Torque,
            A_Value   => Output,
            A_Sdi     => Sdi,
            A_Ssm     => Ssm,
            A_Discretes_Value => No_Discretes);
        if  not Bat_Bus_28_Vdc or edm_fail then
          Ssm := Arinc_429_Types.Failure;
        elsif EDM_Test then
          Ssm := Arinc_429_Types.Functional_Test;
        elsif dc_volts >= 36.0 then
          Ssm := Arinc_429_Types.No_Computed_Data;
        else
          Ssm := Arinc_429_Types.Normal;
        end if;

        Next_Instance.A429_Tx_Edm_336_Engine_Torque(Edm_T.Channel_B) :=
          A429.Label_336.Pack
          ( A_Value           => Output,
            A_Sdi             => Channel,
            A_Ssm             => Ssm,
            A_Discretes_Value => No_Discretes);
      end;
      -- Next_Instance.A429_Tx_Edm_336_Engine_Torque(Edm_T.Channel_A)           := A429_RX_PMU_A_336_Engine_Torque;
      -- Next_Instance.A429_Tx_Edm_336_Engine_Torque(Edm_T.Channel_B)           := A429_RX_PMU_B_336_Engine_Torque;

      -- Engine Rpm from PMU Arinc Bus
      declare
        Output           : Float;
        Ssm              : Arinc_429_Types.Bnr_Ssm_Type;
        Channel          : Edm_T.Edm_Bnr_Sdi_Type := Edm_T.Channel_A;
      begin
        A429.Label_344.Unpack
          ( A_Message => A429_RX_PMU_A_344_Engine_Rpm,
            A_Value   => Output,
            A_Sdi     => Sdi,
            A_Ssm     => Ssm,
            A_Discretes_Value => No_Discretes);
        if  not Gen_Bus_28_Vdc or edm_fail then
          Ssm := Arinc_429_Types.Failure;
        elsif EDM_Test then
          Ssm := Arinc_429_Types.Functional_Test;
        elsif dc_volts >= 36.0 then
          Ssm := Arinc_429_Types.No_Computed_Data;
        else
          Ssm := Arinc_429_Types.Normal;
        end if;

        Next_Instance.A429_Tx_Edm_344_Engine_Rpm(Edm_T.Channel_A):=
          A429.Label_344.Pack
          ( A_Value           => Output,
            A_Sdi             => Channel,
            A_Ssm             => Ssm,
            A_Discretes_Value => No_Discretes);
      end;

      declare
        Output             : Float;
        Ssm                : Arinc_429_Types.Bnr_Ssm_Type;
        Channel            : Edm_T.Edm_Bnr_Sdi_Type := Edm_T.Channel_B;
      begin
        A429.Label_344.Unpack
          ( A_Message => A429_RX_PMU_B_344_Engine_Rpm,
            A_Value   => Output,
            A_Sdi     => Sdi,
            A_Ssm     => Ssm,
            A_Discretes_Value => No_Discretes);
        if  not Bat_Bus_28_Vdc or edm_fail then
          Ssm := Arinc_429_Types.Failure;
        elsif EDM_Test then
          Ssm := Arinc_429_Types.Functional_Test;
        elsif dc_volts >= 36.0 then
          Ssm := Arinc_429_Types.No_Computed_Data;
        else
          Ssm := Arinc_429_Types.Normal;
        end if;

        Next_Instance.A429_Tx_Edm_344_Engine_Rpm(Edm_T.Channel_B) :=
          A429.Label_344.Pack
          ( A_Value           => Output,
            A_Sdi             => Channel,
            A_Ssm             => Ssm,
            A_Discretes_Value => No_Discretes);
      end;

      -- Engine ITT from PMU Arinc Bus
      declare
        Output             : Float;
        Ssm                : Arinc_429_Types.Bnr_Ssm_Type;
        Channel            : Edm_T.Edm_Bnr_Sdi_Type := Edm_T.Channel_A;
      begin
        A429.Label_345.Unpack
          ( A_Message => A429_RX_PMU_A_345_Engine_ITT,
            A_Value   => Output,
            A_Sdi     => Sdi,
            A_Ssm     => Ssm,
            A_Discretes_Value => No_Discretes);
        if  not Gen_Bus_28_Vdc or edm_fail then
          Ssm := Arinc_429_Types.Failure;
        elsif EDM_Test then
          Ssm := Arinc_429_Types.Functional_Test;
        elsif dc_volts >= 36.0 then
          Ssm := Arinc_429_Types.No_Computed_Data;
        else
          Ssm := Arinc_429_Types.Normal;
        end if;

        Next_Instance.A429_Tx_Edm_345_Engine_ITT(Edm_T.Channel_A) :=
          A429.Label_345.Pack
          ( A_Value           => Output,
            A_Sdi             => Channel,
            A_Ssm             => Ssm,
            A_Discretes_Value => No_Discretes);
      end;

      declare
        Output             : Float;
        Ssm                : Arinc_429_Types.Bnr_Ssm_Type;
        Channel            : Edm_T.Edm_Bnr_Sdi_Type := Edm_T.Channel_B;
      begin
        A429.Label_345.Unpack
          ( A_Message => A429_RX_PMU_B_345_Engine_ITT,
            A_Value   => Output,
            A_Sdi     => Sdi,
            A_Ssm     => Ssm,
            A_Discretes_Value => No_Discretes);
        if  not Bat_Bus_28_Vdc or edm_fail then
          Ssm := Arinc_429_Types.Failure;
        elsif EDM_Test then
          Ssm := Arinc_429_Types.Functional_Test;
        elsif dc_volts >= 36.0 then
          Ssm := Arinc_429_Types.No_Computed_Data;
        else
          Ssm := Arinc_429_Types.Normal;
        end if;

        Next_Instance.A429_Tx_Edm_345_Engine_ITT(Edm_T.Channel_B) :=
          A429.Label_345.Pack
          ( A_Value           => Output,
            A_Sdi             => Channel,
            A_Ssm             => Ssm,
            A_Discretes_Value => No_Discretes);
      end;

      -- Propeller Rpm from PMU Arinc Bus
      declare
        Output             : Float;
        Ssm                : Arinc_429_Types.Bnr_Ssm_Type;
        Channel            : Edm_T.Edm_Bnr_Sdi_Type := Edm_T.Channel_A;
      begin
        A429.Label_346.Unpack
          ( A_Message => A429_RX_PMU_A_346_Propeller_Rpm,
            A_Value   => Output,
            A_Sdi     => Sdi,
            A_Ssm     => Ssm,
            A_Discretes_Value => No_Discretes);
        if  not Gen_Bus_28_Vdc or edm_fail then
          Ssm := Arinc_429_Types.Failure;
        elsif EDM_Test then
          Ssm := Arinc_429_Types.Functional_Test;
        elsif dc_volts >= 36.0 then
          Ssm := Arinc_429_Types.No_Computed_Data;
        else
          Ssm := Arinc_429_Types.Normal;
        end if;

        Next_Instance.A429_Tx_Edm_346_Propeller_Rpm(Edm_T.Channel_A) :=
          A429.Label_346.Pack
          ( A_Value           => Output,
            A_Sdi             => Channel,
            A_Ssm             => Ssm,
            A_Discretes_Value => No_Discretes);
      end;
      declare
        Output             : Float;
        Ssm                : Arinc_429_Types.Bnr_Ssm_Type;
        Channel            : Edm_T.Edm_Bnr_Sdi_Type := Edm_T.Channel_B;
      begin
        A429.Label_346.Unpack
          ( A_Message => A429_RX_PMU_B_346_Propeller_Rpm,
            A_Value   => Output,
            A_Sdi     => Sdi,
            A_Ssm     => Ssm,
            A_Discretes_Value => No_Discretes);
        if  not Bat_Bus_28_Vdc or edm_fail then
          Ssm := Arinc_429_Types.Failure;
        elsif EDM_Test then
          Ssm := Arinc_429_Types.Functional_Test;
        elsif dc_volts >= 36.0 then
          Ssm := Arinc_429_Types.No_Computed_Data;
        else
          Ssm := Arinc_429_Types.Normal;
        end if;

        Next_Instance.A429_Tx_Edm_346_Propeller_Rpm(Edm_T.Channel_B) :=
          A429.Label_346.Pack
          ( A_Value           => Output,
            A_Sdi             => Channel,
            A_Ssm             => Ssm,
            A_Discretes_Value => No_Discretes);
      end;

      for Channel in Edm_T.Channel_A .. Edm_T.Channel_B loop

        if Channel = Edm_T.Channel_A then
          Edm_Power :=  Gen_Bus_28_Vdc;
        else
          Edm_Power := Bat_Bus_28_Vdc;
        end if;
        --------------------------------------
        -- DC Amps
    Label_101 :
        declare
          Output                 : Electrical_Units_Types.Amps;
          -- Label_101_Discrete     : A429.Label_101_Discrete_Type;
          Ssm                    : Arinc_429_Types.Bnr_Ssm_Type;
        begin

          -- Set Ouput label
          if  Dc_Amps >= 380.0 then
            Output := 380.0;
          elsif Dc_Amps <= -380.0 then
            Output := -380.0;
          else
            Output := Dc_Amps;
          end if;

          -- set SSM bits
          if  not Edm_Power or Edm_Fail then
            Ssm := Arinc_429_Types.Failure;
          elsif EDM_Test then
            Ssm:= Arinc_429_Types.Functional_Test;
          elsif Dc_Amps >= 360.0 or Dc_Amps <= -360.0 then
            Ssm := Arinc_429_Types.No_Computed_Data;
          else
            Ssm := Arinc_429_Types.Normal;
          end if;


          Next_Instance.A429_Tx_Edm_101_Dc_Amps(Channel) :=
            A429.Label_101.Pack
            ( A_Value => Output,
              A_Sdi   => Channel,
              A_Ssm   => Ssm,
              A_Discretes_Value => No_Discretes);

        end Label_101;
        ----------------------------------------
        -- DC Volts
    Label_103 :

        declare
          Output                 : Electrical_Units_Types.Volts_Dc;
          -- Label_103_Discrete     : A429.Label_103_Discrete_Type;
          Ssm                    : Arinc_429_Types.Bnr_Ssm_Type;
        begin
          -- Set Ouput label
          if Dc_Volts >= 40.00 then
            Output := 40.00;
          elsif Dc_Volts <= 0.0 then
            Output := 0.0;
          else
            Output := Dc_Volts;
          end if;

          -- set SSM bits
          if  not Edm_Power or edm_fail then
            Ssm := Arinc_429_Types.Failure;
          elsif EDM_Test then
            Ssm := Arinc_429_Types.Functional_Test;
          elsif dc_volts >= 36.0 then
            Ssm := Arinc_429_Types.No_Computed_Data;
          else
            Ssm := Arinc_429_Types.Normal;
          end if;

          Next_Instance.A429_TX_EDM_103_Dc_Volts(Channel) :=
            A429.Label_103.Pack
            ( A_Value => Output,
              A_Sdi   => Channel,
              A_Ssm   => Ssm,
              A_Discretes_Value => No_Discretes);
        end Label_103;

        ----------------------------------------
        -- Hydraulic Pressure
    Label_174 :

        declare
          Output                 : Force_Types.Psi;
          -- Label_174_Discrete     : A429.Label_174_Discrete_Type;
          Ssm                    : Arinc_429_Types.Bnr_Ssm_Type;
        begin
          -- Set Ouput label
          if Hyd_Press_Input_Hi >= 4095.0 then
            Output := 4095.0;
          elsif Hyd_Press_Input_Hi <= 0.0 then
            Output := 0.0;
          else
            Output := Hyd_Press_Input_Hi;
          end if;

          -- set SSM bits
          if  not Edm_Power or Edm_Fail then
            Ssm := Arinc_429_Types.Failure;
          elsif EDM_Test then
            Ssm := Arinc_429_Types.Functional_Test;
          elsif Hyd_Press_Input_Hi > 4095.0 then
            Ssm := Arinc_429_Types.No_Computed_Data;
          else
            Ssm := Arinc_429_Types.Normal;
          end if;

          Next_Instance.A429_TX_EDM_174_Hydraulic_Pressure(Channel) :=
            A429.Label_174.Pack
            ( A_Value           => Output,
              A_Sdi             => Channel,
              A_Ssm             => Ssm,
              A_Discretes_Value => No_Discretes);
        end Label_174;
        ----------------------------------------
        -- Cockpit Altitude
    Label_203 :

        declare
          Output                 : Length_Types.Feet;
          -- Label_203_Discrete     : A429.Label_203_Discrete_Type;
          -- Sdi                    : Edm_T.Edm_Bnr_Sdi_Type;
          Ssm                    : Arinc_429_Types.Bnr_Ssm_Type;
        begin
          -- Set Ouput label
          if Cockpit_Pressure_Altitude >= 40000.0 then
            Output :=  40000.0;
          elsif Cockpit_Pressure_Altitude <= -1000.0 then
            Output := -1000.0;
          else
            Output := Cockpit_Pressure_Altitude;
          end if;

          -- set SSM bits
          if  not Edm_Power or Edm_Fail then
            Ssm := Arinc_429_Types.Failure;
          elsif EDM_Test then
            SSM :=Arinc_429_Types.Functional_Test;
          elsif Cockpit_Pressure_Altitude > 40000.0  or Cockpit_Pressure_Altitude < -1000.0 then
            SSM := Arinc_429_Types.No_Computed_Data;
          else
            Ssm := Arinc_429_Types.Normal;
          end if;

          Next_Instance.A429_TX_EDM_203_Cockpit_Pressure_Altitude(Channel) :=
            A429.Label_203.Pack
            ( A_Value           => Output,
              A_Sdi             => Channel,
              A_Ssm             => Ssm,
              A_Discretes_Value => No_Discretes);
        end Label_203;
        ----------------------------------------
        -- Cockpit Differential Pressure
    Label_243 :

        declare
          Output                 : Force_Types.Psi;
          -- Label_243_Discrete     : A429.Label_243_Discrete_Type;
          -- Sdi                    : Edm_T.Edm_Bnr_Sdi_Type;
          Ssm                    : Arinc_429_Types.Bnr_Ssm_Type;
        begin
          -- Set Ouput label
          if Cockpit_Differential_Pressure >= 15.0 then
            Output := 15.0;
          elsif Cockpit_Differential_Pressure <= 0.0 then
            Output := 0.0;
          else
            Output := Cockpit_Differential_Pressure;
          end if;

          -- set SSM bits
          if  not Edm_Power or Edm_Fail then
            Ssm := Arinc_429_Types.Failure;
          elsif EDM_Test then
            SSM := Arinc_429_Types.Functional_Test;
          elsif Cockpit_Differential_Pressure > 5.0 then
            SSM := Arinc_429_Types.No_Computed_Data;
          else
            Ssm := Arinc_429_Types.Normal;
          end if;

          Next_Instance.A429_TX_EDM_243_Cockpit_Differential_Pressure(Channel) :=
            A429.Label_243.Pack
            ( A_Value           => Output,
              A_Sdi             => Channel,
              A_Ssm             => Ssm,
              A_Discretes_Value => No_Discretes);
        end Label_243;
        ----------------------------------------
        -- Fuel Flow transmission
    Label_244 :

        declare
          Output                 : Mass_Types.Pph;
          -- Label_244_Discrete     : A429.Label_244_Discrete_Type;
          -- Sdi                    : Edm_T.Edm_Bnr_Sdi_Type;
          Ssm                    : Arinc_429_Types.Bnr_Ssm_Type;
        begin
          -- Set Ouput label
          if Fuel_Flow_Input_Hi >= 1023.75 then
            Output := 1023.75;
          elsif Fuel_Flow_Input_Hi <= 25.0 then
            Output := 0.0;
          else
            Output := Fuel_Flow_Input_Hi;
          end if;

          -- set SSM bits
          if  not Edm_Power or Edm_Fail then
            Ssm := Arinc_429_Types.Failure;
          elsif EDM_Test then
            Ssm := Arinc_429_Types.Functional_Test;
          elsif Fuel_Flow_Input_Hi > 800.0 then
            Ssm := Arinc_429_Types.No_Computed_Data;
          else
            Ssm := Arinc_429_Types.Normal;
          end if;

          Next_Instance.A429_TX_EDM_244_Fuel_Flow(Channel) :=
            A429.Label_244.Pack
            ( A_Value           => Output,
              A_Sdi             => Channel,
              A_Ssm             => Ssm,
              A_Discretes_Value => No_Discretes);
        end Label_244;
        ----------------------------------------
        -- Left Fuel Quantity
    Label_256 :
        declare

          Output                 : Mass_Types.Lbm;
          -- Label_256_Discrete     : A429.Label_256_Discrete_Type;
          -- Sdi                    : Edm_T.Edm_Bnr_Sdi_Type;
          Ssm                    : Arinc_429_Types.Bnr_Ssm_Type;
        begin
          -- Set Ouput label
          if Next_Instance.Left_Fuel_Quantity >= 1023.75 then
            Output := 1023.75;
          elsif Next_Instance.Left_Fuel_Quantity <= 0.0 then
            Output := 0.0;
          else
            Output := Next_Instance.Left_Fuel_Quantity;
          end if;

          -- set SSM bits
          if  not Edm_Power or Edm_Fail then
            Ssm := Arinc_429_Types.Failure;
          elsif EDM_Test then
            Ssm := Arinc_429_Types.Functional_Test;
          elsif Next_Instance.Left_Fuel_Quantity > 800.0 then
            Ssm := Arinc_429_Types.No_Computed_Data;
          else
            Ssm := Arinc_429_Types.Normal;
          end if;

          Next_Instance.A429_TX_EDM_256_Left_Fuel_Quantity(Channel) :=
            A429.Label_256.Pack
            ( A_Value           => Output,
              A_Sdi             => Channel,
              A_Ssm             => Ssm,
              A_Discretes_Value => No_Discretes);

        end Label_256;

        ---------------------------------------------
        -- Right Fuel Quantity
    Label_257 :
        declare
          Output                 : Mass_Types.Lbm;
          -- Label_257_Discrete     : A429.Label_257_Discrete_Type;
          -- Sdi                    : Edm_T.Edm_Bnr_Sdi_Type;
          Ssm                    : Arinc_429_Types.Bnr_Ssm_Type;
        begin
          -- Set Ouput label
          if Next_Instance.Right_Fuel_Quantity >= 1023.75 then
            Output := 1023.75;
          elsif Next_Instance.Right_Fuel_Quantity <= 0.0 then
            Output := 0.0;
          else
            Output := Next_Instance.Right_Fuel_Quantity;
          end if;

          -- set SSM bits
          if  not Edm_Power or Edm_Fail then
            Ssm := Arinc_429_Types.Failure;
          elsif EDM_Test then
            Ssm :=Arinc_429_Types.Functional_Test;
          elsif Next_Instance.Right_Fuel_Quantity > 800.0 then
            Ssm := Arinc_429_Types.No_Computed_Data;
          else
            Ssm := Arinc_429_Types.Normal;
          end if;

          Next_Instance.A429_TX_EDM_257_Right_Fuel_Quantity(Channel) :=
            A429.Label_257.Pack
            ( A_Value           => Output,
              A_Sdi             => Channel,
              A_Ssm             => Ssm,
              A_Discretes_Value => No_Discretes);
        end Label_257;
        ----------------------------------------
        -- Raw Itt
    Label_301 :
        declare
          Output                 : Temperature_Types.Celsius;
          --               Label_301_Discrete     : A429.Label_301_Discrete_Type;
          --       Sdi                    : Edm_T.Edm_Bnr_Sdi_Type;
          Ssm                    : Arinc_429_Types.Bnr_Ssm_Type;
        begin
          -- Set Ouput label
          if Alternate_Itt_Input_Hi >= 1300.00 then
            Output := 1300.0;
          elsif Alternate_Itt_Input_Hi <= 0.0 then
            Output := 0.0;
          else
            Output := Alternate_Itt_Input_Hi;
          end if;

          -- set SSM bits
          if  not Edm_Power or Edm_Fail then
            Ssm := Arinc_429_Types.Failure;
          elsif EDM_Test then
            Ssm := Arinc_429_Types.Functional_Test;
          elsif Alternate_Itt_Input_Hi > 1200.0 or Alternate_Itt_Input_Hi < -70.0 then
            Ssm := Arinc_429_Types.No_Computed_Data;
          else
            Ssm := Arinc_429_Types.Normal;
          end if;

          Next_Instance.A429_TX_EDM_301_ITT(Channel) :=
            A429.Label_301.Pack
            ( A_Value           => Output,
              A_Sdi             => Channel,
              A_Ssm             => Ssm,
              A_Discretes_Value => No_Discretes);
        end Label_301;
        ----------------------------------------
        -- Alternate N1 in Percent
    Label_302 :
        declare
          Output                 : Angle_Types.Rpm;
          --               Label_302_Discrete     : A429.Label_302_Discrete_Type;
          --       Sdi                    : Edm_T.Edm_Bnr_Sdi_Type;
          Ssm                    : Arinc_429_Types.Bnr_Ssm_Type;
          Alt_N1                 : Normalized_Types.Normalize;
        begin
          -- Set Ouput label
          Alt_N1 := Alternate_N1_Input_Hi/374.68;

          if Alt_N1 >= 160.0  then
            Output := 160.0;
          elsif Alt_N1  <= 8.0 then
            Output := 0.0;
          else
            --               Output := 86.0;
            Output := Alt_N1;
          end if;

          -- set SSM bits
          if  not Edm_Power or Edm_Fail then
            Ssm := Arinc_429_Types.Failure;
          elsif EDM_Test then
            Ssm := Arinc_429_Types.Functional_Test;
          elsif Alt_N1 > 150.0 then
            Ssm := Arinc_429_Types.No_Computed_Data;
          else
            Ssm := Arinc_429_Types.Normal;
          end if;

          Next_Instance.A429_TX_EDM_302_N1(Channel) :=
            A429.Label_302.Pack
            ( A_Value           => Output,
              A_Sdi             => Channel,
              A_Ssm             => Ssm,
              A_Discretes_Value => No_Discretes);

        end Label_302;
        ----------------------------------------
        -- Oil Temperature
    Label_316 :
        declare
          Output                 : Temperature_Types.Celsius;
          --               Label_316_Discrete     : A429.Label_316_Discrete_Type;
          --       Sdi                    : Edm_T.Edm_Bnr_Sdi_Type;
          Ssm                    : Arinc_429_Types.Bnr_Ssm_Type;
        begin
          -- Set Ouput label
          if Oil_Temp_Signal_Hi >= 127.9688 then
            Output := 127.9688;
          elsif Oil_Temp_Signal_Hi <= -77.0 then
            Output := -77.0;
          else
            Output := Oil_Temp_Signal_Hi;
          end if;

          -- set SSM bits
          if  not Edm_Power or Edm_Fail then
            Ssm := Arinc_429_Types.Failure;
          elsif EDM_Test then
            Ssm := Arinc_429_Types.Functional_Test;
          elsif Oil_Temp_Signal_Hi > 128.0 or Oil_Temp_Signal_Hi < -70.0 then
            Ssm := Arinc_429_Types.No_Computed_Data;
          else
            Ssm := Arinc_429_Types.Normal;
          end if;

          Next_Instance.A429_TX_EDM_316_Oil_Temperature(Channel) :=
            A429.Label_316.Pack
            ( A_Value           => Output,
              A_Sdi             => Channel,
              A_Ssm             => Ssm,
              A_Discretes_Value => No_Discretes);

        end Label_316;
        ----------------------------------------
        -- Oil Pressure
    Label_317 :

        declare
          Output                 : Force_Types.Psi;
          --                Label_317_Discrete     : A429.Label_317_Discrete_Type;
          --        Sdi                    : Edm_T.Edm_Bnr_Sdi_Type;
          Ssm                    : Arinc_429_Types.Bnr_Ssm_Type;
        begin
          -- Set Ouput label
          if Oil_Press_Input_Hi >= 255.9375 then
            Output := 255.9375;
          elsif Oil_Press_Input_Hi <= 0.0 then
            Output := 0.0;
          else
            Output := Oil_Press_Input_Hi;
          end if;

          -- set SSM bits
          if  not Edm_Power or Edm_Fail then
            Ssm := Arinc_429_Types.Failure;
          elsif EDM_Test then
            Ssm := Arinc_429_Types.Functional_Test;
          elsif Oil_Press_Input_Hi >= 230.0 then
            Ssm := Arinc_429_Types.No_Computed_Data;
          else
            Ssm := Arinc_429_Types.Normal;
          end if;

          Next_Instance.A429_TX_EDM_317_Oil_Pressure(Channel) :=
            A429.Label_317.Pack
            ( A_Value           => Output,
              A_Sdi             => Channel,
              A_Ssm             => Ssm,
              A_Discretes_Value => No_Discretes);

        end Label_317;
        ----------------------------------------
        -- Diagnostics

        Next_Instance.Oil_Press_Lo_Signal     := ( (Oil_Press_Input_Hi < 40.0 and Power_Lever_Angle >= 32.0) or
                                                   (Oil_Press_Input_Hi < 15.0 and Power_Lever_Angle < 32.0));
        Next_Instance.Hyd_Press_Lo_Output     := Hyd_Press_Input_Hi < 2900.0;
        Next_Instance.Cockpit_Press_Hi_Output := Cockpit_Differential_Pressure > 3.9;
        Next_Instance.Cockpit_Alt_Hi_Output   := Cockpit_Pressure_Altitude > 19000.0;

        if Weight_On_Wheels then
          Next_Instance.Weight_On_Wheels_Timer := This_Instance.Weight_On_Wheels_Timer + Iconst;
          Next_Instance.L_Wheel_Speed_30 := L_Wheel_Spd_Sig_Hi > 30.0 or This_Instance.L_Wheel_Speed_30 or This_Instance.First_Pass;
          Next_Instance.R_Wheel_Speed_30 := R_Wheel_Spd_Sig_Hi > 30.0 or This_Instance.R_Wheel_Speed_30 or This_Instance.First_Pass;
        else
          This_Instance.Weight_On_Wheels_Timer := 0.0;
          Next_Instance.Weight_On_Wheels_Timer := 0.0;
          Next_Instance.L_Wheel_Speed_30 := False;
          This_Instance.L_Wheel_Speed_30 := False;
          Next_Instance.R_Wheel_Speed_30 := False;
          This_Instance.R_Wheel_Speed_30 := False;
        end if;

        Next_Instance.Nws_Enable := Next_Instance.Weight_On_Wheels_Timer > 5.0 and
          Next_Instance.L_Wheel_Speed_30 and
          Next_Instance.R_Wheel_Speed_30 and
          L_Wheel_Spd_Sig_Hi < 30.0 and
          R_Wheel_Spd_Sig_Hi < 30.0;

        ----------------------------------------------------
        -- Discrete Diagnostics

    Label_350 :
        declare
          Label_350_Discrete     : A429.Label_350_Discrete_Type;
          --       Sdi                    : Edm_T.Edm_Bnr_Sdi_Type;
          Ssm                    : Arinc_429_Types.Bnr_Ssm_Type;
        begin
          -- Set Ouput label

            Label_350_Discrete.Left_Ob_Probe_Fail            := L_Fuel_Probe_5_Signal_Malfunction;
            Label_350_Discrete.Left_Mid_Probe_Fail           := L_Fuel_Probe_3_Signal_Malfunction;
            Label_350_Discrete.Left_Ib_Probe_Fail            := L_Fuel_Probe_1_Signal_Malfunction;
            Label_350_Discrete.Right_Ob_Probe_Fail           := R_Fuel_Probe_6_Signal_Malfunction;
            Label_350_Discrete.Right_Mid_Probe_Fail          := R_Fuel_Probe_4_Signal_Malfunction;
            Label_350_Discrete.Right_Ib_Probe_Fail           := R_Fuel_Probe_2_Signal_Malfunction;
            Label_350_Discrete.Collector_Probe_Fail          := Collector_Probe_0_Signal_Malfunction;


          -- set SSM bits
          if  not Edm_Power or Edm_Fail then
            Ssm := Arinc_429_Types.Failure;
          elsif EDM_Test then
            Ssm := Arinc_429_Types.Functional_Test;
            --               elsif Fuel_Flow_Input_Hi >= 800.0 then
            --                 Ssm := Arinc_429_Types.No_Computed_Data;
          else
            Ssm := Arinc_429_Types.Normal;
          end if;

          Next_Instance.A429_TX_EDM_350_Discrete_Diagnostics(Channel) :=
            A429.Label_350.Pack
            ( A_Discretes_Value => Label_350_Discrete,
              A_Sdi             => Channel,
              A_Ssm             => Ssm );

        end Label_350;
        ----------------------------------------------------
        -- Engine Data Discretes

    Label_270 :
        declare
          Label_270_Discrete     : A429.Label_270_Discrete_Type;
          --       Sdi                    : Edm_T.Edm_Bnr_Sdi_Type;
          Ssm                    : Arinc_429_Types.Bnr_Ssm_Type;
        begin
          -- Set Ouput label

          Label_270_Discrete.Vm_Fail          := False;
          Label_270_Discrete.Vp_Fail          := False;
          Label_270_Discrete.Vdd_Fail         := False;
          Label_270_Discrete.Vin_Fail         := False;
          Label_270_Discrete.Edm_Channel_Id   := (Sdi = Edm_T.Channel_A);
          Label_270_Discrete.Anti_Ice         := True; -- Anti_Ice_On;
          Label_270_Discrete.Weight_On_Wheels := Weight_On_Wheels;

          -- set SSM bits
          if  not Edm_Power or Edm_Fail then
            Ssm := Arinc_429_Types.Failure;
          elsif EDM_Test then
            Ssm := Arinc_429_Types.Functional_Test;
            --               elsif Fuel_Flow_Input_Hi >= 800.0 then
            --                 Ssm := Arinc_429_Types.No_Computed_Data;
          else
            Ssm := Arinc_429_Types.Normal;
          end if;

          Next_Instance.A429_TX_EDM_270_Engine_Data_Discretes(Channel) :=
            A429.Label_270.Pack
            ( A_Discretes_Value => Label_270_Discrete,
              A_Sdi             => Channel,
              A_Ssm             => Ssm );

        end Label_270;
      end loop;
    Next_Instance.First_Pass := False;
  end Update;

  procedure Initialize
    (An_Instance : in out Instance) is
  begin
    An_Instance.First_Pass                                    := True;
    An_Instance.Auto_Balance_Fail_Output                      := False;
    An_Instance.Cockpit_Alt_Hi_Output                         := False;
    An_Instance.Cockpit_Press_Hi_Output                       := False;
    An_Instance.Hyd_Press_Lo_Output                           := False;
    An_Instance.Left_Transfer_Valve_Out                       := False;
    An_Instance.Right_Transfer_Valve_Out                      := False;
    An_Instance.Oil_Press_Lo_Signal                           := False;
    An_Instance.NWS_Enable                                    := False;
    An_Instance.Collector_Fuel_Quantity                       := Mass_Types.Lbm(40.0);
    An_Instance.Left_Fuel_Quantity                            := Mass_Types.Lbm(530.0);
    An_Instance.Right_Fuel_Quantity                           := Mass_Types.Lbm(530.0);
    An_Instance.Power_Out_Time                                := 0.0;
    An_Instance.Data_Available_Timer                          := 0.0;
    An_Instance.Data_Available                                := True;
    An_Instance.L_Wheel_Speed_30                              := False;
    An_Instance.R_Wheel_Speed_30                              := False;
    An_Instance.Normalized_Capacitance                        := 0.0;
    An_Instance.Weight_On_Wheels_Timer                        := 0.0;
    An_Instance.Auto_Balance_Mode                             := True;
    An_Instance.Auto_Balance_Mode_Timer_20                    := 0.0;
    An_Instance.Auto_Balance_Mode_Timer_30                    := 0.0;
  end Initialize;


end Engine_Data_Manager;
