-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                 JPATS T-6A Texan-II Flight Training Device
--
--
--  Engineer:  Keith H. Rehm
--
--  Revision:  (Number and date inserted by Clearcase)
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------

with Interpolation_Table.Singly_Indexed;
with Interpolation_Table.Doubly_Indexed;
with Simulation_Dictionary;
with Log;
with Ada.Numerics;

package body Power_Supply.Generator is

   Voltage_Table        : aliased Interpolation_Table.Doubly_Indexed.Instance;
   Starter_Load_Table   : aliased Interpolation_Table.Singly_Indexed.Instance;
   Starter_Torque_Table : aliased Interpolation_Table.Doubly_Indexed.Instance;

   procedure Update
     (An_Instance     : in out Instance;
      Starter_Engaged : in     Boolean;
      Voltage_In      : in     Electrical_Units_Types.Volts_Dc;
      Gen_Shaft_Rpm   : in     Angle_Types.Rpm)

   is

     package Ele_U  renames Electrical_Units_Types;
     package Single renames Interpolation_Table.Singly_Indexed;
     package Double renames Interpolation_Table.Doubly_Indexed;

     Voltage         : Ele_U.Volts_Dc      renames An_Instance.Voltage;
     Power_Available : Boolean             renames An_Instance.Power_Available;
     Starter_Load    : Ele_U.Amps          renames An_Instance.Starter_Load;
     Torque          : Torque_Types.Ft_Lbf renames An_Instance.Torque;
     Load            : Ele_U.Amps          renames An_Instance.Load;
     Rpm             : Angle_Types.Rpm     renames An_Instance.Rpm;

     TorqueL         : Torque_Types.Ft_Lbf := 0.0;

   begin

      Rpm := Gen_Shaft_Rpm;

      if Starter_Engaged then
         Voltage := 0.0;
      else
         Voltage := Ele_U.Volts_Dc(Double.Interpolate (Float(Rpm),Float(Load),Voltage_Table'Access));
      end if;

      Power_Available := Voltage > 5.0;

      if Starter_Engaged then
         Torque := Torque_Types.Ft_Lbf(Double.Interpolate (Float(Voltage_In),Float(Rpm), Starter_Torque_Table'Access));
         if Voltage_In /= 0.0 then
            Starter_Load := Ele_U.Amps(Single.Interpolate(Float(Rpm),Starter_Load_Table'Access));
         else
            Starter_Load := 0.0;
         end if;
      else

         if Rpm > 0.0 then
            TorqueL := Torque_Types.Ft_Lbf ( - (Float(Voltage) * Float(Load)) / ( 0.7 * 746.0 * Float(Rpm)));
         end if;

         Torque := TorqueL * 550.0 * 60.0 / (2.0 * Ada.Numerics.Pi);

         Starter_Load := 0.0;
      end if;


   exception
      when others =>
         Log.Report("Power_Supply.Generator.Update()");
         raise;
   end Update;

   procedure Initialize
     (An_Instance : in out Instance)
   is
   begin
      An_Instance.Voltage         := 0.0;
      An_Instance.Power_Available := False;
      An_Instance.Load            := 0.0;
      An_Instance.Rpm             := 0.0;
      An_Instance.Starter_Load    := 0.0;
      An_Instance.Torque          := 0.0;
   exception
      when others =>
         Log.Report("Power_Supply.Generator.Initialize()");
         raise;
   end Initialize;

   function Starter_Load
     (An_Instance : in Instance)
     return Electrical_Units_Types.Amps
   is
   begin
      return An_Instance.Starter_Load;
   exception
      when others =>
         Log.Report("Power_Supply.Generator.Starter_Load()");
         raise;
   end Starter_Load;

   function Torque
     (An_Instance : in Instance)
     return Torque_Types.Ft_Lbf
   is
   begin
      return An_Instance.Torque;
   exception
      when others =>
         Log.Report("Power_Supply.Generator.Torque()");
         raise;
   end Torque;

   procedure Read_Tables
   is
   begin
      Interpolation_Table.Read
        (File_Name => Simulation_Dictionary.Lookup ("Electrical_Dir") & "genvdc.ito",
         Table     => Voltage_Table);
      Interpolation_Table.Read
        (File_Name => Simulation_Dictionary.Lookup ("Electrical_Dir") & "strload.ito",
         Table     => Starter_Load_Table);
      Interpolation_Table.Read
        (File_Name => Simulation_Dictionary.Lookup ("Electrical_Dir") & "strtorq.ito",
         Table     => Starter_Torque_Table);
   exception
      when others =>
         Log.Report("Power_Supply.Generator.Read_Tables()");
         raise;
   end Read_Tables;

end Power_Supply.Generator;


