-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                 JPATS T-6A Texan-II Flight Training Device
--
--
--  Engineer:  Keith H. Rehm
--
--  Revision:  (Number and date inserted by Clearcase)
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------

with Interpolation_Table.Singly_Indexed;
with Interpolation_Table.Doubly_Indexed;
with Simulation_Dictionary;
with Log;

package body Power_Supply.Auxiliary_Battery is

   Voltage_Table        : aliased Interpolation_Table.Doubly_Indexed.Instance;
   Charge_Current_Table : aliased Interpolation_Table.Singly_Indexed.Instance;

   procedure Update
     (An_Instance             : in out Instance;
      Iconst                  : in     Float;
      Outside_Air_Temperature : in     Temperature_Types.Fahrenheit;
      Positive_Post_Voltage   : in     Electrical_Units_Types.Volts_Dc)
   is

      package Ele_U  renames Electrical_Units_Types;
      package Temp   renames Temperature_Types;
      package Single renames Interpolation_Table.Singly_Indexed;
      package Double renames Interpolation_Table.Doubly_Indexed;

      Capacity        : Float           renames An_Instance.Capacity;
      Load            : Ele_U.Amps      renames An_Instance.Load;
      Charge_Current  : Ele_U.Amps      renames An_Instance.Charge_Current;
      Is_Charging     : Boolean         renames An_Instance.Is_Charging;
      Voltage         : Ele_U.Volts_Dc  renames An_Instance.Voltage;
      Power_Available : Boolean         renames An_Instance.Power_Available;
      Temperature     : Temp.Fahrenheit renames An_Instance.Temperature;

      Oat : constant Temp.Fahrenheit := Outside_Air_Temperature;

   begin


     Is_Charging := Positive_Post_Voltage > Voltage + 0.2;
     if Is_Charging then
        Charge_Current := Single.Interpolate (Capacity,Charge_Current_Table'Access);
     else
        Charge_Current := 0.0;
     end if;

     Capacity := Float'Min(100.0,Float'Max(0.0,Capacity - (Load - Charge_Current) * (6.9444e-4 * Iconst)));

     Voltage := 24.0*Double.Interpolate (Capacity,Load,Voltage_Table'Access);

     Power_Available := Voltage > 5.0;

     Temperature := Temperature + (2.0e-4 * (Oat - Temperature + 7.0e-4 * (Capacity + Load)**2));

   exception
      when others =>
         Log.Report("Power_Supply.Auxiliary_Battery.Update()");
         raise;
   end Update;


   procedure Set_Capacity
     (An_Instance : in out Instance;
      C           : in     Float)
   is
      Capacity : Float renames An_Instance.Capacity;
   begin
      if C > 100.0 then
         capacity := 100.0;
      elsif C < 0.0 then
         Capacity := 0.0;
      else
         Capacity := C;
      end if;
   exception
      when others =>
         Log.Report("Power_Supply.Auxiliary_Battery.Set_Capacity()");
         raise;
   end Set_Capacity;

   function Is_Charging
     (An_Instance : in Instance)
     return Boolean
   is
   begin
      return An_Instance.Is_Charging;
   exception
      when others =>
         Log.Report("Power_Supply.Auxiliary_Battery.Is_Charging()");
         raise;
   end Is_Charging;

   procedure Read_Tables
   is
   begin
      Interpolation_Table.Read
        (File_Name => Simulation_Dictionary.Lookup ("Electrical_Dir") & "auxbatv.ito",
         Table     => Voltage_Table);
      Interpolation_Table.Read
        (File_Name => Simulation_Dictionary.Lookup ("Electrical_Dir") & "auxbatcc.ito",
         Table     => Charge_Current_Table);
   exception
      when others =>
         Log.Report("Power_Supply.Auxiliary_Battery.Read_Tables()");
         raise;
   end Read_Tables;

   procedure Initialize
     (An_Instance : in out Instance)
   is
   begin
      An_Instance.Voltage         := 24.0;
      An_Instance.Load            := 0.0;
      An_Instance.Power_Available := True;
      An_Instance.Is_Charging     := False;
      An_Instance.Capacity        := 100.0;
      An_Instance.Charge_Current  := 0.0;
      An_Instance.Temperature     := 71.0;
   exception
      when others =>
         Log.Report("Power_Supply.Auxiliary_Battery.Initialize()");
         raise;
   end Initialize;

end Power_Supply.Auxiliary_Battery;

