-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                 JPATS T-6A Texan-II Flight Training Device
--
--
--  Engineer:  Keith H. Rehm
--
--  Revision:  (Number and date inserted by Clearcase)
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------

with Jpats_Electrical.Container;
with Power_Supply.Battery;
with Power_Supply.Generator;
with Power_Supply.External_Power;
with Electrical_Units_Types;
with Ada.Numerics.Float_Random;
with Log;
with Jpats_Powerplant;

package body Jpats_Electrical.Voltmeter_Controller is

   EXT_PWR_START_LOAD           : constant Float := 6.0;
   Batt_Start_Ext_Pwr_Last_Pass : Boolean := False;
   Save_Load                    : Float := 0.0;
   Start_Timer                  : Integer := 0;

   procedure Initialize
   is
      package Ctnr   renames Container;
      package Vm     renames Ctnr.Voltmeter;
      The_Vm : Vm.Instance renames Ctnr.This_Subsystem.The_Voltmeter;
   begin
      Vm.Initialize (The_Vm);
   exception
      when others =>
         Log.Report ("Jpats_Electrical.Voltmeter_Controller.Initialize()");
         raise;
   end Initialize;

   Noise          : Float             renames Container.This_Subsystem.VM_Noise;
   Noise_Tmr      : Float             renames Container.This_Subsystem.VM_Noise_Tmr;
   Noise_Duration : Float             renames Container.This_Subsystem.VM_Noise_Duration;
   Five_Min_Timer : Float             renames Container.This_Subsystem.VM_Five_Min_Timer;
   Battery_Charging_Off_Gen : Boolean renames Container.This_Subsystem.VM_Battery_Charging_Off_Gen;

   procedure Update
     (Iconst : in Float)
   is
      use Ada.Numerics.Float_Random;
      G : Generator;

      package Ctnr   renames Container;
      package Ele    renames Jpats_Electrical_Types;
      package Ele_U  renames Electrical_Units_Types;
      package Relays renames Ctnr.Coil_Relay_Collection;
      package Vm     renames Ctnr.Voltmeter;

      The_Batt    : Power_Supply.Battery.Instance        renames Ctnr.This_Subsystem.The_Battery;
      The_Gen     : Power_Supply.Generator.Instance      renames Ctnr.This_Subsystem.The_Generator;
      The_Ext_Pwr : Power_Supply.External_Power.Instance renames Ctnr.This_Subsystem.The_External_Power;
      The_Relays  : Relays.Instance                      renames Ctnr.This_Subsystem.The_Relays;
      The_Vm      : Vm.Instance                          renames Ctnr.This_Subsystem.The_Voltmeter;

      Batt_Voltage    : constant Ele_U.Volts_Dc := Power_Supply.Voltage (The_Batt);
      Gen_Voltage     : constant Ele_U.Volts_Dc := Power_Supply.Voltage (The_Gen);
      Ext_Pwr_Voltage : constant Ele_U.Volts_Dc := Power_Supply.Voltage (The_Ext_Pwr);

      Batt_Power_Available    : constant Boolean := Power_Supply.Power_Available (The_Batt);
      Gen_Power_Available     : constant Boolean := Power_Supply.Power_Available (The_Gen);
      Ext_Pwr_Power_Available : constant Boolean := Power_Supply.Power_Available (The_Ext_Pwr);

      Batt_Load    : constant Ele_U.Amps := Power_Supply.Load (The_Batt);
      Gen_Load     : constant Ele_U.Amps := Power_Supply.Load (The_Gen);
      Ext_Pwr_Load : constant Ele_U.Amps := Power_Supply.Load (The_Ext_Pwr);

      Batt_Rly_Closed    : constant Boolean :=
        Relays.Is_Closed
        (An_Instance => The_Relays,
         Relay_Name  => Ele.Battery_Relay);

      Ext_Pwr_Rly_Closed : constant Boolean :=
        Relays.Is_Closed
        (An_Instance => The_Relays,
         Relay_Name  => Ele.External_Power_Relay);

      Bus_Tie_Rly_Closed : constant Boolean :=
        Relays.Is_Closed
        (An_Instance => The_Relays,
         Relay_Name  => Ele.Bus_Tie_Relay);

      Gen_Rly_Closed     : constant Boolean :=
        Relays.Is_Closed
        (An_Instance => The_Relays,
         Relay_Name  => Ele.Generator_Relay);

      Voltmeter_Voltage : Ele_U.Volts_Dc := 0.0;
      Voltmeter_Load    : Ele_U.Amps     := 0.0;

      Batt_Path    : constant Boolean := Batt_Power_Available and then Batt_Rly_Closed;
      Gen_Path     : constant Boolean := Gen_Power_Available and then Gen_Rly_Closed and then Bus_Tie_Rly_Closed;
      Ext_Pwr_Path : constant Boolean := Ext_Pwr_Power_Available and then Ext_Pwr_Rly_Closed;

      Io  : Ctnr.Io_Interface_Instance renames Ctnr.This_Io_Interface;
      Ios : Ctnr.Ios_Interface_Instance renames Ctnr.This_Ios_Interface;

      Battery_Charge_Current : constant Ele_U.Amps := Power_Supply.Battery.Charge_Current(The_Batt);

      Voltmeter_Load_Junk : Ele_U.Amps     := 0.0;
      Voltage_Offset : Ele_U.Volts_Dc := 0.0;


      Ext_Pwr_Rly        : constant Boolean := Relays.Is_Closed (The_Relays,Ele.External_Power_Relay);
      Batt_Rly           : constant Boolean := Relays.Is_Closed (The_Relays,Ele.Battery_Relay);
      Bus_Tie_Rly        : constant Boolean := Relays.Is_Closed (The_Relays,Ele.Bus_Tie_Relay);
      Gen_Rly            : constant Boolean := Relays.Is_Closed (The_Relays,Ele.Generator_Relay);
      
      Str_Rly            : constant Boolean := Jpats_Powerplant.Start_Relay;
      Pwr_From_Ext_Pwr   : constant Boolean := Str_Rly and then (Ext_Pwr_Power_Available and Ext_Pwr_Rly);
      Pwr_From_Batt      : constant Boolean := Str_Rly and then (Batt_Power_Available and Batt_Rly);
      Pwr_From_Gen       : constant Boolean := Str_Rly and then (Gen_Power_Available and Gen_Rly and Bus_Tie_Rly);
      Battery_Start      : constant Boolean := Pwr_From_Batt and not (Pwr_From_Ext_Pwr or Pwr_From_Gen);
      Batt_Start_Ext_Pwr : constant Boolean := Pwr_From_Batt and Pwr_From_Ext_Pwr and not Pwr_From_Gen;

   begin

      -- find voltage to be displayed
      if Ext_Pwr_Path then
         Voltmeter_Voltage := Ext_Pwr_Voltage;
      end if;

      if Batt_Path then
         Voltage_Offset := 4.0*(Power_Supply.Generator.Starter_Load (The_Gen)/200.0);
         if Voltage_Offset > 4.0 then Voltage_Offset := 4.0; end if;
         Voltmeter_Voltage := Ele_U.Volts_Dc'Max (Voltmeter_Voltage,Batt_Voltage - Voltage_Offset);
      end if;

      if Gen_Path then
         Voltmeter_Voltage := Ele_U.Volts_Dc'Max (Voltmeter_Voltage,Gen_Voltage);
      end if;

      -- find load to be displayed
      if Gen_Path then
         Five_Min_Timer := Float'Min(300.0, Five_Min_Timer + Iconst);
      end if;

      if Batt_Path and not Ext_Pwr_Path and not Gen_Path then
         Voltmeter_Load := -Batt_Load;
      elsif Batt_Path and Ext_Pwr_Path and not Gen_Path then

         -- If this is a battery only start with external power supplied then
         -- add a 6 amp load when starter is engaged (as seen in video from aircraft)
         -- then return to starting load amperage after 5 seconds.
         if Batt_Start_Ext_Pwr then

            if not Batt_Start_Ext_Pwr_Last_Pass then
               Batt_Start_Ext_Pwr_Last_Pass := true;
               Start_Timer := 0;
               if Ext_Pwr_Voltage > Batt_Voltage then
                  Voltmeter_load := -Ext_Pwr_Load;
               else
                  Voltmeter_Load := -Batt_Load;
               end if;
               Save_Load := Voltmeter_Load + EXT_PWR_START_LOAD;
            else
               if Start_Timer < 75 then -- 5 seconds
                  Start_Timer := Start_Timer + 1;
                  Voltmeter_Load := Save_Load;
               else
                  Voltmeter_Load := Save_Load - EXT_PWR_START_LOAD;
               end if;
            end if;

         else

            if Batt_Start_Ext_Pwr_Last_Pass then
               Batt_Start_Ext_Pwr_Last_Pass := False;
            end if;
   
            if Ext_Pwr_Voltage > Batt_Voltage then
               Voltmeter_Load := -Ext_Pwr_Load;
            else
               Voltmeter_Load := -Batt_Load;
            end if;

         end if;

      elsif Batt_Path and Gen_Path and not Ext_Pwr_Path then

         if Gen_Voltage > Batt_Voltage then
            Voltmeter_Load := Battery_Charge_Current;
            Battery_Charging_Off_Gen := True;
         else
            Voltmeter_Load := -Batt_Load;
         end if;

      elsif not Batt_Path and Gen_Path and not Ext_Pwr_Path then

         Voltmeter_Load := 0.0;

      elsif not Batt_Path and Ext_Pwr_Path then -- should not exist
         Voltmeter_Load := -Ext_Pwr_Load;
      end if;

      Noise_Tmr := Noise_Tmr + Iconst;
      if Noise_Tmr >= Noise_Duration then
         Reset (G);
         Noise_Tmr := 0.0;
         Noise_Duration := 2.5 * Random (G);
         Noise := -5.0 + 10.0 * Random (G);
      end if;

      if Battery_Charging_Off_Gen then
         if Noise > 2.0 then
            Noise := 2.0;
         elsif Noise < -2.0 then
            Noise := -2.0;
         end if;
         if Battery_Charge_Current < 10.0 then
            Noise := Noise * Battery_Charge_Current / 10.0;
         end if;
         Voltmeter_Load_Junk := abs(Voltmeter_Load + Ele_U.Amps (Noise));
         if Voltmeter_Load_Junk < 2.0 then
            Voltmeter_Load := 2.0;
         else
            Voltmeter_Load := Voltmeter_Load_Junk;
         end if;
      else
         if Batt_Path then
            Voltmeter_Load := Voltmeter_Load + Ele_U.Amps (Noise);
         end if;
      end if;

      Battery_Charging_Off_Gen := False;

      -- update voltmeter
      Vm.Update (The_Vm, Voltmeter_Voltage, Voltmeter_Load);

      -- drive instrument
      Ios.Display_Voltage := Float (Voltmeter_Voltage);
      Ios.Display_Load    := Float (Voltmeter_Load);

   exception
      when others =>
         Log.Report ("Jpats_Electrical.Voltmeter_Controller.Update()");
         raise;
   end Update;

end Jpats_Electrical.Voltmeter_Controller;
