-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                 JPATS T-6A Texan-II Flight Training Device
--
--
--  Engineer:  Keith H. Rehm
--
--  Revision:  (Number and date inserted by Clearcase)
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------

with Jpats_Electrical.Container;
with Power_Supply;
with Power_Supply.Auxiliary_Battery;
with Power_Supply.Battery;
with Power_Supply.External_Power;
with Power_Supply.Generator;
with JPATS_Reposition;
with Jpats_Aircraft_Body;
with Jpats_Atmosphere;
with Jpats_Powerplant;
with Jpats_Simulated_Aircraft;
with Length_Types;
with Angle_Types;
with Electrical_Units_Types;
with Temperature_Types;
with Interpolation_Table.Singly_Indexed;
with Simulation_Dictionary;
with Log;

package body Jpats_Electrical.Power_Supply_Controller is

   Batt_Capacity_Table : aliased Interpolation_Table.Singly_Indexed.Instance;
   Starting            : Boolean := False;

   procedure Update_Sources_Loads
     (Iconst : in Float)
   is

      package Ctnr     renames Container;
      package Ele      renames Jpats_Electrical_Types;
      package Ele_U    renames Electrical_Units_Types;
      package Batt     renames Power_Supply.Battery;
      package Gen      renames Power_Supply.Generator;
      package Aux_Batt renames Power_Supply.Auxiliary_Battery;
      package Ext_Pwr  renames Power_Supply.External_Power;
      package Buses    renames Ctnr.Bus_Collection;
      package Paths    renames Ctnr.Path_Collection;

      Ios : Ctnr.Ios_Interface_Instance renames Ctnr.This_Ios_Interface;

      The_Batt     : Batt.Instance     renames Ctnr.This_Subsystem.The_Battery;
      The_Gen      : Gen.Instance      renames Ctnr.This_Subsystem.The_Generator;
      The_Aux_Batt : Aux_Batt.Instance renames Ctnr.This_Subsystem.The_Auxiliary_Battery;
      The_Ext_Pwr  : Ext_Pwr.Instance  renames Ctnr.This_Subsystem.The_External_Power;

      The_Buses : Buses.Instance renames Ctnr.This_Subsystem.The_Buses;
      The_Paths : Paths.Instance renames Ctnr.This_Subsystem.The_Paths;

      Bus_Loads        : array (ele.Bus_Name_Type) of Ele_U.Amps := (others => 0.0);
      Pwr_Source_Bus_Loads : array (Ctnr.Power_Source_Name_Type) of Ele_U.Amps := (others => 0.0);
      Pwr_Source_Loads : array (Ctnr.Power_Source_Name_Type) of Ele_U.Amps := (others => 0.0);

      Ksh : Electrical_Units_Types.Volts_Dc;

      use type Ctnr.Power_Source_Name_Type;

   begin

      for Bus in ele.Bus_Name_Type
      loop
         Bus_Loads (Bus) :=
           Buses.Voltage (The_Buses, Bus) * 3.50877e-2 * Buses.Load (The_Buses, Bus);

         Ksh := 0.0;
         for Pwr_Source in Ctnr.Power_Source_Name_Type
         loop
            if Pwr_Source = Ctnr.Battery then
               if Paths.Is_Complete (The_Paths,Pwr_Source,Bus) and not
                 Batt.Is_Charging (The_Batt) then
                  Ksh := Ksh + Paths.Voltage (The_Paths,Pwr_Source,Bus);
               end if;
            else
               if Paths.Is_Complete (The_Paths,Pwr_Source,Bus) then
                  Ksh := Ksh + Paths.Voltage (The_Paths,Pwr_Source,Bus);
               end if;
            end if;
            if Ksh > 0.0 then
               Pwr_Source_Bus_Loads (Pwr_Source) :=
                 (Paths.Voltage (The_Paths,Pwr_Source,Bus) / Ksh) * Bus_Loads (Bus);
            else
               Pwr_Source_Bus_Loads (Pwr_Source) := 0.0;
            end if;
            Pwr_Source_Loads (Pwr_Source) :=
              Pwr_Source_Loads (Pwr_Source) + Pwr_Source_Bus_Loads (Pwr_Source);
         end loop;
      end loop;

      Power_Supply.Set_Load (The_Batt, Pwr_Source_Loads (Ctnr.Battery));
      Power_Supply.Set_Load (The_Aux_Batt, Pwr_Source_Loads (Ctnr.Auxiliary_Battery));
      Power_Supply.Set_Load (The_Gen, Pwr_Source_Loads (Ctnr.Generator));
      Power_Supply.Set_Load (The_Ext_Pwr, Pwr_Source_Loads (Ctnr.External_Power));

      Ios.BattLoad    := Pwr_Source_Loads (Ctnr.Battery);
      Ios.GenLoad     := Pwr_Source_Loads (Ctnr.Generator);
      Ios.AuxBattLoad := Pwr_Source_Loads (Ctnr.Auxiliary_Battery);
      Ios.ExtPwrLoad  := Pwr_Source_Loads (Ctnr.External_Power);


   exception
      when others =>
         Log.Report("Jpats_Electrical.Power_Supply_Controller.Update_Sources_Loads()");
         raise;
   end Update_Sources_Loads;


--   Low_Battery_Voltage_Malf_Lp : Boolean := False;

   procedure Update_Battery
     (Iconst : in Float)
   is

      package Ctnr  renames Container;
      Package Ele   renames Jpats_Electrical_Types;
      package Ele_U renames Electrical_Units_Types;
      package Atm   renames Jpats_Atmosphere;
      package Temp  renames Temperature_Types;
      package Batt  renames Power_Supply.Battery;
      package Buses renames Ctnr.Bus_Collection;
      package Cbs   renames Ctnr.Circuit_Breaker_Collection;

      package Relays  renames Ctnr.Coil_Relay_Collection;
      package Pwr_Src renames Power_Supply;
      package Gen     renames Power_Supply.Generator;
      package Ext_Pwr renames Power_Supply.External_Power;

      package Single renames Interpolation_Table.Singly_Indexed;

      The_Batt  : Batt.Instance  renames Ctnr.This_Subsystem.The_Battery;
      The_Buses : Buses.Instance renames Ctnr.This_Subsystem.The_Buses;
      The_Cbs   : Cbs.Instance renames Ctnr.This_Subsystem.The_Circuit_Breakers;

      Hot_Batt_Bus_Voltage : constant Ele_U.Volts_Dc := Buses.Voltage (The_Buses,ele.Hot_Battery_Bus);

      Oat : Temp.Rankine := Temp.Rankine_To_Fahrenheit (Atm.Temperature);

      Io : Ctnr.Io_Interface_Instance renames Ctnr.This_Io_Interface;

      Ios : Ctnr.Ios_Interface_Instance renames Ctnr.This_Ios_Interface;

      Host : Ctnr.Host_Interface_Instance renames Ctnr.This_Host_Interface;

      Batt_Sw_Cb : constant Boolean := Cbs.Is_Powered(The_Cbs,Ele.Batt_Sw_Cb);

      Low_Battery_Voltage_Malf_Lp : Boolean renames Ctnr.This_Subsystem.Low_Battery_Voltage_Malf_Lp;

      Batt_Bus_Light_Start_Malf_Lp : Boolean renames Ctnr.This_Subsystem.Batt_Bus_Light_Start_Malf_Lp;

      Batcap : Float := 0.0;

      -- TSCO 15-175 NEW CODE NEEDED TO DETERMINE IF WE ARE IN A BATTERY START ATTEMPT
      The_Relays         : Relays.Instance  renames Ctnr.This_Subsystem.The_Relays;
      Ext_Pwr_Rly        : constant Boolean := Relays.Is_Closed (The_Relays,Ele.External_Power_Relay);
      Batt_Rly           : constant Boolean := Relays.Is_Closed (The_Relays,Ele.Battery_Relay);
      Bus_Tie_Rly        : constant Boolean := Relays.Is_Closed (The_Relays,Ele.Bus_Tie_Relay);
      Gen_Rly            : constant Boolean := Relays.Is_Closed (The_Relays,Ele.Generator_Relay);
      The_Gen            : Gen.Instance     renames Ctnr.This_Subsystem.The_Generator;
      The_Ext_Pwr        : Ext_Pwr.Instance renames Ctnr.This_Subsystem.The_External_Power;
      Ext_Pwr_Available  : constant Boolean := Pwr_Src.Power_Available (The_Ext_Pwr);
      Batt_Pwr_Available : constant Boolean := Pwr_Src.Power_Available (The_Batt);
      Gen_Pwr_Available  : constant Boolean := Pwr_Src.Power_Available (The_Gen);
      Str_Rly            : constant Boolean := Jpats_Powerplant.Start_Relay;
      Pwr_From_Ext_Pwr   : constant Boolean := Str_Rly and then (Ext_Pwr_Available and Ext_Pwr_Rly);
      Pwr_From_Batt      : constant Boolean := Str_Rly and then (Batt_Pwr_Available and Batt_Rly);
      Pwr_From_Gen       : constant Boolean := Str_Rly and then (Gen_Pwr_Available and Gen_Rly and Bus_Tie_Rly);
      Battery_Start      : constant Boolean := Pwr_From_Batt and not (Pwr_From_Ext_Pwr or Pwr_From_Gen);

   begin

      -- Check to see if we have starter engage from a battery only start.
      -- Used for Batt Bus Light during Battery Start EP.
      if Battery_Start then
         Starting := true;
      end if;

      if Ios.Battery_Reset or Host.Battery_Reset_Command then
         Batt.Set_Capacity (The_Batt, 100.0);
         Ios.Battery_Reset := False;
         Host.Battery_Reset_Command := False;
      end if;

      -- If EP just now activated
      if (Ios.Low_Battery_Voltage and not Low_Battery_Voltage_Malf_Lp) or
          (Ios.Batt_Bus_Light_Start and not Batt_Bus_Light_Start_Malf_Lp) then
         Batcap := Single.Interpolate(Power_Supply.Load(The_Batt),Batt_Capacity_Table'Access);
         Batt.Set_Capacity (The_Batt, Batcap); 
      elsif (not Ios.Low_Battery_Voltage and Low_Battery_Voltage_Malf_Lp) or
            (not Ios.Batt_Bus_Light_Start and Batt_Bus_Light_Start_Malf_Lp) then
         Batt.Set_Capacity (The_Batt, 100.0);
         Starting := False;
      end if;

      Low_Battery_Voltage_Malf_Lp := Ios.Low_Battery_Voltage;
      Batt_Bus_Light_Start_Malf_Lp := Ios.Batt_Bus_Light_Start;

      Batt.Update
        (An_Instance             => The_Batt,
         Iconst                  => Iconst,
         Outside_Air_Temperature => Oat,
         Positive_Post_Voltage   => Hot_Batt_Bus_Voltage,
         Batt_Bus_Light_Start    => Ios.Batt_Bus_Light_Start and Starting);

      Io.Master_Bat_On := Batt_Sw_Cb;

      Ios.BattIsCharging := Batt.Is_Charging(The_Batt);

   exception
      when others =>
         Log.Report("Jpats_Electrical.Power_Supply_Controller.Update_Battery()");
         raise;
   end Update_Battery;

   procedure Update_Generator
     (Iconst : in Float)
   is

      package Ctnr    renames Container;
      package Ele     renames Jpats_Electrical_Types;
      package Pwr_Src renames Power_Supply;
      package Gen     renames Power_Supply.Generator;
      package Batt    renames Power_Supply.Battery;
      package Ext_Pwr renames Power_Supply.External_Power;
      package Relays  renames Ctnr.Coil_Relay_Collection;

      Ios : Ctnr.Ios_Interface_Instance renames Ctnr.This_Ios_Interface;

      The_Gen     : Gen.Instance     renames Ctnr.This_Subsystem.The_Generator;
      The_Batt    : Batt.Instance    renames Ctnr.This_Subsystem.The_Battery;
      The_Ext_Pwr : Ext_Pwr.Instance renames Ctnr.This_Subsystem.The_External_Power;
      The_Relays  : Relays.Instance  renames Ctnr.This_Subsystem.The_Relays;

      Ext_Pwr_Available  : constant Boolean := Pwr_Src.Power_Available (The_Ext_Pwr);
      Batt_Pwr_Available : constant Boolean := Pwr_Src.Power_Available (The_Batt);
      Gen_Pwr_Available  : constant Boolean := Pwr_Src.Power_Available (The_Gen);

      Str_Rly : constant Boolean := Jpats_Powerplant.Start_Relay;

      Ext_Pwr_Rly : constant Boolean := Relays.Is_Closed (The_Relays,Ele.External_Power_Relay);
      Batt_Rly    : constant Boolean := Relays.Is_Closed (The_Relays,Ele.Battery_Relay);
      Bus_Tie_Rly : constant Boolean := Relays.Is_Closed (The_Relays,Ele.Bus_Tie_Relay);
      Gen_Rly     : constant Boolean := Relays.Is_Closed (The_Relays,Ele.Generator_Relay);

      Pwr_From_Ext_Pwr : constant Boolean := Str_Rly and then (Ext_Pwr_Available and Ext_Pwr_Rly);
      Pwr_From_Batt    : constant Boolean := Str_Rly and then (Batt_Pwr_Available and Batt_Rly);
      Pwr_From_Gen     : constant Boolean := Str_Rly and then (Gen_Pwr_Available and Gen_Rly and Bus_Tie_Rly);

      Str_Engaged : constant Boolean := Pwr_From_Batt or Pwr_From_Ext_Pwr or Pwr_From_Gen;

      Shaft_Rpm : constant Angle_Types.Rpm := Jpats_Powerplant.Generator_Shaft_Rpm;

      Vdc_Into_Gen : Electrical_Units_Types.Volts_Dc := 0.0;

   begin

      if Pwr_From_Ext_Pwr then
         Vdc_Into_Gen := Pwr_Src.Voltage (The_Ext_Pwr);
      end if;
      if Pwr_From_Batt then
         Vdc_Into_Gen := Electrical_Units_Types.Volts_Dc'Max(Pwr_Src.Voltage (The_Batt), Vdc_Into_Gen);
      end if;
      if Pwr_From_Gen then
         Vdc_Into_Gen := Electrical_Units_Types.Volts_Dc'Max(Pwr_Src.Voltage (The_Gen), Vdc_Into_Gen);
      end if;

      Gen.Update
        (An_Instance     => The_Gen,
         Starter_Engaged => Str_Engaged,
         Voltage_In      => Vdc_Into_Gen,
         Gen_Shaft_Rpm   => Shaft_Rpm);

--      Ios.StarterLoad := Gen.Starter_Load(The_Gen);
      Ios.StarterLoad
        := Ctnr.Bus_Collection.Voltage(Ctnr.This_Subsystem.The_Buses,
                                       Jpats_Electrical_Types.Starter_Distribution_Bus);

   exception
      when others =>
         Log.Report("Jpats_Electrical.Power_Supply_Controller.Update_Generator()");
         raise;
   end Update_Generator;

   procedure Update_Generator_Control_Unit
     (Iconst : in Float)
   is

      package Ctnr    renames Container;
      package Ele     renames Jpats_Electrical_Types;
      package Ele_U   renames Electrical_Units_Types;
      package Cbs     renames Ctnr.Circuit_Breaker_Collection;
      package Gcu     renames Ctnr.Generator_Control_Unit;
      package Pwr_Src renames Power_Supply;
      package Batt    renames Pwr_Src.Battery;
      package Gen     renames Pwr_Src.Generator;
      package Buses   renames Ctnr.Bus_Collection;

      Io : Ctnr.Io_Interface_Instance renames Ctnr.This_Io_Interface;
      Ios : Ctnr.Ios_Interface_Instance renames Ctnr.This_Ios_Interface;

      The_Gcu : Gcu.Instance renames Ctnr.This_Subsystem.The_Generator_Control_Unit;
      The_Cbs : Cbs.Instance renames Ctnr.This_Subsystem.The_Circuit_Breakers;

      The_Buses : Buses.Instance renames Ctnr.This_Subsystem.The_Buses;

      The_Gen         : Gen.Instance   renames Ctnr.This_Subsystem.The_Generator;
      The_Gen_Voltage : Ele_U.Volts_Dc renames Pwr_Src.Voltage (The_Gen);
      The_Gen_Load    : Ele_U.Amps     renames Pwr_Src.Load (The_Gen);

      Temp_Boolean : Boolean    := False;
      Temp_Load    : Ele_U.Amps := 0.0;

      Str_Rly_Open : constant Boolean := not Jpats_Powerplant.Start_Relay;

      Str_Cb       : constant Boolean := Cbs.Is_Powered (The_Cbs, Ele.Start_Cb);
      Bus_Sense_Cb : constant Boolean := Cbs.Is_Powered (The_Cbs, Ele.Bus_Sense_Cb);
      Gen_Sw_Cb    : constant Boolean := Cbs.Is_Powered (The_Cbs, Ele.Gen_Sw_Cb);
      Gen_Sense_Cb : constant Boolean := Cbs.Is_Powered (The_Cbs, Ele.Generator_Sense_Cb);

      Reverse_Current : Ele_U.Amps     := 0.0;
      Voltage         : Ele_U.Volts_Dc := 0.0;
      Load            : Ele_U.Amps     := 0.0;

      Line_Control : constant Boolean := Gcu.Line_Control (The_Gcu);

      Reset_In : constant Boolean := Io.Gen_Reset_Sw and Str_Rly_Open and Gen_Sense_Cb;

      Volts_In : Ele_U.Volts_Dc := 0.0;

   begin

      if Io.Master_Gen_On_Sw and not Io.Gen_Reset_Sw and Str_Rly_Open and Gen_Sense_Cb then
         Volts_In := The_Gen_Voltage;
      end if;

      Gcu.Update
        (An_Instance        => The_Gcu,
         Iconst             => Iconst,
         Voltage_In         => Volts_In,
         Bus_Sense_Voltage  => Buses.Voltage (The_Buses, ele.Generator_Bus),
         Current_In         => The_Gen_Load,
         Reverse_Current_In => Reverse_Current,
         Gen_Reset_In       => Reset_In,
         Fail_Demanded      => Ios.Gcu_Fail);

      Io.Master_Gen_On := Str_Cb or Gen_Sw_Cb;

   exception
      when others =>
         Log.Report("Jpats_Electrical.Power_Supply_Controller.Update_Generator_Control_Unit()");
         raise;
   end Update_Generator_Control_Unit;


--   Dead_Aux_Batt_Commanded_Last_Pass : Boolean := False;
   procedure Update_Auxiliary_Battery
     (Iconst : in Float)
   is

      package Ele          renames Jpats_Electrical_Types;
      package Ele_U        renames Electrical_Units_Types;
      package Ctnr         renames Container;
      package Aux_Batt     renames Power_Supply.Auxiliary_Battery;
      package Temp         renames Temperature_Types;
      package Atm          renames Jpats_Atmosphere;
      package Cbs          renames Ctnr.Circuit_Breaker_Collection;
      package Buses        renames Ctnr.Bus_Collection;
      package Aux_Batt_Mnt renames Ctnr.Auxiliary_Battery_Mount;

      Ios : Ctnr.Ios_Interface_Instance renames Ctnr.This_Ios_Interface;

      The_Cbs          : Cbs.Instance          renames Ctnr.This_Subsystem.The_Circuit_Breakers;
      The_Buses        : Buses.Instance        renames Ctnr.This_Subsystem.The_Buses;
      The_Aux_Batt     : Aux_Batt.Instance     renames Ctnr.This_Subsystem.The_Auxiliary_Battery;
      The_Aux_Batt_Mnt : Aux_Batt_Mnt.Instance renames Ctnr.This_Subsystem.The_Auxiliary_Battery_Mount;

      Oat : Temp.Fahrenheit := Temp.Rankine_To_Fahrenheit (Atm.Temperature);

      Volts_In : Ele_U.Volts_Dc := Aux_Batt_Mnt.Voltage_In (The_Aux_Batt_Mnt);

      Dead_Aux_Batt_Commanded_Last_Pass : Boolean renames Ctnr.This_Subsystem.Dead_Aux_Batt_Commanded_Last_Pass;

   begin

      if Ios.Dead_Aux_Batt then
         Aux_Batt.Set_Capacity(The_Aux_Batt,0.0);
      end if;

      if not Ios.Dead_Aux_Batt and Dead_Aux_Batt_Commanded_Last_Pass then
         Aux_Batt.Set_Capacity(The_Aux_Batt,100.0);
      end if;

      Dead_Aux_Batt_Commanded_Last_Pass := Ios.Dead_Aux_Batt;


      Aux_Batt.Update
        (An_Instance             => The_Aux_Batt,
         Iconst                  => Iconst,
         Outside_Air_Temperature => Oat,
         Positive_Post_Voltage   => Volts_In);

      Ios.AuxBattIsCharging := Aux_Batt.Is_Charging(The_Aux_Batt);

   exception
      when others =>
         Log.Report("Jpats_Electrical.Power_Supply_Controller.Update_Auxiliary_Battery()");
         raise;
   end Update_Auxiliary_Battery;

   procedure Update_Auxiliary_Battery_Mount
     (Iconst : in Float)
   is

      Temp_Voltage : Electrical_Units_Types.Volts_Dc := 0.0;
      Temp_Boolean : Boolean := False;

      package Ele          renames Jpats_Electrical_Types;
      package Ele_U        renames Electrical_Units_Types;
      package Ctnr         renames Container;
      package Aux_Batt_Mnt renames Ctnr.Auxiliary_Battery_Mount;
      package Pwr_Src      renames Power_Supply;
      package Aux_Batt     renames Pwr_Src.Auxiliary_Battery;
      package Cbs          renames Ctnr.Circuit_Breaker_Collection;
      package Buses        renames Ctnr.Bus_Collection;
      package Relays       renames Ctnr.Coil_Relay_Collection;

      The_Aux_Batt_Mnt : Aux_Batt_Mnt.Instance renames Ctnr.This_Subsystem.The_Auxiliary_Battery_Mount;
      The_Aux_Batt     : Aux_Batt.Instance     renames Ctnr.This_Subsystem.The_Auxiliary_Battery;
      The_Cbs          : Cbs.Instance          renames Ctnr.This_Subsystem.The_Circuit_Breakers;
      The_Buses        : Buses.Instance        renames Ctnr.This_Subsystem.The_Buses;
      The_Relays       : Relays.Instance       renames Ctnr.This_Subsystem.The_Relays;

      Aux_Batt_Voltage : constant Ele_U.Volts_Dc := Pwr_Src.Voltage (The_Aux_Batt);

      Io : Ctnr.Io_Interface_Instance renames Ctnr.This_Io_Interface;

      Aux_Batt_Cb : constant Boolean := Cbs.Is_Powered (The_Cbs,Ele.Aux_Battery_Cb);
      Stby_Lts_Cb : constant Boolean := Cbs.Is_Powered (The_Cbs,Ele.Stby_Lts_Cb);
      Volts_In    : Ele_U.Volts_Dc   := 0.0;
      Lights_In   : Boolean          := Stby_Lts_Cb;

   begin

      if Aux_Batt_Cb then
         Volts_In := Buses.Voltage (The_Buses,ele.Forward_Battery_Bus);
      end if;

      Aux_Batt_Mnt.Update
        (An_Instance                  => The_Aux_Batt_Mnt,
         Volts_28_In                  => Volts_In,
         Volts_5_In                   => Lights_In,
         Auxiliary_Battery_Voltage_In => Aux_Batt_Voltage);

      Io.Aux_Batt_Lamp := (Io.Aux_Batt_Test_Sw
                           and Cbs.Is_Powered(The_Cbs,Ele.Aux_Batt_Cb)
                           and not Relays.Is_Closed(The_Relays,Ele.Battery_Relay));


   exception
      when others =>
         Log.Report("Jpats_Electrical.Power_Supply_Controller.Update_Auxiliary_Battery_Mount()");
         raise;
   end Update_Auxiliary_Battery_Mount;

   procedure Update_External_Power
     (Iconst : in Float)
   is

      package Ele     renames Jpats_Electrical_Types;
      package Ele_U   renames Electrical_Units_Types;
      package Ctnr    renames Container;
      package Pwr_Src renames Power_Supply;
      package Ext_Pwr renames Pwr_Src.External_Power;
      package Sim_Acft renames Jpats_Simulated_Aircraft;

      The_Ext_Pwr : Ext_Pwr.Instance renames Ctnr.This_Subsystem.The_External_Power;

      Ios : Ctnr.Ios_Interface_Instance renames Ctnr.This_Ios_Interface;
      Plugged_In_And_Power_On : Boolean;
      Altitude     : constant Length_Types.Feet  := Sim_Acft.Get_Aircraft_Height_Above_Local_Terrain;
      Ground_Speed : constant Length_Types.Feet_per_Sec := Sim_Acft.Get_Ground_Speed;
      Ground_Service_Available : constant Boolean := Altitude < 10.0 and Ground_Speed < 5.0;

   begin

      if JPATS_Reposition.Reposition_Command.Active then
         Ctnr.This_Subsystem.Ext_Pwr_Connector_In := Ios.The_Gpu_Connector_In;
         Ctnr.This_Subsystem.Ext_Pwr_On           := Ios.The_Gpu_Power_On;
      else

         if Ground_Service_Available then
            Ctnr.This_Subsystem.Ext_Pwr_Connector_In := Ios.The_Gpu_Connector_In;
            Ctnr.This_Subsystem.Ext_Pwr_On           := Ios.The_Gpu_Power_On;
         else
            Ctnr.This_Subsystem.Ext_Pwr_Connector_In := False;
            Ctnr.This_Subsystem.Ext_Pwr_On           := False;
            Ios.The_Gpu_Power_On                     := False;
         end if;

      end if;

--      Plugged_In_And_Power_On
--        := Ctnr.This_Subsystem.Ext_Pwr_Connector_In and Ctnr.This_Subsystem.Ext_Pwr_On;
      Plugged_In_And_Power_On := Ctnr.This_Subsystem.Ext_Pwr_On;




      Ext_Pwr.Update
         (An_Instance => The_Ext_Pwr,
          Iconst      => Iconst,
          Power_On    => Plugged_In_And_Power_On and Ground_Service_Available);

   exception
      when others =>
         Log.Report("Jpats_Electrical.Power_Supply_Controller.Update_External_Power()");
         raise;
   end Update_External_Power;

   procedure Update
     (Iconst : in Float)
   is
   begin

      Update_Sources_Loads (Iconst => Iconst);
      Update_Generator (Iconst => Iconst);
      Update_Generator_Control_Unit (Iconst => Iconst);
      Update_Battery (Iconst => Iconst);
      Update_Auxiliary_Battery (Iconst => Iconst);
      Update_Auxiliary_Battery_Mount (Iconst => Iconst);
      Update_External_Power (Iconst => Iconst);

   exception
      when others =>
         Log.Report("Jpats_Electrical.Power_Supply_Controller.Update()");
         raise;
   end Update;


   procedure Read_Tables
   is
   begin
      Interpolation_Table.Read
        (File_Name => Simulation_Dictionary.Lookup ("Electrical_Dir") & "battcap.ito",
         Table     => Batt_Capacity_Table);
   exception
      when others =>
         Log.Report("Jpats_Electrical.Power_Supply_Controller.Read_Tables()");
         raise;
   end Read_Tables;


   procedure Initialize
   is

      package Ctnr         renames Container;
      package Batt         renames Power_Supply.Battery;
      package Gen          renames Power_Supply.Generator;
      package Aux_Batt     renames Power_Supply.Auxiliary_Battery;
      package Ext_Pwr      renames Power_Supply.External_Power;
      package Gcu          renames Ctnr.Generator_Control_Unit;
      package Aux_Batt_Mnt renames Ctnr.Auxiliary_Battery_Mount;

      The_Batt         : Batt.Instance         renames Ctnr.This_Subsystem.The_Battery;
      The_Gen          : Gen.Instance          renames Ctnr.This_Subsystem.The_Generator;
      The_Aux_Batt     : Aux_Batt.Instance     renames Ctnr.This_Subsystem.The_Auxiliary_Battery;
      The_Ext_Pwr      : Ext_Pwr.Instance      renames Ctnr.This_Subsystem.The_External_Power;
      The_Gcu          : Gcu.Instance          renames Ctnr.This_Subsystem.The_Generator_Control_Unit;
      The_Aux_Batt_Mnt : Aux_Batt_Mnt.Instance renames Ctnr.This_Subsystem.The_Auxiliary_Battery_Mount;
      Dead_Aux_Batt_Commanded_Last_Pass : Boolean renames Ctnr.This_Subsystem.Dead_Aux_Batt_Commanded_Last_Pass;

   begin

      Batt.Initialize (The_Batt);
      Gen.Initialize (The_Gen);
      Aux_Batt.Initialize (The_Aux_Batt);
      Ext_Pwr.Initialize (The_Ext_Pwr);
      Gcu.Initialize (The_Gcu);
      Aux_Batt_Mnt.Initialize (The_Aux_Batt_Mnt);

      Batt.Read_Tables;
      Aux_Batt.Read_Tables;
      Gen.Read_Tables;
      Read_Tables;

      Dead_Aux_Batt_Commanded_Last_Pass := False;
   exception
      when others =>
         Log.Report("Jpats_Electrical.Power_Supply_Controller.Initialize()");
         raise;
   end Initialize;

end Jpats_Electrical.Power_Supply_Controller;



