-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                 JPATS T-6A Texan-II Flight Training Device
--
--
--  Engineer:  Keith H. Rehm
--
--  Revision:  (Number and date inserted by Clearcase)
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------
with Jpats_Electrical.Container;
with Power_Supply.Battery;
with Power_Supply.Auxiliary_Battery;
with Power_Supply.Generator;
with Power_Supply.External_Power;
with Jpats_Environmental;
with Jpats_Environmental_Types;
with Jpats_Lights;
with Jpats_Secondary_Flight_Controls;
with Jpats_Landing_Gear;
with Jpats_Fuel;
with Jpats_Powerplant;
with Jpats_Flight_Instruments;
with Jpats_Electrical_Types;
with Electrical_Units_Types;
with Log;

package body Jpats_Electrical.Bus_Controller is

   procedure Initialize
   is
      package Ctnr  renames Container;
      package Paths renames Ctnr.Path_Collection;
      package Buses renames Ctnr.Bus_Collection;
      The_Paths : Paths.Instance renames Ctnr.This_Subsystem.The_Paths;
      The_Buses : Buses.Instance renames Ctnr.This_Subsystem.The_Buses;
   begin
      Paths.Initialize (The_Paths);
      Buses.Initialize (The_Buses);
   exception
      when others =>
         Log.Report("Jpats_Electrical.Bus_Controller.Initialize()");
         raise;
   end Initialize;

   procedure Update_Paths
     (Iconst : in Float)
   is

      package Ele_T            renames Jpats_Electrical_Types;
      package Ctnr             renames Container;
      package Relays           renames Ctnr.Coil_Relay_Collection;
      package Current_Limiters renames Ctnr.Current_Limiter_Collection;

      Jpats : Ctnr.Instance renames Ctnr.This_Subsystem;

      Path : Ctnr.Path_Array_Of_Boolean_Type;

      Pwr_Source_Voltage : Ctnr.Array_Of_Power_Source_Voltage_Type;

      Io  : Ctnr.Io_Interface_Instance  renames Ctnr.This_Io_Interface;
      Ios : Ctnr.Ios_Interface_Instance renames Ctnr.This_Ios_Interface;

      The_Batt         : Power_Supply.Battery.Instance           renames Jpats.The_Battery;
      The_Aux_Batt     : Power_Supply.Auxiliary_Battery.Instance renames Jpats.The_Auxiliary_Battery;
      The_Gen          : Power_Supply.Generator.Instance         renames Jpats.The_Generator;
      The_Ext_Pwr      : Power_Supply.External_Power.Instance    renames Jpats.The_External_Power;
      The_Aux_Batt_Mnt : Ctnr.Auxiliary_Battery_Mount.Instance   renames Jpats.The_Auxiliary_Battery_Mount;

      From_Batt     : constant Ctnr.Power_Source_Name_Type := Ctnr.Battery;
      From_Aux_Batt : constant Ctnr.Power_Source_Name_Type := Ctnr.Auxiliary_Battery;
      From_Gen      : constant Ctnr.Power_Source_Name_Type := Ctnr.Generator;
      From_Ext_Pwr  : constant Ctnr.Power_Source_Name_Type := Ctnr.External_Power;

      To_Fwd_Batt_Bus     : constant Ele_T.Bus_Name_Type := Ele_T.Forward_Battery_Bus;
      To_Fwd_Gen_Bus      : constant Ele_T.Bus_Name_Type := Ele_T.Forward_Generator_Bus;
      To_Fwd_Avi_Batt_Bus : constant Ele_T.Bus_Name_Type := Ele_T.Forward_Avionics_Battery_Bus;
      To_Fwd_Avi_Gen_Bus  : constant Ele_T.Bus_Name_Type := Ele_T.Forward_Avionics_Generator_Bus;
      To_Hot_Batt_Bus     : constant Ele_T.Bus_Name_Type := Ele_T.Hot_Battery_Bus;
      To_Gen_Bus          : constant Ele_T.Bus_Name_Type := Ele_T.Generator_Bus;
      To_Aft_Batt_Bus     : constant Ele_T.Bus_Name_Type := Ele_T.Aft_Battery_Bus;
      To_Aft_Gen_Bus      : constant Ele_T.Bus_Name_Type := Ele_T.Aft_Generator_Bus;
      To_Aft_Aux_Batt_Bus : constant Ele_T.Bus_Name_Type := Ele_T.Aft_Auxiliary_Battery_Bus;
      To_Fwd_Aux_Batt_Bus : constant Ele_T.Bus_Name_Type := Ele_T.Forward_Auxiliary_Battery_Bus;
      To_Aft_Avi_Gen_Bus  : constant Ele_T.Bus_Name_Type := Ele_T.Aft_Avionics_Generator_Bus;
      To_Aft_Avi_Batt_Bus : constant Ele_T.Bus_Name_Type := Ele_T.Aft_Avionics_Battery_Bus;
      To_Str_Distrib_Bus  : constant Ele_T.Bus_Name_Type := Ele_T.Starter_Distribution_Bus;

      Batt_Fwd_Avi_Cb : constant Boolean := Io.Fwd_Avi_Batt_Di or Ios.Kcbin;
      Gen_Fwd_Avi_Cb  : constant Boolean := Io.Fwd_Avi_Gen_Di  or Ios.Kcbin;
      Aft_Batt_Cb     : constant Boolean := Io.Aft_Bat_Di      or Ios.Kcbin;
      Aft_Gen_Cb      : constant Boolean := Io.Aft_Gen_Di      or Ios.Kcbin;
      Aft_Inst_Cb     : constant Boolean := Io.Aft_Inst_Di     or Ios.Kcbin;
      Aft_Avi_Gen_Cb  : constant Boolean := Io.Aft_Avi_Gen_Di  or Ios.Kcbin;
      Aft_Avi_Batt_Cb : constant Boolean := Io.Aft_Avi_Batt_Di or Ios.Kcbin;

      Batt_Rly_Closed        : constant Boolean :=
        Relays.Is_Closed (Jpats.The_Relays, Ele_T.Battery_Relay);

      Gen_Rly_Closed         : constant Boolean :=
        Relays.Is_Closed (Jpats.The_Relays, Ele_T.Generator_Relay);

      Bus_Tie_Rly_Closed     : constant Boolean :=
        Relays.Is_Closed (Jpats.The_Relays, Ele_T.Bus_Tie_Relay);

      Aux_Batt_Rly_Open      : constant Boolean :=
        Relays.Is_Open (Jpats.The_Relays, Ele_T.Auxiliary_Battery_Relay);

      Aux_Batt_Rly_Closed    : constant Boolean :=
        Relays.Is_Closed (Jpats.The_Relays, Ele_T.Auxiliary_Battery_Relay);

      Ext_Pwr_Rly_Closed     : constant Boolean :=
        Relays.Is_Closed (Jpats.The_Relays, Ele_T.External_Power_Relay);

      Avi_Mstr_Batt_Rly_Open : constant Boolean :=
        Relays.Is_Open (Jpats.The_Relays, Ele_T.Avionics_Master_Battery_Relay);

      Avi_Mstr_Gen_Rly_Open  : constant Boolean :=
        Relays.Is_Open (Jpats.The_Relays, Ele_T.Avionics_Master_Generator_Relay);

      Batt_Current_Limiter_Closed : constant Boolean :=
        Current_Limiters.Is_Closed
        (Jpats.The_Current_Limiters, Ctnr.Battery_Bus_Current_Limiter);

      Gen_Current_Limiter_Closed  : constant Boolean :=
        Current_Limiters.Is_Closed
        (Jpats.The_Current_Limiters, Ctnr.Generator_Bus_Current_Limiter);

      Batt_Pwr_Available : constant Boolean := Power_Supply.Power_Available (The_Batt);
      Gen_Pwr_Available  : constant Boolean := Power_Supply.Power_Available (The_Gen);
      Ext_Pwr_Available  : constant Boolean := Power_Supply.Power_Available (The_Ext_Pwr);

   begin

      -- battery to buses
      Path (From_Batt, To_Hot_Batt_Bus) := Batt_Pwr_Available;

      Path (From_Batt, To_Fwd_Batt_Bus) :=
        Path (From_Batt, To_Hot_Batt_Bus) and then Batt_Rly_Closed and then
        Batt_Current_Limiter_Closed;

      Path (From_Batt, To_Fwd_Gen_Bus) :=
        Path (From_Batt, To_Hot_Batt_Bus) and then Batt_Rly_Closed and then
        Bus_Tie_Rly_Closed and then Gen_Current_Limiter_Closed;

      Path (From_Batt, To_Fwd_Avi_Batt_Bus) :=
        Path (From_Batt, To_Fwd_Batt_Bus) and then Batt_Fwd_Avi_Cb and then
        Avi_Mstr_Batt_Rly_Open;

      Path (From_Batt, To_Fwd_Avi_Gen_Bus) :=
        Path (From_Batt, To_Fwd_Gen_Bus) and then Gen_Fwd_Avi_Cb and then
        Avi_Mstr_Gen_Rly_Open;

      Path (From_Batt, To_Gen_Bus) :=
        Path (From_Batt, To_Hot_Batt_Bus) and then Batt_Rly_Closed and then
        Bus_Tie_Rly_Closed;

      Path (From_Batt, To_Aft_Batt_Bus) :=
        Path (From_Batt, To_Fwd_Batt_Bus) and then Aft_Batt_Cb;

      Path (From_Batt, To_Aft_Gen_Bus) :=
        Path (From_Batt, To_Fwd_Gen_Bus) and then Aft_Gen_Cb;

      Path (From_Batt, To_Fwd_Aux_Batt_Bus) :=
        Path (From_Batt, To_Fwd_Batt_Bus) and then Aux_Batt_Rly_Open;

      Path (From_Batt, To_Aft_Aux_Batt_Bus) :=
        Path (From_Batt, To_Fwd_Aux_Batt_Bus) and then Aft_Inst_Cb;

      Path (From_Batt, To_Aft_Avi_Gen_Bus) :=
        Path (From_Batt, To_Fwd_Avi_Gen_Bus) and then Aft_Avi_Gen_Cb;

      Path (From_Batt, To_Aft_Avi_Batt_Bus) :=
        Path (From_Batt, To_Fwd_Avi_Batt_Bus) and then Aft_Avi_Batt_Cb;

      Path(From_Batt, To_Str_Distrib_Bus) :=
        Path (From_Batt, To_Hot_Batt_Bus) and then Batt_Rly_Closed;

      -- generator to buses
      Path (From_Gen, To_Gen_Bus) := Gen_Pwr_Available and then Gen_Rly_Closed;

      Path (From_Gen, To_Fwd_Batt_Bus) :=
        Path (From_Gen, To_Gen_Bus) and then Bus_Tie_Rly_Closed and then
        Batt_Current_Limiter_Closed;

      Path (From_Gen, To_Fwd_Gen_Bus) :=
        Path (From_Gen, To_Gen_Bus) and then Gen_Current_Limiter_Closed;

      Path (From_Gen, To_Fwd_Avi_Batt_Bus) :=
        Path (From_Gen, To_Fwd_Batt_Bus) and then Batt_Fwd_Avi_Cb and then
        Avi_Mstr_Batt_Rly_Open;

      Path (From_Gen, To_Fwd_Avi_Gen_Bus) :=
        Path (From_Gen, To_Fwd_Gen_Bus) and then Gen_Fwd_Avi_Cb and then
        Avi_Mstr_Gen_Rly_Open;

      Path (From_Gen, To_Hot_Batt_Bus) :=
        Path (From_Gen, To_Gen_Bus) and then Bus_Tie_Rly_Closed and then
        Batt_Rly_Closed;

      Path (From_Gen, To_Aft_Batt_Bus) :=
        Path (From_Gen, To_Fwd_Batt_Bus) and then Aft_Batt_Cb;

      Path (From_Gen, To_Aft_Gen_Bus) :=
        Path (From_Gen, To_Fwd_Gen_Bus) and then Aft_Gen_Cb;

      Path (From_Gen, To_Fwd_Aux_Batt_Bus) :=
        Path (From_Gen, To_Fwd_Batt_Bus) and then Aux_Batt_Rly_Open;

      Path (From_Gen, To_Aft_Aux_Batt_Bus) :=
        Path (From_Gen, To_Fwd_Aux_Batt_Bus) and then Aft_Inst_Cb;

      Path (From_Gen, To_Aft_Avi_Gen_Bus) :=
        Path (From_Gen, To_Fwd_Avi_Gen_Bus) and then Aft_Avi_Gen_Cb;

      Path (From_Gen, To_Aft_Avi_Batt_Bus) :=
        Path (From_Gen, To_Fwd_Avi_Batt_Bus) and then Aft_Avi_Batt_Cb;

      Path (From_Gen, To_Str_Distrib_Bus) :=
        Path (From_Gen, To_Gen_Bus) and then Bus_Tie_Rly_Closed;

      -- auxiliary battery to buses
      Path (From_Aux_Batt, To_Fwd_Batt_Bus)     := False;
      Path (From_Aux_Batt, To_Fwd_Gen_Bus)      := False;
      Path (From_Aux_Batt, To_Fwd_Avi_Batt_Bus) := False;
      Path (From_Aux_Batt, To_Fwd_Avi_Gen_Bus)  := False;
      Path (From_Aux_Batt, To_Hot_Batt_Bus)     := False;
      Path (From_Aux_Batt, To_Gen_Bus)          := False;
      Path (From_Aux_Batt, To_Aft_Batt_Bus)     := False;
      Path (From_Aux_Batt, To_Aft_Gen_Bus)      := False;
      Path (From_Aux_Batt, To_Aft_Avi_Gen_Bus)  := False;
      Path (From_Aux_Batt, To_Aft_Avi_Batt_Bus) := False;
      Path (From_Aux_Batt, To_Str_Distrib_Bus)  := False;

      Path (From_Aux_Batt, To_Fwd_Aux_Batt_Bus) :=
        Ctnr.Auxiliary_Battery_Mount.Is_Powered (The_Aux_Batt_Mnt) and then
        Aux_Batt_Rly_Closed;

      Path (From_Aux_Batt, To_Aft_Aux_Batt_Bus) :=
        Path (From_Aux_Batt, To_Fwd_Aux_Batt_Bus) and then Aft_Inst_Cb;


      -- external power to buses
      Path (From_Ext_Pwr, To_Fwd_Batt_Bus) :=
        Ext_Pwr_Available and then Ext_Pwr_Rly_Closed and then
        Batt_Current_Limiter_Closed;

      Path (From_Ext_Pwr, To_Fwd_Gen_Bus) :=
        Ext_Pwr_Available and then Ext_Pwr_Rly_Closed and then
        Bus_Tie_Rly_Closed and then Gen_Current_Limiter_Closed;

      Path (From_Ext_Pwr, To_Fwd_Avi_Batt_Bus) :=
        Path (From_Ext_Pwr, To_Fwd_Batt_Bus) and then
        Batt_Fwd_Avi_Cb and then Avi_Mstr_Batt_Rly_Open;

      Path (From_Ext_Pwr, To_Fwd_Avi_Gen_Bus) :=
        Path (From_Ext_Pwr, To_Fwd_Gen_Bus) and then
        Gen_Fwd_Avi_Cb and then Avi_Mstr_Gen_Rly_Open;

      Path (From_Ext_Pwr, To_Hot_Batt_Bus) :=
        Ext_Pwr_Available and then Ext_Pwr_Rly_Closed and then
        Batt_Rly_Closed;

      Path (From_Ext_Pwr, To_Gen_Bus) :=
        Ext_Pwr_Available and then Ext_Pwr_Rly_Closed and then
        Bus_Tie_Rly_Closed;

      Path (From_Ext_Pwr, To_Aft_Batt_Bus) :=
        Path (From_Ext_Pwr, To_Fwd_Batt_Bus) and then Aft_Batt_Cb;

      Path (From_Ext_Pwr, To_Aft_Gen_Bus) :=
        Path (From_Ext_Pwr, To_Fwd_Gen_Bus) and then Aft_Gen_Cb;

      Path (From_Ext_Pwr, To_Fwd_Aux_Batt_Bus) :=
        Path (From_Ext_Pwr, To_Fwd_Batt_Bus) and then Aux_Batt_Rly_Open;

      Path (From_Ext_Pwr, To_Aft_Aux_Batt_Bus) :=
        Path (From_Ext_Pwr, To_Fwd_Aux_Batt_Bus) and then Aft_Inst_Cb;

      Path (From_Ext_Pwr, To_Aft_Avi_Gen_Bus) :=
        Path (From_Ext_Pwr, To_Fwd_Avi_Gen_Bus) and then Aft_Avi_Gen_Cb;

      Path (From_Ext_Pwr, To_Aft_Avi_Batt_Bus) :=
        Path (From_Ext_Pwr, To_Fwd_Avi_Batt_Bus) and then Aft_Avi_Batt_Cb;

      Path (From_Ext_Pwr, To_Str_Distrib_Bus) :=
        Ext_Pwr_Available and then Ext_Pwr_Rly_Closed;

      -- power source voltages
      Pwr_Source_Voltage (From_Batt)     := Power_Supply.Voltage (The_Batt);

      Pwr_Source_Voltage (From_Gen)      := Power_Supply.Voltage (The_Gen);

      Pwr_Source_Voltage (From_Aux_Batt) := Power_Supply.Voltage (The_Aux_Batt);

      Pwr_Source_Voltage (From_Ext_Pwr)  := Power_Supply.Voltage (The_Ext_Pwr);

      -- update paths
      Ctnr.Path_Collection.Update
        (An_Instance          => Jpats.The_Paths,
         Path_Is_Complete     => Path,
         Power_Source_Voltage => Pwr_Source_Voltage);

   exception
      when others =>
         Log.Report("Jpats_Electrical.Bus_Controller.Update_Paths()");
         raise;
   end Update_Paths;


   procedure Update_Buses
     (Iconst : in Float)
   is

      package Ele_T     renames Jpats_Electrical_Types;
      package Ele_U     renames Electrical_Units_Types;
      package Ctnr      renames Container;
      package Cbs       renames Ctnr.Circuit_Breaker_Collection;
      package Pwrplnt   renames Jpats_Powerplant;
      package Env       renames Jpats_Environmental;
      package Env_T     renames Jpats_Environmental_Types;
      package Fuel      renames Jpats_Fuel;
      package Ldg_Gr    renames Jpats_Landing_Gear;
      package Lts       renames Jpats_Lights;
      package Sfc       renames Jpats_Secondary_Flight_Controls;
      package Paths     renames Ctnr.Path_Collection;
      package Jflin     renames Jpats_Flight_Instruments;

      The_Cbs    : Cbs.Instance   renames Ctnr.This_Subsystem.The_Circuit_Breakers;
      The_Paths  : Paths.Instance renames Ctnr.This_Subsystem.The_Paths;
      The_Buses  : Ctnr.Bus_Collection.Instance renames Ctnr.This_Subsystem.The_Buses;
      Io         : Ctnr.Io_Interface_Instance  renames Ctnr.This_Io_Interface;

      Bus_Voltages : Ctnr.Array_Of_Bus_Voltages := (others => 0.0);
      Bus_Loads    : Ctnr.Array_Of_Bus_Loads    := (others => 0.0);

      Fwd_Batt_Bus_Load              : Ele_U.Amps := 0.0;
      Fwd_Gen_Bus_Load               : Ele_U.Amps := 0.0;
      Fwd_Avi_Batt_Bus_Load          : Ele_U.Amps := 0.0;
      Fwd_Avi_Gen_Bus_Load           : Ele_U.Amps := 0.0;
      Hot_Batt_Bus_Load              : Ele_U.Amps := 0.0;
      Gen_Bus_Load                   : Ele_U.Amps := 0.0;
      Aft_Batt_Bus_Load              : Ele_U.Amps := 0.0;
      Aft_Gen_Bus_Load               : Ele_U.Amps := 0.0;
      Aft_Aux_Batt_Bus_Load          : Ele_U.Amps := 0.0;
      Fwd_Aux_Batt_Bus_Load          : Ele_U.Amps := 0.0;
      Aft_Avi_Gen_Bus_Load           : Ele_U.Amps := 0.0;
      Aft_Avi_Batt_Bus_Load          : Ele_U.Amps := 0.0;
      Starter_Distribution_Bus_Load  : Ele_U.Amps := 0.0;

   begin

      -- load bus voltage array
      for Bus_Name in Ele_T.Bus_Name_Type
      loop
         for Pwr_Source_Name in Ctnr.Power_Source_Name_Type
         loop
            if Paths.Voltage (The_Paths, Pwr_Source_Name, Bus_Name) > Bus_Voltages (Bus_Name) then
               Bus_Voltages (Bus_Name) := Paths.Voltage (The_Paths, Pwr_Source_Name, Bus_Name);
            end if;
         end loop;
      end loop;

      -- find loads on buses

      -- Forward_Battery_Bus
      if Ctnr.Bus_Collection.Is_Powered (The_Buses, Ele_T.Forward_Battery_Bus) then
         if Cbs.Is_Powered (The_Cbs, Ele_T.Ail_El_Trim_Cb) then
            Fwd_Batt_Bus_Load := Fwd_Batt_Bus_Load  +
              Sfc.Aileron_Trim_Load                 +
              Sfc.Elevator_Trim_Load;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Rud_Trim_Cb) then
            Fwd_Batt_Bus_Load := Fwd_Batt_Bus_Load + Sfc.Rudder_Trim_Load;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Ldg_Gr_Cont_Cb) then
            Fwd_Batt_Bus_Load := Fwd_Batt_Bus_Load         +
              Ldg_Gr.Left_Landing_Gear_A1_WOW_Relay_Load   +
              Ldg_Gr.Right_Landing_Gear_A1_WOW_Relay_Load  +
              Ldg_Gr.Left_Landing_Gear_A2_WOW_Relay_Load   +
              Ldg_Gr.Right_Landing_Gear_A2_WOW_Relay_Load  +
              Ldg_Gr.Q_Switch_Control_Relay_Load           +
              Ldg_Gr.Landing_Gear_Down_And_Locked_Rly_Load +
              Ldg_Gr.Landing_Gear_Control_Assembly_Load;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Flap_Cont_Cb) then
            Fwd_Batt_Bus_Load := Fwd_Batt_Bus_Load      +
              Sfc.Flaps_Electrical_Load                 +
              Sfc.Primary_Flap_Selector_Manifold_Load   +
              Sfc.Emergency_Flap_Selector_Manifold_Load;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Util_Lt_Cb) then
            Fwd_Batt_Bus_Load := Fwd_Batt_Bus_Load + Lts.Utility_Light_Load;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Coll_Lts_Cb) then
            Fwd_Batt_Bus_Load := Fwd_Batt_Bus_Load + Lts.Anti_Collision_Lights_Load;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Inst_Lts_Cb) then
            Fwd_Batt_Bus_Load := Fwd_Batt_Bus_Load + Lts.Instrument_Lights_Load;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Ldg_Lts_Cb) then
            Fwd_Batt_Bus_Load := Fwd_Batt_Bus_Load + Lts.Landing_Light_Load;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Cws_Cb_Fwd_Batt_Bus) then
            Fwd_Batt_Bus_Load := Fwd_Batt_Bus_Load + 0.01;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Fuel_Qty_Lo_Cb) then
            Fwd_Batt_Bus_Load := Fwd_Batt_Bus_Load +
              Fuel.Left_Fuel_Low_Sensor_Load       +
              Fuel.Right_Fuel_Low_Sensor_Load;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Pmu_Cb) then
            Fwd_Batt_Bus_Load := Fwd_Batt_Bus_Load +
              Pwrplnt.PMU_Fail_Relay_Load          +
              Pwrplnt.PMU_Load;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Boost_Pump_Cb) then
            Fwd_Batt_Bus_Load := Fwd_Batt_Bus_Load + Fuel.Boost_Pump_Load;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Ign_Cb) then
            Fwd_Batt_Bus_Load := Fwd_Batt_Bus_Load +
              Pwrplnt.Auto_Ignition_Relay_Load     +
              Pwrplnt.Ignition_Exciter_Load;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Start_Cb) then
            Fwd_Batt_Bus_Load := Fwd_Batt_Bus_Load + 0.01;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Oil_Trx_Cb) then
            Fwd_Batt_Bus_Load := Fwd_Batt_Bus_Load + 0.01;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Batt_Sw_Cb) then
            Fwd_Batt_Bus_Load := Fwd_Batt_Bus_Load + 0.01;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Avi_Mstr_Cb) then
            Fwd_Batt_Bus_Load := Fwd_Batt_Bus_Load + 0.01;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Fld_Lts_Cb) then
            Fwd_Batt_Bus_Load := Fwd_Batt_Bus_Load + Lts.Flood_Lights_Load;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Alt_Eng_Dis_Cb) then
            Fwd_Batt_Bus_Load := Fwd_Batt_Bus_Load + 0.01;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Edm_Cb_Fwd_Batt_Bus) then
            Fwd_Batt_Bus_Load := Fwd_Batt_Bus_Load + 0.01;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Prop_Sys_Cb) then
            Fwd_Batt_Bus_Load := Fwd_Batt_Bus_Load + 0.01;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Fdr_Cb) then
            Fwd_Batt_Bus_Load := Fwd_Batt_Bus_Load + 0.01;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Aoa_Cb) then
            Fwd_Batt_Bus_Load := Fwd_Batt_Bus_Load + 0.01;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Inflow_Sys_Cb) then
            Fwd_Batt_Bus_Load := Fwd_Batt_Bus_Load +
              Env.Load(Env_T.Auto_Pressure_Relay) +
              Env.Load(Env_T.Heat_Exchanger_Blower_Relay) +
              Env.Load(Env_T.Bleed_Air_Flow_Relay_1) +
              Env.Bi_Level_Flow_Control_Valve_Load +
              Env.Anti_Gravity_Shutoff_Valve_Load  +
              Env.Canopy_Seal_Valve_Load;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Hyd_Sys_Cb) then
            Fwd_Batt_Bus_Load := Fwd_Batt_Bus_Load + 0.01;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Audio_Cb) then
            Fwd_Batt_Bus_Load := Fwd_Batt_Bus_Load + 0.01;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Aft_Battery_Cb) then
            Fwd_Batt_Bus_Load := Fwd_Batt_Bus_Load + 0.01;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Aux_Battery_Cb) then
            Fwd_Batt_Bus_Load := Fwd_Batt_Bus_Load + 0.01;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Fwd_Avi_Cb_Fwd_Batt_Bus) then
            Fwd_Batt_Bus_Load := Fwd_Batt_Bus_Load + 0.01;
         end if;
      end if;

      -- Forward_Generator_Bus
      if Ctnr.Bus_Collection.Is_Powered (The_Buses, Ele_T.Forward_Generator_Bus) then
         if Cbs.Is_Powered (The_Cbs, Ele_T.Air_Cond_Cb) then
            Fwd_Gen_Bus_Load := Fwd_Gen_Bus_Load + Env.Load(Env_T.Condensor_Blower_Relay);
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Ckpt_Temp_Cb) then
            Fwd_Gen_Bus_Load := Fwd_Gen_Bus_Load      +
              Env.Load(Env_T.Cool_Command_Relay)      +
              Env.Load(Env_T.Defog_Select_Relay)      +
              Env.Bleed_Air_Bypass_Valve_Load         +
              Env.Cockpit_Temperature_Controller_Load;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Aural_Warn_Cb) then
            Fwd_Gen_Bus_Load := Fwd_Gen_Bus_Load + 0.01;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Evap_Blower_Cb) then
            Fwd_Gen_Bus_Load := Fwd_Gen_Bus_Load + Env.Evaporator_Blower_Load;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Aoa_Ht_Cb) then
            Fwd_Gen_Bus_Load := Fwd_Gen_Bus_Load + Jflin.Get_Angle_of_Attack_Vane_Probe_Heater_Load;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Pitot_Ht_Cb) then
            Fwd_Gen_Bus_Load := Fwd_Gen_Bus_Load + Jflin.Get_Primary_Pitot_Probe_Heater_Load +
                                Jflin.Get_Secondary_Pitot_Probe_Heater_Load;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Fuel_Bal_Cb) then
            Fwd_Gen_Bus_Load := Fwd_Gen_Bus_Load +
              Fuel.Left_Wing_Transfer_Valve_Load +
              Fuel.Left_Wing_Transfer_Valve_Load;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Side_Lts_Cb) then
            Fwd_Gen_Bus_Load := Fwd_Gen_Bus_Load + 0.01;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Nav_Lts_Cb) then
            Fwd_Gen_Bus_Load := Fwd_Gen_Bus_Load +
              Lts.Navigation_Lights_Load         +
              Lts.Position_Lights_Load;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Taxi_Lts_Cb) then
            Fwd_Gen_Bus_Load := Fwd_Gen_Bus_Load + Lts.Taxi_Light_Load;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Speedbrake_Cb) then
            Fwd_Gen_Bus_Load := Fwd_Gen_Bus_Load + Sfc.Speedbrake_Electrical_Load;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Nws_Cb) then
            Fwd_Gen_Bus_Load := Fwd_Gen_Bus_Load + 0.01;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Test_Lts_Cb) then
            Fwd_Gen_Bus_Load := Fwd_Gen_Bus_Load + 0.01;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Trim_Ind_Cb) then
            Fwd_Gen_Bus_Load := Fwd_Gen_Bus_Load + 0.01;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Edm_Cb_Fwd_Gen_Bus) then
            Fwd_Gen_Bus_Load := Fwd_Gen_Bus_Load + 0.01;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Gen_Sw_Cb) then
            Fwd_Gen_Bus_Load := Fwd_Gen_Bus_Load + 0.01;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Fwd_Avi_Cb_Fwd_Gen_Bus) then
            Fwd_Gen_Bus_Load := Fwd_Gen_Bus_Load + 0.01;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Fire_Detect_Cb_Fwd_Gen_Bus) then
            Fwd_Gen_Bus_Load := Fwd_Gen_Bus_Load + 0.01;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Eng_Sys_Dis_Cb) then
            Fwd_Gen_Bus_Load := Fwd_Gen_Bus_Load + 0.01;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Pri_Eng_Dis_Cb) then
            Fwd_Gen_Bus_Load := Fwd_Gen_Bus_Load + 0.01;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Tad_Cb) then
            Fwd_Gen_Bus_Load := Fwd_Gen_Bus_Load + 0.01;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Seat_Adj_Cb) then
            Fwd_Gen_Bus_Load := Fwd_Gen_Bus_Load + Env.Seat_Motor_Load;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Cws_Cb_Fwd_Gen_Bus) then
            Fwd_Gen_Bus_Load := Fwd_Gen_Bus_Load + 0.01;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Aft_Gen_Cb) then
            Fwd_Gen_Bus_Load := Fwd_Gen_Bus_Load + 0.01;
         end if;
      end if;

      -- Forward_Avionics_Battery_Bus
      if Ctnr.Bus_Collection.Is_Powered (The_Buses, Ele_T.Forward_Avionics_Battery_Bus) then
         if Cbs.Is_Powered (The_Cbs, Ele_T.Eadi_Cb) then
            Fwd_Avi_Batt_Bus_Load := Fwd_Avi_Batt_Bus_Load + 0.05;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Gps_Cb) then
            Fwd_Avi_Batt_Bus_Load := Fwd_Avi_Batt_Bus_Load + 0.05;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Ahrs_Cb) then
            Fwd_Avi_Batt_Bus_Load := Fwd_Avi_Batt_Bus_Load + 0.05;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Aft_Avi_Cb_Fwd_Avi_Batt_Bus) then
            Fwd_Avi_Batt_Bus_Load := Fwd_Avi_Batt_Bus_Load + 0.05;
         end if;
      end if;

      -- Forward_Avionics_Generator_Bus
      if Ctnr.Bus_Collection.Is_Powered (The_Buses, Ele_T.Forward_Avionics_Generator_Bus) then
         if Cbs.Is_Powered (The_Cbs, Ele_T.Adc_Cb) then
            Fwd_Avi_Gen_Bus_Load := Fwd_Avi_Gen_Bus_Load + 0.03;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Ehsi_Cb) then
            Fwd_Avi_Gen_Bus_Load := Fwd_Avi_Gen_Bus_Load + 0.03;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Xpdr_Cb) then
            Fwd_Avi_Gen_Bus_Load := Fwd_Avi_Gen_Bus_Load + 0.03;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Tas_Cb) then
            Fwd_Avi_Gen_Bus_Load := Fwd_Avi_Gen_Bus_Load + 0.03;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Vhf_Comm_Cb) then
            Fwd_Avi_Gen_Bus_Load := Fwd_Avi_Gen_Bus_Load + 0.03;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Asi_Cb) then
            Fwd_Avi_Gen_Bus_Load := Fwd_Avi_Gen_Bus_Load + 0.03;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Altm_Cb_Fwd_Avi_Gen_Bus) then
            Fwd_Avi_Gen_Bus_Load := Fwd_Avi_Gen_Bus_Load + 0.03;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Vhf_Nav_Cb) then
            Fwd_Avi_Gen_Bus_Load := Fwd_Avi_Gen_Bus_Load + 0.03;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Dme_Cb) then
            Fwd_Avi_Gen_Bus_Load := Fwd_Avi_Gen_Bus_Load + 0.03;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Radio_Rlys_Cb_Fwd_Avi_Gen_Bus) then
            Fwd_Avi_Gen_Bus_Load := Fwd_Avi_Gen_Bus_Load + 0.03;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Vvi_Cb) then
            Fwd_Avi_Gen_Bus_Load := Fwd_Avi_Gen_Bus_Load + 0.03;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Rmu_Cb) then
            Fwd_Avi_Gen_Bus_Load := Fwd_Avi_Gen_Bus_Load + 0.03;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Aft_Avi_Cb_Fwd_Avi_Gen_Bus) then
            Fwd_Avi_Gen_Bus_Load := Fwd_Avi_Gen_Bus_Load + 0.03;
         end if;
      end if;

      -- Hot_Battery_Bus
      if Ctnr.Bus_Collection.Is_Powered (The_Buses, Ele_T.Hot_Battery_Bus) then
         if Cbs.Is_Powered (The_Cbs, Ele_T.Spare_Cb) then
            Hot_Batt_Bus_Load := Hot_Batt_Bus_Load + 0.01;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Ram_Air_Valve_Cb) then
            Hot_Batt_Bus_Load := Hot_Batt_Bus_Load + Env.Ram_Air_Valve_Load;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Battery_Cb) then
            Hot_Batt_Bus_Load := Hot_Batt_Bus_Load + 0.01;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Emerg_Flaps_Cb) then
            Hot_Batt_Bus_Load := Hot_Batt_Bus_Load + 0.01;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Clocks_Cb) then
            Hot_Batt_Bus_Load := Hot_Batt_Bus_Load + 0.01;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Elt_Cb) then
            Hot_Batt_Bus_Load := Hot_Batt_Bus_Load + 0.01;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Obogs_Cb) then
            Hot_Batt_Bus_Load := Hot_Batt_Bus_Load +
              Env.OBOGS_Regulator_Load +
              Env.OBOGS_Concentrator_Load +
              Env.OBOGS_Shutoff_Valve_Load;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Fdr_Maint_Epa_Fuel_Pump_Cb) then
            Hot_Batt_Bus_Load := Hot_Batt_Bus_Load + 0.01;
         end if;
      end if;

      -- Generator_Bus
      if Ctnr.Bus_Collection.Is_Powered (The_Buses, Ele_T.Generator_Bus) then
         if Cbs.Is_Powered (The_Cbs, Ele_T.Cond_Blower_Cb) then
            Gen_Bus_Load := Gen_Bus_Load + Env.Condensor_Blower_Load;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Heat_Exch_Blower_Cb) then
            Gen_Bus_Load := Gen_Bus_Load + Env.Heat_Exchanger_Blower_Load;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Bus_Sense_Cb) then
            Gen_Bus_Load := Gen_Bus_Load + 0.01;
         end if;
      end if;

      -- Aft_Battery_Bus
      if Ctnr.Bus_Collection.Is_Powered (The_Buses, Ele_T.Aft_Battery_Bus) then
         if Cbs.Is_Powered (The_Cbs, Ele_T.Alt_Eng_Dis_Cb_Aft_Batt_Bus) then
            Aft_Batt_Bus_Load := Aft_Batt_Bus_Load + 0.01;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Audio_Cb_Aft_Batt_Bus) then
            Aft_Batt_Bus_Load := Aft_Batt_Bus_Load + 0.01;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Util_Lt_Cb_Aft_Batt_Bus) then
            Aft_Batt_Bus_Load := Aft_Batt_Bus_Load + 0.01;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Inst_Lts_Cb_Aft_Batt_Bus) then
            Aft_Batt_Bus_Load := Aft_Batt_Bus_Load + 0.01;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Fld_Lts_Cb_Aft_Batt_Bus) then
            Aft_Batt_Bus_Load := Aft_Batt_Bus_Load + 0.01;
         end if;
      end if;

      -- Aft_Generator_Bus
      if Ctnr.Bus_Collection.Is_Powered (The_Buses, Ele_T.Aft_Generator_Bus) then
         if Cbs.Is_Powered (The_Cbs, Ele_T.Evap_Blower_Cb_Aft_Gen_Bus) then
            Aft_Gen_Bus_Load := Aft_Gen_Bus_Load + 0.01;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Seat_Adj_Cb_Aft_Gen_Bus) then
            Aft_Gen_Bus_Load := Aft_Gen_Bus_Load + 0.01;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Side_Lts_Cb_Aft_Gen_Bus) then
            Aft_Gen_Bus_Load := Aft_Gen_Bus_Load + 0.01;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Eng_Sys_Dis_Cb_Aft_Gen_Bus) then
            Aft_Gen_Bus_Load := Aft_Gen_Bus_Load + 0.01;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Pri_Eng_Dis_Cb_Aft_Gen_Bus) then
            Aft_Gen_Bus_Load := Aft_Gen_Bus_Load + 0.01;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Trim_Ind_Cb_Aft_Gen_Bus) then
            Aft_Gen_Bus_Load := Aft_Gen_Bus_Load + 0.01;
         end if;
      end if;

      -- Aft_Auxiliary_Battery_Bus
      if Ctnr.Bus_Collection.Is_Powered (The_Buses, Ele_T.Aft_Auxiliary_Battery_Bus) then
         if Cbs.Is_Powered (The_Cbs, Ele_T.Altm_Cb_Aft_Aux_Batt_Bus) then
            Aft_Aux_Batt_Bus_Load := Aft_Aux_Batt_Bus_Load + 0.01;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Gyro_Cb_Aft_Aux_Batt_Bus) then
            Aft_Aux_Batt_Bus_Load := Aft_Aux_Batt_Bus_Load + 0.01;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Trn_Rt_Cb_Aft_Aux_Batt_Bus) then
            Aft_Aux_Batt_Bus_Load := Aft_Aux_Batt_Bus_Load + 0.01;
         end if;
      end if;

      -- Forward_Auxiliary_Battery_Bus
      if Ctnr.Bus_Collection.Is_Powered (The_Buses, Ele_T.Forward_Auxiliary_Battery_Bus) then
         if Cbs.Is_Powered (The_Cbs, Ele_T.Gyro_Cb) then
            Fwd_Aux_Batt_Bus_Load := Fwd_Aux_Batt_Bus_Load + 0.01;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Altm_Cb_Fwd_Aux_Batt_Bus) then
            Fwd_Aux_Batt_Bus_Load := Fwd_Aux_Batt_Bus_Load + 0.01;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Trn_Rt_Cb) then
            Fwd_Aux_Batt_Bus_Load := Fwd_Aux_Batt_Bus_Load + 0.01;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Uhf_Comm_Cb) then
            Fwd_Aux_Batt_Bus_Load := Fwd_Aux_Batt_Bus_Load + 0.01;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Radio_Rlys_Cb_Fwd_Aux_Batt_Bus) then
            Fwd_Aux_Batt_Bus_Load := Fwd_Aux_Batt_Bus_Load + 0.01;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Fire_Detect_Cb_Fwd_Aux_Batt_Bus) then
            Fwd_Aux_Batt_Bus_Load := Fwd_Aux_Batt_Bus_Load + 0.01;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Stby_Lts_Cb) then
            Fwd_Aux_Batt_Bus_Load := Fwd_Aux_Batt_Bus_Load + 0.01;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Aft_Stby_Cb) then
            Fwd_Aux_Batt_Bus_Load := Fwd_Aux_Batt_Bus_Load + 0.01;
         end if;
      end if;

      -- Aft_Avionics_Generator_Bus
      if Ctnr.Bus_Collection.Is_Powered (The_Buses, Ele_T.Aft_Avionics_Generator_Bus) then
         if Cbs.Is_Powered (The_Cbs, Ele_T.Ehsi_Cb_Aft_Avi_Gen_Bus) then
            Aft_Avi_Gen_Bus_Load := Aft_Avi_Gen_Bus_Load + 0.01;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Rmu_Cb_Aft_Avi_Gen_Bus) then
            Aft_Avi_Gen_Bus_Load := Aft_Avi_Gen_Bus_Load + 0.01;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Gps_Cb_Aft_Avi_Gen_Bus) then
            Aft_Avi_Gen_Bus_Load := Aft_Avi_Gen_Bus_Load + 0.01;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Altm_Cb_Aft_Avi_Gen_Bus) then
            Aft_Avi_Gen_Bus_Load := Aft_Avi_Gen_Bus_Load + 0.01;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Asi_Cb_Aft_Avi_Gen_Bus) then
            Aft_Avi_Gen_Bus_Load := Aft_Avi_Gen_Bus_Load + 0.01;
         end if;
         if Cbs.Is_Powered (The_Cbs, Ele_T.Vvi_Cb_Aft_Avi_Gen_Bus) then
            Aft_Avi_Gen_Bus_Load := Aft_Avi_Gen_Bus_Load + 0.01;
         end if;
      end if;

      -- Aft_Avionics_Battery_Bus
      if Ctnr.Bus_Collection.Is_Powered (The_Buses, Ele_T.Aft_Avionics_Battery_Bus) then
         if Cbs.Is_Powered (The_Cbs, Ele_T.Eadi_Cb_Aft_Avi_Batt_Bus) then
            Aft_Avi_Batt_Bus_Load := Aft_Avi_Batt_Bus_Load + 0.01;
         end if;
      end if;

      -- starter distribution bus
      if Ctnr.Bus_Collection.Is_Powered (The_Buses, Ele_T.Starter_Distribution_Bus) then
         if Jpats_Powerplant.Start_Relay then
            Starter_Distribution_Bus_Load :=
              Power_Supply.Generator.Starter_Load (Ctnr.This_Subsystem.The_Generator);
         end if;
      end if;

      -- load bus load array
      Bus_Loads (Ele_T.Forward_Battery_Bus)            := Fwd_Batt_Bus_Load;
      Bus_Loads (Ele_T.Forward_Generator_Bus)          := Fwd_Gen_Bus_Load;
      Bus_Loads (Ele_T.Forward_Avionics_Battery_Bus)   := Fwd_Avi_Batt_Bus_Load;
      Bus_Loads (Ele_T.Forward_Avionics_Generator_Bus) := Fwd_Avi_Gen_Bus_Load;
      Bus_Loads (Ele_T.Hot_Battery_Bus)                := Hot_Batt_Bus_Load;
      Bus_Loads (Ele_T.Generator_Bus)                  := Gen_Bus_Load;
      Bus_Loads (Ele_T.Aft_Battery_Bus)                := Aft_Batt_Bus_Load;
      Bus_Loads (Ele_T.Aft_Generator_Bus)              := Aft_Gen_Bus_Load;
      Bus_Loads (Ele_T.Aft_Auxiliary_Battery_Bus)      := Aft_Aux_Batt_Bus_Load;
      Bus_Loads (Ele_T.Forward_Auxiliary_Battery_Bus)  := Fwd_Aux_Batt_Bus_Load;
      Bus_Loads (Ele_T.Aft_Avionics_Generator_Bus)     := Aft_Avi_Gen_Bus_Load;
      Bus_Loads (Ele_T.Aft_Avionics_Battery_Bus)       := Aft_Avi_Batt_Bus_Load;
      Bus_Loads (Ele_T.Starter_Distribution_Bus)       := Starter_Distribution_Bus_Load;

      -- update buses
      Ctnr.Bus_Collection.Update
        (The_Buses, Bus_Voltages, Bus_Loads);

   exception
      when others =>
         Log.Report("Jpats_Electrical.Bus_Controller.Update_Buses()");
         raise;
   end Update_Buses;


   procedure Update
     (Iconst : in Float)
   is
   begin
      Update_Paths (Iconst => Iconst);
      Update_Buses (Iconst => Iconst);
   exception
      when others =>
         Log.Report("Jpats_Electrical.Bus_Controller.Update()");
         raise;
   end Update;

end Jpats_Electrical.Bus_Controller;
