-------------------------------------------------------------------------------
--|
--|            FlightSafety International Simulation Systems Division;
--|                     Broken Arrow, OK  USA  918-259-4000
--|
--|                  JPATS T-6A Texan-II Flight Training Device
--|
--|
--|   Engineer:  Howard Landmann
--|
--|   Revision:  (Number and date inserted by Clearcase)
--|
--|
--|  DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
--|  Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
--|  technical or operational data or information from automatic dissemination
--|  under the International Exchange Program or by other means.  This protection
--|  covers information required solely for administrative or operational
--|  purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--|
--|  WARNING:  This document contains technical data whose export is restricted
--|  by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
--|  Executive Order 12470.  Violation of these export control laws is subject
--|  to severe criminal penalties.  Dissemination of this document is controlled
--|  under DOD Directive 5230.25
--|
-------------------------------------------------------------------------------
--| Reference: Kimball, D., "Flight Simulation Mathematical Model of
--|            the Beech MkII Joint Primary Aircraft Training System
--|            (JPATS)", Document 133E794 Rev. B, 11 November, 1998.
--|            Flight Manual Dated April 30,1998.
--|                          AIR FORCE TO 1T-6A-1,
--|                          NAVY (NAVAIR) A1-T6AAA-NFM-100
--|            Standards from Flight Dynamics Group
--|            Flight Safety International, Inc., System Simulation Division
--|            Broken Arrow, OK 74012
-------------------------------------------------------------------------------
--|
with Ada.Text_IO;                              use Ada.Text_IO;
with Ada.Float_Text_IO;                        use Ada.Float_Text_IO;
with Ada.Integer_Text_IO;                      use Ada.Integer_Text_IO;
with Ada.Long_Float_Text_IO;                   use Ada.Long_Float_Text_IO;
with Log;
with Ada.Numerics.Elementary_Functions;

with JPATS_DCLS.Container;
with JPATS_DCLS.Ios_If;
with Jpats_Dcls_Types;
with Vibrations;
with JPATS_IO;
with Jpats_Io_Types;
with Io_Types;

with Angle_Types;
with Coordinate_Types;
with JPATS_Auto_Test;
with JPATS_Atmosphere;
with JPATS_Aircraft_Body;
with JPATS_Secondary_Flight_Controls;
with JPATS_Simulated_Aircraft;
with JPATS_Powerplant.Test;
with JPATS_Avionics;
with Interpolation_Table.Singly_Indexed;

use Ada.Numerics.Elementary_Functions;

package body JPATS_DCLS.Dynamic_Seat_Controller is

   Ios : Container.Ios_Interface_Instance renames Container.This_Ios_Interface;

   Bool_01_N1              : Boolean := False;
   Bool_02_N1              : Boolean := False;
   Bool_03_N1              : Boolean := False;

   procedure Zero_Output_To_Dcls;

   procedure Update
     (This_Subsystem  : in out JPATS_DCLS.Container.Instance;
      Iconst          : in     Float) is

      Sine_Vibration        : array(1..20) of Float := (others => 0.0);
      White_Vibration       : array(1..5)  of Float := (others => 0.0);

      Vibration_Magnitude     : Float;
      Sine_Wave_Multipliers   : Coordinate_Types.Array_Type(1..20);
      White_Noise_Multipliers : Coordinate_Types.Array_Type(1..5);

      Flat_Tire_Multiplier    : Float := 4.0;
      Tar_Strip_Multiplier    : Float := 10.0;
      Gear_Cycle_Multiplier   : Float := 3.0;
      Touchdown_Multiplier    : Float := 2.0;

   begin


      --------------------------------------------------------------------------
      --                      Begin Update Vibrations                         --
      --------------------------------------------------------------------------

      --| for Runway Rumble
      Vibration_Magnitude     := JPATS_Aircraft_Body.Get_Aircraft_Runway_Rumble;
      Sine_Wave_Multipliers   := JPATS_DCLS_Types.Runway_Rumble_Sine_Multiplier;
      White_Noise_Multipliers :=
        JPATS_DCLS_Types.Runway_Rumble_White_Noise_Multiplier;

      Vibrations.Set_Sine_Wave
        (Vibration_Magnitude  ,
         Sine_Wave_Multipliers,
         An_Instance => Container.This_Subsystem.The_Runway_Rumble_Vibration);
      Vibrations.Set_White_Noise
        (Vibration_Magnitude    ,
         White_Noise_Multipliers,
         An_Instance => Container.This_Subsystem.The_Runway_Rumble_Vibration);

      --| for Speedbrake
      Vibration_Magnitude     :=
        JPATS_Secondary_Flight_Controls.Speedbrake_Buffet;
      Sine_Wave_Multipliers   :=
        JPATS_DCLS_Types.Speedbrake_Sine_Multiplier;
      White_Noise_Multipliers :=
        JPATS_DCLS_Types.Speedbrake_White_Noise_Multiplier;

      Vibrations.Set_Sine_Wave
        (Vibration_Magnitude  ,
         Sine_Wave_Multipliers,
         An_Instance => Container.This_Subsystem.The_Speedbrake_Vibration);
      Vibrations.Set_White_Noise
        (Vibration_Magnitude    ,
         White_Noise_Multipliers,
         An_Instance => Container.This_Subsystem.The_Speedbrake_Vibration);

      --| for Wheel Scuffing
      Vibration_Magnitude     :=
        JPATS_Aircraft_Body.Nose_Wheel_scuffing;
      Sine_Wave_Multipliers   :=
        JPATS_DCLS_Types.Wheel_Scuffing_Sine_Multiplier;
      White_Noise_Multipliers :=
        JPATS_DCLS_Types.Wheel_Scuffing_White_Noise_Multiplier;

      Vibrations.Set_Sine_Wave
        (Vibration_Magnitude  ,
         Sine_Wave_Multipliers,
         An_Instance => Container.This_Subsystem.The_Wheel_Scuffing_Vibration);

      Vibrations.Set_White_Noise
        (Vibration_Magnitude    ,
         White_Noise_Multipliers,
         An_Instance => Container.This_Subsystem.The_Wheel_Scuffing_Vibration);

      --| for Landing Gear Buffet
      Vibration_Magnitude     :=
        JPATS_Aircraft_Body.Get_Aircraft_Landing_Gear_Buffet;
      Sine_Wave_Multipliers   :=
        JPATS_DCLS_Types.Landing_Gear_Sine_Multiplier;
      White_Noise_Multipliers :=
        JPATS_DCLS_Types.Landing_Gear_White_Noise_Multiplier;

      Vibrations.Set_Sine_Wave
        (Vibration_Magnitude  ,
         Sine_Wave_Multipliers,
         An_Instance => Container.This_Subsystem.The_Landing_Gear_Vibration);
      Vibrations.Set_White_Noise
        (Vibration_Magnitude    ,
         White_Noise_Multipliers,
         An_Instance => Container.This_Subsystem.The_Landing_Gear_Vibration);

      --| for Engine
      Vibration_Magnitude     :=
        JPATS_Powerplant.Engine_Vibration;
      Sine_Wave_Multipliers   :=
        JPATS_DCLS_Types.Engine_Sine_Multiplier;
      White_Noise_Multipliers :=
        JPATS_DCLS_Types.Engine_White_Noise_Multiplier;

      Vibrations.Set_Sine_Wave
        (Vibration_Magnitude  ,
         Sine_Wave_Multipliers,
         An_Instance => Container.This_Subsystem.The_Engine_Vibration);
      Vibrations.Set_White_Noise
        (Vibration_Magnitude    ,
         White_Noise_Multipliers,
         An_Instance => Container.This_Subsystem.The_Engine_Vibration);

      --| for Flap
      Vibration_Magnitude     :=
        JPATS_Secondary_Flight_Controls.Flap_Buffet;
      Sine_Wave_Multipliers   :=
        JPATS_DCLS_Types.Flap_Sine_Multiplier;
      White_Noise_Multipliers :=
        JPATS_DCLS_Types.Flap_White_Noise_Multiplier;

      Vibrations.Set_Sine_Wave
        (Vibration_Magnitude  ,
         Sine_Wave_Multipliers,
         An_Instance => Container.This_Subsystem.The_Flap_Vibration);
      Vibrations.Set_White_Noise
        (Vibration_Magnitude    ,
         White_Noise_Multipliers,
         An_Instance => Container.This_Subsystem.The_Flap_Vibration);

      --| for Stall
      Vibration_Magnitude     :=
        JPATS_Simulated_Aircraft.Get_Stall_Buffet;
      Sine_Wave_Multipliers   :=
        JPATS_DCLS_Types.Stall_Sine_Multiplier;
      White_Noise_Multipliers :=
        JPATS_DCLS_Types.Stall_White_Noise_Multiplier;

      Vibrations.Set_Sine_Wave
        (Vibration_Magnitude  ,
         Sine_Wave_Multipliers,
         An_Instance => Container.This_Subsystem.The_Stall_Vibration);
      Vibrations.Set_White_Noise
        (Vibration_Magnitude    ,
         White_Noise_Multipliers,
         An_Instance => Container.This_Subsystem.The_Stall_Vibration);

      ----------------------------------------------------------------------------
      --| for Turbulence
      -- the involved equation here is because the difference between 30 and
      -- 60 percent turbulence was not as noticeable as between 10 and 30, so we
      -- "smoothed" in a curve over the linear value.
      ----------------------------------------------------------------------------
      Vibration_Magnitude     :=
        JPATS_Atmosphere.Get_Total_Turbulence_Intensity
        -10.0 * Sin(JPATS_Atmosphere.Get_Total_Turbulence_Intensity,100.0);
      Sine_Wave_Multipliers   :=
        JPATS_DCLS_Types.Turbulence_Sine_Multiplier;
      White_Noise_Multipliers :=
        JPATS_DCLS_Types.Turbulence_White_Noise_Multiplier;

      Vibrations.Set_Sine_Wave
        (Vibration_Magnitude  ,
         Sine_Wave_Multipliers,
         An_Instance => Container.This_Subsystem.The_Turbulence_Vibration);
      Vibrations.Set_White_Noise
        (Vibration_Magnitude    ,
         White_Noise_Multipliers,
         An_Instance => Container.This_Subsystem.The_Turbulence_Vibration);

      --| for Structural Damage Vibration
      Vibration_Magnitude     :=
        Float(Boolean'Pos(JPATS_Atmosphere.Midair));
      Sine_Wave_Multipliers   :=
        JPATS_DCLS_Types.Structural_Damage_Sine_Multiplier;
      White_Noise_Multipliers :=
        JPATS_DCLS_Types.Structural_Damage_White_Noise_Multiplier;

      Vibrations.Set_Sine_Wave
        (Vibration_Magnitude  ,
         Sine_Wave_Multipliers,
         An_Instance => Container.This_Subsystem.The_Structural_Damage_Vibration);
      Vibrations.Set_White_Noise
        (Vibration_Magnitude    ,
         White_Noise_Multipliers,
         An_Instance => Container.This_Subsystem.The_Structural_Damage_Vibration);

      -- End Update Vibrations


      --------------------------------------------------------------------------
      --                      Begin Vibration Discretes                       --
      --------------------------------------------------------------------------

      Container.This_IO_Interface.DCO_Seat_Prim_FltAZA := 0.0;

      if Ios.Dcls_Bool_01 and not Bool_01_N1 then
         Ios.Dcls_Bool_02 := False;
         Ios.Dcls_Bool_03 := False;
      elsif Ios.Dcls_Bool_02 and not Bool_02_N1 then
         Ios.Dcls_Bool_01 := False;
         Ios.Dcls_Bool_03 := False;
      elsif Ios.Dcls_Bool_03 and not Bool_03_N1 then
         Ios.Dcls_Bool_01 := False;
         Ios.Dcls_Bool_02 := False;
      end if;

      Bool_01_N1 := Ios.Dcls_Bool_01;
      Bool_02_N1 := Ios.Dcls_Bool_02;
      Bool_03_N1 := Ios.Dcls_Bool_03;

      if IOS.DCLS_Bool_01 then
         Container.This_Io_Interface.Dco_Seat_Z_Axis_Spef := Ios.DCLS_Float_26;
      else
         Container.This_Io_Interface.Dco_Seat_Z_Axis_Spef :=
           Tar_Strip_Multiplier
           * (JPATS_Aircraft_Body.Tar_Strip_Bump)

           + Flat_Tire_Multiplier
           * (JPATS_Aircraft_Body.Get_Flat_Nose_Amplitude
              + JPATS_Aircraft_Body.Get_Flat_Left_Amplitude
              + JPATS_Aircraft_Body.Get_Flat_Right_Amplitude)

           + Touchdown_Multiplier
           * (JPATS_Aircraft_Body.Get_Left_Td_Bump
              + JPATS_Aircraft_Body.Get_Right_Td_Bump
              + JPATS_Aircraft_Body.Get_Nose_Td_Bump)

           + Gear_Cycle_Multiplier
           * (JPATS_Aircraft_Body.Get_Cycle_Bump);

      end if;
      ----------------------------------------------------------------------------
      -- Stop seat motion with freezes or crash
      ----------------------------------------------------------------------------
      Container.This_IO_Interface.DCO_Seat_Freeze_Flag :=
        JPATS_Simulated_Aircraft.Get_Flight_Freeze
        or JPATS_Simulated_Aircraft.Slew_In_Progress
        or JPATS_Simulated_Aircraft.Get_Crash_Sink_Rate
        or JPATS_Simulated_Aircraft.Get_Crash_Vertical_Load
        or JPATS_Simulated_Aircraft.Get_Crash_Excess_Airspeed
        or JPATS_Simulated_Aircraft.Get_Crash_Mountain_Impact;

      -- Add Sine Vibrations for all categories in each array value

      for I in 1..20 loop
         Sine_Vibration(I) :=
           Vibrations.Get_Sine_Wave
           (An_Instance =>
              Container.This_Subsystem.The_Runway_Rumble_Vibration)(i)
           + Vibrations.Get_Sine_Wave
           (An_Instance =>
              Container.This_Subsystem.The_Speedbrake_Vibration)(i)
           + Vibrations.Get_Sine_Wave
           (An_Instance =>
              Container.This_Subsystem.The_Wheel_Scuffing_Vibration)(i)
           + Vibrations.Get_Sine_Wave
           (An_Instance => Container.This_Subsystem.The_Landing_Gear_Vibration)(i)
           + Vibrations.Get_Sine_Wave
           (An_Instance => Container.This_Subsystem.The_Engine_Vibration)(i)
           + Vibrations.Get_Sine_Wave
           (An_Instance => Container.This_Subsystem.The_Flap_Vibration)(i)
           + Vibrations.Get_Sine_Wave
           (An_Instance => Container.This_Subsystem.The_Stall_Vibration)(i)
           + Vibrations.Get_Sine_Wave
           (An_Instance => Container.This_Subsystem.The_Turbulence_Vibration)(i)
           + Vibrations.Get_Sine_Wave
           (An_Instance =>
              Container.This_Subsystem.The_Structural_Damage_Vibration)(i);
      end loop;

      -- Add White Noise Vibrations for each category in each array value
      for I in 1..5 loop
         White_Vibration(I) :=
           Vibrations.Get_White_Noise
           (An_Instance =>
              Container.This_Subsystem.The_Runway_Rumble_Vibration)(i)
           + Vibrations.Get_White_Noise
           (An_Instance => Container.This_Subsystem.The_Speedbrake_Vibration)(i)
           + Vibrations.Get_White_Noise
           (An_Instance =>
              Container.This_Subsystem.The_Wheel_Scuffing_Vibration)(i)
           + Vibrations.Get_White_Noise
           (An_Instance => Container.This_Subsystem.The_Landing_Gear_Vibration)(i)
           + Vibrations.Get_White_Noise
           (An_Instance => Container.This_Subsystem.The_Engine_Vibration)(i)
           + Vibrations.Get_White_Noise
           (An_Instance => Container.This_Subsystem.The_Flap_Vibration)(i)
           + Vibrations.Get_White_Noise
           (An_Instance => Container.This_Subsystem.The_Stall_Vibration)(i)
           + Vibrations.Get_White_Noise
           (An_Instance => Container.This_Subsystem.The_Turbulence_Vibration)(i)
           + Vibrations.Get_White_Noise
           (An_Instance =>
              Container.This_Subsystem.The_Structural_Damage_Vibration)(i);
      end loop;

      ----------------------------------------------------------------------
      --                     SIN WAVE OUTPUTS                             --
      ----------------------------------------------------------------------
      if IOS.DCLS_Bool_01 then

         Container.This_IO_Interface.DCO_Seat_Sine_1_Ampl  := IOS.DCLS_Float_01;
         Container.This_IO_Interface.DCO_Seat_Sine_2_Ampl  := IOS.DCLS_Float_02;
         Container.This_IO_Interface.DCO_Seat_Sine_3_Ampl  := IOS.DCLS_Float_03;
         Container.This_IO_Interface.DCO_Seat_Sine_4_Ampl  := IOS.DCLS_Float_04;
         Container.This_IO_Interface.DCO_Seat_Sine_5_Ampl  := IOS.DCLS_Float_05;
         Container.This_IO_Interface.DCO_Seat_Sine_6_Ampl  := IOS.DCLS_Float_06;
         Container.This_IO_Interface.DCO_Seat_Sine_7_Ampl  := IOS.DCLS_Float_07;
         Container.This_IO_Interface.DCO_Seat_Sine_8_Ampl  := IOS.DCLS_Float_08;
         Container.This_IO_Interface.DCO_Seat_Sine_9_Ampl  := IOS.DCLS_Float_09;
         Container.This_IO_Interface.DCO_Seat_Sine_10_Ampl := IOS.DCLS_Float_10;
         Container.This_IO_Interface.DCO_Seat_Sine_11_Ampl := IOS.DCLS_Float_11;
         Container.This_IO_Interface.DCO_Seat_Sine_12_Ampl := IOS.DCLS_Float_12;
         Container.This_IO_Interface.DCO_Seat_Sine_13_Ampl := IOS.DCLS_Float_13;
         Container.This_IO_Interface.DCO_Seat_Sine_14_Ampl := IOS.DCLS_Float_14;
         Container.This_IO_Interface.DCO_Seat_Sine_15_Ampl := IOS.DCLS_Float_15;
         Container.This_IO_Interface.DCO_Seat_Sine_16_Ampl := IOS.DCLS_Float_16;
         Container.This_IO_Interface.DCO_Seat_Sine_17_Ampl := IOS.DCLS_Float_17;
         Container.This_IO_Interface.DCO_Seat_Sine_18_Ampl := IOS.DCLS_Float_18;
         Container.This_IO_Interface.DCO_Seat_Sine_19_Ampl := IOS.DCLS_Float_19;
         Container.This_IO_Interface.DCO_Seat_Sine_20_Ampl := IOS.DCLS_Float_20;

      else

         Container.This_IO_Interface.DCO_Seat_Sine_1_Ampl  :=
           Float'Max(0.0,Float'Min(0.3,Sine_Vibration(1)));    -- ECLS_O_0047

         Container.This_IO_Interface.DCO_Seat_Sine_2_Ampl  :=
           Float'Max(0.0,Float'Min(0.3,Sine_Vibration(2)));    -- ECLS_O_0048

         Container.This_IO_Interface.DCO_Seat_Sine_3_Ampl  :=
           Float'Max(0.0,Float'Min(0.3,Sine_Vibration(3)));    -- ECLS_O_0049

         Container.This_IO_Interface.DCO_Seat_Sine_4_Ampl  :=
           Float'Max(0.0,Float'Min(0.3,Sine_Vibration(4)));    -- ECLS_O_0050

         Container.This_IO_Interface.DCO_Seat_Sine_5_Ampl  :=
           Float'Max(0.0,Float'Min(0.3,Sine_Vibration(5)));    -- ECLS_O_0051

         Container.This_IO_Interface.DCO_Seat_Sine_6_Ampl  :=
           Float'Max(0.0,Float'Min(0.3,Sine_Vibration(6)));    -- ECLS_O_0052

         Container.This_IO_Interface.DCO_Seat_Sine_7_Ampl  :=
           Float'Max(0.0,Float'Min(0.3,Sine_Vibration(7)));    -- ECLS_O_0053

         Container.This_IO_Interface.DCO_Seat_Sine_8_Ampl  :=
           Float'Max(0.0,Float'Min(0.3,Sine_Vibration(8)));    -- ECLS_O_0054

         Container.This_IO_Interface.DCO_Seat_Sine_9_Ampl  :=
           Float'Max(0.0,Float'Min(0.3,Sine_Vibration(9)));    -- ECLS_O_0055

         Container.This_IO_Interface.DCO_Seat_Sine_10_Ampl :=
           Float'Max(0.0,Float'Min(0.3,Sine_Vibration(10)));   -- ECLS_O_0056

         Container.This_IO_Interface.DCO_Seat_Sine_11_Ampl :=
           Float'Max(0.0,Float'Min(0.3,Sine_Vibration(11)));   -- ECLS_O_0057

         Container.This_IO_Interface.DCO_Seat_Sine_12_Ampl :=
           Float'Max(0.0,Float'Min(0.3,Sine_Vibration(12)));   -- ECLS_O_0058

         Container.This_IO_Interface.DCO_Seat_Sine_13_Ampl :=
           Float'Max(0.0,Float'Min(0.3,Sine_Vibration(13)));   -- ECLS_O_0059

         Container.This_IO_Interface.DCO_Seat_Sine_14_Ampl :=
           Float'Max(0.0,Float'Min(0.3,Sine_Vibration(14)));   -- ECLS_O_0060

         Container.This_IO_Interface.DCO_Seat_Sine_15_Ampl :=
           Float'Max(0.0,Float'Min(0.3,Sine_Vibration(15)));   -- ECLS_O_0061

         Container.This_IO_Interface.DCO_Seat_Sine_16_Ampl :=
           Float'Max(0.0,Float'Min(0.3,Sine_Vibration(16)));   -- ECLS_O_0062

         Container.This_IO_Interface.DCO_Seat_Sine_17_Ampl :=
           Float'Max(0.0,Float'Min(0.3,Sine_Vibration(17)));   -- ECLS_O_0063

         Container.This_IO_Interface.DCO_Seat_Sine_18_Ampl :=
           Float'Max(0.0,Float'Min(0.3,Sine_Vibration(18)));   -- ECLS_O_0064

         Container.This_IO_Interface.DCO_Seat_Sine_19_Ampl :=
           Float'Max(0.0,Float'Min(0.3,Sine_Vibration(19)));   -- ECLS_O_0065

         Container.This_IO_Interface.DCO_Seat_Sine_20_Ampl :=
           Float'Max(0.0,Float'Min(0.3,Sine_Vibration(20)));   -- ECLS_O_0066

   end if;

      ----------------------------------------------------------------------
      --                   WHITE NOISE VIBRATIONS                         --
      ----------------------------------------------------------------------

      if IOS.DCLS_Bool_01 then

         Container.This_IO_Interface.DCO_Seat_White_1_Var := IOS.Dcls_Float_21;
         Container.This_IO_Interface.DCO_Seat_White_2_Var := IOS.Dcls_Float_22;
         Container.This_IO_Interface.DCO_Seat_White_3_Var := IOS.Dcls_Float_23;
         Container.This_IO_Interface.DCO_Seat_White_4_Var := IOS.Dcls_Float_24;
         Container.This_IO_Interface.DCO_Seat_White_5_Var := IOS.Dcls_Float_25;

      else

         -- ECLS_O_0067
         Container.This_IO_Interface.DCO_Seat_White_1_Var :=
           Float'Max(0.0,Float'Min(0.3,White_Vibration(1)));

         -- ECLS_O_0068
         Container.This_IO_Interface.DCO_Seat_White_2_Var :=
           Float'Max(0.0,Float'Min(0.3,White_Vibration(2)));

         -- ECLS_O_0069
         Container.This_IO_Interface.DCO_Seat_White_3_Var :=
           Float'Max(0.0,Float'min(0.3,White_Vibration(3)));

         -- ECLS_O_0070
         Container.This_IO_Interface.DCO_Seat_White_4_Var :=
           Float'Max(0.0,Float'min(0.3,White_Vibration(4)));

         -- ECLS_O_0071
         Container.This_IO_Interface.DCO_Seat_White_5_Var :=
           Float'Max(0.0,Float'min(0.3,White_Vibration(5)));

      end if;

      ----------------------------------------------------------------
      -- Read inputs for debug purposes
      ----------------------------------------------------------------

      if Ios.Dcls_Bool_02 then
         Ios.Dcls_Float_01 := Container.This_IO_Interface.Dco_Seat_Sine_1_Ampl;
         Ios.Dcls_Float_02 := Container.This_IO_Interface.Dco_Seat_Sine_2_Ampl;
         Ios.Dcls_Float_03 := Container.This_IO_Interface.Dco_Seat_Sine_3_Ampl;
         Ios.Dcls_Float_04 := Container.This_IO_Interface.Dco_Seat_Sine_4_Ampl;
         Ios.Dcls_Float_05 := Container.This_IO_Interface.Dco_Seat_Sine_5_Ampl;
         Ios.Dcls_Float_06 := Container.This_IO_Interface.Dco_Seat_Sine_6_Ampl;
         Ios.Dcls_Float_07 := Container.This_IO_Interface.Dco_Seat_Sine_7_Ampl;
         Ios.Dcls_Float_08 := Container.This_IO_Interface.Dco_Seat_Sine_8_Ampl;
         Ios.Dcls_Float_09 := Container.This_IO_Interface.Dco_Seat_Sine_9_Ampl;
         Ios.Dcls_Float_10 := Container.This_IO_Interface.Dco_Seat_Sine_10_Ampl;
         Ios.Dcls_Float_11 := Container.This_IO_Interface.Dco_Seat_Sine_11_Ampl;
         Ios.Dcls_Float_12 := Container.This_IO_Interface.Dco_Seat_Sine_12_Ampl;
         Ios.Dcls_Float_13 := Container.This_IO_Interface.Dco_Seat_Sine_13_Ampl;
         Ios.Dcls_Float_14 := Container.This_IO_Interface.Dco_Seat_Sine_14_Ampl;
         Ios.Dcls_Float_15 := Container.This_IO_Interface.Dco_Seat_Sine_15_Ampl;
         Ios.Dcls_Float_16 := Container.This_IO_Interface.Dco_Seat_Sine_16_Ampl;
         Ios.Dcls_Float_17 := Container.This_IO_Interface.Dco_Seat_Sine_17_Ampl;
         Ios.Dcls_Float_18 := Container.This_IO_Interface.Dco_Seat_Sine_18_Ampl;
         Ios.Dcls_Float_19 := Container.This_IO_Interface.Dco_Seat_Sine_19_Ampl;
         Ios.Dcls_Float_20 := Container.This_IO_Interface.Dco_Seat_Sine_20_Ampl;
         Ios.Dcls_Float_21 := Container.This_IO_Interface.Dco_Seat_White_1_Var;
         Ios.Dcls_Float_22 := Container.This_IO_Interface.Dco_Seat_White_2_Var;
         Ios.Dcls_Float_23 := Container.This_IO_Interface.Dco_Seat_White_3_Var;
         Ios.Dcls_Float_24 := Container.This_IO_Interface.Dco_Seat_White_4_Var;
         Ios.Dcls_Float_25 := Container.This_IO_Interface.Dco_Seat_White_5_Var;
         Ios.Dcls_Float_26 := Container.This_Io_Interface.Dco_Seat_Z_Axis_Spef;
      end if;

      if Ios.Dcls_Bool_03 then
         Ios.Dcls_Float_01 := JPATS_Aircraft_Body.Get_Aircraft_Runway_Rumble;
         Ios.Dcls_Float_02 := JPATS_Secondary_Flight_Controls.Speedbrake_Buffet;
         Ios.Dcls_Float_03 := JPATS_Aircraft_Body.Nose_Wheel_scuffing;
         Ios.Dcls_Float_04 := JPATS_Aircraft_Body.Get_Aircraft_Landing_Gear_Buffet;
         Ios.Dcls_Float_05 := JPATS_Powerplant.Engine_Vibration;
         Ios.Dcls_Float_06 := JPATS_Secondary_Flight_Controls.Flap_Buffet;
         Ios.Dcls_Float_07 := JPATS_Simulated_Aircraft.Get_Stall_Buffet;
         Ios.Dcls_Float_08 := JPATS_Atmosphere.Get_Total_Turbulence_Intensity;
         Ios.Dcls_Float_09 := Float(Boolean'Pos(JPATS_Atmosphere.Midair));
         Ios.Dcls_Float_10 := JPATS_Aircraft_Body.Tar_Strip_Bump;
         Ios.Dcls_Float_11 := JPATS_Aircraft_Body.Get_Flat_Nose_Amplitude;
         Ios.Dcls_Float_12 := JPATS_Aircraft_Body.Get_Flat_Left_Amplitude;
         Ios.Dcls_Float_13 := JPATS_Aircraft_Body.Get_Flat_Right_Amplitude;
         Ios.Dcls_Float_14 := JPATS_Aircraft_Body.Get_Left_Td_Bump;
         Ios.Dcls_Float_15 := JPATS_Aircraft_Body.Get_Right_Td_Bump;
         Ios.Dcls_Float_16 := JPATS_Aircraft_Body.Get_Nose_Td_Bump;
         Ios.Dcls_Float_17 := JPATS_Aircraft_Body.Get_Cycle_Bump;
         Ios.Dcls_Float_18 := 0.0;
         Ios.Dcls_Float_19 := 0.0;
         Ios.Dcls_Float_20 := 0.0;
         Ios.Dcls_Float_21 := 0.0;
         Ios.Dcls_Float_22 := 0.0;
         Ios.Dcls_Float_23 := 0.0;
         Ios.Dcls_Float_24 := 0.0;
         Ios.Dcls_Float_25 := 0.0;
         Ios.Dcls_Float_26 := 0.0;
      end if;

      ----------------------------------------------------------------
      -- DTP using dynamic seat module
      -- 901, 902, 903 :Latency test
      -- 701, 702, 703 :Dynamic Seat Test
      ----------------------------------------------------------------
      if Jpats_auto_test.At_Phase > 0 then
         -- zero all outputs to dcls
         Zero_Output_To_Dcls;

         if JPATS_Aircraft_Body.Latency_Trigger then
            -- Latency test for dynamic seat
            Container.This_Io_Interface.Dco_Seat_Cyl_Discr_Input :=
              Jpats_Simulated_Aircraft.Load_factor.Z * 0.2;
         elsif Jpats_auto_test.Test_Number =702 then
            case  Jpats_auto_test.Seat_Frequency is
               when  1 => Container.This_IO_Interface.DCO_Seat_Sine_1_Ampl  := Jpats_auto_test.Seat_Position;
               when  2 => Container.This_IO_Interface.DCO_Seat_Sine_2_Ampl  := Jpats_auto_test.Seat_Position;
               when  3 => Container.This_IO_Interface.DCO_Seat_Sine_3_Ampl  := Jpats_auto_test.Seat_Position;
               when  4 => Container.This_IO_Interface.DCO_Seat_Sine_4_Ampl  := Jpats_auto_test.Seat_Position;
               when  5 => Container.This_IO_Interface.DCO_Seat_Sine_5_Ampl  := Jpats_auto_test.Seat_Position;
               when  6 => Container.This_IO_Interface.DCO_Seat_Sine_6_Ampl  := Jpats_auto_test.Seat_Position;
               when  7 => Container.This_IO_Interface.DCO_Seat_Sine_7_Ampl  := Jpats_auto_test.Seat_Position;
               when  8 => Container.This_IO_Interface.DCO_Seat_Sine_8_Ampl  := Jpats_auto_test.Seat_Position;
               when  9 => Container.This_IO_Interface.DCO_Seat_Sine_9_Ampl  := Jpats_auto_test.Seat_Position;
               when 10 => Container.This_IO_Interface.DCO_Seat_Sine_10_Ampl := Jpats_auto_test.Seat_Position;
               when 11 => Container.This_IO_Interface.DCO_Seat_Sine_11_Ampl := Jpats_auto_test.Seat_Position;
               when 12 => Container.This_IO_Interface.DCO_Seat_Sine_12_Ampl := Jpats_auto_test.Seat_Position;
               when 13 => Container.This_IO_Interface.DCO_Seat_Sine_13_Ampl := Jpats_auto_test.Seat_Position;
               when 14 => Container.This_IO_Interface.DCO_Seat_Sine_14_Ampl := Jpats_auto_test.Seat_Position;
               when 15 => Container.This_IO_Interface.DCO_Seat_Sine_15_Ampl := Jpats_auto_test.Seat_Position;
               when 16 => Container.This_IO_Interface.DCO_Seat_Sine_16_Ampl := Jpats_auto_test.Seat_Position;
               when 17 => Container.This_IO_Interface.DCO_Seat_Sine_17_Ampl := Jpats_auto_test.Seat_Position;
               when 18 => Container.This_IO_Interface.DCO_Seat_Sine_18_Ampl := Jpats_auto_test.Seat_Position;
               when 19 => Container.This_IO_Interface.DCO_Seat_Sine_19_Ampl := Jpats_auto_test.Seat_Position;
               when 20 => Container.This_IO_Interface.DCO_Seat_Sine_20_Ampl := Jpats_auto_test.Seat_Position;
               when others => null;
            end case;
         elsif Jpats_auto_test.Test_Number = 701 or
           Jpats_auto_test.Test_Number =703 then
            Container.This_Io_Interface.Dco_Seat_Cyl_Discr_Input := Jpats_auto_test.Seat_Position;
         else
            Container.This_Io_Interface.Dco_Seat_Cyl_Discr_Input := 0.0;
         end if;
      end if;

   end Update;

   procedure Initialize
     (A_Subsystem : in out JPATS_DCLS.Container.Instance) is

   begin

      --Register IOS variables:
      JPATS_dcls.Ios_If.Register_Ios_Variables;


      JPATS_IO.Map_To_Icd_Mnemonic
        (An_Interface    => Jpats_Io_Types.Control_Loading,
         An_Icd_Mnemonic => "DCI_seat_cyl_POS",
         A_Direction     => Io_Types.Input,
         An_Address      => Container.This_Io_Interface.DCI_Seat_Cyl_Pos'Address,
         A_Size          => Container.This_Io_Interface.DCI_Seat_Cyl_Pos'Size );

      Jpats_Io.Map_To_Icd_Mnemonic
       (An_Interface    => Jpats_Io_Types.Control_Loading,
        An_Icd_Mnemonic => "DCI_SEAT_ACCELEROMETER",
        A_Direction     => Io_Types.Input,
        An_Address      => Container.This_Io_Interface.DCI_seat_accelerometer'Address,
        A_Size          => Container.This_Io_Interface.DCI_seat_accelerometer'Size );


      JPATS_IO.Map_To_Icd_Mnemonic
        (An_Interface    => Jpats_Io_Types.Control_Loading,
         An_Icd_Mnemonic => "DCO_Seat_Prim_Fltaza",
         A_Direction     => Io_Types.Output,
         An_Address      => Container.This_Io_Interface.DCO_Seat_Prim_Fltaza'Address,
         A_Size          => Container.This_Io_Interface.DCO_Seat_Prim_Fltaza'Size );


      JPATS_IO.Map_To_Icd_Mnemonic
        (An_Interface    => Jpats_Io_Types.Control_Loading,
         An_Icd_Mnemonic => "DCO_Seat_z_axis_spef",
         A_Direction     => Io_Types.Output,
         An_Address      => Container.This_Io_Interface.DCO_Seat_Z_Axis_Spef'Address,
         A_Size          => Container.This_Io_Interface.DCO_Seat_Z_Axis_Spef'Size );


      JPATS_IO.Map_To_Icd_Mnemonic
        (An_Interface    => Jpats_Io_Types.Control_Loading,
         An_Icd_Mnemonic => "DCO_Seat_Cyl_Discr_Input",
         A_Direction     => Io_Types.Output,
         An_Address      => Container.This_Io_Interface.DCO_Seat_Cyl_Discr_Input'Address,
         A_Size          => Container.This_Io_Interface.DCO_Seat_Cyl_Discr_Input'Size );


      JPATS_IO.Map_To_Icd_Mnemonic
        (An_Interface    => Jpats_Io_Types.Control_Loading,
         An_Icd_Mnemonic => "DCO_Seat_freeze_flag",
         A_Direction     => Io_Types.Output,
         An_Address      => Container.This_Io_Interface.DCO_Seat_Freeze_Flag'Address,
         A_Size          => Container.This_Io_Interface.DCO_Seat_Freeze_Flag'Size );


      JPATS_IO.Map_To_Icd_Mnemonic
        (An_Interface    => Jpats_Io_Types.Control_Loading,
         An_Icd_Mnemonic => "DCO_Seat_Sine_1_Ampl",
         A_Direction     => Io_Types.Output,
         An_Address      => Container.This_Io_Interface.Dco_Seat_Sine_1_Ampl'Address,
         A_Size          => Container.This_Io_Interface.Dco_Seat_Sine_1_Ampl'Size );


      JPATS_IO.Map_To_Icd_Mnemonic
        (An_Interface    => Jpats_Io_Types.Control_Loading,
         An_Icd_Mnemonic => "dco_seat_sine_2_Ampl",
         A_Direction     => Io_Types.Output,
         An_Address      => Container.This_Io_Interface.Dco_Seat_Sine_2_Ampl'Address,
         A_Size          => Container.This_Io_Interface.Dco_Seat_Sine_2_Ampl'Size );


      JPATS_IO.Map_To_Icd_Mnemonic
        (An_Interface    => Jpats_Io_Types.Control_Loading,
         An_Icd_Mnemonic => "dco_seat_sine_3_Ampl",
         A_Direction     => Io_Types.Output,
         An_Address      => Container.This_Io_Interface.Dco_Seat_Sine_3_Ampl'Address,
         A_Size          => Container.This_Io_Interface.Dco_Seat_Sine_3_Ampl'Size );


      JPATS_IO.Map_To_Icd_Mnemonic
        (An_Interface    => Jpats_Io_Types.Control_Loading,
         An_Icd_Mnemonic => "dco_seat_sine_4_Ampl",
         A_Direction     => Io_Types.Output,
         An_Address      => Container.This_Io_Interface.Dco_Seat_Sine_4_Ampl'Address,
         A_Size          => Container.This_Io_Interface.Dco_Seat_Sine_4_Ampl'Size );


      JPATS_IO.Map_To_Icd_Mnemonic
        (An_Interface    => Jpats_Io_Types.Control_Loading,
         An_Icd_Mnemonic => "dco_seat_sine_5_Ampl",
         A_Direction     => Io_Types.Output,
         An_Address      => Container.This_Io_Interface.Dco_Seat_Sine_5_Ampl'Address,
         A_Size          => Container.This_Io_Interface.Dco_Seat_Sine_5_Ampl'Size );


      JPATS_IO.Map_To_Icd_Mnemonic
        (An_Interface    => Jpats_Io_Types.Control_Loading,
         An_Icd_Mnemonic => "dco_seat_sine_6_Ampl",
         A_Direction     => Io_Types.Output,
         An_Address      => Container.This_Io_Interface.Dco_Seat_Sine_6_Ampl'Address,
         A_Size          => Container.This_Io_Interface.Dco_Seat_Sine_6_Ampl'Size );


      JPATS_IO.Map_To_Icd_Mnemonic
        (An_Interface    => Jpats_Io_Types.Control_Loading,
         An_Icd_Mnemonic => "dco_seat_sine_7_Ampl",
         A_Direction     => Io_Types.Output,
         An_Address      => Container.This_Io_Interface.Dco_Seat_Sine_7_Ampl'Address,
         A_Size          => Container.This_Io_Interface.Dco_Seat_Sine_7_Ampl'Size );


      JPATS_IO.Map_To_Icd_Mnemonic
        (An_Interface    => Jpats_Io_Types.Control_Loading,
         An_Icd_Mnemonic => "dco_seat_sine_8_Ampl",
         A_Direction     => Io_Types.Output,
         An_Address      => Container.This_Io_Interface.Dco_Seat_Sine_8_Ampl'Address,
         A_Size          => Container.This_Io_Interface.Dco_Seat_Sine_8_Ampl'Size );


      JPATS_IO.Map_To_Icd_Mnemonic
        (An_Interface    => Jpats_Io_Types.Control_Loading,
         An_Icd_Mnemonic => "dco_seat_sine_9_Ampl",
         A_Direction     => Io_Types.Output,
         An_Address      => Container.This_Io_Interface.Dco_Seat_Sine_9_Ampl'Address,
         A_Size          => Container.This_Io_Interface.Dco_Seat_Sine_9_Ampl'Size );


      JPATS_IO.Map_To_Icd_Mnemonic
        (An_Interface    => Jpats_Io_Types.Control_Loading,
         An_Icd_Mnemonic => "dco_seat_sine_10_Ampl",
         A_Direction     => Io_Types.Output,
         An_Address      => Container.This_Io_Interface.Dco_Seat_Sine_10_Ampl'Address,
         A_Size          => Container.This_Io_Interface.Dco_Seat_Sine_10_Ampl'Size );


      JPATS_IO.Map_To_Icd_Mnemonic
        (An_Interface    => Jpats_Io_Types.Control_Loading,
         An_Icd_Mnemonic => "dco_seat_sine_11_Ampl",
         A_Direction     => Io_Types.Output,
         An_Address      => Container.This_Io_Interface.Dco_Seat_Sine_11_Ampl'Address,
         A_Size          => Container.This_Io_Interface.Dco_Seat_Sine_11_Ampl'Size );


      JPATS_IO.Map_To_Icd_Mnemonic
        (An_Interface    => Jpats_Io_Types.Control_Loading,
         An_Icd_Mnemonic => "dco_seat_sine_12_Ampl",
         A_Direction     => Io_Types.Output,
         An_Address      => Container.This_Io_Interface.Dco_Seat_Sine_12_Ampl'Address,
         A_Size          => Container.This_Io_Interface.Dco_Seat_Sine_12_Ampl'Size );


      JPATS_IO.Map_To_Icd_Mnemonic
        (An_Interface    => Jpats_Io_Types.Control_Loading,
         An_Icd_Mnemonic => "dco_seat_sine_13_Ampl",
         A_Direction     => Io_Types.Output,
         An_Address      => Container.This_Io_Interface.Dco_Seat_Sine_13_Ampl'Address,
         A_Size          => Container.This_Io_Interface.Dco_Seat_Sine_13_Ampl'Size );


      JPATS_IO.Map_To_Icd_Mnemonic
        (An_Interface    => Jpats_Io_Types.Control_Loading,
         An_Icd_Mnemonic => "dco_seat_sine_14_Ampl",
         A_Direction     => Io_Types.Output,
         An_Address      => Container.This_Io_Interface.Dco_Seat_Sine_14_Ampl'Address,
         A_Size          => Container.This_Io_Interface.Dco_Seat_Sine_14_Ampl'Size );


      JPATS_IO.Map_To_Icd_Mnemonic
        (An_Interface    => Jpats_Io_Types.Control_Loading,
         An_Icd_Mnemonic => "dco_seat_sine_15_Ampl",
         A_Direction     => Io_Types.Output,
         An_Address      => Container.This_Io_Interface.Dco_Seat_Sine_15_Ampl'Address,
         A_Size          => Container.This_Io_Interface.Dco_Seat_Sine_15_Ampl'Size );


      JPATS_IO.Map_To_Icd_Mnemonic
        (An_Interface    => Jpats_Io_Types.Control_Loading,
         An_Icd_Mnemonic => "dco_seat_sine_16_Ampl",
         A_Direction     => Io_Types.Output,
         An_Address      => Container.This_Io_Interface.Dco_Seat_Sine_16_Ampl'Address,
         A_Size          => Container.This_Io_Interface.Dco_Seat_Sine_16_Ampl'Size );


      JPATS_IO.Map_To_Icd_Mnemonic
        (An_Interface    => Jpats_Io_Types.Control_Loading,
         An_Icd_Mnemonic => "dco_seat_sine_17_Ampl",
         A_Direction     => Io_Types.Output,
         An_Address      => Container.This_Io_Interface.Dco_Seat_Sine_17_Ampl'Address,
         A_Size          => Container.This_Io_Interface.Dco_Seat_Sine_17_Ampl'Size );


      JPATS_IO.Map_To_Icd_Mnemonic
        (An_Interface    => Jpats_Io_Types.Control_Loading,
         An_Icd_Mnemonic => "dco_seat_sine_18_Ampl",
         A_Direction     => Io_Types.Output,
         An_Address      => Container.This_Io_Interface.Dco_Seat_Sine_18_Ampl'Address,
         A_Size          => Container.This_Io_Interface.Dco_Seat_Sine_18_Ampl'Size );


      JPATS_IO.Map_To_Icd_Mnemonic
        (An_Interface    => Jpats_Io_Types.Control_Loading,
         An_Icd_Mnemonic => "dco_seat_sine_19_Ampl",
         A_Direction     => Io_Types.Output,
         An_Address      => Container.This_Io_Interface.Dco_Seat_Sine_19_Ampl'Address,
         A_Size          => Container.This_Io_Interface.Dco_Seat_Sine_19_Ampl'Size );


      JPATS_IO.Map_To_Icd_Mnemonic
        (An_Interface    => Jpats_Io_Types.Control_Loading,
         An_Icd_Mnemonic => "dco_seat_sine_20_Ampl",
         A_Direction     => Io_Types.Output,
         An_Address      => Container.This_Io_Interface.Dco_Seat_Sine_20_Ampl'Address,
         A_Size          => Container.This_Io_Interface.Dco_Seat_Sine_20_Ampl'Size );


      JPATS_IO.Map_To_Icd_Mnemonic
        (An_Interface    => Jpats_Io_Types.Control_Loading,
         An_Icd_Mnemonic => "DCO_seat_white_1_var",
         A_Direction     => Io_Types.Output,
         An_Address      => Container.This_Io_Interface.DCO_Seat_White_1_Var'Address,
         A_Size          => Container.This_Io_Interface.DCO_Seat_White_1_Var'Size );


      JPATS_IO.Map_To_Icd_Mnemonic
        (An_Interface    => Jpats_Io_Types.Control_Loading,
         An_Icd_Mnemonic => "DCO_seat_white_2_var",
         A_Direction     => Io_Types.Output,
         An_Address      => Container.This_Io_Interface.DCO_Seat_White_2_Var'Address,
         A_Size          => Container.This_Io_Interface.DCO_Seat_White_2_Var'Size );


      JPATS_IO.Map_To_Icd_Mnemonic
        (An_Interface    => Jpats_Io_Types.Control_Loading,
         An_Icd_Mnemonic => "DCO_seat_white_3_var",
         A_Direction     => Io_Types.Output,
         An_Address      => Container.This_Io_Interface.DCO_Seat_White_3_Var'Address,
         A_Size          => Container.This_Io_Interface.DCO_Seat_White_3_Var'Size );


      JPATS_IO.Map_To_Icd_Mnemonic
        (An_Interface    => Jpats_Io_Types.Control_Loading,
         An_Icd_Mnemonic => "DCO_seat_white_4_var",
         A_Direction     => Io_Types.Output,
         An_Address      => Container.This_Io_Interface.DCO_Seat_White_4_Var'Address,
         A_Size          => Container.This_Io_Interface.DCO_Seat_White_4_Var'Size );


      JPATS_IO.Map_To_Icd_Mnemonic
        (An_Interface    => Jpats_Io_Types.Control_Loading,
         An_Icd_Mnemonic => "DCO_seat_white_5_var",
         A_Direction     => Io_Types.Output,
         An_Address      => Container.This_Io_Interface.DCO_Seat_White_5_Var'Address,
         A_Size          => Container.This_Io_Interface.DCO_Seat_White_5_Var'Size );

   end Initialize;

   procedure Read_Tables is

   begin
      null;

   end Read_Tables;

   procedure Zero_Output_To_Dcls is
   begin
      Container.This_Io_Interface.DCO_Seat_Prim_FltAZA        := 0.0;
      Container.This_Io_Interface.DCO_Seat_Z_Axis_Spef        := 0.0;
      Container.This_Io_Interface.DCO_Seat_Cyl_Discr_Input    := 0.0;

      Container.This_Io_Interface.DCO_Seat_Sine_1_Ampl        := 0.0;
      Container.This_Io_Interface.DCO_Seat_Sine_2_Ampl        := 0.0;
      Container.This_Io_Interface.DCO_Seat_Sine_3_Ampl        := 0.0;
      Container.This_Io_Interface.DCO_Seat_Sine_4_Ampl        := 0.0;
      Container.This_Io_Interface.DCO_Seat_Sine_5_Ampl        := 0.0;
      Container.This_Io_Interface.DCO_Seat_Sine_6_Ampl        := 0.0;
      Container.This_Io_Interface.DCO_Seat_Sine_7_Ampl        := 0.0;
      Container.This_Io_Interface.DCO_Seat_Sine_8_Ampl        := 0.0;
      Container.This_Io_Interface.DCO_Seat_Sine_9_Ampl        := 0.0;
      Container.This_Io_Interface.DCO_Seat_Sine_10_Ampl       := 0.0;
      Container.This_Io_Interface.DCO_Seat_Sine_11_Ampl       := 0.0;
      Container.This_Io_Interface.DCO_Seat_Sine_12_Ampl       := 0.0;
      Container.This_Io_Interface.DCO_Seat_Sine_13_Ampl       := 0.0;
      Container.This_Io_Interface.DCO_Seat_Sine_14_Ampl       := 0.0;
      Container.This_Io_Interface.DCO_Seat_Sine_15_Ampl       := 0.0;
      Container.This_Io_Interface.DCO_Seat_Sine_16_Ampl       := 0.0;
      Container.This_Io_Interface.DCO_Seat_Sine_17_Ampl       := 0.0;
      Container.This_Io_Interface.DCO_Seat_Sine_18_Ampl       := 0.0;
      Container.This_Io_Interface.DCO_Seat_Sine_19_Ampl       := 0.0;
      Container.This_Io_Interface.DCO_Seat_Sine_20_Ampl       := 0.0;
      Container.This_Io_Interface.DCO_Seat_White_1_Var        := 0.0;
      Container.This_Io_Interface.DCO_Seat_White_2_Var        := 0.0;
      Container.This_Io_Interface.DCO_Seat_White_3_Var        := 0.0;
      Container.This_Io_Interface.DCO_Seat_White_4_Var        := 0.0;
      Container.This_Io_Interface.DCO_Seat_White_5_Var        := 0.0;
   end Zero_Output_To_Dcls;


end JPATS_DCLS.Dynamic_Seat_Controller;







