-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                 JPATS T-6A Texan-II Flight Training Device
--
--
--  Engineer:  JK Reynolds
--
--  Revision:  (Number and date inserted by Clearcase)
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------
with Text_Io,
  Jpats_Avionics_Types;
use Text_Io,
  Jpats_Avionics_Types;

with ada.Numerics.Elementary_Functions,
  Radio_Utilities;
with JPATS_Atmosphere;

use ada.Numerics.Elementary_Functions;

package body Threat_Control is

   package Ru renames Radio_Utilities;

   Rtemp : Float;

   Traffic_On  : array(Jat.Intruders'Range) of Boolean;
   Warning_Timer : Float;
   Traffic_On_N1 : array(Jat.Intruders'Range) of Boolean;
   Latch_Hdg : Float;
   Latch_Vs : Float;
   Traff_Alti : array(Jat.Intruders'Range) of Float;
   Upper_Altlim : constant := 57999.0;
   Deg_To_Nm : constant := 60.0;
   Ft_Per_Nm : constant := 6076.1;
   Fps_To_Kts : constant := 0.59249;
   Nm_To_Ft : constant := 6076.1;
   Ft_To_Nm : constant := 1.0/Nm_To_Ft;
   Nm_To_Deg : constant := 1.0/Deg_To_Nm;
   Ta_Pad : constant := 7.5;
   Enhanced_Delay : constant := 8.0;
   Reverse_Delay : constant := 4.0;
   Ft_Per_Deg : constant := 6076.1 * 60.0;
   Delta_Lat : array(Jat.Intruders'Range) of Float;
   Traff_Lati : array(Jat.Intruders'Range) of Float;
   Coslat : Float;
   Delta_Lon : array(Jat.Intruders'Range) of Float;
   Traff_Loni : array(Jat.Intruders'Range) of Float;
   Traff_Dst : array(Jat.Intruders'Range) of Float;
   Groundspeed : Float;
   Traff_Spd : array(Jat.Intruders'Range) of Float;
   Traff_Hdg : array(Jat.Intruders'Range) of Float;
   Traff_Hdg_Init : array(Jat.Intruders'Range) of Float;
   Traff_Vs : array(Jat.Intruders'Range) of Float;
   Nxios_Act : Boolean;
   Nx_Time_N1 : Float;
   Nxios_Act_N1 : Boolean;
   Enhanced_Timer : array(Jat.Intruders'Range) of Float;
   Own_Clmb_Dscnd : Integer;
   Track_Own : array(Jat.Intruders'Range) of Boolean;
   Enhanced : array(Jat.Intruders'Range) of Boolean;
   Reversal : array(Jat.Intruders'Range) of Boolean;
   Enhanced_Track : array(Jat.Intruders'Range) of Boolean;
   Reverse_Manuever : array(Jat.Intruders'Range) of Boolean;
   Nx_Init_Vs : array(Jat.Intruders'Range) of Float;
   Nx_Ios_Vel : array(Jat.Intruders'Range) of Float;
   Own_Alt_Final : Float;
   Final_Alt_Saved : array(Jat.Intruders'Range) of Float;
   Own_Sensitivity : Integer;
   Closure_To_Cpa : Float;
   Dmd_Final_Dst : Float;
   Init_Time : array(Jat.Intruders'Range) of Float;
   Thtlat_Final : array(Jat.Intruders'Range) of Float;
   Thtlon_Final : array(Jat.Intruders'Range) of Float;
   Thtalt_Final : array(Jat.Intruders'Range) of Float;
   Tht_Dist_To_Final : array(Jat.Intruders'Range) of Float;
   Ios_Start_N1 : array(Jat.Intruders'Range) of Boolean;
   Nxiostrt_Scen : array(Jat.Intruders'Range) of Boolean;
   Decoy : array(Jat.Intruders'Range) of Boolean;
   Decoy_On : array(Jat.Intruders'Range) of Boolean;
   Decoy_On_N1 : array(Jat.Intruders'Range) of Boolean;
   Decoy_Escape : array(Jat.Intruders'Range) of Boolean;
   Run_Clock : array(Jat.Intruders'Range) of Float;
   Time_Left : array(Jat.Intruders'Range) of Float;
   Solution_Found_N1 : array(Jat.Intruders'Range) of Integer;
   Solution_Reverse_N1 : array(Jat.Intruders'Range) of Boolean;
   Cancel_Threat : array(Jat.Intruders'Range) of Boolean;
   Manuever_Time : array(Jat.Intruders'Range) of Float;
   Alt_Diff : array(Jat.Intruders'Range) of Float;
   Drive_Active_N1 : array(Jat.Intruders'Range) of Boolean;
   --replaces NXIOIALT--old IOS initial altitude input
   Initial_Alt : array(Jat.Intruders'Range) of Float;

   --the following could/should become inputs from the ios for threat
   --control.
   Nx_Time : constant := 5.0;  --ios to collision minutes
   No_Alt : array(Jat.Intruders'Range) of Boolean;
   No_Brg : array(Jat.Intruders'Range) of Boolean;
   No_Txp : array(Jat.Intruders'Range) of Boolean;
   Warn_Flag : Boolean;

   procedure Set_Ground_Elevation
     (An_Instance      : in out Instance;
      Ground_Elevation :        Float) is
   begin
      An_Instance.Ground_Elevation := Ground_Elevation;
   end;

   procedure Set_Aircraft_Position
     (An_Instance       : in out Instance;
      Aircraft_Position :       Lat_Long_Types.Position ) is
   begin
      An_Instance.Aircraft_Position := Aircraft_Position;
   end;

   procedure Set_True_Heading
     (An_Instance  : in out Instance;
      True_Heading :        Jat.Bearing_Type) is
   begin
      An_Instance.True_Heading := True_Heading;
   end;

   procedure Set_Reposition_Active
     (An_Instance       : in out Instance;
      Reposition_Active :        Boolean) is
   begin
      An_Instance.Reposition_Active := Reposition_Active;
   end;

   procedure Set_Earth_Axis_Velocity
     (An_Instance         : in out Instance;
      Earth_Axis_Velocity :        Coordinate_Types.Cartesian) is
   begin
      An_Instance.Earth_Axis_Velocity := Earth_Axis_Velocity;
   end;

   procedure Set_Body_Axis_Velocity
     (An_Instance        : in out Instance;
      Body_Axis_Velocity :        Coordinate_Types.Cartesian) is
   begin
      An_Instance.Body_Axis_Velocity := Body_Axis_Velocity;
   end;

   procedure Set_Rate_Of_Climb
     (An_Instance   : in out Instance;
      Rate_Of_Climb :        Float) is
   begin
      An_Instance.Rate_Of_Climb := Rate_Of_Climb;
   end;

   procedure Set_Weight_On_Wheels
     (An_Instance      : in out Instance;
      Weight_On_Wheels :        Boolean) is
   begin
      An_Instance.Weight_On_Wheels := Weight_On_Wheels;
   end;

   procedure Set_Threat_Active
     (An_Instance  : in out Instance;
      Threat_Active :        Boolean) is
   begin
      An_Instance.Active := Threat_Active;
   end;

   procedure Set_Threat_Page_Start_Button
     (An_Instance             : in out Instance;
      Threat_Page_Start_Button :        Jat.Ios_Boolean_Type) is
   begin
      An_Instance.Threat_Page_Start_Button := Threat_Page_Start_Button;
   end;

   procedure Set_Threat_Page_Aircraft_Type
     (An_Instance              : in out Instance;
      Threat_Page_Aircraft_Type :        Jat.Ios_Aircraft_Type) is
   begin
      An_Instance.Threat_Page_Aircraft_Type := Threat_Page_Aircraft_Type;
   end;

   procedure Set_Threat_Page_Bearing
     (An_Instance        : in out Instance;
      Threat_Page_Bearing :        Jat.Ios_Float_Type) is
   begin
      An_Instance.Threat_Page_Bearing := Threat_Page_Bearing;
   end;

   procedure Set_Threat_Page_Distance
     (An_Instance         : in out Instance;
      Threat_Page_Distance :        Jat.Ios_Float_Type) is
   begin
      An_Instance.Threat_Page_Distance := Threat_Page_Distance;
   end;

   procedure Set_Threat_Page_Altitude
     (An_Instance         : in out Instance;
      Threat_Page_Altitude :        Jat.Ios_Float_Type) is
   begin
      An_Instance.Threat_Page_Altitude := Threat_Page_Altitude;
   end;

   procedure Set_Threat_Page_Time
     (An_Instance     : in out Instance;
      Threat_Page_Time :        Jat.Ios_Float_Type) is
   begin
      An_Instance.Threat_Page_Time := Threat_Page_Time;
   end;

   procedure Set_Threat_Page_Vert_Pos
     (An_Instance         : in out Instance;
      Threat_Page_Vert_Pos :        Jat.Ios_Vertical_Type) is
   begin
      An_Instance.Threat_Page_Vert_Pos := Threat_Page_Vert_Pos;
   end;

   procedure Set_Threat_Page_Speed
     (An_Instance      : in out Instance;
      Threat_Page_Speed :        Jat.Ios_Float_Type) is
   begin
      An_Instance.Threat_Page_Speed := Threat_Page_Speed;
   end;

   procedure Set_Threat_Page_Closure
     (An_Instance        : in out Instance;
      Threat_Page_Closure :        Jat.Ios_Float_Type) is
   begin
      An_Instance.Threat_Page_Closure := Threat_Page_Closure;
   end;

   procedure Set_Flight_Freeze
     (An_Instance   : in out Instance;
      Flight_Freeze :        Boolean) is
   begin
      An_Instance.Flight_Freeze := Flight_Freeze;
   end;

   procedure Set_Solution_Found
     (An_Instance    : in out Instance;
      Solution_Found :        Jat.Ios_Integer_Type) is
   begin
      An_Instance.Solution_Found := Solution_Found;
   end;

   procedure Set_Solution_Reverse
     (An_Instance      : in out Instance;
      Solution_Reverse :        Jat.Ios_Boolean_Type) is
   begin
      An_Instance.Solution_Reverse := Solution_Reverse;
   end;

   procedure Set_Intruder_Drive
     (An_Instance    : in out Instance;
      Intruder_Drive :        Jat.Intruder_Drive_Array_Type) is
   begin
      An_Instance.Intruder_Drive := Intruder_Drive;
   end;

   function Ground_Elevation
     (An_Instance : in Instance)
      return Float is
   begin
      return An_Instance.Ground_Elevation;
   end;

   function Aircraft_Position
     (An_Instance : in Instance)
      return Lat_Long_Types.Position is
   begin
      return An_Instance.Aircraft_Position;
   end;

   function True_Heading
     (An_Instance : in Instance)
      return Jat.Bearing_Type is
   begin
      return An_Instance.True_Heading;
   end;

   function Reposition_Active
     (An_Instance : in Instance)
      return Boolean is
   begin
      return An_Instance.Reposition_Active;
   end;

   function Earth_Axis_Velocity
     (An_Instance : in Instance)
      return Coordinate_Types.Cartesian is
   begin
      return An_Instance.Earth_Axis_Velocity;
   end;

   function Rate_Of_Climb
     (An_Instance : in Instance)
      return Float is
   begin
      return An_Instance.Rate_Of_Climb;
   end;

   function Weight_On_Wheels
     (An_Instance : in Instance)
      return Boolean is
   begin
      return An_Instance.Weight_On_Wheels;
   end;

   function Threat_Active
     (An_Instance : in Instance)
      return Boolean is
   begin
      return An_Instance.Active;
   end;

   function Threat_Page_Start_Button
     (An_Instance : in Instance)
      return Jat.Ios_Boolean_Type is
   begin
      return An_Instance.Threat_Page_Start_Button;
   end;

   function Threat_Page_Aircraft_Type
     (An_Instance : in Instance)
      return Jat.Ios_Aircraft_Type is
   begin
      return An_Instance.Threat_Page_Aircraft_Type;
   end;

   function Threat_Page_Bearing
     (An_Instance : in Instance)
      return Jat.Ios_Float_Type is
   begin
      return An_Instance.Threat_Page_Bearing;
   end;

   function Threat_Page_Distance
     (An_Instance : in Instance)
      return Jat.Ios_Float_Type is
   begin
      return An_Instance.Threat_Page_Distance;
   end;

   function Threat_Page_Altitude
     (An_Instance : in Instance)
      return Jat.Ios_Float_Type is
   begin
      return An_Instance.Threat_Page_Altitude;
   end;

   function Threat_Page_Time
     (An_Instance : in Instance)
      return Jat.Ios_Float_Type is
   begin
      return An_Instance.Threat_Page_Time;
   end;

   function Threat_Page_Vert_Pos
     (An_Instance : in Instance)
      return Jat.Ios_Vertical_Type is
   begin
      return An_Instance.Threat_Page_Vert_Pos;
   end;

   function Threat_Page_Speed
     (An_Instance : in Instance)
      return Jat.Ios_Float_Type is
   begin
      return An_Instance.Threat_Page_Speed;
   end;

   function Threat_Page_Closure
     (An_Instance : in Instance)
      return Jat.Ios_Float_Type is
   begin
      return An_Instance.Threat_Page_Closure;
   end;

   function Solution_Found
     (An_Instance : in Instance)
      return Jat.Ios_Integer_Type is
   begin
      return An_Instance.Solution_Found;
   end;

   function Solution_Reverse
     (An_Instance : in Instance)
      return Jat.Ios_Boolean_Type is
   begin
      return An_Instance.Solution_Reverse;
   end;

   function Intruder_Drive
     (An_Instance : in Instance)
      return Jat.Intruder_Drive_Array_Type is
   begin
      return An_Instance.Intruder_Drive;
   end;


   -- Method Init
   procedure Init(An_Instance : in out Instance) is

      Variable_Registered : Boolean;

   begin

      An_Instance.Ground_Elevation                  := 0.0;
      An_Instance.Aircraft_Position                 := (0.0,0.0,0.0);
      An_Instance.True_Heading                      := 0.0;
      An_Instance.Reposition_Active                 := False;
      An_Instance.Earth_Axis_Velocity               := (0.0,0.0,0.0);
      An_Instance.Body_Axis_Velocity                := (0.0,0.0,0.0);
      An_Instance.Rate_Of_Climb                     := 0.0;
      An_Instance.Weight_On_Wheels                  := False;
      An_Instance.Active                            := False;

      for I in Jat.Intruders'Range loop
         An_Instance.Threat_Page_Start_Button(I)           := False;
         An_Instance.Threat_Page_Aircraft_Type(I)          := Jat.T6a_Navy;
         An_Instance.Threat_Page_Bearing(I)                := 0.0;
         An_Instance.Threat_Page_Distance(I)               := 0.0;
         An_Instance.Threat_Page_Altitude(I)               := 0.0;
         An_Instance.Threat_Page_Time(I)                   := 0.0;
         An_Instance.Threat_Page_Vert_Pos(I)               := Jat.Level;
         An_Instance.Threat_Page_Speed(I)                  := 0.0;
         An_Instance.Threat_Page_Closure(I)                := 0.0;
         An_Instance.Intruder_Drive(I).Aircraft           := Jat.T6a_Navy;
         An_Instance.Intruder_Drive(I).Lat                := 0.0;
         An_Instance.Intruder_Drive(I).Lon                := 0.0;
         An_Instance.Intruder_Drive(I).Active             := False;
         An_Instance.Intruder_Drive(I).Altitude           := 0.0;
         An_Instance.Intruder_Drive(I).Heading            := 0.0;
         An_Instance.Intruder_Drive(I).Completion_Percent := 0.0;
         An_Instance.Intruder_Drive(I).Brg_From_Sim       := 0.0;
         An_Instance.Intruder_Drive(I).Dst_From_Sim       := 0.0;
         An_Instance.Intruder_Drive(I).Roll               := 0.0;
         An_Instance.Intruder_Drive(I).Pitch              := 0.0;

         --locals:
         Traffic_On(I) := False;
         Traffic_On_N1(I) := False;
         Traff_Alti(I) := 0.0;
         Delta_Lat(I) := 0.0;
         Traff_Lati(I) := 0.0;
         Delta_Lon(I) := 0.0;
         Traff_Loni(I) := 0.0;
         Traff_Dst(I) := 0.0;
         Traff_Spd(I) := 0.0;
         Traff_Hdg(I) := 0.0;
         Traff_Hdg_Init(I) := 0.0;
         Traff_Vs(I) := 0.0;
         Enhanced_Timer(I) := 0.0;
         Track_Own(I) := False;
         Enhanced(I) := False;
         Reversal(I) := False;
         Enhanced_Track(I) := False;
         Reverse_Manuever(I) := False;
         Nx_Init_Vs(I) := 0.0;
         Nx_Ios_Vel(I) := 0.0;
         Final_Alt_Saved(I) := 0.0;
         Init_Time(I) := 0.0;
         Thtlat_Final(I) := 0.0;
         Thtlon_Final(I) := 0.0;
         Thtalt_Final(I) := 0.0;
         Tht_Dist_To_Final(I) := 0.0;
         Ios_Start_N1(I) := False;
         Nxiostrt_Scen(I) := False;
         Run_Clock(I) := 0.0;
         Time_Left(I) := 0.0;
         Solution_Found_N1(I) := 0;
         Solution_Reverse_N1(I) := False;
         Cancel_Threat(I) := False;
         Manuever_Time(I) := 0.0;
         Alt_Diff(I) := 0.0;
         Drive_Active_N1(I) := False;
         Initial_Alt(I) := 0.0;
         No_Alt(I) := False;
         No_Brg(I) := False;
         No_Txp(I) := False;

      end loop;

      Warning_Timer := 0.0;
      Latch_Hdg := 0.0;
      Latch_Vs := 0.0;
      Coslat := 0.0;
      Groundspeed := 0.0;
      Nxios_Act := True;
      Nx_Time_N1 := 0.0;
      Nxios_Act_N1 := True;
      Own_Clmb_Dscnd := 0;
      Own_Alt_Final := 0.0;
      Own_Sensitivity := 0;
      Closure_To_Cpa := 0.0;
      Dmd_Final_Dst := 0.0;
      Warn_Flag := False;


   end Init;

   -- Method Update
   procedure Update(Iconst      : in     Float;
                    An_Instance : in out Instance) is

   begin

-- If instructor changes the scenario, or time to CPA chnages,
-- cancel the scenario, if selected.
--


--***************************************************************************
--I am not allowing a threat to be active on the Main TCAS(normal) ios
--page and the scenario page at the same time. Activating these at the
--same time defeats the purpose of having the 12 scenarios and would
--only cause confusion for the instructor and crew. AIOS is making the
--12 scenario button non-selectable if a threat is active on the main
--TCAS page.
--**************************************************************************
         for I in Jat.Intruders'Range loop
            if An_Instance.Threat_Page_Start_Button(I) and
              not Ios_Start_N1(I) then
               Enhanced_Timer(I) := 0.0;

--**************************************************************************
--            Set flags for threat display manipulation
--**************************************************************************
--this section handles special case scenarios that require certain threat
--info not to be displayed or warnings not given
--******************2********************************************************

--c              nxionoalt(i) set by instructor
--c              nxionobrg(i) set by instructor
--c              nxionotxp(i) set by instructor

               Track_Own(I)        := False;
               Enhanced(I)         := False;
               Reversal(I)         := False;
               Enhanced_Track(I)   := False;
               Reverse_Manuever(I) := False;
--**************************************************************************
--           Set threat horiz position via IOS selection
--**************************************************************************

--c              nx_init_brg(i)  set by instructor

--***********************************************************************
--         Set threat vertical speed from scenario type
--***********************************************************************

--c              nxios_vert(i) set by instructor
--c              nxios_vert_rate(i) set by instructor
               if An_Instance.Threat_Page_Vert_Pos(I) = Jat.Above then --above
                  Nx_Init_Vs(I) :=
                    (-1.0)* An_Instance.Threat_Page_Closure(I);
               elsif An_Instance.Threat_Page_Vert_Pos (I) = Jat.Below then
                  Nx_Init_Vs(I) :=
                    An_Instance.Threat_Page_Closure(I);
               else
                  Nx_Init_Vs(I) := 0.0; --level
               end if;
--*************************************************************************
--*          Set threat speed from instructor selection
--*************************************************************************
--c              nxios_speed(1) set by instructor
--               if (abs(JPATS_Atmosphere.Density) > 0.0001) then
--                  Nx_Ios_Vel(I) :=
--                    Ru.Flimit(An_Instance.Threat_Page_Speed(I),
--                              80.0, 1000.0)/(12.15257*
--                                             Sqrt(JPATS_Atmosphere.Density));
--               else
                  Nx_Ios_Vel(I) :=
                    Ru.Flimit(An_Instance.Threat_Page_Speed(I),
                              80.0, 1000.0)*1.689;
--               end if;

--**************************************************************************
--                 Set threat final altitude and distance
--**************************************************************************
--this section will use a look up table to determine the final altitude
--and distance of the threat for any given scenario selected with factoring
--in the ownship climb/descend status.
--**************************************************************************

--c              nx_final_alt(i) set by instructor
--c              nx_final_dst(1) set by instructor

--**************************************************************************
--                Determine the final positions
--**************************************************************************
--The "final" positions we are calculating here are essentially where the
--threat will be at the "TIME TO COLLISION" selected on the IOS.
--**************************************************************************
               if (An_Instance.Threat_Page_Time(I) <= 0.001) then
                  Init_Time(I) := 0.001;
               else
                  Init_Time(I) := An_Instance.Threat_Page_Time(I);
               end if;

               Thtlat_Final(I) :=
                 Float(An_Instance.Aircraft_Position.Latitude) +
                 (An_Instance.Earth_Axis_Velocity.X*Init_Time(I)*60.0 +
                  An_Instance.Threat_Page_Distance(I)*Nm_To_Ft*
                  Cos(An_Instance.True_Heading+90.0,360.0)) *
                 Ft_To_Nm*Nm_To_Deg;
--Put("Thtlat_Final(1):");
--Put_line(Float'Image(Thtlat_Final(1)));
               Thtlon_Final(I) :=
                 Float(An_Instance.Aircraft_Position.Longitude) +
                 (An_Instance.Earth_Axis_Velocity.Y*Init_Time(I)*60.0 +
                  An_Instance.Threat_Page_Distance(I)*Nm_To_Ft*
                  Sin(An_Instance.True_Heading+90.0,360.0)) *
                 Ft_To_Nm*Nm_To_Deg/
                 Cos(Float(An_Instance.Aircraft_Position.Latitude),360.0);
--Put("Thtlon_Final(1):");
--Put_line(Float'Image(Thtlon_Final(1)));
               Thtalt_Final(I) :=
                 An_Instance.Aircraft_Position.Altitude +
                 An_Instance.Threat_Page_altitude(I) +
                 An_Instance.Rate_Of_Climb * Init_Time(I);
--Put("Thtalt_Final(1):");
--Put_line(Float'Image(Thtalt_Final(1)));
--**************************************************************************
--                Set threat initial and final parameters
--**************************************************************************
--nx_tcas_scenario_no will remain at 0 until a scenario is activated
--this will allow the Main TCAS page to still function as usual.
--Once a scenario is activated I force the first element of the initial
--and final parameters to what is set on the TCAS scenario page. Otherwise,
--this branch is not entered and the initial and final values will come
--from the Main TCAS page.
--**************************************************************************

-- Initial position is dependent upon instructor's demanded event time.
-- Event time assumes that the ownship and the threat are on a collision
-- course.  Threat distance is not to ownship, but to ownship final position.
--
               Tht_Dist_To_Final(I) := Nx_Ios_Vel(I) * Init_Time(I)*60.0 *
                 Ft_To_Nm;
               Initial_Alt(I) := (-1.0) * Nx_Init_Vs(I)* Init_Time(I);

            end if;
            --N-1 value for start button
            Ios_Start_N1(I) := An_Instance.Threat_Page_Start_Button(I);
         end loop;

--**************************************************************************
--n-1 for activation.... the data in the branch above need only be set once
--**************************************************************************

--*******************************************************************
      --start from threat page or scenario
      Nxiostrt_Scen(1) := An_Instance.Threat_Page_Start_Button(1);
      Nxiostrt_Scen(2) := An_Instance.Threat_Page_Start_Button(2);

--*******************************************************************
--# main loop
--*******************************************************************
--
--
      for I in Jat.Intruders'Range loop

--*******************************************************************
--# control flags
--*******************************************************************
-- these threat control flags are driven by the "threat start" button
-- on the aios page.
--
--         if An_Instance.Weight_On_Wheels then
--            if (Nxiostrt_Scen(I)) then Warning_Timer := 3.0; end if;
--            if Warning_Timer > 0.0 then
--               Warning_Timer := Warning_Timer - Iconst;
--            end if;
--            Warn_Flag := Warning_Timer > 0.0;
--            Traffic_On(I) := False;
--            Nxiostrt_Scen(I) := False;
--            An_Instance.Threat_Page_Start_Button(I) := False;
--            if I = 1 then Nxios_Act := False; end if;
--         else
         Traffic_On(I) := Nxiostrt_Scen(I) and
           not An_Instance.Reposition_Active;

         Warn_Flag := False;
--         end if;

--*******************************************************************
--# flight profiles
--*******************************************************************
--the threats are controlled by the instructor.  the initial/final bearing,
-- distance, and altitude terms inputed by the instructor will be
-- used to calculate the starting point, collision point, and threat flight
-- profile terms.  The decoy starting point and decoy flight profile terms
-- are based on the threats flight profile terms so decoy interference
-- will be kept minimal.
--
         if Traffic_On(I) and not Traffic_On_N1(I) then
-- latch terms for breakaway
            Latch_Hdg := An_Instance.True_Heading;
            Latch_Vs  := An_Instance.Rate_Of_Climb;
-- Reset the run_clk(i)
            Run_Clock(I) := 0.0;
--## initial point of threat
--calculate initial point of threat with given initial bearing and
-- distance.
--
            Traff_Alti(I) :=
              Ru.Flimit(Thtalt_Final(I) + Initial_Alt(I),
                        An_Instance.Ground_Elevation,Upper_Altlim);
--Put("Traff_Alti(1):");
--Put_line(Float'Image(Traff_Alti(1)));
            Traff_Lati(I) :=
              Thtlat_Final(I) +
              Tht_Dist_To_Final(I) * Nm_To_Deg *
              Cos(An_Instance.True_Heading +
                  An_Instance.Threat_Page_Bearing(I),360.0);
--Put("Traff_Lati(1):");
--Put_line(Float'Image(Traff_Lati(1)));
            Coslat := Cos(Traff_Lati(I),360.0);
            Traff_Loni(I) :=
              Thtlon_Final(I) +
              (Tht_Dist_To_Final(I) * Nm_To_Deg *
               Sin(An_Instance.True_Heading +
                   An_Instance.Threat_Page_Bearing(I),360.0))/Coslat;
--Put("Traff_Loni(1):");
--Put_line(Float'Image(Traff_Loni(1)));

--## ownship time to conflict
-- time to conflict is based on ownship velocity and distance to
--collision point.  time to conflict is in seconds.
--
            Coslat :=
              Cos((Float(An_Instance.Aircraft_Position.Latitude) +
                   Traff_Lati(I))/2.0,360.0);
            Groundspeed := Sqrt(An_Instance.Earth_Axis_Velocity.X ** 2 +
                                An_Instance.Earth_Axis_Velocity.Y ** 2);
            if Groundspeed = 0.0 then Groundspeed := 0.000001; end if;

            Delta_Lat(I) := (Traff_Lati(I) - Thtlat_Final(I)) * Deg_To_Nm;
            Delta_Lon(I) := (Traff_Loni(I) - Thtlon_Final(I)) * Deg_To_Nm *
              Coslat;

--## threat flight profile terms
--
            Traff_Spd(I) := Nx_Ios_Vel(I);
--Put("Delta_Lon(1),Delta_Lat(1):");
--Put(Float'Image(Delta_Lon(1)));
            --Put_line(Float'Image(Delta_Lat(1)));
            Rtemp := Delta_Lat(I);
            if Rtemp = 0.0 then
               Rtemp := 0.00001;
            end if;
            Traff_Hdg(I) :=
              Ru.Xn180(Arctan((-1.0)*Delta_Lon(I),(-1.0)*Rtemp,360.0));
--Put("Traff_Hdg(1):");
--Put_line(Float'Image(Traff_Hdg(1)));
            Traff_Hdg_Init(I) := Traff_Hdg(I);

            Traff_Vs(I) :=
              (Thtalt_Final(I) - Traff_Alti(I)) /(Init_Time(I)* 60.0);
--Put("Traff_Vs(1):");
--Put_line(Float'Image(Traff_Vs(1)));
         end if;

         Traffic_On_N1(I) := Traffic_On(I);



--**************************************************************
--# Send threat aircraft type to v8_model
--**************************************************************
--
         An_Instance.Intruder_Drive(I).Aircraft :=
           An_Instance.Threat_Page_Aircraft_Type(I);
--      end loop;

--**************************************************************
--# run threat and decoys
--*************************************************************
--
--      for I in Jat.Intruders'Range loop

-- If flight freeze is on, don't allow dynamic changes.
         if An_Instance.Flight_Freeze then
            null;
            -- No action

         -- Otherwise, if traffic is selected, run threats
         elsif Traffic_On(I) then
            if Run_Clock(I) = 0.0 then
               An_Instance.Intruder_Drive(I).Altitude := Traff_Alti(I);
               An_Instance.Intruder_Drive(I).Heading := Traff_Hdg(I);
               An_Instance.Intruder_Drive(I).Lat := Long_Float(Traff_Lati(I));
               An_Instance.Intruder_Drive(I).Lon := Long_Float(Traff_Loni(I));
               Coslat :=
                 Cos(Float(An_Instance.Aircraft_Position.Latitude +
                           An_Instance.Intruder_Drive(I).Lat)/2.0,360.0);
               An_Instance.Intruder_Drive(I).Dst_From_Sim :=
                 Sqrt(Float(An_Instance.Aircraft_Position.Latitude -
                            An_Instance.Intruder_Drive(I).Lat)**2 +
                      (Float(An_Instance.Aircraft_Position.Longitude -
                             An_Instance.Intruder_Drive(I).Lon)*Coslat)**2) *
                 Deg_To_Nm;
               An_Instance.Intruder_Drive(I).Active :=
                 An_Instance.Intruder_Drive(I).Dst_From_Sim < 127.0 and
                 abs(An_Instance.Intruder_Drive(I).Altitude -
                     An_Instance.Aircraft_Position.Altitude) < 10_000.0;

               An_Instance.Intruder_Drive(I).Roll  := 0.0;
               An_Instance.Intruder_Drive(I).Pitch := 0.0;
               Run_Clock(I) := Run_Clock(I) + Iconst;
               Time_Left(I) := Init_Time(I) - Run_Clock(I)/60.0;
               Solution_Found_N1(I) := An_Instance.Solution_Found(I);
--
-- Continue threat until it goes out of range.  Note that the threat
-- may start out of range, but it is not canceled until it comes
-- into range.  To prevent threats from staying out of range forever,
-- cancel any threat greater than 150 nm out.
--
            elsif not Cancel_Threat(I) and
              An_Instance.Intruder_Drive(I).Dst_From_Sim < 150.0 and
              (An_Instance.Intruder_Drive(I).Altitude -
               An_Instance.Ground_Elevation) > 0.0 then
               Run_Clock(I) := Run_Clock(I) + Iconst;
               Time_Left(I) := Init_Time(I) - Run_Clock(I)/60.0;
               if (Time_Left(I) = 0.0) then
                  Time_Left(I) := 0.001;
               end if;
--
-- Until we get an RA command, and while some time remains
-- before CPA, continue making adjustments to heading and
-- vertical speed to ensure that we still
-- come fairly close to reaching are desired final distance
-- and altitude.
--
               if Track_Own(I) then
--Put(Boolean'Image(Track_Own(1)));
                  --Recalculate projected final position
                  Thtlat_Final(I) :=
                    Float(An_Instance.Aircraft_Position.Latitude) +
                    (An_Instance.Earth_Axis_Velocity.X*Time_Left(I)*60.0 +
                     An_Instance.Threat_Page_Distance(I)*Nm_To_Ft*
                     Cos(An_Instance.True_Heading + 90.0,360.0))*
                    Ft_To_Nm*Nm_To_Deg;

                  Thtlon_Final(I) :=
                    Float(An_Instance.Aircraft_Position.Longitude) +
                    (An_Instance.Earth_Axis_Velocity.Y*Time_Left(I)*60.0 +
                     An_Instance.Threat_Page_Distance(I)*Nm_To_Ft*
                     Sin(An_Instance.True_Heading+90.0,360.0)) *
                    Ft_To_Nm*Nm_To_Deg/
                    Cos(Float(An_Instance.Aircraft_Position.Latitude),360.0);

                  Thtalt_Final(I) :=
                    An_Instance.Aircraft_Position.Altitude +
                    An_Instance.Threat_Page_Altitude(I) +
                    An_Instance.Rate_Of_Climb * Time_Left(I);

                  Coslat :=
                    Cos((Thtlat_Final(I) +
                         Float(An_Instance.Intruder_Drive(I).Lat))/2.0,360.0);

                  Delta_Lat(I) :=
                    (Float(An_Instance.Intruder_Drive(I).Lat) -
                     Thtlat_Final(I)) * Deg_To_Nm;

                  Delta_Lon(I) :=
                    ((Float(An_Instance.Intruder_Drive(I).Lon) -
                     Thtlon_Final(I)) * Deg_To_Nm) * Coslat;

                  Traff_Dst(I) := Sqrt(Delta_Lat(I)**2 + Delta_Lon(I)**2);

                  Traff_Spd(I) := Traff_Dst(I)* Ft_Per_Nm/(Time_Left(I)*60.0);
--Put("Traff_Dst(I):");
--Put_Line(Float'Image(Traff_Dst(1)));
--Put("Ft_Per_Nm:");
--Put_Line(Float'Image(Ft_Per_Nm));
--Put("Time_Left(I):");
--Put_Line(Float'Image(Time_Left(1)));
                  Rtemp := Delta_Lat(I);
                  if Rtemp = 0.0 then
                     Rtemp := 0.00001;
                  end if;
                  Traff_Hdg(I) :=
                    Ru.Xn180(Arctan((-1.0)*Delta_Lon(I),
                                    (-1.0)*Rtemp,360.0));

                  Traff_Vs(I) :=
                    (Thtalt_Final(I) - An_Instance.Intruder_Drive(I).Altitude)/
                    (Time_Left(I)*60.0);

               end if;

-- If the time left is very small, then discontinue tracking
-- the ownship.  This is necessary for TA's and proximate traffic.
-- Also, if the threat is too close to the calculated
-- final distance, discontinue tracking.
-- Finally, if the threat speed speed increases or decreases by more
-- than 30%, or heading or vertical speed vary too much, stop tracking
-- ownship.  This will only happen if the ownship pilot fails to
-- maintain fairly steady airspeed/heading/altitude.
               if Time_Left(I) < 0.1                                    or
                 (No_Alt(I) and Time_Left(I) < 0.8)                     or
                 Traff_Dst(I) < 0.1                                     or
                 abs(Traff_Spd(I) - Nx_Ios_Vel(I)) > Nx_Ios_Vel(I)*0.3  or
                 abs(Ru.Xn180(Traff_Hdg(I) - Traff_Hdg_Init(I))) > 20.0 then

                  Track_Own(I) := False;
                  Enhanced(I)  := False;
                  Reversal(I)  := False;
               end if;

               An_Instance.Intruder_Drive(I).Heading := Traff_Hdg(I);
               An_Instance.Intruder_Drive(I).Roll := 0.0;
               Rtemp := Traff_Spd(I);
               if Rtemp = 0.0 then
                  Rtemp := 0.00001;
               end if;
               An_Instance.Intruder_Drive(I).Pitch :=
                 Ru.Flimit(Ru.Xn180(Arctan(Traff_Vs(I),rtemp,360.0)+
                                    2.0),-20.0,20.0);
               An_Instance.Intruder_Drive(I).Lat :=
                 An_Instance.Intruder_Drive(I).Lat +
                 Long_Float(Traff_Spd(I)/Ft_Per_Deg*Iconst*
                            Cos(An_Instance.Intruder_Drive(I).Heading,360.0));
--Put("An_Instance.Intruder_Drive(1).Lat:");
--Put_Line(Long_Float'Image(An_Instance.Intruder_Drive(1).Lat));
--Put("Traff_Spd(1):");
--Put_Line(Float'Image(Traff_Spd(1)));
--Put("Ft_Per_Deg:");
--Put_Line(Float'Image(Ft_Per_Deg));
--Put("Iconst:");
--Put_Line(Float'Image(Iconst));
--Put("Cos(An_Instance.Intruder_Drive(I).Heading,360.0):");
--Put_Line(Float'Image(Cos(An_Instance.Intruder_Drive(1).Heading,360.0)));


               Coslat :=
                 Cos(Float(An_Instance.Aircraft_Position.Latitude +
                           An_Instance.Intruder_Drive(I).Lat)/2.0,360.0);

               An_Instance.Intruder_Drive(I).Lon :=
                 An_Instance.Intruder_Drive(I).Lon +
                 Long_Float(Traff_Spd(I)/Ft_Per_Deg*Iconst*
                            Sin(An_Instance.Intruder_Drive(I).Heading,360.0) /
                            Cos(Float(An_Instance.Intruder_Drive(I).Lat),
                                      360.0));

               An_Instance.Intruder_Drive(I).Dst_From_Sim :=
                 Sqrt(Float(An_Instance.Aircraft_Position.Latitude -
                            An_Instance.Intruder_Drive(I).Lat)**2  +
                      (Float(An_Instance.Aircraft_Position.Longitude -
                             An_Instance.Intruder_Drive(I).Lon)*Coslat)**2) *
                      Deg_To_Nm;


               An_Instance.Intruder_Drive(I).Altitude :=
                 Ru.Flimit(An_Instance.Intruder_Drive(I).Altitude +
                           Traff_Vs(I)*Iconst,An_Instance.Ground_Elevation,
                           Upper_Altlim);

               An_Instance.Intruder_Drive(I).Active :=
                 An_Instance.Intruder_Drive(I).Dst_From_Sim < 127.0 and
                 abs(An_Instance.Intruder_Drive(I).Altitude -
                     An_Instance.Aircraft_Position.Altitude) < 10_000.0;

               Rtemp := Float(An_Instance.Intruder_Drive(I).Lat -
                             An_Instance.Aircraft_Position.Latitude);
               if Rtemp = 0.0 then
                  Rtemp := 0.00001;
               end if;
               An_Instance.Intruder_Drive(I).Brg_From_Sim :=
                 Ru.Xn180
                 (Arctan(Float(An_Instance.Intruder_Drive(I).Lon -
                               An_Instance.Aircraft_Position.Longitude)*Coslat,
                         Rtemp,
                         360.0)- An_Instance.True_Heading);

            else

               An_Instance.Intruder_Drive(I).Dst_From_Sim := 0.0;
               An_Instance.Intruder_Drive(I).Brg_From_Sim := 0.0;
               An_Instance.Intruder_Drive(I).Altitude     := 0.0;
               An_Instance.Intruder_Drive(I).Heading      := 0.0;
               An_Instance.Intruder_Drive(I).Active       := False;
               An_Instance.Intruder_Drive(I).Lat          := 0.0;
               An_Instance.Intruder_Drive(I).Lon          := 0.0;
               An_Instance.Intruder_Drive(I).Roll         := 0.0;
               An_Instance.Intruder_Drive(I).Pitch        := 0.0;
               Nxiostrt_Scen(I)                           := False;
               An_Instance.Threat_Page_Start_Button(I)     := False;
               Reverse_Manuever(I)                        := False;

-- For right now, scenario's only use one threat at a time
-- if that changes in the future, then this following line
-- of code will have to modified.

               Decoy_On(I) := False;
            end if;
         else

            Coslat := 0.0;
            Traff_Alti(I) := 0.0;
            Traff_Lati(I) := 0.0;
            Traff_Loni(I) := 0.0;
            Traff_Dst(I)  := 0.0;
            Traff_Spd(I)  := 0.0;
            Traff_Hdg(I)  := 0.0;
            Traff_Vs(I)   := 0.0;
            Run_Clock(I)  := 0.0;
            An_Instance.Intruder_Drive(I).Active       := False;
            An_Instance.Intruder_Drive(I).Dst_From_Sim := 0.0;
            An_Instance.Intruder_Drive(I).Brg_From_Sim := 0.0;
            An_Instance.Intruder_Drive(I).Altitude     := 0.0;
            An_Instance.Intruder_Drive(I).Heading      := 0.0;
            An_Instance.Intruder_Drive(I).Lat          := 0.0;
            An_Instance.Intruder_Drive(I).Lon          := 0.0;

            Decoy_On(I) := False;
            An_Instance.Threat_Page_Start_Button(I)     := False;

            Reverse_Manuever(I) := False;

-- For right now, scenario's only use one threat at a time
-- if that changes in the future, then this following line
-- of code will have to modified.

         end if;

-- Cancel threat if the threat was displayed on the displays, but
-- is now out of range
--
         Cancel_Threat(I) :=
           Drive_Active_N1(I) and not An_Instance.Intruder_Drive(I).Active;
         Drive_Active_N1(I) := An_Instance.Intruder_Drive(I).Active;

      end loop;

   end Update;


end Threat_Control;
